/*
GAME PROGRAMMING TUTORIAL
Copyright (C) 1996 Emmanuel Lagare

DIRTY RECTANGLES

Double buffering produces shear and flicker free animation. However, it can
be very slower than directly accessing the video buffer since we are
completely redrawing the whole screen even though only some portions have
actually changed. The solution is of course drawing only these portions
that have changed.

Everytime a portion of the double buffer is modified, we store the upper
left and lower right coordinates of the smallest rectangular area (the
dirty rectangle) containing the portion that has been modified. We also
increase the count of dirty rectangles. We shall call this function 
register_dirty_rect(). In our example, we will embed this function in our
draw_sprite and erase_sprite functions. You can of course do anything you
want with it.

Then instead of calling show_double_buffer, we call a new function which
will copy these dirty rectangles from the double buffer into the video
buffer during refresh time. We also reset the count of dirty rectangles.
We shall call this function show_dirty_rects().

Also, we must set an upper bound on the number of rectangles, so that when
we reach that bound we simply show the double buffer. This is useful if
most of the screen has been modified.

Below is the usual example.
*/

#include <sys/nearptr.h>
#include <go32.h> /* for _dos_ds */
#include <dos.h>

#define GRAPHICS        0x013
#define TEXT            0x03

char *video_buffer = (char *)0xa0000;
char *double_buffer;

typedef struct sprite_type {
        unsigned x, y;                  /* location */
        unsigned x_vel, y_vel;          /* velocity */
        char width, height;             /* dimensions */
        char num_frames;                /* number of frames */
        char cur_frame;                 /* current frame */
        char *background, **frame;
} sprite;

void set_video_mode(int mode)
{
        union REGS regs;

        regs.x.ax = mode;
        int86(0x10, &regs, &regs);
}

void put_pixel(short int x, short int y, char color)
{
        double_buffer[(y << 8) + (y << 6) + x] = color;
}

void bit_blt(int x, int y, char *bitmap)
{
        int yindex, offset = (y << 8) + (y << 6) + x;
        char width = bitmap[0], height = bitmap[1];

        bitmap += 2; /* skip the first two bytes */
        for (yindex = 0; yindex < height; yindex++) {
                memcpy((char *)double_buffer+offset,bitmap,width);
                offset += 320; /* next line of double buffer */
                bitmap += width; /* next line of bitmap */
        }
}

void reverse_bit_blt(int x, int y, char *bitmap)
{
        int yindex, offset = (y << 8) + (y << 6) + x;
        char width = bitmap[0], height = bitmap[1];

        bitmap += 2; /* skip the first two bytes */
        for (yindex = 0; yindex < height; yindex++) {
                memcpy(bitmap,(char *)double_buffer+offset,width);
                offset += 320; /* next line of double buffer */
                bitmap += width; /* next line of bitmap */
        }
}

void transparent_bit_blt(int x, int y, char *buffer)
{
        int xindex, yindex, offset = (y << 8) + (y << 6) + x;
        char width = buffer[0], height = buffer[1];

        buffer += 2; /* skip the first two bytes */
        for (yindex = 0; yindex < height; yindex++) {
                for (xindex = 0; xindex < width; xindex++) {
                        if (buffer[xindex]) {
                                double_buffer[offset+xindex] = buffer[xindex];
                        }
                }
                offset += 320; /* next line of double buffer */
                buffer += width; /* next line of bitmap */
        }
}

void clear(char *buffer, char color)
{
        memset(buffer,color,64000);
}

void wait_for_vsync(void)
{
        while (inportb(0x3da) & 0x08); /* vga is in retrace */
        while (!(inportb(0x3da) & 0x08)); /* wait for start of retrace */
}

void show_double_buffer(void)
{
        wait_for_vsync();
        memcpy((char *)video_buffer,(char *)double_buffer,64000);
}

#define MAXDIRTYRECTS 40

typedef struct rect_type {
        int x1, y1, x2, y2;
} rect;

rect dirty_rect[MAXDIRTYRECTS];
int num_dirty_rects = 0;

/* new */
void register_dirty_rect(int x1, int y1, int x2, int y2)
{
        rect *tmp_rect;

        if (num_dirty_rects > MAXDIRTYRECTS) return;
        tmp_rect = &dirty_rect[num_dirty_rects];
        tmp_rect->x1 = x1;
        tmp_rect->y1 = y1;
        tmp_rect->x2 = x2;
        tmp_rect->y2 = y2;
        num_dirty_rects++;
}

/* new */
void show_dirty_rects(void)
{
        int rectindex, xindex, yindex, width, height, offset;
        rect *tmp_rect;

        if (num_dirty_rects > MAXDIRTYRECTS) {
                show_double_buffer();
                return;
        }
        for (rectindex = 0; rectindex < num_dirty_rects; rectindex++) {
                tmp_rect = &dirty_rect[rectindex];
                width = tmp_rect->x2 - tmp_rect->x1;
                height = tmp_rect->y2 - tmp_rect->y1;
                offset = (tmp_rect->y1 << 8) + (tmp_rect->y1 << 6) +
                        tmp_rect->x1;
                for (yindex = 0; yindex < height; yindex++) {
                        memcpy((char *)video_buffer+offset,
                                (char *)double_buffer+offset,
                                width);
                        offset += 320; /* next line */
                }
        }
        num_dirty_rects = 0;
}

void draw_sprite(sprite *sprite)
{
        sprite->x += sprite->x_vel; sprite->y += sprite->y_vel;
        reverse_bit_blt(sprite->x,sprite->y,sprite->background);
        transparent_bit_blt(sprite->x,sprite->y,sprite->frame[sprite->cur_frame]);
        register_dirty_rect(sprite->x,sprite->y,
                sprite->x+(int)sprite->width,sprite->y+(int)sprite->height);
}

void erase_sprite(sprite *sprite)
{
        bit_blt(sprite->x,sprite->y,sprite->background);
        register_dirty_rect(sprite->x,sprite->y,
                sprite->x+(int)sprite->width,sprite->y+(int)sprite->height);
}

char bitmap[102] = {
        10,10,                          /* width and height of block */
        15,15,15,15,15,15,15,15,15,15,  /* strip 1 */
        15,00,00,00,00,00,00,00,00,15,
        15,00,00,00,00,00,00,00,00,15,
        15,00,00,00,00,00,00,00,00,15,
        15,00,00,00,00,00,00,00,00,15,
        15,00,00,00,00,00,00,00,00,15,
        15,00,00,00,00,00,00,00,00,15,
        15,00,00,00,00,00,00,00,00,15,
        15,00,00,00,00,00,00,00,00,15,
        15,15,15,15,15,15,15,15,15,15   /* strip 10 */
};

void main(void)
{
        unsigned count;
        sprite sprite;

        /* disable all memory protection */
        __djgpp_nearptr_enable();
        video_buffer += __djgpp_conventional_base;

        /* setup double buffer */
        double_buffer = (char *)malloc(64000);
        clear(double_buffer,0);

        /* go to graphics mode */
        set_video_mode(GRAPHICS);

        /* initialize our sprite */
        sprite.x = 0; sprite.y = 95;
        sprite.x_vel = 1; sprite.y_vel = 0;
        sprite.width = 10; sprite.height = 10;
        sprite.num_frames = 1;
        sprite.cur_frame = 0;
        sprite.background = (char *)malloc(sprite.width*sprite.height+2);
        sprite.background[0] = 10; sprite.background[1] = 10;
        sprite.frame = (char **)malloc(sizeof(char *)*sprite.num_frames);
        sprite.frame[0] = (char *)&bitmap;

        /* fill up the screen with random pixels */
        for(count = 0; count < 32000; count++) {
                put_pixel(rand() % 320, rand() % 200, rand() % 256);
        }

        show_double_buffer();

        /* move sprite from left to right */
        for(count = 0; count < 310; count++) {
                draw_sprite(&sprite);
                show_dirty_rects();
                delay(10);
                erase_sprite(&sprite);
        }

        /* go back to text mode */
        set_video_mode(TEXT);

        /* reset double buffer */
        free(double_buffer);

        /* reenable memory protection */
        __djgpp_nearptr_disable();
}
