/*
GAME PROGRAMMING TUTORIAL
Copyright (C) 1996 Emmanuel Lagare

GETTING INPUT FROM THE JOYSTICK

Special thanks to Steve Carter and Thrustmaster, Inc. for the joystick
programming info.

First of all, let's get reading the joystick buttons out of the way. To read
a joystick button you simply write a 0 to the joystick port at 0x201. Then
read the port, then invert the value you get. The upper 4 bits, let's call it 
the button nibble (4 bits) which are set, i.e. has a value of 1,
corresponds to a pressed button. So to get the  value of a specific button,
you simply AND the button nibble with the bit masks 0x10, 0x20, 0x40, and
0x80 for joystick 1 button A, joystick 1 button B, joystick 2 button A, and
joystick 2 button B, respectively. Pretty simple.

Getting input from the joystick axes, on the other hand, is a complicated
beast. To start you write a 0 to the joystick port. You then read the
joystick port. The lower 4 bits, let's call them the axes nibble, is 1.
Increment a counter and read the joystick port again until the bit you are
watching for reverts to 0. This process is called polling.

So how do you know if it has reverted to 0? You AND the axes nibble with
the bit masks 0x01, 0x02, 0x04, and 0x08 for joystick 1 x-axis, joystick 1 
y-axis, joystick 2 x-axis, and joystick 2 y-axis, respectively. Then if the
result is non-zero, you can go on polling. Better yet, you can read the 
x and y axes of a joystick at the same time (this is what the code does).
You can even read all the axes at the same time. You wouldn't want to do this
however, since the absence of the second joystick would cause long delays
caused by the timeout.

Timeout? What timeout? Well, there's got to be a point when you have to stop
counting. At that point, there probably is no joystick. In fact, the timeout
is what we will be using to detect for the presence of a joystick.

Below is the implementation and example. 
*/

#include <dpmi.h>
#include <go32.h>
#include <pc.h>

#define JOYPORT         0x201   /* game port is at port 0x201 */

#define _1_A            0x10    /* joystick 1, button A */
#define _1_B            0x20    /* joystick 1, button B */
#define _2_A            0x40    /* joystick 2, button A */
#define _2_B            0x80    /* joystick 2, button B */

#define _1_X            0x01    /* joystick 1, x axis */
#define _1_Y            0x02    /* joystick 1, y axis */
#define _2_X            0x04    /* joystick 2, x axis */
#define _2_Y            0x08    /* joystick 2, y axis */

/* joystick */
unsigned int joy_1_max_x, joy_1_max_y,
	joy_1_min_x, joy_1_min_y,
	joy_1_cx, joy_1_cy,
	joy_2_max_x, joy_2_max_y,
	joy_2_min_x, joy_2_min_y,
	joy_2_cx, joy_2_cy;

unsigned char joystick_button(unsigned char button)
{
	outportb(JOYPORT,0); /* clear the latch and request a sample */

	/* invert button then and with request */
	return( ~inportb(JOYPORT) & button);
}

int joystick_pos(int *x, int *y)
{
	unsigned char portvalue, xfire, yfire;

	(*x) = (*y) = 0;
	asm("cli");
	outportb(JOYPORT,0); /* clear the latch and request a sample */
	do {
		portvalue = inportb(JOYPORT);
		if (portvalue & _1_X) {
			(*x)++;
		}
		if (portvalue & _1_Y) {
		       (*y)++;
		}
	} while ((portvalue & 0x03) && ((*y) < 0xffff) && ((*x) < 0xffff));
	asm("sti");

	if ((*x >= 0xffff) || (*y >= 0xffff)) return 0; /* timout */

	return 1;
}

unsigned int detect_joystick(void)
{
	int x, y;

	return joystick_pos(&x,&y);
}

void joystick_calibrate(void)
{
	unsigned int x_new,y_new; // temp joystick positions

	joy_1_min_x = joy_1_min_y = 0xffff;
	joy_1_max_x = joy_1_max_y = 0;

	printf("Calibrating Joystick\n");

	printf("Swirl the joystick and press a button\n");
	do {
		joystick_pos(&x_new,&y_new);
		if (x_new < joy_1_min_x) joy_1_min_x = x_new;
		if (x_new > joy_1_max_x) joy_1_max_x = x_new;
		if (y_new < joy_1_min_y) joy_1_min_y = y_new;
		if (y_new > joy_1_max_y) joy_1_max_y = y_new;
	} while(!joystick_button(_1_A | _1_B));

	while(joystick_button(_1_A | _1_B)) {
	}

	printf("Center the joystick and press a button\n");
	while(!joystick_button(_1_A | _1_B)) {
		joystick_pos(&joy_1_cx,&joy_2_cx);
	}

	while(joystick_button(_1_A | _1_B)) {
	}
}


void main(void)
{
	int x = 0, y = 0;

	if (detect_joystick()) {
		printf("Joystick found - press buttons 1 and 2 to quit\n");
		while(!(joystick_button(_1_A) && joystick_button(_1_B))) {
			joystick_pos(&x,&y);
			printf("%9d %9d ",x,y);
			if (joystick_button(_1_A)) printf("A ");
			else printf("a ");
			if (joystick_button(_1_B)) printf("B");
			else printf("b");
			printf("\n");
		}
	} else printf("No joystick found");
}
