/*
GAME PROGRAMMING TUTORIAL
Copyright (C) 1996 Emmanuel Lagare

SIMPLE LINES

To further understand the memory layout of the VGA buffer, we will now
implement functions for plotting horizontal and vertical lines.

Plotting horizontal lines is simply a matter of fixing the y value and
plotting pixels from the first x to the second x. Vertical lines are
fixed in the x value and plotting pixels from the first y to the second y.

But no. We'll not use the pixel plotting function. Instead for the
horizontal line, we take advantage of the video buffer's (in Mode 13h)
memory structure. First we compute the offset in the video buffer of the
line's first point (the way we did using pixels), and then the length.
Then it's a simple matter of calling the memset() function.

For the vertical line also compute the offset, plot a pixel then increment
the offset with the width of the screen. We keep on doing this until we 
reach the next y value. This way we can save a lot on the stack operations
which are necessary when you call functions.

Note that the horizontal line drawing function assume that the first x value
is larger than the second x value. The same goes for the vertical line
drawing function. Why? Since you know that you will be plotting a vertical or
horizontal line, chances are you know the proper way to feed the values.

Below is the implementation and example.
*/

#include <sys/nearptr.h>
#include <go32.h> /* for _dos_ds */
#include <dos.h>

#define GRAPHICS        0x013
#define TEXT            0x03

char *video_buffer = (char *)0xa0000;

void set_video_mode(int mode)
{
        union REGS regs;

        regs.x.ax = mode;
        int86(0x10, &regs, &regs);
}

/* new */
void horizontal_line(short int x1, short int x2, short int y, char color)
{
        unsigned short int offset = (y << 8) + (y << 6) + x1;
        
        memset(&video_buffer[offset], color, x2-x1);
}

/* new */
void vertical_line(short int x, short int y1, short int y2, char color)
{
        unsigned short int offset = (y1 << 8) + (y1 << 6) + x;
        
        while (y1 <= y2) {
                video_buffer[offset] = color;
                y1++; offset += 320;
        }
}

void main(void)
{
        unsigned count;

        /* disable all memory protection */
        __djgpp_nearptr_enable();
        video_buffer += __djgpp_conventional_base;

        /* go to graphics mode */
        set_video_mode(GRAPHICS);

        /* fill up the screen with horizontal lines */
        for(count = 0; count < 200; count++) {
                horizontal_line(0, 319, count, rand() % 256);        
        }
        
        /* fill up the screen with vertical lines */
        for(count = 0; count < 320; count++) {
                vertical_line(count, 0, 199, rand() % 256);        
        }
        
        /* go back to text mode */
        set_video_mode(TEXT);

        /* reenable memory protection */
        __djgpp_nearptr_disable();
}
