/*
GAME PROGRAMMING TUTORIAL
Copyright (C) 1996 Emmanuel Lagare

COMPLEX LINES

Last time, we looked at ways to plot simple vertical and horizontal lines.
What if we want to draw lines of arbitrary slopes? No problem, it was 
already solved by a guy named Bresenham. Bresenham's algorithm-- okay, it
should be "Bresenham's Line Drawing Algorithm" but I felt that would be too
long. Besides, The line drawing algorithm seems to be the most famous of
his algorithms-- allows us to draw lines of arbitrary slopes. Theoretically
drawing such a line is simple. For example, if we are to plot a line from
(0,0) to (11,2) we would simply divide 12 (pixel 0 to 11) by 3 (pixel 0 to 2) 
to get 4. We then plot pixels 4 times, move down, plot another 4 pixels and
so on.

(0,0) ****
          ****
              **** (11,2)

However, due to the nature of the video buffer, thing get more complicated
when you don't get a clean divide, for example 13 divided by 3 is equal to
4.333. How do you plot that?

The solution is some sort of iterative division. We plot pixels along the 
major axis (the axis where you travel longer) while incrementing (or
decrement, whichever applies) a counter by the smaller number (3 in our
example) until the counter is greater than or equal to the bigger number
(12 in our example). When that happens, we do an iteration of the iterative
division by subtracting the bigger number from the counter and move the pen
(down in our example). The remainder (which will only happen is we don't get
a clean divide if we were to actually divide it) is left over for the next
iteration.

So if we were to plot a line from (0,0) to (12,2) using Bresenham's line
drawing algorithm, we would get the following line:

(0,0) *****
           ****
               **** (12,2)

Vertical and horizontal line cases can also be checked, but that is left to
the reader as an exercise :).

Below is the implementation and example.
*/

#include <sys/nearptr.h>
#include <go32.h> /* for _dos_ds */
#include <dos.h>

#define GRAPHICS        0x013
#define TEXT            0x03

char *video_buffer = (char *)0xa0000;

void set_video_mode(int mode)
{
        union REGS regs;

        regs.x.ax = mode;
        int86(0x10, &regs, &regs);
}

/* new */
void line(short int x1, short int y1, short int x2, short int y2, char color)
{
        short int dx, dy, sdx, sdy, x, y, px, py;

        /* the change in x and y */
        dx = x2 - x1;
        dy = y2 - y1;

        /* signs of dx and dy, for getting absolute value */
        /* and for moving our pen */
        sdx = (dx < 0) ? -1 : 1;
        sdy = (dy < 0) ? -1 : 1;

        /* the values to be used for incrementing/decrementing counters */
        dx = sdx * dx + 1;
        dy = sdy * dy + 1;

        /* the counters */
        x = y = 0;

        /* the pen */
        px = x1;
        py = y1;

        if (dx >= dy) { /* x axis s major */
                for (x = 0; x < dx; x++) {
                        video_buffer[(py << 8) + (py << 6) + px] = color;
                        y += dy;
                        if (y >= dx) {
                                y -= dx;
                                py += sdy;
                        }
                        px += sdx;
                }
        } else { /* y axis is major */
                for (y = 0; y < dy; y++) {
                        video_buffer[(py << 8) + (py << 6) + px] = color;
                        x += dx;
                        if (x >= dy) {
                                x -= dy;
                                px += sdx;
                        }
                        py += sdy;
                }
        }
}

void main(void)
{
        unsigned count;

        /* disable all memory protection */
        __djgpp_nearptr_enable();
        video_buffer += __djgpp_conventional_base;

        /* go to graphics mode */
        set_video_mode(GRAPHICS);

        /* fill up the screen with lines */
        for(count = 0; count < 32000; count++) {
                line(rand()%320,rand()%200,rand()%320,rand()%200,rand()%256);
        }
        
        /* go back to text mode */
        set_video_mode(TEXT);

        /* reenable memory protection */
        __djgpp_nearptr_disable();
}
