/*
GAME PROGRAMMING TUTORIAL
Copyright (C) 1996 Emmanuel Lagare

DETECTING SIMULTANEOUS KEYPRESSES

Thanks to Mr. Eric Domazlicky (siberia@prodigy.com) for helping me with this.

Keypresses trigger a "make" code which tell the computer that a key has been
pressed. When the key is released a corresponding "break" code is sent.
Break code is the make code plus 128.

So to detect simultaneous keypresses all we need to do is keep the state
of the keys. Setting the states of individual keys when make is detected
and resetting when break is detected.

To do this we need to set up our own keyboard interrupt handler (explanations
of which are available in most DOS programming books) where we do the things
we want to do. Below is a sample code to demonstrate this.
*/

#include <dpmi.h>
#include <go32.h>
#include <pc.h>

#define UP           0
#define DOWN         1
#define LEFT         2
#define RIGHT        3

unsigned char raw_key, key_table[4];
	
_go32_dpmi_seginfo old_key_handler,new_key_handler;

void key_handler(void)
{
	unsigned char al, ah;

	asm("cli; pusha");
	
	raw_key = inportb(0x60);
		
	al = inportb(0x61); 
	al |= 0x82;
	outportb(0x61, al);       
	al &= 0x7f;
	outportb(0x61, al);

	/* you have the option of putting this outside */
	switch(raw_key) {
		/* make codes */
		case 72: /* up */
			key_table[UP] = 1;
			break;
		case 80: /* down */
			key_table[DOWN] = 1;
			break;
		case 75: /* left */
			key_table[LEFT] = 1;
			break;
		case 77: /* right */
			key_table[RIGHT] = 1;
			break;
		/* break codes = make code + 128 */
		case 200: /* up release */
			key_table[UP] = 0;
			break;
		case 208: /* down release */
			key_table[DOWN] = 0;
			break;
		case 203: /* left release */
			key_table[LEFT] = 0;
			break;
		case 205: /* right release */
			key_table[RIGHT] = 0;
			break;
		default:
			break;
	}

	outportb(0x20, 0x20);
	asm("popa; sti");
}

void key_init(void) /* function to swap state */
{
	new_key_handler.pm_offset   = (int)key_handler;
	new_key_handler.pm_selector = _go32_my_cs();
	_go32_dpmi_get_protected_mode_interrupt_vector(0x9, &old_key_handler);
	_go32_dpmi_allocate_iret_wrapper(&new_key_handler);
	_go32_dpmi_set_protected_mode_interrupt_vector(0x9,&new_key_handler);
}

void key_delete(void)
{
	_go32_dpmi_set_protected_mode_interrupt_vector(0x9,&old_key_handler);
}

void main(void)
{
	/* install keyboard handler */
	key_init();
	
	/* clear key_table */
	key_table[UP] = key_table[DOWN] = key_table[LEFT] = key_table[RIGHT] = 0;

	/* while not all keys pressed */
	do {
		if (key_table[UP])    printf("UP    "); 
		else printf("      ");
		if (key_table[DOWN])  printf("DOWN  "); 
		else printf("      ");
		if (key_table[LEFT])  printf("LEFT  "); 
		else printf("      ");
		if (key_table[RIGHT]) printf("RIGHT \n"); 
		else printf("      \n");
	} while(!(key_table[UP]&&key_table[DOWN]&&key_table[LEFT]&&key_table[RIGHT]));
	
	/* delete keyboard handler */
	key_delete();
}
