/*
GAME PROGRAMMING TUTORIAL
Copyright (C) 1996 Emmanuel Lagare

SPRITES

Now we want to move a bitmap across a background without disturbing it.
We move into the realm of sprites. Sprites in the simplest terms bitmaps
which don't disturb the background. Of course we could make the sprite
more complicated by adding a lot of other info (location, velocity, timers).
For our purposes we'll just add the most basic info.

Sprites in some other computers are implemented in hardware. In the PC
world, we are not so fortunate. We have to implement them in software.
Hence the complexity of this thing we're about to do. But it's not that
bad actually.

We need a sprite structure, additional functions draw_sprite, erase_sprite,
and a bit blitter that saves a block from the screen.

The sprite structure contains all pertinent information regarding the
sprite:
x, y : the x and y location of the sprite on the screen
x_vel, y_vel : the current x and y velocities of the sprite
width, height : the dimensions of the sprite's bitmaps
num_frames : the number of bitmaps the sprite contains, useful for animation
cur_frame : the current bitmap to draw
background : this is where we save the area to be overdrawn
frame[] : this array are pointers to the bitmaps

draw_sprite computes the new location of the sprite, saves the area to be
overdrawn by the bitmap, then draws the bitmap onto the new location (using
bit_blt or transparent_bit_blt, depending on your needs). erase_sprite
simply restores the saved area (using bit_blt).

We will call the bit blitter that saves a block from the screen into a
bitmap as reverse_bit_blitter. As it's name suggests, we just modify the
bit_blt function by reversing the source and the destination in the memcpy.
We assume that the first two bytes (the width and the height) are already
set beforehand (preferably in the sprite initialization stage).

Actually, we could recode our bitblt to include a source and destination.
but that would mean additional things to put onto the stack on a call which
is slower.

Below is the usual example.
*/

#include <sys/nearptr.h>
#include <go32.h> /* for _dos_ds */
#include <dos.h>

#define GRAPHICS        0x013
#define TEXT            0x03

char *video_buffer = (char *)0xa0000;

typedef struct sprite_type {
	unsigned x, y;                  /* location */
	unsigned x_vel, y_vel;          /* velocity */
	char width, height;             /* dimensions */
	char num_frames;                /* number of frames */
	char cur_frame;                 /* current frame */
	char *background, **frame;
} sprite;

void set_video_mode(int mode)
{
	union REGS regs;

	regs.x.ax = mode;
	int86(0x10, &regs, &regs);
}

void put_pixel(short int x, short int y, char color)
{
	video_buffer[(y << 8) + (y << 6) + x] = color;
}

void bit_blt(int x, int y, char *bitmap)
{
	int yindex, offset = (y << 8) + (y << 6) + x;
	char width = bitmap[0], height = bitmap[1];

	bitmap += 2; /* skip the first two bytes */
	for (yindex = 0; yindex < height; yindex++) {
		memcpy((char *)video_buffer+offset,bitmap,width);
		offset += 320; /* next line of video buffer */
		bitmap += width; /* next line of bitmap */
	}
}

/* new */
void reverse_bit_blt(int x, int y, char *bitmap)
{
	int yindex, offset = (y << 8) + (y << 6) + x;
	char width = bitmap[0], height = bitmap[1];

	bitmap += 2; /* skip the first two bytes */
	for (yindex = 0; yindex < height; yindex++) {
		memcpy(bitmap,(char *)video_buffer+offset,width);
		offset += 320; /* next line of video buffer */
		bitmap += width; /* next line of bitmap */
	}
}

void transparent_bit_blt(int x, int y, char *buffer)
{
	int xindex, yindex, offset = (y << 8) + (y << 6) + x;
	char width = buffer[0], height = buffer[1];

	buffer += 2; /* skip the first two bytes */
	for (yindex = 0; yindex < height; yindex++) {
		for (xindex = 0; xindex < width; xindex++) {
			if (buffer[xindex]) {
				video_buffer[offset+xindex] = buffer[xindex];
			}
		}
		offset += 320; /* next line of video buffer */
		buffer += width; /* next line of bitmap */
	}
}

/* new */
void draw_sprite(sprite *sprite)
{
	sprite->x += sprite->x_vel; sprite->y += sprite->y_vel;
	reverse_bit_blt(sprite->x,sprite->y,sprite->background);
	transparent_bit_blt(sprite->x,sprite->y,sprite->frame[sprite->cur_frame]);
}

/* new */
void erase_sprite(sprite *sprite)
{
	bit_blt(sprite->x,sprite->y,sprite->background);
}

char bitmap[102] = {
	10,10,                          /* width and height of block */
	15,15,15,15,15,15,15,15,15,15,  /* strip 1 */
	15,00,00,00,00,00,00,00,00,15,
	15,00,00,00,00,00,00,00,00,15,
	15,00,00,00,00,00,00,00,00,15,
	15,00,00,00,00,00,00,00,00,15,
	15,00,00,00,00,00,00,00,00,15,
	15,00,00,00,00,00,00,00,00,15,
	15,00,00,00,00,00,00,00,00,15,
	15,00,00,00,00,00,00,00,00,15,
	15,15,15,15,15,15,15,15,15,15   /* strip 10 */
};

void main(void)
{
	unsigned count;
	sprite sprite;

	/* disable all memory protection */
	__djgpp_nearptr_enable();
    video_buffer += __djgpp_conventional_base;

	/* go to graphics mode */
	set_video_mode(GRAPHICS);

	/* initialize our sprite */
	sprite.x = 0; sprite.y = 95;
	sprite.x_vel = 1; sprite.y_vel = 0;
	sprite.width = 10; sprite.height = 10;
	sprite.num_frames = 1;
	sprite.cur_frame = 0;
	sprite.background = (char *)malloc(sprite.width*sprite.height+2);
	sprite.background[0] = 10; sprite.background[1] = 10;
	sprite.frame = (char **)malloc(sizeof(char *)*sprite.num_frames);
	sprite.frame[0] = (char *)&bitmap;

	/* fill up the screen with random pixels */
	for(count = 0; count < 32000; count++) {
		put_pixel(rand() % 320, rand() % 200, rand() % 256);
	}

	/* move sprite from left to right */
	for(count = 0; count < 310; count++) {
		draw_sprite(&sprite);
		delay(10);
		erase_sprite(&sprite);
	}

	/* go back to text mode */
	set_video_mode(TEXT);

	/* reenable memory protection */
	__djgpp_nearptr_disable();
}