/*
GAME PROGRAMMING TUTORIAL
Copyright (C) 1996 Emmanuel Lagare

STEREO PANNING

Sound is cool but stereo sound is even cooler. Using stereo sound can 
provide "sound-sourcing". That is, we can simulate where a sound source is
by simply varying the volumes of the left and right sound channels. The
only problem is that most speakers are mounted in front of the player
resulting in only frontal sound sourcing. For the best effect, headphones
(or side mounted speakers) are necessary.

SB_LIB provides for a way to manipulate the sound channels via the variables
right_volume and left_volume of the sound structure. Their values vary from
0 (mute) to 64 (loudest).

To have a sound source on the left we set the left_volume to 64 and the
right_volume to 0. To have one on the right we set the left_volume to 0 and
the right_volume to 64. To have one on the center we set both left_volume
and right_volume to 32. By having two channels pumping out sound at a volume
of 32, we get an effective volumes of 64.

In the example, we will pan a sound from left to right. The're is only one
minor problem: how to assure that we get to the rightmost panning position
at the exact moment the sample finishes playing. The solution is simple, use
delay() then pan then delay(). So how do we compute the length of the delay?

The sound structure has a variable length which is the number of bytes in
the sound sample. If we divide that by our playback rate (11000 in our case)
we will get the number of seconds to play the sample. To get the number of
milliseconds, we multiply it by 1000. We then divide the result by the number
of sample positions (64 in our case). The final result will be the time we
need to wait before moving to the next panning position.

Normally, we wouldn't need to go through all these computations. In cases 
when the samples are short bursts (like the growl of a monster), we simple
calculate the panning position and play the sample. But in cases when the
samples are extended (like the sound of a TIE fighter passing by) these
computations become essential.

Here is the example. Remember to add sb_lib.a when compiling, i.e. use
"gcc -o sound sound.c sb_lib.a".
*/
#include "sb_lib.h"

void callback(void)
{
        /* we'll retain this for good practice */
}

void main(int argc, char *argv[])
{
        sb_sample *sound;
        int delaylength;

        /* setup driver: make sure sample is in the specified frequency */
        /* if not, convert the sample using sb_convert_frequency() */
        sb_install_driver(11000);

        /* setup WAV file specified in command line */
        sound=sb_load_sample(argv[1],_SB_WAV);
        sound->callback=callback;
        
        /* play sample: you can also use _sb_queue_sample() */
        sb_mix_sample(sound);

        /* pan the sound from left to right */
        delaylength = (sound->length / 11000) * 1000 / 64; /* for clarity */
        sound->right_volume = 0;
        sound->left_volume = 64;
        while(sound->left_volume > 32) {
                delay(delaylength);
                sound->right_volume++;
                sound->left_volume--;
        }
        while(sound->right_volume < 64) {
                delay(delaylength);
                sound->right_volume++;
                sound->left_volume--;
        }
        
        /* clean up */
        sb_free_sample(sound);
        sb_uninstall_driver();
}
