/*

NanoTech - a 3d game engine
Copyright (C) 1996  Sean Lane Fuller

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

Sean Lane Fuller
124 Autumn Lane
Tullahoma, TN 37388
615-393-4550
email: fuller@edge.net

*/


#include <math.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <dos.h>
#include "pix.h"
#include "sbuffer.h"

int was_hit = 0;
extern int do_time;

extern int background;

//#undef MAXX
//#undef MAXY
//#undef MIDX
//#undef MIDY
//#define MAXX 319
//#define MAXY 199
//#define MIDX (MAXX/2)
//#define MIDY (MAXY/2)

#define fixtoint(x) ((int)(x))
#define fixtodbl(x) (x)
#define inttofix(x) ((float)(x))
#define dbltofix(x) (x)

extern int last_id;

#define fixmul(a,b) (a)*(b)
#define fixdiv(a,b) (a)/(b)

// define this for directional lighting (doesn't work)
//#define LIGHTING

#define NOT_IN_VIEW 2
#define ALL_IN_VIEW 1
#define PART_IN_VIEW 0
struct obj_struct *add_obj(int id, unsigned char type);
int del_obj(int id);
typedef unsigned char angle;
#define NANGLES 256
#define maxv 3000
#define maxf 1600
#define maxlist 4000

fixed mx, my, mz;
#define far_clip inttofix(-30000)
fixed yaw, pitch, roll;
fixed sin_yaw, cos_yaw;
fixed cos_roll, sin_roll;
fixed cos_pitch, sin_pitch;

char area_type;
char mod_name[80];

#define horiz CLIPMAXX
#define vert CLIPMAXY

#define maxpoly 20
int num_visible;
char wire_frame = 3;
fixed epsilon = dbltofix(.0001);
struct vector3 { fixed x, y, z; };
fixed MID_X; // = inttofix(MAXX/2) + inttofix(1);
fixed MID_Y; // = inttofix(MAXY/2) + inttofix(1);
int firstfree=0, nov=0, ntv=0, nof=0, ntf=0; /* counts */
struct corner { int i; fixed u, v; };
struct vector3 direct, eye, zero = { 0, 0, 0 };
fixed ppd;
short inside[maxpoly];
int ksize;
struct corner kfacet[2][maxpoly];
struct vector3 precomp_norm[5];
int horiz_plane[5];
fixed precomp_stth[5];
fixed precomp_nabstth[5];

#define FLOOR 0
#define PYRAMID 1
#define CUBE 2
#define SPHERE 3
#define CYLINDER 4
#define RECTANGLE 5
#define TRIANGLE 6
#define LETTER_N 7
#define LETTER_A 8
#define LETTER_O 9
#define LETTER_T 10
#define LETTER_E 11
#define LETTER_C 12
#define LETTER_H 13
#define TORUS 14
#define FACE 15
#define TREE 16
#define CHECKER 17
#define WORLD 18
#define BIT_CRITTER 19
#define GEM 20
char *shapes[] = { "floor.plg", "pyramid.plg", "cube.plg",
   "sphere.plg", "cylinder.plg", "rectangl.plg", "triangle.plg",
   "n.plg", "a.plg", "o.plg", "t.plg", "e.plg", "c.plg", "h.plg",
   "mtorus.plg", "face.plg", "tree.plg", "checker.plg", "world.plg",
   "bit.plg", "gem.plg",
   0 };

#define MAXVAR 3
#define max_objs 200
struct obj_struct
{
   int id;
   unsigned char type;
   unsigned char clipped;
   unsigned char off;
   long state;
   int var[MAXVAR];
   int first_fac;
   int nof;
   int first_vert;
   int nov;
   int first_list;
   int nol;
   angle yaw;
   angle roll;
   angle pitch;
   fixed x, y, z;
   struct vector3 *act;
   unsigned char marked;
   fixed radius;
   fixed midx, midy, midz;  // transformed centroid
   fixed obsx, obsy, obsz;  // observed position of centroid
   fixed actx, acty, actz;  // untransformed centroid
};
#define NULLOBJ ((struct obj_struct*)0)
int num_objs = 0;
int num_sorted = 0;
struct obj_struct *obj[max_objs];

float get_obj_radius(int i) { return obj[i]->radius; }
float get_obj_midy  (int i) { return obj[i]->midy; }
int get_obj_id       (int i) { return obj[i]->id; }
unsigned char get_obj_type (int i) { return obj[i]->type; }

void strip_comments(char *buff)
{
   char *p;
   if ((p = strchr(buff, '\n')) != NULL) *p = '\0';  /* trim newline */
   if ((p = strchr(buff, '#')) != NULL) *p = '\0';   /* trim comment */
   if ((p = strchr(buff, '*')) != NULL) *p = '\0';   /* future expansion */
}

#ifdef LIGHTING
long long_sqrt(long v)
{
   int i;
   unsigned long result, tmp, low, high;
   if (v <= 1L) return 0L;
   low = v;
   high = 0L;
   result = 0;
   for (i = 0; i < 16; i++)
   {
      result += result;
      high = (high << 2) | ((low >>30) & 0x3);
      low <<= 2;
      tmp = result + result + 1;
      if (high >= tmp)
      {
         result++;
         high -= tmp;
      }
   }
   if (v - (result * result) >= (result - 1)) result++;
   return result;
}
#endif

int draw_shadows(char *b, int w, int h)
{
   int x, z, i, p, j, k, q;
   for (i=1; i<num_objs; i++)
   {
      x = (((float)(obj[i]->midx + 1000.0)/2000.0)) * (float)h;
      z = (1.0 - ((float)(obj[i]->midz + 1000.0)/2000.0)) * (float)w;
      p = z * w + x;
      for (j=-3; j<4; j++)
      {
         for (k=-3; k<4; k++)
         {
            q = p + k + j * w;
            if (q >= 0 && q <= w * h && b[q] > 1) b[q] -= 2;
         }
      }
   }
}

int collision(fixed midx, fixed midy, fixed midz, fixed delta, int id)
{
   int i;
   fixed r;
   for (i=1; i<num_objs; i++)
   {
      if (obj[i]->off) continue;
      if (obj[i]->id == id) continue;
      if (obj[i]->type == RECTANGLE) continue;
      r = obj[i]->radius;
      if (midx + delta < obj[i]->midx - r
         || midx - delta > obj[i]->midx + r
         || midy < obj[i]->midy - r
         || midy - delta > obj[i]->midy + r
         || midz + delta < obj[i]->midz - r
         || midz - delta > obj[i]->midz + r) continue;
      return i;
   }
   if (id >= 0)
   {
      for (i=1; i<num_objs; i++)
      {
         if (obj[i]->id == id)
         {
            if (-mx + delta < obj[i]->midx - r
               || -mx - delta > obj[i]->midx + r
               || -my < obj[i]->midy - r
            || -my - delta > obj[i]->midy + r
            || -mz + delta < obj[i]->midz - r
            || -mz - delta > obj[i]->midz + r)
               break;
            else
               return -999;
         }
      }
   }
   return -1;
}

struct vector3 act[maxv];
struct vector3 obs[maxv];
struct vector3 pro[maxv];
unsigned char projected[maxv];
struct obj_struct *sorted[max_objs];
int color[maxf];
int facfront[maxf];
int size[maxf];
int nfac[maxf];
fixed psin[NANGLES];
fixed pcos[NANGLES];
struct corner faclist[maxlist];
#ifdef LIGHTING
   vector3 normals[maxf];
   vector3 normals2[maxf];
#endif

void init_3d()
{
   int i;
   float a, s;
   for (i=0; i<max_objs; i++) obj[i] = NULLOBJ;
   s = (float)(3.14159267 * 2)/NANGLES;
   for (a=0; a<NANGLES; a++)
   {
      psin[(int)a] = sin(a * s);
      pcos[(int)a] = cos(a * s);
   }
   num_objs = 0;
   MID_X = MAXX/2 + 1;
   MID_Y = MAXY/2 + 1;
}

void term_3d()
{
}

#define sign(f) (((f) < 0) ? -1 : 1)

/*****************************************************/
/* Compute dot product of 2 3d vectors               */
/*****************************************************/
#define dot3(p1, p2) (p1).x * (p2).x + (p1).y * (p2).y + (p1).z * (p2).z

#ifdef LIGHTING
void  vectorproduct(struct vector3 p, struct vector3 q, struct vector3 *v)
{
   v->x = fixmul(p.y, q.z) - fixmul(p.z, q.y);
   v->y = fixmul(p.z, q.x) - fixmul(p.x, q.z);
   v->z = fixmul(p.x, q.y) - fixmul(p.y, q.x);
}
#endif

/*****************************************************/
/* Clip a facet                                      */
/*****************************************************/
static int clip(int k)
{
   /* clips facet k */
   int i, j, h, n, f, s, insect, inter, kfi, l1, l2, nnv, np[maxpoly];
   fixed rval, stth, nabstth, a, b, dotprod1, dotprod2, mustore;
   int facfrontk;
   struct vector3 base, dir, ipt, norm;
   char texture_mapped;
   int vf, vs, c;
   struct vector3 *precomp_norm_ptr = precomp_norm + 1;
   /* nnv is total number of vertices prior to clipping facet k */
   nnv = ntv;
   /* copy pointers to facet vertices into first section of kfacet */
   ksize = size[k];
   l1 = 0;
   facfrontk = facfront[k];
   texture_mapped = (color[k] >> 8) && (wire_frame == 0 || wire_frame == 1);
   texture_mapped = 1;
   for (i=0; i<ksize; i++)
   {
      kfacet[l1][i].i = faclist[facfrontk].i;
      kfacet[l1][i].u = faclist[facfrontk].u;
      kfacet[l1][i].v = faclist[facfrontk].v;
      facfrontk++;
   }
   /* loop through clipping planes 1 through 4 */
   for (i=1; i<5; i++)
   {
      n = 0;
      /* find normal norm vector of clipping plane and in value of verteces */
      norm = *precomp_norm_ptr;
      precomp_norm_ptr++;
      c = 0;
      stth = precomp_stth[i];
      nabstth = precomp_nabstth[i];
      if (horiz_plane[i])
      {
         for (h=0; h<ksize; h++)
         {
            j = kfacet[l1][h].i;
            a = fixmul(obs[j].y, stth);
            b = fixmul(nabstth, obs[j].z);
            if (a < b) { inside[h] = 1; c++; }
            else if (b < a) inside[h] = -1;
            else inside[h] = 0;
         }
         if (c == 0) return 2;
      }
      /* Vertical clipping plane */
      else
      {
         for (h=0; h<ksize; h++)
         {
            j = kfacet[l1][h].i;
            a = fixmul(obs[j].x, stth);
            b = fixmul(nabstth, obs[j].z);
            if (a < b) { inside[h] = 1; c++; }
            else if (b < a) inside[h] = -1;
            else inside[h] = 0;
         }
         if (c == 0) return 2;
      }
      l2 = 1 - l1; f = ksize - 1;
      /* slice the facet defined by kfacet array with clipping plane i */
      /* consider facet edge joining vertices f(first) and s(second) */
      for (j=0; j<ksize; j++)
      {
         s=j;
         /* If vertex f is inside then include in new facet */
         if (inside[f] > 0) /* was >= 0 */
         {
            kfacet[l2][n].i = kfacet[l1][f].i;
            kfacet[l2][n].u = kfacet[l1][f].u;
            kfacet[l2][n].v = kfacet[l1][f].v;
            n++;
         }
         /* If vertices f and s are on opposite sides of the plane then */
         /* find the intersection of the edge with  the plane and include */
         if ((inside[f] < 0 && inside[s] > 0)
            || (inside[f] > 0 && inside[s] < 0))
         {
            vf = kfacet[l1][f].i;
            vs = kfacet[l1][s].i;
            base.x = obs[vf].x;
            base.y = obs[vf].y;
            base.z = obs[vf].z;
            dir.x = obs[vs].x - base.x;
            dir.y = obs[vs].y - base.y;   
            dir.z = obs[vs].z - base.z;
            //*** find intersection of plane and line
            dotprod1 = dot3(dir, norm);
            dotprod2 = dot3(base, norm);
            mustore = -dotprod2 / dotprod1;
            ipt.x = base.x + mustore * dir.x;
            ipt.y = base.y + mustore * dir.y;
            ipt.z = base.z + mustore * dir.z;
            obs[ntv].x = ipt.x;
            obs[ntv].y = ipt.y;
            obs[ntv].z = ipt.z;
            kfacet[l2][n].i = ntv;
            if (texture_mapped)
            {
               kfacet[l2][n].u = kfacet[l1][f].u +
                  mustore * (kfacet[l1][s].u - kfacet[l1][f].u);
               kfacet[l2][n].v = kfacet[l1][f].v +
                  mustore * (kfacet[l1][s].v - kfacet[l1][f].v);
            }
            n++;
            ntv++;
         }
         f = s;
      }
      /* if new facet empty the stop */
      if (n <= 2)
      {
         ntv = nnv;
         return 2;
      }
      ksize = n;
      l1 = l2;
   }
   /* Reach here if non-empty facet remains.  If new vertices have been */
   /* created then sort them and store new facet */
   if (ntv > nnv)
   {
      for (i=0; i<ksize; i++)
         np[i] = i;
      n = nnv;
      facfront[ntf] = firstfree;
      size[ntf] = ksize;
      /* Storage of vertices with garbage collection */
      /* Sort contents of kfacet array into increasing order */
      for (i=0; i<ksize; i++)
      {
         for (j=i+1; j<ksize; j++)
         {
            if (kfacet[l1][np[i]].i > kfacet[l1][np[j]].i)
            {
               inter = np[i]; np[i] = np[j];
               np[j] = inter;
            }
         }
         kfi = kfacet[l1][np[i]].i;
         /* If vertex new (i.e. kfi>=nnv) then place in next available */
         /* location else refer to old location */
         if (kfi >= nnv)
         {
            faclist[facfront[ntf] + np[i]].i = n;
            faclist[facfront[ntf] + np[i]].u = kfacet[l1][np[i]].u;
            faclist[facfront[ntf] + np[i]].v = kfacet[l1][np[i]].v;
            obs[n] = obs[kfi];
            n++;
         }
         else
         {
            faclist[facfront[ntf] + np[i]].i = kfi;
            faclist[facfront[ntf] + np[i]].u = kfacet[l1][np[i]].u;
            faclist[facfront[ntf] + np[i]].v = kfacet[l1][np[i]].v;
         }
      }
      ntv = n;
      firstfree += size[ntf];
      ntf++;
      return 3;
   }
   return 1;
}
/******************************************************************************
 * is object outside of clipping cone
 *****************************************************************************/
static int outside_cone(fixed obsx, fixed obsy, fixed obsz, fixed radius)
{
   fixed b;
   b = fixmul(precomp_nabstth[3], obsz);
   if (fixmul(obsy + radius, precomp_stth[3]) > b) return 1;
   b = fixmul(precomp_nabstth[4], obsz);
   if (fixmul(obsx + radius, precomp_stth[4]) > b) return 1;
   b = fixmul(precomp_nabstth[1], obsz);
   if (fixmul(obsy - radius, precomp_stth[1]) > b) return 1;
   b = fixmul(precomp_nabstth[2], obsz);
   if (fixmul(obsx - radius, precomp_stth[2]) > b) return 1;
   return 0;
}

/******************************************************************************
 * is object inside clipping cone
 *****************************************************************************/
static int inside_cone(fixed obsx, fixed obsy, fixed obsz, fixed radius)
{
   fixed b;
   b = fixmul(precomp_nabstth[3], obsz);
   if (fixmul(obsy - radius, precomp_stth[3]) > b) return 0;
   b = fixmul(precomp_nabstth[4], obsz);
   if (fixmul(obsx - radius, precomp_stth[4]) > b) return 0;
   b = fixmul(precomp_nabstth[1], obsz);
   if (fixmul(obsy + radius, precomp_stth[1]) > b) return 0;
   b = fixmul(precomp_nabstth[2], obsz);
   if (fixmul(obsx + radius, precomp_stth[2]) > b) return 0;
   return 1;
}

/******************************************************************************
 * Clip scene by object
 *****************************************************************************/
void clip_scene_obj()
{
   int i, j, clipindex, last_fac;
   firstfree = facfront[nof-1] + size[nof - 1];
   for (i=0; i<num_objs; i++)
   {
      struct obj_struct *o = obj[i];
      last_fac = o->first_fac + o->nof;
      if (o->obsz - o->radius > 0) // quickly test for behind observer
      {
         for (j=o->first_fac; j<last_fac; j++) nfac[j] = -1;
         obj[i]->clipped = NOT_IN_VIEW;
      }
      else if (o->off)
      {
         for (j=o->first_fac; j<last_fac; j++) nfac[j] = -1;
         obj[i]->clipped = NOT_IN_VIEW;
      }
      else if (o->obsz + o->radius < far_clip)
      {
         for (j=o->first_fac; j<last_fac; j++) nfac[j] = -1;
         obj[i]->clipped = NOT_IN_VIEW;
      }
      else if (outside_cone(o->obsx, o->obsy, o->obsz, o->radius))
      {
         for (j=o->first_fac; j<last_fac; j++) nfac[j] = -1;
         obj[i]->clipped = NOT_IN_VIEW;
      }
      else if (inside_cone(o->obsx, o->obsy, o->obsz, o->radius))
      {
         for (j=o->first_fac; j<last_fac; j++) nfac[j] = j;
         obj[i]->clipped = ALL_IN_VIEW;
      }
      else obj[i]->clipped = PART_IN_VIEW;
   }
}

void clip_scene_vert()
{
   int i, j, clipindex, last_fac;
   for (i=0; i<num_objs; i++)
   {
      struct obj_struct *o = obj[i];
      if (o->clipped == PART_IN_VIEW)
      {
         last_fac = o->first_fac + o->nof;
         for (j=o->first_fac; j<last_fac; j++)
         {
               clipindex = clip(j);
               if (clipindex == 1) nfac[j] = j;       /* not modified */
               else if (clipindex == 2) nfac[j] = -1; /* degenerate */
               else nfac[j] = ntf - 1;                /* modified */
         }
      }
   }
}

void clear_scene()
{
   int i;
   fixed tth, stth, nabstth;
   while (num_objs > 0)
   {
      del_obj(obj[0]->id);
   }
   ppd = MAXX; //256.0;
   firstfree = 0;
   num_objs = 0;
   nov = 0;
   nof = 0;
   ntv = 0;
   ntf = 0;
   precomp_frustrum();
}

void precomp_frustrum()
{
   int i;
   fixed tth, stth, nabstth;
   ppd = (CLIPMAXX + CLIPMAXY)/2; //256.0;
   for (i=1; i<5; i++)
   {
      precomp_norm[i].x = inttofix(-((i-1) % 2));
      precomp_norm[i].y = inttofix(-(i % 2));
      if (precomp_norm[i].x < epsilon && precomp_norm[i].x > -epsilon)
      {
         horiz_plane[i] = 1;
         precomp_norm[i].z = (float)vert * 0.5 * ((float)(i - 2)/ppd);
      }
      else
      {
         horiz_plane[i] = 0;
         precomp_norm[i].z = (float)horiz * 0.5 * ((float)(i-3)/ppd);
      }
      // find normal norm vector of clipping plane and in vaue of verteces
      tth = -precomp_norm[i].z;
      if (tth < 0) { stth = -1; nabstth = tth; }
      else if (tth == 0) { stth = 0; nabstth = 0; }
      else { stth = 1; nabstth = -tth; }
      precomp_stth[i] = stth;
      precomp_nabstth[i] = nabstth;
   }
}

void  draw_scene()
{
   polygon2D_t screenpoly;
   int j, i, k, v1, v2, l, c, sizej, z, f_fac, l_fac, obj_i;
   int mmi;
   fixed miny, maxy;
   char buf[132];
   int xx[maxpoly], yy[maxpoly];
   int color_adj;
   fixed zz[maxpoly];
   int pid;
   fixed nz, nx, ny;
   fixed xxx[3], yyy[3], zzz[3];
   fixed u[maxpoly], v[maxpoly];
   ClearEdgeLists();
      for (obj_i=0; obj_i<num_sorted; obj_i++)
      {
         f_fac = sorted[obj_i]->first_fac;
         l_fac = f_fac + sorted[obj_i]->nof;
      for (i=f_fac; i<l_fac; i++) {
         j = nfac[i];
         if (j == -1) continue;
         sizej = size[j];
         l = facfront[j];
         if (sorted[obj_i]->id == 0) z = -10000;
         else
            z = fixtoint(pro[faclist[l].i].z);
         for (k=0; k<3; k++, l++)
         {
            v2 = faclist[l].i;
            xxx[k] = pro[v2].x;
            yyy[k] = pro[v2].y;
            zzz[k] = pro[v2].z;
            u[k] = ((float)faclist[l].u);
            v[k] = ((float)faclist[l].v);
         }
         if ((xxx[1] - xxx[0]) * (yyy[2] - yyy[1]) -
            (yyy[1] - yyy[0]) * (xxx[2] - xxx[1]) >= 0)
         {
            for (k=0; k<3; k++)
            {
               xx[k] = fixtoint(xxx[k]);
               yy[k] = fixtoint(yyy[k]);
               zz[k] = zzz[k];
            }
            for (; k<sizej; k++, l++)
            {
               v2 = faclist[l].i;
               xx[k] = fixtoint(pro[v2].x);
               yy[k] = fixtoint(pro[v2].y);
               zz[k] = pro[v2].z;
               u[k] = ((float)faclist[l].u);
               v[k] = ((float)faclist[l].v);
            }
   screenpoly.num_verts = size[j];
   l = facfront[j];
   for (k=0; k<sizej; k++, l++)
   {
      v2 = faclist[l].i;
      screenpoly.verts[k].x = pro[v2].x;
      screenpoly.verts[k].y = pro[v2].y;
      screenpoly.verts[k].z = -pro[v2].z;
      screenpoly.verts[k].u = faclist[l].u;
      screenpoly.verts[k].v = faclist[l].v;
   }
   if (color[j] & 0xff0000) AddPolygonEdges(&screenpoly, (color[j] & 0xff) | 0x400);
   else AddPolygonEdges(&screenpoly, color[j]);
        }
      }
   }
   ScanEdges();
   DrawSpans();
}

int comp_obj(const void *a, const void *b)
{
   struct obj_struct *o1, *o2;
   o1 = *((struct obj_struct **)a);
   o2 = *((struct obj_struct **)b);
   if (o1->obsz == o2->obsz) return 0;
   if (o1->obsz < o2->obsz) return -1;
   return 1;
}

void sort_obj()
{
   int i, j;
   num_sorted = 0;
   for (i=0; i<nof; i++) {
      j = nfac[i];
      if (j != -1)
      {
         color[j] = color[i];
#ifdef LIGHTING
         normals2[j] = normals[i];
#endif
      }
   }
   for (i=0; i<num_objs; i++)
   {
      if (obj[i]->clipped == PART_IN_VIEW || obj[i]->clipped == ALL_IN_VIEW)
      {
         sorted[num_sorted++] = obj[i];
      }
   }
}

void obs_scene_obj()
{
   int i;
   fixed xx, yy, zz, xa, ya, za, x, y, z, temp;
   for (i=0; i<num_objs; i++)
   {
      struct obj_struct *o = obj[i];
      /* no yaw, pitch, or roll */
      if (sin_yaw == 0 && sin_pitch == 0 && sin_roll == 0)
      {
         o->obsx = o->midx + mx;
         o->obsy = o->midy + my;
         o->obsz = o->midz + mz;
      }
      /* for just yaw */
      else if (sin_roll == 0 && sin_pitch == 0)
      {
         x = o->midx + mx;
         o->obsy = o->midy + my;
         z = o->midz + mz;
         o->obsx = fixmul(cos_yaw, x) - fixmul(sin_yaw, z);
         o->obsz = fixmul(sin_yaw, x) + fixmul(cos_yaw, z);
      }
      /* for yaw and pitch only */
      else if (sin_roll == 0)
      {
         x = o->midx + mx; y = o->midy + my; z = o->midz + mz;
         o->obsx = fixmul(cos_yaw, x) - fixmul(sin_yaw, z);
         za = fixmul(sin_yaw, x) + fixmul(cos_yaw, z);
         o->obsz = fixmul(cos_pitch, za) - fixmul(sin_pitch, y);
         o->obsy = fixmul(sin_pitch, za) + fixmul(cos_pitch, y);
      }
      /* for yaw pitch and roll */
      else
      {
         x = o->midx + mx; y = o->midy + my; z = o->midz + mz;
         xa = fixmul(cos_yaw, x) - fixmul(sin_yaw, z);
         za = fixmul(sin_yaw, x) + fixmul(cos_yaw, z);
         o->obsz = fixmul(cos_pitch, za) - fixmul(sin_pitch, y);
         ya = fixmul(sin_pitch, za) + fixmul(cos_pitch, y);
         o->obsx = fixmul(cos_roll, xa) + fixmul(sin_roll, ya);
         o->obsy = fixmul(cos_roll, ya) - fixmul(sin_roll, xa);
      }
   }
}

void obs_scene_vert()
{
   int i, f, l, j;
   fixed xx, yy, zz, xa, ya, za, x, y, z, temp;
   for (j=0; j<num_objs; j++)
   {
      if (obj[j]->clipped == NOT_IN_VIEW) continue;  /* no part in view */
      f = obj[j]->first_vert;
      l = obj[j]->nov + f;
      for (i=f; i<=l; i++)
      {
         if (sin_yaw == 0 && sin_pitch == 0 && sin_roll == 0)
         {
            obs[i].x = act[i].x + mx;
            obs[i].y = act[i].y + my;
            obs[i].z = act[i].z + mz;
         }
         /* to do yaw only */
         else if (sin_pitch == 0 && sin_roll == 0)
         {
            x = act[i].x + mx; y = act[i].y + my; z = act[i].z + mz;
            obs[i].x = fixmul(cos_yaw, x) - fixmul(sin_yaw, z);
            obs[i].z = fixmul(sin_yaw, x) + fixmul(cos_yaw, z);
            obs[i].y = y;
         }
         /* do yaw and pitch */
         else if (sin_roll == 0)
         {
            x = act[i].x + mx; y = act[i].y + my; z = act[i].z + mz;
            obs[i].x = fixmul(cos_yaw, x) - fixmul(sin_yaw, z);
            za = fixmul(sin_yaw, x) + fixmul(cos_yaw, z);
            obs[i].z = fixmul(cos_pitch, za) - fixmul(sin_pitch, y);
            obs[i].y = fixmul(sin_pitch, za) + fixmul(cos_pitch, y);
         }
         /* to do roll, pitch, and yaw */
         else
         {
            x = act[i].x + mx; y = act[i].y + my; z = act[i].z + mz;
            xa = fixmul(cos_yaw, x) - fixmul(sin_yaw, z);
            za = fixmul(sin_yaw, x) + fixmul(cos_yaw, z);
            obs[i].z = fixmul(cos_pitch, za) - fixmul(sin_pitch, y);
            ya = fixmul(sin_pitch, za) + fixmul(cos_pitch, y);
            obs[i].x = fixmul(cos_roll, xa) + fixmul(sin_roll, ya);
            obs[i].y = fixmul(cos_roll, ya) - fixmul(sin_roll, xa);
         }
      }
   }
}

void pro_obj()
{
   int i, k, f_fac, l_fac, obj_n, fac_n, fac_i, front;
   fixed scale, temp;
   for (i=0; i<ntv; i++) projected[i] = 0;
   // for each object in view
   for (obj_n=0; obj_n<num_sorted; obj_n++)
   {
      // for each facet in object
      f_fac = sorted[obj_n]->first_fac;
      l_fac = sorted[obj_n]->nof + f_fac;
      for (fac_n=f_fac; fac_n<l_fac; fac_n++)
      {
         fac_i = nfac[fac_n];
         if (fac_i == -1) continue;
         front = facfront[fac_i];
         for (k=0; k<size[fac_i]; k++, front++)
         {
            i = faclist[front].i;
            if (projected[i] == 0)
            {
               if (obs[i].z > inttofix(1) || obs[i].z < inttofix(-1))
                  scale = fixdiv(ppd, obs[i].z);
               else scale = inttofix(1);
               pro[i].x = fixmul(-obs[i].x, scale) + MID_X;
               pro[i].y = fixmul(obs[i].y, scale) + MID_Y;
               pro[i].z = obs[i].z;
               projected[i] = 1;
            }
         }
      }
   }
}

void  make_scene()
{
   ntf = nof;
   ntv = nov;
   obs_scene_obj();
   clip_scene_obj();
   obs_scene_vert();
   clip_scene_vert();
   sort_obj();
   pro_obj();
   firstfree = facfront[nof-1] + size[nof-1];
}

/******************************************************************************
 * find object pointer given an id
 *****************************************************************************/
struct obj_struct *find_obj(int id)
{
   register int i;
   for (i=0; i<num_objs; i++)
   {
      if (obj[i]->id == id) return obj[i];
   }
   return NULLOBJ;
}

#ifdef LIGHTING
void normalize3(struct vector3 *v)
{
   fixed d = long_sqrt((fixmul(v->x,v->x)+fixmul(v->y,v->y)+fixmul(v->z,v->z))) << 16;
   if (d != 0) {
      v->x = fixdiv(v->x, d);
      v->y = fixdiv(v->y, d);
      v->z = fixdiv(v->z, d);
   }
   else
   {
      v->x = v->y = v->z = 0;
   }
}

/******************************************************************************
 * compute normals for object facets
 *****************************************************************************/
void compute_normals(struct obj_struct *o)
{
   struct vector3 p1, p2, p3, d1, d2;
   int fac_no, vert_no, vert_idx, fac_size, normal_no, last_fac;
   last_fac = o->first_fac + o->nof;
   for (normal_no=0, fac_no=o->first_fac;
         fac_no<last_fac; fac_no++, normal_no++)
   {
      vert_idx = facfront[fac_no];
      vert_no = faclist[vert_idx];
      fac_size = size[fac_no];
      if (fac_size < 3) continue;
      vert_no = faclist[vert_idx];
      p1.x = act[vert_no].x; p1.y = act[vert_no].y; p1.z = act[vert_no].z;
      vert_idx++;
      vert_no = faclist[vert_idx];
      p2.x = act[vert_no].x; p2.y = act[vert_no].y; p2.z = act[vert_no].z;
      vert_idx++;
      vert_no = faclist[vert_idx];
      p3.x = act[vert_no].x; p3.y = act[vert_no].y; p3.z = act[vert_no].z;
      d1.x = p2.x - p1.x;
      d1.y = p2.y - p1.y;
      d1.z = p2.z - p1.z;
      d2.x = p3.x - p2.x;
      d2.y = p3.y - p2.y;
      d2.z = p3.z - p2.z;
      vectorproduct(d1, d2, &(normals[fac_no]));
      normalize3(&(normals[fac_no]));
//      normals[fac_no].z = fac_no;
   }
}
#endif

/******************************************************************************
 * compute object mid point and radius and facet normals
 *****************************************************************************/
void compute_obj(struct obj_struct *o)
{
   int i;
   float lx, ly, lz, hx, hy, hz;
   float dx, dy, dz;
   lx = o->act[0].x;
   ly = o->act[0].y;
   lz = o->act[0].z;
   hx = lx;
   hy = ly;
   hz = lz;
   for (i=1; i<o->nov; i++)
   {
      if (o->act[i].x > hx) hx = o->act[i].x;
      else if (o->act[i].x < lx) lx = o->act[i].x;
      if (o->act[i].y > hy) hy = o->act[i].y;
      else if (o->act[i].y < ly) ly = o->act[i].y;
      if (o->act[i].z > hz) hz = o->act[i].z;
      else if (o->act[i].z < lz) lz = o->act[i].z;
   }
   o->actx = (lx + hx)/2.0;
   o->acty = (ly + hy)/2.0;
   o->actz = (lz + hz)/2.0;
   o->midx = o->actx;
   o->midy = o->acty;
   o->midz = o->actz;
   dx = o->actx - lx;
   dy = o->acty - ly;
   dz = o->actz - lz;
   o->radius = sqrt(dx * dx + dy * dy + dz * dz);
}

extern void strip_comments(char *buf);
struct obj_struct *add_obj(int id, unsigned char type)
{
   struct obj_struct *o = find_obj(id);
   int j;
   long c;
   char *s;
   char name[32];
   char line[129];
   FILE *fp;
   int cnof;
   int cnov;
   float x, y, z;
   int i;
   if (o != NULLOBJ) return 0;
   o = (struct obj_struct *)malloc(sizeof(struct obj_struct));
   if (o == NULLOBJ) return 0;
   obj[num_objs] = o;
   num_objs++;
   o->id = id;
   o->state = 0;
   o->type = type;
   o->off = 0;
   o->nof = 0;
   o->nov = 0;
   o->nol = 0;
   o->first_vert = nov;
   o->first_fac = nof;
   o->first_list = firstfree;
   o->marked = 0;
   o->x = 0;
   o->y = 0;
   o->z = 0;
   o->yaw = 0;
   o->pitch = 0;
   o->roll = 0;
   fp = fopen(shapes[type], "r");
   if (fp == NULL) return 0;
   do {
      if (fgets(line, sizeof(line), fp) == NULL) { fclose(fp); return 0; }
      strip_comments(line);
   } while (line[0] == '\0');
   if (sscanf(line, "%s %d %d", name, &cnov, &cnof) != 3) return 0;
   if (cnov == 0 || cnof == 0) return o;
   if (cnof + nof >= maxf) return 0;
   if (cnov + nov >= maxv) return 0;
   o->act = (struct vector3 *)malloc(sizeof(struct vector3) * cnov);
   if (o->act == (struct vector3*)0) return 0;
   for (i=0; i<MAXVAR; i++) o->var[i] = 0;
   for (i=0; i<cnov; i++)
   {
      do {
         if (fgets(line, sizeof(line), fp) == NULL) { fclose(fp); return 0; }
         strip_comments(line);
      } while (line[0] == '\0');
      sscanf(line, "%f %f %f", &x, &y, &z);
      o->act[o->nov].x = dbltofix(x);
      o->act[o->nov].y = dbltofix(y);
      o->act[o->nov].z = dbltofix(z);
      act[nov + i] = o->act[o->nov];
      o->nov++;
   }
   for (i=0; i<cnof; i++)
   {
      do {
         if (fgets(line, sizeof(line), fp) == NULL) { fclose(fp); return 0; }
         strip_comments(line);
      } while (line[0] == '\0');
      c = strtoul(line, &s, 0);
#ifdef LIGHTING
      color[nof + i] = c | 0x0f;
#else
      color[nof + i] = c;
#endif
      if ((s = strtok(s, " \t")) == NULL) { fclose(fp); return 0; }
      size[nof + i] = atoi(s);
      facfront[nof + i] = firstfree;
      o->nof++;
      for (j=0; j<size[nof + i]; j++)
      {
         if ((s = strtok(NULL, " \t")) == NULL) { fclose(fp); return 0; }
         faclist[firstfree].i = atoi(s) + nov;
         switch (j)
         {
            case 3:
               faclist[firstfree].u = 1;
               faclist[firstfree].v = 1;
               break;
            case 2:
               faclist[firstfree].u = (TMAPH-1)/4;
               faclist[firstfree].v = 1;
               break;
            case 1:
               faclist[firstfree].u = (TMAPH-1)/4;
               faclist[firstfree].v = (TMAPW-1)/4;
               break;
            case 0:
               faclist[firstfree].u = 1;
               faclist[firstfree].v = (TMAPW-1)/4;
               break;
            default:
               faclist[firstfree].u = 1;
               faclist[firstfree].v = 1;
               break;
         }
         firstfree++;
         o->nol++;
      }
   }
   fclose(fp);
   nof += cnof;
   nov += cnov;
   compute_obj(o);
   return o;
}

int upd_obj(int id, angle yaw, angle pitch, angle roll,
            fixed tx, fixed ty, fixed tz);
int load_area_by_name(char *filename)
{
   FILE *fp;
   int i;
   char line[80];
   struct obj_struct *o;
   fixed x, y, z;
   int type;
   int yaw, pitch, roll, v, j;
   int num_objs;
   fp = fopen(filename, "r");
   if (fp == NULL) return 1;
   fgets(line, sizeof(line) - 1, fp);
   area_type = line[0];
   fgets(mod_name, 255, fp);
   fscanf(fp, "%d %d\n", &background, &num_objs);
   for (i=0; i<num_objs; i++)
   {
      fscanf(fp, "%d", &type);
      o = add_obj(i, type);
      fscanf(fp, "%f %f %f ", &x, &y, &z);
      fscanf(fp, "%d %d %d% ", &yaw, &pitch, &roll);
      for (j=0; j<MAXVAR; j++) fscanf(fp, "%d", &(o->var[j]));
      if (o->type == PYRAMID) o->off = o->var[0];
      fscanf(fp, "%*c");
      upd_obj(i, yaw, pitch, roll, x, y, z);
   }
   last_id = i;
   fclose(fp);
   
   return 0;
}

int load_area(int l)
{
   char name[8];
   sprintf(name, "%03d.wld", l);
   return load_area_by_name(name);
}

#ifndef PRODUCTION
int save_area_by_name(char *filename)
{
   int j;
   int i;
   FILE *fp;
   fp = fopen(filename, "w");
   if (fp == NULL) return 1;
   fprintf(fp, "%c\n", area_type);
   fprintf(fp, "%s", mod_name);
   fprintf(fp, "%d %d\n", background, num_objs);
   for (i=0; i<num_objs; i++)
   {
      fprintf(fp, "%d ", (int)obj[i]->type);
      fprintf(fp, "%f %f %f ", obj[i]->x, obj[i]->y, obj[i]->z);
      fprintf(fp, "%d %d %d ", (int)obj[i]->yaw,
            (int)obj[i]->pitch, (int)obj[i]->roll);
      for (j=0; j<MAXVAR; j++)
      {
         fprintf(fp, "%d", obj[i]->var[j]);
         if (j == MAXVAR - 1) fprintf(fp, "\n");
            else fprintf(fp, " ");
      }
   }
   fclose(fp);
   return 0;
}

int save_area(int l)
{
   char name[8];
   sprintf(name, "%03d.wld", l);
   return save_area_by_name(name);
}
#endif

int upd_obj(int id, angle yaw, angle pitch, angle roll,
            fixed tx, fixed ty, fixed tz)
{
   fixed xa, ya, za, x, y, z;
   struct obj_struct *o = find_obj(id);
   int v;
   int i;
   fixed sin_yaw, cos_yaw, sin_pitch, cos_pitch, sin_roll, cos_roll;
   if (o == NULLOBJ) return 0;
   v = o->first_vert;
   sin_yaw = psin[yaw];
   cos_yaw = pcos[yaw];
   sin_pitch = psin[pitch];
   cos_pitch = pcos[pitch];
   sin_roll = psin[roll];
   cos_roll = pcos[roll];
   x = o->actx; y = o->acty; z = o->actz;
   xa = fixmul(cos_yaw, x) - fixmul(sin_yaw, z);
   za = fixmul(sin_yaw, x) + fixmul(cos_yaw, z);
   z = fixmul(cos_pitch, za) - fixmul(sin_pitch, y);
   ya = fixmul(sin_pitch, za) + fixmul(cos_pitch, y);
   x = fixmul(cos_roll, xa) + fixmul(sin_roll, ya);
   y = fixmul(cos_roll, ya) - fixmul(sin_roll, xa);
   o->midx = x + tx;
   o->midy = y + ty;
   o->midz = z + tz;
   for (i=0; i<o->nov; i++, v++)
   {
      /* for just yaw */
      if (pitch == 0 && roll == 0)
      {
         act[v].x = fixmul(cos_yaw, o->act[i].x)
            - fixmul(sin_yaw, o->act[i].z) + tx;
         act[v].z = fixmul(sin_yaw, o->act[i].x)
            + fixmul(cos_yaw, o->act[i].z) + tz;
         act[v].y = o->act[i].y + ty;
      }
      /* for yaw and pitch */
      else if (roll == 0)
      {
         x = o->act[i].x; y = o->act[i].y; z = o->act[i].z;
         za = cos_pitch * z - sin_pitch * y;
         ya = sin_pitch * z + cos_pitch * y;
         act[v].x = cos_yaw * x - sin_yaw * za + tx;
         act[v].y = ya + ty;
         act[v].z = sin_yaw * x + cos_yaw * za + tz;
      }
      /* for yaw, pitch, and roll */
      else {
         x = o->act[i].x; y = o->act[i].y; z = o->act[i].z;
         za = cos_pitch * z - sin_pitch * y;
         ya = sin_pitch * z + cos_pitch * y;
         xa = cos_yaw * x - sin_yaw * za;
         z = sin_yaw * x + cos_yaw * za;

         x = fixmul(cos_roll, xa) + fixmul(sin_roll, ya);
         y = fixmul(cos_roll, ya) - fixmul(sin_roll, xa);
         act[v].x = x + tx;
         act[v].y = y + ty;
         act[v].z = z + tz;
      }
   }
   o->yaw = yaw;
   o->pitch = pitch;
   o->roll = roll;
   o->x = tx;
   o->y = ty;
   o->z = tz;
#ifdef LIGHTING
   compute_normals(o);
#endif
   return 1;
}

int rotate_obj(int id, int dyaw, int dpitch, int droll)
{
   struct obj_struct *o = find_obj(id);
   if (o == NULLOBJ) return 0;
   return upd_obj(id, o->yaw + dyaw, o->pitch + dpitch, o->roll + droll,
                  o->x, o->y, o->z);
}

int translate_obj(int id, fixed dx, fixed dy, fixed dz)
{
   struct obj_struct *o = find_obj(id);
   if (o == NULLOBJ) return 0;
   return upd_obj(id, o->yaw, o->pitch, o->roll,
                  o->x + dx, o->y + dy, o->z + dz);
}

int scale_obj(int id, fixed sx, fixed sy, fixed sz)
{
   int i;
   struct obj_struct *o = find_obj(id);
   if (o == NULLOBJ) return 0;
   for (i=0; i<o->nov; i++)
   {
      o->act[i].x = fixmul(o->act[i].x, sx);
      o->act[i].y = fixmul(o->act[i].y, sy);
      o->act[i].z = fixmul(o->act[i].z, sz);
   }
   compute_obj(o);
   return upd_obj(id, o->yaw, o->pitch, o->roll, o->x, o->y, o->z);
}

#ifndef PRODUCTION
int locate_obj(int x, int y)
{
   fixed xx, yy, zz;
   struct obj_struct *o, *the_object;
   int i, j, last_fac, vert_no;
   fixed minx, miny, maxx, maxy, maxz;
   int fac_size, k, vert_idx, fac_no;
   xx = inttofix(x);
   yy = inttofix(y);
   the_object = NULLOBJ;
   //*** look through objects
   for (i=0; i<num_objs; i++)
   {
      o = obj[i];
      last_fac = o->first_fac + o->nof;
      //*** examine objects facets
      for (j=o->first_fac; j<last_fac; j++)
      {
         fac_no = nfac[j];
         if (fac_no == -1) continue;
         vert_idx = facfront[fac_no];
         vert_no = faclist[vert_idx].i;
         //*** compute min, max x and y and max z (for closest object)
         minx = pro[vert_no].x;
         maxx = minx;
         miny = pro[vert_no].y;
         maxy = miny;
         maxz = pro[vert_no].z;
         fac_size = size[fac_no];
         for (k=1; k<fac_size; k++)
         {
            vert_idx++;
            vert_no = faclist[vert_idx].i;
            if (pro[vert_no].x > maxx) maxx = pro[vert_no].x;
            else if (pro[vert_no].x < minx) minx = pro[vert_no].x;
            if (pro[vert_no].y > maxy) maxy = pro[vert_no].y;
            else if (pro[vert_no].y < miny) miny = pro[vert_no].y;
            if (pro[vert_no].z > maxz) maxz = pro[vert_no].z;
         }
         if (xx >= minx && xx <= maxx && yy >= miny && yy <= maxy)
         {
            if (the_object == NULLOBJ)
            {
               zz = maxz;
               the_object = o;
            }
            else if (zz < maxz)
            {
               zz = maxz;
               the_object = o;
            }
         }
      }
   }
   if (the_object == NULLOBJ) return -1;
   return the_object->id;
}

void flash_obj(struct obj_struct *o)
{
   int last_fac, i;
   if (o == NULLOBJ) return;
   last_fac = o->first_fac + o->nof;
   for (i=o->first_fac; i<last_fac; i++)
   {
      color[i] = (color[i] & 0xff00) + (((color[i] & 0xff) + 1) & 0xff);
   }
}

int mark_obj(int id)
{
   int last_fac, i;
   struct obj_struct *o;
   o = find_obj(id);
   if (o == NULLOBJ) return 0;
   last_fac = o->first_fac + o->nof;
   for (i=o->first_fac; i<last_fac; i++)
   {
      color[i] |= 0xff0000;
   }
   o->marked = 1;
   return 1;
}

int color_obj(int id, int d)
{
   int last_fac, i;
   struct obj_struct *o;
   o = find_obj(id);
   if (o == NULLOBJ) return 0;
   last_fac = o->first_fac + o->nof;
   for (i=o->first_fac; i<last_fac; i++)
   {
      color[i] = (color[i] & 0xff00) + (((color[i] & 0xff) + d) & 0xff);
   }
   o->marked = 1;
   return 1;
}

int make_textured(int id, int t)
{
   int last_fac, i;
   struct obj_struct *o;
   o = find_obj(id);
   if (o == NULLOBJ) return 0;
   last_fac = o->first_fac + o->nof;
   for (i=o->first_fac; i<last_fac; i++)
   {
      color[i] = (color[i] & 0xff) | ((0x3 + t) << 8);
   }
   o->marked = 1;
   return 1;
}

int spin_obj_toggle(int id)
{
   struct obj_struct *o;
   o = find_obj(id);
   if (o == NULLOBJ) return 0;
   if (o->var[1]) o->var[1] = 0; else o->var[1] = 1;
   return 1;
}

int unmark_obj(int id)
{
   int last_fac, i;
   struct obj_struct *o;
   o = find_obj(id);
   if (o == NULLOBJ) return 0;
   last_fac = o->first_fac + o->nof;
   for (i=o->first_fac; i<last_fac; i++)
   {
      color[i] &= 0xffff;
   }
   o->marked = 0;
   return 1;
}

int is_obj_marked(int id)
{
   struct obj_struct *o = find_obj(id);
   if (o == NULLOBJ) return 0;
   return o->marked;
}
#endif

int del_obj(int id)
{
   int i, j;
   struct obj_struct *o = find_obj(id);
   if (o == NULLOBJ) return 0;

   //*** delete faclist items
   for (j=o->first_list; j<firstfree - o->nol + 1; j++)
   {
      faclist[j].i = faclist[j + o->nol].i - o->nov;
   }

   //*** change the facfront of all facets that start after it
   for (j=o->first_fac + o->nof - 1; j<nof; j++)
   {
      facfront[j] -= o->nol;
   }

   //*** delete vert items
   for (j=o->first_vert; j<nov - o->nov + 1; j++)
   {
      act[j] = act[j + o->nov];
   }

   //*** delete fac items (size, color, facfront)
   for (j=o->first_fac; j<nof - o->nof + 1; j++)
   {
      size[j] = size[j + o->nof];
      color[j] = color[j + o->nof];
      facfront[j] = facfront[j + o->nof];
   }

   //*** delete the object
   for (i=0; i<num_objs; i++)
   {
      if (obj[i]->id == o->id)
      {
         for (j=i; j<num_objs - 1; j++)
         {
            obj[j] = obj[j+1];
            obj[j]->first_vert = obj[j]->first_vert - o->nov;
            obj[j]->first_list = obj[j]->first_list - o->nol;
            obj[j]->first_fac  = obj[j]->first_fac  - o->nof;
         }
         break;
      }
   }
   nof -= o->nof;
   nov -= o->nov;
   num_objs--;
   if (o->act) free(o->act);
   free(o);
   return 1;
}

unsigned char reflect(unsigned char a, unsigned char b)
{
/*
                                 0
                           224   |    32
                               \ |  /
                      192--------+--------64
                               / |  \
                           160   |    96
                                128
*/
   unsigned char c;
   // first normalize b to the face that a strikes
   if (abs((int)a - (int)b) > 128) b -= 128;
   // next figure out if b is clockwise or counterclockwise of a
   c = ((int)b - (int)a) + (int)a;
   return c;
}

int time_slice()
{
   extern fixed jump_accel, up_accel;
   int flag, retries;
   fixed x, y, z, xa, ya, za, cx, cy, cz, aim;
   float dx, dz, dy, tx, ty, tz;
   fixed cos_yaw, sin_yaw, cos_pitch, sin_pitch;
   const fixed low = inttofix(-2000);
   const fixed high = inttofix(2000);
   unsigned char ny, np, nr;
   int obj_ix;
   int i;
   extern int forced_turn;
   extern int sitting_on;
   forced_turn = 0;
   if (was_hit > 0) was_hit--;
   for (i=0; i<num_objs; i++)
   {
      struct obj_struct *o = obj[i];
      switch (o->type)
      {
      case PYRAMID: case CUBE:
         if (o->var[1] != 0)
         {
            upd_obj(o->id, o->yaw + 1, o->pitch, o->roll, o->x, o->y, o->z);
            if (sitting_on == o->id)
            {
               forced_turn = 1;
            }
         }
         break;
//      case RECTANGLE:    // rain like behavior
//         if (o->midy < inttofix(-100))
//         {
//               upd_obj(o->id, o->yaw, o->pitch, o->roll,
//                     inttofix((rand()%2000)-1000), inttofix(1000),
//                     inttofix((rand()%2000)-1000));
//         }
//         else
//         {
//            upd_obj(o->id, o->yaw + 3, o->pitch + 3,
//                        o->roll + 3, o->x, o->y - inttofix(5), o->z);
//         }
//         break;
      case FACE:
         upd_obj(o->id, o->yaw - 3, o->pitch, o->roll, o->x, o->y, o->z);
         break;
      case TREE:
         upd_obj(o->id, o->yaw + 1, o->pitch, o->roll, o->x, o->y, o->z);
         break;
      case GEM:
         upd_obj(o->id, o->yaw + 4, o->pitch + 4,
            o->roll + 4, o->x, o->y, o->z);
         flash_obj(o);
         break;
      case SPHERE:
         upd_obj(o->id, o->yaw + 4, o->pitch + 4,
            o->roll + 4, o->x, o->y, o->z);
         break;
      case TORUS: case WORLD:
         upd_obj(o->id, o->yaw + 2, o->pitch,
            o->roll, o->x, o->y, o->z);
         break;
      case BIT_CRITTER:  //*************************************************
         // move critter along z axis with collision avoidance
         flag = 0;
         x = 0;
         z = 16;
         y = 0;
         o->roll = 0;
         for (retries = 0; retries < 5 && flag == 0; retries++)
         {
            cos_yaw = pcos[o->yaw]; sin_yaw = psin[o->yaw];
            cos_pitch = pcos[o->pitch]; sin_pitch = psin[o->pitch];
            za = cos_pitch * z - sin_pitch * y;
            ya = sin_pitch * z + cos_pitch * y;
            cx = cos_yaw * x - sin_yaw * za + o->x;
            cy = ya + o->y;
            cz = sin_yaw * x + cos_yaw * za + o->z;

            if (cx > high || cx < low || cy > high/2 || cy < 30
               || cz > high || cz < low)
            {
               o->yaw = rand() % 256;
               o->pitch = rand() % 256;
            }
            else if ((obj_ix = collision(cx, cy, cz, o->radius * 2, o->id)) != -1)
            {
               if (obj_ix >= 0
                  && obj[obj_ix]->id != o->id
                  && obj_ix != o->id)
               {
                  o->yaw = rand() % 256;
                  o->pitch = rand() % 256;
               }
               else if (obj_ix == -999)
               {
                  o->yaw = rand() % 256;
                  o->pitch = rand() % 256;
                  if (was_hit == 0) { play_sound("land.wav", 25); was_hit = 10; }
                  up_accel = jump_accel;
               }
            }
            else if (rand() % 256 == 0)
            {
                dy = -my - o->y;
                dx = -mx - o->x;
                dz = -mz - o->z;
                aim = atan2(dz, dx);
                o->yaw = 192.0 + aim * (256.0 / 6.283);
                aim = atan2(dy, sqrt(dx*dx + dz*dz));
                o->pitch = aim * (256.0 / 6.283);
            }
            else flag = 1;
         }
         upd_obj(o->id, o->yaw, o->pitch, o->roll, cx, cy, cz);
         break;
      case TRIANGLE: //****************************************************
            //********** move missile along its z axis ******************
            x = 0; z = 6; y = 0;
            cos_yaw = pcos[o->yaw]; sin_yaw = psin[o->yaw];
            cos_pitch = pcos[o->pitch]; sin_pitch = psin[o->pitch];
            xa = cos_yaw * x - sin_yaw * z;
            za = sin_yaw * x + cos_yaw * z;
            cz = fixmul(cos_pitch, za) - fixmul(sin_pitch, y) + o->z;
            cy = fixmul(sin_pitch, za) + fixmul(cos_pitch, y) + o->y;
            cx = xa + o->x;
            //*********** if off board remove missile ****************
            if (cx > high || cx < low || cy > high || cy < low
               || cz > high || cz < low)
            {
               struct obj_struct *cannon = find_obj(o->state);
               if (cannon) cannon->state = 0;
               del_obj(o->id);
            }
//            //********** if collision delete or reflect missile *************
//            // unless collision is with cannon that fired it
//            /* else */ if ((obj_ix = collision(cx, cy, cz, o->radius, o->id)) != -1
//               && obj[obj_ix]->id != o->state
//               && obj[obj_ix]->id != o->id
//               && obj_ix != o->id
//               && obj[obj_ix]->type != o->type)
//            {
//               // deflected unless it hits it flat on
//               if (obj[obj_ix]->type == RECTANGLE
//                  && (obj[obj_ix]->yaw != o->yaw
//                     || obj[obj_ix]->pitch != o->pitch
//                     || obj[obj_ix]->roll != o->roll))
//               {
//                  if (obj[obj_ix]->yaw == o->yaw) ny = o->yaw;
//                  else ny = reflect(o->yaw, obj[obj_ix]->yaw);
//                  if (obj[obj_ix]->pitch == o->pitch) np = o->pitch;
//                  else np = reflect(o->pitch, obj[obj_ix]->pitch);
//                  if (obj[obj_ix]->roll == o->roll) nr = o->roll;
//                  else nr = reflect(o->roll, obj[obj_ix]->roll);
//                  upd_obj(o->id, ny, np, nr, o->x, o->y, o->z);
//               }
//               else
//               {
//                  struct obj_struct *cannon = find_obj(o->state);
//                  if (cannon) cannon->state = 0;
//                  del_obj(o->id);
//               }
//            }
//            else
//            {
             upd_obj(o->id, o->yaw, o->pitch, o->roll, cx, cy, cz);
//            }
         break;
      }
   }
}

