/*

NanoTech - a 3d game engine
Copyright (C) 1996  Sean Lane Fuller

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

Sean Lane Fuller
124 Autumn Lane
Tullahoma, TN 37388
615-393-4550
email: fuller@edge.net

*/

#include <stdio.h>
#include <signal.h>
#include <math.h>
#include <dos.h>
#include <graph.h>
#include "keyboard.h"
#include "pix.h"
#include "3d.h"
#include "timer.h"
#include "sound.h"

int MAXX, MAXY, NUMXPIXELS, NUMYPIXELS, MIDX, MIDY, VIDMODE, CLIPMAXX, CLIPMAXY;

int score = 0;
typedef float fixed;
int no_perspective_correct = 0;
int no_gravity = 0;
int god_mode = 0;
int no_sound = 0;
int do_time = 0;
int no_music = 0;
int no_stars = 0;
int no_textures = 0;
int do_test = 0;
int no_intro = 0;
int do_ccpp = 0;
static char msgbuf[256];

#define fixtoint(x) ((int)(x))
#define fixtodbl(x) (x)
#define inttofix(x) ((float)(x))
#define dbltofix(x) (x)
#define fixmul(a,b) ((a)*(b))

extern int last_id;

fixed up_accel = inttofix(0);
fixed jump_accel = 32, ground_my = -100;
const fixed jump_decel = 4;
const fixed turn_accel = .060, max_turn = 3.14159267/15.0;
int the_sphere;
extern float midi_tempoadjust;
int last_id;
int frame = 16;
float fps = 16;
int level = 0;
int quit = 0;
int sitting_on = 0;
int forced_turn = 0;
int first_sphere;
int last_sphere;
char *msg = 0;
int msg_timer = 0;

MOD far *song;
SAMPLE *sound1, *rumble;
DWORD length1, rumble_length;
DWORD id1;
char errstring[100];

MIDI *midi;

int start_sound()
{
   int i;
   if (no_sound) return;
   midi_data = 0;
   mod_data = 0;
   if(!SBSetUp()) {
      printf("Sound Blaster setup failed with rc = %d\n", sberr);
      FreeMod(song);
      exit(1);
   }
   if (!no_music)
   {
      song = LoadMod("terri.mod",errstring);
      if(!song) {
         printf("ERROR loading MOD:  %s\n",errstring);
         exit(1);
      }
   }
   SetSampleRate(10000);
   TimeVSync();
   GoVarmint();
   sound1 = rumble = 0;
   if (!no_music)
   {
      mod_data = song;
      ModCommand(v_play);
   }
}

int stop_sound()
{
   if (no_sound) return;
  DropDeadVarmint();
//  SBCleanUp();
//  if (song) FreeMod(song);
//  if (midi) FreeMidi(midi);
}

int new_song(char *name)
{
   if (no_music) return;
   if (mod_data)
   {
      ModCommand(v_stop);
      mod_data = (MOD *)0;
      FreeMod(song);
   }
   if (midi_data)
   {
      MidiCommand(v_stop);
      midi_data = (MIDI *)0;
      FreeMidi(midi);
   }
   if (strstr(name, ".mod") || strstr(name, ".s3m")
         || strstr(name, ".MOD") || strstr(name, ".S3M"))
   {
      song = LoadMod(name,errstring);
      if(!song) {
         pix_term();
         DropDeadVarmint();                      // Stop the SB interrupt.
         SBCleanUp();                            // Clean up sound stuff.
         printf("ERROR loading MOD:  %s\n",errstring);
         exit(0);
      }
      mod_data = song;
      ModCommand(v_play);
   }
   else
   {
      midi = LoadMidi(name, errstring);
      if (!midi)
      {
         pix_term();
         DropDeadVarmint();                      // Stop the SB interrupt.
         SBCleanUp();                            // Clean up sound stuff.
         printf("ERROR loading MIDI:  %s\n",errstring);
         exit(0);
      }
      midi_data = midi;
      MidiCommand(v_play);
   }
}

void load_mod()
{
   int i;
   extern char mod_name[256];
   char name[256];
   if (no_music) return;
   for (i=0; i<255 && mod_name[i] != 0 && mod_name[i] != '\n'; i++)
         name[i] = mod_name[i];
   name[i] = 0;
   new_song(name);
}

int play_sound(char *name, int vol)
{
   if (no_sound) return;
   if (sound1)
   {
      AlterSoundEffect(id1, v_stop, 0);
      free(sound1);
   }
   sound1 = LoadWave(name,&length1);
   id1 = PlaySound(sound1,length1,v_fancy);      // Play a sound effect
//   AlterSoundEffect(id1,v_setrate,1.0);    // Change the frequency
//   AlterSoundEffect(id1,v_setvolume,vol);    // Change the volume
}

static DWORD loopid;
void start_rumble()
{
   if (no_sound) return;
    if (rumble) {
      AlterSoundEffect(loopid, v_resume, 0);
    }
    else {
       rumble = LoadWave("rumble.wav", &rumble_length);
       loopid = PlaySound(rumble,rumble_length,v_fancy);
       AlterSoundEffect(loopid,v_setrepeat,-1);
       AlterSoundEffect(loopid,v_setvolume,32);
    }
}

void stop_rumble()
{
   if (no_sound) return;
   AlterSoundEffect(loopid,v_pause,0);
}

void add_pieces()
{
   struct obj_struct *o = find_obj(the_sphere);
   struct obj_struct *tri;
   int i;
   first_sphere = last_id;
//if (o == 0)
//{
//   printf("the_sphere was not found\n");
//}
   for (i=0; i<25; i++)
   {
      tri = add_obj(last_id, 6);
      translate_obj(last_id, o->x, o->y, o->z);
      rotate_obj(last_id, rand() % 256, rand() % 256, 0);
      tri->state = o->id;
      last_sphere = last_id;
      last_id++;
      time_slice();
   }
   del_obj(the_sphere);
}

void align_viewer()
{
   float delta = .1;
   while (fabs(roll) >= delta || fabs(pitch) >= delta || fabs(yaw) >= delta)
   {
      if (roll < -delta) roll += delta;
      else if (roll > delta) roll -= delta;
      if (pitch < -delta) pitch += delta;
      else if (pitch > delta) pitch -= delta;
      if (yaw < -delta) yaw += delta;
      else if (yaw > delta) yaw -= delta;
      sin_pitch = sin(pitch);
      cos_pitch = cos(pitch);
      sin_yaw = sin(yaw);
      cos_yaw = cos(yaw);
      pix_refresh();
      ClearEdgeLists();
      ScanEdges();
      DrawSpans();
   }
}

void hit_sphere()
{
   short i, j;
   float speed;
   short which_four = 0;
   WORD f;
   float fr;

   char    inst[2][11] = {
      { 0x00,0x00,0x00,0x00,0x80,0x80,0x00,0x00,0x06,0x03,0x00},
      { 0x02,0x06,0x94,0x0A,0x80,0x80,0x00,0x00,0x00,0x00,0x00}};
   short vints[8] = {0,1,0,1,0,1,0,1}; // Start/end frequencies for intro
   WORD startfreq[8] ={0xa000,0x6000,0xa000,0x2000,0x1000,0x9000,0xa000,0x6000};
   WORD endfreq[8] = {0x0800,0x2800,0x4800,0x6800,0x8800,0xa800,0xc800,0xe800};

   add_pieces();
   if (!no_sound)
   {
      FMReset();
      play_sound("land.wav", 60);
      for(i = 0; i < 8; i++) {             // initialize music voices
         FMSetVoice(i,inst[vints[i]]);
         FMSetFrequency(i,endfreq[i]);
         FMSetVolume(i,0xff);
         FMKeyOn(i);
      }
   }

   speed = inttofix(20);
   for(i = 0; i<1000; i += 20) {
      mx -= speed * sin_yaw * 2;
      mz -= speed * cos_yaw * 2;
//      roll += dbltofix(.05);
//      cos_roll = dbltofix(cos(fixtodbl(roll)));
//      sin_roll = dbltofix(sin(fixtodbl(roll)));

      if (which_four == 0) which_four = 4; else which_four = 0;
      if (!no_sound)
      {
         for (j=0; j<8; j++)
         {
            FMSetFrequency(j, (rand() % 0x2000) + 0x1000);
            FMSetVolume(j, 0xff - i/20);
         }
      }
      time_slice();
      make_scene();
      draw_scene();
      pix_refresh();
   }
   if (!no_sound) for (j=0; j<8; j++) FMSetVolume(j, 0);
   speed += speed;
   if (!no_sound)
   {
      for(i = 0; i < 8; i++) {             // initialize music voices
         FMSetVoice(i,inst[vints[i]]);
         FMSetFrequency(i,startfreq[i]);
         FMSetVolume(i,0x00);
         FMKeyOn(i);
      }
   }
   for(i = 0; i < 1000; i += 20) {
      fr = (float)i/1000.0;
      for(j = 0; j < 8; j++) {
         f = startfreq[j]+((float)endfreq[j]-(float)startfreq[j]) * fr;
         f = f * .2;
         if (!no_sound)
         {
            FMSetFrequency(j,f);
            FMSetVolume(j,fr*0x3f );
            FMKeyOn(j);
         }
      }
      mx += speed * sin_yaw * 2;
      mz += speed * cos_yaw * 2;
//      roll += dbltofix(.05);
//      cos_roll = dbltofix(cos(fixtodbl(roll)));
//      sin_roll = dbltofix(sin(fixtodbl(roll)));
      time_slice();
      make_scene();
      draw_scene();
      pix_refresh();
   }
   if (!no_sound) FMReset();
}

void sound_intro()
{
   extern void init_viewer();
   char    inst[2][11] = {
      { 0x00,0x00,0x00,0x00,0x80,0x80,0x00,0x00,0x06,0x03,0x00},
      { 0x02,0x06,0x94,0x0A,0x80,0x80,0x00,0x00,0x00,0x00,0x00}};
   short vints[8] = {0,1,0,1,0,1,0,1}; // Start/end frequencies for intro
   WORD startfreq[8] ={0xa000,0x6000,0xa000,0x2000,0x1000,0x9000,0xa000,0x6000};
   WORD endfreq[8] = {0x0800,0x2800,0x4800,0x6800,0x8800,0xa800,0xc800,0xe800};
   short i,j;
   float fr;
   WORD f;
   DWORD id;
   if (!no_sound)
   {
      FMReset();
      for(i = 0; i < 8; i++) {             // initialize music voices
         FMSetVoice(i,inst[vints[i]]);
         FMSetFrequency(i,startfreq[i]);
         FMSetVolume(i,0x00);
         FMKeyOn(i);
      }
   }
   for(i = 0; i < 1000; i+= 20) {
//    _disable();                       // Disable interrupts to protect
                                      // Floating point operations.
      fr = (float)i/1000.0;
      for(j = 0; j < 8; j++) {
         f = startfreq[j]+((float)endfreq[j]-(float)startfreq[j]) * fr;
         if (!no_sound)
         {
            FMSetFrequency(j,f);
            FMSetVolume(j,fr*0x3f );
         }
      }
      mz = dbltofix(-(1000.0 - (float)i) * 12);
      time_slice();
      make_scene();
      draw_scene();
      pix_refresh();
   }
   if (!no_sound)
   {
      for(i = 0; i < 8; i++) {
         FMSetVolume(i,0x00);
         FMKeyOff(i);
      }
   }
   init_viewer();
   make_scene();
   draw_scene();
   pix_refresh();
}

void rotate_it(int id)
{
   make_scene();
   draw_scene();
   pix_refresh();
   while (!keys[ESC])
   {
      if (keys[UP]) rotate_obj(id, 1, 0, 0);
      if (keys[DOWN]) rotate_obj(id, -1, 0, 0);
      if (keys[LEFT]) rotate_obj(id, 0, 1, 0);
      if (keys[RIGHT]) rotate_obj(id, 0, -1, 0);
      if (keys[PGUP]) rotate_obj(id, 0, 0, 1);
      if (keys[PGDN]) rotate_obj(id, 0, 0, -1);
      make_scene();
      draw_scene();
      pix_refresh();
   }
   keys[ESC] = 0;
}

void options()
{
   start_rumble();
   draw_scene();
   pix_pop_msg("                          \n"
               "----------Options---------\n"
               " (t)extures               \n"
               " (m)usic                  \n"
               " (s)tars                  \n"
               " (p)erspective correction \n"
               "                          ", 255, 34);
   pix_refresh();
   do {
      if (keys[ESC]) { keys[ESC] = 0; stop_rumble(); return; }
   } while (!keys[KB_T] && !keys[KB_M] && !keys[KB_S] && !keys[KB_P]);
   if (keys[KB_T]) { no_textures = (no_textures) ? 0 : 1; }
   if (keys[KB_P]) { no_perspective_correct = (no_perspective_correct)? 0 : 1;}
   if (keys[KB_M]) { no_music = (no_music) ? 0 : 1;
      if (no_music)
      {
         if (mod_data)
         {
            ModCommand(v_stop);
            FreeMod(song);
            mod_data = (MOD *)0;
         }
         if (midi_data)
         {
            MidiCommand(v_stop);
            FreeMidi(midi);
            midi_data = (MIDI *)0;
         }
      }
      else
      {
         load_mod();
      }
   }
   if (keys[KB_S]) { no_stars = (no_stars) ? 0 : 1; }
   keys[KB_T] = 0;
   keys[KB_M] = 0;
   keys[KB_S] = 0;
   keys[KB_P] = 0;
   stop_rumble();
}

void add_it(fixed mx, fixed my, fixed mz)
{
   int key, type;
   start_rumble();
   draw_scene();
   pix_pop_msg("               \n"
               "------Add------\n"
               " (c)ube        \n"
               " (p)yramid     \n"
               " (s)phere      \n"
               " (r)ectangle   \n"
               " (t)riangle    \n"
               " c(y)linder    \n"
               " (b)it critter \n"
               " (g)em         \n"
               "               ", 255, 34);
   pix_refresh();
   do {
      if (keys[ESC]) { keys[ESC] = 0; return; }
   } while (!keys[KB_C] && !keys[KB_P] && !keys[KB_S] && !keys[KB_R]
      && !keys[KB_Y] && !keys[KB_T] && !keys[KB_B] && !keys[KB_G]);
   if (keys[KB_C]) type = 2;
   else if (keys[KB_P]) type = 1;
   else if (keys[KB_S]) type = 3;
   else if (keys[KB_R]) type = 5;
   else if (keys[KB_Y]) type = 4;
   else if (keys[KB_T]) type = 6;
   else if (keys[KB_B]) type = 19;
   else if (keys[KB_G]) type = 20;
   add_obj(last_id, type);
//   if (type == 19)
//      rotate_obj(last_id, rand() % 256, rand() % 256, rand() % 256);
   translate_obj(last_id++, mx, my, mz);
   keys[KB_C] = 0; keys[KB_P] = 0; keys[KB_S] = 0; keys[KB_R] = 0;
   keys[KB_Y] = 0; keys[KB_T] = 0; keys[KB_B] = 0; keys[KB_G] = 0;
   stop_rumble();
}

void move_it(int id)
{
   int key, axis;
   fixed d = inttofix(10);
   make_scene();
   draw_scene();
   pix_refresh();
   while (!keys[ESC])
   {
      if (keys[RIGHT]) translate_obj(id, d, 0, 0);
      if (keys[UP]) translate_obj(id, 0, d, 0);
      if (keys[PGUP]) translate_obj(id, 0, 0, d);
      if (keys[LEFT]) translate_obj(id, -d, 0, 0);
      if (keys[DOWN]) translate_obj(id, 0, -d, 0);
      if (keys[PGDN]) translate_obj(id, 0, 0, -d);
      make_scene();
      draw_scene();
      pix_refresh();
   }
   keys[ESC] = 0;
}

int save()
{
   int key;
   start_rumble();
   keys[ESC] = 0;
   make_scene();
   draw_scene();
   pix_pop_msg(
      "             \n"
      " Save? (y/n) \n"
      "             ", 255, 34);
   pix_refresh();
   while (!keys[KB_Y] && !keys[KB_N] && !keys[ESC]) { ; }
   stop_rumble();
   if (keys[KB_Y]) return 1; else { keys[ESC] = 0; return 0; }
}

int escape()
{
   int key;
   start_rumble();
   keys[ESC] = 0;
   make_scene();
   draw_scene();
   pix_pop_msg(
      "             \n"
      " Quit? (y/n) \n"
      "             ", 255, 34);
   pix_refresh();
   while (!keys[KB_Y] && !keys[KB_N] && !keys[ESC]) { ; }
   stop_rumble();
   if (keys[KB_Y]) return 1; else { keys[ESC] = 0; return 0; }
}

void info()
{
   char buf[1024];
   extern int num_objs, nof, nov;
   int pps = fps * nof;
   start_rumble();
   draw_scene();
   sprintf(buf, 
      "                     \n"
      "      Scene Info     \n"
      "---------------------\n"
      " %5d objects       \n"
      " %5d facets        \n"
      " %5d vertices      \n"
      " %5d frames/second \n"
      " %5d facets/second \n"
      " %5d Level         \n"
      "                     \n"
      "  press <escape>     \n"
      "                     ",
      num_objs, nof, nov, (int)fps, pps, level);
   pix_pop_msg(buf, 255, 34);
   pix_refresh();
   while (!keys[ESC]) { ; }
   keys[ESC] = 0;
   stop_rumble();
}

void help()
{
   start_rumble();
   draw_scene();
   pix_pop_msg("                                                       \n"
               "          Nanotech 2.6 - by Sean Lane Fuller           \n"
               "-------------------------------------------------------\n"
               " up arrow ...... move forward     a .... add object *  \n"
               " down arrow .... move backward    tab .. modify object*\n"
               " left arrow .... turn left        i .... show info *   \n"
               " right arrow ... turn right       c .... show compass* \n"
               " page up ....... look up          l .... show location*\n"
               " page down ..... look down        s .... save map *    \n"
               " spacebar ...... jump             o .... toggle option*\n"
               " escape ........ quit to DOS      u,d .. up, down *    \n"
               "                                  g .... gravity *     \n"
               "                                                       \n"
               " Try to collide with the spinning sphere to exit a     \n"
               " level.  The cubes are switches and landing on one may \n"
               " toggle one or more platforms.  (*=only in edit mode)  \n"
               " Collect gems for score.                               \n"
               "                    press <escape>                     \n"
               "                                                       ", 255, 34);
   pix_refresh();
   while (!keys[ESC]) { ; }
   keys[ESC] = 0;
   stop_rumble();
}

void init_viewer()
{
   mx = 0; my = ground_my; mz = 0;

   yaw = 0;
   pitch = 0;
   roll = 0;

   cos_yaw = pcos[0];
   sin_yaw = psin[0];
   sin_pitch = psin[0];
   cos_pitch = pcos[0];
   sin_roll = psin[0];
   cos_roll = pcos[0];

}

void intro(int long_version)
{
   struct obj_struct *l[9];
   unsigned char a;
   int i, j, k;
   float f;
   const int near_z = -290;
   const int far_z = -2000;
   const int spin_y = 100;
   clear_scene();
   pix_setup_palette(1.0);
   
   init_viewer();
   l[1] = add_obj(1, 7);
   l[2] = add_obj(2, 8);
   l[3] = add_obj(3, 7);
   l[4] = add_obj(4, 9);
   l[5] = add_obj(5, 10);
   l[6] = add_obj(6, 11);
   l[7] = add_obj(7, 12);
   l[8] = add_obj(8, 13);
   add_obj(9, 3);
   upd_obj(9, 0, 0, 0, 0, spin_y, -370);
//   add_obj(10, 14);
//   upd_obj(10, 0, 0, 0, 0, spin_y, -350);
//   add_obj(11, 14);
//   upd_obj(11, 0, 0, 0, 150, 30, -500);
   if (long_version == 0)
   {
      float brightness = 0.0;
      for (j=far_z; j<=-400; j+=((long_version)?100:10))
      {
//         pix_setup_palette(brightness);
         if (keys[ESC]) { break; }
         upd_obj(9, 0, 0, 0, 0, spin_y, inttofix(j));
         time_slice();
         make_scene(); draw_scene(); pix_refresh();
         if (brightness < 1.0) brightness += 0.01;
      }
//      pix_setup_palette(1.0);
   }
   last_id = 10;
   the_sphere = 9;
   // start far away and move closer
   for (j=far_z; j<=near_z; j+=((long_version)?100:10))
   {
      if (keys[ESC]) { break; }
      for (i=1, k=-145; i<9; i++, k+=40)
      { upd_obj(i, 0, 0, 0, inttofix(k), spin_y, inttofix(j)); }
      time_slice();
      make_scene(); draw_scene(); pix_refresh();
   }
   if (long_version)
   {
      // spin letters around all axes
      for (i=0; i<=256; i+=2)
      {
         if (keys[ESC]) { break; }
         for (j=1; j<9; j++) rotate_obj(j, 1, 1, 1);
         time_slice();
         make_scene(); draw_scene(); pix_refresh();
      }
      a = 64;
      // spin word around z axis
      for (j=0; j<=64; j++, a+=8)
      {
         if (keys[ESC]) { break; }
         for (i=0, k=-185; i<9; i++, k+=40)
         {  upd_obj(i, 0, 0, 0, inttofix(k) * psin[a],
               spin_y - inttofix(k) * pcos[a], inttofix(near_z)); }
         time_slice();
         make_scene(); draw_scene(); pix_refresh();
      }
      // spin word around y axis
      for (j=0; j<=64; j++, a+=8)
      {
         if (keys[ESC]) { break; }
         for (i=0, k=-185; i<9; i++, k+=40)
         {  upd_obj(i, 0, 0, 0, inttofix(k) * psin[a],
               spin_y,inttofix(near_z) - inttofix(k) * pcos[a]); }
         time_slice();
         make_scene(); draw_scene(); pix_refresh();
      }
      // spin word around x axis
      for (i=0; i<=256; i+=8)
      {
         if (keys[ESC]) { break; }
         for (j=1; j<9; j++) rotate_obj(j, 0, 1, 0);
         time_slice();
         make_scene(); draw_scene(); pix_refresh();
      }
      // move away
      for (j=near_z; j>=far_z; j-=200)
      {
         if (keys[ESC]) { break; }
         for (i=1, k=-145; i<9; i++, k+=40)
         { upd_obj(i, 0, 0, 0, inttofix(k), spin_y, inttofix(j)); }
         time_slice();
         make_scene(); draw_scene(); pix_refresh();
      }
      // start far away and move closer
      for (j=far_z; j<=near_z; j+=200)
      {
         if (keys[ESC]) { break; }
         for (i=1, k=-145; i<9; i++, k+=40)
         { upd_obj(i, 0, 0, 0, inttofix(k), spin_y, inttofix(j)); }
         time_slice();
         make_scene(); draw_scene(); pix_refresh();
      }
   }
//   add_pieces();
//   play_sound("land.wav", 60);
//   for (i=0; i<2; i++) { time_slice(); make_scene(); draw_scene(); pix_refresh(); }
//   the_sphere = 10;
//   add_pieces();
//   play_sound("land.wav", 60);
//   for (i=0; i<2; i++) { time_slice(); make_scene(); draw_scene(); pix_refresh(); }
//   the_sphere = 11;
   hit_sphere();
   // clear intro objects and display intro text and picture
   clear_scene();
//   if (long_version)
//   {
//      pix_cls(0);
//      pix_refresh();
//      pix_loadpcx("intro.pcx", 1);
//      pix_refresh();
//      while (!keys[ESC]) { ; }
//      pix_setup_palette(0.0);
//   }
   pix_cls(0);
   pix_refresh();
//   pix_setup_palette(0.0);
}

void manipulate(int id)
{
   int i;
   char buf[1024];
   start_rumble();
   for (i=0; i<4; i++)
   {
      make_scene();
      draw_scene();  
      pix_refresh();
   }
   make_scene();
   draw_scene();
   sprintf(buf,
      "            \n"
      "---Modify---\n"
      " (m)ove     \n"
      " (r)otate   \n"
      " (d)elete   \n"
      " (s)pin     \n"
      "            \n"
      "  id #%03d   \n"
      " type #%03d  \n"
      "            ", id, find_obj(id)->type);
   pix_pop_msg(buf, 255, 34);
   pix_refresh();
   do {
   } while (!keys[ESC] && !keys[KB_M] && !keys[KB_R] && !keys[KB_D]
         && !keys[KB_S]);
   if (keys[KB_M]) move_it(id);
   if (keys[KB_R]) rotate_it(id);
   if (keys[KB_D]) del_obj(id);
   if (keys[KB_S]) spin_obj_toggle(id);
   keys[ESC] = 0;
   keys[KB_D] = 0;
   keys[KB_M] = 0;
   keys[KB_R] = 0;
   keys[KB_S] = 0;
   stop_rumble();
}

void win()
{
   float f;
   init_textures();
   align_viewer();
   init_viewer();
   intro(0);
   quit = 1;
return;
   new_song("w.mod");
   pix_cls(0);
//   pix_loadpcx("win.pcx", 1);
   pix_refresh();
   while (!keys[ESC]) { ; }
   quit = 1;
   pix_cls(0);
   pix_pulldown();
//   pix_setup_palette(1.0);
}

void next_level()
{
   start_rumble();
   new_song("tunnel.mod");
   hit_sphere();
   align_viewer();
   clear_scene();
   init_viewer();
   stop_rumble();
//   tunnel();
   if (VIDMODE == _MRES256COLOR) warp();
   if (load_area(++level)) { win(); return; }
   init_textures();
   sound_intro();
   load_mod();
   align_viewer();
   init_viewer();
   frame = 16;
   fps = 16;
   pix_setup_palette(1.0);
}

int do_switch(int id)
{
   int i;
   int rc = 0;
   struct obj_struct *o = find_obj(id);
   struct obj_struct *p;
   if (o)
   {
      for (i=0; i<MAXVAR; i++)
      {
         if (o->var[i] > 0)
         {
            p = find_obj(o->var[i]);
            if (p->off) p->off = 0; else p->off = 1;
            rc = 1;
         }
      }
   }
   return rc;
}

void observe()
{
   int last_sitting_on;
   const fixed extra_jump = dbltofix(.3);
   int id;
   fixed xa, ya, za, x, y, z;
   int show_compass = 0;
   int i, rc;
   float pal_level = 0.0;
   float pal_inc = 0.1;
   fixed cx, cy, cz;
   fixed ocx[3], ocy[3], ocz[3];

   unsigned int last_clock_tick;
   fixed yaw_turn = 0, pitch_turn = 0;
   const float momentum = 10.0;
   const float max_momentum = 50.0;
   float speed = 0;

   float temp1, temp2, temp3;
   const float arms_length = 200.0;
   int show_location = 0;
   char buf[256];
   int key;
   int inst_n = 6;
   extern int clock_cycles;

   ocx[0] = inttofix(40);  ocy[0] = 0;  ocz[0] = 0;
   ocx[1] = 0;  ocy[1] = inttofix(40);  ocz[1] = 0;
   ocx[2] = 0;  ocy[2] = 0;  ocz[2] = inttofix(40);

   init_viewer();
   pix_cls(0);
   do {
      if (area_type == 'b' && frame % 30 == 0 && !god_mode)
      {
         pal_level += pal_inc;
         if (pal_level > 1.0 || pal_level < 0.0)
         {
            pal_inc *= -1;
            pal_level += pal_inc;
         }
         pix_setup_palette(pal_level);
      }
      if (!no_gravity) time_slice();
      make_scene();
      draw_scene();
      frame++;
      if (show_location)
      {
         sprintf(buf, "x:%5d y:%5d z:%5d fps:%3d ccpp:%3d", (int)mx, (int)my,
                  (int)mz, (int)fps, (int)clock_cycles);
         pix_disp_str(buf, 10, 0, 31, 0);
      }

      if (msg_timer > 0)
      {
         msg_timer--;
         pix_disp_str(msg, 0, 0, 31, 0);
      }

//      sprintf(buf, "level %2d  time %7d  fps %3d  location %5d %5d %5d",
//            level, timer_tick, (int)fps, (int)mx, (int)my, (int)mz);
      sprintf(buf, "level %2d  Score %8d", level, score);
      pix_disp_str(buf, 0, MAXY - 8, 31, 0);

      if (god_mode)
      {
         pix_disp_str("Editing", NUMXPIXELS - 38, NUMYPIXELS - 8, 33 + frame, 0);
      }

      if (last_clock_tick != clock_tick)
      {
         last_clock_tick = clock_tick;
      }
      if (clock_tick > 16)
      {
         clock_tick = 0;
         fps = frame;
         frame = 0;
      }
      //***************
      // COMPASS
      //***************
      if (show_compass)
      {
         for (i=0; i<3; i++)
         {
            x = ocx[i]; y = ocy[i]; z = ocz[i];
            xa = fixmul(cos_yaw, x) - fixmul(sin_yaw, z);
            za = fixmul(sin_yaw, x) + fixmul(cos_yaw, z);
            ya = -fixmul(sin_pitch, za) - fixmul(cos_pitch, y);
            cx = fixmul(cos_roll, xa) + fixmul(sin_roll, ya) + inttofix(250);
            cy = fixmul(cos_roll, ya) - fixmul(sin_roll, xa) + inttofix(70);
            pix_lineclip(250, 70, fixtoint(cx), fixtoint(cy), 120 + i);
            if (i == 0) *buf = 'x';
            else if (i == 1) *buf = 'y';
            else *buf = 'z';
            pix_disp_char(*buf, fixtoint(cx), fixtoint(cy), 255, 0);
         }
      }
      pix_refresh();
      // ***************************************************************
      // * FORWARD/BACKWARD MOMENTUM
      // ***************************************************************
      if (keys[DOWN])
      {
         if (speed > -max_momentum) speed -= momentum;
      }
      else if (keys[UP])
      {
         if (speed < max_momentum) speed += momentum;
      }
      else if (speed > 0)
      {
         speed -= momentum; if (speed < 0) speed = 0;
      }
      else if (speed < 0)
      {
         speed += momentum; if (speed > 0) speed = 0;
      }

      //***************************************************************
      // * FORWARD/BACKWARD
      // ***************************************************************
       temp1 = mx; temp2 = mz;
       mx += speed * sin_yaw * (16.0/fps);
       mz += speed * cos_yaw * (16.0/fps);
       if ((rc = collision(-mx, -my, -mz, max_momentum/2, -1)) >= 0 && !no_gravity)
       {
//          sprintf(msgbuf, "collision with %d type %d", rc, get_obj_type(rc));
//          msg = msgbuf;
//          msg_timer = 200;
          if (get_obj_type(rc) == 3)
          {
               the_sphere = rc;
               next_level();
               up_accel = 0;
               yaw_turn = 0;
               pitch_turn = 0;
               speed = 0;
               if (quit) break;
               continue;
          }
          else if (!god_mode && get_obj_type(rc) == 20)
          {
             score++;
             del_obj(get_obj_id(rc));
             play_sound("gem.wav", 25);
             continue;
          }
          mx = temp1;
          mz = temp2;
          speed = -(speed);
          play_sound("land.wav", 25);
       }
       else if (mx > inttofix(1000) || mz > inttofix(1000)
            || mx < inttofix(-1000) || mz < inttofix(-1000))
      {
         keys[DOWN] = 0;
         speed = 0;
         mx = temp1;
         mz = temp2;
      }

      //***************************************************************
      // * TURN MOMENTUM
      // ***************************************************************
      if (forced_turn)
      {
         yaw_turn = turn_accel * -2;
      }
      if (keys[RIGHT])
      {
         if (yaw_turn > -max_turn) yaw_turn -= turn_accel;
      }
      else if (keys[LEFT])
      {
         if (yaw_turn < max_turn) yaw_turn += turn_accel;
      }
      else if (yaw_turn > 0)
      {
         yaw_turn -= turn_accel; if (yaw_turn < 0) yaw_turn = 0;
      }
      else if (yaw_turn < 0)
      {
         yaw_turn += turn_accel; if (yaw_turn > 0) yaw_turn = 0;
      }
      if (keys[PGDN])
      {
         if (pitch_turn > -max_turn) pitch_turn -= turn_accel;
      }
      else if (keys[PGUP])
      {
         if (pitch_turn < max_turn) pitch_turn += turn_accel;
      }
      else if (pitch_turn > 0)
      {
         pitch_turn -= turn_accel; if (pitch_turn < 0) pitch_turn = 0;
      }
      else if (pitch_turn < 0)
      {
         pitch_turn += turn_accel; if (pitch_turn > 0) pitch_turn = 0;
      }

      //***************************************************************
      // * TURN YAW, PITCH, ROLL
      // ***************************************************************
      if (pitch_turn)
      {
            pitch += pitch_turn;
            if (pitch > dbltofix(6.28319)) pitch -= dbltofix(6.28319);
            else if (pitch < 0) pitch += dbltofix(6.28319);
            sin_pitch = dbltofix(sin(fixtodbl(pitch)));
            cos_pitch = dbltofix(cos(fixtodbl(pitch)));
      }
      if (yaw_turn)
      {
            yaw += yaw_turn * (16.0/fps);
            if (yaw > dbltofix(6.28319)) yaw -= dbltofix(6.28319);
            else if (yaw < 0) yaw += dbltofix(6.28319);
            sin_yaw = sin(yaw);
            cos_yaw = cos(yaw);
      }

      //***************************************************************
      // * GRAVITY
      // ***************************************************************
      if (god_mode && no_gravity)
      {
         if (keys[KB_U])
         {
            my -= jump_accel *  (16.0 / fps);
         }
         if (keys[KB_D])
         {
            my += jump_accel *  (16.0 / fps);
         }
      }
      else
      {
         temp1 = my;
         my -= up_accel * (16.0/fps);
         if ((rc = collision(-mx, -my, -mz, max_momentum/2 + 1, -1)) >= 0
            || my >= ground_my)
         {
             if (rc >= 0 && get_obj_type(rc) == 3)
             {
                  the_sphere = rc;
                  next_level();
                  if (quit) break;
                  continue;
             }
             else if (!god_mode && get_obj_type(rc) == 20)
             {
               score++;
               del_obj(rc);
               play_sound("gem.wav", 25);
               continue;
            }
            last_sitting_on = sitting_on;
            if (up_accel != 0) sitting_on = 0;
            if (temp1 != my && up_accel != 0)
            {
                  if (rc > 0 && get_obj_type(rc) == 2)
                  {
                     if (do_switch(rc)) play_sound("toggle.wav", 25);
                     else {
                        if (last_sitting_on != get_obj_id(rc))
                           play_sound("land.wav", 25);
                     }
                  }
                  else
                  {
                     if (last_sitting_on != get_obj_id(rc))
                        play_sound("land.wav", 25);
                  }
                  sitting_on = get_obj_id(rc);
            }
            if (up_accel < 0)
            {
               if (rc >= 0 && -my > get_obj_midy(rc))
               {
                  my = -(get_obj_midy(rc) + get_obj_radius(rc) + max_momentum/2 + 1);
               }
               else my = ground_my;
               up_accel = 0;
            }
            else up_accel = -up_accel;
            if ((keys[SPACE]) && up_accel == 0)
            {
               play_sound("jump.wav", 25);
               up_accel = jump_accel;
               sitting_on = 0;
            }
         }
         else
         {
            if ((keys[SPACE]) && up_accel > 0) up_accel += extra_jump * (16.0/fps);
            if (my < ground_my)
            {
               up_accel -= jump_decel * (16.0/fps);
            }
            else
            {
               up_accel = 0;
            }
         }
      }

      if (keys[ESC])
      {
            if (escape()) keys[ESC] = 1; else keys[ESC] = 0;
      }
      if (god_mode && keys[KB_G])
      {
         if (no_gravity) no_gravity = 0; else no_gravity = 1;
         keys[KB_G] = 0;
      }
      if (keys[KB_C])
      {
         show_compass = !show_compass;
         keys[KB_C] = 0;
      }
      if (keys[KB_I])
      {
         keys[KB_I] = 0;
         info();
      }
      if (keys[TAB] && god_mode)
      {
         keys[TAB] = 0;
         id = locate_obj(160, 120);
         if (id >= 0)
         {
           if (is_obj_marked(id)) unmark_obj(id);
            else { mark_obj(id); manipulate(id); unmark_obj(id); }
         }
      }
      if (keys[KB_O])
      {
         keys[KB_O] = 0;
         options();
      }
      if (keys[KB_A] && god_mode)
      {
         keys[KB_A] = 0;
         add_it(-(mx + fixmul(arms_length, sin_yaw)), -my,
                     -(mz + fixmul(arms_length, cos_yaw)));
      }
      if (keys[KB_S] && god_mode)
      {
         if (save()) save_area(level);
         keys[KB_S] = 0;
         keys[ESC] = 0;
      }
      if (keys[KB_L])
      {
         show_location = !show_location;
         keys[KB_L] = 0;
      }
      if (keys[F1])
      {
         keys[F1] = 0;
         help();
      }
      if (keys[PLUS])
      {
         if (CLIPMAXY < MAXY) { CLIPMAXX++; CLIPMAXY++; }
         precomp_frustrum();
      }
      else if (keys[MINUS])
      {
         if (CLIPMAXY > 10) { CLIPMAXY--; CLIPMAXX--; }
         precomp_frustrum();
      }
   } while (!keys[ESC] && !quit);
}

void main(int argc, char *argv[])
{
   const fixed extra_jump = dbltofix(.3);
   int id;
   fixed xa, ya, za, x, y, z;
   int show_compass = 0;
   int i, rc;
   float pal_level = 0.0;
   float pal_inc = 0.1;
   fixed cx, cy, cz;
   fixed ocx[3], ocy[3], ocz[3];

   unsigned int last_clock_tick;
   fixed up_accel = inttofix(0);
   fixed yaw_turn = 0, pitch_turn = 0;
   fixed speed = inttofix(0);

   fixed temp1, temp2, temp3;
   const fixed arms_length = inttofix(200);
   int frame = 0, show_location = 0;
   char buf[256];
   int key;
   int inst_n = 6;

   ocx[0] = inttofix(40);  ocy[0] = 0;  ocz[0] = 0;
   ocx[1] = 0;  ocy[1] = inttofix(40);  ocz[1] = 0;
   ocx[2] = 0;  ocy[2] = 0;  ocz[2] = inttofix(40);

   msg = "Press F1 for help at any time.  Good luck!";
   msg_timer = 5 * 16;

   VIDMODE = _MRES256COLOR;
   MAXX = 319;
   MAXY = 199;
   CLIPMAXX = 319;
   CLIPMAXY = 199;

//   signal(SIGFPE, SIG_IGN);
//   signal(SIGSEGV, SIG_IGN);
   

   for (i=1; i<argc; i++)
   {
      if (argv[i][0] == '-')
      {
         if (strcmp(argv[i], "-lowres") == 0)
         {
            VIDMODE = _MRES256COLOR;
            CLIPMAXX = MAXX = 319;
            CLIPMAXY = MAXY = 199;
         }
         else if (strcmp(argv[i], "-mediumres") == 0)
         {
            VIDMODE = _URES256COLOR;
            CLIPMAXX = MAXX = 639;
            CLIPMAXY = MAXY = 399;
         }
         else if (strcmp(argv[i], "-highres") == 0)
         {
            VIDMODE = _XRES256COLOR;
            CLIPMAXX = MAXX = 1023;
            CLIPMAXY = MAXY = 767;
         }
         else if (strcmp(argv[i], "-ccpp") == 0)
         {
            do_ccpp = 1;
         }
         else if (strcmp(argv[i], "-nosound") == 0)
         {
            no_sound = 1;
            no_music = 1;
         }
         else if (strcmp(argv[i], "-nointro") == 0)
         {
            no_intro = 1;
         }
         else if (strcmp(argv[i], "-help") == 0)
         {
printf("ntech [-fastest] [-test] [-nomusic] [-nosound] [-notextures] [-nostars] [-edit]\n");
exit(0);
         }
         else if (strcmp(argv[i], "-fastest") == 0)
         {
            no_music = 1;
            no_perspective_correct = 1;
            no_sound = 1;
            no_textures = 1;
            no_stars = 1;
            VIDMODE = _MRES256COLOR;
            MAXX = 319;
            MAXY = 199;
            CLIPMAXX = 319;
            CLIPMAXY = 199;
         }
         else if (strcmp(argv[i], "-time") == 0)
         {
            do_time = 1;
            no_sound = 1;
            no_music = 1;
            no_intro = 1;
            no_stars = 1;
            show_location = 1;
         }
         else if (strcmp(argv[i], "-test") == 0)
         {
            do_test = 1;
            no_sound = 1;
            no_music = 1;
         }
         else if (strcmp(argv[i], "-noperspective") == 0)
         {
            no_perspective_correct = 1;
         }
         else if (strcmp(argv[i], "-nomusic") == 0)
         {
            no_music = 1;
         }
         else if (strcmp(argv[i], "-nostars") == 0)
         {
            no_stars = 1;
         }
         else if (strcmp(argv[i], "-notextures") == 0)
         {
            no_textures = 1;
         }
         else if (strcmp(argv[i], "-edit") == 0)
         {
            god_mode = 1;
            no_gravity = 1;
            no_sound = 1;
            no_music = 1;
            no_intro = 1;
            no_stars = 1;
         }
      }
      else level = atoi(argv[i]);
   }
            MIDX = MAXX/2;
            MIDY = MAXY/2; 
            NUMXPIXELS = MAXX + 1;
            NUMYPIXELS = MAXY + 1;


/*
   while (!keys[ESC])
   {
      for (i=0; i<256; i++)
      {
         if (keys[i] && i != 132 && i != 140 && (i < 144 || i > 151)) printf("%03d ", i);
      }
   }
   UninstallKey();
   exit(0);
*/

   printf("NanoTech is Copyright (C) 1996 Sean Lane Fuller\n");
   printf("NanoTech comes with ABSOLUTELY NO WARRANTY; for details\n");
   printf("see the file COPYING.txt.  This is free software, and you\n");
   printf("are welcome to redistribute it under the conditions\n");
   printf("stated in the README.txt file.\n\n");
   printf("Varmit's Audio Tools by Eric Jorgensen, PMODE Bryan Wilkins ...");
   start_sound();
   printf("\n");
   printf("running shaders to initialize textures ...");
   init_textures();
   printf("\n");
   printf("precalcing 3d lookup tables and constants ...");
   fflush(stdout);
   init_3d();
   printf("\n");
   printf("initializing S-Buffer ...");
   fflush(stdout);
   init_sbuffer();
   printf("\n");
   printf("initializing timer ISR ...");
   fflush(stdout);
   Init_Timer();
   printf("\n");
   printf("initializing scene database ...");
   fflush(stdout);
   clear_scene();
   printf("\n");
   printf("initializing keyboard ISR ...");
   fflush(stdout);
   InitKey();
   printf("\n");
   printf("Setting graphics mode ...");
   fflush(stdout);
   pix_init();
   printf("\n");
   last_id = 0;
   if (!(do_test) && !no_intro) intro(1); 
   last_id = num_objs;
   init_viewer();
   make_scene();
   draw_scene();
   pix_refresh();
   keys[ESC] = 0;
   if (do_test) { intro(1); goto OUT; }
   load_area(level);
   init_textures();
   new_song("tunnel.mod");
   if (!god_mode) sound_intro();
   load_mod();
   observe();
OUT:
   init_viewer();
   term_3d();
   term_sbuffer();
   pix_term();
   Close_Timer();
   UninstallKey();
   stop_sound();

printf("\n");
printf("\n");
printf("    \n");
printf("                                                                        \n");
printf("   NanoTech is Copyright (C) 1996  Sean Lane Fuller                     \n");
printf("                                                                        \n");
printf("   NanoTech comes with ABSOLUTELY NO WARRANTY; for details see the file \n");
printf("   COPYING.txt.  This is free software, and you are welcome to          \n");
printf("   redistribute it under the conditions stated in the README.txt file.  \n");
printf("                                                                        \n");
printf("   Thanks to Eric Jorgensen & Bryan Wilkins for Varmint's Audio Tools.  \n");
printf("   Music written using Sami Tammilethto's Digiplayer & Scream Tracker.  \n");
printf("                                                                        \n");
printf("   Sean Lane Fuller      Nanotech Home Page:                            \n");
printf("   124 Autumn Lane          http://edge.edge.net/~fuller/nanotech.htm   \n");
printf("   Tullahoma, TN 37388                                                  \n");
printf("   fuller@edge.net                                                      \n");
printf("                                                                        \n");
printf("  \n");
printf("   \n");

}
