/* ========================================================================= */
/*  GBDIS 1.0                                                                */
/* ========================================================================= */
/*  A more complex disassembler for GB.                                      */
/*  I included a simple "code-analyser" to seperate code and data chunks     */
/* ========================================================================= */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "..\nintendo\nintendo.h"

typedef unsigned char uchar;
typedef unsigned short ushort;
typedef unsigned long ulong;
typedef unsigned int uint;
typedef char bool;

#define TRUE 1
#define FALSE 0

char *cOpNames[]=
{"adc",  "add", "and", "bit", "call", "ccf",  "cmp", "cmp",
 "cpl",  "daa", "dec", "di",  "ei",   "halt", "inc", "jp",
 "ld",   "ldi", "ldd", "nop", "or",   "pop",  "push","res",
 "ret",  "reti","rl",  "rlc", "rol",  "ror",  "rr",  "rrc",
 "rst",  "sbb", "sbc", "scf", "set",  "sla",  "sra", "srl",
 "shl",  "shr", "swap","sub", "xor"};

uchar ucOpConfig[][2] = /* len,mask */
{{ 1, 0377 }, /*NO      */
 { 1, 0300 }, /*RR      */
 { 2, 0307 }, /*RN      */
 { 1, 0377 }, /*AIBC    */
 { 1, 0377 }, /*AIDE    */
 { 3, 0377 }, /*AINN    */
 { 2, 0377 }, /*IHA     */
 { 2, 0377 }, /*AIH     */
 { 1, 0377 }, /*ICA     */
 { 1, 0377 }, /*AIC     */
 { 1, 0377 }, /*IBCA    */
 { 1, 0377 }, /*IDEA    */
 { 3, 0377 }, /*INNA    */
 { 3, 0317 }, /*DDNN    */
 { 3, 0377 }, /*HLINN   */
 { 3, 0377 }, /*INNHL   */
 { 1, 0317 }, /*QQ      */
 { 1, 0377 }, /*SPHL    */
 { 1, 0370 }, /*AR      */
 { 2, 0377 }, /*AN      */
 { 1, 0307 }, /*R       */
 { 1, 0317 }, /*HLSS    */
 { 1, 0317 }, /*SS      */
 { 3, 0377 }, /*NN      */
 { 3, 0347 }, /*CCNN    */
 { 1, 0377 }, /*IHL     */
 { 1, 0377 }, /*A       */
 { 1, 0347 }, /*CC      */
 { 1, 0307 }, /*T       */
 { 1, 0377 }, /*IHLA    */
 { 1, 0377 }, /*AIHL    */
 { 2, 0377 }, /*REN     */
 { 2, 0347 }, /*CCREN   */
 { 2, 0377 }, /*SPN	*/
 { 3, 0377 }, /*INNSP   */
 { 2, 0377 }, /*HLSPN   */
 { 2, 0300 }}; /*BR      */

/* Adressing modes */
#define NO      0
#define RR      1
#define RN      2
#define AIBC    3
#define AIDE    4
#define AINN    5
#define IHHA    6
#define AIHH    7
#define ICA     8
#define AIC     9
#define IBCA    10
#define IDEA    11
#define INNA    12
#define DDNN    13
#define HLINN   14
#define INNHL   15
#define QQ      16
#define SPHL    17
#define AR      18
#define AN      19
#define R       20
#define HLSS    21
#define SS      22
#define NN      23
#define CCNN    24
#define IHL     25
#define A       26
#define CC      27
#define T       28
#define IHLA    29
#define AIHL    30
#define REN	31
#define CCREN	32
#define SPN	33
#define INNSP   35
#define HLSPN   36
#define BR	37

#define ENDOFLIST 0xFF

uchar ucOpTable[][3] =
/* Base/Param/Name */
{{0000,NO,19},       /* NOP          */
 {0x27,NO,9},        /* DAA          */
 {0x2F,NO,8},        /* CPL          */
 {0x3F,NO,5},        /* CCF          */
 {0x37,NO,35},       /* SCF          */
 {0xF3,NO,11},       /* DI           */
 {0xFB,NO,12},       /* EI           */
 {0x76,NO,13},       /* HALT         */
 {0x40,RR,16},       /* LD R,R       */
 {0x06,RN,16},       /* LD R,N       */
 {0x0A,AIBC,16},     /* LD A,(BC)    */
 {0x1A,AIDE,16},     /* LD A,(DE)    */
 {0xFA,AINN,16},     /* LD A,(NN)    */
 {0340,IHHA,16},     /* LD (HH),A    */
 {0360,AIHH,16},     /* LD A,(HH)    */
 {0342,ICA,16},      /* LD (C),A     */
 {0362,AIC,16},      /* LD A,(C)     */
 {0x02,IBCA,16},     /* LD (BC),A    */
 {0x12,IDEA,16},     /* LD (DE),A    */
 {0xEA,INNA,16},     /* LD (NN),A    */
 {0x01,DDNN,16},     /* LD DD,NN     */
 {0x2a,HLINN,16},    /* LD HL,(NN)   */
 {0x22,INNHL,16},    /* LD (NN),HL   */
 {0xF9,SPHL,16},     /* LD SP,HL     */
 {0305,QQ,22},       /* PUSH QQ      */
 {0301,QQ,21},       /* POP QQ       */
 {0210,AR,0},        /* ADC A,R      */
 {0316,AN,0},        /* ADC A,N      */
 {0200,AR,1},        /* ADD A,R      */
 {0306,AN,1},        /* ADD A,N      */
 {0240,AR,2},        /* AND A,R      */
 {0346,AN,2},        /* AND A,N      */
 {0260,AR,20},       /* OR A,R       */
 {0366,AN,20},       /* OR A,N       */
 {0220,AR,43},       /* SUB A,R      */
 {0326,AN,43},       /* SUB A,N      */
 {0230,AR,34},       /* SBC A,R      */
 {0336,AN,34},       /* SBC A,N      */
 {0250,AR,44},       /* XOR A,R      */
 {0356,AN,44},       /* XOR A,N      */
 {0270,AR,7},        /* CMP A,R      */
 {0376,AN,7},        /* CMP A,N      */
 {0004,R,14},        /* INC R        */
 {0005,R,10},        /* DEC R        */
 {0011,HLSS,1},      /* ADD HL,SS    */
 {0003,SS,14},       /* INC SS       */
 {0013,SS,10},       /* DEC SS       */
 {0303,NN,15},       /* JP NN        */
 {0302,CCNN,15},     /* JP CC,NN     */
 {0030,REN,15},      /* JR RN        */
 {0040,CCREN,15},    /* JR CC,RN     */
 {0351,IHL,15},      /* JP (HL)      */
 {0007,A,27},        /* RLC A        */
 {0027,A,26},        /* RL A         */
 {0017,A,31},        /* RRC A        */
 {0037,A,30},        /* RR A         */
 {0315,NN,4},        /* CALL NN      */
 {0304,CCNN,4},      /* CALL CC,NN   */
 {0311,NO,24},       /* RET          */
 {0300,CC,24},       /* RET CC       */
 {0331,NO,25},       /* RETI         */
 {0307,T,32},        /* RST t        */
 {0062,IHLA,18},     /* LDD (HL),A   */
 {0042,IHLA,17},     /* LDI (HL),A   */
 {0074,AIHL,18},     /* LDD A,(HL)   */
 {0064,AIHL,17},     /* LDI A,(HL)   */
 {0xE8,SPN,1},	     /* ADD SP,N     */
 {0x08,INNSP,16},    /* LD (NN),SP   */
 {0xF8,HLSPN,16},    /* LD HL,SP+N   */
 {0000,ENDOFLIST}};

uchar ucOpCBTable[][3] =
{{0100,BR,3},        /* BIT B,R      */
 {0200,BR,23},       /* RES B,R      */
 {0300,BR,36},       /* SET B,R      */
 {0000,ENDOFLIST}};

int error;

/* misc. */

#define _GETWORD(x) (((uchar)*(x))|((uchar)*((x)+1)<<8))

ulong eval (char *s)
{
  ulong x;char *p;char v[32];
  x=0;strupr(s);
  if ((p=strchr(s,'H'))!=NULL) {
    *p=0;
    sprintf(v,"0X%s",s);flushall();
    sscanf(v,"%X",&x);
  } else {
    if ((p=strchr(s,'$'))!=NULL) {
      strcpy(p,p+1);
      sprintf(v,"0X%s",s);flushall();
      sscanf(v,"%X",&x);
    } else sscanf(s,"%u",&x);
  }
  return x;
}

/* label management */

typedef struct _LABEL {
  char *name;
  char ltype;
  ushort adress;
  ushort size;
} LABEL;

ulong labelnum,labelmax;
LABEL *(labels);

void label_init()
{
  labelmax=512;labelnum=0;
  labels=(LABEL *)malloc(labelmax*sizeof(LABEL));
}

void label_done()
{
  free(labels);
}

void label_store(char *name,char ltype,ushort adress,ushort size)
{
  if (labelnum==labelmax) {
    labelmax+=512;
    labels=(LABEL *)realloc(labels,labelmax*sizeof(LABEL));
  }
  if (labels!=NULL) {
    if ((labels[labelnum].name=malloc(strlen(name)+1))!=NULL) {
      labels[labelnum].ltype=ltype;
      labels[labelnum].adress=adress;
      labels[labelnum].size=size;
      strcpy(labels[labelnum].name,name);
      labelnum++;
    } else {
      fprintf(stderr,"Out of memory !\n");
      error++;
    }
  } else {
    fprintf(stderr,"Out of memory !\n");
    error++;
  }
}

#define _LABEL   0
#define _JUMP    1
#define _HEADER  2
#define _JTABLE  3
#define _LTABLE  4
#define _CODEBLK 5

int label_find(ushort adr)
{
  int c;
  for (c=0;c<labelnum;c++) {
    if ((labels[c].ltype!=_CODEBLK)&&(adr>=labels[c].adress)&&(adr<(ulong)(((ulong)labels[c].adress+labels[c].size)))) return c;
  }
  return -1;
}

void label_read(char *s)
{
  FILE *f;
  if ((f=fopen(s,"rt"))!=NULL) {
    int cnt;
    cnt=0;
    while (!feof(f)) {
      char line[256];
      char *p;
      fgets(line,sizeof(line),f);cnt++;
      if ((p=strchr(line,';'))!=NULL) *p=0;
      if ((p=strchr(line,'\n'))!=NULL) *p=0;
      while ((p=strstr(line,"  "))!=NULL) strcpy(p,p+1);
      while (line[strlen(line)]==32) line[strlen(line)]=0;
      while (line[0]==32) strcpy(&line[0],&line[1]);
      if (line[0]!=0) {
        char name[32];char substr[32];
        char ltype;
        ushort adress;
        ushort size;
        size=1;
        strncpy(substr,line,min(strchr(line,32)-&line[0],32));
        substr[min(strchr(line,32)-&line[0],32)]=0;
        strupr(substr);
        if (strcmp(".LABEL",substr)==0) ltype=_LABEL; else
        if (strcmp(".JUMP",substr)==0) ltype=_JUMP; else
        if (strcmp(".HEADER",substr)==0) ltype=_HEADER; else
        if (strcmp(".JTABLE",substr)==0) ltype=_JTABLE; else
        if (strcmp(".LTABLE",substr)==0) ltype=_LTABLE; else {
          fprintf(stderr,"%s/%u:unknown type %s ignored !\n",s,cnt,substr);
          ltype=-1;
        }
        if (ltype>=0) {
          char *r;
          r=strchr(line,32)+1;
          while ((p=strchr(r,32))!=NULL) strcpy(p,p+1);
          if (ltype==_HEADER) {
            strcpy(name,"HEADER");
            adress=eval(r);
            size=0x50;
            label_store(name,ltype,adress,size);
          } else {
            if ((p=strchr(r,','))!=NULL) {
              strncpy(name,r,min(p-r,32));
              name[min(p-r,32)]=0;
              r=p+1;
              if ((p=strchr(r,','))==NULL) p=strlen(r)+r;
              strncpy(substr,r,min(p-r,32));
              substr[min(p-r,32)]=0;r=p;
              adress=eval(substr);
              if (*p!=0) {
                strncpy(substr,r+1,32);
                substr[min(strlen(p+1),32)]=0;
                size=eval(substr);
              }
              if (size==0) size++;
              label_store(name,ltype,adress,size);
            } else {
              fprintf(stderr,"%s/%u:name expected !\n",s,cnt);
              error++;
            }
          }
        }
      }
    }
  } else {
    fprintf(stderr,"%s not found !\n",s);
    error++;
  }
}

void label_sort()
{
  int c;
  bool done;
  LABEL swap;
  do {
    done=TRUE;
    for (c=0;c<(labelnum-1);c++) {
      if ((labels[c].adress>labels[c+1].adress)||((labels[c].ltype>labels[c+1].ltype)&&((labels[c].adress==labels[c+1].adress)))) {
        memcpy((void *)&swap,(void *)(labels+c),sizeof(LABEL));
	memcpy((void *)(labels+c),(void *)(labels+c+1),sizeof(LABEL));
	memcpy((void *)(labels+c+1),(void *)&swap,sizeof(LABEL));
        done=FALSE;
      }
    }
  } while (!done);
}

/* disassembler */

void fline(FILE *stream)
{
  fprintf(stream,"; ---------------------------------------------------------------------------\n");
}

int findcode(uchar code)
{
  int c;
  for (c=0;ucOpTable[c][1]!=ENDOFLIST;c++)
    if ((code & ucOpConfig[ucOpTable[c][1]][1])==ucOpTable[c][0]) return c;
  return -1;
}

int findcodeCB(uchar code)
{
  int c;
  for (c=0;ucOpCBTable[c][1]!=ENDOFLIST;c++)
    if ((code & ucOpConfig[ucOpCBTable[c][1]][1])==ucOpCBTable[c][0]) return c;
  return -1;
}

int opsize(uchar code)
{
  int c;
  if (code==0xCB) return 2;
  c=findcode(code);
  return (c<0)?1:ucOpConfig[ucOpTable[c][1]][0];
}

char *_r[] = { "b","c","d","e","h","l","(hl)","a" };
char *_qq[] = { "bc","de","hl","af" };
char *_dd[] = { "bc","de","hl","sp" };
char *_cc[] = { "nz","z","nc","c" };
#define _ss _dd

char *gl(ushort adr,bool j, char *buffer)
{
  int c;
  if ((c=label_find(adr))>=0) {
    if (!j) {
      if ((adr-labels[c].adress)>0)
        sprintf(buffer,"%s+%u",labels[c].name,adr-labels[c].adress);
      else
        sprintf(buffer,"%s",labels[c].name);
      return buffer;
    } else
      if (adr==labels[c].adress)
      {
        sprintf(buffer,"%s",labels[c].name);
        return buffer;
      }
  }
  sprintf(buffer,"$%4.4X",adr);
  return buffer;
}

bool fprintasm(FILE *stream,char *data,int pos)
{
  int c;
  ushort _nn;uchar _n;ushort _rn;uchar _m;
  char nl[32];
  char params[80];
  if ((uchar)*(data+pos)!=0xCB)
    c=findcode(*(data+pos));
  else {
    c=findcodeCB(*(data+pos+1));
    if (c>=0) c+=512;
  }
  if (c<0) {
    fprintf(stream, "%-39s", "$$$");
    return FALSE;
  } else {
    if (c<512)
      if (ucOpConfig[ucOpTable[c][1]][0]>1) {
        _n=*(data+pos+1);
        _nn=((unsigned)((uchar)*(data+pos+2))<<8)|((uchar)*(data+pos+1));
        _rn=2+pos+(signed char)*(data+pos+1);
      }
    strcpy(params,"");
    if (c<512)
      _m=ucOpTable[c][1];
    else
      _m=ucOpCBTable[c-512][1];
    switch (_m) {
      case RR      : sprintf(params, " %s,%s",_r[*(data+pos)>>3&07],_r[*(data+pos)&07]); break;
      case RN      : sprintf(params, " %s,$%2.2X",_r[*(data+pos)>>3&07],_n); break;
      case AIBC    : sprintf(params, " a,(bc)"); break;
      case AIDE    : sprintf(params, " a,(de)"); break;
      case AINN    : sprintf(params, " a,(%s)",gl(_nn,FALSE,nl)); break;
      case IHHA    : sprintf(params, " (%s),a",gl(0xFF00+_n,FALSE,nl)); break;
      case AIHH    : sprintf(params, " a,(%s)",gl(0xFF00+_n,FALSE,nl)); break;
      case ICA     : sprintf(params, " (c),a"); break;
      case AIC     : sprintf(params, " a,(c)"); break;
      case IBCA    : sprintf(params, " (bc),a"); break;
      case IDEA    : sprintf(params, " (de),a"); break;
      case INNA    : sprintf(params, " (%s),a",gl(_nn,FALSE,nl)); break;
      case DDNN    : sprintf(params, " %s,%s",_ss[*(data+pos)>>4&03],gl(_nn,FALSE,nl)); break;
      case HLINN   : sprintf(params, " hl,(%s)",gl(_nn,FALSE,nl)); break;
      case INNHL   : sprintf(params, " (%s),hl",gl(_nn,FALSE,nl)); break;
      case QQ      : sprintf(params, " %s",_qq[*(data+pos)>>4&03]); break;
      case SPHL    : sprintf(params, " sp,hl"); break;
      case AR      : sprintf(params, " a,%s",_r[*(data+pos)&07]); break;
      case AN      : sprintf(params, " a,$%2.2X",_n); break;
      case R       : sprintf(params, " %s",_r[*(data+pos)>>3&07]); break;
      case HLSS    : sprintf(params, " hl,%s",_ss[*(data+pos)>>4&03]); break;
      case SS      : sprintf(params, " %s",_ss[*(data+pos)>>4&03]); break;
      case NN      : sprintf(params, " %s",gl(_nn,TRUE,nl)); break;
      case CCNN    : sprintf(params, " %s,%s",_cc[*(data+pos)>>3&03],gl(_nn,TRUE,nl)); break;
      case REN     : sprintf(params, " %s",gl(_rn,TRUE,nl)); break;
      case CCREN   : sprintf(params, " %s,%s",_cc[*(data+pos)>>3&03],gl(_rn,TRUE,nl)); break;
      case IHL     : sprintf(params, " (hl)"); break;
      case A       : sprintf(params, " A"); break;
      case CC      : sprintf(params, " %s",_cc[*(data+pos)>>3&03]); break;
      case T       : sprintf(params, " %s",gl(*(data+pos)&070,TRUE,nl)); break;
      case IHLA    : sprintf(params, " (hl),a"); break;
      case AIHL    : sprintf(params, " a,(hl)"); break;
      case SPN     : sprintf(params, " sp+%s",gl(_n,FALSE,nl)); break;
      case INNSP   : sprintf(params, " (%s),sp",gl(_nn,FALSE,nl)); break;
      case HLSPN   : sprintf(params, " hl,sp+%s",gl(_n,FALSE,nl)); break;
      case BR      : sprintf(params, " %u,%s",(*(data+pos+1)>>3)&07,_r[*(data+pos)&07]); break;
    }
    if (c<512)
      fprintf(stream, "%-8s %-30s",cOpNames[ucOpTable[c][2]],params);
    else
      fprintf(stream, "%-8s %-30s",cOpNames[ucOpCBTable[c-512][2]],params);
    return TRUE;
  }
}

void fdisass(FILE *stream,char *data,int pos,int size)
{
  int n,n2,os,c;
  char label[32];

  fline(stream);
  while (size>0) {
    strcpy(label,"");
    if ((c=label_find(pos))>=0)
      if (labels[c].adress==pos) {
        strcpy(label,labels[c].name); strcat(label,":");
      }
    fprintf(stream,"%-12s ",label);
    fprintasm(stream,data,pos);
    fprintf(stream,"; %4.4X  ",pos);
    os=opsize(*(data+pos));
    for (n2=0;n2<os;n2++)
      fprintf(stream,"%2.2X ",(unsigned char)*(data+pos+n2));
    fprintf(stream,"\n");
    pos+=os;
    size-=os;
  }
}

void fdecodeheader(FILE *stream,char *data)
{
  int x,y;

  fline(stream);
  data-=4;
  fprintf(stream,"; title: %.15s\n",(data+0x34));
  fprintf(stream,"; version: %u\n",(unsigned char)*(data+0x4c));
  if ((unsigned char)*(data+1)==0xC3)
    fprintf(stream,"; start: 0%4.4Xh\n",(unsigned short)((unsigned char)*(data+2)|((unsigned char)*(data+3)<<8)));
  fprintf(stream,"; configuration: ");
  switch (*(data+0x47)) {
    case 0 : fprintf(stream,"ROM ONLY\n"); break;
    case 1 : fprintf(stream,"MBC1\n"); break;
    case 2 : fprintf(stream,"MBC1+RAM\n"); break;
    case 3 : fprintf(stream,"MBC1+RAM+BATTERY\n"); break;
    case 5 : fprintf(stream,"MBC2\n"); break;
    case 6 : fprintf(stream,"MBC2+BATTERY\n"); break;
    default: fprintf(stream,"unknown\n"); break;
  }
  fprintf(stream,"; ROM: %u kByte\n",0x20 << *(data+0x48));
  if (*(data+0x49)!=0)
    fprintf(stream,"; RAM: %u kByte\n",0x2 << ((*(data+0x49)-1)<<1));
  fprintf(stream,"; maker code:");
  if (*(data+0x4b)==51)
    printf("(%2.2s) %s\n",data+0x44,GetNameForID(data+0x44));
  else
    printf("(%2.2X) %s\n",(unsigned)*(data+0x4B),GetNameForNum(*(data+0x4B)));
  fprintf(stream,"; complement: %2.2X\n",(unsigned char)*(data+0x4d));
  fprintf(stream,"; checksum: %4.4X\n",(unsigned short)((unsigned char)*(data+0x4e)|((unsigned char)*(data+0x4f)<<8)));
  fline(stream);
  for (y=0;y<8;y++) {
    fprintf(stream,"; ");
    for (x=0;x<48;x++)
      if ((*(data+4+(y>>2)*24+((x>>1)&0xfe)+((y>>1)&1))<<((x&3)+((y&1)<<2)))&128)
	fprintf(stream,"#");
      else
	fprintf(stream," ");
    fprintf(stream,"\n");
  }
}

void fdump(FILE *stream,char *data,ushort adr,ushort size)
{
  int c;char nl[32];
  fline(stream);
  if (((c=label_find(adr))>=0)&&(labels[c].adress==adr)) printf("%s\n",labels[c].name);
  c=0;
  while (c<size) {
    int cc;
    fprintf(stream,".db ");
    for (cc=0;(cc<16)&&(c++<size);cc++)
      fprintf(stream," %2.2X%s",(uchar)*(data+adr+cc),((cc<15)&&(c<size))?(","):(" "));
    while ((cc++)<16)
      fprintf(stream,"    ");
    fprintf(stream,";  %4.4X\n",adr);
    adr+=16;
  }
}

void ftable(FILE *stream,char *data,ushort adr,ushort size)
{
  int c;char nl[32];
  for (c=0;c<size;c+=2)
    fprintf(stream,".DW %s\n",gl(_GETWORD(data+c+adr),FALSE,nl));
}

/***************************************************************************/
/* Codescanner                                                             */
/***************************************************************************/

bool checkjump(ushort adr)
{
  int c;
  for (c=0;c<labelnum;c++)
    if ((adr>=labels[c].adress)&&(adr<(ulong)(((ulong)labels[c].adress+labels[c].size)))&&(labels[c].ltype==_CODEBLK))
      return TRUE;
  return FALSE;
}

bool jumpexist(ushort adr)
{
  int c;
  for (c=0;c<labelnum;c++)
    if ((adr==labels[c].adress)&&(labels[c].ltype==_JUMP)) return TRUE;
  return FALSE;
}

void new_jump(ushort adr)
{
  char name[32];
  if ((!jumpexist(adr)) && (adr<0x8000)) {
    sprintf(name,"PROC_%4.4X",adr);
    label_store(name,_JUMP,adr,1);
  }
}

void new_segment(char *data,ushort adr)
{
  ushort size;
  ushort _nn,_rn,_n;
  bool brk;
  uchar code;
  size=0;
  do {
    brk=FALSE;
    code=*(data+adr+size);
    _nn=((unsigned)((uchar)*(data+adr+size+2))<<8)|((uchar)*(data+adr+size+1));
    _rn=2+adr+size+(signed char)*(data+adr+size+1);
    switch (code) {
      case 0xC3: new_jump(_nn);brk=TRUE;break;
      case 0x18: new_jump(_rn);brk=TRUE;break;
      case 0x38: new_jump(_rn);break;
      case 0x30: new_jump(_rn);break;
      case 0x28: new_jump(_rn);break;
      case 0x20: new_jump(_rn);break;
      case 0xC2: new_jump(_nn);break;
      case 0xCA: new_jump(_nn);break;
      case 0xD2: new_jump(_nn);break;
      case 0xDA: new_jump(_nn);break;
      case 0xC9: brk=TRUE;break;
      case 0xD9: brk=TRUE;break;
      case 0xE9: brk=TRUE;break;
      case 0xCD: new_jump(_nn);break;
      case 0xC4: new_jump(_nn);break;
      case 0xCC: new_jump(_nn);break;
      case 0xD4: new_jump(_nn);break;
      case 0xDC: new_jump(_nn);break;
    } 
    if ((code&0307)==0307) {
      new_jump(code & 0070);
      brk=TRUE;
    }
    size+=opsize(code);
    brk|=checkjump(adr+size);
  } while (!brk);
  label_store("CODESEG",_CODEBLK,adr,size);
/*
  fprintf(stdout,"codeseg %4.4X - %4.4X\n",adr,adr+size);
*/
}

void codescan(char *data)
{
  int c,cc;
  bool done;
  ushort num;

  done=FALSE;
  for (c=0;(c<labelnum)&&(error==0);c++)
    if (labels[c].ltype==_JTABLE) {
      num=0;
      while (num<labels[c].size) {
        ushort x;
        x=_GETWORD(data+labels[c].adress+num);
        if (!checkjump(x)) {
          new_jump(x);
          done=FALSE;
        }
        num+=2;
      }
    }
  while ((!done)&&(error==0)) {
    done=TRUE;
    printf(".");
    for (c=0;(c<labelnum)&&(error==0);c++)
      if (labels[c].ltype==_JUMP)
        if (!checkjump(labels[c].adress)) {
          new_segment(data,labels[c].adress);
          done=FALSE;
        }
    }
  printf("\n");
}

/***************************************************************************/

char *fname = "c:\\GBSYS\\GB\\TETRIS.GBS";

char *load_program(char *s)
{
  char *data;
  FILE *f;
  size_t fsize;

  if ((f=fopen(s,"rb"))!=NULL)
  {
    /* Read the data */
    fseek(f,0,SEEK_END);
    fsize=ftell(f);
    fseek(f,0,SEEK_SET);
    if ((data=malloc(fsize))!=NULL)
      fread(data,fsize,1,f);
    fclose(f);
  } else
    data=NULL;
  return data;
}

void fwritedata(FILE *stream,char *data)
{
  int c;ushort pos;
  pos=0;
  for (c=0;(c<labelnum)&&(pos<0x8000)&&(error==0);c++) {
    while ((labels[c].adress<pos)&&(c<labelnum)) c++;
    if (labels[c].adress>pos) fdump(stream,data,pos,labels[c].adress-pos);
    pos=labels[c].adress;
/*  fprintf(stderr,"%s\n",labels[c].name); */
    switch (labels[c].ltype) {
      case _HEADER : fdecodeheader(stdout,data+labels[c].adress);pos+=labels[c].size;break;
      case _CODEBLK : fdisass(stdout,data,labels[c].adress,labels[c].size);pos+=labels[c].size;break;
      case _LTABLE :
      case _JTABLE : ftable(stdout,data,labels[c].adress,labels[c].size);pos+=labels[c].size;break;
    }
  }
  fline(stdout);
}

#pragma argsused
int main(int argc,char *argv[])
{
  char *data;

  error=0;

  label_init();
  label_read("GAMEBOY.LAB");
  label_read("TETRIS.LAB");

  data=load_program(fname);

  printf("analysing code ");
  codescan(data);

  label_sort();

  fline(stdout);
  fprintf(stdout,"; created with GBDIS V1.0");
  fprintf(stdout,"; file: %s\n",fname);

  fwritedata(stdout,data);

  label_done();

  if (error>0) fprintf(stderr,"%u errors",error);

  return 0;
}
