#include <time.h>
#include <stdarg.h>

#include "defines.hpp"
#include "input.hpp"
#include "video.hpp"
#include "mesh.hpp"
#include "file.hpp"

// Copyright (c) 1996 by Kerrigan Burgess, all rights reserved.

INPUTCLASS      *Input;
VIDEOCLASS      *Video;
MESHCLASS       *Mesh;
FILESYSTEMCLASS *File;
    
void Error(char *fmt, ...);        // Always put.

void Build(const int Device, int Mode);       // Your own functions.
void TearDown(void);
void Handle3DInput(void);
void SetupCFG(char *filename);


void main(int argc, char **argv)
{
   if (argc < 2)
     cout<<"USAGE: 3dcfg filname.cfg\n";
   else
   {   

     SetupCFG(argv[1]);
   
     float FrameRate;
     long start,end;
     long count=0;
    
     start = clock ();
     while (!Input->Is_Key(SCAN_ESC))
     {

        Video->ClearBuffer();
          Handle3DInput();
          Mesh->_3DPipeLine();         // do the rendering pipeline.
        Video->Buf2Video();

        count++;
     }
    end = clock ();
    
    TearDown();

    FrameRate = (float)(CLOCKS_PER_SEC*count/(float)(end-start));   
    cout<<"Frame rate was " << FrameRate << " fps \n";
   }
}


void Build(const int Device, int Mode)
{
    Video  = NULL;
    Mesh   = NULL;
    File   = NULL;
    Input  = NULL;

    Video        = new VIDEOCLASS;
    if (Video==NULL)
      Error("Not enough memory\n");      

    switch (Device)
     {
       case KEYBOARD:
         Input = new KEYBOARDCLASS;
         if (Input==NULL)
           Error("Not enough memory\n");
       break;
       case JOYSTICK:
       break;
       case MOUSE:
       break;
       default:
       break;
     }

    Video->InitVideo(Mode);                   // initialize video mode.
    Video->CreateBuffer(255);                   // create double buffer, flood with white color.
}

void TearDown()
{
    if (Input != NULL)
      delete Input;   
    if (Video != NULL)
      delete Video;
    if (Mesh != NULL)
      delete Mesh;
    if (File != NULL)
      delete File;
}

char meshfile[10][30];
char String[30],MorphFlag[30],ShadowFlag[30];
char TransOn[30],HazeOn[30];
char Backgroundfile[30],TransTblfile[30];
char HazeTblfile[30],PaletteLinearfile[30],PalettePhongfile[30],Texturefile[30];
int XMinClip,YMinClip,XMaxClip,YMaxClip,YonClip,HitherClip;
int VidMode,ControlMode,nummesh;
int rx,ry,rz;
float wx,wy,wz;
float cx,cy,cz,cax,cay,caz;
float lx,ly,lz;

void SetupCFG(char *filename)
{
  FILE *fptr;

  fptr=fopen(filename,"r");
  if (fptr == NULL)
    Error("File does not exist\n");

   char *type;
   type=strrchr(filename,'.');          // look for filename extension.
   type++;                           
   
   if (strncmp(type,"cfg",3) !=0 )
     Error("Invalid file. File must be in .cfg format\n");

   fscanf(fptr,"DEVICES\n");
   cout<<"DEVICES\n";
     fscanf(fptr,"  BEGIN\n");
     cout<<"  BEGIN\n";
       fscanf(fptr,"    VIDEOMODE = %s\n",String);
       if (strcmp(String,"MCGA_320_200_256")==0)
         VidMode = MCGA_320_200_256;
       else
         Error("Video mode not yet implemented");
       cout<<"    VIDEO MODE = "<<String<<"\n";
       fscanf(fptr,"    CONTROL = %s\n",String);
       if (strcmp(String,"KEYBOARD")==0)
         ControlMode = KEYBOARD;
       else
         Error("Control mode not yet implemented");
       cout<<"    CONTROL MODE = "<<String<<"\n";
     fscanf(fptr,"  END\n");
     cout<<"  END\n\n";
   
   fscanf(fptr,"ENVIROMENT\n");
   cout<<"ENVIROMENT\n";
     fscanf(fptr,"  BEGIN\n");
     cout<<"  BEGIN\n";
       fscanf(fptr,"    HITHER PLANE = %d\n",&HitherClip);
         cout<<"    HITHER PLANE = "<<HitherClip<<"\n";
       fscanf(fptr,"    YON PLANE = %d\n",&YonClip);
         cout<<"    YON PLANE = "<<YonClip<<"\n";
       fscanf(fptr,"    MINCLIPX = %d\n",&XMinClip);
         cout<<"    MINCLIPX = "<<XMinClip<<"\n";
       fscanf(fptr,"    MINCLIPY = %d\n",&YMinClip);
         cout<<"    MINCLIPY = "<<YMinClip<<"\n";
       fscanf(fptr,"    MAXCLIPX = %d\n",&XMaxClip);
         cout<<"    MAXCLIPX = "<<XMaxClip<<"\n";
       fscanf(fptr,"    MAXCLIPY = %d\n",&YMaxClip);
         cout<<"    MAXCLIPY = "<<YMaxClip<<"\n";
       fscanf(fptr,"    BACKGROUND = %s\n",Backgroundfile);
         cout<<"    BACKGROUND = "<<Backgroundfile<<"\n";
     fscanf(fptr,"  END\n");
     cout<<"  END\n\n";

   fscanf(fptr,"MESH\n");
   cout<<"MESH\n";   
     fscanf(fptr,"  BEGIN\n");
     cout<<"  BEGIN\n";
       fscanf(fptr,"    MORPH = %s\n",MorphFlag);
       cout<<"    MORPH = "<<MorphFlag<<"\n";
       fscanf(fptr,"    OBJECT NUM = %d\n",&nummesh);
       cout<<"    OBJECT NUM = "<<nummesh<<"\n";
       int count;
       for (count=0;count<nummesh;count++)
       {
         fscanf(fptr,"      OBJECT = %s\n",meshfile[count]);
         cout<<"      OBJECT = "<<meshfile[count]<<"\n";
       }
       fscanf(fptr,"    ROTATION = %d %d %d\n",&rx,&ry,&rz);
       cout<<"    ROTATION = "<<rx<<" "<<ry<<" "<<rz<<"\n";
       fscanf(fptr,"    POSITION = %f %f %f\n",&wx,&wy,&wz);               
       cout<<"    POSITION = "<<wx<<" "<<wy<<" "<<wz<<"\n";
       fscanf(fptr,"    SHADOW = %s\n",ShadowFlag);
       cout<<"    SHADOW = "<<ShadowFlag<<"\n";
     fscanf(fptr,"  END\n");
     cout<<"END\n\n";
       
     fscanf(fptr,"MATERIAL\n");
     cout<<"MATERIAL\n";
       fscanf(fptr,"  BEGIN\n");
       cout<<"  BEGIN\n";
         fscanf(fptr,"    TRANSPARENCY = %s\n",TransOn);
         fscanf(fptr,"      TABLE = %s\n",TransTblfile);
         cout<<"    TRANSPARENCY = "<<TransOn<<"\n";
         cout<<"      TABLE = "<<TransTblfile<<"\n";
         fscanf(fptr,"    HAZE = %s\n",HazeOn);
         fscanf(fptr,"      TABLE = %s\n",HazeTblfile);
         cout<<"    HAZE = "<<HazeOn<<"\n";
         cout<<"      TABLE = "<<HazeTblfile<<"\n";
         fscanf(fptr,"    PALETTE = %s\n",String);
         fscanf(fptr,"      TABLE = %s\n",PaletteLinearfile);
         cout<<"    PALETTE = "<<String<<"\n";
         cout<<"      TABLE = "<<PaletteLinearfile<<"\n";
         fscanf(fptr,"    PALETTE = %s\n",String);
         fscanf(fptr,"      TABLE = %s\n",PalettePhongfile);
         cout<<"    PALETTE = "<<String<<"\n";
         cout<<"      TABLE = "<<PalettePhongfile<<"\n";
         fscanf(fptr,"    TEXTURE = %s\n",Texturefile);
         cout<<"    TEXTURE = "<<Texturefile<<"\n";
       fscanf(fptr,"END\n");
       cout<<"  END\n\n";

     fscanf(fptr,"CAMERA\n");
     cout<<"CAMERA\n";
       fscanf(fptr,"  BEGIN\n");
       cout<<"  BEGIN\n";
         fscanf(fptr,"    POSITION = %f %f %f\n",&cx,&cy,&cz);
         cout<<"    POSITION = "<<cx<<" "<<cy<<" "<<cz<<"\n";
         fscanf(fptr,"    ANGLE = %f %f %f\n",&cax,&cay,&caz);
         cout<<"    ANGLE = "<<cax<<" "<<cay<<" "<<caz<<"\n";
       fscanf(fptr,"  END\n");
       cout<<"  END\n\n";

     fscanf(fptr,"LIGHT\n");
     cout<<"LIGHT\n";
       fscanf(fptr,"  BEGIN\n");
       cout<<"  BEGIN\n";
         fscanf(fptr,"    POSITION = %f %f %f\n",&lx,&ly,&lz);
         cout<<"    POSITION = "<<lx<<" "<<ly<<" "<<lz<<"\n";
       fscanf(fptr,"  END\n");
       cout<<"  END\n\n";
       
       fclose(fptr);
       
                     // LOAD DEVICES

  Build(ControlMode,VidMode);  

                     // LOAD ENVIROMENT

  Video->SetClipExtents(XMinClip,YMinClip,XMaxClip,YMaxClip,HitherClip,YonClip);

                     // LOAD MESH
                     
  Mesh       = new MESHCLASS;
    if (Mesh==NULL)
      Error("Not enough memory\n");      

  File       = new FILESYSTEMCLASS;
    if (File==NULL)
      Error("Not enough memory\n");      

  int ObjectId;
  ObjectId=File->CreateDataBase( Mesh,meshfile[0],PARENT );
  Mesh->SetWorldXYZ( ObjectId,wx,wy,wz );
  for (count=1;count<nummesh;count++)
  {
    ObjectId=File->CreateDataBase( Mesh,meshfile[count],CHILD );   // if object has children, load in.
    Mesh->SetWorldXYZ( ObjectId,wx,wy,wz );
  }
  Mesh->CreateMeshList();                   // MUST HAVE:  List to hold all polygons.
  Mesh->SetTSRVectors( rx,ry,rz,0,0,0 );        // start object rotating
  if (strcmp(MorphFlag,"YES")==0)
    Mesh->SetMorph( Morph );
  else
    Mesh->SetMorph( NoMorph );
  if (strcmp(ShadowFlag,"YES")==0)
    Mesh->SetShadow( Shadow );
  else
    Mesh->SetShadow( NoShadow );

                     // LOAD MATERIAL
                     
  if (strcmp(TransOn,"YES")==0)
  {
    Mesh->SetMaxTransparency( Video->LoadTransTable( TransTblfile ) );       // load in transparency table.
  }    
  if (strcmp(HazeOn,"YES")==0)
  {
    if (strcmp(TransOn,"YES")==0)
      Error("Can't have both Hazing and Transparency on. Check Config file\n");
    else
    {
      Video->LoadHazeTable( HazeTblfile );
      Mesh->SetHazing( Haze );
    }
  }
  else
  if (strcmp(Backgroundfile,"NULL") != 0)         // only load if NOT hazed or if not NULL.
    Video->LoadBackground( Backgroundfile );
    
  Video->LoadTextureMap( Texturefile );
  Video->LoadPalTable( PaletteLinearfile );       // load predefined linear palette
  Video->LoadPalPhongTable( PalettePhongfile );     // load predefined phong palette.
  Video->LoadPhongTBL("data/phong.tbl");            // NOTE: must be loaded if you want phong shading. load precomputed angle table. Cos*NUM_SHADES.

                     // LOAD CAMERA

  Mesh->Camera[0].SetViewPoint(cx,cy,cz);
  Mesh->Camera[0].SetViewAngle(cax,cay,caz);       

                     // LOAD LIGHT

  Mesh->Light[0].SetLight( lx,ly,lz);
}

void Handle3DInput(void)
{
   int camangle_x=0,camangle_y=0,camangle_z=0;
   int rotangle_x=0,rotangle_y=0,rotangle_z=0;
   int transx=0,transy=0,transz=0;
   float viewx=0,viewy=0,viewz=0;

  if (Input->Is_Key(SCAN_ANY))            // first see if any key is pressed!
  {
    if (Input->Is_Key(SCAN_F1))      // 4 virtual cameras!
      Mesh->SetCurrentCamera(0);
    else
    if (Input->Is_Key(SCAN_F2))
      Mesh->SetCurrentCamera(1);
    else
    if (Input->Is_Key(SCAN_F3))
      Mesh->SetCurrentCamera(2);
    else
    if (Input->Is_Key(SCAN_F4))
      Mesh->SetCurrentCamera(3);

    if (Input->Is_Key(SCAN_INSERT))       // +X
      rotangle_x=1;
    else
    if (Input->Is_Key(SCAN_DELETE))       // -X
     rotangle_x=-1;

    if (Input->Is_Key(SCAN_HOME))         // -Y
      rotangle_y=-1;
    else
    if (Input->Is_Key(SCAN_END))          // +Y
      rotangle_y=1;
    
    if (Input->Is_Key(SCAN_PGUP))         // +Z
      rotangle_z=1;
    else
    if (Input->Is_Key(SCAN_PGDOWN))       // -Z
     rotangle_z=-1;

    if (Input->Is_Key(SCAN_UP))           // +X
      camangle_x=3;
    else
    if (Input->Is_Key(SCAN_DOWN))         // -X
     camangle_x=-3;
    
    if (Input->Is_Key(SCAN_LEFT))         // -Y
      camangle_y=-3;
    else
    if (Input->Is_Key(SCAN_RIGHT))        // +Y
      camangle_y=3;
    
    if (Input->Is_Key(SCAN_A))
      viewz=20;
    else
    if (Input->Is_Key(SCAN_Z))
      viewz=-20;

    Mesh->SetTSRVectors(rotangle_x,rotangle_y,rotangle_z,transx,transy,transz);
    Mesh->MoveCamera(camangle_x,camangle_y,camangle_z,viewx,viewy,viewz);
 
    if (Input->Is_Key(SCAN_W))
      Mesh->SetShading(WireFrame);
    if (Input->Is_Key(SCAN_C))
      Mesh->SetShading(Constant);
    if (Input->Is_Key(SCAN_L))
      Mesh->SetShading(Lambert);     
    if (Input->Is_Key(SCAN_G))
      Mesh->SetShading(Gouraud);
    if (Input->Is_Key(SCAN_P))
      Mesh->SetShading(Phong);

    if (Input->Is_Key(SCAN_T))
      Mesh->SetTextureMapping(Texture);
    if (Input->Is_Key(SCAN_N))
      Mesh->SetTextureMapping(NoTexture);

    if (Input->Is_Key(SCAN_MINUS))
      Mesh->SetTransparencyLevel(-1);
    else
    if (Input->Is_Key(SCAN_EQUAL))
      Mesh->SetTransparencyLevel(1);
  }

}

void Error(char *fmt, ...)   
{
  char msg[80];
  va_list argptr;

  va_start(argptr, fmt);
  vsprintf(msg,fmt,argptr);
  va_end(argptr);
  TearDown();
  cout<<msg;
  exit(1);
}

/*

   fscanf(fptr,"ENVIROMENT\n");
   cout<<"ENVIROMENT\n";
     fscanf(fptr,"  BEGIN\n");
     cout<<"  BEGIN\n";
       fscanf(fptr,"    HITHER PLANE = %d\n",&HITHER_Z);
         cout<<"    HITHER PLANE = "<<HITHER_Z<<"\n";
       fscanf(fptr,"    YON PLANE = %d\n",&YON_Z);
         cout<<"    YON PLANE = "<<YON_Z<<"\n";
//       fscanf(fptr,"VIEW DISTANCE = %d\n",&VIEWDISTANCE);
//         cout<<"    VIEW DISTANCE = "<<VIEWDISTANCE<<"\n";
       fscanf(fptr,"    MINCLIPX = %d\n",&POLY_MIN_CLIP_X);
         cout<<"    MINCLIPX = "<<POLY_MIN_CLIP_X<<"\n";
       fscanf(fptr,"    MINCLIPY = %d\n",&POLY_MIN_CLIP_Y);
         cout<<"    MINCLIPY = "<<POLY_MIN_CLIP_Y<<"\n";
       fscanf(fptr,"    MAXCLIPX = %d\n",&POLY_MAX_CLIP_X);
         cout<<"    MAXCLIPX = "<<POLY_MAX_CLIP_X<<"\n";
       fscanf(fptr,"    MAXCLIPY = %d\n",&POLY_MAX_CLIP_Y);
         cout<<"    MAXCLIPY = "<<POLY_MAX_CLIP_Y<<"\n";
       fscanf(fptr,"    BACKGROUND = %s\n",Backgroundfile);
         cout<<"    BACKGROUND = "<<Backgroundfile<<"\n";
     fscanf(fptr,"  END\n");
     cout<<"  END\n\n";
*/
     
