#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#include "defines.hpp"
#include "input.hpp"
#include "video.hpp"
#include "voxel.hpp"
#include "mesh.hpp"
#include "file.hpp"
#include "bmapgfx.hpp"

// Copyright (c) 1996 by Kerrigan Burgess, all rights reserved.

VIDEOCLASS        *Video;
VOXELCLASS        *Voxel;
MESHCLASS         *Mesh;
FILESYSTEMCLASS   *File;
BITMAPGFXCLASS    *Bitmap;
INPUTCLASS        *Input;

void Error(char *fmt, ...);        // Always put.

void Build(const int Device, int Mode);
void TearDown(void);
void Demo1(void);
void Demo2(void);
void Demo3(void);
void Handle3DInput(void);
void ShowMesh(int shade,int mode);
void ShowMeshEffects(int effect);


void main(int argc, char **argv)
{

   cout<<"\n\n\n\n\n ROTATION MOVEMENTS:\n";
   cout<<"  +X    --> < INSERT >     +Y --> < HOME >      +Z --> < PGUP >\n";
   cout<<"  -X    --> < DELETE >     -Y --> < END >       -Z --> < PGDN >\n\n";
   cout<<"                     ALL STOP: < F1 >\n\n";
   cout<<"  CAMERA MOVEMENTS:\n";
   cout<<"  < LEFT ARROW > < RIGHT ARROW > < UP ARROW > < DOWN ARROW >\n\n\n\n\n";
   
   cout<<"   Please press < esc >  >>> EACH <<< time in order to continue demo\n\n";
   while (!kbhit());
   
   Build(KEYBOARD,MCGA_320_200_256);

     Demo1();
     Demo2();
     Demo3();

   TearDown();

   cout<<"\n\n              Hope you liked my Demo\n\n";
    
}

void Error(char *fmt, ...)   
{
  char msg[80];
  va_list argptr;

  va_start(argptr, fmt);
  vsprintf(msg,fmt,argptr);
  va_end(argptr);
  TearDown();
  cout<<msg;
  exit(1);
}

void Build(const int Device, int Mode)
{
    Video  = NULL;
    Voxel  = NULL;
    Mesh   = NULL;
    File   = NULL;
    Bitmap = NULL;
    Input  = NULL;

    Video        = new VIDEOCLASS;
    if (Video==NULL)
      Error("Not enough memory\n");      

    switch (Device)
     {
       case KEYBOARD:
         Input = new KEYBOARDCLASS;
         if (Input==NULL)
           Error("Not enough memory\n");
       break;
       case JOYSTICK:
       break;
       case MOUSE:
       break;
       default:
       break;
     }

    Video->InitVideo(Mode);                   // initialize video mode.
    Video->CreateBuffer(0);                   // create double buffer, flood with black color.
}

void TearDown(void)
{
    if (Input != NULL)
      delete Input;   
    if (Video != NULL)
      delete Video;
    if (Voxel != NULL)
      delete Voxel;      
    if (Bitmap != NULL)
      delete Bitmap;
    if (Mesh != NULL)
      delete Mesh;
    if (File != NULL)
      delete File;
}

void Demo1(void)
{
    
    Video->LoadBackground("images/intro.pcx");
    Video->SetAllRgbPalette();

    Video->FadeToColor();
      WaitFor(20);
    Video->FadeToBlack();
    Video->DeleteBackground();          // don't need background any more.

    Bitmap     = new BITMAPGFXCLASS;
      if (Bitmap==NULL)
        Error("Not enough memory\n");
  
    Bitmap->Create( Video->GetStats(), Video->LoadPcxFile("images/biglake.pcx") );
    Video->SetAllRgbPalette();

    float scale=0.05;

    while ( (!Input->Is_Key(SCAN_ESC)) && (scale < 1.95) )             // SCALING OF BITMAPS
    {
       Video->ClearBuffer();
         scale+=0.05;
         Bitmap->Scale(Video->GetBuffer(),scale);
       Video->Buf2Video();
    }

    WaitFor(15);
         
    while ( (!Input->Is_Key(SCAN_ESC)) && (scale > 0.05) )
    {
       Video->ClearBuffer();
         scale-=0.05;
         Bitmap->Scale(Video->GetBuffer(),scale);
       Video->Buf2Video();
    }

    Input->ClearKeys();      // clear all pending key presses.
    
    while ( (!Input->Is_Key(SCAN_ESC)) && (scale < 1.95) )              // ROTATION OF BITMAPS
    {
       Video->ClearBuffer();
         scale+=0.04;
         Bitmap->Rotate(Video->GetBuffer(),3,scale);
       Video->Buf2Video();
    }

    while ( (!Input->Is_Key(SCAN_ESC)) && (scale > 0.05) )
    {
       Video->ClearBuffer();
         scale-=0.04;
         Bitmap->Rotate(Video->GetBuffer(),3,scale);
       Video->Buf2Video();
    }

    while ( (!Input->Is_Key(SCAN_ESC)) && (scale < 0.95) )   
    {
       Video->ClearBuffer();
         scale+=0.04;
         Bitmap->Rotate(Video->GetBuffer(),3,scale);
       Video->Buf2Video();
    }

   WaitFor(15);                  // let viewer admire scenery. (Yah right!)
   Video->FadeToBlack();
}

void Demo2(void)
{

    Video->LoadBackground("images/3dtrans.pcx");
    Video->SetAllRgbPalette();

    Video->FadeToColor();
      WaitFor(20);
    Video->FadeToBlack();
    Video->DeleteBackground();          // don't need background any more.

    Mesh       = new MESHCLASS;
      if (Mesh==NULL)
        Error("Not enough memory\n");      

    File       = new FILESYSTEMCLASS;
      if (File==NULL)
        Error("Not enough memory\n");      

    File->CreateDataBase(Mesh,"meshes/duck.asc",PARENT);
    File->CreateDataBase(Mesh,"meshes/sphere1.asc",PARENT);
    File->CreateDataBase(Mesh,"meshes/sphere2.asc",CHILD);
    File->CreateDataBase(Mesh,"meshes/sphere3.asc",CHILD);

    Mesh->CreateMeshList();                   // List to hold all polygons.
    Mesh->SetMaxTransparency( Video->LoadTransTable("data/trans.tbl") );       // load in transparency table.

    Video->LoadPalTable("data/standard.pal");          // load predefined linear palette.
    Video->LoadPalPhongTable("data/phong.pal");     // load predefined phong palette.
    Video->LoadPhongTBL("data/phong.tbl");             // load precomputed angle table. Cos*NUM_SHADES.
    Video->LoadHazeTable("data/haze.tbl");

    Mesh->SetTSRVectors(12,8,5,0,0,0);        // start object rotating
    Input->ClearKeys();            // reset all pending key presses

    ShowMesh(WireFrame,NoTexture);
    ShowMesh(Lambert,NoTexture);
    ShowMesh(Gouraud,NoTexture);
    ShowMesh(Phong,NoTexture);
    ShowMesh(Lambert,Texture);
    ShowMesh(Gouraud,Texture);
    ShowMesh(Phong,Texture);

    Input->ClearKeys();            // reset all pending key presses

    ShowMeshEffects(Shadow);
    ShowMeshEffects(Morph);
    ShowMeshEffects(Transparent);
    ShowMeshEffects(Haze);

    Video->FadeToBlack();
}

void Demo3(void)
{
    
    Video->LoadBackground("images/voxelgr.pcx");
    Video->SetAllRgbPalette();

    Video->FadeToColor();
      WaitFor(10);
    Video->FadeToBlack();
    Video->DeleteBackground();          // don't need background any more.


    Voxel        = new VOXELCLASS;
    if (Voxel==NULL)
      Error("Not enough memory\n");      

    Voxel->CreateDataBase(Video);

    Input->ClearKeys();            // reset all pending key presses
    while (!Input->Is_Key(SCAN_ESC))
    {
       Video->ClearBuffer();
          Voxel->HandleInput(Input);
          Voxel->RenderView( Video->GetBuffer() );
       Video->Buf2Video();
    }

  Video->FadeToBlack();
}

void Handle3DInput(void)
{
   int rotangle_x=0,rotangle_y=0,rotangle_z=0;
   float viewx=0,viewy=0,viewz=0;

   if (Input->Is_Key(SCAN_ANY))            // first see if any key is pressed!
   {
     if (Input->Is_Key(SCAN_F1))           // all stop!
       Mesh->ResetTSRVectors();

     if (Input->Is_Key(SCAN_INSERT))       // +X
       rotangle_x=1;
     else
     if (Input->Is_Key(SCAN_DELETE))       // -X
       rotangle_x=-1;

     if (Input->Is_Key(SCAN_HOME))         // -Y
       rotangle_y=-1;
     else
     if (Input->Is_Key(SCAN_END))          // +Y
       rotangle_y=1;
    
     if (Input->Is_Key(SCAN_PGUP))         // +Z
       rotangle_z=1;
     else
     if (Input->Is_Key(SCAN_PGDOWN))       // -Z
       rotangle_z=-1;
    
    if (Input->Is_Key(SCAN_F))
       viewz=25;
     else
     if (Input->Is_Key(SCAN_B))
       viewz=-25;

    Mesh->SetTSRVectors(rotangle_x,rotangle_y,rotangle_z,0,0,0);
    Mesh->MoveCamera(0,0,0,viewx,viewy,viewz);
 
    if (Input->Is_Key(SCAN_MINUS))
      Mesh->SetTransparencyLevel(-1);
    else
    if (Input->Is_Key(SCAN_EQUAL))
      Mesh->SetTransparencyLevel(1);
   }
}

void ShowMesh(int shade,int mode)
{

 Input->ClearKeys();            // reset all pending key presses

  switch ( shade | mode )
  {
    case WireFrame:
      Video->LoadBackground("images/sunwr.pcx");   // creates a double buffer, size depending on mode. Allocates space for MeshList.
    break;
    case Lambert:
      Video->LoadBackground("images/sunlb.pcx");   // creates a double buffer, size depending on mode. Allocates space for MeshList.
    break;
    case Gouraud:
      Video->LoadBackground("images/sungr.pcx");   // creates a double buffer, size depending on mode. Allocates space for MeshList.
    break;
    case Phong:
      Video->LoadBackground("images/sunpg.pcx");   // creates a double buffer, size depending on mode. Allocates space for MeshList.
    break;
    case Lambert_Texture:
      Video->LoadBackground("images/sunlbtx.pcx");   // creates a double buffer, size depending on mode. Allocates space for MeshList.
    break;
    case Gouraud_Texture:
      Video->LoadBackground("images/sungrtx.pcx");   // creates a double buffer, size depending on mode. Allocates space for MeshList.
    break;
    case Phong_Texture:
      Video->LoadBackground("images/sunpgtx.pcx");   // creates a double buffer, size depending on mode. Allocates space for MeshList.
    break;
    default:
    break;
  }
        
    Mesh->SetShading( shade );
    Mesh->SetTextureMapping( mode );
    
    while (!Input->Is_Key(SCAN_ESC))
    {
        Video->ClearBuffer();
          Handle3DInput();
          Mesh->_3DPipeLine();         // do the rendering pipeline.
        Video->Buf2Video();
    }
  Input->ClearKeys();            // reset all pending key presses
}

void ShowMeshEffects(int effect)
{

 Input->ClearKeys();            // reset all pending key presses

  switch (effect)
  {
     case Shadow:
     
       Video->LoadBackground("images/sunshd.pcx"); 

       Mesh->SetShadow( Shadow );                // enable effects.
       Mesh->SetShading( Phong );
       Mesh->SetTextureMapping( NoTexture );
       
       Mesh->Light[0].SetLight(-100,0,-1);     // move light to better see shadow.
              
       while (!Input->Is_Key(SCAN_ESC))
       {
          Video->ClearBuffer();   
            Handle3DInput();
            Mesh->_3DPipeLine();         // do the rendering pipeline.
          Video->Buf2Video();
       }

     Mesh->SetShadow( NoShadow );
     Mesh->Light[0].SetLight(0,0,-1);       // reset light source.
     break;

     case Morph:

       Video->LoadBackground("images/sunmph.pcx"); 

       Mesh->SetMorph( Morph );
       Mesh->SetShading( Phong );
       Mesh->SetTextureMapping( Texture );
       
       while (!Input->Is_Key(SCAN_ESC))
       {
          Video->ClearBuffer();
            Handle3DInput();
            Mesh->_3DPipeLine();         // do the rendering pipeline.
          Video->Buf2Video();
       }

       Mesh->SetMorph( NoMorph );   // shut off.
     break;

     case Transparent:

       Video->LoadBackground("images/suntrn.pcx"); 

       Mesh->SetTransparency( Transparent );       
       Mesh->SetShading( Phong );
       Mesh->SetTextureMapping( Texture );

       while (!Input->Is_Key(SCAN_ESC))
       {
          Video->ClearBuffer();   
            Handle3DInput();
            Mesh->_3DPipeLine();         // do the rendering pipeline.
          Video->Buf2Video();
       }

     Mesh->SetTransparency( Opaque );
     break;

     case Haze:

       Video->LoadBackground("images/sunhz.pcx");  

       Mesh->SetHazing( Haze );
       Mesh->SetShading( Gouraud );
       Mesh->SetTextureMapping( Texture );
       Mesh->MoveCamera(0,0,0,0,0,-150);    // move object into furthur into haze ( easier to see it fogged ).

       while (!Input->Is_Key(SCAN_ESC))
       {
          Video->ClearBuffer();
            Handle3DInput();
            Mesh->_3DPipeLine();         // do the rendering pipeline.
          Video->Buf2Video();
       }

       Mesh->SetHazing( NoHaze );
     break;

     default:
     break;
  }
 Input->ClearKeys();            // reset all pending key presses
}

