#include "setpal.h"

int GetMode(void);
#pragma aux GetMode =\
    "mov ah,0x0F",\
    "int 0x10,"\
    "and eax,0xFF",\

int SetMode(int);
#pragma aux SetMode =\
    "int 0x10",\
    parm [eax]\

DrawLine(int x1, int y1, int x2, int y2, unsigned char color)
{
    int dx,dy,index,xdir,ydir,error=0;
    unsigned char *screen;

    screen = (unsigned char *)0xa0000;
    dx = x2 - x1;
    dy = y2 - y1;

    if (dx>=0)
    {
	xdir = 1;
    }
    else
    {
	xdir = -1;
	dx = -dx;
    }

    if (dy>=0)
    {
	ydir = SCREENX;
    }
    else
    {
	ydir = -SCREENX;
	dy = -dy;
    }

    screen+=(y1<<8)+(y1<<6)+x1;
    
    if (dx>dy)
    {
     for (index=0;index<dx;index++)
      {
	*screen=color;
	error+=dy;
	if (error>dx)
	{
	  screen+=ydir;
	  error-=dx;
	}
	screen+=xdir;
      }
    }
    else
    {
     for (index=0;index<dy;index++)
      {
	*screen=color;
	error+=dx;
	if (error>0)
	{
	  screen+=xdir;
	  error-=dy;
	}
	screen+=ydir;
      }
    }
}

void LoadPcxFile(char *filename,PcxFile *pcx)
{
  unsigned long int i;
  signed short int mode=NORMAL,nbytes;
  unsigned char abyte,*p;
  FILE *f;
     
  f=fopen(filename,"rb");
  if (f==NULL)
  {
    printf("Couldn't find PCX file\n");
    exit(0);
  }

  fread(&pcx->hdr,sizeof(PcxHeader),1,f);
  pcx->width=1+pcx->hdr.xmax-pcx->hdr.xmin;
  pcx->height=1+pcx->hdr.ymax-pcx->hdr.ymin;
  pcx->imagebytes=( unsigned long int )(pcx->width*pcx->height);
  pcx->bitmap=( unsigned char *)malloc(pcx->imagebytes);

  p=pcx->bitmap;
  for(i=0;i<pcx->imagebytes;i++)
  {
    if(mode == NORMAL)
    {
      abyte=fgetc(f);
      if(( unsigned char )abyte > 0xbf)
      {
	nbytes=abyte & 0x3f;
	abyte=fgetc(f);
	if(--nbytes > 0)
	  mode=RLE;
      }
    }
    else if(--nbytes == 0)
      mode=NORMAL;
    *p++=abyte;
  }

  fseek(f,-768L,SEEK_END);      // get palette from pcx file
  fread(pcx->pal,768,1,f);
  p=pcx->pal;
  for(i=0;i<768;i++)            // bit shift palette
  *p++=*p >>2;
  fclose(f);
}

void SetAllRgbPalette(unsigned char *pal)
{
      struct SREGS s;
      union REGS r;

      segread(&s);                    // get current segment values
      s.es=FP_SEG((void far*)pal);    // point ES to pal array
      r.x.edx=FP_OFF((void far*)pal);  // get offset to pal array
      r.x.eax=0x1012;                  // BIOS func 10h sub 12h
      r.x.ebx=0;                       // starting DAC register
      r.x.ecx=256;                     // ending DAC register
      int386x(0x10,&r,&r,&s);          // call video BIOS
}


void GetColor(int index,unsigned char *red,unsigned char *green,unsigned char *blue)
{
   *red   = AuxTable[index].red;
   *green = AuxTable[index].green;
   *blue  = AuxTable[index].blue;   
}

void SetupAuxTable(void)          // populate AuxTable so others functions can access rgb values without
{                                 // using IO ports.
  int index;

  for (index=0;index<256;index++)
  {  
    outp(COLOR_MASK,0xff);
    outp(COLOR_REGISTER_RD, index);

    AuxTable[index].red=  inp(COLOR_DATA);
    AuxTable[index].green=inp(COLOR_DATA);
    AuxTable[index].blue= inp(COLOR_DATA);
  }
}

void SaveOldPalette()
{
  int index,reg;

  reg=0;
  for (index=0;index<256;index++)
  {  
    outp(COLOR_MASK,0xff);
    outp(COLOR_REGISTER_RD, index);

    OldPalette[reg]=inp(COLOR_DATA);      // RED
    reg++;
    OldPalette[reg]=inp(COLOR_DATA);      // GREEN
    reg++; 
    OldPalette[reg]=inp(COLOR_DATA);      // BLUE
    reg++;
  }
}

void SetupMaps()
{    
  SaveOldPalette();                 // save old palette because we're changing it.
  LoadPcxFile(texturemap,&pcx);     // read in background bitmap
  SetAllRgbPalette(pcx.pal);
  SetupAuxTable();     
}

void ExitGracefully()
{
  SetAllRgbPalette(OldPalette);      // put old palette back.
  free(&pcx.bitmap);                
}

int SearchPal(unsigned char red, unsigned char green,
                        unsigned char blue)
{
  int index, reg;
  unsigned char r,g,b;
  long test_error, error=200000;    // some extremely large number.
   
  for (index=0;index<256;index++)
  {
     GetColor(index,&r,&g,&b);

     test_error=( (red-r)*(red-r) + (green-g)*(green-g) + (blue-b)*(blue-b) );

     if (test_error<error)
     {
	error=test_error;
        reg=index;	
	if (test_error == 0)    // found it exactly!
	  return reg;
     }
  }
 return reg;
}
		
void CreateLookup(int Shades, int Ambient, float percentage)  // makes a Linear Palette.
{
  unsigned char red,green,blue;
  int index,shd;
  float diffuse,step,r,g,b;
  float AmbientPercentage,AmbientRed,AmbientGreen,AmbientBlue;
  
  Look_Pal = (unsigned char *)calloc(256*Shades,sizeof(char));
  if (Look_Pal == NULL)
   {
     printf("ERROR: Not enough memory for Lookup Table\n");
     exit(-1);
   }
  
  colorindex=0;
  AmbientPercentage=Ambient/(float)100;
  step=percentage/(float)Shades;
  
  for (index=0;index<256;index++)
  {

    GetColor(index,&red,&green,&blue);

    AmbientRed   = red*AmbientPercentage;
    AmbientGreen = green*AmbientPercentage;
    AmbientBlue  = blue*AmbientPercentage;

    diffuse = 0;
    r=0;
    g=0;
    b=0;
    
      for (shd=0;shd<Shades;shd++)   // Shades from black to color.
      {

	r= AmbientRed + red*diffuse;
	  if (r>255)
	    r=255;
	g= AmbientGreen + green*diffuse;
	  if (g>255)
	    g=255;
	b= AmbientBlue + blue*diffuse;
	  if (b>255)
	    b=255;

	Look_Pal[colorindex]=SearchPal((unsigned char)r,(unsigned char)g,(unsigned char)b);

	colorindex++;
        diffuse+=step;
	
      }
  }
}

void CreateSpecularTable(int Shades,int Shiny)
{    
  float angle, ANGULAR_INCR,DEG_TO_RAD;
  double ANG;
  int index;

  DEG_TO_RAD = 3.1459/(float)180;
  
           // make Look up for cos(@)^n 
  Specular = (float *)malloc(sizeof(float)*Shades);
  Diffuse  = (float *)malloc(sizeof(float)*Shades);

  ANGULAR_INCR = 90/(float)Shades;
  index=0;
      
  for (angle=90;angle>0;angle-=ANGULAR_INCR)   // go from 0 to 1.
  {
      
     ANG = -angle*DEG_TO_RAD;
     
     ANG = cos( ANG );
     
     Diffuse[index]  = ANG;
     Specular[index] = (float)pow( ANG, Shiny );
     index++;
  }  
}

void CreatePhongPalette(int Shades, int Ambient, int SpecularK)
{
  unsigned char red,green,blue;
  int index,shd;
  float r,g,b,AmbientRed,AmbientGreen,AmbientBlue,AmbientPercentage;

  Look_Pal = (unsigned char *)calloc(256*Shades,sizeof(char));
  if (Look_Pal == NULL)
   {
     printf("ERROR: Not enough memory for Lookup Table\n");
     exit(-1);
   }
  
  AmbientPercentage=Ambient/(float)100;
  colorindex=0;
  
  for (index=0;index<256;index++)
  {

    GetColor(index,&red,&green,&blue);  // find shades of this color.

    AmbientRed   = red*AmbientPercentage;
    AmbientGreen = green*AmbientPercentage;
    AmbientBlue  = blue*AmbientPercentage;
    
    for (shd=0;shd<Shades;shd++)   // Shades from black to color.
    {

        r= AmbientRed + red*Diffuse[shd] + SpecularK*Specular[shd];
	if (r>255)
	  r=255;

	g= AmbientGreen + green*Diffuse[shd] + SpecularK*Specular[shd];
	if (g>255)
	  g=255;

	b= AmbientBlue + blue*Diffuse[shd] + SpecularK*Specular[shd];
	if (b>255)
	  b=255;
	
	Look_Pal[colorindex]=SearchPal((unsigned char)r,(unsigned char)g,(unsigned char)b);
	colorindex++;

    }
  }
}

void CreateTransTable(int Levels)
{
   unsigned char RED,GREEN,BLUE,Rf,Gf,Bf,Rb,Gb,Bb;
   int count,findex,bindex,index;
   float Trans,TransDelta;

   TransTable = (unsigned char *)calloc(256*256*Levels,sizeof(char));    // malloc lookup table.
   if (TransTable == NULL)
   {
     printf("ERROR: Not enough memory for Trans Table\n");
     exit(-1);
   }

   index=0;
   TransDelta = 1/(float)(Levels+1); // don't want fully opaque or transp.

   Trans=0.0;                       
   
   for (count=0;count<Levels;count++)    
   {                                            
     Trans+=TransDelta;   
     for (findex=0;findex<256;findex++)
     {
        GetColor(findex,&Rf,&Gf,&Bf);                     // get forground color.
        for (bindex=0;bindex<256;bindex++)
        {
	   GetColor(bindex,&Rb,&Gb,&Bb);                  // get backround color.

	   RED   = (Rb*Trans) + Rf*(1-Trans);          // mix throughly, add water.
	   GREEN = (Gb*Trans) + Gf*(1-Trans);
	   BLUE  = (Bb*Trans) + Bf*(1-Trans);
	 
	   TransTable[index] = SearchPal(RED,GREEN,BLUE);
	   index++;
	}
     }
   }
}
	 
void CreateHazeTable(int Levels,unsigned char HAZERED,
                     unsigned char HAZEGREEN,unsigned char HAZEBLUE)
{
   unsigned char RED,GREEN,BLUE; 
   float REDf,GREENf,BLUEf,Dr,Dg,Db,kludge;
   int reg,haze,index;

   kludge=0.50;
   
   HazeTable = (unsigned char *)calloc(256*Levels,sizeof(char));    // malloc lookup table.
   if (HazeTable == NULL)
   {
     printf("ERROR: Not enough memory for Haze Table\n");
     exit(-1);
   }

   index=0;
   for (reg=0;reg<256;reg++)
   {
     GetColor(reg,&RED,&GREEN,&BLUE);
     
     Dr=(HAZERED-RED)/(float)(Levels*kludge);
     Dg=(HAZEGREEN-GREEN)/(float)(Levels*kludge);
     Db=(HAZEBLUE-BLUE)/(float)(Levels*kludge);

     REDf=(float)RED;
     GREENf=(float)GREEN;
     BLUEf=(float)BLUE;
          
     for (haze=0;haze<Levels;haze++)
     {                   
        HazeTable[index] = SearchPal( (unsigned char)REDf,
	                              (unsigned char)GREENf,
				      (unsigned char)BLUEf);
        REDf+=Dr;
	GREENf+=Dg;
	BLUEf+=Db;
	index++;
     }
   }
}

void CreateShadowVox(float percentage)
{
  unsigned char red,green,blue;
  int index;
  float r,g,b;

  ShadowPAL = (unsigned char *)calloc(256*2,sizeof(char));  //256*2 for shadow on/off
  if (ShadowPAL == NULL)
   {
     printf("ERROR: Not enough memory for Lookup Table\n");
     exit(-1);
   }
  
  colorindex=0;
  
  for (index=0;index<256;index++)
  {

    GetColor(index,&red,&green,&blue);

    r=red*percentage;
    g=green*percentage;
    b=blue*percentage;
    
    ShadowPAL[colorindex]=index;         // shadow is off, so use same color
    colorindex++;
                                        // find shadow color when shadow is on.
    ShadowPAL[colorindex]=SearchPal((unsigned char)r,(unsigned char)g,(unsigned char)b);
    colorindex++;
  }
}

void SaveLookPal(int Shades,char *filename)
{
  FILE *fptr;
  int index;

  fptr=fopen(filename,"wb");

  fprintf(fptr,"Palette ");
  fprintf(fptr,"%4d",Shades);
  fprintf(fptr,"Texture Map ");
  fwrite((void *)texturemap,sizeof(char),30,fptr);
  fwrite((void *)Look_Pal,sizeof(char),Shades*256,fptr);

 fclose(fptr);
}

void SaveTransTable(int Levels)
{
  FILE *fptr;
  int index;

  fptr=fopen("trans.tbl","wb");

  fprintf(fptr,"Transparency Table ");
  fprintf(fptr,"%2d",Levels);                 // max of 10 levels.
  fprintf(fptr,"Texture Map ");
  fwrite((void *)texturemap,sizeof(char),30,fptr);
  fwrite((void *)TransTable,sizeof(char),Levels*256*256,fptr);  // you got to be
                                              // kidding! Max 640k Lookup table?
 fclose(fptr);
}

void SaveHazeTable(int Levels)
{
  FILE *fptr;
  int index;

  fptr=fopen("haze.tbl","wb");

  fprintf(fptr,"Haze Table ");
  fprintf(fptr,"%4d",Levels);
  fprintf(fptr,"Texture Map ");
  fwrite((void *)texturemap,sizeof(char),30,fptr);
  fwrite((void *)HazeTable,sizeof(char),256*Levels,fptr);  
                                              
 fclose(fptr);
}

void SaveShadowVox()
{
  FILE *fptr;
  int index;

  fptr=fopen("shadow.pal","wb");

  fprintf(fptr,"Shadow Palette ");
  fprintf(fptr,"Texture Map ");
  fwrite((void *)texturemap,sizeof(char),30,fptr);
  fwrite((void *)ShadowPAL,sizeof(char),256*2,fptr);

 fclose(fptr);
}

ReadPal(char *filename)
{

  FILE *fptr;
  char buffer[30];
  int Shades;
       
  fptr=fopen(filename,"rb");
  if (fptr==NULL)
  {
     SetMode(prevmode);
     printf("Couldn't find filename\n");
     exit(0);
  }
    
  fread((void *)buffer,sizeof(char),8,fptr);      // header.
  fscanf(fptr,"%4d",&Shades);                     // num of shades.
  
  Look_Pal  = (unsigned char *)calloc(256*Shades,sizeof(char));    // malloc lookup table.

  fread((void *)buffer,sizeof(char),12,fptr);      // header.
  fread((void *)texturemap,sizeof(char),30,fptr);
  
  fread((void *)Look_Pal,sizeof(unsigned char),Shades*256,fptr);

 fclose(fptr);

 SetupMaps();                // load in correct texture map.
 DisplayShades(Shades);
}

void DisplayShades(int Shades)       // Note: this is geared towards showing
{                                    // 64 shades of each color. If more shades
  int x=10,y=40,colorindex=0;        // It might atempt to write outside of VidMem. 
  int index,shade;

  if (Shades>64)
    Shades=64;
    
  for (index=0;index<128;index++)
  {
    for (shade=0;shade<Shades;shade++)
    {      	  	 
       DrawLine(x,y,x+2,y,(unsigned char)Look_Pal[colorindex]);
       colorindex++;
       x+=2;
    }
    y++;
    x=10;
  }

  x=170;
  y=40;
  for (index=0;index<128;index++)
  {
    for (shade=0;shade<Shades;shade++)
    {      	  	 
       DrawLine(x,y,x+2,y,(unsigned char)Look_Pal[colorindex]);
       colorindex++;
       x+=2;
    }
    x=170;
    y++;
  }  
}

void DisplayHaze(int Levels)
{
  int x=10,y=10,colorindex=0;
  int index,shade;

  for (index=128;index<256;index++)
  {
    for (shade=0;shade<Levels;shade++)
    {      	  	 
       DrawLine(x,y,x+1,y,(unsigned char)Look_Pal[colorindex]);
       colorindex++;
       x++;
    }
    y++;
    x=10;
  }
}

void main(int argc, char **argv)
{
  unsigned char red,green,blue;  
  float percentage;
  int Shades,Ambient,SpecularK,Shiny,Levels,Depth=1;

  if (argc<2)
  {
     printf("\n\n\n\nUSAGE: setpal <read> | <write> | <phong> | <transparency> | <haze> | <voxel>\n");
  }
  else
  if (strcmp(argv[1],"write")==0)       // defaults to linear palette.
  {
      printf("Num of shades:\n");
      scanf("%d",&Shades);
      printf("Ambient Level:\n");
      scanf("%d",&Ambient);
      printf("Percentage: (0 - 200%) of color \n");
      scanf("%f",&percentage);
      printf("Colormap: filename.pcx\n");
      scanf("%s",texturemap);
      printf("\n\n");
        
      SetupMaps();
      
      percentage/=100;
      CreateLookup(Shades,Ambient,percentage); 

      SaveLookPal(Shades,"standard.pal");
      ExitGracefully();
  }
  else
  if (strcmp(argv[1],"read")==0)
  {
      if (argc<3)
      {
	printf("\n\n\n\nUSAGE: setpal read filename.pal\n");
	exit(0);
      }
      else
      {	
        prevmode=GetMode();
        SetMode(0x13);

        ReadPal(argv[2]);

        while (!kbhit())
        ;

        SetMode(prevmode);
      }
  }
  else
  if (strcmp(argv[1],"phong")==0)
  {      
      printf("Num of shades:\n");
      scanf("%d",&Shades);
      printf("Ambient Level:\n");
      scanf("%d",&Ambient);
      printf("Shinyness: (1 -> ?)\n");
      scanf("%d",&Shiny);
      printf("Specular Constant: \n");
      scanf("%d",&SpecularK);
      printf("From palette: filename.pcx\n");
      scanf("%s",texturemap);
      printf("\n\n");        

      CreateSpecularTable(Shades,Shiny);
      printf("Please wait this may take some time\n");
      SetupMaps();
      CreatePhongPalette(Shades,Ambient,SpecularK);
      SaveLookPal(Shades,"phong.pal");      
      ExitGracefully();
  }
  if (strcmp(argv[1],"transparency")==0)
  {
      printf("Num Levels of transparency:\n");
      scanf("%d",&Levels);
      printf("From palette: filename.pcx\n");
      scanf("%s",texturemap);
      printf("\n\n");
        
      Levels--;            // don't want fully opaque or fully transparent.
      if (Levels > 10)
        Levels = 10;

      printf("Please wait this may take some time\n");
      SetupMaps();
      CreateTransTable(Levels);
      SaveTransTable(Levels);
      ExitGracefully();
  }
  else
  if (strcmp(argv[1],"haze")==0)
  {
      printf("Num Levels of Haze:\n");
      scanf("%d",&Levels);
      printf("From palette: filename.pcx\n");
      scanf("%s",texturemap);
      printf("Haze color: <r,g,b>\n");
      scanf("%d %d %d",&red,&green,&blue);
      printf("\n\n");
        
      printf("Please wait this may take some time\n");
      SetupMaps();
      CreateHazeTable(Levels,red,green,blue);
      SaveHazeTable(Levels);
      ExitGracefully();
  }
  else
  if (strcmp(argv[1],"voxel")==0)
  {
      printf("Colormap: filename.pcx\n");
      scanf("%s",texturemap);
      printf("Shadow Percentage: \n");
      scanf("%f",&percentage);
      printf("\n\n");
        
      SetupMaps();
      
      percentage/=100;
      CreateShadowVox(percentage); 

      SaveShadowVox();
      ExitGracefully();
  }           
}
