/* maze.c

   Win32 program to demonstrate drawing from a 2D BSP tree. Allows
   navigation around and viewing of a world made of walls based on
   a 2D-BSP tree, to illustrate the use of BSP trees for surface
   visibility.

   Hardwired to display the BSP tree stored in bsptree.txt in the
   current directory.

   Derived from the VC++ generic sample application.
   Tested with VC++ 2.0 running on Windows NT 3.5. */

#include <windows.h>   	// required for all Windows applications
#include <stdlib.h>
#include <stdio.h>
#include "maze.h"  		// specific to this program

#define DBG	0			// set to 0 for no debug code, 1 for debug
#define INITIAL_DIB_WIDTH  	320		// initial dimensions of DIB
#define INITIAL_DIB_HEIGHT	240		//  into which we'll draw
#define NO_CHILD_TREE	-1	// index for front or back child if no
                            //  tree in that direction
#define FIXEDPOINT(x)	((FIXEDPOINT)(((long)x)*((long)0x10000)))
#define FIXTOINT(x)		((int)(x >> 16))
#define ANGLE(x)		((long)x)
#define STANDARD_SPEED	(FIXEDPOINT(20))
#define STANDARD_ROTATION (ANGLE(4))
#define MAX_NUM_NODES	2000
#define MAX_NUM_EXTRA_VERTICES	2000
#define WORLD_MIN_X 	(FIXEDPOINT(-16000))
#define WORLD_MAX_X 	(FIXEDPOINT(16000))
#define WORLD_MIN_Y 	(FIXEDPOINT(-16000))
#define WORLD_MAX_Y 	(FIXEDPOINT(16000))
#define WORLD_MIN_Z 	(FIXEDPOINT(-16000))
#define WORLD_MAX_Z 	(FIXEDPOINT(16000))
#define PROJECTION_RATIO (2.0/1.0)  // controls field of view; the
                                    // bigger this is, the narrower
                                    // the field of view
typedef long FIXEDPOINT;

typedef struct _VERTEX
{
	FIXEDPOINT x;
	FIXEDPOINT z;
	FIXEDPOINT viewx;
	FIXEDPOINT viewz;
} VERTEX, *PVERTEX;

typedef struct _POINT2
{
	FIXEDPOINT x;
	FIXEDPOINT z;
} POINT2, *PPOINT2;

typedef struct _POINT2INT
{
	int	x;
	int y;
} POINT2INT, *PPOINT2INT;

typedef long ANGLE;		// angles are stored in degrees

typedef struct _NODE
{
	VERTEX *   	pstartvertex;
	VERTEX *	pendvertex;
	FIXEDPOINT	walltop;
	FIXEDPOINT	wallbottom;
	FIXEDPOINT 	tstart;
	FIXEDPOINT 	tend;
	FIXEDPOINT	clippedtstart;
	FIXEDPOINT	clippedtend;
	struct _NODE *fronttree;
	struct _NODE *backtree;
	int			color;
	FIXEDPOINT	screenxstart;
	FIXEDPOINT	screenxend;
	FIXEDPOINT	screenytopstart;
	FIXEDPOINT	screenybottomstart;
	FIXEDPOINT	screenytopend;
	FIXEDPOINT	screenybottomend;
	int			isVisible;
} NODE, *PNODE;

FIXEDPOINT sintable[] = {
#include "sintable.h"
};

BITMAPINFO * pbmiDIB;		// pointer to the BITMAPINFO
char * pDIB;				// pointer to DIB section we'll draw into
HBITMAP hDIBSection;        // handle of DIB section
HINSTANCE hInst;        // current instance
char szAppName[] = "Maze";   // The name of this application
char szTitle[]   = "3D multiperson maze"; // The title bar text
HPALETTE hpalold, hpalDIB;
int iteration = 0;
int WorldIsRunning = 1;
HWND hwndOutput;
int DIBWidth, DIBHeight;
FIXEDPOINT fxHalfDIBWidth, fxHalfDIBHeight;
int DIBPitch;
VERTEX *pvertexlist;
VERTEX *pextravertexlist;
NODE *pnodelist;
int numvertices;
int numnodes;
POINT2 currentlocation;
ANGLE currentangle;
POINT2 currentdirection;
POINT2 currentorientation;
FIXEDPOINT currentspeed;
FIXEDPOINT fxViewerY;
FIXEDPOINT FrontClipPlane = FIXEDPOINT(10);
FIXEDPOINT currentYSpeed;

#if DBG
int ShowDrawingSteps = 0;
#endif

void UpdateWorld(void);
int LoadBSPTree(void);
void DisplayError(char *text);
FIXEDPOINT FixedMul(FIXEDPOINT x, FIXEDPOINT y);
FIXEDPOINT FixedDiv(FIXEDPOINT x, FIXEDPOINT y);
FIXEDPOINT FixedSin(ANGLE angle);
FIXEDPOINT FixedCos(ANGLE angle);
extern int FillConvexPolygon(POINT2INT * VertexPtr, int Color);

/////////////////////////////////////////////////////////////////////
// WinMain
/////////////////////////////////////////////////////////////////////
int APIENTRY WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
    LPSTR lpCmdLine, int nCmdShow)
{
    MSG msg;
    HANDLE hAccelTable;

    if (!hPrevInstance) {       // Other instances of app running?
        if (!InitApplication(hInstance)) { // Initialize shared things
            return (FALSE);     // Exits if unable to initialize
        }
    }

    // Perform initializations that apply to a specific instance
    if (!InitInstance(hInstance, nCmdShow)) {
        return (FALSE);
    }

    hAccelTable = LoadAccelerators (hInstance, szAppName);

    // Acquire and dispatch messages until a WM_QUIT message is
    // received
    for (;;) {
		if (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE)) {
			do {
	    		if (msg.message == WM_QUIT) {
					goto Done;
				}
	      		if (!TranslateAccelerator (msg.hwnd, hAccelTable,
	      		        &msg)) {
            		TranslateMessage(&msg);// xlates virt keycodes
       	        	DispatchMessage(&msg); // Dispatches msg to window
	       	   	}
			} while (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE));
		}

        // Update the world if updating has begun
		if (WorldIsRunning)
			UpdateWorld();
	}

Done:
    return (msg.wParam); // Returns the value from PostQuitMessage

    lpCmdLine; // This will prevent 'unused formal parameter' warnings
}

/////////////////////////////////////////////////////////////////////
// InitApplication
/////////////////////////////////////////////////////////////////////
BOOL InitApplication(HINSTANCE hInstance)
{
        WNDCLASS  wc;

        // Fill in window class structure with parameters that
        // describe the main window.
        wc.style         = CS_HREDRAW | CS_VREDRAW;
        wc.lpfnWndProc   = (WNDPROC)WndProc;
        wc.cbClsExtra    = 0;
        wc.cbWndExtra    = 0;
        wc.hInstance     = hInstance;
        wc.hIcon         = LoadIcon (hInstance, szAppName);
        wc.hCursor       = LoadCursor(NULL, IDC_ARROW);
        wc.hbrBackground = (HBRUSH)(COLOR_WINDOW+1);
        wc.lpszMenuName  = szAppName;
        wc.lpszClassName = szAppName;

        // Register the window class and return success/failure code.
        return (RegisterClass(&wc));
}

/////////////////////////////////////////////////////////////////////
// InitInstance
/////////////////////////////////////////////////////////////////////
BOOL InitInstance(
        HINSTANCE       hInstance,
        int             nCmdShow)
{
        HWND            hwnd; // Main window handle.
		HDC				hdc;
		INT				i, j, k;
		LOGPALETTE *	plogpal;
		PUSHORT 		pusTemp;
		HPALETTE		hpal;
		RECT			rctmp;
		int				screenx, screeny;

        // Save the instance handle in static variable, which will be
        // used in many subsequent calls from this application to
        // Windows
        hInst = hInstance; // Store inst handle in our global variable

        // Create a main window for this application instance
		DIBWidth = INITIAL_DIB_WIDTH;
		fxHalfDIBWidth = FIXEDPOINT(DIBWidth/2);
		DIBPitch = DIBWidth;
		DIBHeight = INITIAL_DIB_HEIGHT;
		fxHalfDIBHeight = FIXEDPOINT(DIBHeight/2);
	   	rctmp.left = 0;
		rctmp.top = 0;
		rctmp.right = DIBWidth;
		rctmp.bottom = DIBHeight;
   
	  	AdjustWindowRect(&rctmp, WS_OVERLAPPEDWINDOW, 1);

		screenx = GetSystemMetrics(SM_CXSCREEN);
		screeny = GetSystemMetrics(SM_CYSCREEN);

        hwnd = CreateWindow(
                szAppName,           // See RegisterClass() call.
                szTitle,             // Text for window title bar.
                WS_OVERLAPPEDWINDOW,// Window style.
                screenx - (rctmp.right - rctmp.left),
                screeny - (rctmp.bottom - rctmp.top),
				rctmp.right - rctmp.left,
				rctmp.bottom - rctmp.top,
                NULL,
                NULL,
                hInstance,
                NULL
        );

        // If window could not be created, return "failure"
        if (!hwnd) {
        	return (FALSE);
        }

		// Make the window visible and draw it
        ShowWindow(hwnd, nCmdShow); // Show the window
        UpdateWindow(hwnd);         // Sends WM_PAINT message

        hdc = GetDC(hwnd);

        // For 256-color mode, set up the palette for maximum speed
        // in copying to the screen. If not a 256-color mode, the
        // adapter isn't palettized, so we'll just use the default
        // palette. However, we will run a lot slower in this case
        // due to translation while copying
        if (GetDeviceCaps(hdc, RASTERCAPS) & RC_PALETTE) {
            // This is a 256-color palettized mode.
    		// Set up and realize our palette and the identity color
    		// table for the DIB (identity means that DIB color
            // indices and screen palette indices match exactly, so
            // GDI doesn't have to do any translation when copying
            // from DIB to screen. This helps performance a lot)
		    plogpal = malloc(sizeof(LOGPALETTE) +
	    	        256 * sizeof(PALETTEENTRY));

    		if (plogpal == NULL) {
		    	return(FALSE);
    		}

	    	// Take up all physical palette entries, to flush out
	    	// anything that's currently in the palette
	    	plogpal->palVersion = 0x300;
		    plogpal->palNumEntries = 236;
	    	for (i=0; i<236; i++) {
		    	plogpal->palPalEntry[i].peRed = i;
			    plogpal->palPalEntry[i].peGreen = 0;
    			plogpal->palPalEntry[i].peBlue = 0;
	    		plogpal->palPalEntry[i].peFlags =
		    	        PC_RESERVED | PC_NOCOLLAPSE;
    		}

	    	hpal = CreatePalette(plogpal);

    		if (hpal == 0) {
		    	return(FALSE);
    		}

		    hpalold = SelectPalette(hdc, hpal, FALSE);

    		if (hpalold == 0) {
		    	return(FALSE);
    		}

    		if (RealizePalette(hdc) != plogpal->palNumEntries) {
			    return(FALSE);
		    }

    		if (SelectPalette(hdc, hpalold, FALSE) == 0) {
		    	return(FALSE);
    		}

	    	if (!DeleteObject(hpal)) {
			    return(FALSE);
    		}

		    // Now set up the 6value-6value-6value RGB palette,
		    // followed by 20 gray levels, that we want to work with
		    // into the physical palette
	    	for (i=0; i<6; i++) {
			    for (j=0; j<6; j++) {
	    			for (k=0; k<6; k++) {
			    		plogpal->palPalEntry[i*36+j*6+k].peRed =
                                i*255/6;
					    plogpal->palPalEntry[i*36+j*6+k].peGreen =
					            j*255/6;
    					plogpal->palPalEntry[i*36+j*6+k].peBlue =
	   				            k*255/6;
	    				plogpal->palPalEntry[i*36+j*6+k].peFlags =
					        PC_RESERVED | PC_NOCOLLAPSE;
		    		}
    			}
	    	}

		    for (i=0; i<20; i++) {
    			plogpal->palPalEntry[i+216].peRed = i*255/20;
	    		plogpal->palPalEntry[i+216].peGreen = i*255/20;
			    plogpal->palPalEntry[i+216].peBlue = i*255/20;
    			plogpal->palPalEntry[i+216].peFlags =
	    		        PC_RESERVED | PC_NOCOLLAPSE;
		    }

    		hpal = CreatePalette(plogpal);

	    	if (hpal == 0) {
			    return(FALSE);
    		}

	    	if (SelectPalette(hdc, hpal, FALSE) == 0) {
			    return(FALSE);
    		}

	    	if (RealizePalette(hdc) != plogpal->palNumEntries) {
			    return(FALSE);
    		}

		    // Get back the 256 colors now in the physical palette,
		    // which are the 236 we just selected, plus the 20 static
		    // colors
    		if (GetSystemPaletteEntries(hdc, 0, 256,
    		        plogpal->palPalEntry) != 256) {
	    		return(FALSE);
		    }

    		for (i=10; i<246; i++) {
		    	plogpal->palPalEntry[i].peFlags =
			            PC_RESERVED | PC_NOCOLLAPSE;
    		}

	    	// Now create a logical palette that exactly matches the
		    // physical palette, and realize it. This is the palette
    		// into which the DIB pixel values will be indices
    		plogpal->palNumEntries = 256;

	    	hpalDIB = CreatePalette(plogpal);

    		if (hpalDIB == 0)
		    	return(FALSE);

	    	if (SelectPalette(hdc, hpalDIB, FALSE) == 0)
			    return(FALSE);

	    	DeleteObject(hpal);

    		if (RealizePalette(hdc) != plogpal->palNumEntries)
		    	return(FALSE);

	    	if (SelectPalette(hdc, hpalold, FALSE) == FALSE)
			    return(FALSE);

       		free(plogpal);
        }

		// Finally, set up the DIB section
		pbmiDIB = malloc(sizeof(BITMAPINFO) - 4 + 256*sizeof(USHORT));

		if (pbmiDIB == NULL)
			return(FALSE);


		pbmiDIB->bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
		pbmiDIB->bmiHeader.biWidth = DIBPitch;
		pbmiDIB->bmiHeader.biHeight = DIBHeight;
		pbmiDIB->bmiHeader.biPlanes = 1;
		pbmiDIB->bmiHeader.biBitCount = 8;
		pbmiDIB->bmiHeader.biCompression = BI_RGB;
		pbmiDIB->bmiHeader.biSizeImage = 0;
		pbmiDIB->bmiHeader.biXPelsPerMeter = 0;
		pbmiDIB->bmiHeader.biYPelsPerMeter = 0;
		pbmiDIB->bmiHeader.biClrUsed = 256;
		pbmiDIB->bmiHeader.biClrImportant = 256;

		pusTemp = (PUSHORT) pbmiDIB->bmiColors;

		for (i=0; i<256; i++) {
			*pusTemp++ = i;
		}

        hDIBSection = CreateDIBSection (hdc, pbmiDIB, DIB_PAL_COLORS,
                          &pDIB, NULL, 0);

        if (!hDIBSection) {
            free(pbmiDIB);
            return(FALSE);
        }

		// Clear the DIB
		memset(pDIB, 0, DIBPitch*DIBHeight);

		ReleaseDC(hwnd, hdc);

		// Read in the BSP tree describing the walls.
		if (!LoadBSPTree())
			return(FALSE);

		hwndOutput = hwnd;

		// Set the initial location, direction, and speed
		currentlocation.x = FIXEDPOINT(0);
		currentlocation.z = FIXEDPOINT(-1500);

		currentangle = ANGLE(90.0);

		currentdirection.x = FIXEDPOINT(0);
		currentdirection.z = FIXEDPOINT(1);
		currentorientation = currentdirection;
		currentYSpeed = FIXEDPOINT(0);

		fxViewerY = FIXEDPOINT(120);

		currentspeed = 0;

        return (TRUE);              // We succeeded...
}

/////////////////////////////////////////////////////////////////////
// WndProc
/////////////////////////////////////////////////////////////////////
LRESULT CALLBACK WndProc(
                HWND hwnd,         // window handle
                UINT message,      // type of message
                WPARAM uParam,     // additional information
                LPARAM lParam)     // additional information
{
    int wmId, wmEvent;
	UINT fwSizeType;
	int oldDIBWidth, oldDIBHeight, oldDIBPitch;
    HBITMAP holdDIBSection;
    HDC hdc;

    switch (message) {

    case WM_COMMAND:  // message: command from application menu

    	wmId    = LOWORD(uParam);
        wmEvent = HIWORD(uParam);

        switch (wmId) {

     	case IDM_EXIT:
        	DestroyWindow (hwnd);
            break;

        default:
        	return (DefWindowProc(hwnd, message, uParam, lParam));
        }
        break;

	case WM_KEYDOWN:
		switch (uParam) {

        case VK_DOWN:
			currentspeed = -STANDARD_SPEED;
			break;

        case VK_UP:
			currentspeed = STANDARD_SPEED;
			break;

        case VK_LEFT:
			currentangle += STANDARD_ROTATION;
			if (currentangle > ANGLE(360.0))
				currentangle -= ANGLE(360.0);
			currentdirection.x = FixedCos(currentangle);
			currentdirection.z = FixedSin(currentangle);
			currentorientation = currentdirection;
			break;

        case VK_RIGHT:
			currentangle -= STANDARD_ROTATION;
			if (currentangle < ANGLE(0.0))
				currentangle += ANGLE(360.0);
			currentdirection.x = FixedCos(currentangle);
			currentdirection.z = FixedSin(currentangle);
			currentorientation = currentdirection;
			break;

		case VK_SUBTRACT:
			currentYSpeed += STANDARD_SPEED;
			break;

		case VK_ADD:
			currentYSpeed -= STANDARD_SPEED;
			break;

		default:
			break;
		}
		return(0);

	case WM_KEYUP:
		switch (uParam) {

        case VK_DOWN:
			currentspeed = 0;
			break;

        case VK_UP:
			currentspeed = 0;
			break;

        case VK_LEFT:
			break;

        case VK_RIGHT:
			break;

		case VK_SUBTRACT:
			currentYSpeed = FIXEDPOINT(0);
			break;

		case VK_ADD:
			currentYSpeed = FIXEDPOINT(0);
			break;

		default:
			break;
		}
		return(0);

	case WM_SIZE:	// window size changed
		fwSizeType = uParam;
		if (fwSizeType != SIZE_MINIMIZED) {
            // Skip when this is called before the first DIB
            // section is created
            if (hDIBSection == 0)
                break;

			oldDIBWidth = DIBWidth;
			oldDIBPitch = DIBPitch;
			oldDIBHeight = DIBHeight;

			DIBWidth = LOWORD(lParam);
			DIBPitch = (DIBWidth + 3) & ~3;
			DIBHeight = HIWORD(lParam);

			// Resize the DIB section to the new size
            holdDIBSection = hDIBSection;
			pbmiDIB->bmiHeader.biWidth = DIBPitch;
			pbmiDIB->bmiHeader.biHeight = DIBHeight;

            hdc = GetDC(hwnd);
            hDIBSection = CreateDIBSection (hdc, pbmiDIB,
                    DIB_PAL_COLORS, &pDIB, NULL, 0);

            if (hDIBSection) {
                // Success
                DeleteObject(holdDIBSection);
            } else {
                // Failed, just use old size
    			pbmiDIB->bmiHeader.biWidth = oldDIBPitch;
	    		pbmiDIB->bmiHeader.biHeight = oldDIBHeight;
    			DIBWidth = oldDIBWidth;
				DIBPitch = oldDIBPitch;
    			DIBHeight = oldDIBHeight;
            }

			// Clear the DIB
			memset(pDIB, 0, DIBPitch*DIBHeight);

			fxHalfDIBWidth = FIXEDPOINT(DIBWidth / 2);
			fxHalfDIBHeight = FIXEDPOINT(DIBHeight / 2);
		}
		break;

	case WM_DESTROY:  // message: window being destroyed
		free(pbmiDIB);
        DeleteObject(hDIBSection);
		DeleteObject(hpalold);
                        
        PostQuitMessage(0);
        break;

    default:          // Passes it on if unproccessed
		return (DefWindowProc(hwnd, message, uParam, lParam));
    }
    return (0);
}

/////////////////////////////////////////////////////////////////////
// Parses and loads the BSP tree describing the walls.
/////////////////////////////////////////////////////////////////////
int LoadBSPTree()
{
	int i;
	int version;
	int startvertex, endvertex;
	int fronttree, backtree;
	FILE *infile;
#define BUFLEN	80
	char tempbuf[BUFLEN+1];

	infile = fopen("bsptree.txt", "r");
	if (infile == NULL) {
		DisplayError("Couldn't open BSP file bsptree.txt");
		return(0);
	}

	// Read in the header
	if (fscanf(infile, "%[a-zA-Z0-9 ], %d\n", tempbuf, &version) !=
	        2) {
		DisplayError("Error reading BSP file");
		fclose(infile);
		return(0);
	}

	if (strcmp(tempbuf, "2D BSP Tree Version") != 0) {
		DisplayError("Bad BSP file");
		fclose(infile);
		return(0);
	}

	// Check the version
	if (version != CURRENT_VERSION) {
		sprintf(tempbuf, "Bad BSP file version; expected %d",
		        CURRENT_VERSION);
		DisplayError(tempbuf);
		fclose(infile);
		return(0);
	}

	// Skip the vertices section header, and read the number of
	// vertices
	for (i=0; i<3; i++) {
		if (fgets(tempbuf, BUFLEN, infile) == NULL) {
			DisplayError("Error reading BSP file");
			fclose(infile);
			return(0);
		}
	}

	if (fscanf(infile, "%[a-z :#], %d\n", tempbuf, &numvertices) !=
	        2) {
		DisplayError("Error reading BSP file");
		fclose(infile);
		return(0);
	}

	if (strcmp(tempbuf, "# vertices:") != 0) {
		DisplayError("Bad BSP file");
		fclose(infile);
		return(0);
	}

	if (numvertices < 1) {
		sprintf(tempbuf, "Bad # vertices: %d", numvertices);
		DisplayError(tempbuf);
		fclose(infile);
		return(0);
	}

	pextravertexlist = (VERTEX *)
	        malloc(sizeof(VERTEX)*MAX_NUM_EXTRA_VERTICES);
	if (pextravertexlist == NULL) {
		DisplayError("Couldn't get memory for extra vertex list");
		fclose(infile);
		return(0);
	}

	// Read in the lines
	pvertexlist = (VERTEX *)malloc(sizeof(VERTEX)*numvertices);
	if (pvertexlist == NULL) {
		DisplayError("Couldn't get memory for vertex list");
		fclose(infile);
		return(0);
	}

	for (i=0; i<numvertices; i++) {
		if (fscanf(infile, "%ld,%ld\n",
				&pvertexlist[i].x,
				&pvertexlist[i].z) != 2) {
			DisplayError("Bad line entry");
			free(pvertexlist);
			fclose(infile);
			return(0);
		}
	}

	// Skip the nodes section header, and read the number of nodes
	for (i=0; i<3; i++) {
		if (fgets(tempbuf, BUFLEN, infile) == NULL) {
			DisplayError("Error reading BSP file");
			free(pvertexlist);
			fclose(infile);
			return(0);
		}
	}

	if (fscanf(infile, "%[a-z :#], %d\n", tempbuf, &numnodes) != 2) {
		DisplayError("Error reading BSP file");
		free(pvertexlist);
		fclose(infile);
		return(0);
	}

	if (strcmp(tempbuf, "# nodes:") != 0) {
		DisplayError("Bad BSP file");
		free(pvertexlist);
		fclose(infile);
		return(0);
	}

	// Read in the nodes
	pnodelist = (NODE *)malloc(sizeof(NODE)*numnodes);
	if (pnodelist == NULL) {
		DisplayError("Couldn't get memory for node list");
		free(pvertexlist);
		fclose(infile);
		return(0);
	}

	for (i=0; i<numnodes; i++) {
		if (fscanf(infile, "%d,%d, %ld,%ld, %ld,%ld, %d,%d, %d\n",
				&startvertex,
				&endvertex,
				&pnodelist[i].walltop,
				&pnodelist[i].wallbottom,
				&pnodelist[i].tstart,
				&pnodelist[i].tend,
				&fronttree,
				&backtree,
				&pnodelist[i].color) != 9) {
			DisplayError("Bad node entry");
			free(pvertexlist);
			free(pnodelist);
			fclose(infile);
			return(0);
		}

		// Convert indices into pointers
		pnodelist[i].pstartvertex = &pvertexlist[startvertex];
		pnodelist[i].pendvertex = &pvertexlist[endvertex];
		if (fronttree == NO_CHILD_TREE)
			pnodelist[i].fronttree = NULL;
		else
			pnodelist[i].fronttree = &pnodelist[fronttree];

		if (backtree == NO_CHILD_TREE)
			pnodelist[i].backtree = NULL;
		else
			pnodelist[i].backtree = &pnodelist[backtree];
	}

	// Make sure that's the end of the file, and we're done
	if (fgets(tempbuf, BUFLEN, infile) == NULL) {
		DisplayError("Error reading BSP file");
		free(pvertexlist);
		free(pnodelist);
		fclose(infile);
		return(0);
	}

	if (fgets(tempbuf, BUFLEN, infile) == NULL) {
		DisplayError("Error reading BSP file");
		free(pvertexlist);
		free(pnodelist);
		fclose(infile);
		return(0);
	}

	if (strcmp(tempbuf, "End of file\n") != 0) {
		DisplayError("Bad BSP file");
		free(pvertexlist);
		free(pnodelist);
		fclose(infile);
		return(0);
	}

	fclose(infile);

	return(1);
}

/////////////////////////////////////////////////////////////////////
// Displays a message box.
/////////////////////////////////////////////////////////////////////
void DisplayError(char *text)
{
	MessageBox(hwndOutput, text, NULL, MB_OK);
}

/////////////////////////////////////////////////////////////////////
// Performs a fixed-point multiplication of two 16.16 fixed-point
// numbers.
/////////////////////////////////////////////////////////////////////
FIXEDPOINT FixedMul(FIXEDPOINT x, FIXEDPOINT y)
{
	FIXEDPOINT temp;

	_asm {
		mov 	eax,x
		imul	y
		shrd    eax,edx,16
		mov 	temp,eax
	}
	return(temp);
}

/////////////////////////////////////////////////////////////////////
// Performs a fixed-point multiplication of two 16.16 fixed-point
// numbers, followed by a division by another 16.16 fixed-point
// number. The intermediate result can be bigger than 32 bits (in
// fact, that's why this routine exists), but the result must fit in
// 16.16.
/////////////////////////////////////////////////////////////////////
FIXEDPOINT FixedMulDiv(FIXEDPOINT x, FIXEDPOINT y, FIXEDPOINT z)
{
	FIXEDPOINT temp;

	_asm {
		mov 	eax,x
		imul	y
		idiv	z
		mov 	temp,eax
	}
	return(temp);
}

/////////////////////////////////////////////////////////////////////
// Performs a fixed-point division of two 16.16 fixed-point numbers.
/////////////////////////////////////////////////////////////////////
FIXEDPOINT FixedDiv(FIXEDPOINT x, FIXEDPOINT y)
{
	FIXEDPOINT temp;

	_asm {
		mov 	eax,x
		mov 	edx,eax
		sar 	edx,16	// bias up by 2^16 so result is 16.16
		shl 	eax,16
		idiv	y
		mov 	temp,eax
	}
	return(temp);
}

/////////////////////////////////////////////////////////////////////
// Returns the fixedpoint sine of the angle.
/////////////////////////////////////////////////////////////////////
FIXEDPOINT FixedSin(ANGLE angle)
{
	if (angle > ANGLE(180.0)) {
		if (angle > ANGLE(270.0))
			return(-sintable[ANGLE(360.0) - angle]);
		else
			return(-sintable[angle - ANGLE(180.0)]);
	} else {
		if (angle > ANGLE(90.0))
			return(sintable[ANGLE(180.0) - angle]);
		else
			return(sintable[angle]);
	}
}

/////////////////////////////////////////////////////////////////////
// Returns the fixedpoint cosine of the angle.
/////////////////////////////////////////////////////////////////////
FIXEDPOINT FixedCos(ANGLE angle)
{
	if (angle > ANGLE(180.0)) {
		if (angle > ANGLE(270.0))
			return(sintable[ANGLE(90.0) - (ANGLE(360.0) - angle)]);
		else
			return(-sintable[ANGLE(90.0) - (angle - ANGLE(180.0))]);
	} else {
		if (angle > ANGLE(90.0))
			return(-sintable[ANGLE(90.0) - (ANGLE(180.0) - angle)]);
		else
			return(sintable[ANGLE(90.0)-angle]);
	}
}

/////////////////////////////////////////////////////////////////////
// Returns nonzero if a wall is facing the viewer, 0 else.
/////////////////////////////////////////////////////////////////////
int WallFacingViewer(NODE * pwall)
{
	FIXEDPOINT viewxstart = pwall->pstartvertex->viewx;
	FIXEDPOINT viewzstart = pwall->pstartvertex->viewz;
	FIXEDPOINT viewxend = pwall->pendvertex->viewx;
	FIXEDPOINT viewzend = pwall->pendvertex->viewz;
	int Temp;

/*  // equivalent C code
	if (( ((pwall->pstartvertex->viewx >> 16) *
			((pwall->pendvertex->viewz -
			 pwall->pstartvertex->viewz) >> 16)) +
			((pwall->pstartvertex->viewz >> 16) *
			 ((pwall->pstartvertex->viewx -
			   pwall->pendvertex->viewx) >> 16)) )
					< 0)
		return(1);
	else
		return(0);
*/
	_asm {
		mov 	eax,viewzend
		sub 	eax,viewzstart
		imul    viewxstart
		mov 	ecx,edx
		mov 	ebx,eax
		mov 	eax,viewxstart
		sub 	eax,viewxend
		imul	viewzstart
		add 	eax,ebx
		adc 	edx,ecx
		mov     eax,0
		jns     short WFVDone
		inc 	eax
WFVDone:
		mov     Temp,eax
	}
	return(Temp);
}

/////////////////////////////////////////////////////////////////////
// Update the viewpoint position as needed.
/////////////////////////////////////////////////////////////////////
void UpdateViewPos()
{
	if (currentspeed != 0) {
		currentlocation.x += FixedMul(currentdirection.x,
		                              currentspeed);
		if (currentlocation.x <= WORLD_MIN_X)
			currentlocation.x = WORLD_MIN_X;
		if (currentlocation.x >= WORLD_MAX_X)
			currentlocation.x = WORLD_MAX_X - 1;
		currentlocation.z += FixedMul(currentdirection.z,
		                              currentspeed);
		if (currentlocation.z <= WORLD_MIN_Z)
			currentlocation.z = WORLD_MIN_Z;
		if (currentlocation.z >= WORLD_MAX_Z)
			currentlocation.z = WORLD_MAX_Z - 1;
	}

	if (currentYSpeed != 0) {
		fxViewerY += currentYSpeed;
		if (fxViewerY <= WORLD_MIN_Y)
			fxViewerY = WORLD_MIN_Y;
		if (fxViewerY >= WORLD_MAX_Y)
			fxViewerY = WORLD_MAX_Y - 1;
	}
}

/////////////////////////////////////////////////////////////////////
// Transform all vertices into viewspace.
/////////////////////////////////////////////////////////////////////
void TransformVertices()
{
	VERTEX *pvertex;
	FIXEDPOINT tempx, tempz;
    int vertex;

	pvertex = pvertexlist;
	for (vertex = 0; vertex < numvertices; vertex++) {
		// Translate the vertex according to the viewpoint
		tempx = pvertex->x - currentlocation.x;
		tempz = pvertex->z - currentlocation.z;

		// Rotate the vertex so viewpoint is looking down z axis
		pvertex->viewx = FixedMul(FixedMul(tempx,
		                                   currentorientation.z) +
						 FixedMul(tempz, -currentorientation.x),
						 FIXEDPOINT(PROJECTION_RATIO));
		pvertex->viewz = FixedMul(tempx, currentorientation.x) +
						 FixedMul(tempz, currentorientation.z);
		pvertex++;
	}
}

/////////////////////////////////////////////////////////////////////
// 3D clip all walls. If any part of each wall is still visible,
// transform to perspective viewspace.
/////////////////////////////////////////////////////////////////////
void ClipWalls()
{
	NODE *pwall;
    int wall;
	FIXEDPOINT tempstartx, tempendx, tempstartz, tempendz;
	FIXEDPOINT tempstartwalltop, tempstartwallbottom;
	FIXEDPOINT tempendwalltop, tempendwallbottom;
	VERTEX *pstartvertex, *pendvertex;
	VERTEX *pextravertex = pextravertexlist;

	pwall = pnodelist;
	for (wall = 0; wall < numnodes; wall++) {
		// Assume the wall won't be visible
		pwall->isVisible = 0;

		// Generate the wall endpoints, accounting for t values and
		// clipping

		// Calculate the viewspace coordinates for this wall
		pstartvertex = pwall->pstartvertex;
		pendvertex = pwall->pendvertex;

		// Look for z clipping first
		// Calculate start and end z coordinates for this wall
		if (pwall->tstart == FIXEDPOINT(0))
			tempstartz = pstartvertex->viewz;
		else
			tempstartz = pstartvertex->viewz +
					FixedMul((pendvertex->viewz-pstartvertex->viewz),
					pwall->tstart);

		if (pwall->tend == FIXEDPOINT(1))
			tempendz = pendvertex->viewz;
		else
			tempendz = pstartvertex->viewz +
					FixedMul((pendvertex->viewz-pstartvertex->viewz),
					pwall->tend);

		// Clip to the front plane
		if (tempendz < FrontClipPlane) {
			if (tempstartz < FrontClipPlane) {
				// Fully front-clipped
				goto NextWall;
			} else {
				pwall->clippedtstart = pwall->tstart;

				// Clip the end point to the front clip plane
				pwall->clippedtend =
						FixedDiv(pstartvertex->viewz - FrontClipPlane,
							   pstartvertex->viewz-pendvertex->viewz);
				tempendz = pstartvertex->viewz +
					FixedMul((pendvertex->viewz-pstartvertex->viewz),
					pwall->clippedtend);
			}
		} else {
			pwall->clippedtend = pwall->tend;

			if (tempstartz < FrontClipPlane) {
				// Clip the start point to the front clip plane
				pwall->clippedtstart =
						FixedDiv(FrontClipPlane - pstartvertex->viewz,
							   pendvertex->viewz-pstartvertex->viewz);
				tempstartz = pstartvertex->viewz +
					FixedMul((pendvertex->viewz-pstartvertex->viewz),
					pwall->clippedtstart);
			} else {
				pwall->clippedtstart = pwall->tstart;
			}
		}

		// Calculate x coordinates
		if (pwall->clippedtstart == FIXEDPOINT(0))
			tempstartx = pstartvertex->viewx;
		else
			tempstartx = pstartvertex->viewx +
					FixedMul((pendvertex->viewx-pstartvertex->viewx),
					pwall->clippedtstart);

		if (pwall->clippedtend == FIXEDPOINT(1))
			tempendx = pendvertex->viewx;
		else
			tempendx = pstartvertex->viewx +
					FixedMul((pendvertex->viewx-pstartvertex->viewx),
					pwall->clippedtend);

		// Clip in x as needed
		if ((tempstartx > tempstartz) || (tempstartx < -tempstartz)) {
			// The start point is outside the view triangle in x;
			// perform a quick test for trivial rejection by seeing if
			// the end point is outside the view triangle on the same
			// side as the start point
			if (((tempstartx>tempstartz) && (tempendx>tempendz)) ||
				((tempstartx<-tempstartz) && (tempendx<-tempendz)))
				// Fully clipped--trivially reject
				goto NextWall;

			// Clip the start point
			if (tempstartx > tempstartz) {
				// Clip the start point on the right side
				pwall->clippedtstart =
					FixedDiv(pstartvertex->viewx-pstartvertex->viewz,
							 pendvertex->viewz-pstartvertex->viewz -
							 pendvertex->viewx+pstartvertex->viewx);
				tempstartx = pstartvertex->viewx +
					FixedMul((pendvertex->viewx-pstartvertex->viewx),
						 	  pwall->clippedtstart);
				tempstartz = tempstartx;
			} else {
				// Clip the start point on the left side
				pwall->clippedtstart =
					FixedDiv(-pstartvertex->viewx-pstartvertex->viewz,
							 pendvertex->viewx+pendvertex->viewz -
							 pstartvertex->viewz-pstartvertex->viewx);
				tempstartx = pstartvertex->viewx +
					FixedMul((pendvertex->viewx-pstartvertex->viewx),
						 	  pwall->clippedtstart);
				tempstartz = -tempstartx;
			}
		}

		// See if the end point needs clipping
		if ((tempendx > tempendz) || (tempendx < -tempendz)) {
			// Clip the end point
			if (tempendx > tempendz) {
				// Clip the end point on the right side
				pwall->clippedtend =
					FixedDiv(pstartvertex->viewx-pstartvertex->viewz,
							 pendvertex->viewz-pstartvertex->viewz -
							 pendvertex->viewx+pstartvertex->viewx);
				tempendx = pstartvertex->viewx +
					FixedMul((pendvertex->viewx-pstartvertex->viewx),
						 	  pwall->clippedtend);
				tempendz = tempendx;
			} else {
				// Clip the end point on the left side
				pwall->clippedtend =
					FixedDiv(-pstartvertex->viewx-pstartvertex->viewz,
							 pendvertex->viewx+pendvertex->viewz -
							 pstartvertex->viewz-pstartvertex->viewx);
				tempendx = pstartvertex->viewx +
					FixedMul((pendvertex->viewx-pstartvertex->viewx),
						 	  pwall->clippedtend);
				tempendz = -tempendx;
			}
		}

		tempstartwalltop = FixedMul((pwall->walltop - fxViewerY),
				FIXEDPOINT(PROJECTION_RATIO));
		tempendwalltop = tempstartwalltop;
		tempstartwallbottom = FixedMul((pwall->wallbottom-fxViewerY),
				FIXEDPOINT(PROJECTION_RATIO));
		tempendwallbottom = tempstartwallbottom;

		// Partially clip in y (the rest is done later in 2D)
		// Check for trivial accept
		if ((tempstartwalltop > tempstartz) ||
			(tempstartwallbottom < -tempstartz) ||
			(tempendwalltop > tempendz) ||
			(tempendwallbottom < -tempendz)) {
			// Not trivially unclipped; check for fully clipped
			if ((tempstartwallbottom > tempstartz) &&
				(tempstartwalltop < -tempstartz) &&
				(tempendwallbottom > tempendz) &&
				(tempendwalltop < -tempendz)) {
				// Outside view triangle, trivially clipped
				goto NextWall;
			}

			// Partially clipped in Y; we'll do Y clipping at
			// drawing time
		}

		// The wall is visible; mark it as such and project it.
		// +1 on scaling because of bottom/right exclusive polygon
		// filling
		pwall->isVisible = 1;

		pwall->screenxstart =
			(FixedMulDiv(tempstartx, fxHalfDIBWidth+FIXEDPOINT(0.5),
				tempstartz) + fxHalfDIBWidth + FIXEDPOINT(0.5));
		pwall->screenytopstart =
				(FixedMulDiv(tempstartwalltop,
				fxHalfDIBHeight + FIXEDPOINT(0.5), tempstartz) +
				fxHalfDIBHeight + FIXEDPOINT(0.5));
		pwall->screenybottomstart =
				(FixedMulDiv(tempstartwallbottom,
				fxHalfDIBHeight + FIXEDPOINT(0.5), tempstartz) +
				fxHalfDIBHeight + FIXEDPOINT(0.5));

		pwall->screenxend =
				(FixedMulDiv(tempendx, fxHalfDIBWidth+FIXEDPOINT(0.5),
				tempendz) + fxHalfDIBWidth + FIXEDPOINT(0.5));
		pwall->screenytopend =
				(FixedMulDiv(tempendwalltop,
				fxHalfDIBHeight + FIXEDPOINT(0.5), tempendz) +
				fxHalfDIBHeight + FIXEDPOINT(0.5));
		pwall->screenybottomend =
				(FixedMulDiv(tempendwallbottom,
				fxHalfDIBHeight + FIXEDPOINT(0.5), tempendz) +
				fxHalfDIBHeight + FIXEDPOINT(0.5));
NextWall:
		pwall++;
	}
}

/////////////////////////////////////////////////////////////////////
// Walk the tree back to front; backface cull whenever possible,
// and draw front-facing walls in back-to-front order.
/////////////////////////////////////////////////////////////////////
void DrawWallsBackToFront()
{
	NODE *pFarChildren, *pNearChildren, *pwall;
	NODE *pendingnodes[MAX_NUM_NODES];
	NODE **pendingstackptr;
	POINT2INT apoint[4];

	pwall = pnodelist;
	pendingnodes[0] = (NODE *)NULL;
	pendingstackptr = pendingnodes + 1;

	for (;;) {
		for (;;) {
			// Descend as far as possible toward the back,
			// remembering the nodes we pass through on the way

           	// Figure whether this wall is facing frontward or
			// backward; do in viewspace because non-visible walls
			// aren't projected into screenspace, and we need to
            // traverse all walls in the BSP tree, visible or not,
            // in order to find all the visible walls
			if (WallFacingViewer(pwall)) {
				// We're on the forward side of this wall, do the back
				// children first
				pFarChildren = pwall->backtree;
			} else {
				// We're on the back side of this wall, do the front
				// children first
				pFarChildren = pwall->fronttree;
			}

			if (pFarChildren == NULL)
				break;
#if DBG
			if (pendingstackptr >= (pendingnodes + MAX_NUM_NODES)) {
				DisplayError("UpdateWorld: Pending stack overflow");
				pendingstackptr = pendingnodes + MAX_NUM_NODES - 1;
									// fix up well enough to not crash
			}
#endif
			*pendingstackptr = pwall;
			pendingstackptr++;
			pwall = pFarChildren;
		}

		for (;;) {
			// See if the wall is even visible
			if (pwall->isVisible)
			{
				// See if we can backface cull this wall
				if (pwall->screenxstart < pwall->screenxend)
				{
					// Draw the wall
					apoint[0].x = FIXTOINT(pwall->screenxstart);
					apoint[1].x = FIXTOINT(pwall->screenxstart);
					apoint[2].x = FIXTOINT(pwall->screenxend);
					apoint[3].x = FIXTOINT(pwall->screenxend);
					apoint[0].y = FIXTOINT(pwall->screenytopstart);
					apoint[1].y = FIXTOINT(pwall->screenybottomstart);
					apoint[2].y = FIXTOINT(pwall->screenybottomend);
					apoint[3].y = FIXTOINT(pwall->screenytopend);
					FillConvexPolygon(apoint, pwall->color);
				}
			}

			// If there's a near tree from this node, draw it;
			// otherwise, work back up to the last-pushed parent
			// node of the branch we just finished; we're done if
			// there are no pending parent nodes

			// Figure whether this wall is facing frontward or
			// backward; do in viewspace because non-visible walls
			// aren't projected into screenspace, and we need to
            // traverse all walls in the BSP tree, visible or not,
            // in order to find all the visible walls
			if (WallFacingViewer(pwall)) {
				// We're on the forward side of this wall, do the
				// front children now
				pNearChildren = pwall->fronttree;
			} else {
				// We're on the back side of this wall, do the back
				// children now
				pNearChildren = pwall->backtree;
			}

			// Walk the near subtree of this wall
			if (pNearChildren != NULL)
				goto WalkNearTree;
#if DBG
			if (pendingstackptr <= pendingnodes) {
				DisplayError("UpdateWorld: Pending stack underflow");
				pendingstackptr = pendingnodes + 1;
								// fix up well enough to not crash
			}
#endif      
			// Pop the last-pushed wall
			pendingstackptr--;
			pwall = *pendingstackptr;
			if (pwall == NULL)
				goto NodesDone;
		}

WalkNearTree:

		pwall = pNearChildren;
	}

NodesDone:
;
}

/////////////////////////////////////////////////////////////////////
// Render the current state of the world to the screen.
/////////////////////////////////////////////////////////////////////
void UpdateWorld()
{
	HPALETTE holdpal;
    HDC hdcScreen, hdcDIBSection;
    HBITMAP holdbitmap;

    UpdateViewPos();
    
	memset(pDIB, 0, DIBPitch*DIBHeight);    // clear frame

    TransformVertices();

    ClipWalls();

    DrawWallsBackToFront();

	// We've drawn the frame; copy it to the screen
	hdcScreen = GetDC(hwndOutput);
	holdpal = SelectPalette(hdcScreen, hpalDIB, FALSE);
	RealizePalette(hdcScreen);

    hdcDIBSection = CreateCompatibleDC(hdcScreen);
    holdbitmap = SelectObject(hdcDIBSection, hDIBSection);

    BitBlt(hdcScreen, 0, 0, DIBWidth, DIBHeight, hdcDIBSection,
           0, 0, SRCCOPY);

	SelectPalette(hdcScreen, holdpal, FALSE);
	ReleaseDC(hwndOutput, hdcScreen);
    SelectObject(hdcDIBSection, holdbitmap);
    ReleaseDC(hwndOutput, hdcDIBSection);
	iteration++;
}
