;//////////////////////
;// Object structure //
;//////////////////////
Object STRUC

       ID_Number                dw ?    ;engine version
       Number_Of_Vertices       dw ?    ;number of vertices
       Number_Of_Faces          dw ?    ;number of faces

       Size_Of_Vertices         dd ?    ;ammount of memory needed for vertices
       Size_Of_Faces            dd ?    ;ammount of memory needed for faces

       Shade_Limit              db ?    ;0=flat 1=gouraud 2=texture 3=envmap
       Eye_Distance             dd ?    ;distance to eye
       Z_OffSet                 dd ?    ;Z offset (define size of object)

       Angle_X                  dd ?    ;rotating angles of the object
       Angle_Y                  dd ?
       Angle_Z                  dd ?

       OffSet_VArray            dd ?    ;offset to vertices
       OffSet_NArray            dd ?    ;offset to normals
       OffSet_VArray_Rotated    dd ?    ;offset to rotated vertices
       OffSet_NArray_Rotated    dd ?    ;offset to rotated normals
       OffSet_VArray_Projected  dd ?    ;offset to projected vertices
       OffSet_Faces             dd ?    ;offset to faces
       OffSet_FNormals          dd ?    ;offset to face normals
       OffSet_FNormals_Rotated  dd ?    ;offset to rotated face normals
       OffSet_Texture           dd ?    ;offset to texture map

Object ENDS

;;
;                                                                          ;
; Name       : Load_Object                                                 ;
; Description: Load 3D object from disk                                    ;
; Input      : EDX = file name                                             ;
; Output     : /                                                           ;
; Uses       : (E)AX,(E)BX,ECX,EDX                                         ;
;                                                                          ;
;;
            align 4
Load_Object proc near
            ;=== OPEN FILE ===
            mov        ax,3d00h
            int        21h
            jc         @@Error_Opening_File
            mov        [File_Handle],ax
            ;=== READ HEADER ===
            mov        ax,3f00h
            mov        bx,[File_Handle]
            mov        ecx,6
            mov        edx,OFFSET Object_3D
            int        21h
            jc         @@Error_Reading_File
            mov        [Object_3D.Eye_Distance],256
            mov        [Object_3D.Z_OffSet],256
            ;=== CHECK VERSION OF ENGINE ===
            movzx      eax,[Object_3D.ID_Number]
            cmp        eax,[ENGINE_VERSION]
            jne        @@Error_Version
            ;=== CALCULATE AMMOUNT OF MEMORY NEEDED FOR VERICES AND FACES ===
            movzx      edx,[Object_3D.Number_Of_Vertices]
            mov        eax,edx
            add        edx,edx
            add        edx,eax
            shl        edx,2
            mov        [Object_3D.Size_Of_Vertices],edx
            movzx      edx,[Object_3D.Number_Of_Faces]
            mov        eax,edx
            add        edx,edx
            add        edx,eax
            shl        edx,2
            mov        [Object_3D.Size_Of_Faces],edx
            call       Init_Object
            ;=== LOAD VERTICES ===
            mov        ax,3f00h
            mov        bx,[File_Handle]
            mov        ecx,[Object_3D.Size_Of_Vertices]
            mov        edx,[Object_3D.OffSet_VArray]
            int        21h
            jc         @@Error_Reading_File
            ;=== LOAD VERTEX NORMALS ===
            mov        ax,3f00h
            mov        bx,[File_Handle]
            mov        ecx,[Object_3D.Size_Of_Vertices]
            mov        edx,[Object_3D.OffSet_NArray]
            int        21h
            jc         @@Error_Reading_File
            ;=== LOAD FACES ===
            mov        ax,3f00h
            mov        bx,[File_Handle]
            mov        ecx,[Object_3D.Size_Of_Faces]
            mov        edx,[Object_3D.OffSet_Faces]
            int        21h
            jc         @@Error_Reading_File
            ;=== LOAD FACE NORMALS ===
            mov        ax,3f00h
            mov        bx,[File_Handle]
            mov        ecx,[Object_3D.Size_Of_Faces]
            mov        edx,[Object_3D.OffSet_FNormals]
            int        21h
            jc         @@Error_Reading_File
            ;=== CLOSE FILE ===
            mov        ax,3e00h
            mov        bx,[File_Handle]
            jc         @@Error_Closing_File
            int        21h
            ret
            ;=== ERROR WHILE OPENING FILE ===
            @@Error_Opening_File:
            lea        edx,Error_Opening_File
            call       Error
            ;=== ERROR WHILE READING FROM FILE ===
            @@Error_Reading_File:
            lea        edx,Error_Reading_File
            call       Error
            ;=== ERROR WHILE CLOSING FILE ===
            @@Error_Closing_File:
            lea        edx,Error_Closing_File
            call       Error
            ;=== ERROR IN VERSION OF ENGINE ===
            @@Error_Version:
            lea        edx,Error_Version
            call       Error
Load_Object endp

;;
;                                                                          ;
; Name       : Init_Object                                                 ;
; Description: Init memory for object                                      ;
; Input      : /                                                           ;
; Output     : /                                                           ;
; Uses       : (E)AX,EDX                                                   ;
;                                                                          ;
;;
            align 4
Init_Object proc near
            ;=== ALLOCATE MEMORY FOR VERTICES ===
            mov        ax,0ee42h
            mov        edx,[Object_3D.Size_Of_Vertices]
            int        31h
            jc         @@Error_Allocating_Memory
            mov        Object_3D.OffSet_VArray,edx
            ;=== ALLOCATE MEMORY FOR VERTEX NORMALS ===
            mov        ax,0ee42h
            mov        edx,[Object_3D.Size_Of_Vertices]
            int        31h
            jc         @@Error_Allocating_Memory
            mov        Object_3D.OffSet_NArray,edx
            ;=== ALLOCATE MEMORY FOR FACES ===
            mov        ax,0ee42h
            mov        edx,[Object_3D.Size_Of_Faces]
            int        31h
            jc         @@Error_Allocating_Memory
            mov        Object_3D.OffSet_Faces,edx
            ;=== ALLOCATE MEMORY FOR FACE NORMALS ===
            mov        ax,0ee42h
            mov        edx,[Object_3D.Size_Of_Faces]
            int        31h
            jc         @@Error_Allocating_Memory
            mov        Object_3D.OffSet_FNormals,edx
            ;=== ALLOCATE MEMORY FOR ROTATED VERTICES ===
            mov        ax,0ee42h
            mov        edx,[Object_3D.Size_Of_Vertices]
            int        31h
            jc         @@Error_Allocating_Memory
            mov        Object_3D.OffSet_VArray_Rotated,edx
            ;=== ALLOCATE MEMORY FOR ROTATED VERTEX NAORMALS ===
            mov        ax,0ee42h
            mov        edx,[Object_3D.Size_Of_Vertices]
            int        31h
            jc         @@Error_Allocating_Memory
            mov        Object_3D.OffSet_NArray_Rotated,edx
            ;=== ALLOCATE MEMORY FOR PROJECTED VERTICES ===
            mov        ax,0ee42h
            movzx      edx,[Object_3D.Number_Of_Vertices]
            shl        edx,3
            int        31h
            jc         @@Error_Allocating_Memory
            mov        Object_3D.OffSet_VArray_Projected,edx
            ;=== ALLOCATE MEMORY FOR ROTATE FACE NORMALS ===
            mov        ax,0ee42h
            mov        edx,[Object_3D.Size_Of_Faces]
            int        31h
            jc         @@Error_Allocating_Memory
            mov        Object_3D.OffSet_FNormals_Rotated,edx
            ret
            ;=== ERROR WHILE ALLOCATING MEMORY ===
            @@Error_Allocating_Memory:
            lea        edx,Error_Allocating_Memory
            call       Error
Init_Object endp


;;
;                                                                          ;
; Name       : Load_Texture                                                ;
; Description: Allocate memory and load texture into allocated memory      ;
; Input      : EDX = file name                                             ;
; Output     : /                                                           ;
; Uses       : (E)AX,EDX                                                   ;
;                                                                          ;
;;
             align 4
Load_Texture proc near
             ;=== OPEN FILE ===
             mov        ax,3d00h
             int        21h
             jc         @@Error_Opening_File
             mov        [File_Handle],ax
             ;=== ALLOCATE MEMORY FOR TEXTURE MAP ===
             mov        ax,0ee42h
             mov        edx,[Texture_256x256_Size]
             int        31h
             jc         @@Error_Allocating_Memory
             mov        Object_3D.OffSet_Texture,edx
             ;=== LOAD TEXTURE ===
             mov        ax,3f00h
             mov        bx,[File_Handle]
             mov        ecx,[Texture_256x256_Size]
             mov        edx,[Object_3D.OffSet_Texture]
             int        21h
             jc         @@Error_Reading_File
             ;=== CLOSE FILE ===
             mov        ax,3e00h
             mov        bx,[File_Handle]
             jc         @@Error_Closing_File
             int        21h
             ret

Load_Texture endp



File_Handle          dw     ?
Object_3D            Object ?
Texture_256x256_Size equ 65535

include ../includes/error.inc
include ../includes/flattri.inc
include ../includes/gourtri.inc
include ../includes/sorty.inc
include ../includes/palette.inc
include ../includes/random.inc
include ../includes/rotate3d.inc
include ../includes/texttri.inc
include ../includes/variable.inc
include ../includes/clip.inc
include ../data/sincos.dat
