'*****************************************************************************
'                                                                            *
'  EFFECTS.BAS                                                               *
'                                                                            *
'  This program demonstrates several methods of fading in an image from an   *
'  off-screen video page using either Fastgraph or Fastgraph/Light.  The set *
'  of routines provided herein are written for 320 x 200 graphics video      *
'  modes, but they could easily be extended to work in other resolutions.    *
'                                                                            *
'  The examples are by no means all inclusive.  Rather, their purpose is to  *
'  illustrate a few methods of creating special effects with Fastgraph or    *
'  Fastgraph/Light.                                                          *
'                                                                            *
'  To compile this program and link it with Fastgraph:                       *
'                                                                            *
'     BC EFFECTS;                                                            *
'     LINK EFFECTS,,NUL,FGQB                                                 *
'                                                                            *
'  This program also can be linked with Fastgraph/Light if you replace the   *
'  FGQB library reference with FGLQB.                                        *
'                                                                            *
'  Fastgraph (tm) and Fastgraph/Light (tm) are graphics libraries published  *
'  by Ted Gruber Software.  For more info, please call, write, or FAX.       *
'                                                                            *
'  Ted Gruber Software                           orders/info (702) 735-1980  *
'  PO Box 13408                                          FAX (702) 735-4603  *
'  Las Vegas, NV  89112                                  BBS (702) 796-7134  *
'                                                                            *
'*****************************************************************************

REM $INCLUDE: 'fastgraf.bi'

DEFINT A-Z

REM function declarations

DECLARE SUB Announce (Message$)
DECLARE FUNCTION Irandom (Min, Max)

DECLARE SUB Curtain (Delay)
DECLARE SUB DiagonalFade (Delay)
DECLARE SUB HorizontalRandomFade (Delay)
DECLARE SUB InwardTunnelEffect (Delay)
DECLARE SUB OutwardTunnelEffect (Delay)
DECLARE SUB SpiralDual (Delay)
DECLARE SUB SpiralLayered (Delay)
DECLARE SUB SpiralNormal (Delay)
DECLARE SUB SplitScreen (Delay)
DECLARE SUB Unveil (Delay)
DECLARE SUB VenetianBlind (Delay)

REM global variables

COMMON SHARED Delay, ScrollDelay

REM start of main program

REM make sure a 320 x 200 color graphics mode is available

NewMode = FGbestmode(320,200,2)
IF (NewMode < 0) OR (NewMode = 12) THEN
   PRINT "This program requires a 320 x 200 color graphics mode."
   STOP
END IF

REM determine the number of delay units per half clock tick

Delay = FGmeasure / 2

REM initialize Fastgraph for the selected video mode

OldMode = FGgetmode
FGsetmode NewMode
Status = FGallocate(1)

REM display a packed pixel run file on a hidden page

FGsethpage 1
FGsetpage 1
FGmove 0, 199
FGdispfile "FG.PPR"+CHR$(0), 320, 1
FGsetpage 0

REM compute the number of delay units needed to make the text scroll
REM down at the same rate, regardless of the CPU speed or video mode

Count = 0
FGwaitfor 1
StartTime& = FGgetclock
DO
   FGscroll 0, 319, 0, 7, 4, 1
   Count = Count + 1
LOOP WHILE (FGgetclock = StartTime&)

ScrollDelay = (Delay / 8) - (Delay * 2) / Count
IF (ScrollDelay < 0) THEN ScrollDelay = 0

REM demonstrate the inward tunnel effect

Announce "inward tunnel effect"
InwardTunnelEffect 0
FGwaitfor 27
Announce "inward tunnel effect with delay"
InwardTunnelEffect Delay
FGwaitfor 27

REM demonstrate the outward tunnel effect

Announce "outward tunnel effect"
OutwardTunnelEffect 0
FGwaitfor 27
Announce "outward tunnel effect with delay"
OutwardTunnelEffect Delay
FGwaitfor 27

REM demonstrate the diagonal fade

Announce "diagonal fade"
DiagonalFade 0
FGwaitfor 27
Announce "diagonal fade with delay"
DiagonalFade Delay/2
FGwaitfor 27

REM demonstrate the horizontal random fade

Announce "horizontal random fade"
HorizontalRandomFade Delay
FGwaitfor 27

REM demonstrate the curtain effect

Announce "curtain"
Curtain Delay/8
FGwaitfor 27

REM demonstrate the spiral effect

Announce "spiral"
SpiralNormal Delay*2
FGwaitfor 27

REM demonstrate the layered spiral effect

Announce "layered spiral"
SpiralLayered Delay
FGwaitfor 27

REM demonstrate the dual spiral effect

Announce "dual spiral"
SpiralDual Delay/2
FGwaitfor 27

REM demonstrate the split screen effect

Announce "split screen"
SplitScreen Delay/2
FGwaitfor 27

REM demonstrate the unveil effect

Announce "unveil"
Unveil Delay/4
FGwaitfor 27

REM demonstrate the "venetian blind" effect

Announce "venetian blind"
VenetianBlind Delay
FGwaitfor 27

REM restore the original video mode and screen attributes

Status = FGfreepage(1)
FGsetmode OldMode
FGreset

END

'*****************************************************************************
'                                                                            *
'  Announce                                                                  *
'                                                                            *
'  Display the name of the special effect we're about to see.                *
'                                                                            *
'*****************************************************************************

SUB Announce (Message$)

SHARED Delay, ScrollDelay

REM clear the screen

FGerase

REM display the specified message at the top row

FGsetcolor 15
Length = LEN(Message$)
FGlocate 0, 20-Length/2
FGtext Message$, Length

REM scroll the message to the center of the screen

FGsetcolor 0

FOR Y = 0 TO 95 STEP 4
   FGscroll 0, 319, Y, Y+7, 4, 1
   FGstall ScrollDelay
NEXT

REM wait 1.5 seconds

FGwaitfor 27

END SUB

'*****************************************************************************
'                                                                            *
'  Irandom                                                                   *
'                                                                            *
'  Random number generator used in some of the effects.  It returns an       *
'  integer between min and max inclusive.                                    *
'                                                                            *
'*****************************************************************************

FUNCTION Irandom (Min, Max)

Irandom = ((RND*32767) MOD (Max-Min+1)) + Min

END FUNCTION

'*****************************************************************************
'                                                                            *
'  Curtain                                                                   *
'                                                                            *
'  Reveal each row, one at a time, starting from the bottom and proceeding   *
'  to the top.  This gives the effect of a curtain rising, hence the name.   *
'                                                                            *
'*****************************************************************************

SUB Curtain (Delay)

FOR Y = 199 TO 0 STEP -1
   FGrestore 0, 319, Y, Y
   FGstall Delay
NEXT

END SUB

'*****************************************************************************
'                                                                            *
'  DiagonalFade                                                              *
'                                                                            *
'  This reveals the hidden page in two diagonal segments, separated by an    *
'  imaginary line extending from the lower left corner to the upper right    *
'  corner of the screen.  We start with the top line of the left segment and *
'  the bottom line of the right segment, and continue until the entire       *
'  screen is revealed.                                                       *
'                                                                            *
'*****************************************************************************

SUB DiagonalFade (Delay)

Xmin = 0
Xmax = 319
Ymin = 0
Ymax = 199

WHILE (Xmax > 0)
   FGrestore 0, Xmax, Ymin, Ymin+4
   FGrestore Xmin, 319, Ymax-4, Ymax
   FGstall Delay

   Xmin = Xmin + 8
   Xmax = Xmax - 8
   Ymin = Ymin + 5
   Ymax = Ymax - 5
WEND

END SUB

'*****************************************************************************
'                                                                            *
'  HorizontalRandomFade                                                      *
'                                                                            *
'  In this effect, the screen is divided into a series of two-pixel high     *
'  rows.  Each row is revealed in random parts from left to right.  This     *
'  process repeats 20 times, once for each row.  At the end, a call to the   *
'  FGrestore routine guarantees that all rows are transferred.               *
'                                                                            *
'*****************************************************************************

SUB HorizontalRandomFade (Delay)

DIM Xpos(100)

FOR J = 0 TO 99
   Xpos(J) = 0
NEXT

FOR I = 1 TO 20
   FOR J = 0 TO 99
      Xmin = Xpos(J)
      IF (Xmin < 320) THEN
         Xmax = Xmin + Irandom(1,10) * 8
         IF (Xmax > 320) THEN Xmax = 320
         Y = J * 2
         FGrestore Xmin, Xmax-1, Y, Y+1
         Xpos(J) = Xmax
      END IF
   NEXT
   FGstall Delay
NEXT

REM make sure we got them all

FGrestore 0, 319, 0, 199

END SUB

'*****************************************************************************
'                                                                            *
'  InwardTunnelEffect                                                        *
'                                                                            *
'  Starting at the screen edges, reveal the screen through a series of       *
'  concentric hollow rectangles.                                             *
'                                                                            *
'*****************************************************************************

SUB InwardTunnelEffect (Delay)

Xmin = 0
Xmax = 319
Ymin = 0
Ymax = 199

WHILE (Xmin < Xmax)
   FGrestore 0, 319, Ymin, Ymin+4
   FGrestore Xmax-7, Xmax, 0, 199
   FGrestore 0, 319, Ymax-4, Ymax
   FGrestore Xmin, Xmin+7, 0, 199
   FGstall Delay

   Xmin = Xmin + 8
   Xmax = Xmax - 8
   Ymin = Ymin + 5
   Ymax = Ymax - 5
WEND

END SUB

'*****************************************************************************
'                                                                            *
'  OutwardTunnelEffect                                                       *
'                                                                            *
'  Starting at the screen center, reveal the screen through a series of      *
'  concentric hollow rectangles.                                             *
'                                                                            *
'*****************************************************************************

SUB OutwardTunnelEffect (Delay)

Xmin = 152
Xmax = 167
Ymin = 95
Ymax = 104

WHILE (Xmin >= 0)
   FGrestore Xmin, Xmax, Ymin, Ymin+5
   FGrestore Xmax-7, Xmax, Ymin, Ymax
   FGrestore Xmin, Xmax, Ymax-4, Ymax
   FGrestore Xmin, Xmin+7, Ymin, Ymax
   FGstall Delay

   Xmin = Xmin - 8
   Xmax = Xmax + 8
   Ymin = Ymin - 5
   Ymax = Ymax + 5
WEND

END SUB

'*****************************************************************************
'                                                                            *
'  SpiralDual                                                                *
'                                                                            *
'  In this effect, we reveal the screen through two spirals.  One spiral     *
'  emanates clockwise from the screen edges to the screen center, while the  *
'  other emanates counterclockwise from the center to the screen edges.      *
'                                                                            *
'*****************************************************************************

SUB SpiralDual (Delay)

XminOuter = 0
XmaxOuter = 319
YminOuter = 0
YmaxOuter = 199

XminInner = 152
XmaxInner = 167
YminInner = 95
YmaxInner = 104

WHILE (XminOuter < XminInner)
   FGrestore XminOuter, XmaxOuter, YminOuter, YminOuter+4
   FGstall Delay
   FGrestore XminInner, XmaxInner, YmaxInner-4, YmaxInner
   FGstall Delay
   FGrestore XmaxOuter-7, XmaxOuter, YminOuter, YmaxOuter
   FGstall Delay
   FGrestore XmaxInner+1, XmaxInner+8, YminInner, YmaxInner
   FGstall Delay
   FGrestore XminOuter, XmaxOuter, YmaxOuter-4, YmaxOuter
   FGstall Delay
   FGrestore XminInner-8, XmaxInner, YminInner, YminInner+4
   FGstall Delay
   FGrestore XminOuter, XminOuter+7, YminOuter, YmaxOuter
   FGstall Delay
   FGrestore XminInner-8, XminInner-1, YminInner, YmaxInner+5
   FGstall Delay

   XminOuter = XminOuter + 8
   XmaxOuter = XmaxOuter - 8
   YminOuter = YminOuter + 5
   YmaxOuter = YmaxOuter - 5

   XminInner = XminInner - 8
   XmaxInner = XmaxInner + 8
   YminInner = YminInner - 5
   YmaxInner = YmaxInner + 5
WEND

END SUB

'*****************************************************************************
'                                                                            *
'  SpiralLayered                                                             *
'                                                                            *
'  This effect is similar to the normal spiral.  Instead of revealing the    *
'  screen in one iteration, this effect does so in four iterations (layers), *
'  each moving more toward the screen center.                                *
'                                                                            *
'*****************************************************************************

SUB SpiralLayered (Delay)

FOR I = 0 TO 3
   Xmin = I * 8
   Xmax = 319 - Xmin
   Ymin = I * 5
   Ymax = 199 - Ymin

   WHILE (Xmin < Xmax)
      FGrestore Xmin, Xmax, Ymin, Ymin+4
      FGstall Delay
      FGrestore Xmax-7, Xmax, Ymin, Ymax
      FGstall Delay
      FGrestore Xmin, Xmax, Ymax-4, Ymax
      FGstall Delay
      FGrestore Xmin, Xmin+7, Ymin, Ymax
      FGstall Delay

      Xmin = Xmin + 32
      Xmax = Xmax - 32
      Ymin = Ymin + 20
      Ymax = Ymax - 20
   WEND
NEXT

END SUB

'*****************************************************************************
'                                                                            *
'  SpiralNormal                                                              *
'                                                                            *
'  This is a spiral effect in which we reveal the screen as a series of      *
'  rectangles, emanating from the screen edges and proceeding clockwise to   *
'  the center of the screen.                                                 *
'                                                                            *
'*****************************************************************************

SUB SpiralNormal (Delay)

Xmin = 0
Xmax = 319
Ymin = 0
Ymax = 199

WHILE (Xmin < Xmax)
   FGrestore Xmin, Xmax, Ymin, Ymin+19
   FGstall Delay
   FGrestore Xmax-31, Xmax, Ymin, Ymax
   FGstall Delay
   FGrestore Xmin, Xmax, Ymax-19, Ymax
   FGstall Delay
   FGrestore Xmin, Xmin+31, Ymin, Ymax
   FGstall Delay

   Xmin = Xmin + 32
   Xmax = Xmax - 32
   Ymin = Ymin + 20
   Ymax = Ymax - 20
WEND

END SUB

'*****************************************************************************
'                                                                            *
'  SplitScreen                                                               *
'                                                                            *
'  Reveal the top half of from left to right while revealing the bottom half *
'  from right to left.                                                       *
'                                                                            *
'*****************************************************************************

SUB SplitScreen (Delay)

Xmin = 0
Xmax = 319

WHILE (Xmax > 0)
   FGrestore Xmin, Xmin+7, 0, 99
   FGrestore Xmax-7, Xmax, 100, 199
   FGstall Delay
   Xmin = Xmin + 8
   Xmax = Xmax - 8
WEND

END SUB

'*****************************************************************************
'                                                                            *
'  Unveil                                                                    *
'                                                                            *
'  Starting at the center, reveal the screen in small horizontal increments  *
'  until we reach the left and right edges.                                  *
'                                                                            *
'*****************************************************************************

SUB Unveil (Delay)

Xmin = 152
Xmax = 167

WHILE (Xmin >= 0)
   FGrestore Xmin, Xmin+7, 0, 199
   FGrestore Xmax-7, Xmax, 0, 199
   FGstall Delay
   Xmin = Xmin - 8
   Xmax = Xmax + 8
WEND

END SUB

'*****************************************************************************
'                                                                            *
'  VenetianBlind                                                             *
'                                                                            *
'  Reveal the screen in four iterations, each revealing every fourth row.    *
'  The effect produced resembles opening a Venetian blind.                   *
'                                                                            *
'*****************************************************************************

SUB VenetianBlind (Delay)

FOR Y = 0 TO 199 STEP 4
   FGrestore 0, 319, Y, Y
NEXT
FGstall Delay

FOR Y = 1 TO 199 STEP 4
   FGrestore 0, 319, Y, Y
NEXT
FGstall Delay

FOR Y = 2 TO 199 STEP 4
   FGrestore 0, 319, Y, Y
NEXT
FGstall Delay

FOR Y = 3 TO 199 STEP 4
   FGrestore 0, 319, Y, Y
NEXT
FGstall Delay

END SUB
