/****************************************************************************\
*                                                                            *
*  FISHTANK.C                                                                *
*                                                                            *
*  This program shows how to perform simple animation using Fastgraph for    *
*  Windows. Several types of tropical fish swim back and forth against a     *
*  coral reef background. The background image and fish sprites are stored   *
*  in standard pixel run (SPR) files.                                        *
*                                                                            *
*  The application window is initially set to 320x200 pixels, the same size  *
*  as the CORAL.SPR image. When the window size changes, the image is scaled *
*  to fill the window.                                                       *
*                                                                            *
\****************************************************************************/

#include <fgwin.h>
#include <stdlib.h>

#define MAX(x,y) ((x) > (y)) ? (x) : (y)
#define MIN(x,y) ((x) < (y)) ? (x) : (y)

#define NFISH 11

long WINAPI _export WindowProc(HWND,UINT,UINT,LONG);
void get_fish(void);
void go_fish(void);
void put_fish(int,int,int,int);

int PASCAL WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance,
                    LPSTR lpszCmdParam, int nCmdShow)
{
   static char szAppName[] = "FGfishtank";
   HWND        window;
   MSG         msg;
   WNDCLASS    wndclass;

   if (!hPrevInstance)
   {
      wndclass.style         = CS_HREDRAW | CS_VREDRAW;
      wndclass.lpfnWndProc   = WindowProc;
      wndclass.cbClsExtra    = 0;
      wndclass.cbWndExtra    = 0;
      wndclass.hInstance     = hInstance;
      wndclass.hIcon         = LoadIcon(NULL,IDI_APPLICATION);
      wndclass.hCursor       = LoadCursor(NULL,IDC_ARROW);
      wndclass.hbrBackground = (HBRUSH)GetStockObject(WHITE_BRUSH);
      wndclass.lpszMenuName  = NULL;
      wndclass.lpszClassName = szAppName;
      RegisterClass(&wndclass);
   }

   window = CreateWindow(szAppName,      // window class name
      "Fastgraph Fish Tank",   // window caption
      WS_OVERLAPPEDWINDOW,     // window style
      CW_USEDEFAULT,           // initial x position
      CW_USEDEFAULT,           // initial y position
      320,                     // initial x size
      200,                     // initial y size
      NULL,                    // parent window handle
      NULL,                    // window menu handle
      hInstance,               // program instance handle
      NULL);                   // creation parameters

   ShowWindow(window,nCmdShow);
   UpdateWindow(window);

   // The message loop processes entries placed in the message queue.
   // When no message is ready, call go_fish() to perform one frame of
   // animation.

   while (TRUE)
   {
      if (PeekMessage(&msg,NULL,0,0,PM_REMOVE))
      {
         if (msg.message == WM_QUIT)
            break;
         else
         {
            TranslateMessage(&msg);
            DispatchMessage(&msg);
         }
      }
      else
         go_fish();
   }

   return msg.wParam;
}

/****************************************************************************\
*                                                                            *
*  WindowProc                                                                *
*                                                                            *
*  Window procedure to handle messages sent to the window.                   *
*                                                                            *
\****************************************************************************/

HDC      hdc;
HPALETTE hpal;
UINT     cxClient, cyClient;
int      hvb1, hvb2;

long WINAPI _export WindowProc(HWND window, UINT message,
                               UINT wParam, LONG lParam)
{
   PAINTSTRUCT ps;

   switch (message)
   {
      case WM_CREATE:

         // use the default logical palette

         hdc = GetDC(window);
         fg_setdc(hdc);
         hpal = fg_defpal();
         fg_realize(hpal);

         // create two 320x200 virtual buffers

         fg_vbinit();
         hvb1 = fg_vballoc(320,200);
         hvb2 = fg_vballoc(320,200);

         // display the coral background in virtual buffer #2 (which
         // will always contain a clean copy of the background image)

         fg_vbopen(hvb2);
         fg_vbcolors();
         fg_move(0,199);
         fg_showspr("CORAL.SPR",320);

         // get the fish bitmaps

         get_fish();

         return 0;

      case WM_PAINT:
         BeginPaint(window,&ps);
         fg_vbscale(0,fg_getmaxx(),0,fg_getmaxy(),0,cxClient-1,0,cyClient-1);
         EndPaint(window,&ps);
         return 0;

      case WM_SETFOCUS:
         fg_realize(hpal);
         InvalidateRect(window,NULL,TRUE);
         return 0;

      case WM_SIZE:
         cxClient = LOWORD(lParam);
         cyClient = HIWORD(lParam);
         return 0;

      case WM_DESTROY:
         fg_vbclose();
         fg_vbfree(hvb1);
         fg_vbfree(hvb2);
         fg_vbfin();
         DeleteObject(hpal);
         ReleaseDC(window,hdc);
         PostQuitMessage(0);
         return 0;
   }
   return DefWindowProc(window,message,wParam,lParam);
}

/****************************************************************************\
*                                                                            *
*  get_fish                                                                  *
*                                                                            *
*  Fill the fish bitmap arrays.                                              *
*                                                                            *
\****************************************************************************/

char fish1[56*25];
char fish2[54*38];
char fish3[68*26];
char fish4[56*30];
char fish5[62*22];
char fish6[68*36];

int fish_x[] = {0,   64,128,200,  0, 80}; // location of fish (x)
int fish_y[] = {199,199,199,199,150,150}; // location of fish (y)
int width[]  = { 56, 54, 68, 56, 62, 68}; // size of fish: width
int height[] = { 25, 38, 26, 30, 22, 36}; // size of fish: height

char *fishes[] = {fish1, // for convenience, an array of pointers to fish bitmaps
                  fish2,
                  fish3,
                  fish4,
                  fish5,
                  fish6};

void get_fish()
{
   register int i;

   // get the fish bitmaps from an SPR file

   fg_vbopen(hvb1);
   fg_vbcolors();
   fg_move(0,199);
   fg_showspr("FISH.SPR",320);

   for (i = 0; i < 6; i++)
   {
      fg_move(fish_x[i],fish_y[i]);
      fg_getimage(fishes[i],width[i],height[i]);
   }

   fg_erase();
}

/****************************************************************************\
*                                                                            *
*  go_fish                                                                   *
*                                                                            *
*  Make the fish swim around.                                                *
*                                                                            *
\****************************************************************************/

/*  There are 11 fish total, and 6 different kinds of fish. These
*   arrays keep track of what kind of fish each fish is, and how each
*   fish moves:
*
*   fish[]   -- which fish bitmap applies to this fish?
*   x[]      -- starting x coordinate
*   y[]      -- starting y coordinate
*
*   xmin[]   -- how far left (off screen) the fish can go
*   xmax[]   -- how far right (off screen) the fish can go
*   xinc[]   -- how fast the fish goes left and right
*   dir[]    -- starting direction for each fish
*
*   ymin[]   -- how far up this fish can go
*   ymax[]   -- how far down this fish can go
*   yinc[]   -- how fast the fish moves up or down
*   yturn[]  -- how long fish can go in the vertical direction
*               before stopping or turning around
*   ycount[] -- counter to compare to yturn
*/

int fish[]   = {   1,   1,   2,   3,   3,   0,   0,   5,   4,   2,   3};
int x[]      = {-100,-150,-450,-140,-200, 520, 620,-800, 800, 800,-300};
int y[]      = {  40,  60, 150,  80,  70, 190, 180, 100,  30, 130,  92};

int xmin[]   = {-300,-300,-800,-200,-200,-200,-300,-900,-900,-900,-400};
int xmax[]   = { 600, 600,1100,1000,1000, 750, 800,1200,1400,1200, 900};
int xinc[]   = {   2,   2,   8,   5,   5,  -3,  -3,   7,  -8,  -9,   6};
int dir[]    = {   0,   0,   0,   0,   0,   1,   1,   0,   1,   1,   0};

int ymin[]   = {  40,  60, 120,  70,  60, 160, 160,  80,  30, 110,  72};
int ymax[]   = {  80, 100, 170, 110, 100, 199, 199, 120,  70, 150, 122};
int yturn[]  = {  50,  30,  10,  30,  20,  10,  10,  10,  30,   20, 10};
int ycount[] = {   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0};
int yinc[]   = {   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0};

void go_fish()
{
   register int i;

   // copy the background to the workspace

   fg_copypage(hvb2,hvb1);

   // put all the fish in their new positions

   for (i = 0; i < NFISH; i++)
   {
      ycount[i]++;
      if (ycount[i] > yturn[i])
      {
         ycount[i] = 0;
         yinc[i] = (rand() % 3) - 1;
      }
      y[i] += yinc[i];
      y[i] = MIN(ymax[i],MAX(y[i],ymin[i]));

      if (x[i] >= -72 && x[i] < 320)
         put_fish(fish[i],x[i],y[i],dir[i]);

      x[i] += xinc[i];
      if (x[i] <= xmin[i] || x[i] >= xmax[i])
      {
         xinc[i] = -xinc[i];
         dir[i] = 1 - dir[i];
      }
   }

   // scale the workspace image to fill the client area

   fg_vbscale(0,319,0,199,0,cxClient-1,0,cyClient-1);
}

/****************************************************************************\
*                                                                            *
*  put_fish                                                                  *
*                                                                            *
*  Draw one of the six fish anywhere you want.                               *
*                                                                            *
\****************************************************************************/

void put_fish (int fish_num, int x, int y, int fish_dir)
{
   // move to position where the fish will appear

   fg_move(x,y);

   // draw a left- or right-facing fish, depending on fish_dir

   if (fish_dir == 0)
      fg_flpimage(fishes[fish_num],width[fish_num],height[fish_num]);
   else
      fg_clpimage(fishes[fish_num],width[fish_num],height[fish_num]);
}
