{*****************************************************************************
*                                                                            *
*  FISH.DPR                                                                  *
*  FISHU.PAS                                                                 *
*                                                                            *
*  This program shows how to perform simple animation using Fastgraph for    *
*  Windows. Several types of tropical fish swim back and forth against a     *
*  coral reef background. The background image and fish sprites are stored   *
*  in standard pixel run (SPR) files.                                        *
*                                                                            *
*  The application window is initially set to 320x200 pixels, the same size  *
*  as the CORAL.SPR image. When the window size changes, the image is scaled *
*  to fill the window.                                                       *
*                                                                            *
*  A DOS version of this program is available on our BBS in FGFISH20.ZIP.    *
*                                                                            *
*****************************************************************************}

unit FishU;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, FGWin;

type
  TForm1 = class(TForm)
    procedure AppOnActivate(Sender: TObject);
    procedure AppIdle(Sender: TObject; var Done: Boolean);
    procedure FormActivate(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormPaint(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
  end;

var
  Form1: TForm1;

implementation

{$R *.DFM}
var
  cxClient, cyClient : integer;
  dc   : hDC;
  hpal : hPalette;
  hvb1, hvb2 : integer;

  { fish bitmaps }
  fish1 : array[0..1399] of byte;
  fish2 : array[0..2051] of byte;
  fish3 : array[0..1767] of byte;
  fish4 : array[0..1679] of byte;
  fish5 : array[0..1363] of byte;
  fish6 : array[0..2447] of byte;

const
  { number of fish }
  NFISH = 11;

  { location of fish (x & y) }
  fish_x : array[0..5] of integer = (0,   64,128,200,  0, 80);
  fish_y : array[0..5] of integer = (199,199,199,199,150,150);

  { size of fish (width and height) }
  width   : array[0..5] of integer = ( 56, 54, 68, 56, 62, 68);
  height  : array[0..5] of integer = ( 25, 38, 26, 30, 22, 36);

  { for convenience, an array of pointers to fish bitmaps }
  fishes : array[0..5] of ^byte = (@fish1,@fish2,@fish3,@fish4,@fish5,@fish6);

{*****************************************************************************
*                                                                            *
*  min                                                                       *
*                                                                            *
*  Determine the smaller of two integer values.                              *
*                                                                            *
*****************************************************************************}

function min (value1, value2 : integer) : integer;

begin

  if (value1 <= value2) then
    min := value1
  else
    min := value2;

end;

{*****************************************************************************
*                                                                            *
*  max                                                                       *
*                                                                            *
*  Determine the larger of two integer values.                               *
*                                                                            *
*****************************************************************************}

function max (value1, value2 : integer) : integer;

begin

  if (value1 >= value2) then
    max := value1
  else
    max := value2;

end;

{*****************************************************************************
*                                                                            *
*  get_fish                                                                  *
*                                                                            *
*  Fill the fish bitmap arrays.                                              *
*                                                                            *
*****************************************************************************}

procedure get_fish;

var
  fish_num : integer;

begin

  { get the fish bitmaps from an SPR file }

  fg_vbopen(hvb1);
  fg_vbcolors;
  fg_showspr('FISH.SPR'+chr(0),320);

  for fish_num := 0 to 5 do
  begin
    fg_move(fish_x[fish_num],fish_y[fish_num]);
    fg_getimage(fishes[fish_num]^,width[fish_num],height[fish_num]);
  end;

end;

{*****************************************************************************
*                                                                            *
*  put_fish                                                                  *
*                                                                            *
*  Draw one of the six fish anywhere you want.                               *
*                                                                            *
*****************************************************************************}

procedure put_fish (fish_num, x, y, fish_dir : integer);

begin

  { move to position where the fish will appear }

  fg_move(x,y);

  { draw a left- or right-facing fish, depending on fish_dir }

  if (fish_dir = 0) then
    fg_flpimage(fishes[fish_num]^,width[fish_num],height[fish_num])
  else
    fg_clpimage(fishes[fish_num]^,width[fish_num],height[fish_num]);

end;

{*****************************************************************************
*                                                                            *
*  go_fish                                                                   *
*                                                                            *
*  Make the fish swim around. This procedure is called by the application's  *
*  OnIdle event handler.                                                     *
*                                                                            *
*****************************************************************************}

procedure go_fish;

type

  fish_array = array[0..NFISH-1] of integer;

const
{
*   There are 11 fish total, and 6 different kinds of fish. These
*   arrays keep track of what kind of fish each fish is, and how each
*   fish moves:
*
*   fish()   -- which fish bitmap applies to this fish?
*   x()      -- starting x coordinate
*   y()      -- starting y coordinate
*
*   xmin()   -- how far left (off screen) the fish can go
*   xmax()   -- how far right (off screen) the fish can go
*   xinc()   -- how fast the fish goes left and right
*   dir()    -- starting direction for each fish
*
*   ymin()   -- how far up this fish can go
*   ymax()   -- how far down this fish can go
*   yinc()   -- how fast the fish moves up or down
*   yturn()  -- how long fish can go in the vertical direction
*               before stopping or turning around
*   ycount() -- counter to compare to yturn
}

  fish   : fish_array =
                (   1,   1,   2,   3,   3,   0,   0,   5,   4,   2,   3);
  x      : fish_array =
                (-100,-150,-450,-140,-200, 520, 620,-800, 800, 800,-300);
  y      : fish_array =
                (  40,  60, 150,  80,  70, 190, 180, 100,  30, 130,  92);

  xmin   : fish_array =
                (-300,-300,-800,-200,-200,-200,-300,-900,-900,-900,-400);
  xmax   : fish_array =
                ( 600, 600,1100,1000,1000, 750, 800,1200,1400,1200, 900);
  xinc   : fish_array =
                (   2,   2,   8,   5,   5,  -3,  -3,   7,  -8,  -9,   6);
  dir    : fish_array =
                (   0,   0,   0,   0,   0,   1,   1,   0,   1,   1,   0);

  ymin   : fish_array =
                (  40,  60, 120,  70,  60, 160, 160,  80,  30, 110,  72);
  ymax   : fish_array =
                (  80, 100, 170, 110, 100, 199, 199, 120,  70, 150, 122);
  yturn  : fish_array =
                (  50,  30,  10,  30,  20,  10,  10,  10,  30,   20, 10);
  ycount : fish_array =
                (   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0);
  yinc   : fish_array =
                (   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0);

var
  i : integer;

begin

  { copy the background to the workspace }

  fg_copypage(hvb2,hvb1);

  { put all the fish in their new positions }

  for i := 0 to NFISH-1 do
  begin
    inc(ycount[i]);
    if (ycount[i] > yturn[i]) then
    begin
       ycount[i] := 0;
       yinc[i] := random(3) - 1;
    end;
    inc(y[i],yinc[i]);
    y[i] := min(ymax[i],max(y[i],ymin[i]));

    if (x[i] >= -72) and (x[i] < 320) then
      put_fish(fish[i],x[i],y[i],dir[i]);

    inc(x[i],xinc[i]);
    if (x[i] <= xmin[i]) or (x[i] >= xmax[i]) then
    begin
      xinc[i] := -xinc[i];
      dir[i] := 1 - dir[i];
    end;
  end;

  { scale the workspace image to fill the client area }

  fg_vbscale(0,319,0,199,0,cxClient-1,0,cyClient-1);

end;

{****************************************************************************}

procedure TForm1.AppOnActivate(Sender: TObject);
begin
  fg_realize(hpal);
  Invalidate;
end;

procedure TForm1.AppIdle(Sender: Tobject; var Done: Boolean);
begin
  go_fish;
  Done := False;
end;

procedure TForm1.FormActivate(Sender: TObject);
begin
  fg_realize(hpal);
end;

procedure TForm1.FormCreate(Sender: TObject);
begin
  { use the default logical palette }

  dc := GetDC(Form1.Handle);
  fg_setdc(dc);
  hpal := fg_defpal;
  fg_realize(hpal);

  { create two 320x200 virtual buffers }

  fg_vbinit;
  hvb1 := fg_vballoc(320,200);
  hvb2 := fg_vballoc(320,200);

  { display the coral background in virtual buffer #2 (which  }
  { will always contain a clean copy of the background image) }

  fg_vbopen(hvb2);
  fg_vbcolors;
  fg_move(0,199);
  fg_showspr('CORAL.SPR'+chr(0),320);

  { get the fish bitmaps }

  get_fish;

  Application.OnActivate := AppOnActivate;
  Application.OnIdle := AppIdle;
end;

procedure TForm1.FormPaint(Sender: TObject);
begin
  fg_vbscale(0,fg_getmaxx,0,fg_getmaxy,0,cxClient-1,0,cyClient-1);
end;

procedure TForm1.FormResize(Sender: TObject);
begin
  cxClient := ClientWidth;
  cyClient := ClientHeight;
  Invalidate;
end;

procedure TForm1.FormDestroy(Sender: TObject);
begin
  fg_vbclose;
  fg_vbfree(hvb1);
  fg_vbfree(hvb2);
  fg_vbfin;
  DeleteObject(hpal);
  ReleaseDC(Form1.Handle,dc);
end;

end.
