/******************************************************************\

 sprite.c -- sprite & scrolling code from Quickfire by Diana Gruber

 compile using medium memory model
 requires Fastgraph(tm) or Fastgraph/Light(tm) to link

\******************************************************************/

#include <fastgraf.h>           /* header for the Fastgraph lib   */
#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#ifdef __turboc__               /* Turbo or Borland C             */
  #include <mem.h>
#else
  #include <memory.h>           /* Microsoft C                    */
#endif

#define FALSE     0             /* standard defines               */
#define TRUE      1
#define MAX(x,y) ((x) > (y)) ? (x) : (y)
#define MIN(x,y) ((x) < (y)) ? (x) : (y)

#define KB_ESC    1             /* keyboard defines               */
#define KB_UP    72
#define KB_DOWN  80
#define KB_LEFT  75
#define KB_RIGHT 77

#define MAXROWS 42              /* y extent of tile map           */
#define MAXCOLS 190             /* x extent of tile map           */
#define LOGCOLS 22              /* columns in logical page        */
#define LOGROWS 15              /* rows in logical page           */

unsigned char worldmap[MAXCOLS][MAXROWS]; /* tile map array       */
char layout[2][LOGCOLS][LOGROWS];         /* The layout array     */

int tile_orgx, tile_orgy;       /* x and y origin in tile space   */
int hidden,visual;              /* keep track of page swaps       */
int screen_orgx, screen_orgy;   /* x and y origin in screen space */
int hpageofs,hpagebot;          /* hidden page offset and bottom  */
int vpageofs,vpagebot;          /* visual page offset and bottom  */
int tileofs;                    /* offset of tile area            */
int ncols,nrows;                /* rows and columns in world      */
int scrolled;                   /* scroll flag                    */
unsigned long frames;           /* count the frames               */
int plane_x, plane_y;           /* current location of plane      */
int plane_speed;                /* increment value of plane       */
char terminate_string[50];      /* string displayed at exit       */
unsigned int clockspeed;        /* for benchmarking               */
FILE *stream;                   /* used to read the sprites & map */

int nsprites;                   /* number of sprites              */
int plane_frame;                /* determines which frame is shown*/
typedef struct sprt             /* define a sprite structure      */
{
	char *bitmap;
	int width;
	int height;
   int yoffset;

} SPRITE;
SPRITE *sprite[8];                 /* array of sprites            */
   
/* forward declarations */

struct OBJstruct;
typedef struct OBJstruct OBJ, near *OBJp;

/* data structure for objects */

typedef struct OBJstruct 
{
  int x;
  int y;
  int speed;
  int frame;
  SPRITE *image;
};
OBJp plane;

   /* function declarations */
void  main(void);
void  activate_level(void);        /* main action loop            */
void  adjust_layout(void);         /* keep track of changed tiles */
void  apply_sprite(void);          /* calculate & adjust position */
void  load_sprite(void);           /* read sprites from a file    */
void  load_world_map(void);        /* load world data and tiles   */
void  put_sprite(SPRITE *frame, int x, int y); /* sprite primitive*/
void  put_tile(int i,int j);       /* tile copying primitive      */
void  rebuild_hidden(void);        /* fix tiles that were changed */
void  scroll_right(int  npixels);  /* right horz scroll function  */
void  swap(void);                  /* page flipping primitive     */
void  terminate(void);             /* clean up and exit           */

/************************** main function *************************/

void main()
{
   register int i,j;
   unsigned long time1, time2;

   /* Autodetect VGA. If it isn't there, exit.                    */
   if (fg_testmode(20,4) == 0)
   {
      printf("\nvga required\n");
      exit(0);
   }

   /* initialize the graphics environment and keyboard handler    */
   clockspeed = fg_measure();   /* benchmark the microprocessor   */
   fg_setmode(20);              /* set up for mode 'X'            */
   fg_resize(352,727);          /* resize video memory            */
   fg_kbinit(1);                /* initialize keyboard interface  */

   /* initialize the global variables                             */
   visual = 0;                  /* start with visual page at top  */
   hidden = 1;
   vpageofs = 0;                /* visual page starts at line 0   */
   vpagebot = vpageofs+239;     /* visual page ends at line 239   */
   hpageofs = 240;              /* hidden page starts at ln. 240  */
   hpagebot = hpageofs+239;     /* hidden page ends at line 479   */
   tileofs = 480;               /* tile page starts at line 480   */

   tile_orgx = 0;               /* tile origin startes at 0,6     */
   tile_orgy = 6;
   screen_orgx = 0;             /* screen origin starts at 0,0    */
   screen_orgy = 0;

   load_world_map();            /* load the tiles and tile map    */
   load_sprite();               /* load the sprites from a file   */

   plane = (OBJp)malloc(sizeof(OBJ)+1); /* create the plane object*/
   plane->x = 48;               /* starting x,y position of plane */
   plane->y = 224;
   plane->speed = 5;            /* initial increment value        */
   plane->frame = 0;            /* start with upright plane       */
   plane->image = sprite[plane->frame]; /* pointer to sprite      */

   frames = 0;                  /* frame counter starts at 0      */

   /* rebuild the hidden page by putting all the tiles on it      */
   for (i = 0; i < LOGCOLS; i++)
   {
      for (j = 0; j < LOGROWS; j++)
      {
         put_tile(i,j);
         layout[0][i][j] = FALSE;
         layout[1][i][j] = FALSE;
      }
   }

   /* make the hidden page visual                                 */
   swap();

   /* copy the visual page to the hidden page                     */
   fg_transfer(0,351,vpageofs,vpagebot,0,hpagebot,0,0);

   /* start the frame count                                       */
   fg_waitfor(1);
   time1 = fg_getclock();

   /* start the action -- scroll the world map                    */
   activate_level();

   /* complete the frame count and exit                           */
   time2 = fg_getclock();
   time1 = ((time2-time1) * 10) /182;
   if (time1 > 0)
      frames = frames/time1;
   sprintf(terminate_string,"%lu frames per second.\n",frames);

   terminate();
}

/************************** activate level ************************/

void activate_level()
{
   int tile_x;
   unsigned int stall_time;

   /* calculate a stall factor                                    */
   stall_time = clockspeed/10;

   for(;;)
   {
      /* increment the frame count                                */
      frames++;

      /* periodically check the keyboard                          */
      if (frames%3 == 0)
      {
         if (fg_kbtest(0))
         {

            /* spin the airplane in midair                        */
            if (fg_kbtest(KB_UP))
            {
               plane->frame++;
               if (plane->frame > 7)
                  plane->frame = 0;

            }
            if (fg_kbtest(KB_DOWN))
            {
               plane->frame--;
               if (plane->frame < 0)
                  plane->frame = 7;
            }

            /* increase or decrease flying speed                  */
            if (fg_kbtest(KB_LEFT))
            {
               if (plane->speed > 3)
                  plane->speed--;
            }
            if (fg_kbtest(KB_RIGHT))
            {
               if (plane->speed < 16)
                  plane->speed++;
            }

            /* exit if ESCAPE is pressed                          */
            if (fg_kbtest(KB_ESC))
               break;
         }
         else
         {
            plane->speed = 5;
            if (plane->frame != 0)
               plane->frame++;
            if (plane->frame > 7)
               plane->frame = 0;
         }
         plane->image = sprite[plane->frame];
      }

      /* can't go past far left or far right sides of screen      */
      tile_x = plane->x/16 - tile_orgx;
      if (tile_x < 2)
         plane->speed = MAX(plane->speed,8);
      if (tile_x >= 16)
         plane->speed = MIN(plane->speed,8);

      plane->x += plane->speed;
      scroll_right(8);
      rebuild_hidden();
      apply_sprite();

      /* normalize the frame rate on faster machines              */
      fg_stall(stall_time);

      /* do a page flip                                           */
      swap();

      if (scrolled)
      {
         /* copy the visual page to the hidden page */
         fg_transfer(0,351,vpageofs,vpagebot,0,hpagebot,0,0);

         /* also copy the layout                                  */
         memcpy(layout[hidden],layout[visual],22*15);
         scrolled = FALSE;
      }
   }
}

/********************* adjust the layout array ********************/

void adjust_layout()
{
   register int i;

   /* when you scroll right, adjust the array accordingly         */
   for (i = 0; i < 20; i++)
      memcpy(layout[hidden][i],layout[visual][i+2],15);
}

/************************** apply a sprite ************************/

void apply_sprite()
{
   register int i,j;
   int x,y;
   int tile_x,tile_y;
   int max_tilex, min_tiley;
   char *p;

   /* find the current y position of the sprite                   */
   y = plane->y + sprite[plane->frame]->yoffset;
   x = plane->x;

   /* this is the tile at the lower left corner of the sprite     */
   tile_x = x/16 - tile_orgx;
   tile_y = y/16 - tile_orgy;

   /* calc how many tiles total need to be replaced               */
   max_tilex = (x+sprite[plane->frame]->width)/16 - tile_orgx;
   min_tiley = (y-sprite[plane->frame]->height)/16 - tile_orgy;

   /* don't try to draw a plane beyond the screen borders         */
   if (tile_x < 0 || max_tilex > 21 || tile_y > 14 || min_tiley < 0)
      return;

   /* adjust the layout array, flagging tiles as changed          */
   for (i = tile_x; i <= max_tilex; i++)
   {
      p = layout[hidden][i] + min_tiley;
      for (j = min_tiley; j <= tile_y; j++)
         *p++ = TRUE;
   }

   /* now draw the bitmap */
   put_sprite(plane->image,x,y);
}

/**************************** load sprites ************************/

void load_sprite()
{
   register int i;
   int nbytes;
   char *bitmap;
   int width,height;
   SPRITE *new_sprite;
   static int sprite_offset[] = {-9,-4,0,-3,-5,-3,0,-4};

   /* open the bitmap file and read all the sprites               */
   if ((stream = fopen("plane.bmp","rb")) == NULL)
   {
      sprintf(terminate_string,"plane.bmp not found");
      terminate();
   }

   fread(&nsprites,sizeof(int),1,stream);
   for (i = 0; i < nsprites; i++)
   {
      fread(&width,sizeof(int),1,stream);
      fread(&height,sizeof(int),1,stream);
      nbytes = width * height;

      /* first allocate space for the array                       */
      if ((bitmap = (char *)malloc(nbytes+3)) == (char *)NULL)
      {
         sprintf(terminate_string,"unable to allocate memory");
         terminate();
      }

      /* then allocate space for the structure                    */
      if ((new_sprite = (SPRITE *)malloc(sizeof(SPRITE)+1)) ==
          (SPRITE *)NULL)
      {
         sprintf(terminate_string,"out of sprite memory");
         terminate();
      }

      sprite[i] = new_sprite;
      fread(bitmap,sizeof(char),nbytes,stream);

      /* attach the array and other information to the structure  */
      sprite[i]->bitmap = bitmap;
      sprite[i]->width = width;
      sprite[i]->height = height;
      sprite[i]->yoffset = sprite_offset[i];
   }
   fclose(stream);
}

/******************* load graphics and level data *****************/

void load_world_map()
{
   register int i,j;

   /* display pcx file containing tiles at the tile page offset   */
   fg_move(0,tileofs);
   fg_showpcx("qf.pcx",2);

   /* open the level file containing the map information          */

   if ((stream = fopen("qf.lev","rb")) == NULL)
   {
      sprintf(terminate_string,"qf.lev not found.\n");
      terminate();
   }

   /* read the number of rows and columns                         */
   fread(&ncols,sizeof(int),1,stream);
   fread(&nrows,sizeof(int),1,stream);

   /* read all the rows                                           */
   for (i = 0; i < ncols; i++)
      fread(&worldmap[i][0],sizeof(char),nrows,stream);
   fclose(stream);

   /* rows must be even -- adjust for odd number of rows          */
   if (ncols%2 != 0)
      ncols--;

   /* add 22 columns at the end of the array for circular scroll  */
   for (i = 0; i < LOGCOLS; i++)
   {
      for (j = 0; j < nrows; j++)
         worldmap[ncols+i][j] = worldmap[i+54][j];
   }
   ncols+=22;
}

/**************************** put sprite **************************/

void put_sprite(SPRITE *frame, int world_x, int world_y)
{
   register int x,y;

   /* convert x and y world space to x and y physical space       */
   x = world_x - (tile_orgx*16);
   y = world_y - (tile_orgy*16) + hpageofs;

   /* draw the bitmap at the proper location                      */
   fg_move(x,y);
   fg_drwimage(frame->bitmap,frame->width,frame->height);
}

/***************************** put tile ***************************/

void put_tile(int i,int j)
{
   int x1,x2,x3,y1,y2,y3;
   int tileno;

   /* look up the tile index in the world map                     */
   tileno = (int)worldmap[i+tile_orgx][j+tile_orgy];

   /* find the corners of the tile in the tile page               */
   x1 = tileno%20 *16;
   x2 = x1+15;
   y1 = (tileno/20 * 16) + tileofs;
   y2 = y1+15;

   /* find the destination point in the hidden page               */
   x3 = i*16;
   y3 = j*16+15 + hpageofs;

   /* move the tile to the hidden page                            */
   fg_transfer(x1,x2,y1,y2,x3,y3,0,0);
}

/******************** rebuild offscreen tiles *********************/

void rebuild_hidden()
{
   register int i,j;
   char *p;

   /* point to the beginning of the layout array                  */
   p = layout[hidden][0];

   /* Traverse the layout array, replacing marked tiles           */
   for (i = 0; i < LOGCOLS; i++)
   {
      for (j = 0; j < LOGROWS; j++)
      {
         if (*p)
         {
            put_tile(i,j);
            *p = FALSE;
         }
         p++;
      }
   }
}

/*************************** scroll right *************************/

void scroll_right(int npixels)
{
   register int i;

   /* if possible, move the visible area within the physical page */
   if (screen_orgx <= 32-npixels)
      screen_orgx+=npixels;

   /* the origin is out of range, we need to do a full scroll     */
   else if (tile_orgx < ncols - 22)
   {
      /* transfer the relevant part of visual page to hidden page */
      fg_transfer(32,351,vpageofs,vpagebot,0,hpagebot,0,0);

      /* change the origin in tile space and screen space         */
      tile_orgx+=2;
      screen_orgx-=(32-npixels);

      /* fill in two columns of tiles at right side of page       */
      for(i = 0; i< LOGROWS; i++)
      {
         put_tile(21,i);
         put_tile(20,i);
      }

      /* fix that layout array                                    */
      adjust_layout();

      /* set a global to signal a page copy later in the frame    */
      scrolled = TRUE;
   }

   /* can't scroll right -- just reset the tile origin            */
   else
   {
      /* adjust the plane x coordinate too                        */

      plane->x =  plane->x - tile_orgx*16 + 54*16 - plane->speed;
      tile_orgx = 54;
   }
}

/*************************** swap pages ***************************/

void swap()
{
   /* update the visual and hidden page offsets                   */
   vpageofs = 240 - vpageofs;
   hpageofs = 240 - hpageofs;
   vpagebot = vpageofs+239;
   hpagebot = hpageofs+239;

   /* move the screen origin                                      */
   fg_pan(screen_orgx,screen_orgy+vpageofs);

   /* adjust the globals                                          */
   visual = !visual;
   hidden = !hidden;
}

/******************* terminate graphics and exit ******************/

void terminate()
{
   /* restore the video environment to text mode                  */
   fg_kbinit(0);
   fg_setmode(3);
   fg_reset();

   /* print the exit string and exit                              */
   fg_setcolor(15);
   printf(terminate_string);
   exit(0);
}

