/****************************************************************************\
*                                                                            *
*  RAINBOW.C                                                                 *
*                                                                            *
*  This program demonstrates color palette cycling.                          *
*                                                                            *
\****************************************************************************/

#include <fgwin.h>
#include <string.h>

long WINAPI _export WindowProc(HWND,UINT,UINT,LONG);
void fill_color_palette(void);

char RGBvalues[2*24*3];    // two sets of 24 RGB triplets

int PASCAL WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance,
                    LPSTR lpszCmdParam, int nCmdShow)
{
   static char szAppName[] = "FGrainbow";
   HWND        window;
   MSG         msg;
   WNDCLASS    wndclass;

   if (!hPrevInstance)
   {
      wndclass.style         = CS_HREDRAW | CS_VREDRAW;
      wndclass.lpfnWndProc   = WindowProc;
      wndclass.cbClsExtra    = 0;
      wndclass.cbWndExtra    = 0;
      wndclass.hInstance     = hInstance;
      wndclass.hIcon         = LoadIcon(NULL,IDI_APPLICATION);
      wndclass.hCursor       = LoadCursor(NULL,IDC_ARROW);
      wndclass.hbrBackground = (HBRUSH)GetStockObject(WHITE_BRUSH);
      wndclass.lpszMenuName  = NULL;
      wndclass.lpszClassName = szAppName;
      RegisterClass (&wndclass);
   }

   window = CreateWindow(szAppName,      // window class name
      "Color Cycling",         // window caption
      WS_OVERLAPPEDWINDOW,     // window style
      CW_USEDEFAULT,           // initial x position
      CW_USEDEFAULT,           // initial y position
      CW_USEDEFAULT,           // initial x size
      CW_USEDEFAULT,           // initial y size
      NULL,                    // parent window handle
      NULL,                    // window menu handle
      hInstance,               // program instance handle
      NULL);                   // creation parameters

   ShowWindow(window,nCmdShow);
   UpdateWindow(window);

   SetTimer(window,1,50,NULL);

   while (GetMessage(&msg,NULL,0,0))
   {
      TranslateMessage(&msg);
      DispatchMessage(&msg);
   }
   return msg.wParam;
}

/****************************************************************************\
*                                                                            *
*  WindowProc                                                                *
*                                                                            *
*  Window procedure to handle messages sent to the window.                   *
*                                                                            *
\****************************************************************************/

HDC      hdc;
HPALETTE hpal;
UINT     cxClient, cyClient;
int      hvb;
int      start;

long WINAPI _export WindowProc(HWND window, UINT message,
                               UINT wParam, LONG lParam)
{
   PAINTSTRUCT ps;
   int color, xlen, ylen;

   switch (message)
   {
      case WM_CREATE:

         // create the logical palette

         hdc = GetDC(window);
         fg_setdc(hdc);
         fill_color_palette();
         hpal = fg_logpal(10,24,RGBvalues);

         // create a 640x480 virtual buffer

         fg_vbinit();
         hvb = fg_vballoc(640,480);
         fg_vbopen(hvb);
         fg_vbcolors();

         // construct a crude image of a rainbow

         fg_setcolor(255);
         fg_fillpage();
         fg_setclip(0,639,0,300);
         fg_move(320,300);
         xlen = 240;
         ylen = 120;
         for (color = 10; color < 34; color++)
         {
            fg_setcolor(color);
            fg_ellipsef(xlen,ylen);
            xlen -= 4;
            ylen -= 3;
         }
         fg_setcolor(255);
         fg_ellipsef(xlen,ylen);
         fg_setclip(0,639,0,479);

         // starting index into the array of color values

         start = 0;

         return 0;

      case WM_PAINT:
         BeginPaint(window,&ps);
         fg_vbscale(0,fg_getmaxx(),0,fg_getmaxy(),0,cxClient-1,0,cyClient-1);
         EndPaint(window,&ps);
         return 0;

      case WM_SETFOCUS:
         fg_realize(hpal);
         InvalidateRect(window,NULL,TRUE);
         return 0;

      case WM_SIZE:
         cxClient = LOWORD(lParam);
         cyClient = HIWORD(lParam);
         return 0;

      case WM_TIMER:
         if (GetActiveWindow() == window)
         {
            start = (start + 3) % 72;
            fg_setdacs(10,24,&RGBvalues[start]);
            if (fg_colors() > 8)
               fg_vbscale(0,fg_getmaxx(),0,fg_getmaxy(),0,cxClient-1,0,cyClient-1);
         }
         return 0;

      case WM_DESTROY:
         KillTimer(window,1);
         fg_vbclose();
         fg_vbfree(hvb);
         fg_vbfin();
         DeleteObject(hpal);
         ReleaseDC(window,hdc);
         PostQuitMessage(0);
         return 0;
   }
   return DefWindowProc(window,message,wParam,lParam);
}

/****************************************************************************\
*                                                                            *
*  fill_color_palette                                                        *
*                                                                            *
*  Set up the colors for the application's logical palette in the RGBvalues  *
*  array. The logical palette will contain 24 non-system colors (indices 10  *
*  to 33) defining the initial RGB values for the colors being cycled.       *
*                                                                            *
*  Note that we store two identical sets of 24 RGB triplets in RGBvalues. We *
*  can then perform color cycling without having to worry about wrapping to  *
*  the start of the array because the index pointing to the starting RGB     *
*  triplet never extends beyond the first set of 24 RGB triplets.            *
*                                                                            *
\****************************************************************************/

void fill_color_palette()
{
   static BYTE colors[] = {
      182,182,255, 198,182,255, 218,182,255, 234,182,255, 255,182,255,
      255,182,234, 255,182,218, 255,182,198, 255,182,182, 255,198,182,
      255,218,182, 255,234,182, 255,255,182, 234,255,182, 218,255,182,
      198,255,182, 182,255,182, 182,255,198, 182,255,218, 182,255,234,
      182,255,255, 182,234,255, 182,218,255, 182,198,255};

   // set up two identical sets of the 24 colors in the RGBvalues array

   memcpy(RGBvalues,colors,24*3);
   memcpy(&RGBvalues[24*3],colors,24*3);
}
