/****************************************************************************\
*                                                                            *
*  FISH1.C                                                                   *
*                                                                            *
*  This is the original DOS version of the Fastgraph Fish Tank.              *
*                                                                            *
\****************************************************************************/

#include <fastgraf.h>
#include <stdio.h>
#include <stdlib.h>

void main(void);
void get_fish(void);
void go_fish(void);
void put_fish(int fish_no,int x,int y,int fish_dir);

#define MAX(x,y) ((x) > (y)) ? (x) : (y)
#define MIN(x,y) ((x) < (y)) ? (x) : (y)

#define NFISH 11

/****************************************************************************\
*                                                                            *
*  main program                                                              *
*                                                                            *
\****************************************************************************/

void main()
{
   int old_mode;

   // in case we're compiling for protected mode

   fg_initpm();

   // make sure the system supports video mode 20 with three pages

   if (fg_testmode(20,3) == 0)
   {
      printf("This program requires VGA graphics.\n");
      exit(0);
   }

   // initialize the video environment

   old_mode = fg_getmode();
   fg_setmode(20);

   // display the coral background on page 2 (which will always
   // contain a clean copy of the background image)

   fg_setpage(2);
   fg_move(0,199);
   fg_showppr("CORAL.PPR",320);

   // copy the background to the visual page

   fg_copypage(2,0);

   // get the fish

   get_fish();

   // make the fish go

   go_fish();

   // restore the original video state

   fg_setmode(old_mode);
   fg_reset();
}

/****************************************************************************\
*                                                                            *
*  get_fish                                                                  *
*                                                                            *
*  Fill the fish bitmap arrays.                                              *
*                                                                            *
\****************************************************************************/

char fish1[56*25];
char fish2[54*38];
char fish3[68*26];
char fish4[56*30];
char fish5[62*22];
char fish6[68*36];

int fish_x[] = {0,   64,128,200,  0, 80}; /* location of fish x & y */
int fish_y[] = {199,199,199,199,150,150};
int width[]  = { 56, 54, 68, 56, 62, 68}; /* size of fish: width */
int height[] = { 25, 38, 26, 30, 22, 36}; /* size of fish: height */

char *fishes[] = {fish1, /* for convenience, an array of pointers to fishes */
                  fish2,
                  fish3,
                  fish4,
                  fish5,
                  fish6};

void get_fish()
{
   register int fish_num;

   // get the fish bitmaps from a PPR file

   fg_setpage(1);
   fg_move(0,199);
   fg_showppr("FISH.PPR",320);

   for (fish_num = 0; fish_num < 6; fish_num++)
   {
      fg_move(fish_x[fish_num],fish_y[fish_num]);
      fg_getimage(fishes[fish_num],width[fish_num],height[fish_num]);
   }
}

/****************************************************************************\
*                                                                            *
*  go_fish                                                                   *
*                                                                            *
*  Make the fish swim around.                                                *
*                                                                            *
\****************************************************************************/

/*  There are 11 fish total, and 6 different kinds of fish. These
*   arrays keep track of what kind of fish each fish is, and how each
*   fish moves:
*
*   fish[]   -- which fish bitmap applies to this fish?
*   x[]      -- starting x coordinate
*   y[]      -- starting y coordinate
*
*   xmin[]   -- how far left (off screen) the fish can go
*   xmax[]   -- how far right (off screen) the fish can go
*   xinc[]   -- how fast the fish goes left and right
*   dir[]    -- starting direction for each fish
*
*   ymin[]   -- how far up this fish can go
*   ymax[]   -- how far down this fish can go
*   yinc[]   -- how fast the fish moves up or down
*   yturn[]  -- how long fish can go in the vertical direction
*               before stopping or turning around
*   ycount[] -- counter to compare to yturn
*/

int fish[]   = {   1,   1,   2,   3,   3,   0,   0,   5,   4,   2,   3};
int x[]      = {-100,-150,-450,-140,-200, 520, 620,-800, 800, 800,-300};
int y[]      = {  40,  60, 150,  80,  70, 190, 180, 100,  30, 130,  92};

int xmin[]   = {-300,-300,-800,-200,-200,-200,-300,-900,-900,-900,-400};
int xmax[]   = { 600, 600,1100,1000,1000, 750, 800,1200,1400,1200, 900};
int xinc[]   = {   2,   2,   8,   5,   5,  -3,  -3,   7,  -8,  -9,   6};
int dir[]    = {   0,   0,   0,   0,   0,   1,   1,   0,   1,   1,   0};

int ymin[]   = {  40,  60, 120,  70,  60, 160, 160,  80,  30, 110,  72};
int ymax[]   = {  80, 100, 170, 110, 100, 199, 199, 120,  70, 150, 122};
int yturn[]  = {  50,  30,  10,  30,  20,  10,  10,  10,  30,   20, 10};
int ycount[] = {   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0};
int yinc[]   = {   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0};

void go_fish()
{
   register int i;
   int visual_page, work_page;
   unsigned char key, aux;

   // initially page 0 is the visual page and page 1 is the workspace page

   visual_page = 0;
   work_page = 1;

   // make the fish swim around

   do
   {
      // copy the background to the workspace page

      fg_copypage(2,work_page);

      // put all the fish in their new positions

      for (i = 0; i < NFISH; i++)
      {
         ycount[i]++;
         if (ycount[i] > yturn[i])
         {
            ycount[i] = 0;
            yinc[i] = (rand() % 3) - 1;
         }
         y[i] += yinc[i];
         y[i] = MIN(ymax[i],MAX(y[i],ymin[i]));

         if (x[i] >= -72 && x[i] < 320)
            put_fish(fish[i],x[i],y[i],dir[i]);

         x[i] += xinc[i];
         if (x[i] <= xmin[i] || x[i] >= xmax[i])
         {
            xinc[i] = -xinc[i];
            dir[i] = 1 - dir[i];
         }
      }

      // page flip to display the new frame

      visual_page = work_page;
      fg_setvpage(visual_page);

      // the old visual page becomes the workspace next frame
      work_page = 1 - work_page;
      fg_setpage(work_page);

      // intercept a keystroke, if it is escape exit the program

      fg_intkey(&key,&aux);
   }
   while (key != 27);
}

/****************************************************************************\
*                                                                            *
*  put_fish                                                                  *
*                                                                            *
*  Draw one of the six fish anywhere you want.                               *
*                                                                            *
\****************************************************************************/

void put_fish (int fish_num, int x, int y, int fish_dir)
{
   // move to position where the fish will appear

   fg_move(x,y);

   // draw a left- or right-facing fish, depending on fish_dir

   if (fish_dir == 0)
      fg_flpimage(fishes[fish_num],width[fish_num],height[fish_num]);
   else
      fg_clpimage(fishes[fish_num],width[fish_num],height[fish_num]);
}
