/****************************************************************************\
*                                                                            *
*  FULLSCR.C                                                                 *
*                                                                            *
*  This program shows how to set up a full screen DirectDraw application     *
*  for either blitting or flipping. The selection of blitting or flipping is *
*  controlled by the BLIT and FLIP symbols defined below.                    *
*                                                                            *
\****************************************************************************/

#include <fgwin.h>

#define WIDTH  640
#define HEIGHT 480

// define either BLIT or FLIP, but not both, for blitting or flipping
#define BLIT
//#define FLIP

int hvb;
int status;
BOOL AppIsActive = FALSE;

long WINAPI _export WindowProc(HWND,UINT,UINT,LONG);
void animate(void);

int PASCAL WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance,
                    LPSTR lpszCmdParam, int nCmdShow)
{
   static char szAppName[] = "FGfullscr";
   HWND        window;
   MSG         msg;
   WNDCLASS    wndclass;

   if (!hPrevInstance)
   {
      wndclass.style         = CS_HREDRAW | CS_VREDRAW;
      wndclass.lpfnWndProc   = WindowProc;
      wndclass.cbClsExtra    = 0;
      wndclass.cbWndExtra    = 0;
      wndclass.hInstance     = hInstance;
      wndclass.hIcon         = LoadIcon(NULL,IDI_APPLICATION);
      wndclass.hCursor       = LoadCursor(NULL,IDC_ARROW);
      wndclass.hbrBackground = NULL;
      wndclass.lpszMenuName  = NULL;
      wndclass.lpszClassName = szAppName;
      RegisterClass (&wndclass);
   }

   window = CreateWindowEx(
      WS_EX_TOPMOST,           // extended window style
      szAppName,               // window class name
      "DirectDraw Full Screen",// window caption
      WS_POPUP,                // window style
      0,                       // initial x position
      0,                       // initial y position
      WIDTH,                   // initial x size
      HEIGHT,                  // initial y size
      NULL,                    // parent window handle
      NULL,                    // window menu handle
      hInstance,               // program instance handle
      NULL);                   // creation parameters

   ShowWindow(window,nCmdShow);

   // define the DirectDraw video mode and initialize the Fastgraph for
   // Windows virtual buffer environment for that mode

#ifdef BLIT
   fg_ddsetup(WIDTH,HEIGHT,8,0);
#endif
#ifdef FLIP
   fg_ddsetup(WIDTH,HEIGHT,8,1);
#endif
   fg_vbinit();
   status = fg_ddstatus();
   if (status != 0)
   {
      MessageBox(window,"DirectDraw init error","FULLSCR",MB_OK|MB_ICONSTOP);
      return 0;
   }

   // if blitting, create a virtual buffer the same size as the screen
   // resolution; if flipping, use the primary surface's back buffer

#ifdef BLIT
   hvb = fg_vballoc(WIDTH,HEIGHT);
#endif
#ifdef FLIP
   hvb = 0;
#endif
   fg_vbopen(hvb);
   fg_vbcolors();

   // hide the mouse cursor

   fg_mouseini();
   fg_mousevis(0);

   UpdateWindow(window);

   // The message loop processes entries placed in the message queue.
   // When no message is ready and this is the active application, call
   // animate() to perform one frame of animation.

   while (TRUE)
   {
      if (PeekMessage(&msg,NULL,0,0,PM_REMOVE))
      {
         if (msg.message == WM_QUIT)
            break;
         else
         {
            TranslateMessage(&msg);
            DispatchMessage(&msg);
         }
      }
      else if (AppIsActive)
      {
         animate();
      }
   }

   return msg.wParam;
}

/****************************************************************************\
*                                                                            *
*  WindowProc                                                                *
*                                                                            *
*  Window procedure to handle messages sent to the window.                   *
*                                                                            *
\****************************************************************************/

HDC      hdc;
HPALETTE hpal;

long WINAPI _export WindowProc(HWND window, UINT message,
                               UINT wParam, LONG lParam)
{
   switch (message)
   {
      case WM_ACTIVATEAPP:
         AppIsActive = wParam;
         return 0;

      case WM_CREATE:
         hdc = GetDC(window);
         fg_setdc(hdc);
         hpal = fg_defpal();
         fg_realize(hpal);
         AppIsActive = TRUE;
         return 0;

      case WM_KEYDOWN:
         switch(wParam)
         {
            case VK_ESCAPE:
            case VK_F12:
               PostMessage(window,WM_CLOSE,0,0);
               break;
         }
         return 0;

      case WM_SETFOCUS:
         fg_realize(hpal);
         InvalidateRect(window,NULL,TRUE);
         return 0;

      case WM_DESTROY:
         fg_mousevis(1);
         fg_vbclose();
#ifdef BLIT
         fg_vbfree(hvb);
#endif
         fg_vbfin();
         DeleteObject(hpal);
         ReleaseDC(window,hdc);
         PostQuitMessage(0);
         return 0;
   }
   return DefWindowProc(window,message,wParam,lParam);
}

/****************************************************************************\
*                                                                            *
*  animate                                                                   *
*                                                                            *
*  Construct the next frame of animation and display it with either blitting *
*  or flipping, as directed by the BLIT and FLIP symbols above.              *
*                                                                            *
\****************************************************************************/

void animate(void)
{
   int color;

   // fill drawing surface with the next color

   color = fg_getcolor() + 1;
   if (color > 255) color = 0;
   fg_setcolor(color);
   fg_fillpage();

   // blit or flip surface to the screen

#ifdef BLIT
   fg_vbpaste(0,WIDTH-1,0,HEIGHT-1,0,HEIGHT-1);
#endif
#ifdef FLIP
   fg_ddflip();
#endif
}
