/****************************************************************************\
*                                                                            *
*  GRAPHICS.C                                                                *
*                                                                            *
*  This program demonstrates some of the Fastgraph for Windows graphics      *
*  primitive functions.                                                      *
*                                                                            *
\****************************************************************************/

#include <fgwin.h>
#include "graphics.h"

#define WIDTH  640
#define HEIGHT 480

long WINAPI _export WindowProc(HWND,UINT,UINT,LONG);
void blit(void);
void do_circles(void);
void do_ellipses(void);
void do_lines(void);
void do_paint(void);
void do_points(void);
void do_polygons(void);
void do_rectangles(void);

int PASCAL WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance,
                    LPSTR lpszCmdParam, int nCmdShow)
{
   static char szAppName[] = "FGgraphics";
   HWND        window;
   MSG         msg;
   WNDCLASS    wndclass;

   if (!hPrevInstance)
   {
      wndclass.style         = CS_HREDRAW | CS_VREDRAW;
      wndclass.lpfnWndProc   = WindowProc;
      wndclass.cbClsExtra    = 0;
      wndclass.cbWndExtra    = 0;
      wndclass.hInstance     = hInstance;
      wndclass.hIcon         = LoadIcon(NULL,IDI_APPLICATION);
      wndclass.hCursor       = LoadCursor(NULL,IDC_ARROW);
      wndclass.hbrBackground = (HBRUSH)GetStockObject(WHITE_BRUSH);
      wndclass.lpszMenuName  = szAppName;
      wndclass.lpszClassName = szAppName;
      RegisterClass (&wndclass);
   }

   window = CreateWindow(szAppName, // window class name
      "Graphics Primitives",   // window caption
      WS_OVERLAPPEDWINDOW,     // window style
      CW_USEDEFAULT,           // initial x position
      CW_USEDEFAULT,           // initial y position
      WIDTH,                   // initial x size
      HEIGHT,                  // initial y size
      NULL,                    // parent window handle
      NULL,                    // window menu handle
      hInstance,               // program instance handle
      NULL);                   // creation parameters

   ShowWindow(window,nCmdShow);
   UpdateWindow(window);

   while (GetMessage(&msg,NULL,0,0))
   {
      TranslateMessage(&msg);
      DispatchMessage(&msg);
   }
   return msg.wParam;
}

/****************************************************************************\
*                                                                            *
*  WindowProc                                                                *
*                                                                            *
*  Window procedure to handle messages sent to the window.                   *
*                                                                            *
\****************************************************************************/

HDC      hdc;
HPALETTE hpal;
UINT     cxClient, cyClient;
int      hvb;

long WINAPI _export WindowProc(HWND window, UINT message,
                               UINT wParam, LONG lParam)
{
   PAINTSTRUCT ps;

   switch (message)
   {
      case WM_CREATE:
         hdc = GetDC(window);
         fg_setdc(hdc);
         hpal = fg_defpal();
         fg_realize(hpal);

         fg_vbinit();
         hvb = fg_vballoc(WIDTH,HEIGHT);
         fg_vbopen(hvb);
         fg_vbcolors();

         fg_setcolor(25);
         fg_fillpage();
         return 0;

      case WM_COMMAND:
         switch (wParam)
         {
            case IDM_POINTS:
               do_points();
               return 0;

            case IDM_LINES:
               do_lines();
               return 0;

            case IDM_RECTANGLES:
               do_rectangles();
               return 0;

            case IDM_CIRCLES:
               do_circles();
               return 0;

            case IDM_ELLIPSES:
               do_ellipses();
               return 0;

            case IDM_POLYGONS:
               do_polygons();
               return 0;

            case IDM_PAINT:
               do_paint();
               return 0;

            case IDM_QUIT:
               SendMessage(window,WM_CLOSE,0,0L);
               return 0;
         }
         break;

      case WM_PAINT:
         BeginPaint(window,&ps);
         blit();
         EndPaint(window,&ps);
         return 0;

      case WM_SETFOCUS:
         fg_realize(hpal);
         InvalidateRect(window,NULL,TRUE);
         return 0;

      case WM_SIZE:
         cxClient = LOWORD(lParam);
         cyClient = HIWORD(lParam);
         return 0;

      case WM_DESTROY:
         fg_vbclose();
         fg_vbfree(hvb);
         fg_vbfin();
         DeleteObject(hpal);
         ReleaseDC(window,hdc);
         PostQuitMessage(0);
         return 0;
   }
   return DefWindowProc(window,message,wParam,lParam);
}

/****************************************************************************\
*                                                                            *
*  blit                                                                      *
*                                                                            *
*  Use fg_vbpaste() or fg_vbscale() to display the virtual buffer contents   *
*  in the client area, depending on the size of the client window.           *
*                                                                            *
\****************************************************************************/

void blit()
{
   if (cxClient > WIDTH || cyClient > HEIGHT)   // window larger than 640x480
      fg_vbscale(0,WIDTH-1,0,HEIGHT-1,0,cxClient-1,0,cyClient-1);
   else
      fg_vbpaste(0,WIDTH-1,0,HEIGHT-1,0,cyClient-1);
}

/****************************************************************************\
*                                                                            *
*  do_circles                                                                *
*                                                                            *
*  Draw a series of concentric circles.                                      *
*                                                                            *
\****************************************************************************/

void do_circles()
{
   register int i, radius;

   fg_setcolor(11);
   fg_fillpage();

   // draw 25 concentric circles at the center of the virtual buffer

   fg_move(WIDTH/2,HEIGHT/2);
   radius = 4;
   fg_setcolor(25);
   for (i = 0; i < 25; i++)
   {
      fg_circle(radius);
      radius += 8;
   }

   blit();
}

/****************************************************************************\
*                                                                            *
*  do_ellipses                                                               *
*                                                                            *
*  Draw a series of concentric ellipses.                                     *
*                                                                            *
\****************************************************************************/

void do_ellipses()
{
   register int i;
   int horiz, vert;

   fg_setcolor(11);
   fg_fillpage();

   // draw 80 concentric ellipses at the center of the virtual buffer

   fg_move(WIDTH/2,HEIGHT/2);
   horiz = 4;
   vert  = 1;
   fg_setcolor(25);
   for (i = 0; i < 80; i++)
   {
      fg_ellipse(horiz,vert);
      horiz += 3;
      vert++;
   }

   blit();
}

/****************************************************************************\
*                                                                            *
*  do_lines                                                                  *
*                                                                            *
*  Draw a pattern of solid lines.                                            *
*                                                                            *
\****************************************************************************/

void do_lines()
{
   register int x, y;
   int i, x1, x2, y1;
   static int line_color[] = {12,11,19,21,21,19,11,12};

   fg_setcolor(25);
   fg_fillpage();

   // draw horizontal lines

   for (y = 0; y < HEIGHT; y+=40)
   {
      for (i = 0; i < 8; i++)
      {
         fg_setcolor(line_color[i]);
         y1 = y + 3*i;
         fg_move(0,y1);
         fg_draw(WIDTH-1,y1);
      }
   }

   // draw vertical lines

   for (x = 0; x < WIDTH; x+=60)
   {
      for (i = 0; i < 8; i++)
      {
         fg_setcolor(line_color[i]);
         x1 = x + 3*i;
         fg_move(x1,0);
         fg_draw(x1,HEIGHT-1);
      }
   }

   // draw red diagonal lines

   fg_setcolor(22);
   for (x1 = -640; x1 < 640; x1+=60)
   {
      x2 = x1 + HEIGHT;
      fg_move(x1,0);
      fg_draw(x2,HEIGHT);
   }
   for (x1 = 0; x1 < 1280; x1+=60)
   {
      x2 = x1 - HEIGHT;
      fg_move(x1,0);
      fg_draw(x2,HEIGHT);
   }

   blit();
}

/****************************************************************************\
*                                                                            *
*  do_paint                                                                  *
*                                                                            *
*  Demonstrate region fill.                                                  *
*                                                                            *
\****************************************************************************/

void do_paint()
{
   int x1, x2, y1, y2;

   fg_setcolor(25);
   fg_fillpage();

   // draw a rectangle

   x1 = 40;
   x2 = WIDTH - 40;
   y1 = 20;
   y2 = HEIGHT - 20;
   fg_setcolor(21);
   fg_rect(x1,x2,y1,y2);

   // outline the rectangle

   fg_setcolor(10);
   fg_box(x1,x2,y1,y2);

   // draw the circle

   x1 = WIDTH / 2;
   y1 = HEIGHT / 2;
   fg_move(x1,y1);
   fg_circle(80);

   // draw cross bars in the circle

   fg_move(x1,y1-80);
   fg_draw(x1,y1+80);
   fg_move(x1-80,y1);
   fg_draw(x1+80,y1);

   // paint each quarter of the circle

   fg_setcolor(11);
   fg_paint(x1-6,y1-6);
   fg_setcolor(12);
   fg_paint(x1+6,y1+6);
   fg_setcolor(13);
   fg_paint(x1+6,y1-6);
   fg_setcolor(14);
   fg_paint(x1-6,y1+6);

   // paint the area outside the box

   fg_setcolor(24);
   fg_paint(41,21);

   blit();
}

/****************************************************************************\
*                                                                            *
*  do_points                                                                 *
*                                                                            *
*  Draw a pattern of points.                                                 *
*                                                                            *
\****************************************************************************/

void do_points()
{
   register int x, y;

   // fill the virtual buffer with blue pixels

   fg_setcolor(24);
   fg_fillpage();

   // draw the patterns of points

   fg_setcolor(19);
   for (x = 7; x < WIDTH; x+=20)
      for (y = 0; y < HEIGHT; y+=8)
         fg_point(x,y);

   fg_setcolor(22);
   for (x = 17; x < WIDTH; x+=20)
      for (y = 4; y < HEIGHT; y+=8)
         fg_point(x,y);

   blit();
}

/****************************************************************************\
*                                                                            *
*  do_polygons                                                               *
*                                                                            *
*  Draw a grid of filled polygons.                                           *
*                                                                            *
\****************************************************************************/

void do_polygons()
{
   register int i, j;

   static int xy_dkblue[]  = {0,16, 24,0, 24,40, 0,56};
   static int xy_ltblue[]  = {24,0, 72,0, 72,40, 24,40};
   static int xy_magenta[] = {0,56, 24,40, 72,40, 48,56};
   int work_array[57*2];

   fg_setcolor(25);
   fg_fillpage();

   // draw 225 filled polygons (15 rows of 15)

   for (j = 0; j < 15; j++)
   {
      for (i = 0; i < 15; i++)
      {
         fg_polyoff(i*72-j*24,j*56-i*16);
         fg_setcolor(11);
         fg_polyfill(xy_dkblue,work_array,4);
         fg_setcolor(19);
         fg_polyfill(xy_ltblue,work_array,4);
         fg_setcolor(20);
         fg_polyfill(xy_magenta,work_array,4);
      }
   }

   blit();
}

/****************************************************************************\
*                                                                            *
*  do_rectangles                                                             *
*                                                                            *
*  Draw a grid of filled rectangles.                                         *
*                                                                            *
\****************************************************************************/

void do_rectangles()
{
   register int i, j;
   int color;
   int x1, x2, y1, y2;
   int xinc, yinc;

   x1 = 0;
   xinc = WIDTH / 10;
   x2 = xinc - 1;

   y1 = 0;
   yinc = HEIGHT / 10;
   y2 = yinc - 1;

   color = 10;

   // draw 100 filled rectangles (10 rows of 10)

   for (i = 0; i < 10; i++)
   {
      for (j = 0; j < 10; j++)
      {
         fg_setcolor(color);
         fg_rect(x1,x2,y1,y2);
         color++;
         if (color > 24) color = 10;
         x1 += xinc;
         x2 += xinc;
      }
      x1 = 0;
      x2 = xinc - 1;
      y1 += yinc;
      y2 += yinc;
   }

   blit();
}
