/****************************************************************************\
*                                                                            *
*  IMAGE.C                                                                   *
*                                                                            *
*  This program demonstrates the Fastgraph for Windows image file display    *
*  and creation functions.                                                   *
*                                                                            *
\****************************************************************************/

#include <fgwin.h>
#include <commdlg.h>
#ifdef __TURBOC__
#include <mem.h>
#else
#ifdef __WATCOMC__
#include <mem.h>
#else
#include <memory.h>
#endif
#endif
#include "image.h"

#define OK   1
#define ERR -1

long WINAPI _export WindowProc(HWND,UINT,UINT,LONG);
void do_bmp(UINT);
void do_flic(UINT);
void do_pcx(UINT);
int  get_open_filename(char *, char*);
int  get_save_filename(char *);
void switch_buffers(void);

int PASCAL WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance,
                    LPSTR lpszCmdParam, int nCmdShow)
{
   static char szAppName[] = "FGimage";
   HWND        window;
   MSG         msg;
   WNDCLASS    wndclass;

   if (!hPrevInstance)
   {
      wndclass.style         = CS_HREDRAW | CS_VREDRAW;
      wndclass.lpfnWndProc   = WindowProc;
      wndclass.cbClsExtra    = 0;
      wndclass.cbWndExtra    = 0;
      wndclass.hInstance     = hInstance;
      wndclass.hIcon         = LoadIcon(NULL,IDI_APPLICATION);
      wndclass.hCursor       = LoadCursor(NULL,IDC_ARROW);
      wndclass.hbrBackground = (HBRUSH)GetStockObject(WHITE_BRUSH);
      wndclass.lpszMenuName  = szAppName;
      wndclass.lpszClassName = szAppName;
      RegisterClass (&wndclass);
   }

   window = CreateWindow(szAppName, // window class name
      "Image File Demo",       // window caption
      WS_OVERLAPPEDWINDOW,     // window style
      CW_USEDEFAULT,           // initial x position
      CW_USEDEFAULT,           // initial y position
      CW_USEDEFAULT,           // initial x size
      CW_USEDEFAULT,           // initial y size
      NULL,                    // parent window handle
      NULL,                    // window menu handle
      hInstance,               // program instance handle
      NULL);                   // creation parameters

   ShowWindow(window,nCmdShow);
   UpdateWindow(window);

   while (GetMessage(&msg,NULL,0,0))
   {
      TranslateMessage(&msg);
      DispatchMessage(&msg);
   }
   return msg.wParam;
}

/****************************************************************************\
*                                                                            *
*  WindowProc                                                                *
*                                                                            *
*  Window procedure to handle messages sent to the window.                   *
*                                                                            *
\****************************************************************************/

HDC      hdc;
HMENU    hmenu;
HPALETTE hpal;
int      hvb;
UINT     cxClient, cyClient;
int      cxBuffer, cyBuffer;
int      colors;
char     default_file[256];
char     file_header[128];
char     file_name[256];
char     file_title[16];
char     file_palette[768];
char     open_file[16];
char     mb_text[256];

long WINAPI _export WindowProc(HWND window, UINT message,
                               UINT wParam, LONG lParam)
{
   PAINTSTRUCT ps;

   switch (message)
   {
      case WM_CREATE:
         hdc = GetDC(window);
         fg_setdc(hdc);
         hpal = fg_defpal();
         fg_realize(hpal);

         fg_vbinit();
         hvb = fg_vballoc(1,1);
         fg_vbopen(hvb);
         fg_vbcolors();

         fg_setcolor(25);
         fg_fillpage();
         hmenu = GetMenu(window);
         return 0;

      case WM_COMMAND:
         switch (wParam)
         {
            case IDM_BMPOPEN:
            case IDM_BMPMAKE:
            case IDM_BMPINFO:
               do_bmp(wParam);
               return 0;

            case IDM_PCXOPEN:
            case IDM_PCXMAKE:
            case IDM_PCXINFO:
               do_pcx(wParam);
               return 0;

            case IDM_FLICOPEN:
            case IDM_FLICPLAY:
            case IDM_FLICFRAME:
            case IDM_FLICRESET:
            case IDM_FLICINFO:
               do_flic(wParam);
               return 0;

            case IDM_QUIT:
               SendMessage(window,WM_CLOSE,0,0L);
               return 0;
         }
         break;

      case WM_PAINT:
         BeginPaint(window,&ps);
         fg_vbscale(0,fg_getmaxx(),0,fg_getmaxy(),0,cxClient-1,0,cyClient-1);
         EndPaint(window,&ps);
         return 0;

      case WM_SETFOCUS:
         fg_realize(hpal);
         InvalidateRect(window,NULL,TRUE);
         return 0;

      case WM_SIZE:
         cxClient = LOWORD(lParam);
         cyClient = HIWORD(lParam);
         return 0;

      case WM_DESTROY:
         fg_vbclose();
         fg_vbfree(hvb);
         fg_vbfin();
         DeleteObject(hpal);
         ReleaseDC(window,hdc);
         PostQuitMessage(0);
         return 0;
   }
   return DefWindowProc(window,message,wParam,lParam);
}

/****************************************************************************\
*                                                                            *
*  do_bmp                                                                    *
*                                                                            *
*  Display or create a BMP file.                                             *
*                                                                            *
\****************************************************************************/

void do_bmp(UINT selection)
{
   switch (selection)
   {
      case IDM_BMPOPEN:
         if (get_open_filename("BMP files (*.BMP)","*.BMP") == ERR)
            return;
         if (fg_bmphead(file_name,file_header) < 0)
         {
            wsprintf(mb_text,"%s is not a BMP file.",(LPSTR)file_title);
            MessageBox(GetActiveWindow(),mb_text,"BMP",MB_ICONSTOP|MB_OK);
            return;
         }
         fg_bmpsize(file_header,&cxBuffer,&cyBuffer);
         switch_buffers();
         fg_showbmp(file_name,0);
         fg_vbscale(0,cxBuffer-1,0,cyBuffer-1,0,cxClient-1,0,cyClient-1);
         colors = fg_bmppal(file_name,file_palette);

         EnableMenuItem(hmenu,IDM_BMPMAKE,MF_ENABLED);
         EnableMenuItem(hmenu,IDM_BMPINFO,MF_ENABLED);
         EnableMenuItem(hmenu,IDM_PCXMAKE,MF_ENABLED);
         EnableMenuItem(hmenu,IDM_PCXINFO,MF_GRAYED);
         EnableMenuItem(hmenu,IDM_FLICPLAY,MF_GRAYED);
         EnableMenuItem(hmenu,IDM_FLICFRAME,MF_GRAYED);
         EnableMenuItem(hmenu,IDM_FLICRESET,MF_GRAYED);
         EnableMenuItem(hmenu,IDM_FLICINFO,MF_GRAYED);

         return;

      case IDM_BMPMAKE:
         if (get_save_filename("BMP") == OK)
            fg_makebmp(0,cxBuffer-1,0,cyBuffer-1,colors,file_name);
         return;

      case IDM_BMPINFO:
         wsprintf(mb_text,"%s\n%dx%d pixels\n%d colors",
            (LPSTR)open_file,cxBuffer,cyBuffer,colors);
         MessageBox(GetActiveWindow(),mb_text,"BMP",MB_ICONINFORMATION|MB_OK);
         return;
   }
}

/****************************************************************************\
*                                                                            *
*  do_flic                                                                   *
*                                                                            *
*  Play a flic file one frame at a time, or continuously.                    *
*                                                                            *
\****************************************************************************/

void do_flic(UINT selection)
{
   static char context[16];
   static int  frames;

   switch (selection)
   {
      case IDM_FLICOPEN:
         if (get_open_filename("flic files (*.FLI,*.FLC)","*.FLI;*.FLC") == ERR)
            return;
         if (fg_flichead(file_name,file_header) < 0)
         {
            wsprintf(mb_text,"%s is not an FLI or FLC file.",(LPSTR)file_title);
            MessageBox(GetActiveWindow(),mb_text,"flic",MB_ICONSTOP|MB_OK);
            return;
         }
         fg_flicsize(file_header,&cxBuffer,&cyBuffer);
         switch_buffers();
         fg_flicopen(file_name,context);
         fg_flicplay(context,1,0);
         fg_vbscale(0,cxBuffer-1,0,cyBuffer-1,0,cxClient-1,0,cyClient-1);
         colors = 256;
         memcpy(&frames,&file_header[6],2);

         EnableMenuItem(hmenu,IDM_FLICPLAY,MF_ENABLED);
         EnableMenuItem(hmenu,IDM_FLICFRAME,MF_ENABLED);
         EnableMenuItem(hmenu,IDM_FLICRESET,MF_ENABLED);
         EnableMenuItem(hmenu,IDM_FLICINFO,MF_ENABLED);
         EnableMenuItem(hmenu,IDM_BMPMAKE,MF_ENABLED);
         EnableMenuItem(hmenu,IDM_BMPINFO,MF_GRAYED);
         EnableMenuItem(hmenu,IDM_PCXMAKE,MF_ENABLED);
         EnableMenuItem(hmenu,IDM_PCXINFO,MF_GRAYED);

         return;

      case IDM_FLICPLAY:
         fg_showflic(file_name,0,1);
         fg_flicskip(context,-1);
         return;

      case IDM_FLICFRAME:
         if (fg_flicplay(context,1,0) == 0)
         {
            fg_flicskip(context,-1);
            fg_flicplay(context,1,0);
         }
         fg_vbscale(0,cxBuffer-1,0,cyBuffer-1,0,cxClient-1,0,cyClient-1);
         return;

      case IDM_FLICRESET:
         fg_flicskip(context,-1);
         fg_flicplay(context,1,0);
         fg_vbscale(0,cxBuffer-1,0,cyBuffer-1,0,cxClient-1,0,cyClient-1);
         return;

      case IDM_FLICINFO:
         wsprintf(mb_text,"%s\n%dx%d pixels\n%d frames",
            (LPSTR)open_file,cxBuffer,cyBuffer,frames);
         MessageBox(GetActiveWindow(),mb_text,"FLI/FLC",MB_ICONINFORMATION|MB_OK);
         return;
   }
}

/****************************************************************************\
*                                                                            *
*  do_pcx                                                                    *
*                                                                            *
*  Display or create a PCX file.                                             *
*                                                                            *
\****************************************************************************/

void do_pcx(UINT selection)
{
   int minx, maxx, miny, maxy;

   switch (selection)
   {
      case IDM_PCXOPEN:
         if (get_open_filename("PCX files (*.PCX)","*.PCX") == ERR)
            return;
         if (fg_pcxhead(file_name,file_header) < 0)
         {
            wsprintf(mb_text,"%s is not a PCX file.",(LPSTR)file_title);
            MessageBox(GetActiveWindow(),mb_text,"PCX",MB_ICONSTOP|MB_OK);
            return;
         }
         fg_pcxrange(file_header,&minx,&maxx,&miny,&maxy);
         cxBuffer = maxx - minx + 1;
         cyBuffer = maxy - miny + 1;
         switch_buffers();
         fg_move(0,0);
         fg_showpcx(file_name,2);
         fg_vbscale(0,cxBuffer-1,0,cyBuffer-1,0,cxClient-1,0,cyClient-1);
         colors = fg_pcxpal(file_name,file_palette);

         EnableMenuItem(hmenu,IDM_PCXMAKE,MF_ENABLED);
         EnableMenuItem(hmenu,IDM_PCXINFO,MF_ENABLED);
         EnableMenuItem(hmenu,IDM_BMPMAKE,MF_ENABLED);
         EnableMenuItem(hmenu,IDM_BMPINFO,MF_GRAYED);
         EnableMenuItem(hmenu,IDM_FLICPLAY,MF_GRAYED);
         EnableMenuItem(hmenu,IDM_FLICFRAME,MF_GRAYED);
         EnableMenuItem(hmenu,IDM_FLICRESET,MF_GRAYED);
         EnableMenuItem(hmenu,IDM_FLICINFO,MF_GRAYED);

         return;

      case IDM_PCXMAKE:
         if (get_save_filename("PCX") == OK)
            fg_makepcx(0,cxBuffer-1,0,cyBuffer-1,file_name);
         return;

      case IDM_PCXINFO:
         wsprintf(mb_text,"%s\n%dx%d pixels\n%d colors",
            (LPSTR)open_file,cxBuffer,cyBuffer,colors);
         MessageBox(GetActiveWindow(),mb_text,"PCX",MB_ICONINFORMATION|MB_OK);
         return;
   }
}

/****************************************************************************\
*                                                                            *
*  get_open_filename                                                         *
*                                                                            *
*  Display a dialog box that selects a list of file names that match the     *
*  specified file name. This function uses the Open File dialog box from the *
*  Windows common dialog box library.                                        *
*                                                                            *
*  The selected file name is stored with full path information in the global *
*  variable file_name, and without path information in file_title.           *
*                                                                            *
\****************************************************************************/

int get_open_filename(char *matchtype, char *matchfile) 
{
   OPENFILENAME fn;
   register int i;
   char filter[128];

   // construct file name filter string
   lstrcpy(filter,matchtype);
   i = lstrlen(filter) + 1;
   lstrcpy(&filter[i],matchfile);
   i += lstrlen(&filter[i]) + 1;
   filter[i] = '\0';
   default_file[0] = '\0';

   // fill in structure fields for Open File dialog box
   fn.lStructSize       = sizeof(OPENFILENAME);
   fn.hwndOwner         = GetActiveWindow();
   fn.lpstrFilter       = filter;
   fn.lpstrCustomFilter = NULL;
   fn.nFilterIndex      = 1;
   fn.lpstrFile         = default_file;
   fn.nMaxFile          = 256;
   fn.lpstrFileTitle    = file_title;
   fn.nMaxFileTitle     = 16;
   fn.lpstrInitialDir   = NULL;
   fn.lpstrTitle        = NULL;
   fn.Flags             = 0;
   fn.lpstrDefExt       = &matchfile[2];

   // activate the Open File dialog box
   if (GetOpenFileName(&fn))
   {
      lstrcpy(file_name,default_file);
      lstrcpy(open_file,file_title);
      return(OK);
   }
   else
      return(ERR);
}

/****************************************************************************\
*                                                                            *
*  get_save_filename                                                         *
*                                                                            *
*  Display a dialog box that selects a list of file names that match the     *
*  specified file name. This function uses the Open File dialog box from the *
*  Windows common dialog box library.                                        *
*                                                                            *
*  The selected file name is stored with full path information in the global *
*  variable file_name, and without path information in file_title.           *
*                                                                            *
\****************************************************************************/

int get_save_filename(char *extension)
{
   OPENFILENAME fn;
   register int i;

   // construct default file name for the dialog box edit control

   lstrcpy(default_file,file_title);
   for (i = 0; i < lstrlen(file_title); i++)
   {
      if (default_file[i] == '.')
         default_file[i] = '\0';
   }

   // fill in structure fields for Open File dialog box
   fn.lStructSize       = sizeof(OPENFILENAME);
   fn.hwndOwner         = GetActiveWindow();
   fn.lpstrFilter       = NULL;
   fn.lpstrCustomFilter = NULL;
   fn.lpstrFile         = default_file;
   fn.nMaxFile          = 256;
   fn.lpstrFileTitle    = file_title;
   fn.nMaxFileTitle     = 16;
   fn.lpstrInitialDir   = NULL;
   fn.lpstrTitle        = NULL;
   fn.Flags             = OFN_OVERWRITEPROMPT | OFN_PATHMUSTEXIST;
   fn.lpstrDefExt       = extension;

   // activate the Open File dialog box
   if (GetSaveFileName(&fn))
   {
      lstrcpy(file_name,default_file);
      return(OK);
   }
   else
      return(ERR);
}

/****************************************************************************\
*                                                                            *
*  switch_buffers                                                            *
*                                                                            *
*  Close the and release the active virtual buffer, then create and open a   *
*  new virtual buffer to hold the new image file.                            *
*                                                                            *
\****************************************************************************/

void switch_buffers()
{
   fg_vbclose();
   fg_vbfree(hvb);
   hvb = fg_vballoc(cxBuffer,cyBuffer);
   fg_vbopen(hvb);
   fg_vbcolors();
}
