{*****************************************************************************
*                                                                            *
*  FISH1.PAS                                                                 *
*                                                                            *
*  This is the original DOS version of the Fastgraph Fish Tank.              *
*                                                                            *
*****************************************************************************}

program fishtank;

uses FGmain, FGbitmap, FGmisc, FGpr;

var

  { fish bitmaps }

  fish1 : array[0..1399] of byte;
  fish2 : array[0..2051] of byte;
  fish3 : array[0..1767] of byte;
  fish4 : array[0..1679] of byte;
  fish5 : array[0..1363] of byte;
  fish6 : array[0..2447] of byte;

const

  { number of fish }

  NFISH = 11;

  { location of fish (x & y) }

  fish_x : array[0..5] of integer = (0,   64,128,200,  0, 80);
  fish_y : array[0..5] of integer = (199,199,199,199,150,150);

  { size of fish (width and height) }

  width   : array[0..5] of integer = ( 56, 54, 68, 56, 62, 68);
  height  : array[0..5] of integer = ( 25, 38, 26, 30, 22, 36);

  { for convenience, an array of pointers to fish bitmaps }

  fishes : array[0..5] of ^byte = (@fish1,@fish2,@fish3,@fish4,@fish5,@fish6);

{*****************************************************************************
*                                                                            *
*  min                                                                       *
*                                                                            *
*  Determine the smaller of two integer values.                              *
*                                                                            *
*****************************************************************************}

function min (value1, value2 : integer) : integer;

begin

  if (value1 <= value2) then
    min := value1
  else
    min := value2;

end;

{*****************************************************************************
*                                                                            *
*  max                                                                       *
*                                                                            *
*  Determine the larger of two integer values.                               *
*                                                                            *
*****************************************************************************}

function max (value1, value2 : integer) : integer;

begin

  if (value1 >= value2) then
    max := value1
  else
    max := value2;

end;

{*****************************************************************************
*                                                                            *
*  get_fish                                                                  *
*                                                                            *
*  Fill the fish bitmap arrays.                                              *
*                                                                            *
*****************************************************************************}

procedure get_fish;

var
  fish_num : integer;

begin

  { get the fish bitmaps from an PPR file }

  fg_setpage(1);
  fg_move(0,199);
  fg_showppr('FISH.PPR'+chr(0),320);

  for fish_num := 0 to 5 do
  begin
    fg_move(fish_x[fish_num],fish_y[fish_num]);
    fg_getimage(fishes[fish_num]^,width[fish_num],height[fish_num]);
  end;

end;

{*****************************************************************************
*                                                                            *
*  put_fish                                                                  *
*                                                                            *
*  Draw one of the six fish anywhere you want.                               *
*                                                                            *
*****************************************************************************}

procedure put_fish (fish_num, x, y, fish_dir : integer);

begin

  { move to position where the fish will appear }

  fg_move(x,y);

  { draw a left- or right-facing fish, depending on fish_dir }

  if (fish_dir = 0) then
    fg_flpimage(fishes[fish_num]^,width[fish_num],height[fish_num])
  else
    fg_clpimage(fishes[fish_num]^,width[fish_num],height[fish_num]);

end;

{*****************************************************************************
*                                                                            *
*  go_fish                                                                   *
*                                                                            *
*  Make the fish swim around.                                                *
*                                                                            *
*****************************************************************************}

procedure go_fish;

type

  fish_array = array[0..NFISH-1] of integer;

const
{
*   There are 11 fish total, and 6 different kinds of fish. These
*   arrays keep track of what kind of fish each fish is, and how each
*   fish moves:
*
*   fish()   -- which fish bitmap applies to this fish?
*   x()      -- starting x coordinate
*   y()      -- starting y coordinate
*
*   xmin()   -- how far left (off screen) the fish can go
*   xmax()   -- how far right (off screen) the fish can go
*   xinc()   -- how fast the fish goes left and right
*   dir()    -- starting direction for each fish
*
*   ymin()   -- how far up this fish can go
*   ymax()   -- how far down this fish can go
*   yinc()   -- how fast the fish moves up or down
*   yturn()  -- how long fish can go in the vertical direction
*               before stopping or turning around
*   ycount() -- counter to compare to yturn
}

  fish   : fish_array =
                (   1,   1,   2,   3,   3,   0,   0,   5,   4,   2,   3);
  x      : fish_array =
                (-100,-150,-450,-140,-200, 520, 620,-800, 800, 800,-300);
  y      : fish_array =
                (  40,  60, 150,  80,  70, 190, 180, 100,  30, 130,  92);

  xmin   : fish_array =
                (-300,-300,-800,-200,-200,-200,-300,-900,-900,-900,-400);
  xmax   : fish_array =
                ( 600, 600,1100,1000,1000, 750, 800,1200,1400,1200, 900);
  xinc   : fish_array =
                (   2,   2,   8,   5,   5,  -3,  -3,   7,  -8,  -9,   6);
  dir    : fish_array =
                (   0,   0,   0,   0,   0,   1,   1,   0,   1,   1,   0);

  ymin   : fish_array =
                (  40,  60, 120,  70,  60, 160, 160,  80,  30, 110,  72);
  ymax   : fish_array =
                (  80, 100, 170, 110, 100, 199, 199, 120,  70, 150, 122);
  yturn  : fish_array =
                (  50,  30,  10,  30,  20,  10,  10,  10,  30,   20, 10);
  ycount : fish_array =
                (   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0);
  yinc   : fish_array =
                (   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0);

var
  i : integer;
  visual_page, work_page : integer;
  key, aux : byte;

begin

  { initially page 0 is the visual page and page 1 is the workspace page }

  visual_page := 0;
  work_page := 1;

  { make the fish swim around }

  repeat
  begin
    { copy the background to the workspace }

    fg_copypage(2,work_page);

    { put all the fish in their new positions }

    for i := 0 to NFISH-1 do
    begin
      inc(ycount[i]);
      if (ycount[i] > yturn[i]) then
      begin
         ycount[i] := 0;
         yinc[i] := random(3) - 1;
      end;
      inc(y[i],yinc[i]);
      y[i] := min(ymax[i],max(y[i],ymin[i]));

      if (x[i] >= -72) and (x[i] < 320) then
        put_fish(fish[i],x[i],y[i],dir[i]);

      inc(x[i],xinc[i]);
      if (x[i] <= xmin[i]) or (x[i] >= xmax[i]) then
      begin
        xinc[i] := -xinc[i];
        dir[i] := 1 - dir[i];
      end;
    end;

    { page flip to display the new frame }

    visual_page := work_page;
    fg_setvpage(visual_page);

    { the old visual page becomes the workspace next frame }
    work_page := 1 - work_page;
    fg_setpage(work_page);

    { intercept a keystroke, if it is escape exit the program }

    fg_intkey(key,aux);

  end;
  until (key = 27);

end;

{*****************************************************************************
*                                                                            *
*  main program                                                              *
*                                                                            *
*****************************************************************************}

var
  old_mode : integer;

begin

  { in case we're compiling for protected mode }

  fg_initpm;

  { make sure the system supports video mode 20 with three pages }

  if (fg_testmode(20,3) = 0) then
  begin
     writeln('This program requires VGA graphics.');
     Halt(255);
  end;

  { initialize the video environment }

  old_mode := fg_getmode;
  fg_setmode(20);

  { display the coral background on page 2 (which will always }
  { contain a clean copy of the background image) }

  fg_setpage(2);
  fg_move(0,199);
  fg_showppr('CORAL.PPR'+chr(0),320);

  { copy the background to the visual page }

  fg_copypage(2,0);

  { get the fish }

  get_fish;

  { make the fish go }

  go_fish;

  { restore the original video state }

  fg_setmode(old_mode);
  fg_reset;
end.
