/*
 * xmode.c - general mode-x set and detection routines
 *
 * DESCRIPTION
 * Includes mode-x set, VGA detection, logical screen size set,
 * and vertical retrace detection functions
 *
 * USAGE
 *
 * NOTES
 * Based on xlib60 for BC by Themie Goumas.  Detection and Retrace code
 * adapted from code by Dave Roberts in PC Techniques, V6N1, Apr/May 95.
 *
 * REVISION HISTORY
 * Date         Reason
 * 27 Jun 95    Initial Release
 * 13 Jul 95    Added split-screen functions
 * 31 Aug 95    Updated for clipping variables
 *
 */

/******************************************************/
/* xmode.c - mode-x mode set routines                 */
/******************************************************/
#include <dos.h>
#include <go32.h>
#include <dpmi.h>
#include <pc.h>

#include "defines.h"
#include "xlib.h"

#define SEQU_ADDR               0x3c4
#define CRTC_ADDR               0x3d4
#define MISC_OUTPUT             0x3c2
#define INPUT_STAT_1_REG        0x3DA
#define VR_BIT                  0x08
#define DISPLAY_ENABLE          0x01
#define CRTC_DATA    0x3D5 /* CRT Controller Data register */
#define OVERFLOW     0x07  /* index of CRTC reg holding bit 8 of the
                              line the split screen starts after */
#define MAX_SCAN     0x09  /* index of CRTC reg holding bit 9 of the
                              line the split screen starts after */
#define LINE_COMPARE 0x18  /* index of CRTC reg holding lower 8 bits
                              of line split screen starts after */
#define AC_MODE_CONTROL         0x10
#define AC_INDEX                0x3c0

/* global variables which are initialized in this module */
int Page0_Offs;
int Page1_Offs;
int VisiblePageOffs;
int HiddenPageOffs;
int NonVisibleOffs;
int ScrnPhysicalByteWidth;
int ScrnPhysicalPixelWidth;
int ScrnPhysicalHeight;
int ScrnLogicalByteWidth;
int ScrnLogicalHeight;
int XClipMin;
int YClipMin;
int XClipMax;
int YClipMax;
BOOLEAN InGraphics;
int VisibleOffset;
int Page0_Disp_Offs;
int Page1_Disp_Offs;
int OldVideoMode;
WORD core_select;

int SplitLines;
int SplitStartOffset;
int SplitStartLine;
int NonSplitLines;

extern WORD _core_select;

/* Choose defaults mouse module */
BOOLEAN MouseDriverActive = FALSE;

#define DJGPP_V20

/*************************************************************************
** x_set_mode(int resolution);
** sets the specified mode or returns FALSE if invalid mode requested
** or VGA not detected.
** The crashing problem has been fixed from the v112 version - I think
** I wasn't zeroing all the registers before _go32_dpmi_simulate_interrupt
** Fixed by using __dpmi_int, which does this for you!
*************************************************************************/
int x_set_mode(int resolution)
{
  __dpmi_regs r;
  char in_byte;

  core_select = (unsigned)_dos_ds;

  if ((resolution < 1) || (resolution > 6)) /* Not a valid mode */
    return(FALSE);
  
  if (x_detect_VGA() == FALSE)
    return FALSE;
  
  /* Save old video mode to restore to */
  r.x.ax = 0x0f00;
  __dpmi_int(0x10, &r);
  OldVideoMode = (int)r.h.al;
  
  r.x.ax = 0x0013;
  __dpmi_int(0x10, &r);    /* standard mode 13h */
  
  InGraphics = 1;
  XClipMin = 0;
  YClipMin = 0;
  VisibleOffset = 0;

  switch (resolution)
  {
    case X_320X200:
      outportw(SEQU_ADDR, 0x0604);
      outportw(CRTC_ADDR, 0xe317);
      outportw(CRTC_ADDR, 0X0014);
      x_clear_buffer();         /* asm routine to clear all 256k video ram */
      Page0_Offs = 0; 
      Page1_Offs = 16000;
      VisiblePageOffs = 0;
      HiddenPageOffs = 16000;
      NonVisibleOffs = 32000;
      ScrnPhysicalByteWidth = 80;
      ScrnPhysicalPixelWidth = 320;
      ScrnPhysicalHeight = 200;
      ScrnLogicalByteWidth = 80;
      ScrnLogicalHeight = 200;
      XClipMax = 79;
      YClipMax = 199;
      break;
    case X_320X240:
      outportw(SEQU_ADDR, 0x0604);
      outportw(SEQU_ADDR, 0x0110);
      outportb(MISC_OUTPUT, 0xe3);
      outportw(SEQU_ADDR, 0x0300);
      outportb(CRTC_ADDR, 0x11);
      in_byte = inportb((CRTC_ADDR+1));
      in_byte = (in_byte && 0x7f);
      outportb((CRTC_ADDR+1), in_byte);
      outportw(CRTC_ADDR, 0x0d06);
      outportw(CRTC_ADDR, 0x3e07);
      outportw(CRTC_ADDR, 0x4109); /* 4009 for no vertical doubling */
      outportw(CRTC_ADDR, 0xea10);
      outportw(CRTC_ADDR, 0xac11);
      outportw(CRTC_ADDR, 0xdf12);
      outportw(CRTC_ADDR, 0x0014);
      outportw(CRTC_ADDR, 0xe715);
      outportw(CRTC_ADDR, 0x0616);
      outportw(CRTC_ADDR, 0xe317);
      x_clear_buffer();
      /* asm routine to clear all 256k video ram */
      Page0_Offs = 0; 
      Page1_Offs = 19200;
      VisiblePageOffs = 0;
      HiddenPageOffs = 19200;
      NonVisibleOffs = 38400;
      ScrnPhysicalByteWidth = 80;
      ScrnPhysicalPixelWidth = 320;
      ScrnPhysicalHeight = 240;
      ScrnLogicalByteWidth = 80;
      ScrnLogicalHeight = 240;
      XClipMax = 79;
      YClipMax = 239;
      break;
    case X_320X400:
      outportw(SEQU_ADDR, 0x0604);
      outportw(CRTC_ADDR, 0x4009);
      outportw(CRTC_ADDR, 0xe317);
      outportw(CRTC_ADDR, 0X0014);
      outportb(SEQU_ADDR, 0xe3);
      x_clear_buffer();         /* asm routine to clear all 256k video ram */
      Page0_Offs = 0; 
      Page1_Offs = 32000;
      VisiblePageOffs = 0;
      HiddenPageOffs = 32000;
      NonVisibleOffs = 64000;
      ScrnPhysicalByteWidth = 80;
      ScrnPhysicalPixelWidth = 320;
      ScrnPhysicalHeight = 400;
      ScrnLogicalByteWidth = 80;
      ScrnLogicalHeight = 400;
      XClipMax = 79;
      YClipMax = 399;
      break;
    case X_320X480:
      outportw(SEQU_ADDR, 0x0604);
      outportw(SEQU_ADDR, 0x0110);
      outportb(MISC_OUTPUT, 0xe3);
      outportw(SEQU_ADDR, 0x0300);
      outportb(CRTC_ADDR, 0x11);
      in_byte = inportb((CRTC_ADDR+1));
      in_byte = (in_byte && 0x7f);
      outportb((CRTC_ADDR+1), in_byte);
      outportw(CRTC_ADDR, 0x0d06);
      outportw(CRTC_ADDR, 0x3e07);
      outportw(CRTC_ADDR, 0x4009); /* 4109 for vertical pixel doubling */
      outportw(CRTC_ADDR, 0xea10);
      outportw(CRTC_ADDR, 0xac11);
      outportw(CRTC_ADDR, 0xdf12);
      outportw(CRTC_ADDR, 0x0014);
      outportw(CRTC_ADDR, 0xe715);
      outportw(CRTC_ADDR, 0x0616);
      outportw(CRTC_ADDR, 0xe317);
      x_clear_buffer();         /* asm routine to clear all 256k video ram */
      Page0_Offs = 0; 
      Page1_Offs = 0;           /* NOTE: 1 Page only in 320x480 */
      VisiblePageOffs = 0;
      HiddenPageOffs = 0;
      NonVisibleOffs = 38400;
      ScrnPhysicalByteWidth = 80;
      ScrnPhysicalPixelWidth = 320;
      ScrnPhysicalHeight = 480;
      ScrnLogicalByteWidth = 80;
      ScrnLogicalHeight = 480;
      XClipMax = 79;
      YClipMax = 479;
      break;
    case X_360X240:
      outportw(SEQU_ADDR, 0x0604);
      outportw(SEQU_ADDR, 0x100);
      outportb(MISC_OUTPUT, 0xe7);
      outportw(SEQU_ADDR, 0x300);
      outportb(CRTC_ADDR, 0x11);
      in_byte = inportb((CRTC_ADDR+1));
      in_byte = (in_byte && 0x7f);
      outportb((CRTC_ADDR+1), in_byte);
      outportw(CRTC_ADDR, 0x6b00);
      outportw(CRTC_ADDR, 0x5901);
      outportw(CRTC_ADDR, 0x5a02);
      outportw(CRTC_ADDR, 0x8e03);
      outportw(CRTC_ADDR, 0x5e04);
      outportw(CRTC_ADDR, 0x8a05);
      outportw(CRTC_ADDR, 0x0d06);
      outportw(CRTC_ADDR, 0x3e07);
      outportw(CRTC_ADDR, 0x4109);
      outportw(CRTC_ADDR, 0xea10);
      outportw(CRTC_ADDR, 0xac11);
      outportw(CRTC_ADDR, 0xdf12);
      outportw(CRTC_ADDR, 0x2d13);
      outportw(CRTC_ADDR, 0x0014);
      outportw(CRTC_ADDR, 0xe715);
      outportw(CRTC_ADDR, 0x0616);
      outportw(CRTC_ADDR, 0xe317);
      x_clear_buffer();         /* asm routine to clear all 256k video ram */
      Page0_Offs = 0; 
      Page1_Offs = 21600;
      VisiblePageOffs = 0;
      HiddenPageOffs = 21600;
      NonVisibleOffs = 43200;
      ScrnPhysicalByteWidth = 90;
      ScrnPhysicalPixelWidth = 360;
      ScrnPhysicalHeight = 240;
      ScrnLogicalByteWidth = 90;
      ScrnLogicalHeight = 240;
      XClipMax = 89;
      YClipMax = 239;
      break;
    case X_360X480:
      outportw(SEQU_ADDR, 0x0604);
      outportw(SEQU_ADDR, 0x100);
      outportb(MISC_OUTPUT, 0xe7);
      outportw(SEQU_ADDR, 0x300);
      outportb(CRTC_ADDR, 0x11);
      in_byte = inportb((CRTC_ADDR+1));
      in_byte = (in_byte && 0x7f);
      outportb((CRTC_ADDR+1), in_byte);
      outportw(CRTC_ADDR, 0x6b00);
      outportw(CRTC_ADDR, 0x5901);
      outportw(CRTC_ADDR, 0x5a02);
      outportw(CRTC_ADDR, 0x8e03);
      outportw(CRTC_ADDR, 0x5e04);
      outportw(CRTC_ADDR, 0x8a05);
      outportw(CRTC_ADDR, 0x0d06);
      outportw(CRTC_ADDR, 0x3e07);
      outportw(CRTC_ADDR, 0x4009);
      outportw(CRTC_ADDR, 0xea10);
      outportw(CRTC_ADDR, 0xac11);
      outportw(CRTC_ADDR, 0xdf12);
      outportw(CRTC_ADDR, 0x2d13);
      outportw(CRTC_ADDR, 0x0014);
      outportw(CRTC_ADDR, 0xe715);
      outportw(CRTC_ADDR, 0x0616);
      outportw(CRTC_ADDR, 0xe317);
      x_clear_buffer();         /* asm routine to clear all 256k video ram */
      Page0_Offs = 0; 
      Page1_Offs = 0;
      VisiblePageOffs = 0;
      HiddenPageOffs = 0;
      NonVisibleOffs = 43200;
      ScrnPhysicalByteWidth = 90;
      ScrnPhysicalPixelWidth = 360;
      ScrnPhysicalHeight = 480;
      ScrnLogicalByteWidth = 90;
      ScrnLogicalHeight = 480;
      XClipMax = 89;
      YClipMax = 479;
      break;
  }
  return(TRUE);
}

/***********************************************************
**  Sets the logical screen size, centering the screen
**  in its new coordinates.  x is the number of BYTES to ADD
**  to the physical width, y is the number of rows to add to
**  the physical height.
**  The x value will adjust the row address offset, and the
**  y value will be used to change the PageOffset addresses.
**
**  Note:  x and y should be even values.
**  DO NOT SET VALUES TO LESS THAN PHYSICAL SIZES.
***********************************************************/
void x_set_logical_size(int x, int y)
{
  BYTE byte_width;

  byte_width = (BYTE)ScrnPhysicalByteWidth + (BYTE)x;
  outportb(0x3d4, 0x13);
  outportb(0x3d5, (byte_width >> 1));
  ScrnLogicalByteWidth = (int) byte_width;
  ScrnLogicalHeight = (int) y + ScrnPhysicalHeight;
  VisibleOffset = (int)(x / 2);
  
  Page0_Offs = 0;
  Page1_Offs = ScrnLogicalHeight * ScrnLogicalByteWidth;
  
  /* These are the offsets to centre each page in its virtual screen */
  Page0_Disp_Offs = (y >> 1) * ScrnLogicalByteWidth + 
                    Page0_Offs + VisibleOffset;
  Page1_Disp_Offs = (y >> 1) * ScrnLogicalByteWidth + 
                    Page1_Offs + VisibleOffset;
  
  x_show_page(Page0_Offs);

  /* These point to the top-left corner of each page */
  VisiblePageOffs = 0;
  HiddenPageOffs = ScrnLogicalHeight * ScrnLogicalByteWidth;
}

/* Turn on the split screen at the desired line (minus 1 because the
** split screen starts *after* the line specified by the LINE_COMPARE
** register) (bit 8 of the split screen start line is stored in the
** Overflow register, and bit 9 is in the Maximum Scan Line reg) 
** NOTE:  Always set the virtual screen size before setting up the
** split screen.
** Split Screen code by M. Abrash, PCTechniques, June 1991.
*/
void x_set_split_screen(int startline)
{
   BYTE inbyte;

   SplitStartLine = startline;
   SplitStartOffset = 0;
   SplitLines = ScrnLogicalHeight - SplitStartLine;
   NonSplitLines = ScrnLogicalHeight - SplitLines;

   /* Turn off pixel panning for split screen area */
   inportb(INPUT_STAT_1_REG);
   outportb(AC_INDEX, AC_MODE_CONTROL+0x20);
   inbyte = inportb(AC_INDEX+1);
   inbyte |= 0x20;
   outportb(AC_INDEX, inbyte);

   /* 200 & 240 line modes are double-scanned */
   if (ScrnPhysicalHeight < 400)
     SplitStartLine = SplitStartLine << 1;
     
   outportb(CRTC_ADDR, LINE_COMPARE);
   outportb(CRTC_DATA, (SplitStartLine - 1) & 0xFF);
   outportb(CRTC_ADDR, OVERFLOW);
   outportb(CRTC_DATA, (((((SplitStartLine - 1) & 0x100) >> 8) << 4) |
         (inportb(CRTC_DATA) & ~0x10)));
   outportb(CRTC_ADDR, MAX_SCAN);
   outportb(CRTC_DATA, (((((SplitStartLine - 1) & 0x200) >> 9) << 6) |
         (inportb(CRTC_DATA) & ~0x40)));
   
   /* undo double scanning correction */
   if (ScrnPhysicalHeight < 400)
     SplitStartLine = SplitStartLine >> 1;

   Page0_Offs = SplitLines * ScrnLogicalByteWidth;
   Page1_Offs = ScrnLogicalHeight * ScrnLogicalByteWidth + Page0_Offs;
   x_show_page(Page0_Offs);
   VisiblePageOffs = Page0_Offs;
   HiddenPageOffs = Page1_Offs;
}


/**************************************************** 
**  x_detect_VGA() - tries to detect VGA card
**  RETURNS:  TRUE -  VGA detected
**            FALSE - VGA not detected
****************************************************/
BOOLEAN x_detect_VGA()
{
  __dpmi_regs regs;
    
  /* try a BIOS call only the VGA understands */
  regs.x.bx = 0xffff;
  regs.x.ax = 0x101a;
  __dpmi_int(0x10, &regs);
    
  /* if the card didn't understand, it's not a VGA */
  if (regs.x.bx == 0xffff) 
    return FALSE;
  else 
    return TRUE;
}


/************************************************************************
**  Function: x_wait_vertical_retrace_start
**  Description:
**      Waits for the VGA to begin vertical retrace.  Note that if the
**      VGA is already in vertical retrace, the routine will wait all
**      during the next display frame until vertical retrace begins
**      again.
************************************************************************/
void x_wait_vertical_retrace_start(void)
{
  BYTE InputStatus1;

  /* make sure we aren't in retrace already */
  do 
  {
    InputStatus1 = inportb(INPUT_STAT_1_REG);
  } while (InputStatus1 & VR_BIT);

  /* wait for retrace to start */
  do 
  {
    InputStatus1 = inportb(INPUT_STAT_1_REG);
  } while (!(InputStatus1 & VR_BIT));
}

/**********************************************************************
**  Function: x_wait_vertical_retrace_end
**  Description:
**      Waits for vertical retrace to complete.  If the VGA is not
**      in vertical retrace, it waits until the VGA enters vertical
**      retrace again and the retrace subsequently completes.
**********************************************************************/
void x_wait_vertical_retrace_end(void)
{
  BYTE InputStatus1;

  /* make sure we aren't in display state (= not retrace) */
  do 
  {
    InputStatus1 = inportb(INPUT_STAT_1_REG);
  } while (!(InputStatus1 & VR_BIT));
  
  /* wait for retrace to end */
  do 
  {
    InputStatus1 = inportb(INPUT_STAT_1_REG);
  } while (InputStatus1 & VR_BIT);
}

/**********************************************************************
**  Function: x_wait_display_mode
**  Description:
**      If the VGA is currently in display mode, the function returns
**      immediately, else it waits until diplay mode is entered.
**********************************************************************/
void x_wait_display_mode(void)
{
  BYTE InputStatus1;

  /* wait for retrace to end and display mode to start */
  do 
  {
    InputStatus1 = inportb(INPUT_STAT_1_REG);
  } while (InputStatus1 & DISPLAY_ENABLE);
}

/*********************************************************************
**  Function: x_wait_retrace_mode
**  Description:
**      If the VGA is currently in retrace mode, the function returns
**      immediately, else it waits until retrace mode is entered.
*********************************************************************/
void x_wait_retrace_mode(void)
{
  BYTE InputStatus1;

  /* wait for display mode to end and retrace to start */
  do 
  {
    InputStatus1 = inportb(INPUT_STAT_1_REG);
  } while (!(InputStatus1 & VR_BIT));
}

/********************************************************************
**  Function: x_vertical_retrace_occuring
**  Description:
**      Returns the current state of the VGA vertical retrace.  If
**      retrace is occuring, the function returns TRUE.
********************************************************************/
BOOLEAN x_vertical_retrace_occuring(void)
{
  return (inportb(INPUT_STAT_1_REG) & VR_BIT) >> 3;
}

