/*
 * Copyright (c) 1992 The Regents of the University of California.
 * All rights reserved.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice and the following
 * two paragraphs appear in all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */
/* This file contains C code to implement an ordered dither. */

#include "video.h"
#include "proto.h"
#include "dither.h"

static unsigned char ytab[256+16];
static unsigned char utab[128+16];
static unsigned char vtab[128+16];

static unsigned char uvtab[256*269+270];

/*
 *--------------------------------------------------------------
 *
 *  InitOrderedDither--
 *
 *	Structures intialized for ordered dithering. 
 *
 * Results:
 *	None.
 *
 * Side effects:
 *      None.
 *
 *--------------------------------------------------------------
 */

void
InitOrderedDither()
{
  int i, j, v, bpp=8;

  for (i=-8; i<256+8; i++)
  {
    v = i>>4;
    if (v<1)
      v = 1;
    else if (v>14)
      v = 14;
    ytab[i+8] = v<<4;
  }

  for (i=0; i<128+16; i++)
  {
    v = (i-40)>>4;
    if (v<0)
      v = 0;
    else if (v>3)
      v = 3;
    utab[i] = v<<2;
    vtab[i] = v;
  }
}

/*
 *--------------------------------------------------------------
 *
 * OrderedDitherImage --
 *
 *	Dithers an image using an ordered dither.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

void
OrderedDitherImage (lum, cr, cb, out, h, w)
    unsigned char *lum;
    unsigned char *cr;
    unsigned char *cb;
    unsigned char *out;
    int w, h;
{
  int i,j;
  int y,u,v;
  unsigned char *py,*pu,*pv,*dst;

  py = lum;
  pu = cr;
  pv = cb;
  
#ifdef _WIN32
  dst = out+(h-1)*w;
#else
  dst = out;
#endif

  for (j=0; j<h; j+=4)
  {
    /* line j + 0 */
    for (i=0; i<w; i+=4)
    {
      y = *py++;
      u = *pu++ >> 1;
      v = *pv++ >> 1;
      *dst++ = ytab[y]|utab[u]|vtab[v]	;
      y = *py++;
      *dst++ = ytab[y+8]|utab[u+8]|vtab[v+8]	;
      y = *py++;
      u = *pu++ >> 1;
      v = *pv++ >> 1;
      *dst++ = ytab[y+2]|utab[u+2]|vtab[v+2]	;
      y = *py++;
      *dst++ = ytab[y+10]|utab[u+10]|vtab[v+10]	;
    }

    pu -= w/2;
    pv -= w/2;

#ifdef _WIN32
    dst -= 2*w;
#endif

    /* line j + 1 */
    for (i=0; i<w; i+=4)
    {
      y = *py++;
      u = *pu++ >> 1;
      v = *pv++ >> 1;
      *dst++ = ytab[y+12]|utab[u+12]|vtab[v+12]	;
      y = *py++;
      *dst++ = ytab[y+4]|utab[u+4]|vtab[v+4]	;
      y = *py++;
      u = *pu++ >> 1;
      v = *pv++ >> 1;
      *dst++ = ytab[y+14]|utab[u+14]|vtab[v+14]	;
      y = *py++;
      *dst++ = ytab[y+6]|utab[u+6]|vtab[v+6]	;
    }

#ifdef _WIN32
    dst -= 2*w;
#endif

    /* line j + 2 */
    for (i=0; i<w; i+=4)
    {
      y = *py++;
      u = *pu++ >> 1;
      v = *pv++ >> 1;
      *dst++ = ytab[y+3]|utab[u+3]|vtab[v+3]	;
      y = *py++;
      *dst++ = ytab[y+11]|utab[u+11]|vtab[v+11]	;
      y = *py++;
      u = *pu++ >> 1;
      v = *pv++ >> 1;
      *dst++ = ytab[y+1]|utab[u+1]|vtab[v+1]	;
      y = *py++;
      *dst++ = ytab[y+9]|utab[u+9]|vtab[v+9]	;
    }

    pu -= w/2;
    pv -= w/2;

#ifdef _WIN32
    dst -= 2*w;
#endif

    /* line j + 3 */
    for (i=0; i<w; i+=4)
    {
      y = *py++;
      u = *pu++ >> 1;
      v = *pv++ >> 1;
      *dst++ = ytab[y+15]|utab[u+15]|vtab[v+15]	;
      y = *py++;
      *dst++ = ytab[y+7]|utab[u+7]|vtab[v+7]	;
      y = *py++;
      u = *pu++ >> 1;
      v = *pv++ >> 1;
      *dst++ = ytab[y+13]|utab[u+13]|vtab[v+13]	;
      y = *py++;
      *dst++ = ytab[y+5]|utab[u+5]|vtab[v+5];
    }

#ifdef _WIN32
    dst -= 2*w;
#endif
  }

}


  
