/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                 Test 3D Star Field function                   *
 *                                                               *
 *   Copyright 1995 by David S. Issel, Sacramento, California    *
 *                     Internet:  dissel@nunic.nu.edu            *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <signal.h>
#include <stdlib.h>
#include <conio.h>
#include <stdio.h>
#include <string.h>

#include "svgacc.h"

/* define constants */

#define WHITE 15
#define DARK 8
#define DKGRAY 8
#define GRAY 7
#define BLACK 0
#define RED 4
#define UP 0
#define DOWN 1
#define FALSE 0
#define TRUE 1
#define MAXSTARFIELD 150
#define MAXSPEED 90

/* define global variables */

int fontwidth,fontheight;
int time_to_exit;
int specified_MAXSTARFIELD=100;   /* default 100 stars */
int specified_MAXSPEED=40;        /* default max speed 40 */

/* define special areas of the screen */

typedef struct {
   int x;
   int y;
   int w;
   int h;
} box;
box boxes[10] =
{
   {0,0,0,0},         /* Not found!      0  */
   {100,150,145,20},  /* Stars bar       1  */
   {100,250,145,20},  /* Speed bar       2  */
   {400,325,95,40},   /* Show button     3  */
   {410,385,75,22},   /* Exit button     4  */
   {50,350,245,20},   /* Status box      5  */
   {350,100,200,200}, /* Star field box  6  */
   {65,70,209,15},    /* Program title   7  */
   {40,85,257,15}     /* Copyright Msg   8  */
};
int maxboxes=8;

/*  star field array  */
typedef struct {
   int x;
   int y;
   int z;
   int old_z;
}star_of_star_field;
star_of_star_field star_field[MAXSTARFIELD+1];
int cur_star_field_stars;

/* define function prototypes */

void setup_palette(void);
void fade_to_black(void);
void draw_box(int x, int y, int w, int h, int depth, int inside_color);
void draw_button(int x, int y, int w, int h, char *caption, int down);
void draw_speedbar(int x, int y, int w, int h, int inside_color, int bar_color,int max_val, int cur_val);
void set_status_box_text(char *new_msg);
void fill_boxes_array(void);
int desired_percent(int x, int w, int max_val, int mouseX);
int mouse_over(int mouseX, int mouseY, box *myboxes, int mymaxboxes);
void do_3d_star_field(void);


void main(void)
{
   int vmode;                     /* original video mode           */
   int mousex,mousey,mousebut;    /* mouse location and status     */
   int oldx,oldy,oldmousebut;     /* old mouse location and status */
   int new_stars,new_speed;       /* temp variables                */
   char buf[100];                 /* for message strings           */

   signal(SIGINT,SIG_IGN);  /* Ignore Ctrl-C during program run */

   /* do hardware check */

   if ( whichcpu()<386 ) {
      puts("386 CPU or better required.");
      exit(1);
   }
   vmode = videomodeget();     /* save the original video mode */
   if ( !whichvga() ) {
      puts("Video card not supported.");
      exit(1);
   }
   if ( whichmem() < 512) {
      puts("Not enough memory on video card.");
      exit(1);
   }
   if ( !res640() ) {
      videomodeset(vmode);
      puts("Could not switch to 640 x 480.");
      exit(1);
   }
   if ( !whichmouse() ) {
      videomodeset(vmode);
      puts("Mouse driver not found.");
      exit(1);
   }

   /* initialize the mouse */
   mousecursordefault();
   mouseenter();

   /* get the size of the built-in font */
   fontgetinfo(&fontwidth,&fontheight);

   /* the stars will be drawn in 16 shades of gray */
   /* so we need to create the shades of gray      */
   setup_palette();
   
   /* setup the screen */

   drwfillbox(SET,GRAY,0,0,639,479);  /* fill entire screen with gray */
   drwstring(SET,5,GRAY,"3D Star Field  SVGACC Demo",boxes[7].x,boxes[7].y);
   drwstring(SET,5,GRAY,"Copyright 1995 by David S. Issel",boxes[8].x,boxes[8].y);
   drwstring(SET,DKGRAY,GRAY,"Stars",boxes[1].x+(boxes[1].w/2)-20,boxes[1].y-20);
   drwstring(SET,DKGRAY,GRAY,"Maximum Speed",boxes[2].x+(boxes[2].w/2)-48,boxes[2].y-20);
   drwstring(SET,DKGRAY,GRAY,"Status",boxes[5].x+(boxes[5].w/2)-20,boxes[5].y-20);
   draw_box(boxes[1].x-2,boxes[1].y-2,boxes[1].w+4,boxes[1].h+4,2,GRAY);
   draw_box(boxes[2].x-2,boxes[2].y-2,boxes[2].w+4,boxes[2].h+4,2,GRAY);
   draw_speedbar(boxes[1].x,boxes[1].y,boxes[1].w,boxes[1].h,GRAY,RED,MAXSTARFIELD,specified_MAXSTARFIELD);
   draw_speedbar(boxes[2].x,boxes[2].y,boxes[2].w,boxes[2].h,GRAY,RED,MAXSPEED,specified_MAXSPEED);
   draw_button(boxes[3].x,boxes[3].y,boxes[3].w,boxes[3].h,"SHOW",UP);
   draw_button(boxes[4].x,boxes[4].y,boxes[4].w,boxes[4].h,"EXIT",UP);
   draw_box(boxes[5].x-2,boxes[5].y-2,boxes[5].w+4,boxes[5].h+4,2,GRAY);
   drwbox(SET,BLACK,boxes[5].x,boxes[5].y,boxes[5].x+boxes[5].w,boxes[5].y+boxes[5].h);
   draw_box(boxes[6].x-4,boxes[6].y-4,boxes[6].w+9,boxes[6].h+9,4,BLACK);

   /* here is the main loop (do "something" until time to exit) */

   time_to_exit=FALSE; /* it is not time to exit the first time through */
   do {
      /* nothing needs to be done until the mouse moves */
      while ((mousex==oldx) && (mousey==oldy) && (mousebut==oldmousebut)
               && (!kbhit())) {
         mousestatus(&mousex,&mousey,&mousebut);
      }

      /* mouse has moved, or a button was pressed, so save the new status */
      oldx=mousex;
      oldy=mousey;
      oldmousebut=mousebut;

      /* find out what the mouse is sitting over and branch accordingly */
      switch (mouse_over(mousex,mousey,&boxes[1],maxboxes)) {
         case 1: /* stars bar */
            /* since we know that the mouse is floating over the stars bar */
            /* change the status message accordingly                       */
            sprintf(buf,"Set amount of stars to: %3d   ",desired_percent(boxes[1].x,boxes[1].w,MAXSTARFIELD,mousex));
            set_status_box_text(buf);
            /* if the left mouse button is pressed, change the number of stars */
            if (mousebut & 1) {
               new_stars=desired_percent(boxes[1].x,boxes[1].w,MAXSTARFIELD,mousex);
               /* only redraw the stars bar if it has changed */
               if (new_stars!=specified_MAXSTARFIELD) {
                  specified_MAXSTARFIELD=new_stars;
                  draw_speedbar(boxes[1].x,boxes[1].y,boxes[1].w,boxes[1].h,GRAY,RED,MAXSTARFIELD,specified_MAXSTARFIELD);
               }
            }
            break;
         case 2: /* speed bar */
            /* since we know that the mouse is floating over the speed bar */
            /* change the status message accordingly                       */
            sprintf(buf,"Set max speed of stars to: %2d ",desired_percent(boxes[2].x,boxes[2].w,MAXSPEED,mousex));
            set_status_box_text(buf);
            /* if the left mouse button is pressed, change the max speed */
            if (mousebut & 1) {
               new_speed=desired_percent(boxes[2].x,boxes[2].w,MAXSPEED,mousex);
               /* only redraw the speed bar if it has changed */
               if (new_speed!=specified_MAXSPEED) {
                  specified_MAXSPEED=new_speed;
                  draw_speedbar(boxes[2].x,boxes[2].y,boxes[2].w,boxes[2].h,GRAY,RED,MAXSPEED,specified_MAXSPEED);
               }
            }
            break;
         case 3: /* show button */
            /* since we know that the mouse is floating over the SHOW button */
            /* change the status message accordingly                         */
            set_status_box_text("Execute the 3D Starfield      ");
            /* if the left mouse button is pressed, draw the SHOW button down */
            if (mousebut & 1) {
               draw_button(boxes[3].x,boxes[3].y,boxes[3].w,boxes[3].h,"SHOW",DOWN);
               /* wait until the left mouse button is released */
               while (mousebut & 1) {
                  mousestatus(&mousex,&mousey,&mousebut);
               }
               /* draw the SHOW button up */
               draw_button(boxes[3].x,boxes[3].y,boxes[3].w,boxes[3].h,"SHOW",UP);
               /* run the 3D Star Field Demo */
               set_status_box_text("Running 3D Starfield Demo     ");
               do_3d_star_field();
               set_status_box_text("Execute the 3D Starfield      ");
            }
            break;
         case 4: /* exit button */
            /* since we know that the mouse is floating over the EXIT button */
            /* change the status message accordingly                         */
            set_status_box_text("Exit the Demo Program         ");
            /* if the left mouse button is pressed, draw the EXIT button down */
            if (mousebut & 1) {
               draw_button(boxes[4].x,boxes[4].y,boxes[4].w,boxes[4].h,"EXIT",DOWN);
               /* wait until the left mouse button is released */
               while (mousebut & 1) {
                  mousestatus(&mousex,&mousey,&mousebut);
               }
               /* draw the EXIT button up */
               draw_button(boxes[4].x,boxes[4].y,boxes[4].w,boxes[4].h,"EXIT",UP);
               /* signal to exit the program at the end of the main do loop */
               set_status_box_text("Good bye!                     ");
               time_to_exit=TRUE;
            }
            break;
         case 5: /* status box */
            /* since we know that the mouse is floating over the status box */
            /* change the status message accordingly                        */
            set_status_box_text("This is the status display    ");
            break;
         case 6: /* main window */
            /* since we know that the mouse is floating over the big window */
            /* change the status message accordingly                        */
            set_status_box_text("This is the forward view port ");
            break;
         case 7: /* program title */
            /* since we know that the mouse is floating over the program title */
            /* change the status message accordingly                           */
            set_status_box_text("This is the programs title    ");
            break;
         case 8: /* copyright message */
            /* since we know that the mouse is floating over the copyright msg */
            /* change the status message accordingly                           */
            set_status_box_text("This is the copyright message ");
            break;
         default: /* no where special */
            set_status_box_text("                              ");
            break;
      }
      /* if a key is pressed it is time to quit the program */
      if (kbhit()) {
         time_to_exit=TRUE;
      }
      /* loop back up to the top while not time to exit */
  } while ( !time_to_exit );

  /* get rid of the mouse routines from memory */
  mousehide();
  mouseexit();

  /* fade the screen to black */
  fade_to_black();

  /* return to the original video mode */
  videomodeset(vmode);

  /* exit the program signaling no error */
  exit(0);
}

void setup_palette(void)
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * We will be printing our stars in 16 shades of gray so we need *
 * to change the current palette so that we know what and where  *
 * those colors are.  We will place them just after the color    *
 * bright WHITE (16)                                             *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
{
   PaletteData tmp_pal;
   int i;

   sdelay(1);                 /* wait until a video retrace */
   palget(tmp_pal, 0, 255);   /* set tmp_pal to current palette */
   
   /* setup 16 shades of gray (just after bright WHITE) */
   for (i=1;i<17;i++) {
      tmp_pal[WHITE+i].r = ((tmp_pal[WHITE].r)-(i*4));
      tmp_pal[WHITE+i].g = ((tmp_pal[WHITE].g)-(i*4));
      tmp_pal[WHITE+i].b = ((tmp_pal[WHITE].b)-(i*4));
   }
   
   sdelay(1);              /* wait until a video retrace */
   palset(tmp_pal,0,255);  /* set the current palette to tmp_pal */
   return;
}

void fade_to_black(void)
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * This is a simple routine to fade the current screen to black  *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
{
   PaletteData tmp_pal;

   sdelay(1);                     /* wait until a video retrace */
   palget(tmp_pal, 0, 255);       /* set tmp_pal to current screen palette */
   palioauto(tmp_pal,0,255,-5);   /* fade the palette to BLACK */
   return;
}


void set_status_box_text(char *new_msg)
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * This routine changes the status box message only if the new   *
 * message is different from the one that is currently being     *
 * displayed.                                                    *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
{
   static char old_msg[100];

   /* don't repaint the same message if it is already up */
   /* this method reduces message (and mouse) flicker    */
   if (strcmp(old_msg,new_msg) != 0) {
      /* if this is a new message, save it and update the screen */
      strcpy(old_msg,new_msg);   /* save the new message */
      mousehide();    /* ALWAYS hide the mouse when drawing on screen */
      drwstring(SET,BLACK,GRAY,new_msg,boxes[5].x+4,boxes[5].y+4);
      mouseshow();    /* redisplay mouse quickly to reduce mouse flicker */
   }
   return;
}


void draw_box(int x, int y, int w, int h, int depth, int inside_color)
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * This routine draws a beveled box on the screen and fills it   *
 * with a specified color                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
{
   int i;

   mousehide(); 
   drwfillbox(SET,inside_color,x+depth,y+depth,x+w-depth,y+h-depth);
   for (i=0;i<depth;i++) {
      drwline(SET,DKGRAY,x+i,y+i,x+w-i,y+i);
      drwline(SET,WHITE,x+i,y+h-i,x+w-i,y+h-i);
      drwline(SET,DKGRAY,x+i,y+i,x+i,y+h-i);
      drwline(SET,WHITE,x+w-i,y+i,x+w-i,y+h-i);
   };
   mouseshow();
   return;
}


void draw_button(int x, int y, int w, int h, char *caption, int down)
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * This routine draws a button on the screen, either up or down, *
 * with a specified line of text on the button face              *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
{
   int textx,texty;
   int x2,y2;
    
   textx=x+((w-(strlen(caption)*fontwidth)+1) / 2);
   texty=y+((h-fontheight+1) / 2 ) +1;
   x2=x+w;
   y2=y+h;
   
   mousehide();
   drwfillbox(SET,GRAY,x+1,y+1,x2,y2-1);
   if (down) {
      drwstring(SET,BLACK,GRAY,caption,textx+1,texty+1);
      drwline(SET,GRAY,x+1,y+h,x2-1,y2);
      drwline(SET,DKGRAY,x+1,y,x2-1,y);
      drwline(SET,DKGRAY,x,y+1,x,y2-1);
   } else {
      drwstring(SET,BLACK,GRAY,caption,textx,texty);
      drwline(SET,WHITE,x,y+1,x,y2-1);
      drwline(SET,WHITE,x+1,y,x+1,y2-1);
      drwline(SET,WHITE,x+1,y,x2-1,y);
      drwline(SET,WHITE,x+1,y+1,x2,y+1);
      drwline(SET,DKGRAY,x+1,y2,x2-1,y2);
      drwline(SET,DKGRAY,x+1,y2-1,x2,y2-1);
      drwline(SET,DKGRAY,x2,y+1,x2,y2-1);
      drwline(SET,DKGRAY,x2-1,y+1,x2-1,y2-1);
   }
   drwline(SET,BLACK,x+1,y-1,x2-1,y-1);
   drwline(SET,BLACK,x+1,y2+1,x2-1,y2+1);
   drwline(SET,BLACK,x-1,y+1,x-1,y2-1);
   drwline(SET,BLACK,x2+1,y+1,x2+1,y2-1);
   drwpoint(SET,BLACK,x,y);
   drwpoint(SET,BLACK,x2,y);
   drwpoint(SET,BLACK,x,y2);
   drwpoint(SET,BLACK,x2,y2);

   mouseshow();
   return;
}


void draw_speedbar(int x, int y, int w, int h, int inside_color, int bar_color,int max_val, int cur_val)
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * This routine draws a percentage type box on the screen, you   *
 * give the maximum value and the current value                  *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
{
   int barwidth;
   char buf[10];

   /* calculate width of current value in pixels */
   barwidth= (int)( (long)(((cur_val>max_val)?(long)max_val:(long)cur_val)
                      * ((long)w - ((long)2))) / (long)max_val );
   /* prepare the display value (for the right hand side of the bar) */
   sprintf(buf,"%d   ",cur_val);
   
   mousehide();

   /* draw the slide bar background */
   drwfillbox(SET,inside_color,x+barwidth+1,y+1,x+w-1,y+h-1);
   
   /* draw the slide bar itself (to current value) */
   if (barwidth) {
      drwfillbox(SET,bar_color,x+1,y+1,x+barwidth+1,y+h-1);
   }
         
   /* print the current value next to slide bar */
   drwstring(SET,BLACK,GRAY,buf,x+w+5,y+(h/2)-6);

   /* draw a black frame around the slide bar */
   drwbox(SET,BLACK,x,y,x+w,y+h);

   mouseshow();
   return;
}


int desired_percent(int x, int w, int max_val, int mouseX)
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Given the mouse location, this routine returns how far inside *
 * the percentage slide bar you are                              *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
{
   if ( (mouseX>=(x + 1)) && (mouseX<=(x + w - 1)) ) {
      return( (int)( (long)(((long)mouseX-(long)1-(long)x)*(long)max_val)
                         / (long)((long)w-((long)2)) ) );
   }
   /* handle the special case that the mouse is outside the actual slide bar */
   if ( (mouseX>(x+1)) ) {
      return(max_val);
      }
   return(0);
}


int mouse_over(int mouseX, int mouseY, box *myboxes, int mymaxboxes)
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Given the mouse location, this routine returns which special  *
 * area of the screen the mouse is located in                    *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
{
   int maxboxes;

   maxboxes=mymaxboxes;
   do {
      if ( (mouseX>=myboxes->x) && (mouseX<=(myboxes->x + myboxes->w)) &&
            (mouseY>=myboxes->y) && (mouseY<=(myboxes->y + myboxes->h)) ) {
               return(maxboxes-mymaxboxes+1);
            }
   } while (++myboxes, --mymaxboxes);
   return(0);
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Here is the 3D star field routine.   I wrote them based on   *
 *  an assembly language version that I found on the 'net.  It   *
 *  is much better than mine.  To find it, do an archie search   *
 *  for stars.lzh                                        -Dave   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

void make_star(void)
{
   int i;

   if (cur_star_field_stars<specified_MAXSTARFIELD) {
      /* find an empty slot in the star field array */
      while( (star_field[i].old_z + star_field[i].z > 0) )  {
         i++;
      }
      if (i<MAXSTARFIELD) {
         star_field[i].x=(random(20)-10)*8; /* spread the stars out a little */
         star_field[i].y=(random(20)-10)*8;
         star_field[i].z=1280;              /* set the star to maximum depth */
         star_field[i].old_z=0;             /* old_z set to 0 means a new star */
         cur_star_field_stars++;            /* we just added a new star */
      }
   }
   return;
}

void display_stars(int warp_speed)
{
   int i;
   int xoffset,yoffset;
   int zcolor;
   int depth=256;

   /* set the clipping to the forward view port */
   setview(boxes[6].x,boxes[6].y,boxes[6].x+boxes[6].w,boxes[6].y+boxes[6].h);

   /* locate the center of the forward view port */
   xoffset=boxes[6].x+(boxes[6].w/2);
   yoffset=boxes[6].y+(boxes[6].h/2);
         
   /* display the stars in the middle of the view port */
   for (i=0;i<MAXSTARFIELD;i++) {
      if (star_field[i].old_z != 0) {
         /* erase the old stars location */
         drwpoint(SET,BLACK,((depth*star_field[i].x)/star_field[i].old_z)+xoffset,((depth*star_field[i].y)/star_field[i].old_z)+yoffset);
         /* if star went out of view, decrement the current number of stars showing */
         if (star_field[i].z==0) {
            cur_star_field_stars--;
         }
      }
      /* save the current star's depth so we can erase it later */
      star_field[i].old_z = star_field[i].z;
      if (star_field[i].z != 0) {
         /* set the color of the star based on how far away it is from us */      
         zcolor=WHITE+( star_field[i].z / ((1280/16)+1)  );
         /* draw the stars from center of screen out to sides to create */         
         /* the warping effect */
         drwpoint(SET,zcolor,((depth*star_field[i].x)/star_field[i].z)+xoffset,((depth*star_field[i].y)/star_field[i].z)+yoffset);
         /* if the star went off the screen, set depth to 0 to kill the star */
         if ( abs(((depth*star_field[i].x)/star_field[i].z))>(boxes[6].w/2) ) {
            star_field[i].z = 0;
         } else {
            if ( abs(((depth*star_field[i].y)/star_field[i].z))>(boxes[6].h/2) ) {
               star_field[i].z = 0;
            } else {
               /* star is still on the screen, change its location by the */
               /* current warp speed (i.e. as the star gets closer to us  */
               /* z tends to 0                                            */
               star_field[i].z = (warp_speed<star_field[i].z)?star_field[i].z-warp_speed:0;
            }
         }
      }
      /* loop back and do the next star */
   }
   
   /* set the view back to full screen */
   setview(0,0,maxx,maxy);
   return;
}


void do_3d_star_field(void)
{
   int i,j;

   /* shut off the mouse during the 3D star field effect because if I tried */
   /* to leave it on it would flicker too much                              */
   mousehide();

   /* clear the forward view screen to black */
   if (specified_MAXSPEED) {
      drwfillbox(SET,BLACK,boxes[6].x,boxes[6].y,boxes[6].x+boxes[6].w+1,boxes[6].y+boxes[6].h+1);
   }
   
   /* start warp of stars from slow to fast */ 
   for (i=0;i<specified_MAXSPEED;i++) {
      make_star();
      make_star();
      make_star();
      sdelay(1);
      display_stars(i);
      make_star();
      make_star();
      make_star();
      sdelay(1);
      display_stars(i);
   }

   /* warp at max speed for a short while (if max is > 0) */
   if (specified_MAXSPEED) {
      for (j=0;j<(((90/(specified_MAXSPEED))+100)*5);j++) {
         make_star();
         make_star();
         make_star();
         sdelay(1);
         display_stars(i); /* i is still set from the previous loop */
      }
   }

   /* slow down warp of stars until stop */
   for (i=specified_MAXSPEED;i>0;i--) {
      make_star();
      make_star();
      make_star();
      sdelay(1);
      display_stars(i);
      make_star();
      make_star();
      make_star();
      sdelay(1);
      display_stars(i);
   }
   
   /* show is over now, turn the mouse back on and return */
   mouseshow();
   return;
}
