;; SMOOTH.ASM
;; Smooth scrolling for the EGA/VGA

clr_flg	equ	1
ecd_flg	equ	2
g64_flg	equ	4
vga_flg	equ	8
c_wide	equ	104	;width of the virtual screen
lines	equ	(c_wide-80)*2+25
ATC_add	equ	3c0h	;Attribute Controller address
lc_a	equ	'a'

word_out	macro
;;
;;Most adapter/computer combinations accept a word OUT
;;instruction, however a few combinations balk at this. A good way
;;to handle this is with a macro for all OUT instructions. If someone
;;has a problem with your program, you can compile it with byte out
;;routines simply by changing the macro to the commented section.
;;
	out	DX,AX
;;
;;	out	DX,AL
;;	xchg	AL,AH
;;	inc	DX
;;	out	DX,AL
;;	xchg	AL,AH
;;	dec	DX
;;
	endm

byte_out	macro
;;
	out	DX,AL
	xchg	AL,AH
	out	DX,AL
	xchg	AL,AH
;;
	endm

disp	macro
	local	rt
;;This macro drops through only during a display interval
;;DX must be already set to 3C0h (used in DISPLAY and WAIT_D
;;macros)
;;
rt:	in	al,dx	;;test vertical retrace status bit
	test	al,1000b	;;is it a vertical retrace?
	jnz	rt	;;yes, keep looking for non-retrace
			;;(display) interval

;;
	endm


display	macro
;;
	mov	DX,st_add	;;Input Status Register One
	disp
;;
	endm
	

retr	macro
	local n_rt
	
;;macro which drops through only during a vertical retrace
;;DX must alread be set to 3C0h (used in RETRACE and WAIT_r
;;macros)
;;
n_rt:	in	AL,DX	;;check for the beginning of the
			;;vertical retrace.
	test	AL,1000b	;;Is it a vertical retrace?
	jz	n_rt	;;no, try again.
;;
	endm
	
retrace	macro
;;
	mov	Dx,st_add	;;Input Status Register One
	retr
;;
	endm
	
wait_r	macro
;;
;;This macro will drop through only at the beginning of a vertical
;;retrace. It first waits for a display interval, and only then looks for a
;;retrace
;;
	display
	retr
;;
	endm
	
wait_d	macro
;;
	retrace
	disp
;;
	endm
	
fill_alpha	macro
	local keep,lp
;;
;;This macro fills the data area with the letters of the alphabet
;;and a cycle attribute bytes
;;
lp:	mov	[BX],AX
	add	AX,101h	;;next letter
	add	BX,2	;;next memory address
	cmp	AH,14	;;is the attribute now 14?
	jne	keep	;;no, keep it
	mov	AH,18	;;skip cyan on cyan
keep:	loop	lp
;;
	endm
	
data	segment	public

	ltrs	db	c_wide*2+25*2 dup (?)
;
;The area above will be filled in by the program to save typing
;and will look like:
;
;	ltrs	db	'a.b.c.d.e.f.g.h.i.j.k.l.m.n.o.p.q.r.s.t.u.v.w.
;		db	'x.y.z.'
;		db	'a.b.c.d.e.f.g.h.i.j.k.l.m.n.o.p.q.r.s.t.u.v.w.
;		db	'x.y.z.'
;where "." represents an attribute byte. it is long enough to fill a full
;104 character line starting with any character of the alphabet (offset 0-25).
;
	hdwre	db	?
even
	strt_add	label	byte
	strt_addw	dw	2 dup (0)	;adapter(internal)
						;memory address
	CRT_add	dw	3B4h	;CRTC port
	st_add	dw	?	;input status register
				;one address
	al_seg	dw	0b000h	;alpha mode segment
	ch_hi	dw	13	;character height minus 1
	ch_wi	dw	8	;character width minus 1
	al_mode	db	7	;alpha mode
	
data	ends

code	segment public
	assume	CS:code
	
main	proc	far

start:	push	DS
	sub	AX,AX
	push	AX
	
	mov	AX,data
	mov	DS,AX
	assume	DS:data
	
	mov	BX,offset ltrs	;address of letters
	
;fill the character/attribute data area

	mov	CX,c_wide/26
lp:	push	CX
	mov	CX,26	;all 26 letters of the alphabet
	mov	AL,lc_a	;start with a lowercase 'a'
	mov	AH,1	;start with attribute 1
			;(don't use black on black)
	fill_alpha
	pop	CX
	loop	lp
	
	mov	AL,lc_a	;start with 'a' again
	mov	AH,1	;and attribute 1
	mov	CX,25	;but only go to 'y'
	
	fill_alpha
	
	mov	AX,1a00h	;read display combination
	int	10h	;BIOS video call
	
	cmp	al,1ah	;The VGA(or model 30) will return 1ah
	jne	not_vga
	
	cmp	BL,7	;is it a monochrome VGA?
	je	yes_vga
	cmp	BL,8	;is it a color VGA?
	je	yes_vga
	jmp	not_vga	;could check for EGA for completeness
	
yes_vga:	or	hdwre,vga_flg

not_vga:	mov	BL,10h	;get EGA information
	mov	AH,12h	;alternate functions
	int	10h	;BIOS call
	
	cmp	BH,1	;Is it monochrome
	je	mono	;yes, defaults already set for
			;monochrome, skip setup
			
	cmp	CL,1001b	;Check switch settings.
			;Is it an enhanced display?
	je	ecd
	cmp	CL,0011b	;in hi-res alpha mode
	je	ecd
;must be a normal color display(or ECD in emulation mode)
	mov	ch_hi,7
	jmp	cd
	
ecd:	or	hdwre,ecd_flg	;set the ECD flag

;the following section is common between standard color and ECD

cd:	or	hdwre,clr_flg	;set the color flag
	mov	ch_wi,7		;character width is 8 for non-VGA, or VGA
				;in 8-dot mode.
	mov	al_seg,0b800h	;alpha mode address for all color modes
	mov	al_mode,3	;mode 3 for alpha mode
	mov	CRT_add,3d4h	;CRT address is 3d4h for all color modes
	test	hdwre,vga_flg	;VGA may be in 9 dot mode
	jz	only_8		;if not VGA, don't check
	
	mov	DX,3c4h	;sequencer address register
	mov	AL,1	;clocking mode address
	out	DX,AL
	inc	DX	;point to clocking mode register
	in	AL,DX	;if bit 1=1, the dot clock is 8
	test	AL,1
	jnz	only_8
	mov	ch_wi,8	;character width is 9 for VGA in 9 dot mode
only_8:

mono:	test	hdwre,vga_flg	;VGA may be 16 lines high
	jz	def_hi	;if not VGA, use default height
	
	mov	ch_hi,15	;store height-1 (16 pixels on VGA)
	
def_hi:	mov	AX,CRT_add
	add	AX,6	;Status Address Register One is CRTC+6
	mov	st_add,AX
	
;START ALPHA MODE SCROLLING DEMONSTRATION

	xor	AH,AH	;function call 0 -- set mode
	mov	AL,al_mode	;make sure alpha mode is set
	int	10h	;BIOS video call
	
	mov	AX,al_seg	;set ES:DI to the beginning of
				;alpha memory
	mov	ES,AX
	assume	ES:nothing
	mov	DI,0
	
	cli		;disable interrupts
	
	wait_r		;wait for the vertical retrace
	mov	DX,CRT_add
	mov	AL,13h	;offset register index
	mov	AH,c_wide/2	;use the same virtual width
				;for all alpha modes
	word_out
	
;we will leave the start address at 0(upper left corner)
;although you may want to use a different initial start address
;in your applications
;WRITE THE TEXT TO THE SCREEN THROUGH THE MOVE STRING INSTRUCTION

	mov	BX,offset ltrs	;get the starting letter's address
	xchg	AH,AL	;put word width in AL
	xor	AH,AH	;0 out AH
	shl	AX,1	;multiply by 2 to include attribute bytes
	
	mov	CX,lines
	
n_line:	push	CX	;save outer loop value
	mov	SI,BX	;put starting letter in SI
	mov	CX,AX	;count for entire line width
	rep	movsw	;move a full virtual display line into display mem.
	add	BX,2	;start with the next letter of the alphabet
	cmp	BX,offset ltrs+26*2	; is it past z?
	jne	n_ltr	;no, start next line with next letter
	sub	BX,52	;yes, start with 'a' again
n_ltr:	pop	CX	;restore outer loop value
	loop	n_line	;do next line
	
	sti		;re-enable interrupts
	
;START SMOOTH SCROLLING THROUGH THE DISPLAY (DOWN 1 CHAR & RIGHT 1 CHAR)

	mov	CX,-80	;CX contains width of screen as a negative number
	add	CX,c_wide	;add virtual screen width to
				;get number of characters
				;between the right edge of
				;actual and virtual screen
	push	CX	;save this value for use in second section
	
again:	call	down_c

	call	right_c
	loop	again
	
;START MOVING DOWN ONE CHARACTER AND LEFT ONE CHARACTER

	pop	CX
again2:	call	down_c
	call	left_c
	loop	again2
	
	xor	AH,AH	;function call 0 -- set mode
	mov	AL,al_mode	;end program with alpha mode reset
	int	10h	;BIOS video call
	
	ret
	
main	endp

right_c	proc	near

;Move the display area one character right in text mode

	push	AX
	push	BX
	push	CX
	push	DX
	
	mov	CX,ch_wi
	mov	BX,0033h	;index of horizontal pel panning
			;(start at 0 for 9 bit)
			;bit 5 is set (palette remains unmodified)
	cmp	ch_wi,8	;was it a 9 bit character width (ch_wi=width-1)
	je	is_9	;yes, skip to next step
	inc	BH	;no, start at 1 for 8 bit
	cli		;do not allow interrupts
			;(to prevent register changes)
is_9:	wait_r		;wait for vertical retrace(modifies AX/DX)
			;and resets 3c0h
	mov	DX,ATC_add	;attribute register
	mov	AX,BX
	byte_out		;most cards will accept OUT DX,AX to 3c0h,
			;although it is undocumented
	inc	BH	;next step of horizontal scroll
	loop	is_9	;loop for 1 full character width
	
	cmp	ch_wi,8	;was it a 9 bit character width?
	je	is_9_2	;yes, skip to next step
	mov	bh,0	;no, start at 0 for 8-bit
is_9_2:	display	;Start address must be set during display interval
		;to take effect at the beginning next vertical retrace
		
	inc	strt_addw	;point to the next character
				;for display start address
	mov	DX,CRT_add
	mov	AL,0ch		;index of start address high
	mov	AH,strt_add[1]	;get the most significant byte
	word_out
	inc	AX		;index of start address low
	mov	AH,strt_add	;get the least significant byte
	word_out
	

	retrace	;the above code is during the display interval
		;no need to check again with the WAIT_R macro
	mov	DX,ATC_add
	mov	AX,BX
	byte_out
	
	sti
	
	pop	DX
	pop	CX
	pop	BX
	pop	AX
	
	ret
	
right_c	endp

left_c	proc near
	
;Move the display area one character right in text mode

	push	AX
	push	BX
	push	CX
	push	DX
	
	cli		;do not allow interrupts(to prevent register changes)
	display		;start address must be set during display interval to
			;take effect at the beginning next vertical retrace
	dec	strt_addw	;point to next character for display
				;start address
	mov	DX,CRT_add
	mov	AL,0ch		;index of start address high
	mov	AH,strt_add[1]	;get the most significant byte
	word_out
	inc	AX		;index of start address low
	mov	AH,strt_add	;get the least significant byte

	word_out

	mov	CX,ch_wi
	inc	CX
	mov	BX,0733h	;index of horizontal pel panning (start at 7th pixel)
			;bit 5 is set (palette registers remain unmodified)
			
lp_lft:	wait_r	;waits for vertical retrace(modifies AX/DX)
		;and resets 3c0h
	mov	DX,ATC_add	;attribute register
	mov	AX,BX
	byte_out		;most cards will accept OUT DX,AX to 3c0h,
			;although it is undocumented
	dec	BH	;next step of horizontal scroll
	loop	lp_lft	;loop for 1 full character width
	
	cmp	ch_wi,8	;was it a 9 bit character width?
	jne	not_9	;yes, skip to next step
	mov	bh,0	;no, start at 0 for 8-bit
	wait_r
	mov	DX,ATC_add
	mov	AX,0833h	;set the Pel Pan register to 8 for 9 bit width
	byte_out
	
not_9:	sti

	pop	DX
	pop	CX
	pop	BX
	pop	AX
	
	ret
left_c	endp

down_c	proc	near

;Move the display area one character to the right in text mode

	push	AX
	push	BX
	push	CX
	push	DX
	
	mov	CX,ch_hi
	mov	BX,0108h	;index of preset row scan (start at 1)
	cli		;do not allow interrupts(prevent register changes)
	
lp_d:	wait_r	;wait for retrace interval (modifies AX/DX)
	mov	DX,CRT_add	;CRTC register address
	mov	AX,BX
	word_out
	inc	BH	;next step of vertical scroll
	loop	lp_d	;loop 1 full character height
	
	mov	BX,c_wide	;width of virtual screen
	add	BX,strt_addw	;BX points to next line
	mov	strt_addw,BX	;save new start address
	
	wait_d		;wait for display interval to change start address
	
	mov	DX,CRT_add
	mov	AL,0dh	;index of start address high
	mov	AH,BL	;get the least significant byte
	word_out
	dec	AX	;index of start address low
	mov	AH,BH	;get the least significant byte
	word_out
	

	retrace	;wait for retrace
	mov	DX,CRT_add
	mov	AX,8
	word_out
	wait_d
	
	sti
	
	pop	DX
	pop	CX
	pop	BX
	pop	AX
	
	ret
	
down_c	endp
	
down_p	proc near

;Move the display area one character to the right in text mode

	push	AX
	push	BX
	push	CX
	push	DX
	
	mov	BX,c_wide	;width of virtual screen
	add	BX,strt_addw	;BX points to next line
	mov	strt_addw,BX	;save new start address
	
	wait_d		;wait for display interval to change start address
	
	mov	DX,CRT_add
	mov	AL,0dh	;index of start address high
	mov	AH,BL	;get the least significant byte
	word_out
	dec	AX	;index of start address low
	mov	AH,BH	;get the least significant byte
	word_out
	
	sti
	
	pop	DX
	pop	CX
	pop	BX
	pop	AX
	
	ret
	
down_p	endp

code	ends

end	start
