{$R-,Q-,S-}
Unit XVGA256;

Interface

Const
	TextureShift	= 14;
	TextureShifted	= 1 Shl TextureShift;
	TextureScaler	= 5000;

Type
	TRGB		= Array[1..768] Of Byte;
	TPoint		= Record
		X,Y	: Word;
	End;
	TRes		= 4..8;
	TTranslate	= Array[Byte] Of Byte;
	TCompress	= (cNone,cRLE,cLZH);

Const
	LastLoaded	: TCompress	= cNone;	{ Holds the compression of the last
										  loaded image }

Procedure XSetPoint(Var Point : TPoint; X,Y : Word);
	{ Just fills out the record Point with the X and Y values	}
Procedure XTranslateImage(Var Image; Const TransTable : TTranslate);
	{ Translates the image with the TransTable translation table	}
Procedure XMode(No : Byte);
	{ Sets the video-mode:
		$13	--> X-Mode 320x200x256 (Blacks out palette)
		$03	--> 80x25x16 graphics }
Procedure XDefaultPalette(Var RGB);
	{ Puts the default VGA palette in RGB	}
Procedure XSetDefaultPalette;
	{ Sets the default VGA palette	}
Procedure XGetSortedColors(Var Translate : TTranslate; DarkestFirst : Boolean);
	{ Sorts the color-table according to DarkestFirst, filling out the
	  Translate variable. This is NOT a translation-table yet, but only a
	  list of 256 color-values in sorted order.	}
Procedure XTranslate(Var STable,DTable : TTranslate; Count : Byte);
	{ Translates the table STable Count times, putting the result in DTable	}
Function XTrans(Pixel : Byte) : Byte;
	{ Translates the pixel-value Pixel according to the current translation-
	  table.	}
Function XLookUp(R,G,B : Byte) : Byte;
	{ Finds the closest match to R,G,B and returns it pixel-value	}
Procedure XClr(Color : Byte);
	{ Clears the screen	}
Procedure XPlot(X,Y : Word; Color : Byte);
	{ Plots a pixel	}
Function XGetPixel(X,Y : Word) : Byte;
	{ Returns the pixel-value at the point	}
Procedure XLine(X1,Y1,X2,Y2 : Word; Color : Byte);
	{ Draws a line	}
Procedure XBox(X1,Y1,X2,Y2 : Word; Color : Byte);
	{ Draws a hollow box	}
Procedure XRectangle(X1,Y1,X2,Y2 : Word; Color : Byte);
	{ Draws a filled rectangle	}
Procedure XSetRGB(Index,R,G,B : Byte);
	{ Sets the RGB values for a given color-index. R,G,B cannot be larger
	  than 63. 0,0,0 = Black and 63,63,63 = White	}
Procedure XSetRGBMany(StartIndex : Byte; ColorCount : Word; Var RGBData);
	{ Sets more than one color (ColorCount), from RGBData, starting at the
	  index StartIndex. Format of RGBData is three bytes per color, in the
	  order R, G and B.	}
Procedure XSetRGBBlack(StartIndex : Byte; ColorCount : Word);
	{ Sets ColorCount colors from StartIndex onwards to black	}
Procedure XSetRGBFade(StartIndex : Byte; ColorCount : Word; Var RGBData; Frames : Byte);
	{ Fades the colors from the current ones to RGBData, starting at
	  StartIndex and ColorCount colors onwards, in Frames frames.	}
Procedure XCalcRGBFadeFrame(StartIndex : Byte; ColorCount : Word;
	Var RGB1,RGB2,ResultRGB; CurFrame,Frames : Byte);
	{ Calculates one frame in a fading. Fade goes from RGB1 to RGB2, and the
	  result is put in ResultRGB. CurFrame is 0..Frames-1, and Frames is the
	  total # of frames.	}
Procedure XGetRGB(Index : Byte; Var R,G,B : Byte);
	{ Returns the RGB values for the given color	}
Procedure XGetRGBMany(StartIndex : Byte; ColorCount : Word; Var RGBData);
	{ Returns the RGB values for several colors	}
Procedure XHLine(Y,X1,X2 : Word; Color : Byte);
	{ Draws a horizontal line, using optimal assembly	}
Procedure XVLine(X,Y1,Y2 : Word; Color : Byte);
	{ Draws a vertical line, using optimal assemble	}
Procedure XScrollRight(X1,Y1,X2,Y2 : Word; FillColor : Byte);
	{ Scrolls the area right one pixel	}
Procedure XScrollLeft(X1,Y1,X2,Y2 : Word; FillColor : Byte);
	{ Scrolls the area left one pixel	}
Procedure XScrollUp(X1,Y1,X2,Y2 : Word; FillColor : Byte);
	{ Scrolls the area up one pixel	}
Procedure XScrollDown(X1,Y1,X2,Y2 : Word; FillColor : Byte);
	{ Scrolls the area down one pixel	}
Procedure XAllocImage(X1,Y1,X2,Y2 : Word; Var Data : Pointer);
	{ Allocates memory for a image of the given size/area	}
Procedure XFreeImage(Var Data : Pointer);
	{ Frees up the image taken by Data. Really the same as
		FreeMem(Data,XImageMemSize(Data^));	}
Function XImageSize(X1,Y1,X2,Y2 : Word) : Word;
	{ Returns the number of bytes taken by the image. Similar to
		(X2-X1+1)*(Y2-Y1+1)	}
Function XImageWidth(Var Data) : Word;
	{ Returns the width of the image	}
Function XImageHeight(Var Data) : Word;
	{ Returns the height of the image	}
Function XImageMemSize(Var Data) : Word;
	{ Returns the number of bytes taken by the image in memory. Similar to
		XImageSize(Data)+4	}
Procedure XGetImage(X1,Y1,X2,Y2 : Word; Var Data);
	{ Grabs a portion of the screen, and stores it in Data	}
Procedure XAllocGetImage(X1,Y1,X2,Y2 : Word; Var Data : Pointer);
	{ Allocates memory for the image, and grabs the image from the
	  screen.	}
Procedure XDraw3(X1,Y1,X2,Y2,X3,Y3 : Word; Color : Byte);
	{ Draws the outline of a tetrahedron	}
Procedure XFill3(X1,Y1,X2,Y2,X3,Y3 : Word; Color : Byte);
	{ Draws a filled tetrahedron	}
Procedure XDraw4(X1,Y1,X2,Y2,X3,Y3,X4,Y4 : Word; Color : Byte);
	{ Draws the outline of a quadrahedron	}
Procedure XFill4(X1,Y1,X2,Y2,X3,Y3,X4,Y4 : Word; Color : Byte);
	{ Draws a filled quadrahedron	}
Type
	TPutFn	= (pNORMAL,pXOR,pADD,pSUB,pTRANSPARENT,pOR,pAND,pAPPROACH,pONLY,
		pTRANSLATE,pINC,pDEC);
	{	pNORMAL			: Old:=New
		pXOR			: Old:=Old XOr New
		pADD			: Old:=Old+Add
		pSUB			: Old:=Old-New
		pTRANSPARENT	: Old:=New If Old<>MaskColor
		pOR				: Old:=Old Or New
		pAND			: Old:=Old And New
		pAPPROACH		: The pixel on the screen is incremented or
						  decremented towards the new color
		pONLY			: Draws the image/line/whatever at the screen, only
						  if the pixel on the screen is the MaskColor
		pTRANSLATE		: Translates the pixel-value on the screen
		pINC			: Increments the pixel-value on the screen
		pDEC			: Decrements the pixel-value on the screen	}
Procedure XPutImage(X,Y : Word; Var Data);
	{ Stores the image on the screen	}
Procedure XPutScaledImage(X,Y : Word; Var Image; ScaledWidth,ScaledHeight : Word);
	{ Scales the image while storing it	}
Procedure XAllocBuffer(Var B : Pointer);
	{ Allocates a screen-buffer	}
Function XImageBuffer(Var B : Pointer) : Pointer;
	{ Returns a pointer to a image lying on top of the buffer	}
Procedure XFreeBuffer(Var B : Pointer);
	{ Frees up the memory allocated to the buffer	}
Procedure XUseBuffer(B : Pointer);
	{ Uses the buffer from here on. All drawing-functions will change the
	  buffer and not the screen. Use NIL to use the screen instead of a
	  buffer. }
Procedure XShowBuffer(B : Pointer);
	{ Copies the buffer to the screen	}
Procedure XCopyBuffer(B1,B2 : Pointer; X1,Y1,X2,Y2 : Word);
	{ Copies from the buffer B1 to B2 in the given area. NIL may be used for
	  the screen	}
Procedure XSwapBuffer(B1,B2 : Pointer; X1,Y1,X2,Y2 : Word);
	{ Swaps the buffer B1 with B2 in the given area. NIL may be used for
	  the screen	}
Procedure XDupBuffer(B1,B2 : Pointer);
	{ Copies the buffer B1 to B2. NIL may be used for the screen	}
Function XOkBuffer(B : Pointer) : Boolean;
	{ Returns TRUE if B points to a valid buffer	}
Function XCurrentBuffer : Pointer;
	{ Returns a pointer to the current buffer	}
Function XCurrentSegment : Word;
	{ Returns the segment for the current screen, be it the real screen or
	  a buffer. The Pixels go from offset 0 to 63999, 320 per line.	}
Procedure XSetFn(NewFn : TPutFn; NewMaskColor : Byte);
	{ Sets the current drawing-function and the mask color where
	  appropriate	}
Procedure XSetTranslateTable(Const Translate : TTranslate);
	{ Sets the translation-table. Will not be used if not current function
	  is pTranslate	}
Function XLoadPCX(FileName : String; Var RGB; Var Width,Height : Word) : Integer;
	{ Loads a pcx image to the screen	}
Function XSaveImage(Var Data; FileName : String; Var Offset : LongInt; Compress : TCompress) : Integer;
	{ Saves the image. Offset will be updated to point after the image	}
Function XLoadImage(Var Data : Pointer; FileName : String; Var Offset : LongInt) : Integer;
	{ Loads the image.	}
Function XSavePalette(Var RGB; FileName : String; Var Offset : LongInt) : Integer;
	{ Saves the palette	}
Function XLoadPalette(Var RGB; FileName : String; Var Offset : LongInt) : Integer;
	{ Loads the palette	}
Function XSaveBinary(Var Binary; Size : Word; FileName : String; Var Offset : LongInt) : Integer;
	{ Saves binary data.	}
Function XLoadBinary(Var Binary : Pointer; Var Size : Word; FileName : String; Var Offset : LongInt) : Integer;
	{ Loads binary data.	}
Type
	TAt	= (tImage,tPalette,tBinary,tExeFile,tUnknown,tEndOfFile,tFont);

Function XAt(FileName : String; Offset : LongInt) : TAt;
	{ Returns what is on the given position in the file.	}
Function XNext(FileName : String; Offset : LongInt) : LongInt;
	{ Returns the offset of the next portion.	}
Procedure XSkip(FileName : String; Var Offset : LongInt);
	{ Skips the current portion in the file.	}
Procedure XScaleImage(Var Image1,Image2; ScaledWidth,ScaledHeight : Word);
	{ Scales the image Image1 to Image2. Image2 must be allocated
	  beforehand	}
Procedure XHFlipImage(Var Image);
	{ Flips the image horizontally	}
Procedure XVFlipImage(Var Image);
	{ Flips the image vertically	}
Procedure XVHFlipImage(Var Image);
	{ Flips the image diagonally	}
Procedure VRet;
	{ Waits for a vertical retrace	}
Procedure HRet;
	{ Waits for a horizontal retrace	}
Procedure ScanTime(Index,RealR,RealG,RealB,TimeR,TimeG,TimeB : Byte);
	{ Waits for a vertical retrace, coloring the screen from the current
	  beam-point to the bottom. Useful for timing	}
Procedure CfgRasterCheck(Index,RealR,RealG,RealB,TimeR,TimeG,TimeB : Byte);
	{ Configures the timing check	}
Procedure SetRasterCheck(On : Boolean);
	{ Turns on/off the timing check on the VRet	}
Function RasterCheck : Boolean;
	{ Returns the current setting	}
Procedure XMapTexture(Var Texture; Num : Word; sPoints,dPoints : Array Of TPoint);
	{ Maps/distorts the image Texture in a polygon. sPoints is a polygon in
	  the image, and dPoints is the translated polygon on the screen. Num is
	  number of corners. The polygon must be convex	}
Procedure XDrawPoly(Num : Word; Points : Array Of TPoint; Color : Byte);
	{ Draws the outline of a polygon.	}
Procedure XFillPoly(Num : Word; Points : Array Of TPoint; Color : Byte);
	{ Draws a filled convex polygon	}
Procedure DelFile(FileName : String);
	{ Deletes a file	}
Procedure XCircle(X,Y,R : Word; Color : Byte);
	{ Bresenhams circle algorithm	}
Procedure XFillCircle(X,Y,R : Word; Color : Byte);
	{ Bresenhams patched circle algorithm	}
Procedure XEllipse(X,Y,RW,RH : Word; Color : Byte);
	{ Bresenhams ellipse algorithm	}
Procedure XFillEllipse(X,Y,RW,RH : Word; Color : Byte);
	{ Bresenhams patched ellipse algorithm	}
Const
	jCenter		= 0;
	jRightOf	= 1;	{ Default	}
	jLeftOf		= 2;
	jBelow		= 1;	{ Default	}
	jAbove		= 2;

Type
	PFont	= ^OFont;
	OFont	= Object
		Constructor Init(FontName : String; Height : Word);
			{ Create a new font	}
		Constructor Read(FileName : String; Var Offset : LongInt);
			{ Reads the font from the file	}
		Constructor Load(FileName : String);
			{ Reads from the start	}
		Destructor Done; Virtual;
			{ De-allocates the font from memory	}
		Function GetName : String; Virtual;
			{ Returns the name of the font. Not the filename	}
		Function Write(FileName : String; Var Offset : LongInt) : Integer; Virtual;
			{ Writes the font to the file	}
		Function Save(FileName : String) : Integer; Virtual;
			{ Writes from the start	}
		Function GetLetter(Letter : Char; Var Image : Pointer) : Boolean; Virtual;
			{ Allocates and returns a copy of the image for the letter.	}
		Function Assigned(Letter : Char) : Boolean; Virtual;
			{ Returns TRUE if there exist an image for the letter	}
		Function Assign(Letter : Char; Var Image) : Boolean; Virtual;
			{ Assigns a new image to the letter	}
		Function Grab(Letter : Char; Y,X1,X2 : Word) : Boolean; Virtual;
			{ Grabs a new image for a letter	}
		Procedure UnAssign(Letter : Char); Virtual;
			{ Unassigns a letter	}
		Function GetHeight : Word; Virtual;
			{ Returns the height of the font	}
		Procedure SetScale(MultW,DivW,MultH,DivH : Word); Virtual;
			{ Sets the scale of the font:
				Width=oldWidth*MultW/DivW
				Height=oldHeight*MultH/DivH	}
		Function TextHeight(S : String) : Word; Virtual;
			{ Returns the height in pixels	}
		Function TextWidth(S : String) : Word; Virtual;
			{ Returns the width in pixels	}
		Procedure SetJustify(HorzJustify,VertJustify : Word); Virtual;
			{ Sets the current justification	}
		Procedure Display(X,Y : Word; Text : String); Virtual;
			{ Displays the text	}
		Procedure Mono(Use : Boolean; TranspColor,NewColor : Byte); Virtual;
			{ Sets mono-color	}
	Private
		UseMono	: Boolean;
		TColor	: Byte;
		Color	: Byte;
		H		: Word;
		J1,J2	: Word;
		W1,W2	: Word;
		H1,H2	: Word;
		Name	: PChar;
		Letters	: Array[Char] Of Pointer;
	End;

	PPalette = ^OPalette;
	OPalette = Object
		Constructor Init(GrabCurrent : Boolean);
		Destructor Done; Virtual;

		Procedure SetRGB(Index,R,G,B : Byte); Virtual;

		Procedure Realize; Virtual;
		Procedure Get(Var RGB); Virtual;

		Function Red(Index : Byte) : Byte; Virtual;
		Function Green(Index : Byte) : Byte; Virtual;
		Function Blue(Index : Byte) : Byte; Virtual;

		Procedure Grab; Virtual;
	Private
		Data : Array[Byte,1..3] Of Byte;
	End;

Implementation

Uses
	Dos,Strings,LZH;

Const
	CurBuf				: Pointer	= NIL;
	ScrSeg				: Word		= $A000;
	PutFn				: TPutFn	= pNORMAL;
	MaskColor			: Byte		= 0;

Var
	TransTabl			: TTranslate;
Const
	TransPtr			: Pointer	= @TransTabl;

Procedure Swap(Var A,B; Size : Word); Assembler;
Asm
	PUSH	DI					{ Preserve registers used in procedure		}
	PUSH	SI					{ ...										}
	PUSH	ES					{ ...										}
	PUSH	BX					{ ...										}
	PUSH	AX					{ ...										}
	PUSH	DS					{ ...										}
	CLD							{ LODS/STOS increases the SI/DI registers	}
	LDS		SI,A				{ Set DS:DI to point to A					}
	LES		DI,B				{ Set ES:DI to point to B					}
	MOV		CX,Size				{ Put Size into CX							}
	SHR		CX,1				{ Primarily move whole words at a time		}
	JNC		@SWAPWORDS			{ Size Mod 2<>0 --> @SWAPWORDS				}
	MOV		AL,ES:[DI]			{ Fetch first byte from B					}
	MOVSB						{ Copy first byte from A to B				}
	MOV		[SI-1],AL			{ Put first byte from B into A				}
	JZ		@EXIT				{ Copy more bytes ?							}
@SWAPWORDS:
	MOV		BX,-2				{ Put memory offset into BX					}
	OR		CX,CX				{ Copy any bytes at all ?					}
	JZ		@EXIT				{ No --> @EXIT								}
@SWAPLOOP:
	MOV		AX,ES:[DI]			{ Get word from B							}
	MOVSW						{ Copy word from A to B						}
	MOV		[BX][SI],AX			{ Put word from B into A					}
	LOOP	@SWAPLOOP			{ Decrement CX and do again					}
@EXIT:
	POP		DS					{ Restore registers ...						}
	POP		AX					{ ...										}
	POP		BX					{ ...										}
	POP		ES					{ ...										}
	POP		SI					{ ...										}
	POP		DI					{ ...										}
End;
Procedure XPlot(X,Y : Word; Color : Byte); Assembler;
Asm
	CMP		X,$013F						{ Is X <= 319 (X-range) ?			}
	JA		@EXIT						{ No --> @EXIT						}
	CMP		Y,$00C7						{ Is Y <= 199 (Y-range) ?			}
	JA		@EXIT						{ No --> @EXIT						}
	XOR		DI,DI						{ Set Offset part to $0000			}
	MOV		AX,Y						{ Put Y in AX						}
	MOV		BX,$0140					{ Put $0140 (320) in BX				}
	MUL		BX							{ Multiply AX with BX into DX:AX	}
	ADD		DI,AX						{ Add AX to DI to move down			}
	ADD		DI,X						{ Add X to DI to move right			}
	MOV		ES,ScrSeg					{ Point ES:0 to screen				}
	MOV		AL,Color					{ Put Color in AL					}
	CMP		PutFn,pNORMAL
	JNE		@01
	MOV		AH,AL
	JMP		@EXIT
@01:
	CMP		PutFn,pTRANSPARENT
	JNE		@0
	MOV		AH,AL
	CMP		AL,MaskColor
	JNE		@EXIT
	MOV		AH,ES:[DI]
	JMP		@EXIT
@0:
	CMP		PutFn,pXOR
	JNE		@1
	MOV		AH,ES:[DI]
	XOR		AH,AL
	JMP		@EXIT
@1:
	CMP		PutFn,pADD
	JNE		@2
	MOV		AH,ES:[DI]
	ADD		AH,AL
	JMP		@EXIT
@2:
	CMP		PutFn,pSUB
	JNE		@3
	MOV		AH,ES:[DI]
	SUB		AH,AL
	JMP		@EXIT
@3:
	CMP		PutFn,pOR
	JNE		@4
	MOV		AH,ES:[DI]
	OR		AH,AL
	MOV		AL,AH
	JMP		@EXIT
@4:
	CMP		PutFn,pAND
	JNE		@5
	MOV		AH,ES:[DI]
	AND		AH,AL
	JMP		@EXIT
@5:
	CMP		PutFn,pDEC
	JNE		@5B
	DEC		AH
	JMP		@EXIT
@5B:
	CMP		PutFn,pINC
	JNE		@5C
	INC		AH
	JMP		@EXIT
@5C:
	CMP		PutFn,pTRANSLATE
	JNE		@6
	PUSH	BX
	PUSH	DS
	MOV		AL,ES:[DI]
	LDS		BX,DWORD PTR TransPtr
	XLAT
	POP		DS
	POP		BX
	MOV		AH,AL
	JMP		@EXIT
@6:
	CMP		PutFn,pAPPROACH
	JNE		@7
	MOV		AH,ES:[DI]
	CMP		AH,AL
	JB		@UP
	JA		@DOWN
	JMP		@EXIT
@UP:
	INC		AH
	JMP		@EXIT
@DOWN:
	DEC		AH
	JMP		@EXIT
@7:
	CMP		PutFn,pONLY
	JNE		@EXIT
	MOV		AH,ES:[DI]
	CMP		AH,MaskColor
	JNE		@EXIT
	MOV		AH,AL
@EXIT:
	MOV		ES:[DI],AH
End;

Function XGetPixel(X,Y : Word) : Byte; Assembler;
Asm
	MOV		AL,$00						{ Return color 0 if error			}
	CMP		X,$013F						{ Is X <= 319 (X-range) ?			}
	JA		@EXIT						{ No --> @EXIT						}
	CMP		Y,$00C7						{ Is Y <= 199 (Y-range) ?			}
	JA		@EXIT						{ No --> @EXIT						}
	MOV		AX,$0140
	MUL		Y
	ADD		AX,X
	MOV		DI,AX						{ Set DI to AX to move down			}
	MOV		ES,ScrSeg					{ Point ES:0 to screen				}
	MOV		AL,ES:[DI]					{ Fetch AL (Color) in ES:[DI] (X,Y)	}
@EXIT:
End;

Procedure XLine(X1,Y1,X2,Y2 : Word; Color : Byte); Assembler;

	Procedure XSpecialLine(X1,Y1,X2,Y2 : Word; Color : Byte); Assembler;
	Var
		XAdvance,WholeStep,AdjUp,AdjDown : Word;

		Procedure PutPixel(Color : Byte); Assembler;
		Asm
			MOV		AL,Color
			CMP		PutFn,pNORMAL
			JNE		@01
			MOV		AH,AL
			JMP		@EXIT
		@01:
			CMP		PutFn,pTRANSPARENT
			JNE		@0
			MOV		AH,AL
			CMP		AL,MaskColor
			JNE		@EXIT
			MOV		AH,ES:[DI]
			JMP		@EXIT
		@0:
			CMP		PutFn,pXOR
			JNE		@1
			MOV		AH,ES:[DI]
			XOR		AH,AL
			JMP		@EXIT
		@1:
			CMP		PutFn,pADD
			JNE		@2
			MOV		AH,ES:[DI]
			ADD		AH,AL
			JMP		@EXIT
		@2:
			CMP		PutFn,pSUB
			JNE		@3
			MOV		AH,ES:[DI]
			SUB		AH,AL
			JMP		@EXIT
		@3:
			CMP		PutFn,pOR
			JNE		@4
			MOV		AH,ES:[DI]
			OR		AH,AL
			MOV		AL,AH
			JMP		@EXIT
		@4:
			CMP		PutFn,pAND
			JNE		@5
			MOV		AH,ES:[DI]
			AND		AH,AL
			JMP		@EXIT
		@5:
			CMP		PutFn,pDEC
			JNE		@5B
			DEC		AH
			JMP		@EXIT
		@5B:
			CMP		PutFn,pINC
			JNE		@5C
			INC		AH
			JMP		@EXIT
		@5C:
			CMP		PutFn,pTRANSLATE
			JNE		@6
			PUSH	BX
			PUSH	DS
			MOV     AL,ES:[DI]
			LDS		BX,DWORD PTR TransPtr
			XLAT
			POP		DS
			POP		BX
			MOV		AH,AL
			JMP		@EXIT
		@6:
			CMP		PutFn,pAPPROACH
			JNE		@7
			MOV		AH,ES:[DI]
			CMP		AH,AL
			JB		@UP
			JA		@DOWN
			JMP		@EXIT
		@UP:
			INC		AH
			JMP		@EXIT
		@DOWN:
			DEC		AH
			JMP		@EXIT
		@7:
			CMP		PutFn,pONLY
			JNE		@EXIT
			MOV		AH,ES:[DI]
			CMP		AH,MaskColor
			JNE		@EXIT
			MOV		AH,AL
		@EXIT:
			MOV		ES:[DI],AH
		End;

		Procedure PutPixelA(Color : Byte); Assembler;
		Asm
			MOV		AL,Color
			CMP		PutFn,pNORMAL
			JNE		@01
			MOV		AH,AL
			JMP		@EXIT
		@01:
			CMP		PutFn,pTRANSPARENT
			JNE		@0
			MOV		AH,AL
			CMP		AL,MaskColor
			JNE		@EXIT
			MOV		AH,ES:[DI]
			JMP		@EXIT
		@0:
			CMP		PutFn,pXOR
			JNE		@1
			MOV		AH,ES:[DI]
			XOR		AH,AL
			JMP		@EXIT
		@1:
			CMP		PutFn,pADD
			JNE		@2
			MOV		AH,ES:[DI]
			ADD		AH,AL
			JMP		@EXIT
		@2:
			CMP		PutFn,pSUB
			JNE		@3
			MOV		AH,ES:[DI]
			SUB		AH,AL
			JMP		@EXIT
		@3:
			CMP		PutFn,pOR
			JNE		@4
			MOV		AH,ES:[DI]
			OR		AH,AL
			JMP		@EXIT
		@4:
			CMP		PutFn,pAND
			JNE		@6
			MOV		AH,ES:[DI]
			AND		AH,AL
			JMP		@EXIT
		@5:
			CMP		PutFn,pDEC
			JNE		@5B
			DEC		AH
			JMP		@EXIT
		@5B:
			CMP		PutFn,pINC
			JNE		@5C
			INC		AH
			JMP		@EXIT
		@5C:
			CMP		PutFn,pTRANSLATE
			JNE		@6
			PUSH	BX
			PUSH	DS
			MOV		AL,ES:[DI]
			LDS		BX,DWORD PTR TransPtr
			XLAT
			POP		DS
			POP		BX
			MOV		AH,AL
			JMP		@EXIT
		@6:
			CMP		PutFn,pAPPROACH
			JNE		@7
			MOV		AH,ES:[DI]
			CMP		AH,AL
			JB		@UP
			JA		@DOWN
			JMP		@EXIT
		@UP:
			INC		AH
			JMP		@EXIT
		@DOWN:
			DEC		AH
			JMP		@EXIT
		@7:
			CMP		PutFn,pONLY
			JNE		@EXIT
			MOV		AH,ES:[DI]
			CMP		AH,MaskColor
			JNE		@EXIT
			MOV		AH,AL
		@EXIT:
			XCHG	AL,AH
			STOSB
			XCHG	AL,AH
		End;

	Asm
		CMP		PutFn,pTRANSPARENT
		JNE		@1
		MOV		AL,MaskColor
		CMP		AL,Color
		JNE		@1
		JMP		@EXIT
	@1:
		MOV		AX,X1						{ Put X1 in AX						}
		XOR		AX,X2						{ X1 (AX) = X2 ?					}
		JZ		@VERTICAL					{ Yes --> @VERTICAL					}
		MOV		AX,Y1						{ Put Y1 in AX						}
		XOR		AX,Y2						{ Y1 (AX) = Y2 ?					}
		JZ		@HORIZONTAL					{ Yes --> @HORIZONTAL				}
		MOV		AX,X2						{ Put X2 in AX						}
		CMP		AX,X1						{ AX (X2) > X1 ?					}
		JA		@GET_DIST_X_NORMAL			{ Yes --> @GET_DIST_X_NORMAL		}
		MOV		AX,X1						{ Put X1 in AX						}
		SUB		AX,X2						{ Subtract X2 from AX (X1)			}
		JMP		@GOT_DIST_X					{ Branch to @GOT_DIST_X				}
	@GET_DIST_X_NORMAL:
		SUB		AX,X1						{ Subtract X1 from AX (X2)			}
	@GOT_DIST_X:
		INC		AX							{ Increment AX for length of line	}
		MOV		BX,AX						{ Put AX in BX for later use		}
		MOV		AX,Y2						{ Put Y2 in AX						}
		CMP		AX,Y1						{ Y2 > Y1 ?							}
		JA		@GET_DIST_Y_NORMAL			{ Yes --> @GET_DIST_Y_NORMAL		}
		MOV		AX,Y1						{ Put Y1 in AX						}
		SUB		AX,Y2						{ Subtract Y2 from AX (Y1)			}
		JMP		@GOT_DIST_Y					{ Branch to @GOT_DIST_Y				}
	@GET_DIST_Y_NORMAL:
		SUB		AX,Y1						{ Subtract Y1 from AX (Y2)			}
	@GOT_DIST_Y:
		INC		AX							{ Increment AX for length of line	}
		CMP		AX,BX						{ DeltaX = DeltaY ?					}
		JE		@PERFECT_DIAGONAL			{ Yes --> @PERFECT_DIAGONAL			}
	@DIAGONAL:
		DEC		BX							{ Convert back to Delta				}
		DEC		AX							{ Convert back to Delta				}
		PUSH	BX							{ Save DeltaX on stack				}
		PUSH	AX							{ Save DeltaY on stack				}
		MOV		AX,Y1						{ Put Y1 in AX						}
		CMP		AX,Y2						{ Y1 > Y2 ?							}
		JNA		@ENDS_OK					{ No --> @ENDS_OK					}
	@SWAP_ENDS:
		MOV		AX,X1						{ Put X1 in AX						}
		XCHG	AX,X2						{ Swap AX (X1) And X2				}
		MOV		X1,AX						{ Put AX in X1						}
		MOV		AX,Y1						{ Put Y1 in AX						}
		XCHG	AX,Y2						{ Swap AX (Y1) And Y2				}
		MOV		Y1,AX						{ Put AX in Y1						}
	@ENDS_OK:
		XOR		DI,DI						{ Set offset part to $0000			}
		MOV		AX,Y1						{ Put Y1 in AX						}
		MOV		BX,$0140					{ Put $0140 (320) in BX				}
		MUL		BX							{ Multiply AX with BX into DX:AX	}
		ADD		DI,AX						{ Add AX to DI to move down			}
		ADD		DI,X1						{ Add X1 to DI to move right		}
		MOV		ES,ScrSeg					{ Point ES:0 to screen				}
		POP		CX							{ Restore DeltaY					}
		POP		DX							{ Restore DeltaX					}
		CMP		DX,CX						{ DeltaX > DeltaY ?					}
		JA		@X_MAJOR					{ Yes --> @X_MAJOR					}
	@Y_MAJOR:
		MOV		XAdvance,$0001				{ X is advanced to the right		}
		MOV		AX,X1						{ Put X1 in AX						}
		CMP		AX,X2						{ Is AX (X1) < X2 ?					}
		JB		@XADVANCE_ADJUSTED			{ Yes --> @XADVANCE_ADJUSTED		}
		MOV		XAdvance,$FFFF				{ X is advanced to the left			}
	@XADVANCE_ADJUSTED:
		MOV		AX,CX						{ Put DeltaX in AX					}
		MOV		CX,DX						{ Put DeltaY in CX					}
		XOR		DX,DX						{ Clear DX for division				}
		DIV		CX							{ AX=DX:AX div CX, DX=DX:AX mod CX	}
		MOV		BX,DX						{ Put MOD result in BX				}
		SHL		BX,$01						{ Shift left 1 bit ( * 2)			}
		MOV		AdjUp,BX					{ Put BX in AdjUp					}
		MOV		SI,CX						{ Put DeltaY in SI					}
		SHL		SI,$01						{ Shift left 1 bit ( * 2)			}
		MOV		AdjDown,SI					{ Put SI in AdjDown					}
		SUB		DX,SI						{ Subtract SI from DX into DX		}
		MOV		SI,CX						{ Put DeltaY in SI					}
		MOV		CX,AX						{ Put DIV result in CX				}
		SHR		CX,$01						{ Shift right 1 bit (/ 2)			}
		INC		CX							{ Increment CX						}
		PUSH	CX							{ Save last partial run length		}
		ADD		DX,SI						{ Add SI to DX into DX				}
		TEST	AL,$01						{ Even run length ?					}
		JNZ		@YMAJOR_ADJUSTED			{ No --> @YMAJOR_ADJUSTED			}
		SUB		DX,SI						{ Undo old stuff					}
		AND		BX,BX						{ Adjust up equal to 0 ?			}
		JNZ     @YMAJOR_ADJUSTED			{ No --> @YMAJOR_ADJUSTED			}
		DEC		CX							{ Decrement initial run				}
	@YMAJOR_ADJUSTED:
		MOV		WholeStep,AX				{ Put AX into WholeStep				}
		MOV		AL,Color					{ Put Color in AL					}
		MOV		BX,XAdvance					{ Put X-Advance in BX				}
	@YMAJOR_FIRST:
		PUSH	AX
		PUSH	WORD PTR [BP]
		CALL	PutPixel
		ADD		DI,$0140					{ Move down to next line			}
		LOOP	@YMAJOR_FIRST				{ If more on this run, loop again	}
		ADD		DI,BX						{ Increment X-Axis					}
		CMP		SI,$0001					{ More than 2 runs ?				}
		JNA		@YMAJOR_LAST				{ No --> @YMAJOR_LAST				}
		DEC		DX							{ Decrement DX for carry operations	}
		SHR		SI,$0001					{ Adjust SI for scan-pair runs		}
		JNC		@YMAJOR_ODD					{ If odd --> @YMAJOR_ODD			}
	@YMAJOR_FULL:
		MOV		CX,WholeStep				{ Get length of run					}
		ADD		DX,AdjUp					{ Adjust error term					}
		JNC		@YMAJOR_NOEXTRA				{ No adjustment for errors			}
		INC		CX							{ Increment length of run by 1		}
		SUB		DX,AdjDown					{ Adjust error term down			}
	@YMAJOR_NOEXTRA:
		PUSH	AX
		PUSH	WORD PTR [BP]
		CALL	PutPixel
		ADD		DI,$0140					{ Move down to next line			}
		LOOP	@YMAJOR_NOEXTRA				{ If more on this run, loop again	}
		ADD		DI,BX						{ Increment X-Axis					}
	@YMAJOR_ODD:
		MOV		CX,WholeStep				{ Get length of run					}
		ADD		DX,AdjUp					{ Adjust error term					}
		JNC		@YMAJOR_NOEXTRA2			{ No adjustment for errors			}
		INC		CX							{ Increment length of run by 1		}
		SUB		DX,AdjDown					{ Adjust error term down			}
	@YMAJOR_NOEXTRA2:
		PUSH	AX
		PUSH	WORD PTR [BP]
		CALL	PutPixel
		ADD		DI,$0140					{ Move down to next line			}
		LOOP	@YMAJOR_NOEXTRA2			{ If more on this run, loop again	}
		ADD		DI,BX						{ Increment X-Axis					}
		DEC		SI                   		{ More runs ?						}
		JNZ		@YMAJOR_FULL				{ Yes --> @YMAJOR_FULL				}
	@YMAJOR_LAST:
		POP		CX							{ Restore length of final run		}
	@YMAJOR_LAST_LOOP:
		PUSH	AX
		PUSH	WORD PTR [BP]
		CALL	PutPixel
		ADD		DI,$0140					{ Move down to next line			}
		LOOP	@YMAJOR_LAST_LOOP			{ If more on this run, loop again	}
		JMP		@EXIT						{ Exit								}
	@X_MAJOR:
		CLD									{ Left to right line				}
		MOV		AX,X1						{ Put X1 in AX						}
		CMP		AX,X2						{ AX (X1) < X2 ?					}
		JB		@X_DFSET					{ Yes --> @X_DFSET					}
		STD									{ Right to left line				}
	@X_DFSET:
		MOV		AX,DX						{ Put DX (DeltaX) in AX				}
		XOR		DX,DX						{ Clear DX for division				}
		DIV		CX							{ AX=DX:AX div CX, DX=DX:AX mod CX	}
		MOV		BX,DX						{ Put MOD result in BX				}
		SHL		BX,$01						{ Shift left 1 bit ( * 2)			}
		MOV		AdjUp,BX					{ Store BX in AdjUp					}
		MOV		SI,CX						{ Put CX (DeltaY) in SI				}
		SHL		SI,$01						{ Shift left 1 bit ( * 2)			}
		MOV		AdjDown,SI					{ Store SI in AdjDown				}
		SUB		DX,SI						{ Initial error term				}
		MOV		SI,CX						{ SI = CX (DeltaY)					}
		MOV		CX,AX						{ Minimum run length				}
		SHR		CX,$01						{ Shift left 1 bit ( * 2)			}
		INC		CX							{ Minimum is plus 1					}
		PUSH	CX							{ Save CX (Pixel count)				}
		ADD		DX,SI						{ Add SI to DX into DX				}
		TEST	AL,$01						{ Is run length even ?				}
		JNZ		@XMAJOR_ADJUSTED			{ No --> @XMAJOR_ADJUSTED			}
		SUB		DX,SI						{ Subtract SI from DX into DX		}
		AND		BX,BX						{ Is the adjust up equal to 0 ?		}
		JNZ		@XMAJOR_ADJUSTED			{ No --> @XMAJOR_ADJUSTED			}
		DEC		CX							{ Make initial run 1 shorter		}
	@XMAJOR_ADJUSTED:
		MOV		WholeStep,AX				{ Put AX in WholeStep				}
		MOV		AL,Color
	@LOOP1:
		PUSH	AX
		PUSH	WORD PTR [BP]
		CALL	PutPixelA
		LOOP	@LOOP1
		ADD		DI,$0140					{ Increment to next line			}
		CMP		SI,$0001					{ More than 2 scans ?				}
		JNA		@XMAJOR_LAST				{ No --> @XMAJOR_LAST				}
		DEC		DX							{ Adjust DX so we can use carry		}
		SHR		SI,$01						{ Convert to scan-pair count		}
		JNC		@XMAJOR_ODD					{ If not carry, then @XMAJOR_ODD	}
	@XMAJOR_FULL:
		MOV		CX,WholeStep				{ Get pixel count into CX			}
		ADD		DX,BX						{ Advance the error term			}
		JNC		@XMAJOR_NOEXTRA				{ No extra --> @XMAJOR_NOEXTRA		}
		INC		CX							{ One extra pixel in this run		}
		SUB		DX,AdjDown					{ Reset the error term				}
	@XMAJOR_NOEXTRA:
	@LOOP2:
		PUSH	AX
		PUSH	WORD PTR [BP]
		CALL	PutPixelA
		LOOP	@LOOP2
		ADD		DI,$0140					{ Move down to next line			}
	@XMAJOR_ODD:
		MOV		CX,WholeStep				{ Get pixel count into CX			}
		ADD		DX,BX						{ Advance the error term			}
		JNC		@XMAJOR_NOEXTRA2			{ No extra --> @XMAJOR_NOEXTRA2		}
		INC		CX							{ One extra pixel in this run		}
		SUB		DX,AdjDown					{ Reset the error term				}
	@XMAJOR_NOEXTRA2:
	@LOOP3:
		PUSH	AX
		PUSH	WORD PTR [BP]
		CALL	PutPixelA
		LOOP	@LOOP3
		ADD		DI,$0140					{ Move down to next line			}
		DEC		SI							{ Decrement scan-pair count			}
		JNZ		@XMAJOR_FULL				{ If more --> @XMAJOR_FULL			}
	@XMAJOR_LAST:
		POP		CX							{ Get the final pixel run length	}
	@LOOP4:
		PUSH	AX
		PUSH	WORD PTR [BP]
		CALL	PutPixelA
		LOOP	@LOOP4
		JMP		@EXIT						{ Exit								}
	@PERFECT_DIAGONAL:
		CMP		X1,$013F					{ X1 <= $013F (319) ?				}
		JA		@EXIT						{ No --> @EXIT						}
		CMP		X2,$013F					{ X2 <= $013F (319) ?				}
		JA		@EXIT						{ No --> @EXIT						}
		CMP		Y1,$00C7					{ Y1 <= $00C7 (199) ?				}
		JA		@EXIT						{ No --> @EXIT						}
		CMP		Y2,$00C7					{ Y2 <= $00C7 (199) ?				}
		JA		@EXIT						{ No --> @EXIT						}
		MOV		CX,AX						{ Store length of line in CX		}
		XOR		DI,DI						{ Set offset part to $0000			}
		MOV		AX,Y1						{ Put Y1 in AX						}
		MOV		BX,$0140					{ Put $0140 (320) in BX				}
		MUL		BX							{ Multiply AX with BX into DX:AX	}
		ADD		DI,AX						{ Add AX to DI to move down			}
		ADD		DI,X1						{ Add X1 to DI to move right		}
		MOV		ES,ScrSeg					{ Point ES:0 to screen				}
		MOV		AX,X2						{ Put X2 in AX						}
		CMP		AX,X1						{ AX (X2) < X1 ?					}
		JB		@DIAGONAL_LEFT				{ Yes --> @DIAGONAL_LEFT			}
	@DIAGONAL_RIGHT:
		MOV		DX,$0141					{ Pixel distance = $0141 (321)		}
		JMP		@CHECK_DIRECTION			{ Branch to @CHECK_DIRECTION		}
	@DIAGONAL_LEFT:
		MOV		DX,$013F					{ Pixel distance = $013F (319)		}
	@CHECK_DIRECTION:
		MOV		AX,Y2						{ Put Y2 in AX						}
		CMP		AX,Y1						{ AX (Y2) < Y1 ?					}
		JB		@DIAGONAL_UP				{ Yes --> @DIAGONAL_UP				}
	@DIAGONAL_DOWN:
		MOV		AL,Color					{ Put Color in AL					}
	@DIAGONAL_DOWN_LOOP:
		PUSH	AX
		PUSH	WORD PTR [BP]
		CALL	PutPixel
		ADD		DI,DX						{ Move to next pixel address		}
		LOOP	@DIAGONAL_DOWN_LOOP			{ If more pixels, loop again		}
		JMP		@EXIT						{ Exit								}
	@DIAGONAL_UP:
		MOV		BX,DX						{ Put DX into BX					}
		MOV		DX,$0280					{ Put $0280 in DX (640)				}
		SUB		DX,BX						{ Subtract BX from DX into DX		}
		MOV		AL,Color					{ Put Color in AL					}
	@DIAGONAL_UP_LOOP:
		PUSH	AX
		PUSH	WORD PTR [BP]
		CALL	PutPixel
		SUB		DI,DX						{ Move to next pixel address		}
		LOOP	@DIAGONAL_UP_LOOP			{ If more pixels, loop again		}
		JMP		@EXIT						{ Exit								}
	@HORIZONTAL:
		MOV		AX,X1						{ Put X1 in AX						}
		CMP		AX,X2						{ X1 (AX) <= X2 ?					}
		JBE		@HORIZONTAL_OK				{ Yes --> @HORIZONTAL_OK			}
		XCHG	AX,X2						{ Swap AX (X1) and X2				}
		MOV		X1,AX						{ Store AX in X1					}
	@HORIZONTAL_OK:
		CMP		X1,$013F					{ X1 > $013F (319) ?				}
		JA		@EXIT						{ Yes --> @EXIT						}
		CMP		X2,$013F					{ X2 > $013F (319) ?				}
		JA		@EXIT						{ Yes --> @EXIT						}
		CMP		Y1,$00C7					{ Y1 > $00C7 (199) ?				}
		JA		@EXIT						{ Yes --> @EXIT						}
		XOR		DI,DI						{ Set offset part to $0000			}
		MOV		AX,Y1						{ Put Y1 in AX						}
		MOV		BX,$0140					{ Put $0140 (320) in BX				}
		MUL		BX							{ Multiply AX with BX into DX:AX	}
		ADD		DI,AX						{ Add AX to DI to move down			}
		ADD		DI,X1						{ Add X1 to DI to move right		}
		MOV		CX,X2						{ Put X2 in CX						}
		SUB		CX,X1						{ Calculate distance from X1 to X2	}
		INC		CX							{ Increment for length of line		}
		CLD									{ Store upwards in memory			}
		MOV		AL,Color					{ Put Color in AL					}
		MOV		ES,ScrSeg					{ Point ES:0 to screen				}
	@LOOP5:
		PUSH	AX
		PUSH	WORD PTR [BP]
		CALL	PutPixelA
		LOOP	@LOOP5
		JMP		@EXIT						{ Exit								}
	@VERTICAL:
		MOV		AX,Y1						{ Put Y1 in AX						}
		CMP		AX,Y2						{ Y1 (AX) <= Y2 ?					}
		JBE		@VERTICAL_OK				{ Yes --> @VERTICAL_OK				}
		XCHG	AX,Y2						{ Swap AX (Y1) and Y2				}
		MOV		Y1,AX						{ Store AX in Y1					}
	@VERTICAL_OK:
		CMP		Y1,$00C7					{ Y1 > $00C7 (199) ?				}
		JA		@EXIT						{ Yes --> @EXIT						}
		CMP		Y2,$00C7					{ Y2 > $00C7 (199) ?				}
		JA		@EXIT						{ Yes --> @EXIT						}
		CMP		X1,$013F					{ X1 > $013F (319) ?				}
		JA		@EXIT						{ Yes --> @EXIT						}
		XOR		DI,DI						{ Set offset part to $0000			}
		MOV		AX,Y1						{ Put Y1 in AX						}
		MOV		BX,$0140					{ Put $0140 (320) in BX				}
		MUL		BX							{ Multiply AX with BX into DX:AX	}
		ADD		DI,AX						{ Add AX to DI to move down			}
		ADD		DI,X1						{ Add X1 to DI to move right		}
		MOV		CX,Y2						{ Put X2 in CX						}
		SUB		CX,Y1						{ Calculate distance from X1 to X2	}
		INC		CX							{ Increment for length of line		}
		MOV		AL,Color					{ Put Color in AL					}
		MOV		ES,ScrSeg					{ Point ES:0 to screen				}
	@VERTICAL_LOOP:
		PUSH	AX
		PUSH	WORD PTR [BP]
		CALL	PutPixel
		ADD		DI,$0140					{ Add $0140 (320) to DI = move down	}
		LOOP	@VERTICAL_LOOP				{ If more pixels --> @VERTICAL_LOOP	}
	@EXIT:
	End;

	Procedure XNormalLine(X1,Y1,X2,Y2 : Word; Color : Byte); Assembler;
	Var
		XAdvance,WholeStep,AdjUp,AdjDown : Word;
	Asm
		MOV		AX,X1						{ Put X1 in AX						}
		XOR		AX,X2						{ X1 (AX) = X2 ?					}
		JZ		@VERTICAL					{ Yes --> @VERTICAL					}
		MOV		AX,Y1						{ Put Y1 in AX						}
		XOR		AX,Y2						{ Y1 (AX) = Y2 ?					}
		JZ		@HORIZONTAL					{ Yes --> @HORIZONTAL				}
		MOV		AX,X2						{ Put X2 in AX						}
		CMP		AX,X1						{ AX (X2) > X1 ?					}
		JA		@GET_DIST_X_NORMAL			{ Yes --> @GET_DIST_X_NORMAL		}
		MOV		AX,X1						{ Put X1 in AX						}
		SUB		AX,X2						{ Subtract X2 from AX (X1)			}
		JMP		@GOT_DIST_X					{ Branch to @GOT_DIST_X				}
	@GET_DIST_X_NORMAL:
		SUB		AX,X1						{ Subtract X1 from AX (X2)			}
	@GOT_DIST_X:
		INC		AX							{ Increment AX for length of line	}
		MOV		BX,AX						{ Put AX in BX for later use		}
		MOV		AX,Y2						{ Put Y2 in AX						}
		CMP		AX,Y1						{ Y2 > Y1 ?							}
		JA		@GET_DIST_Y_NORMAL			{ Yes --> @GET_DIST_Y_NORMAL		}
		MOV		AX,Y1						{ Put Y1 in AX						}
		SUB		AX,Y2						{ Subtract Y2 from AX (Y1)			}
		JMP		@GOT_DIST_Y					{ Branch to @GOT_DIST_Y				}
	@GET_DIST_Y_NORMAL:
		SUB		AX,Y1						{ Subtract Y1 from AX (Y2)			}
	@GOT_DIST_Y:
		INC		AX							{ Increment AX for length of line	}
		CMP		AX,BX						{ DeltaX = DeltaY ?					}
		JE		@PERFECT_DIAGONAL			{ Yes --> @PERFECT_DIAGONAL			}
	@DIAGONAL:
		DEC		BX							{ Convert back to Delta				}
		DEC		AX							{ Convert back to Delta				}
		PUSH	BX							{ Save DeltaX on stack				}
		PUSH	AX							{ Save DeltaY on stack				}
		MOV		AX,Y1						{ Put Y1 in AX						}
		CMP		AX,Y2						{ Y1 > Y2 ?							}
		JNA		@ENDS_OK					{ No --> @ENDS_OK					}
	@SWAP_ENDS:
		MOV		AX,X1						{ Put X1 in AX						}
		XCHG	AX,X2						{ Swap AX (X1) And X2				}
		MOV		X1,AX						{ Put AX in X1						}
		MOV		AX,Y1						{ Put Y1 in AX						}
		XCHG	AX,Y2						{ Swap AX (Y1) And Y2				}
		MOV		Y1,AX						{ Put AX in Y1						}
	@ENDS_OK:
		XOR		DI,DI						{ Set offset part to $0000			}
		MOV		AX,Y1						{ Put Y1 in AX						}
		MOV		BX,$0140					{ Put $0140 (320) in BX				}
		MUL		BX							{ Multiply AX with BX into DX:AX	}
		ADD		DI,AX						{ Add AX to DI to move down			}
		ADD		DI,X1						{ Add X1 to DI to move right		}
		MOV		ES,ScrSeg					{ Point ES:0 to screen				}
		POP		CX							{ Restore DeltaY					}
		POP		DX							{ Restore DeltaX					}
		CMP		DX,CX						{ DeltaX > DeltaY ?					}
		JA		@X_MAJOR					{ Yes --> @X_MAJOR					}
	@Y_MAJOR:
		MOV		XAdvance,$0001				{ X is advanced to the right		}
		MOV		AX,X1						{ Put X1 in AX						}
		CMP		AX,X2						{ Is AX (X1) < X2 ?					}
		JB		@XADVANCE_ADJUSTED			{ Yes --> @XADVANCE_ADJUSTED		}
		MOV		XAdvance,$FFFF				{ X is advanced to the left			}
	@XADVANCE_ADJUSTED:
		MOV		AX,CX						{ Put DeltaX in AX					}
		MOV		CX,DX						{ Put DeltaY in CX					}
		XOR		DX,DX						{ Clear DX for division				}
		DIV		CX							{ AX=DX:AX div CX, DX=DX:AX mod CX	}
		MOV		BX,DX						{ Put MOD result in BX				}
		SHL		BX,$01						{ Shift left 1 bit ( * 2)			}
		MOV		AdjUp,BX					{ Put BX in AdjUp					}
		MOV		SI,CX						{ Put DeltaY in SI					}
		SHL		SI,$01						{ Shift left 1 bit ( * 2)			}
		MOV		AdjDown,SI					{ Put SI in AdjDown					}
		SUB		DX,SI						{ Subtract SI from DX into DX		}
		MOV		SI,CX						{ Put DeltaY in SI					}
		MOV		CX,AX						{ Put DIV result in CX				}
		SHR		CX,$01						{ Shift right 1 bit (/ 2)			}
		INC		CX							{ Increment CX						}
		PUSH	CX							{ Save last partial run length		}
		ADD		DX,SI						{ Add SI to DX into DX				}
		TEST	AL,$01						{ Even run length ?					}
		JNZ		@YMAJOR_ADJUSTED			{ No --> @YMAJOR_ADJUSTED			}
		SUB		DX,SI						{ Undo old stuff					}
		AND		BX,BX						{ Adjust up equal to 0 ?			}
		JNZ     @YMAJOR_ADJUSTED			{ No --> @YMAJOR_ADJUSTED			}
		DEC		CX							{ Decrement initial run				}
	@YMAJOR_ADJUSTED:
		MOV		WholeStep,AX				{ Put AX into WholeStep				}
		MOV		AL,Color					{ Put Color in AL					}
		MOV		BX,XAdvance					{ Put X-Advance in BX				}
	@YMAJOR_FIRST:
		MOV		ES:[DI],AL
		ADD		DI,$0140					{ Move down to next line			}
		LOOP	@YMAJOR_FIRST				{ If more on this run, loop again	}
		ADD		DI,BX						{ Increment X-Axis					}
		CMP		SI,$0001					{ More than 2 runs ?				}
		JNA		@YMAJOR_LAST				{ No --> @YMAJOR_LAST				}
		DEC		DX							{ Decrement DX for carry operations	}
		SHR		SI,$0001					{ Adjust SI for scan-pair runs		}
		JNC		@YMAJOR_ODD					{ If odd --> @YMAJOR_ODD			}
	@YMAJOR_FULL:
		MOV		CX,WholeStep				{ Get length of run					}
		ADD		DX,AdjUp					{ Adjust error term					}
		JNC		@YMAJOR_NOEXTRA				{ No adjustment for errors			}
		INC		CX							{ Increment length of run by 1		}
		SUB		DX,AdjDown					{ Adjust error term down			}
	@YMAJOR_NOEXTRA:
		MOV		ES:[DI],AL					{ Display pixel						}
		ADD		DI,$0140					{ Move down to next line			}
		LOOP	@YMAJOR_NOEXTRA				{ If more on this run, loop again	}
		ADD		DI,BX						{ Increment X-Axis					}
	@YMAJOR_ODD:
		MOV		CX,WholeStep				{ Get length of run					}
		ADD		DX,AdjUp					{ Adjust error term					}
		JNC		@YMAJOR_NOEXTRA2			{ No adjustment for errors			}
		INC		CX							{ Increment length of run by 1		}
		SUB		DX,AdjDown					{ Adjust error term down			}
	@YMAJOR_NOEXTRA2:
		MOV		ES:[DI],AL					{ Display pixel						}
		ADD		DI,$0140					{ Move down to next line			}
		LOOP	@YMAJOR_NOEXTRA2			{ If more on this run, loop again	}
		ADD		DI,BX						{ Increment X-Axis					}
		DEC		SI                   		{ More runs ?						}
		JNZ		@YMAJOR_FULL				{ Yes --> @YMAJOR_FULL				}
	@YMAJOR_LAST:
		POP		CX							{ Restore length of final run		}
	@YMAJOR_LAST_LOOP:
		MOV		ES:[DI],AL					{ Display pixel						}
		ADD		DI,$0140					{ Move down to next line			}
		LOOP	@YMAJOR_LAST_LOOP			{ If more on this run, loop again	}
		JMP		@EXIT						{ Exit								}
	@X_MAJOR:
		CLD									{ Left to right line				}
		MOV		AX,X1						{ Put X1 in AX						}
		CMP		AX,X2						{ AX (X1) < X2 ?					}
		JB		@X_DFSET					{ Yes --> @X_DFSET					}
		STD									{ Right to left line				}
	@X_DFSET:
		MOV		AX,DX						{ Put DX (DeltaX) in AX				}
		XOR		DX,DX						{ Clear DX for division				}
		DIV		CX							{ AX=DX:AX div CX, DX=DX:AX mod CX	}
		MOV		BX,DX						{ Put MOD result in BX				}
		SHL		BX,$01						{ Shift left 1 bit ( * 2)			}
		MOV		AdjUp,BX					{ Store BX in AdjUp					}
		MOV		SI,CX						{ Put CX (DeltaY) in SI				}
		SHL		SI,$01						{ Shift left 1 bit ( * 2)			}
		MOV		AdjDown,SI					{ Store SI in AdjDown				}
		SUB		DX,SI						{ Initial error term				}
		MOV		SI,CX						{ SI = CX (DeltaY)					}
		MOV		CX,AX						{ Minimum run length				}
		SHR		CX,$01						{ Shift left 1 bit ( * 2)			}
		INC		CX							{ Minimum is plus 1					}
		PUSH	CX							{ Save CX (Pixel count)				}
		ADD		DX,SI						{ Add SI to DX into DX				}
		TEST	AL,$01						{ Is run length even ?				}
		JNZ		@XMAJOR_ADJUSTED			{ No --> @XMAJOR_ADJUSTED			}
		SUB		DX,SI						{ Subtract SI from DX into DX		}
		AND		BX,BX						{ Is the adjust up equal to 0 ?		}
		JNZ		@XMAJOR_ADJUSTED			{ No --> @XMAJOR_ADJUSTED			}
		DEC		CX							{ Make initial run 1 shorter		}
	@XMAJOR_ADJUSTED:
		MOV		WholeStep,AX				{ Put AX in WholeStep				}
		MOV		AL,Color
		REP		STOSB
		ADD		DI,$0140					{ Increment to next line			}
		CMP		SI,$0001					{ More than 2 scans ?				}
		JNA		@XMAJOR_LAST				{ No --> @XMAJOR_LAST				}
		DEC		DX							{ Adjust DX so we can use carry		}
		SHR		SI,$01						{ Convert to scan-pair count		}
		JNC		@XMAJOR_ODD					{ If not carry, then @XMAJOR_ODD	}
	@XMAJOR_FULL:
		MOV		CX,WholeStep				{ Get pixel count into CX			}
		ADD		DX,BX						{ Advance the error term			}
		JNC		@XMAJOR_NOEXTRA				{ No extra --> @XMAJOR_NOEXTRA		}
		INC		CX							{ One extra pixel in this run		}
		SUB		DX,AdjDown					{ Reset the error term				}
	@XMAJOR_NOEXTRA:
		REP		STOSB						{ Draw the pixel run				}
		ADD		DI,$0140					{ Move down to next line			}
	@XMAJOR_ODD:
		MOV		CX,WholeStep				{ Get pixel count into CX			}
		ADD		DX,BX						{ Advance the error term			}
		JNC		@XMAJOR_NOEXTRA2			{ No extra --> @XMAJOR_NOEXTRA2		}
		INC		CX							{ One extra pixel in this run		}
		SUB		DX,AdjDown					{ Reset the error term				}
	@XMAJOR_NOEXTRA2:
		REP		STOSB						{ Draw the pixel run				}
		ADD		DI,$0140					{ Move down to next line			}
		DEC		SI							{ Decrement scan-pair count			}
		JNZ		@XMAJOR_FULL				{ If more --> @XMAJOR_FULL			}
	@XMAJOR_LAST:
		POP		CX							{ Get the final pixel run length	}
		REP		STOSB						{ Draw the pixel run				}
		JMP		@EXIT						{ Exit								}
	@PERFECT_DIAGONAL:
		CMP		X1,$013F					{ X1 <= $013F (319) ?				}
		JA		@EXIT						{ No --> @EXIT						}
		CMP		X2,$013F					{ X2 <= $013F (319) ?				}
		JA		@EXIT						{ No --> @EXIT						}
		CMP		Y1,$00C7					{ Y1 <= $00C7 (199) ?				}
		JA		@EXIT						{ No --> @EXIT						}
		CMP		Y2,$00C7					{ Y2 <= $00C7 (199) ?				}
		JA		@EXIT						{ No --> @EXIT						}
		MOV		CX,AX						{ Store length of line in CX		}
		XOR		DI,DI						{ Set offset part to $0000			}
		MOV		AX,Y1						{ Put Y1 in AX						}
		MOV		BX,$0140					{ Put $0140 (320) in BX				}
		MUL		BX							{ Multiply AX with BX into DX:AX	}
		ADD		DI,AX						{ Add AX to DI to move down			}
		ADD		DI,X1						{ Add X1 to DI to move right		}
		MOV		ES,ScrSeg					{ Point ES:0 to screen				}
		MOV		AX,X2						{ Put X2 in AX						}
		CMP		AX,X1						{ AX (X2) < X1 ?					}
		JB		@DIAGONAL_LEFT				{ Yes --> @DIAGONAL_LEFT			}
	@DIAGONAL_RIGHT:
		MOV		DX,$0141					{ Pixel distance = $0141 (321)		}
		JMP		@CHECK_DIRECTION			{ Branch to @CHECK_DIRECTION		}
	@DIAGONAL_LEFT:
		MOV		DX,$013F					{ Pixel distance = $013F (319)		}
	@CHECK_DIRECTION:
		MOV		AX,Y2						{ Put Y2 in AX						}
		CMP		AX,Y1						{ AX (Y2) < Y1 ?					}
		JB		@DIAGONAL_UP				{ Yes --> @DIAGONAL_UP				}
	@DIAGONAL_DOWN:
		MOV		AL,Color					{ Put Color in AL					}
	@DIAGONAL_DOWN_LOOP:
		MOV		ES:[DI],AL					{ Display pixel						}
		ADD		DI,DX						{ Move to next pixel address		}
		LOOP	@DIAGONAL_DOWN_LOOP			{ If more pixels, loop again		}
		JMP		@EXIT						{ Exit								}
	@DIAGONAL_UP:
		MOV		BX,DX						{ Put DX into BX					}
		MOV		DX,$0280					{ Put $0280 in DX (640)				}
		SUB		DX,BX						{ Subtract BX from DX into DX		}
		MOV		AL,Color					{ Put Color in AL					}
	@DIAGONAL_UP_LOOP:
		MOV		ES:[DI],AL					{ Display pixel						}
		SUB		DI,DX						{ Move to next pixel address		}
		LOOP	@DIAGONAL_UP_LOOP			{ If more pixels, loop again		}
		JMP		@EXIT						{ Exit								}
	@HORIZONTAL:
		MOV		AX,X1						{ Put X1 in AX						}
		CMP		AX,X2						{ X1 (AX) <= X2 ?					}
		JBE		@HORIZONTAL_OK				{ Yes --> @HORIZONTAL_OK			}
		XCHG	AX,X2						{ Swap AX (X1) and X2				}
		MOV		X1,AX						{ Store AX in X1					}
	@HORIZONTAL_OK:
		CMP		X1,$013F					{ X1 > $013F (319) ?				}
		JA		@EXIT						{ Yes --> @EXIT						}
		CMP		X2,$013F					{ X2 > $013F (319) ?				}
		JA		@EXIT						{ Yes --> @EXIT						}
		CMP		Y1,$00C7					{ Y1 > $00C7 (199) ?				}
		JA		@EXIT						{ Yes --> @EXIT						}
		XOR		DI,DI						{ Set offset part to $0000			}
		MOV		AX,Y1						{ Put Y1 in AX						}
		MOV		BX,$0140					{ Put $0140 (320) in BX				}
		MUL		BX							{ Multiply AX with BX into DX:AX	}
		ADD		DI,AX						{ Add AX to DI to move down			}
		ADD		DI,X1						{ Add X1 to DI to move right		}
		MOV		CX,X2						{ Put X2 in CX						}
		SUB		CX,X1						{ Calculate distance from X1 to X2	}
		INC		CX							{ Increment for length of line		}
		CLD									{ Store upwards in memory			}
		MOV		AL,Color					{ Put Color in AL					}
		MOV		ES,ScrSeg					{ Point ES:0 to screen				}
		REP		STOSB
		JMP		@EXIT						{ Exit								}
	@VERTICAL:
		MOV		AX,Y1						{ Put Y1 in AX						}
		CMP		AX,Y2						{ Y1 (AX) <= Y2 ?					}
		JBE		@VERTICAL_OK				{ Yes --> @VERTICAL_OK				}
		XCHG	AX,Y2						{ Swap AX (Y1) and Y2				}
		MOV		Y1,AX						{ Store AX in Y1					}
	@VERTICAL_OK:
		CMP		Y1,$00C7					{ Y1 > $00C7 (199) ?				}
		JA		@EXIT						{ Yes --> @EXIT						}
		CMP		Y2,$00C7					{ Y2 > $00C7 (199) ?				}
		JA		@EXIT						{ Yes --> @EXIT						}
		CMP		X1,$013F					{ X1 > $013F (319) ?				}
		JA		@EXIT						{ Yes --> @EXIT						}
		XOR		DI,DI						{ Set offset part to $0000			}
		MOV		AX,Y1						{ Put Y1 in AX						}
		MOV		BX,$0140					{ Put $0140 (320) in BX				}
		MUL		BX							{ Multiply AX with BX into DX:AX	}
		ADD		DI,AX						{ Add AX to DI to move down			}
		ADD		DI,X1						{ Add X1 to DI to move right		}
		MOV		CX,Y2						{ Put X2 in CX						}
		SUB		CX,Y1						{ Calculate distance from X1 to X2	}
		INC		CX							{ Increment for length of line		}
		MOV		AL,Color					{ Put Color in AL					}
		MOV		ES,ScrSeg					{ Point ES:0 to screen				}
	@VERTICAL_LOOP:
		MOV		ES:[DI],AL					{ Display pixel						}
		ADD		DI,$0140					{ Add $0140 (320) to DI = move down	}
		LOOP	@VERTICAL_LOOP				{ If more pixels --> @VERTICAL_LOOP	}
	@EXIT:
	End;

Asm
	PUSH	X1
	PUSH	Y1
	PUSH	X2
	PUSH	Y2
	MOV		AL,Color
	PUSH	AX
	PUSH	WORD PTR [BP]
	CMP		PutFn,pNORMAL
	JNE		@SPECIAL
	CALL	XNormalLine
	JMP		@EXIT
@SPECIAL:
	CALL	XSpecialLine
@EXIT:
End;

Procedure XBox(X1,Y1,X2,Y2 : Word; Color : Byte); Assembler;
Var
	WColor : Word;
Asm
	XOR		AX,AX						{ Clear AX register					}
	MOV		AL,Color					{ Put Color in AX					}
	MOV		WColor,AX					{ Put AX in WColor (Color --> Word)	}
	MOV		AX,X1						{ Put X1 in AX						}
	CMP		AX,X2						{ AX (X1) = X2 ?					}
	JE		@VERTICAL_LINE				{ Yes --> @VERTICAL_LINE			}
	MOV		AX,Y1						{ Put Y1 in AX						}
	CMP		AX,Y2						{ AX (Y1) = Y2 ?					}
	JE		@HORIZONTAL_LINE			{ Yes --> @HORIZONTAL_LINE			}
	PUSH	Y1							{ Push Y1 on stack					}
	PUSH	X1							{ Push X1 on stack					}
	MOV		AX,X2						{ Put X2 in AX						}
	DEC		AX							{ Decrement AX						}
	PUSH	AX							{ Push X2-1 on stack				}
	PUSH	WColor						{ Push WColor on stack				}
	CALL	XHLine						{ = Line(X1,Y1,X2-1,Y1,WORD(Color))	}
	PUSH	X2							{ Push X2 on stack					}
	PUSH	Y1							{ Push Y1 on stack					}
	MOV		AX,Y2						{ Put Y2 in AX						}
	DEC		AX							{ Decrement AX						}
	PUSH	AX							{ Push Y2-1 on stack				}
	PUSH	WColor						{ Push WColor on stack				}
	CALL	XVLine						{ = Line(X2,Y1,X2,Y2-1,WORD(Color))	}
	PUSH	Y2							{ Push Y2 on stack					}
	MOV		AX,X1						{ Put X1 in AX						}
	INC		AX							{ Increment AX						}
	PUSH	AX							{ Push X1+1 on stack				}
	PUSH	X2							{ Push X2 on stack					}
	PUSH	WColor						{ Push WColor on stack				}
	CALL	XHLine						{ = Line(X1+1,Y2,X2,Y2,WORD(Color))	}
	PUSH	X1							{ Push X1 on stack					}
	MOV		AX,Y1						{ Put Y1 in AX						}
	INC		AX							{ Increment AX						}
	PUSH	AX							{ Push Y1+1 on stack				}
	PUSH	Y2							{ Push Y2 on stack					}
	PUSH	WColor						{ Push WColor on stack				}
	CALL	XVLine						{ = Line(X1,Y1+1,X1,Y2,WORD(Color))	}
	JMP		@EXIT						{ Exit								}
@HORIZONTAL_LINE:
	PUSH	Y1							{ Push Y1 on stack					}
	PUSH	X1							{ Push X1 on stack					}
	PUSH	X2							{ Push X2 on stack					}
	PUSH	WColor						{ Push WColor on stack				}
	CALL	XHLine						{ = HLine(Y1,X1,X2,WORD(Color))		}
	JMP		@EXIT						{ Exit								}
@VERTICAL_LINE:
	PUSH	X1							{ Push X1 on stack					}
	PUSH	Y1							{ Push Y1 on stack					}
	PUSH	Y2							{ Push Y2 on stack					}
	PUSH	WColor						{ Push WColor on stack				}
	CALL	XVLine						{ = VLine(X1,Y1,Y2,WORD(Color))		}
	JMP		@EXIT						{ Exit								}
@EXIT:
End;

Procedure XRectangle(X1,Y1,X2,Y2 : Word; Color : Byte); Assembler;

	Procedure XNormalRectangle(X1,Y1,X2,Y2 : Word; Color : Byte); Assembler;
	Asm
		PUSH	ES				{ Save ES										}
		MOV		ES,ScrSeg					{ Point ES:0 to screen				}
		CMP		X1,$0140		{ Is X1 inside screen ?							}
		JAE		@EXIT			{ No --> @EXIT									}
		CMP		X2,$0140		{ Is X2 inside screen ?							}
		JB		@OK1			{ Yes --> @OK1									}
		MOV		X2,$013F		{ Move X2 to edge of screen						}
	@OK1:
		CMP		Y1,$00C8		{ Is Y1 inside screen ?							}
		JAE		@EXIT			{ No --> @EXIT									}
		CMP		Y2,$00C8		{ Is Y2 inside screen ?							}
		JB		@OK2			{ Yes --> @OK2									}
		MOV		Y2,$00C7		{ Move Y2 to edge of screen						}
	@OK2:
		MOV		AX,X1			{ Put X1 into AX								}
		CMP		AX,X2			{ Is X1=X2 or X1>X2 ?							}
		JE		@VERT_LINE		{ X1=X2 --> @VERT_LINE							}
		JA		@EXIT			{ X1>X2 --> @EXIT								}
		MOV		AX,Y1			{ Put Y1 into AX								}
		CMP		AX,Y2			{ Is Y1=Y2 or Y1>Y2 ?							}
		JE		@HORZ_LINE		{ Y1=Y2 --> @HORZ_LINE							}
		JA		@EXIT			{ Y1>Y2 --> @EXIT								}
		MOV		AX,$0140		{ Set AX to $0140 (Width of screen in pixels)	}
		MUL		Y1				{ Multiply AX width Y1 to get offset1			}
		ADD		AX,X1			{ Add X1 to AX to get offset2					}
		MOV		DI,AX			{ Put offset of first pixel into DI				}
		MOV		BX,X2			{ Put X2 into BX								}
		SUB		BX,X1			{ Subtract X1 from BX to gain Width-1			}
		INC		BX				{ Increase to get real width of line			}
		MOV		DX,$0140		{ Set DX to $0140 (Width of screen in pixels	}
		SUB		DX,BX			{ Set DX to difference between end-start lines	}
		MOV		CX,Y2			{ Put Y2 into CX								}
		SUB		CX,Y1			{ Subtract Y1 from CX to gain Height-1			}
		INC		CX				{ Increase to get real height of line			}
		CLD						{ Set direction to +							}
		MOV		AL,Color		{ Put color of pixel into AL					}
	@LOOP_Y:
		PUSH	CX				{ Save CX on stack, so CX can be used for width	}
		MOV		CX,BX			{ Fetch width of block into CX					}
		REP		STOSB			{ Store one line on screen						}
		ADD		DI,DX			{ Move to next line								}
		POP		CX				{ Fetch CX from stack							}
		LOOP	@LOOP_Y			{ If more lines, go next --> @LOOP_Y			}
		JMP		@EXIT			{ Exit											}
	@VERT_LINE:
		MOV		AX,$0140		{ Set AX to $0140 (Width of screen in pixels)	}
		MUL		Y1				{ Multiply AX width Y1 to get offset1			}
		ADD		AX,X1			{ Add X1 to AX to get offset2					}
		MOV		DI,AX			{ Put offset of first pixel into DI				}
		MOV		CX,Y2			{ Put Y2 into CX								}
		SUB		CX,Y1			{ Subtract Y1 from CX to gain Height-1			}
		INC		CX				{ Increase to get real height of line			}
		MOV		AL,Color		{ Put color of pixel into AL					}
	@DO_VERT:
		MOV		ES:[DI],AL		{ Store pixel into screen						}
		ADD		DI,$0140		{ Move offset down to next line					}
		LOOP	@DO_VERT		{ If more pixels, do it again --> @DO_VERT		}
		JMP		@EXIT			{ Exit											}
	@HORZ_LINE:
		MOV		AX,$0140		{ Set AX to $0140 (Width of screen in pixels)	}
		MUL		Y1				{ Multiply AX width Y1 to get offset1			}
		ADD		AX,X1			{ Add X1 to AX to get offset2					}
		MOV		DI,AX			{ Put offset of first pixel into DI				}
		MOV		CX,X2			{ Put X2 into CX								}
		SUB		CX,X1			{ Subtract X1 from CX to gain Width-1			}
		INC		CX				{ Increase to get real width of line			}
		MOV		AL,Color		{ Put color of pixel into AL					}
		CLD						{ Set direction to +							}
	@DO_HORZ:
		REP		STOSB			{ Store pixels into screen						}
		JMP		@EXIT			{ Exit											}
	@EXIT:
		POP		ES				{ Restore ES									}
	End;

	Procedure XXOrRectangle(X1,Y1,X2,Y2 : Word; Color : Byte); Assembler;
	Asm
		PUSH	ES				{ Save ES										}
		MOV		ES,ScrSeg					{ Point ES:0 to screen				}
		CMP		X1,$0140		{ Is X1 inside screen ?							}
		JAE		@EXIT			{ No --> @EXIT									}
		CMP		X2,$0140		{ Is X2 inside screen ?							}
		JB		@OK1			{ Yes --> @OK1									}
		MOV		X2,$013F		{ Move X2 to edge of screen						}
	@OK1:
		CMP		Y1,$00C8		{ Is Y1 inside screen ?							}
		JAE		@EXIT			{ No --> @EXIT									}
		CMP		Y2,$00C8		{ Is Y2 inside screen ?							}
		JB		@OK2			{ Yes --> @OK2									}
		MOV		Y2,$00C7		{ Move Y2 to edge of screen						}
	@OK2:
		MOV		AX,X1			{ Put X1 into AX								}
		CMP		AX,X2			{ Is X1=X2 or X1>X2 ?							}
		JA		@EXIT			{ X1>X2 --> @EXIT								}
		MOV		AX,Y1			{ Put Y1 into AX								}
		CMP		AX,Y2			{ Is Y1=Y2 or Y1>Y2 ?							}
		JA		@EXIT			{ Y1>Y2 --> @EXIT								}
		MOV		AX,$0140		{ Set AX to $0140 (Width of screen in pixels)	}
		MUL		Y1				{ Multiply AX width Y1 to get offset1			}
		ADD		AX,X1			{ Add X1 to AX to get offset2					}
		MOV		DI,AX			{ Put offset of first pixel into DI				}
		MOV		BX,X2			{ Put X2 into BX								}
		SUB		BX,X1			{ Subtract X1 from BX to gain Width-1			}
		INC		BX				{ Increase to get real width of line			}
		MOV		DX,$0140		{ Set DX to $0140 (Width of screen in pixels	}
		SUB		DX,BX			{ Set DX to difference between end-start lines	}
		MOV		CX,Y2			{ Put Y2 into CX								}
		SUB		CX,Y1			{ Subtract Y1 from CX to gain Height-1			}
		INC		CX				{ Increase to get real height of line			}
		CLD						{ Set direction to +							}
		MOV		AH,Color		{ Put color of pixel into AL					}
	@LOOP_Y:
		PUSH	CX				{ Save CX on stack, so CX can be used for width	}
		MOV		CX,BX			{ Fetch width of block into CX					}
	@LOOP_X:
		MOV		AL,ES:[DI]
		XOR		AL,AH
		STOSB
		LOOP	@LOOP_X
		ADD		DI,DX			{ Move to next line								}
		POP		CX				{ Fetch CX from stack							}
		LOOP	@LOOP_Y			{ If more lines, go next --> @LOOP_Y			}
	@EXIT:
		POP		ES				{ Restore ES									}
	End;

	Procedure XTranslateRectangle(X1,Y1,X2,Y2 : Word; Color : Byte); Assembler;
	Asm
		PUSH	ES				{ Save ES										}
		MOV		ES,ScrSeg					{ Point ES:0 to screen				}
		CMP		X1,$0140		{ Is X1 inside screen ?							}
		JAE		@EXIT			{ No --> @EXIT									}
		CMP		X2,$0140		{ Is X2 inside screen ?							}
		JB		@OK1			{ Yes --> @OK1									}
		MOV		X2,$013F		{ Move X2 to edge of screen						}
	@OK1:
		CMP		Y1,$00C8		{ Is Y1 inside screen ?							}
		JAE		@EXIT			{ No --> @EXIT									}
		CMP		Y2,$00C8		{ Is Y2 inside screen ?							}
		JB		@OK2			{ Yes --> @OK2									}
		MOV		Y2,$00C7		{ Move Y2 to edge of screen						}
	@OK2:
		MOV		AX,X1			{ Put X1 into AX								}
		CMP		AX,X2			{ Is X1=X2 or X1>X2 ?							}
		JA		@EXIT			{ X1>X2 --> @EXIT								}
		MOV		AX,Y1			{ Put Y1 into AX								}
		CMP		AX,Y2			{ Is Y1=Y2 or Y1>Y2 ?							}
		JA		@EXIT			{ Y1>Y2 --> @EXIT								}
		MOV		AX,$0140		{ Set AX to $0140 (Width of screen in pixels)	}
		MUL		Y1				{ Multiply AX width Y1 to get offset1			}
		ADD		AX,X1			{ Add X1 to AX to get offset2					}
		MOV		DI,AX			{ Put offset of first pixel into DI				}
		MOV		BX,X2			{ Put X2 into BX								}
		SUB		BX,X1			{ Subtract X1 from BX to gain Width-1			}
		INC		BX				{ Increase to get real width of line			}
		MOV		DX,$0140		{ Set DX to $0140 (Width of screen in pixels	}
		SUB		DX,BX			{ Set DX to difference between end-start lines	}
		MOV		CX,Y2			{ Put Y2 into CX								}
		SUB		CX,Y1			{ Subtract Y1 from CX to gain Height-1			}
		INC		CX				{ Increase to get real height of line			}
		CLD						{ Set direction to +							}
		MOV		AH,Color		{ Put color of pixel into AL					}
	@LOOP_Y:
		PUSH	CX				{ Save CX on stack, so CX can be used for width	}
		MOV		CX,BX			{ Fetch width of block into CX					}
	@LOOP_X:
		PUSH	DS
		PUSH	BX
		MOV		AL,ES:[DI]
		LDS		BX,DWORD PTR TransPtr
		XLAT
		POP		BX
		POP		DS
		STOSB
		LOOP	@LOOP_X
		ADD		DI,DX			{ Move to next line								}
		POP		CX				{ Fetch CX from stack							}
		LOOP	@LOOP_Y			{ If more lines, go next --> @LOOP_Y			}
		JMP		@EXIT			{ Exit											}
	@EXIT:
		POP		ES				{ Restore ES									}
	End;

	Procedure XAddRectangle(X1,Y1,X2,Y2 : Word; Color : Byte); Assembler;
	Asm
		PUSH	ES				{ Save ES										}
		MOV		ES,ScrSeg					{ Point ES:0 to screen				}
		CMP		X1,$0140		{ Is X1 inside screen ?							}
		JAE		@EXIT			{ No --> @EXIT									}
		CMP		X2,$0140		{ Is X2 inside screen ?							}
		JB		@OK1			{ Yes --> @OK1									}
		MOV		X2,$013F		{ Move X2 to edge of screen						}
	@OK1:
		CMP		Y1,$00C8		{ Is Y1 inside screen ?							}
		JAE		@EXIT			{ No --> @EXIT									}
		CMP		Y2,$00C8		{ Is Y2 inside screen ?							}
		JB		@OK2			{ Yes --> @OK2									}
		MOV		Y2,$00C7		{ Move Y2 to edge of screen						}
	@OK2:
		MOV		AX,X1			{ Put X1 into AX								}
		CMP		AX,X2			{ Is X1=X2 or X1>X2 ?							}
		JA		@EXIT			{ X1>X2 --> @EXIT								}
		MOV		AX,Y1			{ Put Y1 into AX								}
		CMP		AX,Y2			{ Is Y1=Y2 or Y1>Y2 ?							}
		JA		@EXIT			{ Y1>Y2 --> @EXIT								}
		MOV		AX,$0140		{ Set AX to $0140 (Width of screen in pixels)	}
		MUL		Y1				{ Multiply AX width Y1 to get offset1			}
		ADD		AX,X1			{ Add X1 to AX to get offset2					}
		MOV		DI,AX			{ Put offset of first pixel into DI				}
		MOV		BX,X2			{ Put X2 into BX								}
		SUB		BX,X1			{ Subtract X1 from BX to gain Width-1			}
		INC		BX				{ Increase to get real width of line			}
		MOV		DX,$0140		{ Set DX to $0140 (Width of screen in pixels	}
		SUB		DX,BX			{ Set DX to difference between end-start lines	}
		MOV		CX,Y2			{ Put Y2 into CX								}
		SUB		CX,Y1			{ Subtract Y1 from CX to gain Height-1			}
		INC		CX				{ Increase to get real height of line			}
		CLD						{ Set direction to +							}
		MOV		AH,Color		{ Put color of pixel into AL					}
	@LOOP_Y:
		PUSH	CX				{ Save CX on stack, so CX can be used for width	}
		MOV		CX,BX			{ Fetch width of block into CX					}
	@LOOP_X:
		MOV		AL,ES:[DI]
		ADD		AL,AH
		STOSB
		LOOP	@LOOP_X
		ADD		DI,DX			{ Move to next line								}
		POP		CX				{ Fetch CX from stack							}
		LOOP	@LOOP_Y			{ If more lines, go next --> @LOOP_Y			}
	@EXIT:
		POP		ES				{ Restore ES									}
	End;

	Procedure XIncRectangle(X1,Y1,X2,Y2 : Word; Color : Byte); Assembler;
	Asm
		PUSH	ES				{ Save ES										}
		MOV		ES,ScrSeg					{ Point ES:0 to screen				}
		CMP		X1,$0140		{ Is X1 inside screen ?							}
		JAE		@EXIT			{ No --> @EXIT									}
		CMP		X2,$0140		{ Is X2 inside screen ?							}
		JB		@OK1			{ Yes --> @OK1									}
		MOV		X2,$013F		{ Move X2 to edge of screen						}
	@OK1:
		CMP		Y1,$00C8		{ Is Y1 inside screen ?							}
		JAE		@EXIT			{ No --> @EXIT									}
		CMP		Y2,$00C8		{ Is Y2 inside screen ?							}
		JB		@OK2			{ Yes --> @OK2									}
		MOV		Y2,$00C7		{ Move Y2 to edge of screen						}
	@OK2:
		MOV		AX,X1			{ Put X1 into AX								}
		CMP		AX,X2			{ Is X1=X2 or X1>X2 ?							}
		JA		@EXIT			{ X1>X2 --> @EXIT								}
		MOV		AX,Y1			{ Put Y1 into AX								}
		CMP		AX,Y2			{ Is Y1=Y2 or Y1>Y2 ?							}
		JA		@EXIT			{ Y1>Y2 --> @EXIT								}
		MOV		AX,$0140		{ Set AX to $0140 (Width of screen in pixels)	}
		MUL		Y1				{ Multiply AX width Y1 to get offset1			}
		ADD		AX,X1			{ Add X1 to AX to get offset2					}
		MOV		DI,AX			{ Put offset of first pixel into DI				}
		MOV		BX,X2			{ Put X2 into BX								}
		SUB		BX,X1			{ Subtract X1 from BX to gain Width-1			}
		INC		BX				{ Increase to get real width of line			}
		MOV		DX,$0140		{ Set DX to $0140 (Width of screen in pixels	}
		SUB		DX,BX			{ Set DX to difference between end-start lines	}
		MOV		CX,Y2			{ Put Y2 into CX								}
		SUB		CX,Y1			{ Subtract Y1 from CX to gain Height-1			}
		INC		CX				{ Increase to get real height of line			}
		CLD						{ Set direction to +							}
	@LOOP_Y:
		PUSH	CX				{ Save CX on stack, so CX can be used for width	}
		MOV		CX,BX			{ Fetch width of block into CX					}
	@LOOP_X:
		MOV		AL,ES:[DI]
		INC		AL
		STOSB
		LOOP	@LOOP_X
		ADD		DI,DX			{ Move to next line								}
		POP		CX				{ Fetch CX from stack							}
		LOOP	@LOOP_Y			{ If more lines, go next --> @LOOP_Y			}
	@EXIT:
		POP		ES				{ Restore ES									}
	End;

	Procedure XDecRectangle(X1,Y1,X2,Y2 : Word; Color : Byte); Assembler;
	Asm
		PUSH	ES				{ Save ES										}
		MOV		ES,ScrSeg					{ Point ES:0 to screen				}
		CMP		X1,$0140		{ Is X1 inside screen ?							}
		JAE		@EXIT			{ No --> @EXIT									}
		CMP		X2,$0140		{ Is X2 inside screen ?							}
		JB		@OK1			{ Yes --> @OK1									}
		MOV		X2,$013F		{ Move X2 to edge of screen						}
	@OK1:
		CMP		Y1,$00C8		{ Is Y1 inside screen ?							}
		JAE		@EXIT			{ No --> @EXIT									}
		CMP		Y2,$00C8		{ Is Y2 inside screen ?							}
		JB		@OK2			{ Yes --> @OK2									}
		MOV		Y2,$00C7		{ Move Y2 to edge of screen						}
	@OK2:
		MOV		AX,X1			{ Put X1 into AX								}
		CMP		AX,X2			{ Is X1=X2 or X1>X2 ?							}
		JA		@EXIT			{ X1>X2 --> @EXIT								}
		MOV		AX,Y1			{ Put Y1 into AX								}
		CMP		AX,Y2			{ Is Y1=Y2 or Y1>Y2 ?							}
		JA		@EXIT			{ Y1>Y2 --> @EXIT								}
		MOV		AX,$0140		{ Set AX to $0140 (Width of screen in pixels)	}
		MUL		Y1				{ Multiply AX width Y1 to get offset1			}
		ADD		AX,X1			{ Add X1 to AX to get offset2					}
		MOV		DI,AX			{ Put offset of first pixel into DI				}
		MOV		BX,X2			{ Put X2 into BX								}
		SUB		BX,X1			{ Subtract X1 from BX to gain Width-1			}
		INC		BX				{ Increase to get real width of line			}
		MOV		DX,$0140		{ Set DX to $0140 (Width of screen in pixels	}
		SUB		DX,BX			{ Set DX to difference between end-start lines	}
		MOV		CX,Y2			{ Put Y2 into CX								}
		SUB		CX,Y1			{ Subtract Y1 from CX to gain Height-1			}
		INC		CX				{ Increase to get real height of line			}
		CLD						{ Set direction to +							}
	@LOOP_Y:
		PUSH	CX				{ Save CX on stack, so CX can be used for width	}
		MOV		CX,BX			{ Fetch width of block into CX					}
	@LOOP_X:
		MOV		AL,ES:[DI]
		DEC		AL
		STOSB
		LOOP	@LOOP_X
		ADD		DI,DX			{ Move to next line								}
		POP		CX				{ Fetch CX from stack							}
		LOOP	@LOOP_Y			{ If more lines, go next --> @LOOP_Y			}
	@EXIT:
		POP		ES				{ Restore ES									}
	End;

	Procedure XSubRectangle(X1,Y1,X2,Y2 : Word; Color : Byte); Assembler;
	Asm
		PUSH	ES				{ Save ES										}
		MOV		ES,ScrSeg					{ Point ES:0 to screen				}
		CMP		X1,$0140		{ Is X1 inside screen ?							}
		JAE		@EXIT			{ No --> @EXIT									}
		CMP		X2,$0140		{ Is X2 inside screen ?							}
		JB		@OK1			{ Yes --> @OK1									}
		MOV		X2,$013F		{ Move X2 to edge of screen						}
	@OK1:
		CMP		Y1,$00C8		{ Is Y1 inside screen ?							}
		JAE		@EXIT			{ No --> @EXIT									}
		CMP		Y2,$00C8		{ Is Y2 inside screen ?							}
		JB		@OK2			{ Yes --> @OK2									}
		MOV		Y2,$00C7		{ Move Y2 to edge of screen						}
	@OK2:
		MOV		AX,X1			{ Put X1 into AX								}
		CMP		AX,X2			{ Is X1=X2 or X1>X2 ?							}
		JA		@EXIT			{ X1>X2 --> @EXIT								}
		MOV		AX,Y1			{ Put Y1 into AX								}
		CMP		AX,Y2			{ Is Y1=Y2 or Y1>Y2 ?							}
		JA		@EXIT			{ Y1>Y2 --> @EXIT								}
		MOV		AX,$0140		{ Set AX to $0140 (Width of screen in pixels)	}
		MUL		Y1				{ Multiply AX width Y1 to get offset1			}
		ADD		AX,X1			{ Add X1 to AX to get offset2					}
		MOV		DI,AX			{ Put offset of first pixel into DI				}
		MOV		BX,X2			{ Put X2 into BX								}
		SUB		BX,X1			{ Subtract X1 from BX to gain Width-1			}
		INC		BX				{ Increase to get real width of line			}
		MOV		DX,$0140		{ Set DX to $0140 (Width of screen in pixels	}
		SUB		DX,BX			{ Set DX to difference between end-start lines	}
		MOV		CX,Y2			{ Put Y2 into CX								}
		SUB		CX,Y1			{ Subtract Y1 from CX to gain Height-1			}
		INC		CX				{ Increase to get real height of line			}
		CLD						{ Set direction to +							}
		MOV		AH,Color		{ Put color of pixel into AL					}
	@LOOP_Y:
		PUSH	CX				{ Save CX on stack, so CX can be used for width	}
		MOV		CX,BX			{ Fetch width of block into CX					}
	@LOOP_X:
		MOV		AL,ES:[DI]
		SUB		AL,AH
		STOSB
		LOOP	@LOOP_X
		ADD		DI,DX			{ Move to next line								}
		POP		CX				{ Fetch CX from stack							}
		LOOP	@LOOP_Y			{ If more lines, go next --> @LOOP_Y			}
	@EXIT:
		POP		ES				{ Restore ES									}
	End;

	Procedure XOrRectangle(X1,Y1,X2,Y2 : Word; Color : Byte); Assembler;
	Asm
		PUSH	ES				{ Save ES										}
		MOV		ES,ScrSeg					{ Point ES:0 to screen				}
		CMP		X1,$0140		{ Is X1 inside screen ?							}
		JAE		@EXIT			{ No --> @EXIT									}
		CMP		X2,$0140		{ Is X2 inside screen ?							}
		JB		@OK1			{ Yes --> @OK1									}
		MOV		X2,$013F		{ Move X2 to edge of screen						}
	@OK1:
		CMP		Y1,$00C8		{ Is Y1 inside screen ?							}
		JAE		@EXIT			{ No --> @EXIT									}
		CMP		Y2,$00C8		{ Is Y2 inside screen ?							}
		JB		@OK2			{ Yes --> @OK2									}
		MOV		Y2,$00C7		{ Move Y2 to edge of screen						}
	@OK2:
		MOV		AX,X1			{ Put X1 into AX								}
		CMP		AX,X2			{ Is X1=X2 or X1>X2 ?							}
		JA		@EXIT			{ X1>X2 --> @EXIT								}
		MOV		AX,Y1			{ Put Y1 into AX								}
		CMP		AX,Y2			{ Is Y1=Y2 or Y1>Y2 ?							}
		JA		@EXIT			{ Y1>Y2 --> @EXIT								}
		MOV		AX,$0140		{ Set AX to $0140 (Width of screen in pixels)	}
		MUL		Y1				{ Multiply AX width Y1 to get offset1			}
		ADD		AX,X1			{ Add X1 to AX to get offset2					}
		MOV		DI,AX			{ Put offset of first pixel into DI				}
		MOV		BX,X2			{ Put X2 into BX								}
		SUB		BX,X1			{ Subtract X1 from BX to gain Width-1			}
		INC		BX				{ Increase to get real width of line			}
		MOV		DX,$0140		{ Set DX to $0140 (Width of screen in pixels	}
		SUB		DX,BX			{ Set DX to difference between end-start lines	}
		MOV		CX,Y2			{ Put Y2 into CX								}
		SUB		CX,Y1			{ Subtract Y1 from CX to gain Height-1			}
		INC		CX				{ Increase to get real height of line			}
		CLD						{ Set direction to +							}
		MOV		AH,Color		{ Put color of pixel into AL					}
	@LOOP_Y:
		PUSH	CX				{ Save CX on stack, so CX can be used for width	}
		MOV		CX,BX			{ Fetch width of block into CX					}
	@LOOP_X:
		MOV		AL,ES:[DI]
		OR		AL,AH
		STOSB
		LOOP	@LOOP_X
		ADD		DI,DX			{ Move to next line								}
		POP		CX				{ Fetch CX from stack							}
		LOOP	@LOOP_Y			{ If more lines, go next --> @LOOP_Y			}
	@EXIT:
		POP		ES				{ Restore ES									}
	End;

	Procedure XAndRectangle(X1,Y1,X2,Y2 : Word; Color : Byte); Assembler;
	Asm
		PUSH	ES				{ Save ES										}
		MOV		ES,ScrSeg					{ Point ES:0 to screen				}
		CMP		X1,$0140		{ Is X1 inside screen ?							}
		JAE		@EXIT			{ No --> @EXIT									}
		CMP		X2,$0140		{ Is X2 inside screen ?							}
		JB		@OK1			{ Yes --> @OK1									}
		MOV		X2,$013F		{ Move X2 to edge of screen						}
	@OK1:
		CMP		Y1,$00C8		{ Is Y1 inside screen ?							}
		JAE		@EXIT			{ No --> @EXIT									}
		CMP		Y2,$00C8		{ Is Y2 inside screen ?							}
		JB		@OK2			{ Yes --> @OK2									}
		MOV		Y2,$00C7		{ Move Y2 to edge of screen						}
	@OK2:
		MOV		AX,X1			{ Put X1 into AX								}
		CMP		AX,X2			{ Is X1=X2 or X1>X2 ?							}
		JA		@EXIT			{ X1>X2 --> @EXIT								}
		MOV		AX,Y1			{ Put Y1 into AX								}
		CMP		AX,Y2			{ Is Y1=Y2 or Y1>Y2 ?							}
		JA		@EXIT			{ Y1>Y2 --> @EXIT								}
		MOV		AX,$0140		{ Set AX to $0140 (Width of screen in pixels)	}
		MUL		Y1				{ Multiply AX width Y1 to get offset1			}
		ADD		AX,X1			{ Add X1 to AX to get offset2					}
		MOV		DI,AX			{ Put offset of first pixel into DI				}
		MOV		BX,X2			{ Put X2 into BX								}
		SUB		BX,X1			{ Subtract X1 from BX to gain Width-1			}
		INC		BX				{ Increase to get real width of line			}
		MOV		DX,$0140		{ Set DX to $0140 (Width of screen in pixels	}
		SUB		DX,BX			{ Set DX to difference between end-start lines	}
		MOV		CX,Y2			{ Put Y2 into CX								}
		SUB		CX,Y1			{ Subtract Y1 from CX to gain Height-1			}
		INC		CX				{ Increase to get real height of line			}
		CLD						{ Set direction to +							}
		MOV		AH,Color		{ Put color of pixel into AL					}
	@LOOP_Y:
		PUSH	CX				{ Save CX on stack, so CX can be used for width	}
		MOV		CX,BX			{ Fetch width of block into CX					}
	@LOOP_X:
		MOV		AL,ES:[DI]
		AND		AL,AH
		STOSB
		LOOP	@LOOP_X
		ADD		DI,DX			{ Move to next line								}
		POP		CX				{ Fetch CX from stack							}
		LOOP	@LOOP_Y			{ If more lines, go next --> @LOOP_Y			}
	@EXIT:
		POP		ES				{ Restore ES									}
	End;

	Procedure XApproachRectangle(X1,Y1,X2,Y2 : Word; Color : Byte); Assembler;
	Asm
		PUSH	ES				{ Save ES										}
		MOV		ES,ScrSeg					{ Point ES:0 to screen				}
		CMP		X1,$0140		{ Is X1 inside screen ?							}
		JAE		@EXIT			{ No --> @EXIT									}
		CMP		X2,$0140		{ Is X2 inside screen ?							}
		JB		@OK1			{ Yes --> @OK1									}
		MOV		X2,$013F		{ Move X2 to edge of screen						}
	@OK1:
		CMP		Y1,$00C8		{ Is Y1 inside screen ?							}
		JAE		@EXIT			{ No --> @EXIT									}
		CMP		Y2,$00C8		{ Is Y2 inside screen ?							}
		JB		@OK2			{ Yes --> @OK2									}
		MOV		Y2,$00C7		{ Move Y2 to edge of screen						}
	@OK2:
		MOV		AX,X1			{ Put X1 into AX								}
		CMP		AX,X2			{ Is X1=X2 or X1>X2 ?							}
		JA		@EXIT			{ X1>X2 --> @EXIT								}
		MOV		AX,Y1			{ Put Y1 into AX								}
		CMP		AX,Y2			{ Is Y1=Y2 or Y1>Y2 ?							}
		JA		@EXIT			{ Y1>Y2 --> @EXIT								}
		MOV		AX,$0140		{ Set AX to $0140 (Width of screen in pixels)	}
		MUL		Y1				{ Multiply AX width Y1 to get offset1			}
		ADD		AX,X1			{ Add X1 to AX to get offset2					}
		MOV		DI,AX			{ Put offset of first pixel into DI				}
		MOV		BX,X2			{ Put X2 into BX								}
		SUB		BX,X1			{ Subtract X1 from BX to gain Width-1			}
		INC		BX				{ Increase to get real width of line			}
		MOV		DX,$0140		{ Set DX to $0140 (Width of screen in pixels	}
		SUB		DX,BX			{ Set DX to difference between end-start lines	}
		MOV		CX,Y2			{ Put Y2 into CX								}
		SUB		CX,Y1			{ Subtract Y1 from CX to gain Height-1			}
		INC		CX				{ Increase to get real height of line			}
		CLD						{ Set direction to +							}
		MOV		AH,Color		{ Put color of pixel into AL					}
	@LOOP_Y:
		PUSH	CX				{ Save CX on stack, so CX can be used for width	}
		MOV		CX,BX			{ Fetch width of block into CX					}
	@LOOP_X:
		MOV		AL,ES:[DI]
		CMP		AL,AH
		JB		@UP
		JA		@DOWN
		JMP		@NEXT
	@UP:
		INC		AL
		JMP		@NEXT
	@DOWN:
		DEC		AL
	@NEXT:
		STOSB
		LOOP	@LOOP_X
		ADD		DI,DX			{ Move to next line								}
		POP		CX				{ Fetch CX from stack							}
		LOOP	@LOOP_Y			{ If more lines, go next --> @LOOP_Y			}
	@EXIT:
		POP		ES				{ Restore ES									}
	End;

	Procedure XOnlyRectangle(X1,Y1,X2,Y2 : Word; Color : Byte); Assembler;
	Asm
		PUSH	ES				{ Save ES										}
		MOV		ES,ScrSeg					{ Point ES:0 to screen				}
		CMP		X1,$0140		{ Is X1 inside screen ?							}
		JAE		@EXIT			{ No --> @EXIT									}
		CMP		X2,$0140		{ Is X2 inside screen ?							}
		JB		@OK1			{ Yes --> @OK1									}
		MOV		X2,$013F		{ Move X2 to edge of screen						}
	@OK1:
		CMP		Y1,$00C8		{ Is Y1 inside screen ?							}
		JAE		@EXIT			{ No --> @EXIT									}
		CMP		Y2,$00C8		{ Is Y2 inside screen ?							}
		JB		@OK2			{ Yes --> @OK2									}
		MOV		Y2,$00C7		{ Move Y2 to edge of screen						}
	@OK2:
		MOV		AX,X1			{ Put X1 into AX								}
		CMP		AX,X2			{ Is X1=X2 or X1>X2 ?							}
		JA		@EXIT			{ X1>X2 --> @EXIT								}
		MOV		AX,Y1			{ Put Y1 into AX								}
		CMP		AX,Y2			{ Is Y1=Y2 or Y1>Y2 ?							}
		JA		@EXIT			{ Y1>Y2 --> @EXIT								}
		MOV		AX,$0140		{ Set AX to $0140 (Width of screen in pixels)	}
		MUL		Y1				{ Multiply AX width Y1 to get offset1			}
		ADD		AX,X1			{ Add X1 to AX to get offset2					}
		MOV		DI,AX			{ Put offset of first pixel into DI				}
		MOV		BX,X2			{ Put X2 into BX								}
		SUB		BX,X1			{ Subtract X1 from BX to gain Width-1			}
		INC		BX				{ Increase to get real width of line			}
		MOV		DX,$0140		{ Set DX to $0140 (Width of screen in pixels	}
		SUB		DX,BX			{ Set DX to difference between end-start lines	}
		MOV		CX,Y2			{ Put Y2 into CX								}
		SUB		CX,Y1			{ Subtract Y1 from CX to gain Height-1			}
		INC		CX				{ Increase to get real height of line			}
		CLD						{ Set direction to +							}
		MOV		AH,Color		{ Put color of pixel into AL					}
	@LOOP_Y:
		PUSH	CX				{ Save CX on stack, so CX can be used for width	}
		MOV		CX,BX			{ Fetch width of block into CX					}
	@LOOP_X:
		MOV		AL,ES:[DI]
		CMP		AL,MaskColor
		JNE		@NEXT
		MOV		AL,AH
	@NEXT:
		STOSB
		LOOP	@LOOP_X
		ADD		DI,DX			{ Move to next line								}
		POP		CX				{ Fetch CX from stack							}
		LOOP	@LOOP_Y			{ If more lines, go next --> @LOOP_Y			}
	@EXIT:
		POP		ES				{ Restore ES									}
	End;

Asm
	PUSH	X1
	PUSH	Y1
	PUSH	X2
	PUSH	Y2
	MOV		AL,Color
	PUSH	AX
	PUSH	WORD PTR [BP]
	CMP		PutFn,pNORMAL
	JNE		@01
	CALL	XNormalRectangle
	JMP		@EXIT
@01:
	CMP		PutFn,pTRANSPARENT
	JNE		@1
	MOV		AH,AL
	CMP		AL,MaskColor
	JNE		@EXIT
	MOV		AH,ES:[DI]
	JMP		@EXIT
@1:
	CMP		PutFn,pXOR
	JNE		@2
	CALL	XXOrRectangle
	JMP		@EXIT
@2:
	CMP		PutFn,pADD
	JNE		@3
	CALL	XAddRectangle
	JMP		@EXIT
@3:
	CMP		PutFn,pSUB
	JNE		@4
	CALL	XSubRectangle
	JMP		@EXIT
@4:
	CMP		PutFn,pOR
	JNE		@5
	CALL	XOrRectangle
	JMP		@EXIT
@5:
	CMP		PutFn,pAND
	JNE		@6
	CALL	XAndRectangle
	JMP		@EXIT
@6:
	CMP		PutFn,pINC
	JNE		@6B
	CALL	XIncRectangle
	JMP		@EXIT
@6B:
	CMP		PutFn,pDEC
	JNE		@6C
	CALL	XDecRectangle
	JMP		@EXIT
@6C:
	CMP		PutFn,pTRANSLATE
	JNE		@7
	CALL	XTranslateRectangle
	JMP		@EXIT
@7:
	CMP		PutFn,pAPPROACH
	JNE		@8
	CALL	XApproachRectangle
	JMP		@EXIT
@8:
	CMP		PutFn,pONLY
	JNE		@EXIT
	CALL	XOnlyRectangle
	JMP		@EXIT
@EXIT:
End;

Procedure XVLine(X,Y1,Y2 : Word; Color : Byte); Assembler;
Asm
	PUSH	ES				{ Save ES										}
	MOV		ES,ScrSeg		{ Point ES:0 to screen				}
	MOV		AX,Y1			{ Put Y1 into AX								}
	CMP		AX,Y2			{ Is X1>X2 ?									}
	JA		@EXIT			{ Yes --> @EXIT									}
	MOV		AX,$0140		{ Set AX to $0140 (Width of screen in pixels)	}
	MUL		Y1				{ Multiply AX width Y1 to get offset1			}
	ADD		AX,X			{ Add X1 to AX to get offset2					}
	MOV		DI,AX			{ Put offset of first pixel into DI				}
	CMP		Y1,$00C8		{ Is Y1 inside screen ?							}
	JAE		@EXIT			{ No --> @EXIT									}
	CMP		Y2,$00C8		{ Is Y2 inside screen ?							}
	JB		@OK1			{ Yes --> @OK1									}
	MOV		Y2,$00C7		{ Move Y2 to edge of screen						}
@OK1:
	MOV		CX,Y2			{ Put Y2 into CX								}
	SUB		CX,Y1			{ Subtract Y1 from CX to gain Height-1			}
	INC		CX				{ Increase to get real height of line			}
@DOIT:
	MOV		AL,Color
	CMP		PutFn,pNORMAL
	JNE		@01
	MOV		AH,AL
	JMP		@NEXT
@01:
	CMP		PutFn,pTRANSPARENT
	JNE		@0
	MOV		AH,AL
	CMP		AL,MaskColor
	JNE		@NEXT
	MOV		AH,ES:[DI]
	JMP		@NEXT
@0:
	CMP		PutFn,pXOR
	JNE		@1
	MOV		AH,ES:[DI]
	XOR		AH,AL
	JMP		@NEXT
@1:
	CMP		PutFn,pADD
	JNE		@2
	MOV		AH,ES:[DI]
	ADD		AH,AL
	JMP		@NEXT
@2:
	CMP		PutFn,pSUB
	JNE		@3
	MOV		AH,ES:[DI]
	SUB		AH,AL
	JMP		@NEXT
@3:
	CMP		PutFn,pOR
	JNE		@4
	MOV		AH,ES:[DI]
	OR		AH,AL
	MOV		AL,AH
	JMP		@NEXT
@4:
	CMP		PutFn,pAND
	JNE		@5
	MOV		AH,ES:[DI]
	AND		AH,AL
	JMP		@NEXT
@5:
	CMP		PutFn,pDEC
	JNE		@5B
	DEC		AH
	JMP		@NEXT
@5B:
	CMP		PutFn,pINC
	JNE		@5C
	INC		AH
	JMP		@NEXT
@5C:
	CMP		PutFn,pTRANSLATE
	JNE		@5D
	PUSH	DS
	PUSH	BX
	MOV		AL,ES:[DI]
	LDS		BX,DWORD PTR TransPtr
	XLAT
	MOV		AH,AL
	POP		BX
	POP		DS
	JMP		@NEXT
@5D:
	CMP		PutFn,pAPPROACH
	JNE		@6
	MOV		AH,ES:[DI]
	CMP		AH,AL
	JB		@UP
	JA		@DOWN
	JMP		@NEXT
@UP:
	INC		AH
	JMP		@NEXT
@DOWN:
	DEC		AH
	JMP		@NEXT
@6:
	CMP		PutFn,pONLY
	JNE		@NEXT
	MOV		AH,ES:[DI]
	CMP		AH,MaskColor
	JNE		@NEXT
	MOV		AH,AL
@NEXT:
	MOV		ES:[DI],AH
	ADD		DI,$0140		{ Move offset down to next line					}
	LOOP	@DOIT2			{ If more pixels, do it again --> @DOIT			}
	JMP		@EXIT
@DOIT2:
	JMP		@DOIT
@EXIT:
	POP		ES				{ Restore ES									}
End;

Procedure XHLine(Y,X1,X2 : Word; Color : Byte); Assembler;
Asm
	PUSH	ES				{ Save ES										}
	MOV		ES,ScrSeg		{ Point ES:0 to screen				}
	MOV		AX,X1			{ Put X1 into AX								}
	CMP		AX,X2			{ Is X1>X2 ?									}
	JA		@EXIT			{ Yes --> @EXIT									}
	MOV		AX,$0140		{ Set AX to $0140 (Width of screen in pixels)	}
	MUL		Y				{ Multiply AX width Y1 to get offset1			}
	ADD		AX,X1			{ Add X1 to AX to get offset2					}
	MOV		DI,AX			{ Put offset of first pixel into DI				}
	CMP		X1,$0140		{ Is X1 inside screen ?							}
	JAE		@EXIT			{ No --> @EXIT									}
	CMP		X2,$0140		{ Is X2 inside screen ?							}
	JB		@OK1			{ Yes --> @OK1									}
	MOV		X2,$013F		{ Move X2 to edge of screen						}
@OK1:
	MOV		CX,X2			{ Put X2 into CX								}
	SUB		CX,X1			{ Subtract X1 from CX to gain Width-1			}
	INC		CX				{ Increase to get real width of line			}
	CLD						{ Set direction to +							}
@DOIT:
	MOV		AL,Color
	CMP		PutFn,pNORMAL
	JNE		@01
	REP		STOSB
	JMP		@EXIT
@01:
	CMP		PutFn,pTRANSPARENT
	JNE		@0
	MOV		AH,AL
	CMP		AL,MaskColor
	JNE		@NEXT
	MOV		AH,ES:[DI]
	JMP		@NEXT
@0:
	CMP		PutFn,pXOR
	JNE		@1
	MOV		AH,ES:[DI]
	XOR		AH,AL
	JMP		@NEXT
@1:
	CMP		PutFn,pADD
	JNE		@2
	MOV		AH,ES:[DI]
	ADD		AH,AL
	JMP		@NEXT
@2:
	CMP		PutFn,pSUB
	JNE		@3
	MOV		AH,ES:[DI]
	SUB		AH,AL
	JMP		@NEXT
@3:
	CMP		PutFn,pOR
	JNE		@4
	MOV		AH,ES:[DI]
	OR		AH,AL
	MOV		AL,AH
	JMP		@NEXT
@4:
	CMP		PutFn,pAND
	JNE		@5
	MOV		AH,ES:[DI]
	AND		AH,AL
	JMP		@NEXT
@5:
	CMP		PutFn,pTRANSLATE
	JNE		@5B
	PUSH	DS
	PUSH	BX
	MOV		AL,ES:[DI]
	LDS		BX,DWORD PTR TransPtr
	XLAT
	MOV		AH,AL
	POP		BX
	POP		DS
	JMP		@NEXT
@5B:
	CMP		PutFn,pAPPROACH
	JNE		@6
	MOV		AH,ES:[DI]
	CMP		AH,AL
	JB		@UP
	JA		@DOWN
	JMP		@NEXT
@UP:
	INC		AH
	JMP		@NEXT
@DOWN:
	DEC		AH
	JMP		@NEXT
@6:
	CMP		PutFn,pDEC
	JNE		@6B
	MOV		AH,ES:[DI]
	DEC		AH
	JMP		@NEXT
@6B:
	CMP		PutFn,pINC
	JNE		@6C
	MOV		AH,ES:[DI]
	INC		AH
	JMP		@NEXT
@6C:
	CMP		PutFn,pONLY
	JNE		@NEXT
	MOV		AH,ES:[DI]
	CMP		AH,MaskColor
	JNE		@NEXT
	MOV		AH,AL
@NEXT:
	MOV		AL,AH
	STOSB
	LOOP	@DOIT2
	JMP		@EXIT
@DOIT2:
	JMP		@DOIT
@EXIT:
	POP		ES				{ Restore ES									}
End;

Procedure XMode(No : Byte); Assembler;
Asm
	MOV		AX,$00			{ AH=$00 (Set mode)								}
	MOV		AL,No			{ Put mode-no into AL							}
	INT		$10				{ Call video-interrupt to set mode				}
	CMP		AL,$13			{ Mode $13 ?									}
	JNE		@EXIT			{ No --> @EXIT									}
	PUSH	WORD($0000)		{ Zero all colors								}
	PUSH	WORD($0100)		{ ...											}
	CALL	XSetRGBBlack		{ ...											}
@EXIT:
End;

Procedure XSetRGB(Index,R,G,B : Byte); Assembler;
Asm
	MOV		DX,$03C8		{ DX=$03C8, port $3c8 is set to index			}
	MOV		AL,Index		{ Put Index into AL								}
	OUT		DX,AL			{ Tell VGA which color to set					}
	INC		DX				{ Go to next port ($3C9) to set rgb-values		}
	MOV		AL,R			{ Tell VGA the Red-value of the color			}
	OUT		DX,AL			{ ...											}
	MOV		AL,G			{ Tell VGA the Green-value of the color			}
	OUT		DX,AL			{ ...											}
	MOV		AL,B			{ Tell VGA the Blue-value of the color			}
	OUT		DX,AL			{ ...											}
End;

Procedure XSetRGBMany(StartIndex : Byte; ColorCount : Word; Var RGBData); Assembler;
Asm
	PUSH	DS
	LDS		SI,RGBData
	MOV		DX,$03C8
	MOV		AL,StartIndex
	OUT		DX,AL
	MOV		AX,ColorCount
	MOV		BX,3
	MUL		BX
	MOV		CX,AX
	CLD
	MOV		DX,$03C9
@DOIT:
	LODSB
	OUT		DX,AL
	LOOP	@DOIT
	POP		DS
End;

Procedure XScrollLeft(X1,Y1,X2,Y2 : Word; FillColor : Byte); Assembler;
Asm
	PUSH	ES
	PUSH	DS
	CMP		X1,$0140		{ Is X1 inside screen ?							}
	JAE		@EXIT			{ No --> @EXIT									}
	CMP		X2,$0140		{ Is X2 inside screen ?							}
	JB		@OK1			{ Yes --> @OK1									}
	MOV		X2,$013F		{ Move X2 to edge of screen						}
@OK1:
	CMP		Y1,$00C8		{ Is Y1 inside screen ?							}
	JAE		@EXIT			{ No --> @EXIT									}
	CMP		Y2,$00C8		{ Is Y2 inside screen ?							}
	JB		@OK2			{ Yes --> @OK2									}
	MOV		Y2,$00C7		{ Move Y2 to edge of screen						}
@OK2:
	MOV		AX,X1			{ Put X1 into AX								}
	CMP		AX,X2			{ Is X1=X2 or X1>X2 ?							}
	JA		@EXIT			{ X1>X2 --> @EXIT								}
	MOV		AX,Y1			{ Put Y1 into AX								}
	CMP		AX,Y2			{ Is Y1=Y2 or Y1>Y2 ?							}
	JA		@EXIT			{ Y1>Y2 --> @EXIT								}
	MOV		ES,ScrSeg					{ Point ES:0 to screen				}
	PUSH	ES
	POP		DS
	MOV		AX,$0140		{ Calculate offset into screen					}
	MUL		Y1
	ADD		AX,X1
	MOV		DI,AX
	MOV		SI,DI
	INC		SI
	MOV		CX,Y2			{ Calculate height of XScrolling-area			}
	SUB		CX,Y1
	INC		CX
	MOV		DX,X2			{ Calculate width of XScrolling-area				}
	SUB		DX,X1
	CLD
	MOV		BX,$0140		{ Calculate delta from end-start of lines		}
	SUB		BX,DX
	DEC		BX
@DOIT:
	PUSH	CX				{ Save CX for outer loop						}
	MOV		CX,DX			{ Fetch width of area							}
	REP		MOVSB			{ XScroll one line								}
	MOV		AL,FillColor	{ Blank last pixel								}
	STOSB
	ADD		DI,BX			{ Move on to next line							}
	ADD		SI,BX
	INC		SI
	POP		CX				{ Fetch CX for outer loop						}
	LOOP	@DOIT			{ Loop if more lines							}
@EXIT:
	POP		DS
	POP		ES
End;

Procedure XScrollRight(X1,Y1,X2,Y2 : Word; FillColor : Byte); Assembler;
Asm
	PUSH	ES
	PUSH	DS
	CMP		X1,$0140		{ Is X1 inside screen ?							}
	JAE		@EXIT			{ No --> @EXIT									}
	CMP		X2,$0140		{ Is X2 inside screen ?							}
	JB		@OK1			{ Yes --> @OK1									}
	MOV		X2,$013F		{ Move X2 to edge of screen						}
@OK1:
	CMP		Y1,$00C8		{ Is Y1 inside screen ?							}
	JAE		@EXIT			{ No --> @EXIT									}
	CMP		Y2,$00C8		{ Is Y2 inside screen ?							}
	JB		@OK2			{ Yes --> @OK2									}
	MOV		Y2,$00C7		{ Move Y2 to edge of screen						}
@OK2:
	MOV		AX,X1			{ Put X1 into AX								}
	CMP		AX,X2			{ Is X1=X2 or X1>X2 ?							}
	JA		@EXIT			{ X1>X2 --> @EXIT								}
	MOV		AX,Y1			{ Put Y1 into AX								}
	CMP		AX,Y2			{ Is Y1=Y2 or Y1>Y2 ?							}
	JA		@EXIT			{ Y1>Y2 --> @EXIT								}
	MOV		ES,ScrSeg					{ Point ES:0 to screen				}
	PUSH	ES
	POP		DS
	MOV		AX,$0140		{ Calculate offsets into screen					}
	MUL		Y1
	ADD		AX,X1
	MOV		DI,AX
	MOV		SI,DI
	DEC		SI
	MOV		CX,Y2			{ Calculate height of area						}
	SUB		CX,Y1
	INC		CX
	MOV		DX,X2			{ Calculate width of area						}
	SUB		DX,X1
	MOV		BX,$0140		{ Calculate delta line-to-line					}
	SUB		BX,DX
	DEC		BX
	STD
@DOIT:
	PUSH	CX
	MOV		CX,DX
	ADD		SI,DX
	ADD		DI,DX
	REP		MOVSB
	MOV		AL,FillColor
	STOSB
	ADD		SI,$0140
	ADD		DI,$0140
	INC		DI
	POP		CX
	LOOP	@DOIT
@EXIT:
	POP		DS
	POP		ES
End;

Procedure XScrollUp(X1,Y1,X2,Y2 : Word; FillColor : Byte); Assembler;
Asm
	PUSH	DS
	PUSH	ES
	MOV		ES,ScrSeg					{ Point ES:0 to screen				}
	PUSH	ES
	POP		DS
	MOV		AX,$0140
	MUL		Y1
	ADD		AX,X1
	MOV		DI,AX
	MOV		SI,AX
	ADD		SI,$0140
	MOV		CX,Y2
	SUB		CX,Y1
	MOV		DX,X2
	SUB		DX,X1
	INC		DX
	MOV		BX,$0140
	SUB		BX,DX
	CLD
@DOIT:
	PUSH	CX
	MOV		CX,DX
	REP		MOVSB
	ADD		SI,BX
	ADD		DI,BX
	POP		CX
	LOOP	@DOIT
	MOV		CX,DX
	MOV		AL,FillColor
	REP		STOSB
@EXIT:
	POP		ES
	POP		DS
End;

Procedure XScrollDown(X1,Y1,X2,Y2 : Word; FillColor : Byte); Assembler;
Asm
	PUSH	DS
	PUSH	ES
	MOV		ES,ScrSeg					{ Point ES:0 to screen				}
	PUSH	ES
	POP		DS
	MOV		AX,$0140
	MUL		Y2
	ADD		AX,X2
	MOV		DI,AX
	MOV		SI,AX
	SUB		SI,$0140
	MOV		CX,Y2
	SUB		CX,Y1
	MOV		DX,X2
	SUB		DX,X1
	INC		DX
	MOV		BX,$0140
	SUB		BX,DX
	STD
@DOIT:
	PUSH	CX
	MOV		CX,DX
	REP		MOVSB
	SUB		SI,BX
	SUB		DI,BX
	POP		CX
	LOOP	@DOIT
	MOV		CX,DX
	MOV		AL,FillColor
	REP		STOSB
@EXIT:
	POP		ES
	POP		DS
End;

Procedure XClr(Color : Byte); Assembler;
Asm
	PUSH	ES
	MOV		ES,ScrSeg					{ Point ES:0 to screen				}
	XOR		DI,DI
	MOV		CX,$7D00
	CLD
	MOV		AL,Color
	MOV		AH,Color
	REP		STOSW
	POP		ES
End;

Function XImageSize(X1,Y1,X2,Y2 : Word) : Word; Assembler;
Asm
	MOV		AX,X2			{ Calculate width of area						}
	SUB		AX,X1
	JO		@FAIL
	INC		AX
	MOV		BX,Y2			{ Calculate height of area						}
	SUB		BX,Y1
	JO		@FAIL
	INC		BX
	MUL		BX				{ Calculate area-size in pixels					}
	ADD		AX,$0004		{ Adjust for width/height words					}
	JMP		@EXIT
@FAIL:
	XOR		AX,AX
@EXIT:
End;

Procedure XAllocGetImage(X1,Y1,X2,Y2 : Word; Var Data : Pointer);
Begin
	XAllocImage(X1,Y1,X2,Y2,Data);
	XGetImage(X1,Y1,X2,Y2,Data^);
End;

Procedure XGetImage(X1,Y1,X2,Y2 : Word; Var Data); Assembler;
Asm
	PUSH	ES				{ Save ES										}
	PUSH	DS				{ Save DS										}
	LES		DI,Data			{ Point ES:DI to Data							}
	MOV		DS,ScrSeg		{ Point DS:0 to screen							}
	CMP		X1,$0140		{ Is X1 inside screen ?							}
	JAE		@EXIT			{ No --> @EXIT									}
	CMP		X2,$0140		{ Is X2 inside screen ?							}
	JB		@OK1			{ Yes --> @OK1									}
	MOV		X2,$013F		{ Move X2 to edge of screen						}
@OK1:
	CMP		Y1,$00C8		{ Is Y1 inside screen ?							}
	JAE		@EXIT			{ No --> @EXIT									}
	CMP		Y2,$00C8		{ Is Y2 inside screen ?							}
	JB		@OK2			{ Yes --> @OK2									}
	MOV		Y2,$00C7		{ Move Y2 to edge of screen						}
@OK2:
	MOV		AX,X1			{ Put X1 into AX								}
	CMP		AX,X2			{ Is X1=X2 or X1>X2 ?							}
	JE		@VERT_LINE		{ X1=X2 --> @VERT_LINE							}
	JA		@EXIT			{ X1>X2 --> @EXIT								}
	MOV		AX,Y1			{ Put Y1 into AX								}
	CMP		AX,Y2			{ Is Y1=Y2 or Y1>Y2 ?							}
	JE		@HORZ_LINE		{ Y1=Y2 --> @HORZ_LINE							}
	JA		@EXIT			{ Y1>Y2 --> @EXIT								}
	MOV		AX,$0140		{ Set AX to $0140 (Width of screen in pixels)	}
	MUL		Y1				{ Multiply AX width Y1 to get offset1			}
	ADD		AX,X1			{ Add X1 to AX to get offset2					}
	MOV		SI,AX			{ Put offset of first pixel into SI				}
	MOV		BX,X2			{ Put X2 into BX								}
	SUB		BX,X1			{ Subtract X1 from BX to gain Width-1			}
	INC		BX				{ Increase to get real width of line			}
	MOV		ES:[DI],BX		{ Store width in Data							}
	MOV		DX,$0140		{ Set DX to $0140 (Width of screen in pixels	}
	SUB		DX,BX			{ Set DX to difference between end-start lines	}
	MOV		CX,Y2			{ Put Y2 into CX								}
	SUB		CX,Y1			{ Subtract Y1 from CX to gain Height-1			}
	INC		CX				{ Increase to get real height of line			}
	MOV		ES:[DI+2],CX	{ Store height in Data							}
	ADD		DI,4			{ Move ES:DI past width/height					}
	CLD						{ Set direction to +							}
@LOOP_Y:
	PUSH	CX				{ Save CX on stack, so CX can be used for width	}
	MOV		CX,BX			{ Fetch width of block into CX					}
	REP		MOVSB			{ Fetch one line from the screen				}
	ADD		SI,DX			{ Move to next line								}
	POP		CX				{ Fetch CX from stack							}
	LOOP	@LOOP_Y			{ If more lines, go next --> @LOOP_Y			}
	JMP		@EXIT			{ Exit											}
@VERT_LINE:
	MOV		ES:[DI].WORD,1	{ Store width in Data							}
	MOV		AX,$0140		{ Set AX to $0140 (Width of screen in pixels)	}
	MUL		Y1				{ Multiply AX width Y1 to get offset1			}
	ADD		AX,X1			{ Add X1 to AX to get offset2					}
	MOV		SI,AX			{ Put offset of first pixel into SI				}
	MOV		CX,Y2			{ Put Y2 into CX								}
	SUB		CX,Y1			{ Subtract Y1 from CX to gain Height-1			}
	INC		CX				{ Increase to get real height of line			}
	MOV		ES:[DI+2],CX	{ Store height in Data							}
	ADD		DI,4			{ Move ES:DI past width/height					}
	CLD						{ Set direction to +							}
@DO_VERT:
	MOV		AL,DS:[SI]		{ Fetch pixel from screen						}
	STOSB					{ Store pixel into Data							}
	ADD		SI,$0140		{ Move offset down to next line					}
	LOOP	@DO_VERT		{ If more pixels, do it again --> @DO_VERT		}
	JMP		@EXIT			{ Exit											}
@HORZ_LINE:
	MOV		ES:[DI+2].WORD,1{ Store height in Data							}
	MOV		AX,$0140		{ Set AX to $0140 (Width of screen in pixels)	}
	MUL		Y1				{ Multiply AX width Y1 to get offset1			}
	ADD		AX,X1			{ Add X1 to AX to get offset2					}
	MOV		SI,AX			{ Put offset of first pixel into SI				}
	MOV		CX,X2			{ Put X2 into CX								}
	SUB		CX,X1			{ Subtract X1 from CX to gain Width-1			}
	INC		CX				{ Increase to get real width of line			}
	MOV		ES:[DI],CX		{ Store width in Data							}
	ADD		DI,4			{ Move ES:DI past width/height					}
	CLD						{ Set direction to +							}
@DO_HORZ:
	REP		MOVSB			{ Move pixels screen-->Data						}
	JMP		@EXIT			{ Exit											}
@EXIT:
	POP		DS				{ Restore DS									}
	POP		ES				{ Restore ES									}
End;

Procedure XPutImage(X,Y : Word; Var Data); Assembler;

	Procedure XNormalPutImage(X,Y : Word; Var Data); Assembler;
	Asm
		PUSH	ES
		PUSH	DS
		MOV		ES,ScrSeg		{ Point ES:0 to screen				}
		MOV		AX,$0140
		MUL		Y
		ADD		AX,X
		MOV		DI,AX
		LDS		SI,Data
		CLD
		LODSW
		MOV		DX,AX
		ADD		AX,X
		DEC		AX
		CMP		AX,$0140
		JAE		@EXIT
		LODSW
		MOV		CX,AX
		ADD		AX,Y
		DEC		AX
		CMP		AX,$00C8
		JAE		@EXIT
		MOV		BX,$0140
		SUB		BX,DX
	@DOIT:
		PUSH	CX
		MOV		CX,DX
		REP		MOVSB
		ADD		DI,BX
		POP		CX
		LOOP	@DOIT
	@EXIT:
		POP		DS
		POP		ES
	End;

	Procedure XTransPutImage(X,Y : Word; Var Data; MaskColor : Byte); Assembler;
	Asm
		PUSH	ES
		PUSH	DS
		MOV		ES,ScrSeg		{ Point ES:0 to screen				}
		MOV		AX,$0140
		MUL		Y
		ADD		AX,X
		MOV		DI,AX
		LDS		SI,Data
		CLD
		LODSW
		MOV		DX,AX
		ADD		AX,X
		DEC		AX
		CMP		AX,$0140
		JAE		@EXIT
		LODSW
		MOV		CX,AX
		ADD		AX,Y
		DEC		AX
		CMP		AX,$00C8
		JAE		@EXIT
		MOV		BX,$0140
		SUB		BX,DX
	@DOIT:
		PUSH	CX
		MOV		CX,DX
	@LOOPIT:
		LODSB
		CMP		AL,MaskColor
		JE		@TRANSPARENT
		STOSB
		JMP		@NEXT
	@TRANSPARENT:
		INC		DI
	@NEXT:
		LOOP	@LOOPIT
		JMP		@NEXTLINE
	@NEXTLINE:
		ADD		DI,BX
		POP		CX
		LOOP	@DOIT
	@EXIT:
		POP		DS
		POP		ES
	End;

	Procedure XXOrPutImage(X,Y : Word; Var Data); Assembler;
	Asm
		PUSH	ES
		PUSH	DS
		MOV		ES,ScrSeg		{ Point ES:0 to screen				}
		MOV		AX,$0140
		MUL		Y
		ADD		AX,X
		MOV		DI,AX
		LDS		SI,Data
		CLD
		LODSW
		MOV		DX,AX
		ADD		AX,X
		DEC		AX
		CMP		AX,$0140
		JAE		@EXIT
		LODSW
		MOV		CX,AX
		ADD		AX,Y
		DEC		AX
		CMP		AX,$00C8
		JAE		@EXIT
		MOV		BX,$0140
		SUB		BX,DX
	@DOIT:
		PUSH	CX
		MOV		CX,DX
	@LOOPIT:
		LODSB
		XOR		AL,ES:[DI]
		STOSB
		LOOP	@LOOPIT
		ADD		DI,BX
		POP		CX
		LOOP	@DOIT
	@EXIT:
		POP		DS
		POP		ES
	End;

	Procedure XTranslatePutImage(X,Y : Word; Var Data; TransPtr : Pointer); Assembler;
	Asm
		PUSH	ES
		PUSH	DS
		MOV		ES,ScrSeg		{ Point ES:0 to screen				}
		MOV		AX,$0140
		MUL		Y
		ADD		AX,X
		MOV		DI,AX
		LDS		SI,Data
		CLD
		LODSW
		MOV		DX,AX
		ADD		AX,X
		DEC		AX
		CMP		AX,$0140
		JAE		@EXIT
		LODSW
		MOV		CX,AX
		ADD		AX,Y
		DEC		AX
		CMP		AX,$00C8
		JAE		@EXIT
		MOV		BX,$0140
		SUB		BX,DX
	@DOIT:
		PUSH	CX
		MOV		CX,DX
	@LOOPIT:
		LODSB
		PUSH	DS
		PUSH	BX
		LDS		BX,DWORD PTR TransPtr
		XLAT
		POP		BX
		POP		DS
		STOSB
		LOOP	@LOOPIT
		ADD		DI,BX
		POP		CX
		LOOP	@DOIT
	@EXIT:
		POP		DS
		POP		ES
	End;

	Procedure XAddPutImage(X,Y : Word; Var Data); Assembler;
	Asm
		PUSH	ES
		PUSH	DS
		MOV		ES,ScrSeg		{ Point ES:0 to screen				}
		MOV		AX,$0140
		MUL		Y
		ADD		AX,X
		MOV		DI,AX
		LDS		SI,Data
		CLD
		LODSW
		MOV		DX,AX
		ADD		AX,X
		DEC		AX
		CMP		AX,$0140
		JAE		@EXIT
		LODSW
		MOV		CX,AX
		ADD		AX,Y
		DEC		AX
		CMP		AX,$00C8
		JAE		@EXIT
		MOV		BX,$0140
		SUB		BX,DX
	@DOIT:
		PUSH	CX
		MOV		CX,DX
	@LOOPIT:
		LODSB
		ADD		AL,ES:[DI]
		STOSB
		LOOP	@LOOPIT
		ADD		DI,BX
		POP		CX
		LOOP	@DOIT
	@EXIT:
		POP		DS
		POP		ES
	End;

	Procedure XSubPutImage(X,Y : Word; Var Data); Assembler;
	Asm
		PUSH	ES
		PUSH	DS
		MOV		ES,ScrSeg		{ Point ES:0 to screen				}
		MOV		AX,$0140
		MUL		Y
		ADD		AX,X
		MOV		DI,AX
		LDS		SI,Data
		CLD
		LODSW
		MOV		DX,AX
		ADD		AX,X
		DEC		AX
		CMP		AX,$0140
		JAE		@EXIT
		LODSW
		MOV		CX,AX
		ADD		AX,Y
		DEC		AX
		CMP		AX,$00C8
		JAE		@EXIT
		MOV		BX,$0140
		SUB		BX,DX
	@DOIT:
		PUSH	CX
		MOV		CX,DX
	@LOOPIT:
		LODSB
		SUB		AL,ES:[DI]
		STOSB
		LOOP	@LOOPIT
		ADD		DI,BX
		POP		CX
		LOOP	@DOIT
	@EXIT:
		POP		DS
		POP		ES
	End;

	Procedure XIncPutImage(X,Y : Word; Var Data); Assembler;
	Asm
		PUSH	ES
		PUSH	DS
		MOV		ES,ScrSeg		{ Point ES:0 to screen				}
		MOV		AX,$0140
		MUL		Y
		ADD		AX,X
		MOV		DI,AX
		LDS		SI,Data
		CLD
		LODSW
		MOV		DX,AX
		ADD		AX,X
		DEC		AX
		CMP		AX,$0140
		JAE		@EXIT
		LODSW
		MOV		CX,AX
		ADD		AX,Y
		DEC		AX
		CMP		AX,$00C8
		JAE		@EXIT
		MOV		BX,$0140
		SUB		BX,DX
	@DOIT:
		PUSH	CX
		MOV		CX,DX
	@LOOPIT:
		LODSB
		INC		AL
		STOSB
		LOOP	@LOOPIT
		ADD		DI,BX
		POP		CX
		LOOP	@DOIT
	@EXIT:
		POP		DS
		POP		ES
	End;

	Procedure XDecPutImage(X,Y : Word; Var Data); Assembler;
	Asm
		PUSH	ES
		PUSH	DS
		MOV		ES,ScrSeg		{ Point ES:0 to screen				}
		MOV		AX,$0140
		MUL		Y
		ADD		AX,X
		MOV		DI,AX
		LDS		SI,Data
		CLD
		LODSW
		MOV		DX,AX
		ADD		AX,X
		DEC		AX
		CMP		AX,$0140
		JAE		@EXIT
		LODSW
		MOV		CX,AX
		ADD		AX,Y
		DEC		AX
		CMP		AX,$00C8
		JAE		@EXIT
		MOV		BX,$0140
		SUB		BX,DX
	@DOIT:
		PUSH	CX
		MOV		CX,DX
	@LOOPIT:
		LODSB
		DEC		AL
		STOSB
		LOOP	@LOOPIT
		ADD		DI,BX
		POP		CX
		LOOP	@DOIT
	@EXIT:
		POP		DS
		POP		ES
	End;

	Procedure XApproachPutImage(X,Y : Word; Var Data); Assembler;
	Asm
		PUSH	ES
		PUSH	DS
		MOV		ES,ScrSeg		{ Point ES:0 to screen				}
		MOV		AX,$0140
		MUL		Y
		ADD		AX,X
		MOV		DI,AX
		LDS		SI,Data
		CLD
		LODSW
		MOV		DX,AX
		ADD		AX,X
		DEC		AX
		CMP		AX,$0140
		JAE		@EXIT
		LODSW
		MOV		CX,AX
		ADD		AX,Y
		DEC		AX
		CMP		AX,$00C8
		JAE		@EXIT
		MOV		BX,$0140
		SUB		BX,DX
	@DOIT:
		PUSH	CX
		MOV		CX,DX
	@LOOPIT:
		LODSB
		MOV		AH,ES:[DI]
		CMP		AH,AL
		JA		@DOWN
		JB		@UP
		MOV		AL,AH
		JMP		@NEXT
	@UP:
		INC		AH
		MOV		AL,AH
		JMP		@NEXT
	@DOWN:
		DEC		AH
		MOV		AL,AH
	@NEXT:
		STOSB
		LOOP	@LOOPIT
		ADD		DI,BX
		POP		CX
		LOOP	@DOIT
	@EXIT:
		POP		DS
		POP		ES
	End;

	Procedure XOnlyPutImage(X,Y : Word; Var Data; Color : Byte); Assembler;
	Asm
		PUSH	ES
		PUSH	DS
		MOV		ES,ScrSeg		{ Point ES:0 to screen				}
		MOV		AX,$0140
		MUL		Y
		ADD		AX,X
		MOV		DI,AX
		LDS		SI,Data
		CLD
		LODSW
		MOV		DX,AX
		ADD		AX,X
		DEC		AX
		CMP		AX,$0140
		JAE		@EXIT
		LODSW
		MOV		CX,AX
		ADD		AX,Y
		DEC		AX
		CMP		AX,$00C8
		JAE		@EXIT
		MOV		BX,$0140
		SUB		BX,DX
	@DOIT:
		PUSH	CX
		MOV		CX,DX
	@LOOPIT:
		LODSB
		MOV		AH,ES:[DI]
		CMP		AH,Color
		JE		@NEXT
		MOV		AL,AH
	@NEXT:
		STOSB
		LOOP	@LOOPIT
		ADD		DI,BX
		POP		CX
		LOOP	@DOIT
	@EXIT:
		POP		DS
		POP		ES
	End;

Asm
	PUSH	X
	PUSH	Y
	PUSH	WORD PTR Data+2
	PUSH	WORD PTR Data
	PUSH	WORD PTR [BP]
	CMP		PutFn,pNORMAL
	JNE		@1
	CALL	XNormalPutImage
	JMP		@EXIT
@1:
	CMP		PutFn,pXOR
	JNE		@2
	CALL	XXOrPutImage
	JMP		@EXIT
@2:
	CMP		PutFn,pADD
	JNE		@3
	CALL	XAddPutImage
	JMP		@EXIT
@3:
	CMP		PutFn,pSUB
	JNE		@4
	CALL	XSubPutImage
	JMP		@EXIT
@4:
	CMP		PutFn,pTRANSPARENT
	JNE		@5
	POP		BX
	MOV		AL,MaskColor
	PUSH	AX
	PUSH	BX
	CALL	XTransPutImage
	JMP		@EXIT
@5:
	CMP		PutFn,pTRANSLATE
	JNE		@5B
	POP		BX
	PUSH	WORD PTR TransPtr+2
	PUSH	WORD PTR TransPtr
	PUSH	BX
	CALL	XTranslatePutImage
	JMP		@EXIT
@5B:
	CMP		PutFn,pAPPROACH
	JNE		@6
	CALL	XApproachPutImage
	JMP		@EXIT
@6:
	CMP		PutFn,pINC
	JNE		@6B
	CALL	XIncPutImage
	JMP		@EXIT
@6B:
	CMP		PutFn,pDEC
	JNE		@6C
	CALL	XDECPutImage
	JMP		@EXIT
@6C:
	CMP		PutFn,pONLY
	JNE		@7
	POP		BX
	MOV		AL,MaskColor
	PUSH	AX
	PUSH	BX
	CALL	XOnlyPutImage
	JMP		@EXIT
@7:
	POP		AX
	POP		AX
	POP		AX
	POP		AX
@EXIT:
End;

Const
	Signature	: Array[1..4] Of Char = #207#179#98#246;

Function XImageBuffer(Var B : Pointer) : Pointer;
Const
	Data	: Array[1..2] Of Word =
		(320,200);
Begin
	If XOkBuffer(B) Then
		XImageBuffer:=PChar(Ptr(Seg(B^),12))
	Else Begin
		XImageBuffer:=PChar(Ptr($BFFF,12));
		Move(Data,Ptr($BFFF,12)^,4);
	End;
End;

Procedure XAllocBuffer(Var B : Pointer);
Var
	P	: PChar;
Const
	Data	: Array[1..2] Of Word =
		(320,200);
Begin
	GetMem(B,64032);
	FillChar(B^,64032,#0);
	Move(Signature,B^,4);
	P:=Ptr(Seg(B^),12);
	Move(Data,P^,4);
End;

Procedure XFreeBuffer(Var B : Pointer);
Begin
	If XCurrentBuffer=B Then
		XUseBuffer(NIL);
	FreeMem(B,64032);
	B:=NIL;
End;

Function XOkBuffer(B : Pointer) : Boolean;
Var
	Temp	: Array[1..4] Of Char;
	OK		: Boolean;
	I		: Word;
Begin
	If B=NIL Then Begin
		ScrSeg:=$A000;
		CurBuf:=NIL;
		Exit;
	End;
	Move(B^,Temp,4);
	OK:=TRUE;
	For I:=1 To 4 Do
		If Temp[I]<>Signature[I] Then
			OK:=FALSE;
	XOkBuffer:=OK;
End;

Procedure XUseBuffer(B : Pointer);
Var
	Temp	: Array[1..4] Of Char;
	I		: Byte;
	OK		: Boolean;
Begin
	If B=NIL Then Begin
		ScrSeg:=$A000;
		CurBuf:=NIL;
		Exit;
	End;
	Move(B^,Temp,4);
	OK:=TRUE;
	For I:=1 To 4 Do
		If Temp[I]<>Signature[I] Then
			OK:=FALSE;
	If Not OK Then Begin
		ScrSeg:=$A000;
		CurBuf:=NIL;
		Exit;
	End;
	ScrSeg:=Seg(B^)+1;
	CurBuf:=B;
End;

Procedure XShowBuffer(B : Pointer);
Var
	Temp	: Array[1..4] Of Char;
	I		: Byte;
Begin
	Move(B^,Temp,4);
	For I:=1 To 4 Do
		If Temp[I]<>Signature[I] Then
			Exit;
	Move(Ptr(Seg(B^)+1,0)^,Ptr($A000,0)^,64000);
End;

Procedure XSetFn(NewFn : TPutFn; NewMaskColor : Byte); Assembler;
Asm
	MOV		AL,NewFn
	MOV		PutFn,AL
	MOV		AL,NewMaskColor
	MOV		MaskColor,AL
End;

Function XLoadPCX(FileName : String; Var RGB; Var Width,Height : Word) : Integer;
Const
	BufferSize	= 8192;
Var
	Temp			: Integer;
	PH				: Array[1..16] Of Byte;
	OldMode			: Byte;
	Buffer			: Pointer;
	F				: File;
	BytesToGo		: Word;
	P				: PChar;
	InBuffer		: Word;
	ScreenOffset	: Word;
	Value			: Byte;
	Count			: Word;
	L				: Word;

	Function GetNext : Byte;
	Begin
		If InBuffer=0 Then Begin
			BlockRead(F,Buffer^,BufferSize,InBuffer);
			P:=Buffer;
		End;
		GetNext:=Ord(P^);
		Inc(P);
		Dec(InBuffer);
	End;

Begin
	ScreenOffset:=$0000;
	Assign(F,FileName);
	OldMode:=FileMode;
	FileMode:=0;
	{$I-}
	ReSet(F,1);
	{$I+}
	Temp:=IOResult;
	If Temp<>0 Then Begin
		XLoadPCX:=Temp;
		Exit;
	End;
	FileMode:=OldMode;
	BytesToGo:=$FA00;
	BlockRead(F,PH,SizeOf(PH));
	If Not ((PH[3]=1) And (PH[4]=8) And (PH[5]=0) And (PH[6]=0) And (PH[7]=0) And (PH[8]=0)) Then Begin
		Close(F);
		XLoadPCX:=5;
		Exit;
	End;
	Seek(F,128);
	InBuffer:=0;
	GetMem(Buffer,BufferSize);
	P:=Buffer;
	Move(PH[9],Width,2);
	Move(PH[11],Height,2);
	Inc(Width);
	Inc(Height);
	If Not ((Width<=320) And (Height<=200)) Then Begin
		Close(F);
		XLoadPCX:=5;
		Exit;
	End;
	BytesToGo:=Width*Height;
	L:=Width;
	While BytesToGo>0 Do Begin
		Value:=GetNext;
		If (Value And $C0)=$C0 Then Begin
			Count:=Value And $3F;
			Value:=GetNext;
			While Count>0 Do Begin
				Dec(Count);
				Mem[ScrSeg:ScreenOffset]:=Value;
				Inc(ScreenOffset);
				Dec(BytesToGo);
				Dec(L);
				If L=0 Then Begin
					L:=Width;
					Inc(ScreenOffset,320-L);
				End;
			End;
		End Else Begin
			Mem[ScrSeg:ScreenOffset]:=Value;
			Inc(ScreenOffset);
			Dec(BytesToGo);
			Dec(L);
			If L=0 Then Begin
				L:=Width;
				Inc(ScreenOffset,320-L);
			End;
		End;
	End;
	FreeMem(Buffer,BufferSize);
	Seek(F,FileSize(F)-768);
	BlockRead(F,RGB,768);
	Close(F);
	P:=@RGB;
	For Count:=0 To 767 Do Begin
		P^:=Chr(Ord(P^) Shr 2);
		Inc(P);
	End;
	XLoadPCX:=0;
End;

Procedure XSetRGBBlack(StartIndex : Byte; ColorCount : Word); Assembler;
Asm
	MOV		DX,$03C8
	MOV		AL,StartIndex
	OUT		DX,AL
	MOV		AX,ColorCount
	MOV		BX,3
	MUL		AX
	MOV		CX,AX
	MOV		DX,$03C9
	MOV		AL,$00
@DOIT:
	OUT		DX,AL
	LOOP	@DOIT
End;

Procedure XSetRGBFade(StartIndex : Byte; ColorCount : Word; Var RGBData; Frames : Byte);
Var
	Old,NewData	: Array[1..768] Of Byte;
	I			: Word;
Begin
	XGetRGBMany(StartIndex,ColorCount,Old);
	For I:=0 To Frames-1 Do Begin
		XCalcRGBFadeFrame(StartIndex,ColorCount,Old,RGBData,NewData,I,Frames);
		VRet;
		XSetRGBMany(StartIndex,ColorCount,NewData);
	End;
End;

Procedure XCalcRGBFadeFrame(StartIndex : Byte; ColorCount : Word; Var RGB1,RGB2,ResultRGB; CurFrame,Frames : Byte);
Var
	P1,P2,P3	: PChar;
	J			: Word;
Begin
	P1:=@RGB1;
	P2:=@RGB2;
	P3:=@ResultRGB;
	For J:=0 To 767 Do Begin
{			V1:=Ord(P1^);
		V2:=Ord(P2^);
		P3^:=Chr(V1+((V2-V1)*I) Div 63); }
		P3^:=Chr(Ord(P1^)+(Integer(Ord(P2^))-Ord(P1^))*CurFrame Div (Frames-1));
		Inc(P1);
		Inc(P2);
		Inc(P3);
	End;
End;

Procedure XGetRGB(Index : Byte; Var R,G,B : Byte); Assembler;
Asm
	MOV		AX,$1015
	XOR		BH,BH
	MOV		BL,Index
	INT		$10
	LES		DI,DWORD PTR R
	MOV		ES:[DI],DH
	LES		DI,DWORD PTR G
	MOV		ES:[DI],CH
	LES		DI,DWORD PTR B
	MOV		ES:[DI],CL
End;

Procedure XGetRGBMany(StartIndex : Byte; ColorCount : Word; Var RGBData); Assembler;
Asm
	PUSH	ES
	MOV		AX,$1017
	XOR		BH,BH
	MOV		BL,StartIndex
	MOV		CX,ColorCount
	LES		DX,RGBData
	INT		$10
	POP		ES
End;

Function XImageWidth(Var Data) : Word; Assembler;
Asm
	LES		DI,Data
	MOV		AX,ES:[DI]
End;

Function XImageHeight(Var Data) : Word; Assembler;
Asm
	LES		DI,Data
	MOV		AX,ES:[DI+2]
End;

Function XImageMemSize(Var Data) : Word; Assembler;
Asm
	LES		DI,Data
	MOV		AX,ES:[DI+2]
	MOV		BX,ES:[DI]
	MUL		BX
	ADD		AX,4
End;

Function XCurrentBuffer : Pointer; Assembler;
Asm
	MOV		AX,WORD PTR CurBuf
	MOV		DX,WORD PTR CurBuf+2
End;

Procedure XCopyBuffer(B1,B2 : Pointer; X1,Y1,X2,Y2 : Word); Assembler;
Var
	OldBuffer	: Pointer;
	Seg1,Seg2	: Word;
Asm
	PUSH	DS
	CALL	XCurrentBuffer
	MOV		WORD PTR OldBuffer,AX
	MOV		WORD PTR OldBuffer+2,DX

	PUSH	WORD PTR B1+2
	PUSH	WORD PTR B1
	CALL	XUseBuffer
	MOV		AX,ScrSeg
	MOV		Seg1,AX
	PUSH	WORD PTR B2+2
	PUSH	WORD PTR B2
	CALL	XUseBuffer
	MOV		AX,ScrSeg
	CMP		Seg1,AX
	JE		@EXIT
	MOV		Seg2,AX
	CMP		X1,$0140		{ Is X1 inside screen ?							}
	JAE		@EXIT			{ No --> @EXIT									}
	CMP		X2,$0140		{ Is X2 inside screen ?							}
	JB		@OK1			{ Yes --> @OK1									}
	MOV		X2,$013F		{ Move X2 to edge of screen						}
@OK1:
	CMP		Y1,$00C8		{ Is Y1 inside screen ?							}
	JAE		@EXIT			{ No --> @EXIT									}
	CMP		Y2,$00C8		{ Is Y2 inside screen ?							}
	JB		@OK2			{ Yes --> @OK2									}
	MOV		Y2,$00C7		{ Move Y2 to edge of screen						}
@OK2:
	MOV		AX,$0140
	MUL		Y1
	ADD		AX,X1
	MOV		DS,Seg1
	MOV		SI,AX
	MOV		ES,Seg2
	MOV		DI,AX
	MOV		DX,X2
	SUB		DX,X1
	INC		DX
	MOV		CX,Y2
	SUB		CX,Y1
	INC		CX
	MOV		BX,$0140
	SUB		BX,DX
	CLD
@DOIT:
	PUSH	CX
	MOV		CX,DX
	REP		MOVSB
	ADD		SI,BX
	ADD		DI,BX
	POP		CX
	LOOP	@DOIT
@EXIT:
	POP		DS
	PUSH	WORD PTR OldBuffer+2
	PUSH	WORD PTR OldBuffer
	CALL	XUseBuffer
End;

Procedure XDupBuffer(B1,B2 : Pointer); Assembler;
Var
	OldBuffer	: Pointer;
	Seg1,Seg2	: Word;
Asm
	PUSH	DS
	CALL	XCurrentBuffer
	MOV		WORD PTR OldBuffer,AX
	MOV		WORD PTR OldBuffer+2,DX

	PUSH	WORD PTR B1+2
	PUSH	WORD PTR B1
	CALL	XUseBuffer
	MOV		AX,ScrSeg
	MOV		Seg1,AX
	PUSH	WORD PTR B2+2
	PUSH	WORD PTR B2
	CALL	XUseBuffer
	MOV		AX,ScrSeg
	CMP		Seg1,AX
	JE		@EXIT
	MOV		Seg2,AX
	MOV		DS,Seg1
	MOV		ES,Seg2
	XOR		SI,SI
	XOR		DI,DI
	MOV		CX,$7D00
	CLD
	REP		MOVSW
@EXIT:
	POP		DS
	PUSH	WORD PTR OldBuffer+2
	PUSH	WORD PTR OldBuffer
	CALL	XUseBuffer
End;

Procedure XDifBuffer(B1,B2,B3 : Pointer); Assembler;
	{ Copies from B1 to B2 and B3, if B1<>B2. Useful if B2 is faster than B3 }
Var
	OldBuffer		: Pointer;
	Seg1,Seg2,Seg3	: Word;
Asm
	PUSH	DS
	CALL	XCurrentBuffer
	MOV		WORD PTR OldBuffer,AX
	MOV		WORD PTR OldBuffer+2,DX

	PUSH	WORD PTR B1+2
	PUSH	WORD PTR B1
	CALL	XUseBuffer
	MOV		AX,ScrSeg
	MOV		Seg1,AX
	PUSH	WORD PTR B2+2
	PUSH	WORD PTR B2
	CALL	XUseBuffer
	MOV		AX,ScrSeg
	CMP		Seg1,AX
	JE		@EXIT
	MOV		Seg2,AX
	PUSH	WORD PTR B3+2
	PUSH	WORD PTR B3
	CALL	XUseBuffer
	MOV		AX,ScrSeg
	CMP		Seg1,AX
	JE		@EXIT
	CMP		Seg2,AX
	JE		@EXIT
	MOV		Seg3,AX
	MOV		DS,Seg1
	MOV		ES,Seg2
	XOR		SI,SI
	XOR		DI,DI
	MOV		CX,$7D00
	CLD
@DOIT:
	LODSW
	CMP		AX,ES:[DI]
	JE		@OK
	MOV		ES,Seg3
	MOV		ES:[DI],AX
	MOV		ES,Seg2
	STOSW
	JMP		@NEXT
@OK:
	INC		DI
	INC		DI
@NEXT:
	LOOP	@DOIT
@EXIT:
	POP		DS
	PUSH	WORD PTR OldBuffer+2
	PUSH	WORD PTR OldBuffer
	CALL	XUseBuffer
End;

Procedure XSwapBuffer(B1,B2 : Pointer; X1,Y1,X2,Y2 : Word); Assembler;
Var
	OldBuffer	: Pointer;
	Seg1,Seg2	: Word;
Asm
	PUSH	DS
	CALL	XCurrentBuffer
	MOV		WORD PTR OldBuffer,AX
	MOV		WORD PTR OldBuffer+2,DX

	PUSH	WORD PTR B1+2
	PUSH	WORD PTR B1
	CALL	XUseBuffer
	MOV		AX,ScrSeg
	MOV		Seg1,AX
	PUSH	WORD PTR B2+2
	PUSH	WORD PTR B2
	CALL	XUseBuffer
	MOV		AX,ScrSeg
	CMP		Seg1,AX
	JE		@EXIT
	MOV		Seg2,AX
	CMP		X1,$0140		{ Is X1 inside screen ?							}
	JAE		@EXIT			{ No --> @EXIT									}
	CMP		X2,$0140		{ Is X2 inside screen ?							}
	JB		@OK1			{ Yes --> @OK1									}
	MOV		X2,$013F		{ Move X2 to edge of screen						}
@OK1:
	CMP		Y1,$00C8		{ Is Y1 inside screen ?							}
	JAE		@EXIT			{ No --> @EXIT									}
	CMP		Y2,$00C8		{ Is Y2 inside screen ?							}
	JB		@OK2			{ Yes --> @OK2									}
	MOV		Y2,$00C7		{ Move Y2 to edge of screen						}
@OK2:
	MOV		AX,$0140
	MUL		Y1
	ADD		AX,X1
	MOV		DS,Seg1
	MOV		SI,AX
	MOV		ES,Seg2
	MOV		DI,AX
	MOV		DX,X2
	SUB		DX,X1
	INC		DX
	MOV		CX,Y2
	SUB		CX,Y1
	INC		CX
	MOV		BX,$0140
	SUB		BX,DX
	CLD
@DOIT:
	PUSH	CX
	MOV		CX,DX
@LOOP:
	MOV		AL,DS:[SI]
	MOV		AH,ES:[DI]
	MOV		DS:[SI],AH
	STOSB
	INC		SI
	LOOP	@LOOP
	ADD		SI,BX
	ADD		DI,BX
	POP		CX
	LOOP	@DOIT
@EXIT:
	POP		DS
	PUSH	WORD PTR OldBuffer+2
	PUSH	WORD PTR OldBuffer
	CALL	XUseBuffer
End;

Procedure XTranslateImage(Var Image; Const TransTable : TTranslate); Assembler;
Asm
	LES		DI,DWORD PTR [Image]
	MOV		SI,WORD PTR [TransTable]
	CLD
	MOV		AX,ES:[DI]
	MOV		BX,ES:[DI+$0002]
	ADD		DI,$0004
	MUL		BX
	MOV		CX,AX
	XOR		BX,BX
@@1:
	MOV		BL,ES:[DI]
	MOV		AL,SS:[SI+BX]
	STOSB
	LOOP	@@1
End;

Function XCurrentSegment : Word; Assembler;
Asm
	MOV		AX,ScrSeg
End;

Procedure XDefaultPalette(Var RGB);
Const
	DefaultPaletteTable	: Array[0..767] Of Byte =
		($00,$00,$00,$00,$00,$2A,$00,$2A,$00,$00,$2A,$2A,$2A,$00,$00,
		 $2A,$00,$2A,$2A,$2A,$00,$2A,$2A,$2A,$00,$00,$15,$00,$00,$3F,
		 $00,$2A,$15,$00,$2A,$3F,$2A,$00,$15,$2A,$00,$3F,$2A,$2A,$15,
		 $2A,$2A,$3F,$00,$15,$00,$00,$15,$2A,$00,$3F,$00,$00,$3F,$2A,
		 $2A,$15,$00,$2A,$15,$2A,$2A,$3F,$00,$2A,$3F,$2A,$00,$15,$15,
		 $00,$15,$3F,$00,$3F,$15,$00,$3F,$3F,$2A,$15,$15,$2A,$15,$3F,
		 $2A,$3F,$15,$2A,$3F,$3F,$15,$00,$00,$15,$00,$2A,$15,$2A,$00,
		 $15,$2A,$2A,$3F,$00,$00,$3F,$00,$2A,$3F,$2A,$00,$3F,$2A,$2A,
		 $15,$00,$15,$15,$00,$3F,$15,$2A,$15,$15,$2A,$3F,$3F,$00,$15,
		 $3F,$00,$3F,$3F,$2A,$15,$3F,$2A,$3F,$15,$15,$00,$15,$15,$2A,
		 $15,$3F,$00,$15,$3F,$2A,$3F,$15,$00,$3F,$15,$2A,$3F,$3F,$00,
		 $3F,$3F,$2A,$15,$15,$15,$15,$15,$3F,$15,$3F,$15,$15,$3F,$3F,
		 $3F,$15,$15,$3F,$15,$3F,$3F,$3F,$15,$3F,$3F,$3F,$3F,$1F,$1F,
		 $3F,$27,$1F,$3F,$2F,$1F,$3F,$37,$1F,$3F,$3F,$1F,$37,$3F,$1F,
		 $2F,$3F,$1F,$27,$3F,$1F,$1F,$3F,$1F,$1F,$3F,$27,$1F,$3F,$2F,
		 $1F,$3F,$37,$1F,$3F,$3F,$1F,$37,$3F,$1F,$2F,$3F,$1F,$27,$3F,
		 $2D,$2D,$3F,$31,$2D,$3F,$36,$2D,$3F,$3A,$2D,$3F,$3F,$2D,$3F,
		 $3F,$2D,$3A,$3F,$2D,$36,$3F,$2D,$31,$3F,$2D,$2D,$3F,$31,$2D,
		 $3F,$36,$2D,$3F,$3A,$2D,$3F,$3F,$2D,$3A,$3F,$2D,$36,$3F,$2D,
		 $31,$3F,$2D,$2D,$3F,$2D,$2D,$3F,$31,$2D,$3F,$36,$2D,$3F,$3A,
		 $2D,$3F,$3F,$2D,$3A,$3F,$2D,$36,$3F,$2D,$31,$3F,$00,$00,$1C,
		 $07,$00,$1C,$0E,$00,$1C,$15,$00,$1C,$1C,$00,$1C,$1C,$00,$15,
		 $1C,$00,$0E,$1C,$00,$07,$1C,$00,$00,$1C,$07,$00,$1C,$0E,$00,
		 $1C,$15,$00,$1C,$1C,$00,$15,$1C,$00,$0E,$1C,$00,$07,$1C,$00,
		 $00,$1C,$00,$00,$1C,$07,$00,$1C,$0E,$00,$1C,$15,$00,$1C,$1C,
		 $00,$15,$1C,$00,$0E,$1C,$00,$07,$1C,$0E,$0E,$1C,$11,$0E,$1C,
		 $15,$0E,$1C,$18,$0E,$1C,$1C,$0E,$1C,$1C,$0E,$18,$1C,$0E,$15,
		 $1C,$0E,$11,$1C,$0E,$0E,$1C,$11,$0E,$1C,$15,$0E,$1C,$18,$0E,
		 $1C,$1C,$0E,$18,$1C,$0E,$15,$1C,$0E,$11,$1C,$0E,$0E,$1C,$0E,
		 $0E,$1C,$11,$0E,$1C,$15,$0E,$1C,$18,$0E,$1C,$1C,$0E,$18,$1C,
		 $0E,$15,$1C,$0E,$11,$1C,$14,$14,$1C,$16,$14,$1C,$18,$14,$1C,
		 $1A,$14,$1C,$1C,$14,$1C,$1C,$14,$1A,$1C,$14,$18,$1C,$14,$16,
		 $1C,$14,$14,$1C,$16,$14,$1C,$18,$14,$1C,$1A,$14,$1C,$1C,$14,
		 $1A,$1C,$14,$18,$1C,$14,$16,$1C,$14,$14,$1C,$14,$14,$1C,$16,
		 $14,$1C,$18,$14,$1C,$1A,$14,$1C,$1C,$14,$1A,$1C,$14,$18,$1C,
		 $14,$16,$1C,$00,$00,$10,$04,$00,$10,$08,$00,$10,$0C,$00,$10,
		 $10,$00,$10,$10,$00,$0C,$10,$00,$08,$10,$00,$04,$10,$00,$00,
		 $10,$04,$00,$10,$08,$00,$10,$0C,$00,$10,$10,$00,$0C,$10,$00,
		 $08,$10,$00,$04,$10,$00,$00,$10,$00,$00,$10,$04,$00,$10,$08,
		 $00,$10,$0C,$00,$10,$10,$00,$0C,$10,$00,$08,$10,$00,$04,$10,
		 $08,$08,$10,$0A,$08,$10,$0C,$08,$10,$0E,$08,$10,$10,$08,$10,
		 $10,$08,$0E,$10,$08,$0C,$10,$08,$0A,$10,$08,$08,$10,$0A,$08,
		 $10,$0C,$08,$10,$0E,$08,$10,$10,$08,$0E,$10,$08,$0C,$10,$08,
		 $0A,$10,$08,$08,$10,$08,$08,$10,$0A,$08,$10,$0C,$08,$10,$0E,
		 $08,$10,$10,$08,$0E,$10,$08,$0C,$10,$08,$0A,$10,$0B,$0B,$10,
		 $0C,$0B,$10,$0D,$0B,$10,$0F,$0B,$10,$10,$0B,$10,$10,$0B,$0F,
		 $10,$0B,$0D,$10,$0B,$0C,$10,$0B,$0B,$10,$0C,$0B,$10,$0D,$0B,
		 $10,$0F,$0B,$10,$10,$0B,$0F,$10,$0B,$0D,$10,$0B,$0C,$10,$0B,
		 $0B,$10,$0B,$0B,$10,$0C,$0B,$10,$0D,$0B,$10,$0F,$0B,$10,$10,
		 $0B,$0F,$10,$0B,$0D,$10,$0B,$0C,$10,$00,$00,$07,$00,$00,$06,
		 $00,$00,$05,$00,$00,$04,$00,$00,$03,$00,$00,$02,$00,$00,$01,
		 $00,$00,$00);
Begin
	Move(DefaultPaletteTable,RGB,768);
End;

Procedure XFill3(X1,Y1,X2,Y2,X3,Y3 : Word; Color : Byte); Assembler;
Var
	MinY,MaxY	: Word;
	X,Y			: Word;
	LastX		: Word;
	O			: Word;

	Function Min(A,B : Word) : Word; Assembler;
	Asm
		MOV		AX,A
		CMP		AX,B
		JBE		@EXIT
		MOV		AX,B
	@EXIT:
	End;

	Function Max(A,B : Word) : Word; Assembler;
	Asm
		MOV		AX,A
		CMP		AX,B
		JAE		@EXIT
		MOV		AX,B
	@EXIT:
	End;

	Procedure Check(TY,X1,Y1,X2,Y2 : Word); Assembler;
	Asm
		MOV		AX,TY
		CMP		AX,Y1
		JAE		@OK1
		CMP		AX,Y2
		JAE		@OK1
		JMP		@EXIT
	@OK1:
		CMP		AX,Y1
		JBE		@OK2
		CMP		AX,Y2
		JBE		@OK2
		JMP		@EXIT
	@OK2:
		MOV		AX,Y1
		CMP		AX,Y2
		JNE		@NOTHORZ
		MOV		AX,X1
		CMP		AX,SI
		JAE		@1
		MOV		SI,AX
	@1:
		CMP		AX,DI
		JBE		@EXIT1
		MOV		DI,AX
	@EXIT1:
		MOV		AX,X2
		CMP		AX,SI
		JAE		@2
		MOV		SI,AX
	@2:
		CMP		AX,DI
		JBE		@EXIT2
		MOV		DI,AX
	@EXIT2:
		JMP		@EXIT
	@NOTHORZ:
		MOV		AX,X1
		CMP		AX,X2
		JNE		@NOTVERT
		MOV		AX,X1
		CMP		AX,SI
		JAE		@3
		MOV		SI,AX
	@3:
		CMP		AX,DI
		JBE		@EXIT3
		MOV		DI,AX
	@EXIT3:
		JMP		@EXIT
	@NOTVERT:
		MOV		AX,X2
		SUB		AX,X1
		MOV		BX,Y2
		SUB		BX,Y1
		MOV		DX,TY
		SUB		DX,Y1
		IMUL	DX
		IDIV	BX
		ADD		AX,X1
		CMP		AX,SI
		JAE		@4
		MOV		SI,AX
	@4:
		CMP		AX,DI
		JBE		@EXIT
		MOV		DI,AX
	@EXIT:
	End;

Asm
	MOV		AX,Y1
	CMP		AX,Y2
	JBE		@1
	MOV		AX,Y2
@1:
	CMP		AX,Y3
	JBE		@2
	MOV		AX,Y3
	MOV		MinY,AX
@2:
	MOV		MinY,AX
	MOV		AX,Y1
	CMP		AX,Y2
	JAE		@3
	MOV		AX,Y2
@3:
	CMP		AX,Y3
	JAE		@4
	MOV		AX,Y3
@4:
	MOV		MaxY,AX
	MOV		AX,MinY
	MOV		Y,AX
	XOR		AX,AX
	MOV		AX,$0140
	MUL		Y
	MOV		O,AX
@LOOP:
	MOV		SI,$013F
	MOV		DI,$0000
	PUSH	Y
	PUSH	X1
	PUSH	Y1
	PUSH	X2
	PUSH	Y2
	PUSH	WORD PTR [BP]
	CALL	Check
	PUSH	Y
	PUSH	X2
	PUSH	Y2
	PUSH	X3
	PUSH	Y3
	PUSH	WORD PTR [BP]
	CALL	Check
	PUSH	Y
	PUSH	X3
	PUSH	Y3
	PUSH	X1
	PUSH	Y1
	PUSH	WORD PTR [BP]
	CALL	Check
	MOV		AX,O
	ADD		AX,SI
	MOV		CX,DI
	SUB		CX,SI
	INC		CX
	MOV		DI,AX
	MOV		AL,Color
	CLD
	MOV		ES,ScrSeg
@DOIT:
	MOV		AL,Color
	CMP		PutFn,pNORMAL
	JNE		@101
	REP		STOSB
	JMP		@NEXTLINE
@101:
	CMP		PutFn,pTRANSPARENT
	JNE		@10
	MOV		AH,AL
	CMP		AL,MaskColor
	JNE		@NEXT
	MOV		AH,ES:[DI]
	JMP		@NEXT
@10:
	CMP		PutFn,pXOR
	JNE		@11
	MOV		AH,ES:[DI]
	XOR		AH,AL
	JMP		@NEXT
@11:
	CMP		PutFn,pADD
	JNE		@12
	MOV		AH,ES:[DI]
	ADD		AH,AL
	JMP		@NEXT
@12:
	CMP		PutFn,pSUB
	JNE		@13
	MOV		AH,ES:[DI]
	SUB		AH,AL
	JMP		@NEXT
@13:
	CMP		PutFn,pOR
	JNE		@14
	MOV		AH,ES:[DI]
	OR		AH,AL
	MOV		AL,AH
	JMP		@NEXT
@14:
	CMP		PutFn,pAND
	JNE		@15
	MOV		AH,ES:[DI]
	AND		AH,AL
	JMP		@NEXT
@15:
	CMP		PutFn,pTRANSLATE
	JNE		@15B
	PUSH	DS
	PUSH	BX
	MOV		AL,ES:[DI]
	LDS		BX,DWORD PTR TransPtr
	XLAT
	POP		BX
	POP		DS
	MOV		AH,AL
	JMP		@NEXT
@15B:
	CMP		PutFn,pAPPROACH
	JNE		@16
	MOV		AH,ES:[DI]
	CMP		AH,AL
	JB		@UP
	JA		@DOWN
	JMP		@NEXT
@UP:
	INC		AH
	JMP		@NEXT
@DOWN:
	DEC		AH
	JMP		@NEXT
@16:
	CMP		PutFn,pINC
	JNE		@16B
	INC		AH
	JMP		@NEXT
@16B:
	CMP		PutFn,pDEC
	JNE		@16C
	DEC		AH
	JMP		@NEXT
@16C:
	CMP		PutFn,pONLY
	JNE		@NEXT
	MOV		AH,ES:[DI]
	CMP		AH,MaskColor
	JNE		@NEXT
	MOV		AH,AL
@NEXT:
	MOV		AL,AH
	STOSB
	LOOP	@DOIT2
	JMP		@NEXTLINE
@DOIT2:
	JMP		@DOIT
@NEXTLINE:
	MOV		AX,O
	ADD		AX,$0140
	MOV		O,AX
	INC		Y
	MOV		AX,Y
	CMP		AX,MaxY
	JBE		@LOOP
@EXIT:
End;

Procedure XFill4(X1,Y1,X2,Y2,X3,Y3,X4,Y4 : Word; Color : Byte); Assembler;
Var
	MinY,MaxY	: Word;
	X,Y			: Word;
	O			: Word;

	Function Min(A,B : Word) : Word; Assembler;
	Asm
		MOV		AX,A
		CMP		AX,B
		JBE		@EXIT
		MOV		AX,B
	@EXIT:
	End;

	Function Max(A,B : Word) : Word; Assembler;
	Asm
		MOV		AX,A
		CMP		AX,B
		JAE		@EXIT
		MOV		AX,B
	@EXIT:
	End;

	Procedure Check(TY,X1,Y1,X2,Y2 : Word); Assembler;
	Asm
		MOV		AX,TY
		CMP		AX,Y1
		JAE		@OK1
		CMP		AX,Y2
		JAE		@OK1
		JMP		@EXIT
	@OK1:
		CMP		AX,Y1
		JBE		@OK2
		CMP		AX,Y2
		JBE		@OK2
		JMP		@EXIT
	@OK2:
		MOV		AX,Y1
		CMP		AX,Y2
		JNE		@NOTHORZ
		MOV		AX,X1
		CMP		AX,SI
		JAE		@1
		MOV		SI,AX
	@1:
		CMP		AX,DI
		JBE		@EXIT1
		MOV		DI,AX
	@EXIT1:
		MOV		AX,X2
		CMP		AX,SI
		JAE		@2
		MOV		SI,AX
	@2:
		CMP		AX,DI
		JBE		@EXIT2
		MOV		DI,AX
	@EXIT2:
		JMP		@EXIT
	@NOTHORZ:
		MOV		AX,X1
		CMP		AX,X2
		JNE		@NOTVERT
		MOV		AX,X1
		CMP		AX,SI
		JAE		@3
		MOV		SI,AX
	@3:
		CMP		AX,DI
		JBE		@EXIT3
		MOV		DI,AX
	@EXIT3:
		JMP		@EXIT
	@NOTVERT:
		MOV		AX,X2
		SUB		AX,X1
		MOV		BX,Y2
		SUB		BX,Y1
		MOV		DX,TY
		SUB		DX,Y1
		IMUL	DX
		IDIV	BX
		ADD		AX,X1
		CMP		AX,SI
		JAE		@4
		MOV		SI,AX
	@4:
		CMP		AX,DI
		JBE		@EXIT
		MOV		DI,AX
	@EXIT:
	End;

Asm
	MOV		AX,Y1
	CMP		AX,Y2
	JBE		@1
	MOV		AX,Y2
@1:
	CMP		AX,Y3
	JBE		@2
	MOV		AX,Y3
@2:
	CMP		AX,Y4
	JBE		@3
	MOV		AX,Y4
@3:
	MOV		MinY,AX
	MOV		AX,Y1
	CMP		AX,Y2
	JAE		@4
	MOV		AX,Y2
@4:
	CMP		AX,Y3
	JAE		@5
	MOV		AX,Y3
@5:
	CMP		AX,Y4
	JAE		@6
	MOV		AX,Y4
@6:
	MOV		MaxY,AX
	MOV		AX,MinY
	MOV		Y,AX
	MOV		AX,$0140
	MUL		Y
	MOV		O,AX
@LOOP:
	MOV		SI,$013F
	MOV		DI,$0000
	PUSH	Y
	PUSH	X1
	PUSH	Y1
	PUSH	X2
	PUSH	Y2
	PUSH	WORD PTR [BP]
	CALL	Check
	PUSH	Y
	PUSH	X2
	PUSH	Y2
	PUSH	X3
	PUSH	Y3
	PUSH	WORD PTR [BP]
	CALL	Check
	PUSH	Y
	PUSH	X3
	PUSH	Y3
	PUSH	X4
	PUSH	Y4
	PUSH	WORD PTR [BP]
	CALL	Check
	PUSH	Y
	PUSH	X4
	PUSH	Y4
	PUSH	X1
	PUSH	Y1
	PUSH	WORD PTR [BP]
	CALL	Check
	MOV		AX,O
	ADD		AX,SI
	MOV		CX,DI
	SUB		CX,SI
	INC		CX
	MOV		DI,AX
	MOV		AL,Color
	CLD
	MOV		ES,ScrSeg
@DOIT:
	MOV		AL,Color
	CMP		PutFn,pNORMAL
	JNE		@101
	REP		STOSB
	JMP		@NEXTLINE
@101:
	CMP		PutFn,pTRANSPARENT
	JNE		@10
	MOV		AH,AL
	CMP		AL,MaskColor
	JNE		@NEXT
	MOV		AH,ES:[DI]
	JMP		@NEXT
@10:
	CMP		PutFn,pXOR
	JNE		@11
	MOV		AH,ES:[DI]
	XOR		AH,AL
	JMP		@NEXT
@11:
	CMP		PutFn,pADD
	JNE		@12
	MOV		AH,ES:[DI]
	ADD		AH,AL
	JMP		@NEXT
@12:
	CMP		PutFn,pSUB
	JNE		@13
	MOV		AH,ES:[DI]
	SUB		AH,AL
	JMP		@NEXT
@13:
	CMP		PutFn,pOR
	JNE		@14
	MOV		AH,ES:[DI]
	OR		AH,AL
	MOV		AL,AH
	JMP		@NEXT
@14:
	CMP		PutFn,pAND
	JNE		@15
	MOV		AH,ES:[DI]
	AND		AH,AL
	JMP		@NEXT
@15:
	CMP		PutFn,pTRANSLATE
	JNE		@15B
	PUSH	DS
	PUSH	BX
	MOV		AL,ES:[DI]
	LDS		BX,DWORD PTR TransPtr
	XLAT
	POP		BX
	POP		DS
	MOV		AH,AL
	JMP		@NEXT
@15B:
	CMP		PutFn,pAPPROACH
	JNE		@16
	MOV		AH,ES:[DI]
	CMP		AH,AL
	JB		@UP
	JA		@DOWN
	JMP		@NEXT
@UP:
	INC		AH
	JMP		@NEXT
@DOWN:
	DEC		AH
	JMP		@NEXT
@16:
	CMP		PutFn,pINC
	JNE		@16B
	INC		AH
	JMP		@NEXT
@16B:
	CMP		PutFn,pDEC
	JNE		@16C
	DEC		AH
	JMP		@NEXT
@16C:
	CMP		PutFn,pONLY
	JNE		@NEXT
	MOV		AH,ES:[DI]
	CMP		AH,MaskColor
	JNE		@NEXT
	MOV		AH,AL
@NEXT:
	MOV		AL,AH
	STOSB
	LOOP	@DOIT2
	JMP		@NEXTLINE
@DOIT2:
	JMP		@DOIT
@NEXTLINE:
	MOV		AX,O
	ADD		AX,$0140
	MOV		O,AX
	INC		Y
	MOV		AX,Y
	CMP		AX,MaxY
	JBE		@LOOP
@EXIT:
End;

Procedure XDraw3(X1,Y1,X2,Y2,X3,Y3 : Word; Color : Byte); Assembler;
Asm
	PUSH	X1
	PUSH	Y1
	PUSH	X2
	PUSH	Y2
	MOV		AL,Color
	PUSH	AX
	CALL	XLine
	PUSH	X2
	PUSH	Y2
	PUSH	X3
	PUSH	Y3
	MOV		AL,Color
	PUSH	AX
	CALL	XLine
	PUSH	X3
	PUSH	Y3
	PUSH	X1
	PUSH	Y1
	MOV		AL,Color
	PUSH	AX
	CALL	XLine
End;

Procedure XDraw4(X1,Y1,X2,Y2,X3,Y3,X4,Y4 : Word; Color : Byte); Assembler;
Asm
	PUSH	X1
	PUSH	Y1
	PUSH	X2
	PUSH	Y2
	MOV		AL,Color
	PUSH	AX
	CALL	XLine
	PUSH	X2
	PUSH	Y2
	PUSH	X3
	PUSH	Y3
	MOV		AL,Color
	PUSH	AX
	CALL	XLine
	PUSH	X3
	PUSH	Y3
	PUSH	X4
	PUSH	Y4
	MOV		AL,Color
	PUSH	AX
	CALL	XLine
	PUSH	X4
	PUSH	Y4
	PUSH	X1
	PUSH	Y1
	MOV		AL,Color
	PUSH	AX
	CALL	XLine
End;

Const
	LastFileName	: String	= '';

Function XSaveImage(Var Data; FileName : String; Var Offset : LongInt; Compress : TCompress) : Integer;
Const
	CacheSize	= 4096;
Var
	Temp		: Integer;
	F			: File;
	InBuf		: Byte;
	Buf			: Array[1..4] Of Byte;
	P			: PChar;
	C			: Byte;
	IsPack		: Boolean;
	I			: Word;
	CurSize		: Word;
	MaxSize		: Word;
	PByte		: Byte;
	PCount		: Word;
	Compressed	: Boolean;
	Cache		: PChar;
	Head		: PChar;
	Tail		: PChar;
	ErrorCode	: Integer;
	Freq		: Array[Char] Of Word;
	Size		: LongInt;
Const
	Signature	: Array[1..4] Of Char =
		'GII'#26;
	DLE			: Byte = 26;

	Procedure Flush;
	Var
		Written	: Word;
	Begin
		BlockWrite(F,Cache^,Head-Cache,Written);
		If Written<>Head-Cache Then
			ErrorCode:=101;
		Head:=Cache;
	End;

	Procedure DoWrite(Var Data; Size : Word);
	Var
		P	: PChar;
		I	: Word;

		Procedure WriteByte(B : Char);
		Begin
			If (Head=Tail) Then
				Flush;
			Head^:=B;
			Inc(Head);
		End;

	Begin
		P:=@Data;
		For I:=1 To Size Do Begin
			WriteByte(P^);
			Inc(P);
		End;
	End;

	Function GetByte(Var C : Byte) : Boolean;
	Begin
		If MaxSize>0 Then Begin
			C:=Ord(P^);
			Inc(P);
			Dec(MaxSize);
			GetByte:=TRUE;
		End Else
			GetByte:=FALSE;
	End;

	Function PutByte(Var C : Byte) : Boolean;
	Begin
		DoWrite(C,1);
		PutByte:=TRUE;
	End;

Begin
	If FileName='' Then
		FileName:=LastFileName
	Else
		LastFileName:=FileName;
	Assign(F,FileName);
	{$I-}
	ReSet(F,1);
	{$I+}
	If IOResult<>0 Then Begin
		{$I-}
		ReWrite(F,1);
		Close(F);
		ReSet(F,1);
		{$I+}
	End;
	If Offset>FileSize(F) Then Begin
		Close(F);
		XSaveImage:=5;
		Exit;
	End;
	ErrorCode:=0;
	Seek(F,Offset);
	Temp:=IOResult;
	If Temp<>0 Then Begin
		XSaveImage:=Temp;
		Exit;
	End;
	GetMem(Cache,CacheSize);
	Head:=Cache;
	Tail:=Cache+CacheSize;
	DoWrite(Signature,4);
	DoWrite(Size,4);
	DoWrite(Data,4);
	P:=@Data;
	Inc(P,4);
	MaxSize:=XImageWidth(Data)*XImageHeight(Data);
	CurSize:=0;
	IsPack:=FALSE;
	InBuf:=0;
	Buf[1]:=1;
	Case Compress Of
		cNone		: Begin
			Buf[1]:=0;
			DoWrite(Buf[1],1);
			P:=@Data;
			Inc(P,4);
			DoWrite(P^,MaxSize);
			Compressed:=FALSE;
		End;
		cLZH		: Begin
			Buf[1]:=2;
			DoWrite(Buf[1],1);
			P:=@Data;
			Inc(P,4);
			I:=MaxSize;
			Pack(@GetByte,TRUE,@PutByte,TRUE);
			Compressed:=TRUE;
		End;
		cRLE		: Begin
			DoWrite(Buf[1],1);
			Compressed:=TRUE;
			For I:=1 To MaxSize Do Begin
				C:=Ord(P^);
				Inc(P);
				If IsPack Then Begin
					If C=PByte Then
						Inc(PCount)
					Else Begin
						Inc(CurSize,4);
						DoWrite(DLE,1);
						Buf[1]:=Hi(PCount);
						Buf[2]:=Lo(PCount);
						DoWrite(Buf,2);
						DoWrite(PByte,1);
						InBuf:=1;
						Buf[4]:=C;
						IsPack:=FALSE;
					End;
				End Else Begin
					Buf[1]:=Buf[2];
					Buf[2]:=Buf[3];
					Buf[3]:=Buf[4];
					Buf[4]:=C;
					Inc(InBuf);
					If InBuf=4 Then Begin
						If (Buf[1]=Buf[2]) And (Buf[2]=Buf[3]) And (Buf[3]=Buf[4]) Then Begin
							IsPack:=TRUE;
							PCount:=4;
							PByte:=C;
						End Else Begin
							DoWrite(Buf[1],1);
							Inc(CurSize);
							If Buf[1]=DLE Then Begin
								Buf[1]:=255;
								DoWrite(Buf[1],1);
								Inc(CurSize);
							End;
							Dec(InBuf);
						End;
					End;
				End;
				If CurSize>MaxSize Then Begin
					Buf[1]:=0;
					Flush;
					Seek(F,Offset+8);
					Truncate(F);
					DoWrite(Buf[1],1);
					P:=@Data;
					Inc(P,4);
					DoWrite(P^,MaxSize);
					Compressed:=FALSE;
					Break;
				End;
				If ErrorCode<>0 Then Begin
					Flush;
					FreeMem(Cache,CacheSize);
					Close(F);
					XSaveImage:=ErrorCode;
					Exit;
				End;
			End;
		End;
	End;
	If Compressed And (Compress=cRLE) Then Begin
		If IsPack Then Begin
			DoWrite(DLE,1);
			Buf[1]:=Hi(PCount);
			Buf[2]:=Lo(PCount);
			DoWrite(Buf,2);
			DoWrite(PByte,1);
		End Else Begin
			For I:=5-InBuf To 4 Do Begin
				DoWrite(Buf[I],1);
				If Buf[I]=DLE Then Begin
					Buf[I]:=255;
					DoWrite(Buf[I],1);
				End;
				Dec(InBuf);
			End;
		End;
	End;
	Flush;
	FreeMem(Cache,CacheSize);
	Size:=FilePos(F)-Offset;
	Truncate(F);
	Seek(F,Offset+4);
	BlockWrite(F,Size,4);
	Offset:=FileSize(F);
	Close(F);
	XSaveImage:=0;
End;

Function XSavePalette(Var RGB; FileName : String; Var Offset : LongInt) : Integer;
Var
	F		: File;
	Temp	: Integer;
	Written	: Word;
	Size	: LongInt;
Const
	Signature	: Array[1..4] Of Char =
		'GIP'#26;
Begin
	If FileName='' Then
		FileName:=LastFileName
	Else
		LastFileName:=FileName;
	Assign(F,FileName);
	{$I-}
	ReSet(F,1);
	{$I+}
	If IOResult<>0 Then Begin
		{$I-}
		ReWrite(F,1);
		Close(F);
		ReSet(F,1);
		{$I+}
	End;
	Temp:=IOResult;
	If Offset>FileSize(F) Then Begin
		Close(F);
		XSavePalette:=5;
		Exit;
	End;
	Seek(F,Offset);
	If Temp<>0 Then Begin
		XSavePalette:=Temp;
		Exit;
	End;
	BlockWrite(F,Signature,4);
	Size:=776;
	BlockWrite(F,Size,4);
	BlockWrite(F,RGB,768,Written);
	Offset:=FilePos(F);
	Truncate(F);
	Close(F);
	If Written<>768 Then
		XSavePalette:=101
	Else
		XSavePalette:=0;
End;

Function XLoadPalette(Var RGB; FileName : String; Var Offset : LongInt) : Integer;
Var
	F		: File;
	Temp	: Integer;
	OldMode	: Byte;
	Buf		: Array[1..4] Of Char;
	I		: Byte;
	Read	: Word;
	Size	: LongInt;
Const
	Signature	: Array[1..4] Of Char =
		'GIP'#26;
Begin
	If FileName='' Then
		FileName:=LastFileName
	Else
		LastFileName:=FileName;
	Assign(F,FileName);
	{$I-}
	OldMode:=FileMode;
	FileMode:=0;
	ReSet(F,1);
	FileMode:=OldMode;
	{$I+}
	Temp:=IOResult;
	If Temp<>0 Then Begin
		XLoadPalette:=Temp;
		Exit;
	End;
	If Offset>FileSize(F) Then Begin
		Close(F);
		XLoadPalette:=5;
		Exit;
	End;
	Seek(F,Offset);
	BlockRead(F,Buf,4);
	For I:=1 To 4 Do
		If Buf[I]<>Signature[I] Then Begin
			Close(F);
			XLoadPalette:=5;
			Exit;
		End;
	BlockRead(F,Size,4);
	If Size<>776 Then Begin
		Close(F);
		XLoadPalette:=5;
		Exit;
	End;
	BlockRead(F,RGB,768,Read);
	Offset:=FilePos(F);
	Close(F);
	If Read<>768 Then
		XLoadPalette:=100
	Else
		XLoadPalette:=0;
End;

Function XLoadImage(Var Data : Pointer; FileName : String; Var Offset : LongInt) : Integer;
Const
	CacheSize	= 4096;
Var
	F				: File;
	Cache			: PChar;
	Head			: PChar;
	Tail			: PChar;
	OldMode			: Byte;
	Width			: Word;
	Height			: Word;
	Buf				: Array[1..4] Of Char;
	Temp			: Integer;
	I				: Word;
	P				: PChar;
	C				: Char;
	Size			: Word;
	Count			: Word;
	BlockSize		: LongInt;
	TheBlockSize	: LongInt;
Const
	Signature	: Array[1..4] Of Char =
		'GII'#26;
	DLE			= #26;

	Procedure GetCache;
	Var
		Read	: Word;
	Begin
		BlockRead(F,Cache^,CacheSize,Read);
		Head:=Cache;
		Tail:=Cache+Read;
	End;

	Procedure DoRead(Var Data; Size : Word);
	Var
		P	: PChar;
		I	: Word;

		Procedure ReadByte(Var B : Char);
		Begin
			If (Head=Tail) Then
				GetCache;
			B:=Head^;
			Inc(Head);
		End;

	Begin
		P:=@Data;
		For I:=1 To Size Do Begin
			ReadByte(P^);
			Inc(P);
		End;
	End;

	Function GetByte(Var C : Byte) : Boolean;
	Begin
		If BlockSize>0 Then Begin
			Dec(BlockSize);
			DoRead(C,1);
			GetByte:=TRUE;
		End Else
			GetByte:=FALSE;
	End;

	Function PutByte(Var C : Byte) : Boolean;
	Begin
		P^:=Chr(C);
		Inc(P);
		PutByte:=TRUE;
	End;

Begin
	If FileName='' Then
		FileName:=LastFileName
	Else
		LastFileName:=FileName;
	Assign(F,FileName);
	{$I-}
	OldMode:=FileMode;
	FileMode:=0;
	ReSet(F,1);
	FileMode:=OldMode;
	{$I+}
	Temp:=IOResult;
	If Temp<>0 Then Begin
		XLoadImage:=Temp;
		Exit;
	End;
	If Offset>FileSize(F) Then Begin
		Close(F);
		XLoadImage:=5;
		Exit;
	End;
	Seek(F,Offset);
	BlockRead(F,Buf,4);
	For I:=1 To 4 Do
		If Buf[I]<>Signature[I] Then Begin
			Close(F);
			XLoadImage:=5;
			Exit;
		End;
	BlockRead(F,BlockSize,4);
	TheBlockSize:=BlockSize;
	BlockRead(F,Width,2);
	BlockRead(F,Height,2);
	BlockRead(F,C,1);
	Case C Of
		#0	: Begin
			LastLoaded:=cNone;
			GetMem(Data,Width*Height+4);
			Move(Width,Data^,2);
			Move(Height,(PChar(Data)+2)^,2);
			P:=Data;
			Inc(P,4);
			BlockRead(F,P^,Width*Height);
			Offset:=FilePos(F);
		End;
		#1	: Begin
			LastLoaded:=cRLE;
			GetMem(Cache,CacheSize);
			Head:=Cache;
			Tail:=Cache;
			GetMem(Data,Width*Height+4);
			P:=Data;
			Move(Width,P^,2);
			Inc(P,2);
			Move(Height,P^,2);
			Inc(P,2);
			Size:=Width*Height;
			While Size>0 Do Begin
				DoRead(C,1);
				If C=DLE Then Begin
					DoRead(C,1);
					If C=#255 Then Begin
						P^:=DLE;
						Inc(P);
						Dec(Size);
					End Else Begin
						Count:=Ord(C);
						DoRead(C,1);
						Count:=(Count Shl 8)+Ord(C);
						DoRead(C,1);
						FillChar(P^,Count,C);
						Inc(P,Count);
						Dec(Size,Count);
					End;
				End Else Begin
					P^:=C;
					Inc(P);
					Dec(Size);
				End;
			End;
			Offset:=FilePos(F)-(Tail-Cache)+(Head-Cache);
			FreeMem(Cache,CacheSize);
		End;
		#2	: Begin
			LastLoaded:=cLZH;
			GetMem(Cache,CacheSize);
			Head:=Cache;
			Tail:=Cache;
			GetMem(Data,Width*Height+4);
			P:=Data;
			Move(Width,P^,2);
			Inc(P,2);
			Move(Height,P^,2);
			Inc(P,2);
			Size:=Width*Height;
			Dec(BlockSize,13);
			LZH.UnPack(Size,@GetByte,TRUE,@PutByte,TRUE);
			Inc(Offset,TheBlockSize);
			FreeMem(Cache,CacheSize);
		End;
	End;
	Close(F);
	XLoadImage:=0;
End;

Function XSaveBinary(Var Binary; Size : Word; FileName : String; Var Offset : LongInt) : Integer;
Var
	F		: File;
	Temp	: Integer;
	Written	: Word;
	Size2	: LongInt;
Const
	Signature	: Array[1..4] Of Char =
		'GIB'#26;
Begin
	If FileName='' Then
		FileName:=LastFileName
	Else
		LastFileName:=FileName;
	Assign(F,FileName);
	{$I-}
	ReSet(F,1);
	{$I+}
	If IOResult<>0 Then Begin
		{$I-}
		ReWrite(F,1);
		Close(F);
		ReSet(F,1);
		{$I+}
	End;
	Temp:=IOResult;
	Seek(F,Offset);
	If Temp<>0 Then Begin
		XSaveBinary:=Temp;
		Exit;
	End;
	BlockWrite(F,Signature,4);
	Size2:=Size+10;
	BlockWrite(F,Size2,4);
	BlockWrite(F,Size,2);
	BlockWrite(F,Binary,Size,Written);
	Offset:=FilePos(F);
	Close(F);
	If Written<>Size Then
		XSaveBinary:=101
	Else
		XSaveBinary:=0;
End;


Function XLoadBinary(Var Binary : Pointer; Var Size : Word; FileName : String; Var Offset : LongInt) : Integer;
Var
	F		: File;
	Temp	: Integer;
	OldMode	: Byte;
	Buf		: Array[1..4] Of Char;
	I		: Byte;
	Read	: Word;
	Size2	: LongInt;
Const
	Signature	: Array[1..4] Of Char =
		'GIB'#26;
Begin
	If FileName='' Then
		FileName:=LastFileName
	Else
		LastFileName:=FileName;
	Assign(F,FileName);
	{$I-}
	OldMode:=FileMode;
	FileMode:=0;
	ReSet(F,1);
	FileMode:=OldMode;
	{$I+}
	Seek(F,Offset);
	Temp:=IOResult;
	If Temp<>0 Then Begin
		XLoadBinary:=Temp;
		Exit;
	End;
	BlockRead(F,Buf,4);
	For I:=1 To 4 Do
		If Buf[I]<>Signature[I] Then Begin
			Close(F);
			XLoadBinary:=5;
			Exit;
		End;
	BlockRead(F,Size2,4);
	BlockRead(F,Size,2);
	If Size2<>Size+6 Then Begin
		Close(F);
		XLoadBinary:=5;
		Exit;
	End;
	GetMem(Binary,Size);
	BlockRead(F,Binary^,Size,Read);
	Offset:=FilePos(F);
	Close(F);
	If Read<>Size Then
		XLoadBinary:=100
	Else
		XLoadBinary:=0;
End;

Function XAt(FileName : String; Offset : LongInt) : TAt;
Var
	F		: File;
	OldMode	: Byte;
	Buf		: Array[1..4] Of Char;
	Temp	: Integer;
	I		: Word;
	OK		: Boolean;
Const
	SignatureI	: Array[1..4] Of Char =
		'GII'#26;
	SignatureP	: Array[1..4] Of Char =
		'GIP'#26;
	SignatureB	: Array[1..4] Of Char =
		'GIB'#26;
	SignatureF	: Array[1..4] Of Char =
		'GFF'#26;
Begin
	If FileName='' Then
		FileName:=LastFileName
	Else
		LastFileName:=FileName;
	Assign(F,FileName);
	{$I-}
	OldMode:=FileMode;
	FileMode:=0;
	ReSet(F,1);
	FileMode:=OldMode;
	{$I+}
	Temp:=IOResult;
	If Temp<>0 Then Begin
		XAt:=tUnknown;
		Exit;
	End;
	Seek(F,Offset);
	If Offset>=FileSize(F) Then Begin
		XAt:=tEndOfFile;
		Close(F);
		Exit;
	End;
	BlockRead(F,Buf,4);
	Close(F);
	OK:=TRUE;
	If (Buf[1]='M') And (Buf[2]='Z') Then Begin
		XAt:=tExeFile;
		Exit;
	End;
	For I:=1 To 4 Do
		If (Buf[I]<>SignatureI[I]) Then
			OK:=FALSE;
	If OK Then Begin
		XAt:=tImage;
		Exit;
	End;
	OK:=TRUE;
	For I:=1 To 4 Do
		If (Buf[I]<>SignatureP[I]) Then
			OK:=FALSE;
	If OK Then Begin
		XAt:=tPalette;
		Exit;
	End;
	OK:=TRUE;
	For I:=1 To 4 Do
		If (Buf[I]<>SignatureB[I]) Then
			OK:=FALSE;
	If OK Then Begin
		XAt:=tBinary;
		Exit;
	End;
	OK:=TRUE;
	For I:=1 To 4 Do
		If (Buf[I]<>SignatureF[I]) Then
			OK:=FALSE;
	If OK Then Begin
		XAt:=tFont;
		Exit;
	End;
End;

Procedure XPutScaledImage(X,Y : Word; Var Image; ScaledWidth,ScaledHeight : Word); Assembler;
Var
	Width	: Word;
	Height	: Word;
	O1		: Word;
	X1		: Word;
	Y1		: Word;
	OrgDS	: Word;
Asm
	PUSH	DS
	MOV		OrgDS,DS
	CMP		ScaledWidth,0
	JE		@EXIT
	CMP		ScaledHeight,0
	JE		@EXIT
	MOV		AX,X
	ADD		AX,ScaledWidth
	CMP		AX,$0140
	JA		@EXIT
	MOV		AX,Y
	ADD		AX,ScaledHeight
	CMP		AX,$00C8
	JA		@EXIT
	MOV		ES,ScrSeg
	LDS		SI,Image
	CLD
	LODSW
	MOV		Width,AX
	LODSW
	MOV		Height,AX
	MOV		O1,SI
	MOV		AX,$0140
	MUL		Y
	ADD		AX,X
	MOV		DI,AX
	MOV		DX,ScaledWidth
	MOV		CX,ScaledHeight
	MOV		BX,$0140
	SUB		BX,DX
	MOV		AX,Y
	MOV		Y1,AX
@LOOP1:
	PUSH	CX
	MOV		X1,DI
	MOV		CX,DX
	PUSH	DX
	MOV		AX,Y1
	SUB		AX,Y
	MUL		Height
	DIV		ScaledHeight
	MUL		Width
	MOV		SI,O1
	ADD		SI,AX

@LOOP2:
	MOV		AX,DI
	SUB		AX,X1
	MUL		Width
	DIV		ScaledWidth
	PUSH	SI
	ADD		SI,AX
	MOV		AL,DS:[SI]
	POP		SI

	PUSH	DS
	MOV		DS,OrgDS
	CMP		PutFn,pNORMAL
	JNE		@0
	MOV		AH,AL
	JMP		@NEXT
@0:
	CMP		PutFn,pTRANSPARENT
	JNE		@01
	MOV		AH,AL
	CMP		AL,MaskColor
	JNE		@NEXT
	MOV		AH,ES:[DI]
	JMP		@NEXT
@01:
	CMP		PutFn,pXOR
	JNE		@1
	MOV		AH,ES:[DI]
	XOR		AH,AL
	JMP		@NEXT
@1:
	CMP		PutFn,pADD
	JNE		@2
	MOV		AH,ES:[DI]
	ADD		AH,AL
	JMP		@NEXT
@2:
	CMP		PutFn,pSUB
	JNE		@3
	MOV		AH,ES:[DI]
	SUB		AH,AL
	JMP		@NEXT
@3:
	CMP		PutFn,pOR
	JNE		@4
	MOV		AH,ES:[DI]
	OR		AH,AL
	MOV		AL,AH
	JMP		@NEXT
@4:
	CMP		PutFn,pAND
	JNE		@5
	MOV		AH,ES:[DI]
	AND		AH,AL
	JMP		@NEXT
@5:
	CMP		PutFn,pTRANSLATE
	JNE		@5B
	MOV		AH,ES:[DI]
	PUSH	DS
	PUSH	BX
	MOV		AL,ES:[DI]
	LDS		BX,DWORD PTR TransPtr
	XLAT
	POP		BX
	POP		DS
	MOV		AH,AL
	JMP		@NEXT
@5B:
	CMP		PutFn,pAPPROACH
	JNE		@6
	MOV		AH,ES:[DI]
	CMP		AH,AL
	JB		@UP
	JA		@DOWN
	JMP		@NEXT
@UP:
	INC		AH
	JMP		@NEXT
@DOWN:
	DEC		AH
	JMP		@NEXT
@6:
	CMP		PutFn,pINC
	JNE		@6B
	INC		AH
	JMP		@NEXT
@6B:
	CMP		PutFn,pDEC
	JNE		@6C
	DEC		AH
	JMP		@NEXT
@6C:
	CMP		PutFn,pONLY
	JNE		@NEXT
	MOV		AH,ES:[DI]
	CMP		AH,MaskColor
	JNE		@NEXT
	MOV		AH,AL
@NEXT:
	POP		DS
	MOV		AL,AH
	STOSB

	LOOP	@SKIP2
	JMP		@SKIP3
@SKIP2:
	JMP		@LOOP2
@SKIP3:

	INC		Y1
	ADD		DI,BX
	POP		DX
	POP		CX
	LOOP	@SKIP1
	JMP		@EXIT
@SKIP1:
	JMP		@LOOP1
@EXIT:
	POP		DS
End;

Procedure XScaleImage(Var Image1,Image2; ScaledWidth,ScaledHeight : Word); Assembler;
Var
	Width	: Word;
	Height	: Word;
	O1		: Word;
	X1		: Word;
	Y1		: Word;
Asm
	PUSH	DS
	CMP		ScaledWidth,0
	JE		@EXIT
	CMP		ScaledHeight,0
	JE		@EXIT
	CMP		ScaledWidth,$0140
	JA		@EXIT
	CMP		ScaledHeight,$00C8
	JA		@EXIT
	LDS		SI,Image1
	LES		DI,Image2
	CLD
	LODSW
	MOV		Width,AX
	LODSW
	MOV		Height,AX
	MOV		AX,ScaledWidth
	STOSW
	MOV		AX,ScaledHeight
	STOSW
	MOV		O1,SI
	MOV		DX,ScaledWidth
	MOV		CX,ScaledHeight
	XOR		AX,AX
	MOV		Y1,AX
@LOOP1:
	PUSH	CX
	MOV		X1,DI
	MOV		CX,DX
	PUSH	DX
	MOV		AX,Y1
	MUL		Height
	DIV		ScaledHeight
	MUL		Width
	MOV		SI,O1
	ADD		SI,AX

@LOOP2:
	MOV		AX,DI
	SUB		AX,X1
	MUL		Width
	DIV		ScaledWidth
	PUSH	SI
	ADD		SI,AX
	MOV		AL,DS:[SI]
	POP		SI

	STOSB
	LOOP	@LOOP2
	INC		Y1
	POP		DX
	POP		CX
	LOOP	@LOOP1
@EXIT:
	POP		DS
End;

Procedure XHFlipImage(Var Image); Assembler;
Asm
	PUSH	DS
	LDS		SI,Image
	LES		DI,Image
	CLD
	LODSW
	MOV		DX,AX
	LODSW
	MOV		CX,AX
	MOV		BX,DX
	SHR		BX,$01
@LOOP1:
	PUSH	CX
	PUSH	SI
	MOV		CX,BX
	MOV		DI,SI
	ADD		DI,DX
	DEC		DI
	STD
@LOOP2:
	MOV		AL,DS:[SI]
	MOV		AH,ES:[DI]
	MOV		DS:[SI],AH
	INC		SI
	STOSB
	LOOP	@LOOP2
	POP		SI
	POP		CX
	ADD		SI,DX
	LOOP	@LOOP1
@EXIT:
	POP		DS
End;

Procedure XVFlipImage(Var Image); Assembler;
Asm
	PUSH	DS
	LDS		SI,Image
	LES		DI,Image
	CLD
	LODSW
	MOV		DX,AX
	LODSW
	MOV		CX,AX
	PUSH	DX
	PUSH	CX
	MOV		AX,DX
	DEC		CX
	MUL		CX
	MOV		DI,SI
	ADD		DI,AX
	POP		CX
	SHR		CX,$01
	POP		DX
	CLD
@LOOP1:
	PUSH	CX
	MOV		CX,DX
	PUSH	DI
	PUSH	SI
@LOOP2:
	MOV		AL,DS:[SI]
	MOV		AH,ES:[DI]
	MOV		DS:[SI],AH
	INC		SI
	STOSB
	LOOP	@LOOP2
	POP		SI
	POP		DI
	POP		CX
	ADD		SI,DX
	SUB		DI,DX
	LOOP	@LOOP1
@EXIT:
	POP		DS
End;

Procedure XVHFlipImage(Var Image); Assembler;
Asm
	PUSH	DS
	LDS		SI,Image
	LES		DI,Image
	CLD
	LODSW
	MOV		DX,AX
	LODSW
	MOV		CX,AX
	PUSH	DX
	PUSH	CX
	MOV		AX,DX
	MUL		CX
	DEC		AX
	MOV		DI,SI
	ADD		DI,AX
	POP		CX
	MOV		CX,AX
	INC		CX
	SHR		CX,$01
	POP		DX
	STD
@LOOP1:
	MOV		AL,DS:[SI]
	MOV		AH,ES:[DI]
	MOV		DS:[SI],AH
	INC		SI
	STOSB
	LOOP	@LOOP1
@EXIT:
	POP		DS
End;

Const
	Col	: Byte	= 0;
	R1	: Byte	= 0;	R2	: Byte	= 63;
	G1	: Byte	= 0;	G2	: Byte	= 63;
	B1	: Byte	= 0;	B2	: Byte	= 63;
	CheckIt	: Boolean	= FALSE;

Procedure HRet; Assembler;
Asm
	CLI
	PUSHF
	PUSH	DX
	PUSH	AX
	MOV		DX,$03DA
@WAIT1:
	IN		AL,DX
	RCR		AL,$01
	JC		@WAIT1
@WAIT2:
	IN		AL,DX
	RCR		AL,$01
	JNC		@WAIT2
	POP		AX
	POP		DX
	POPF
	STI
End;

Procedure DoCheck;
Begin
	CheckIt:=FALSE;
	ScanTime(Col,R1,G1,B1,R2,G2,B2);
	CheckIt:=TRUE;
End;

Procedure VRet; Assembler;
Asm
	CMP		CheckIt,TRUE
	JNE		@NORMAL
	CALL	DoCheck
	JMP		@EXIT
@NORMAL:
	CLI
	PUSHF
	PUSH	ES
	PUSH	DX
	PUSH	AX
	PUSH	$0040
	POP		ES
	MOV		DX,ES:[$0063]
	ADD		DX,$0006
@LOOP1:
	IN		AL,DX
	TEST	AL,$08
	JNZ		@LOOP1
@LOOP2:
	IN		AL,DX
	TEST	AL,$08
	JZ		@LOOP2
	POP		AX
	POP		DX
	POP		ES
	POPF
	STI
@EXIT:
End;

Procedure ScanTime(Index,RealR,RealG,RealB,TimeR,TimeG,TimeB : Byte); Assembler;
Asm
	MOV		AH,$00
	MOV		AL,Index
	PUSH	AX
	MOV		AL,TimeR
	PUSH	AX
	MOV		AL,TimeG
	PUSH	AX
	MOV		AL,TimeB
	PUSH	AX
	CALL	XSetRGB
	CALL	VRet
	MOV		AH,$00
	MOV		AL,Index
	PUSH	AX
	MOV		AL,RealR
	PUSH	AX
	MOV		AL,RealG
	PUSH	AX
	MOV		AL,RealB
	PUSH	AX
	CALL	XSetRGB
End;

Procedure CfgRasterCheck(Index,RealR,RealG,RealB,TimeR,TimeG,TimeB : Byte); Assembler;
Asm
	MOV		AL,Index
	MOV		Col,AL
	MOV		AL,RealR
	MOV		R1,AL
	MOV		AL,RealG
	MOV		G1,AL
	MOV		AL,RealG
	MOV		B1,AL
	MOV		AL,TimeR
	MOV		R2,AL
	MOV		AL,TimeG
	MOV		G2,AL
	MOV		AL,TimeG
	MOV		B2,AL
End;

Procedure SetRasterCheck(On : Boolean); Assembler;
Asm
	MOV		AL,On
	MOV		CheckIt,AL
End;

Function RasterCheck : Boolean; Assembler;
Asm
	MOV		AL,CheckIt
End;

Procedure DelFile(FileName : String);
Var
	F	: File;
Begin
	Assign(F,FileName);
	{$I-}
	SetFAttr(F,0);
	Erase(F);
	{$I+}
End;

{ ////////////////////////////////////////////////////////////////////////// }

Function FDefault(Path : String; StandardName : NameStr;
	StandardExtension : ExtStr) : PathStr;
Var
	Dir : DirStr;
	Name : NameStr;
	Ext : ExtStr;
Begin
	FSplit(Path,Dir,Name,Ext);
	If (Dir<>'') And (Dir[Length(Dir)]<>'\') Then
		Dir:=Dir+'\';
	If Name='' Then
		Name:=StandardName;
	If Ext='' Then
		Ext:=StandardExtension;
	FDefault:=Dir+Name+Ext;
End;

Constructor OFont.Init(FontName : String; Height : Word);
Var
	C	: Char;
Begin
	If (Height=0) Or (Height>200) Or (FontName='') Then
		Fail;
	H:=Height;
	GetMem(Name,Length(FontName)+1);
	StrPCopy(Name,FontName);
	For C:=#0 To #255 Do
		Letters[C]:=NIL;
	Mono(FALSE,0,0);
	W1:=1;
	W2:=1;
	H1:=1;
	H2:=1;
	J1:=1;
	J2:=1;
End;

Constructor OFont.Read(FileName : String; Var Offset : LongInt);
Var
	F			: File;
	Buf			: Array[1..4] Of Char;
	I			: Word;
	C			: Char;
	OK			: Boolean;
	OldMode		: Byte;
	S			: String;
	Size		: LongInt;
Const
	Signature	: Array[1..4] Of Char =
		'GFF'#26;

	Function DoRead(Var Data; Size : Word) : Boolean;
	Var
		WasRead	: Word;
	Begin
		BlockRead(F,Data,Size,WasRead);
		DoRead:=FALSE;
		If Size<>WasRead Then Begin
			Close(F);
			Exit;
		End;
		DoRead:=TRUE;
	End;

	Procedure KillLetters(Code : Integer);
	Var
		C	: Char;
	Begin
		C:=#0;
		While Ord(C)<Code Do Begin
			UnAssign(C);
			If C=#255 Then
				Break
			Else
				Inc(C);
		End;
	End;

Begin
	If FileName='' Then
		FileName:=LastFileName
	Else
		LastFileName:=FileName;
	FileName:=FDefault(FExpand(FileName),'FONT','.GFF');
	System.Assign(F,FileName);
	Mono(FALSE,0,0);
	{$I-}
	OldMode:=FileMode;
	FileMode:=0;
	ReSet(F,1);
	FileMode:=OldMode;
	{$I+}
	If IOResult<>0 Then
		Fail;
	If Offset>FileSize(F) Then Begin
		Close(F);
		Fail;
	End;
	Seek(F,Offset);
	If Not DoRead(Buf,4) Then
		Fail;
	For I:=1 To 4 Do
		If Buf[I]<>Signature[I] Then Begin
			Close(F);
			Fail;
		End;
	If Not DoRead(Size,SizeOf(Size)) Then
		Fail;
	If Not DoRead(H,SizeOf(H)) Then
		Fail;
	If Not DoRead(S,1) Then
		Fail;
	If Not DoRead(S[1],Length(S)) Then
		Fail;
	For C:=#0 To #255 Do
		Letters[C]:=NIL;
	For C:=#0 To #255 Do Begin
		If Not DoRead(OK,1) Then Begin
			KillLetters(Ord(C));
			Fail;
		End;
		If OK Then Begin
			If Not DoRead(Buf,4) Then
				Fail;
			GetMem(Letters[C],XImageMemSize(Buf));
			If Letters[C]=NIL Then Begin
				KillLetters(Ord(C));
				Fail;
			End;
			Move(Buf,Letters[C]^,4);
			If Not DoRead((PChar(Letters[C])+4)^,XImageMemSize(Buf)-4) Then
				Fail;
		End;
	End;
	Offset:=FilePos(F);
	Close(F);
	GetMem(Name,Length(S)+1);
	If Name=NIL Then Begin
		KillLetters(256);
		Fail;
	End;
	StrPCopy(Name,S);
	W1:=1;
	W2:=1;
	H1:=1;
	H2:=1;
	J1:=1;
	J2:=1;
End;

Procedure OFont.Mono(Use : Boolean; TranspColor,NewColor : Byte);
Begin
	TColor:=TranspColor;
	Color:=NewColor;
	UseMono:=Use;
End;

Destructor OFont.Done;
Var
	C	: Char;
Begin
	FreeMem(Name,StrLen(Name)+1);
	For C:=#0 To #255 Do
		UnAssign(C);
End;

Function OFont.GetName : String;
Begin
	GetName:=StrPas(Name);
End;

Constructor OFont.Load(FileName : String);
Var
	Offset	: LongInt;
Begin
	Offset:=0;
	If Not Read(FileName,Offset) Then
		Fail;
End;

Function OFont.Save(FileName : String) : Integer;
Var
	Offset	: LongInt;
Begin
	Offset:=0;
	Save:=Write(FileName,Offset);
End;

Function OFont.Write(FileName : String; Var Offset : LongInt) : Integer;
Var
	F			: File;
	Temp		: Integer;
	Written		: Word;
	S			: String;
	C			: Char;
	OK			: Boolean;
	OldOffset	: LongInt;
	Size		: LongInt;
Const
	Signature	: Array[1..4] Of Char =
		'GFF'#26;

	Function DoWrite(Var Data; Size : Word) : Boolean;
	Var
		Written	: Word;
	Begin
		BlockWrite(F,Data,Size,Written);
		DoWrite:=FALSE;
		If Written<>Size Then Begin
			Write:=101;
			Close(F);
			Exit;
		End;
		DoWrite:=TRUE;
	End;

Begin
	If FileName='' Then
		FileName:=LastFileName
	Else
		LastFileName:=FileName;
	FileName:=FDefault(FExpand(FileName),'FONT','.GFF');
	System.Assign(F,FileName);
	{$I-}
	ReSet(F,1);
	{$I+}
	If IOResult<>0 Then Begin
		{$I-}
		ReWrite(F,1);
		{$I+}
	End;
	Temp:=InOutRes;
	If IOResult<>0 Then Begin
		Write:=Temp;
		Exit;
	End;
	If Offset>FileSize(F) Then Begin
		Close(F);
		Exit;
	End;
	OldOffset:=Offset;
	If Not DoWrite(Signature,4) Then
		Exit;
	If Not DoWrite(Size,4) Then
		Exit;
	If Not DoWrite(H,SizeOf(H)) Then
		Exit;
	S:=GetName;
	If Not DoWrite(S,Length(S)+1) Then
		Exit;
	For C:=#0 To #255 Do Begin
		OK:=(Letters[C]<>NIL);
		If Not DoWrite(OK,1) Then
			Exit;
		If OK Then
			If Not DoWrite(Letters[C]^,XImageMemSize(Letters[C]^)) Then
				Exit;
	End;
	Offset:=FilePos(F);
	Size:=Offset-OldOffset;
	Seek(F,OldOffset+4);
	BlockWrite(F,Size,4);
	Close(F);
	Write:=0;
End;

Function OFont.GetLetter(Letter : Char; Var Image : Pointer) : Boolean;
Begin
	If Letters[Letter]<>NIL Then Begin
		GetMem(Image,XImageMemSize(Letters[Letter]^));
		Move(Letters[Letter]^,Image^,XImageMemSize(Letters[Letter]^));
		GetLetter:=TRUE;
	End Else
		GetLetter:=FALSE;
End;

Function OFont.Assign(Letter : Char; Var Image) : Boolean;
Begin
	UnAssign(Letter);
	GetMem(Letters[Letter],XImageMemSize(Image));
	If Letters[Letter]<>NIL Then
		Move(Image,Letters[Letter]^,XImageMemSize(Image));
	Assign:=(Letters[Letter]<>NIL);
End;

Procedure OFont.UnAssign(Letter : Char);
Begin
	If Letters[Letter]<>NIL Then Begin
		FreeMem(Letters[Letter],XImageMemSize(Letters[Letter]^));
		Letters[Letter]:=NIL;
	End;
End;

Function OFont.Grab(Letter : Char; Y,X1,X2 : Word) : Boolean;
Var
	P	: Pointer;
Begin
	Grab:=FALSE;
	GetMem(P,XImageSize(X1,Y,X2,Y+GetHeight-1));
	If P=NIL Then
		Exit;
	XGetImage(X1,Y,X2,Y+GetHeight-1,P^);
	Grab:=Assign(Letter,P^);
	FreeMem(P,XImageMemSize(P^));
End;

Function OFont.GetHeight : Word;
Begin
	GetHeight:=H;
End;

Function OFont.TextHeight(S : String) : Word;
Begin
	TextHeight:=(GetHeight*H1) Div H2;
End;

Function OFont.TextWidth(S : String) : Word;
Var
	TempW	: Word;
	I		: Word;
Begin
	TempW:=0;
	For I:=1 To Length(S) Do Begin
		If Assigned(S[I]) Then
			Inc(TempW,(XImageWidth(Letters[S[I]]^)*W1) Div W2);
	End;
	TextWidth:=TempW;
End;

Procedure OFont.SetScale(MultW,DivW,MultH,DivH : Word);
Begin
	W1:=MultW;	W2:=DivW;
	H1:=MultH;	H2:=DivH;
End;

Function OFont.Assigned(Letter : Char) : Boolean;
Begin
	Assigned:=(Letters[Letter]<>NIL);
End;

Procedure OFont.Display(X,Y : Word; Text : String);
Var
	W		: Word;
	I		: Word;
	Image	: Pointer;
	Trans	: TTranslate;
Begin
	If J1=2 Then Begin
		If X<TextWidth(Text) Then
			Exit
		Else
			Dec(X,TextWidth(Text));
	End Else If J1=0 Then Begin
		If X<TextWidth(Text) Div 2 Then
			Exit
		Else
			Dec(X,TextWidth(Text) Div 2);
	End;
	If J2=2 Then Begin
		If Y<TextHeight(Text) Then
			Exit
		Else
			Dec(Y,TextHeight(Text));
	End Else If J2=0 Then Begin
		If Y<TextHeight(Text) Div 2 Then
			Exit
		Else
			Dec(Y,TextHeight(Text) Div 2);
	End;
	If UseMono Then Begin
		FillChar(Trans,SizeOf(Trans),Color);
		Trans[TColor]:=TColor;
	End;
	For I:=1 To Length(Text) Do
		If Assigned(Text[I]) Then Begin
			Image:=Letters[Text[I]];
			If UseMono Then Begin
				GetMem(Image,XImageMemSize(Letters[Text[I]]^));
				Move(Letters[Text[I]]^,Image^,XImageMemSize(Letters[Text[I]]^));
				XTranslateImage(Image^,Trans);
			End;
			If (W1<>1) Or (W2<>1) Or (H1<>1) Or (H2<>1) Then Begin
				W:=(XImageWidth(Image^)*W1) Div W2;
				XPutScaledImage(X,Y,Image^,
					W,(H*H1) Div H2);
			End Else Begin
				XPutImage(X,Y,Image^);
				W:=XImageWidth(Image^);
			End;
			If UseMono Then
				FreeMem(Image,XImageMemSize(Image^));
			Inc(X,W);
		End;
End;

Procedure OFont.SetJustify(HorzJustify,VertJustify : Word);
Begin
	J1:=HorzJustify;
	J2:=VertJustify;
End;

Constructor OPalette.Init(GrabCurrent : Boolean);
Begin
	FillChar(Data,SizeOf(Data),#0);
	If GrabCurrent Then
		Grab;
End;

Destructor OPalette.Done;
Begin
End;

Procedure OPalette.SetRGB(Index,R,G,B : Byte);
Begin
	Data[Index,1]:=R;
	Data[Index,2]:=G;
	Data[Index,3]:=B;
End;

Procedure OPalette.Realize;
Begin
	XSetRGBMany(0,256,Data);
End;

Function OPalette.Red(Index : Byte) : Byte;
Begin
	Red:=Data[Index,1];
End;

Function OPalette.Green(Index : Byte) : Byte;
Begin
	Green:=Data[Index,2];
End;

Function OPalette.Blue(Index : Byte) : Byte;
Begin
	Blue:=Data[Index,3];
End;

Procedure OPalette.Grab;
Begin
	XGetRGBMany(0,256,Data);
End;

Procedure OPalette.Get(Var RGB);
Begin
	Move(Data,RGB,SizeOf(Data));
End;

Procedure XDrawPoly(Num : Word; Points : Array Of TPoint; Color : Byte);
Var
	I,J	: Word;
Begin
	For I:=0 To Num-1 Do Begin
		J:=(I+1) Mod Num;
		XLine(Points[I].X,Points[I].Y,Points[J].X,Points[J].Y,Color);
	End;
End;

Procedure XFillPoly(Num : Word; Points : Array Of TPoint; Color : Byte);
Var
	P1,P2		: Word;
	Scan1,Scan2	: Word;
	DDX1,DDY1	: LongInt;
	DDX2,DDY2	: LongInt;
	YMin,YMax	: LongInt;
	X1,Y1,X2,Y2	: LongInt;
	I,Y			: Word;
	XX1,XX2		: Word;

	Procedure ProperPoint(Var P,Scans : Word; Var DDX,DDY : LongInt; Direction : Boolean);
	Var
		PN	: Word;
	Begin
		If Direction Then
			PN:=(P+1) Mod Num
		Else
			PN:=(P+Num-1) Mod Num;
		If Points[PN].Y=Points[P].Y Then Begin
			P:=PN;
			ProperPoint(P,Scans,DDX,DDY,Direction);
		End Else Begin
			DDX:=Points[PN].X;
			Dec(DDX,Points[P].X);
			DDX:=DDX Shl TextureShift;
			DDY:=Points[PN].Y;
			Dec(DDY,Points[P].Y);
			Scans:=Abs(DDY);
			DDY:=DDY Shl TextureShift;
			DDX:=DDX Div Scans;
			DDY:=DDY Div Scans;
		End;
	End;

	Procedure Adjust(Var X,Y : LongInt; Var P,Scans : Word; Var DDX,DDY : LongInt; Direction : Boolean);
	Begin
		Inc(X,DDX);
		Inc(Y,DDY);
		Dec(Scans);
		If Scans=0 Then Begin
			If Direction Then
				P:=(P+1) Mod Num
			Else
				P:=(P+Num-1) Mod Num;
			ProperPoint(P,Scans,DDX,DDY,Direction);
			X:=LongInt(Points[P].X) Shl TextureShift;
			Y:=LongInt(Points[P].Y) Shl TextureShift;
		End;
	End;

	Function Min(A,B : Word) : Word; Assembler;
	Asm
		MOV		AX,A
		CMP		AX,B
		JBE		@EXIT
		MOV		AX,B
	@EXIT:
	End;

	Function Max(A,B : Word) : Word; Assembler;
	Asm
		MOV		AX,A
		CMP		AX,B
		JAE		@EXIT
		MOV		AX,B
	@EXIT:
	End;

Begin
	P1:=0;
	P2:=0;
	For I:=2 To Num Do Begin
		If Points[I-1].Y<Points[P1].Y Then
			P1:=I-1;
		If Points[I-1].Y>Points[P2].Y Then
			P2:=I-1;
	End;
	If Points[P1].Y=Points[P2].Y Then Begin
		XX1:=319;
		XX2:=0;
		For I:=1 To Num Do Begin
			If Points[I-1].X<XX1 Then
				XX1:=Points[I-1].X;
			If Points[I-1].X>XX2 Then
				XX2:=Points[I-1].X;
		End;
		XHLine(Points[P1].Y,XX1,XX2,Color);
	End Else Begin
		YMin:=Points[P1].Y;
		YMax:=Points[P2].Y;
		If (YMin<0) Or (YMin>199) Or (YMax<0) Or (YMax>199) Then
			Exit;
		P2:=P1;
		ProperPoint(P1,Scan1,DDX1,DDY1,TRUE);
		ProperPoint(P2,Scan2,DDX2,DDY2,FALSE);
		X1:=LongInt(Points[P1].X) Shl TextureShift;
		Y1:=LongInt(Points[P1].Y) Shl TextureShift;
		X2:=LongInt(Points[P2].X) Shl TextureShift;
		Y2:=LongInt(Points[P2].Y) Shl TextureShift;
		For Y:=YMin To YMax Do Begin
			XX1:=X1 Shr TextureShift;
			XX2:=X2 Shr TextureShift;
			XHLine(Y,Min(XX1,XX2),Max(XX1,XX2),Color);
			Adjust(X1,Y1,P1,Scan1,DDX1,DDY1,TRUE);
			Adjust(X2,Y2,P2,Scan2,DDX2,DDY2,FALSE);
		End;
	End;
End;

Procedure XMapTexture(Var Texture; Num : Word; SPoints,DPoints : Array Of TPoint);
Var
	P1,P2			: Word;
	Scan1,Scan2		: Word;
	SDX1,SDY1		: LongInt;
	SDX2,SDY2		: LongInt;
	DDX1,DDY1		: LongInt;
	DDX2,DDY2		: LongInt;
	YMin,YMax		: LongInt;
	DX1,DY1,DX2,DY2	: LongInt;
	SX1,SY1,SX2,SY2	: LongInt;
	X,Y				: Word;
	XX1,XX2			: Word;
	TDX,TDY			: LongInt;
	TX,TY			: LongInt;
	P				: PChar;
	W,H,I			: Word;
	Degenerate		: Boolean;
	C				: Byte;

	Procedure ProperPoint(Var P,Scans : Word; Var SDX,SDY,DDX,DDY : LongInt; Direction : Boolean);
	Var
		PN	: Word;
	Begin
		If Direction Then
			PN:=(P+1) Mod Num
		Else
			PN:=(P+Num-1) Mod Num;
		If DPoints[PN].Y=DPoints[P].Y Then Begin
			P:=PN;
			ProperPoint(P,Scans,SDX,SDY,DDX,DDY,Direction);
		End Else Begin
			DDX:=DPoints[PN].X;
			Dec(DDX,DPoints[P].X);
			DDX:=DDX Shl TextureShift;
			DDY:=DPoints[PN].Y;
			Dec(DDY,DPoints[P].Y);
			Scans:=Abs(DDY);
			DDY:=DDY Shl TextureShift;
			DDX:=DDX Div Scans;
			DDY:=DDY Div Scans;
			SDX:=SPoints[PN].X;
			Dec(SDX,SPoints[P].X);
			SDX:=SDX Shl TextureShift;
			SDY:=SPoints[PN].Y;
			Dec(SDY,SPoints[P].Y);
			SDY:=SDY Shl TextureShift;
			SDX:=SDX Div Scans;
			SDY:=SDY Div Scans;
		End;
	End;

	Procedure Adjust(Var SX,SY,DX,DY : LongInt; Var P,Scans : Word; Var SDX,SDY,DDX,DDY : LongInt; Direction : Boolean);
	Begin
		Inc(SX,SDX);
		Inc(SY,SDY);
		Inc(DX,DDX);
		Inc(DY,DDY);
		Dec(Scans);
		If Scans=0 Then Begin
			If Direction Then
				P:=(P+1) Mod Num
			Else
				P:=(P+Num-1) Mod Num;
			ProperPoint(P,Scans,SDX,SDY,DDX,DDY,Direction);
			DX:=LongInt(DPoints[P].X) Shl TextureShift;
			DY:=LongInt(DPoints[P].Y) Shl TextureShift;
			SX:=LongInt(SPoints[P].X) Shl TextureShift;
			SY:=LongInt(SPoints[P].Y) Shl TextureShift;
		End;
	End;

	Function Min(A,B : Word) : Word; Assembler;
	Asm
		MOV		AX,A
		CMP		AX,B
		JBE		@EXIT
		MOV		AX,B
	@EXIT:
	End;

	Function Max(A,B : Word) : Word; Assembler;
	Asm
		MOV		AX,A
		CMP		AX,B
		JAE		@EXIT
		MOV		AX,B
	@EXIT:
	End;

	Function GetTranslatedPixel(X,Y : Word) : Byte;
	Begin
		If (X>W-1) Then
			X:=W-1;
		If Y>H-1 Then
			Y:=H-1;
		GetTranslatedPixel:=Ord((P+Y*W+X)^);
	End;

Begin
	P:=@Texture;
	W:=XImageWidth(Texture);
	H:=XImageHeight(Texture);
	Inc(P,4);
	P1:=0;
	P2:=0;
	Degenerate:=TRUE;
	For I:=1 To Num Do Begin
		If SPoints[I-1].X>W Then
			Exit;
		If SPoints[I-1].Y>H Then
			Exit;
		If DPoints[I-1].X>319 Then
			Exit;
		If DPoints[I-1].Y>199 Then
			Exit;
		If DPoints[I-1].Y<DPoints[P1].Y Then
			P1:=I-1;
		If DPoints[I-1].Y>DPoints[P2].Y Then
			P2:=I-1;
		If DPoints[I-1].Y<>DPoints[I Mod Num].Y Then
			Degenerate:=FALSE;
	End;
	If Degenerate Then Begin
		YMin:=DPoints[P1].Y;
		YMax:=DPoints[P2].Y;
		P2:=P1;
		Scan1:=2;
		SDX1:=1;
		SDY1:=1;
		DDX1:=1;
		DDY1:=1;
		Scan2:=2;
		SDX2:=1;
		SDY2:=1;
		DDX2:=1;
		DDY2:=1;
	End Else Begin
		YMin:=DPoints[P1].Y;
		YMax:=DPoints[P2].Y;
		P2:=P1;
		ProperPoint(P1,Scan1,SDX1,SDY1,DDX1,DDY1,TRUE);
		ProperPoint(P2,Scan2,SDX2,SDY2,DDX2,DDY2,FALSE);
	End;
	DX1:=LongInt(DPoints[P1].X) Shl TextureShift;
	DY1:=LongInt(DPoints[P1].Y) Shl TextureShift;
	DX2:=LongInt(DPoints[P2].X) Shl TextureShift;
	DY2:=LongInt(DPoints[P2].Y) Shl TextureShift;
	SX1:=LongInt(SPoints[P1].X) Shl TextureShift;
	SY1:=LongInt(SPoints[P1].Y) Shl TextureShift;
	SX2:=LongInt(SPoints[P2].X) Shl TextureShift;
	SY2:=LongInt(SPoints[P2].Y) Shl TextureShift;
	For Y:=YMin To YMax Do Begin
		XX1:=DX1 Shr TextureShift;
		XX2:=DX2 Shr TextureShift;
		TX:=SX2;
		TY:=SY2;
		If XX2=XX1 Then Begin
			TDX:=0;
			TDY:=0;
		End Else Begin
			TDX:=(SX1-SX2) Div (LongInt(XX1)-LongInt(XX2));
			TDY:=(SY1-SY2) Div (LongInt(XX1)-LongInt(XX2));
		End;
		If XX2>XX1 Then Begin
			XX1:=XX1 XOr XX2;
			XX2:=XX2 XOr XX1;
			XX1:=XX1 XOr XX2;
			TX:=SX1;
			TY:=SY1;
		End;
		For X:=XX2 To XX1 Do Begin
			C:=GetTranslatedPixel(TX Shr TextureShift,TY Shr TextureShift);
			If PutFn=pTRANSLATE Then Begin
				C:=TransTabl[C];
				PutFn:=pNORMAL;
				XPlot(X,Y,C);
				PutFn:=pTRANSLATE;
			End Else
				XPlot(X,Y,C);
			Inc(TX,TDX);
			Inc(TY,TDY);
		End;
		Adjust(SX1,SY1,DX1,DY1,P1,Scan1,SDX1,SDY1,DDX1,DDY1,TRUE);
		Adjust(SX2,SY2,DX2,DY2,P2,Scan2,SDX2,SDY2,DDX2,DDY2,FALSE);
	End;
End;

Procedure XSetDefaultPalette;
Var
	RGB	: TRGB;
Begin
	XDefaultPalette(RGB);
	XSetRGBMany(0,256,RGB);
End;

Procedure XCircle(X,Y,R : Word; Color : Byte);
Var
	X1,Y1	: Word;
	D		: Integer;

	Procedure CirclePoint(X,Y,X1,Y1 : Word);

		Procedure CheckPoint(X,Y : Integer);
		Begin
			If (Y<0) Or (Y>199) Or (X<0) Or (X>319) Then
				Exit;
			XPlot(X,Y,Color);
		End;

	Begin
		CheckPoint(X-X1,Y-Y1);
		CheckPoint(X+X1,Y-Y1);
		CheckPoint(X-X1,Y+Y1);
		CheckPoint(X+X1,Y+Y1);
		CheckPoint(X-Y1,Y-X1);
		CheckPoint(X+Y1,Y-X1);
		CheckPoint(X-Y1,Y+X1);
		CheckPoint(X+Y1,Y+X1);
	End;

Begin
	X1:=0;
	Y1:=R;
	D:=3;
	Dec(D,2*R);
	While X1<Y1 Do Begin
		CirclePoint(X,Y,X1,Y1);
		If D<0 Then
			Inc(D,4*X1+6)
		Else Begin
			Inc(D,4*(X1-Y1)+10);
			Dec(Y1);
		End;
		Inc(X1);
	End;
	If X1=Y1 Then
		CirclePoint(X,Y,X1,Y1);
End;

Procedure XEllipse(X,Y,RW,RH : Word; Color : Byte);
Var
	X1,X2		: Integer;
	X3,X4		: Integer;
	Y1			: Word;
	D			: Integer;
	Min,Max		: Array[-1..200] Of Integer;
	OK			: Array[-1..200] Of Boolean;
	YMin		: Integer;
	YMax		: Integer;
	YMiddle		: Integer;

	Procedure CirclePoint(X,Y,X1,Y1 : Word);

		Procedure CheckPoint(X,Y : Integer);
		Begin
			If (Y<-1) Or (Y>200) Then
				Exit;
			OK[Y]:=TRUE;
			If X<Min[Y] Then
				Min[Y]:=X;
			If X>Max[Y] Then
				Max[Y]:=X;
		End;

	Begin
		CheckPoint(X-X1,Y-(Y1*RH) Div RW);
		CheckPoint(X+X1,Y-(Y1*RH) Div RW);
		CheckPoint(X-X1,Y+(Y1*RH) Div RW);
		CheckPoint(X+X1,Y+(Y1*RH) Div RW);
		CheckPoint(X-Y1,Y-(X1*RH) Div RW);
		CheckPoint(X+Y1,Y-(X1*RH) Div RW);
		CheckPoint(X-Y1,Y+(X1*RH) Div RW);
		CheckPoint(X+Y1,Y+(X1*RH) Div RW);
	End;

Begin
	For Y1:=0 To 199 Do Begin
		Min[Y1]:=319;
		Max[Y1]:=0;
		Ok[Y1]:=FALSE;
	End;
	YMin:=Y-RH;
	YMax:=Y+RH;
	YMiddle:=Y;
	X1:=0;
	Y1:=RW;
	D:=3;
	Dec(D,2*RW);
	While X1<Y1 Do Begin
		CirclePoint(X,Y,X1,Y1);
		If D<0 Then
			Inc(D,4*X1+6)
		Else Begin
			Inc(D,4*(X1-Y1)+10);
			Dec(Y1);
		End;
		Inc(X1);
	End;
	If X1=Y1 Then
		CirclePoint(X,Y,X1,Y1);
	For Y:=0 To 199 Do Begin
		If Ok[Y] Then Begin
			If (Max[Y]>=0) Or (Min[Y]<=319) Then Begin
				If (Y=YMin) Or (Y=YMax) Then Begin
					X1:=Min[Y];
					If X1<0 Then
						X1:=0;
					X2:=Max[Y];
					If X2>319 Then
						X2:=319;
					XHLine(Y,X1,X2,Color);
				End Else Begin
					If Y>YMiddle Then Begin
						If Min[Y+1]-Min[Y]<=1 Then Begin
							If Min[Y]>=0 Then
								XPlot(Min[Y],Y,Color);
							If Max[Y]<=319 Then
								XPlot(Max[Y],Y,Color);
						End Else Begin
							X1:=Min[Y];
							If X1<0 Then
								X1:=0;
							X2:=Max[Y];
							If X2>319 Then
								X2:=319;
							If Min[Y]>=0 Then
								XHLine(Y,X1,Min[Y+1]-1,Color);
							If Max[Y]<=319 Then
								XHLine(Y,Max[Y+1]+1,X2,Color);
						End;
					End Else Begin
						If Min[Y-1]-Min[Y]<=1 Then Begin
							If Min[Y]>=0 Then
								XPlot(Min[Y],Y,Color);
							If Max[Y]<=319 Then
								XPlot(Max[Y],Y,Color);
						End Else Begin
							X1:=Min[Y];
							If X1<0 Then
								X1:=0;
							X2:=Max[Y];
							If X2>319 Then
								X2:=319;
							If Min[Y]>=0 Then
								XHLine(Y,X1,Min[Y-1]-1,Color);
							If Max[Y]<=319 Then
								XHLine(Y,Max[Y-1]+1,X2,Color);
						End;
					End;
				End;
			End;
		End;
	End;
End;

Procedure XFillEllipse(X,Y,RW,RH : Word; Color : Byte);
Var
	X1,X2		: Integer;
	Y1			: Word;
	D			: Integer;
	Min,Max		: Array[0..199] Of Integer;
	OK			: Array[0..199] Of Boolean;
	YMin		: Integer;
	YMax		: Integer;
	YMiddle		: Integer;

	Procedure CirclePoint(X,Y,X1,Y1 : Word);

		Procedure CheckPoint(X,Y : Integer);
		Begin
			If (Y<0) Or (Y>199) Then
				Exit;
			OK[Y]:=TRUE;
			If X<Min[Y] Then
				Min[Y]:=X;
			If X>Max[Y] Then
				Max[Y]:=X;
		End;

	Begin
		CheckPoint(X-X1,Y-(Y1*RH) Div RW);
		CheckPoint(X+X1,Y-(Y1*RH) Div RW);
		CheckPoint(X-X1,Y+(Y1*RH) Div RW);
		CheckPoint(X+X1,Y+(Y1*RH) Div RW);
		CheckPoint(X-Y1,Y-(X1*RH) Div RW);
		CheckPoint(X+Y1,Y-(X1*RH) Div RW);
		CheckPoint(X-Y1,Y+(X1*RH) Div RW);
		CheckPoint(X+Y1,Y+(X1*RH) Div RW);
	End;

Begin
	For Y1:=0 To 199 Do Begin
		Min[Y1]:=319;
		Max[Y1]:=0;
		Ok[Y1]:=FALSE;
	End;
	YMin:=Y-RW;
	YMax:=Y+RW;
	YMiddle:=Y;
	X1:=0;
	Y1:=RW;
	D:=3;
	Dec(D,2*RW);
	While X1<Y1 Do Begin
		CirclePoint(X,Y,X1,Y1);
		If D<0 Then
			Inc(D,4*X1+6)
		Else Begin
			Inc(D,4*(X1-Y1)+10);
			Dec(Y1);
		End;
		Inc(X1);
	End;
	If X1=Y1 Then
		CirclePoint(X,Y,X1,Y1);
	For Y:=0 To 199 Do Begin
		If Ok[Y] Then Begin
			If (Max[Y]>=0) Or (Min[Y]<=319) Then Begin
				X1:=Min[Y];
				If X1<0 Then
					X1:=0;
				X2:=Max[Y];
				If X2>319 Then
					X2:=319;
				XHLine(Y,X1,X2,Color);
			End;
		End;
	End;
End;

Procedure XFillCircle(X,Y,R : Word; Color : Byte);
Var
	X1,X2		: Integer;
	Y1			: Word;
	D			: Integer;
	Min,Max		: Array[0..199] Of Integer;
	OK			: Array[0..199] Of Boolean;
	YMin		: Integer;
	YMax		: Integer;
	YMiddle		: Integer;

	Procedure CirclePoint(X,Y,X1,Y1 : Word);

		Procedure CheckPoint(X,Y : Integer);
		Begin
			If (Y<0) Or (Y>199) Then
				Exit;
			OK[Y]:=TRUE;
			If X<Min[Y] Then
				Min[Y]:=X;
			If X>Max[Y] Then
				Max[Y]:=X;
		End;

	Begin
		CheckPoint(X-X1,Y-Y1);
		CheckPoint(X+X1,Y-Y1);
		CheckPoint(X-X1,Y+Y1);
		CheckPoint(X+X1,Y+Y1);
		CheckPoint(X-Y1,Y-X1);
		CheckPoint(X+Y1,Y-X1);
		CheckPoint(X-Y1,Y+X1);
		CheckPoint(X+Y1,Y+X1);
	End;

Begin
	For Y1:=0 To 199 Do Begin
		Min[Y1]:=319;
		Max[Y1]:=0;
		Ok[Y1]:=FALSE;
	End;
	YMin:=Y-R;
	YMax:=Y+R;
	YMiddle:=Y;
	X1:=0;
	Y1:=R;
	D:=3;
	Dec(D,2*R);
	While X1<Y1 Do Begin
		CirclePoint(X,Y,X1,Y1);
		If D<0 Then
			Inc(D,4*X1+6)
		Else Begin
			Inc(D,4*(X1-Y1)+10);
			Dec(Y1);
		End;
		Inc(X1);
	End;
	If X1=Y1 Then
		CirclePoint(X,Y,X1,Y1);
	For Y:=0 To 199 Do Begin
		If Ok[Y] Then Begin
			If (Max[Y]>=0) Or (Min[Y]<=319) Then Begin
				X1:=Min[Y];
				If X1<0 Then
					X1:=0;
				X2:=Max[Y];
				If X2>319 Then
					X2:=319;
				XHLine(Y,X1,X2,Color);
			End;
		End;
	End;
End;

Procedure XAllocImage(X1,Y1,X2,Y2 : Word; Var Data : Pointer);
Begin
	GetMem(Data,XImageSize(X1,Y1,X2,Y2));
End;

Procedure XFreeImage(Var Data : Pointer);
Begin
	FreeMem(Data,XImageMemSize(Data^));
	Data:=NIL;
End;

Procedure XTranslate(Var STable,DTable : TTranslate; Count : Byte);
Var
	J,I,C	: Byte;
Begin
	For I:=0 To 255 Do Begin
		C:=I;
		For J:=0 To Count Do
			C:=STable[C];
		DTable[I]:=C;
	End;
End;

Function XLookUp(R,G,B : Byte) : Byte;
Var
	RGB			: TRGB;
	Max			: Word;
	Found		: Byte;
	DR,DG,DB	: Word;
	Dist		: Word;
	I			: Byte;
Begin
	XGetRGBMany(0,256,RGB);
	Max:=65535;
	Found:=0;
	For I:=0 To 255 Do Begin
		DR:=Abs(Integer(R)-Integer(RGB[I*3+1]));
		DG:=Abs(Integer(G)-Integer(RGB[I*3+2]));
		DB:=Abs(Integer(B)-Integer(RGB[I*3+3]));
		Dist:=DR*DR+DG*DG+DB*DB;
		If Dist<Max Then Begin
			Max:=Dist;
			Found:=I;
		End;
	End;
	XLookUp:=Found;
End;

Procedure XGetSortedColors(Var Translate : TTranslate; DarkestFirst : Boolean);
Var
	I1,I2	: Byte;
	Pal		: OPalette;
	T2		: TTranslate;
	T3		: Array[Byte] Of Word;
	R,G,B	: Byte;
	Tot		: Word;
	DoSwap	: Boolean;
Begin
	For I1:=0 To 255 Do
		T2[I1]:=I1;
	Pal.Init(TRUE);
	For I1:=0 To 255 Do Begin
		Tot:=Pal.Red(I1);
		Inc(Tot,Pal.Green(I1));
		Inc(Tot,Pal.Blue(I1));
		T3[I1]:=Tot;
	End;
	For I1:=0 To 254 Do
		For I2:=I1+1 To 255 Do Begin
			If DarkestFirst Then
				DoSwap:=T3[I2]<T3[I1]
			Else
				DoSwap:=T3[I2]>T3[I1];
			If DoSwap Then Begin
				Swap(T3[I1],T3[I2],2);
				Swap(T2[I1],T2[I2],1);
			End;
		End;
	Move(T2,Translate,SizeOf(Translate));
End;

Procedure XSetTranslateTable(Const Translate : TTranslate);
Begin
	Move(Translate,TransTabl,SizeOf(Translate));
End;

Function XTrans(Pixel : Byte) : Byte; Assembler;
Asm
	PUSH	DS
	PUSH	BX
	LDS		BX,DWORD PTR TransPtr
	MOV		AL,Pixel
	XLAT
	POP		BX
	POP		DS
End;

Procedure XSetPoint(Var Point : TPoint; X,Y : Word);
Begin
	Point.X:=X;
	Point.Y:=Y;
End;

Function XNext(FileName : String; Offset : LongInt) : LongInt;
Var
	F		: File;
	Size	: LongInt;
	OldMode	: Byte;
	Temp	: Integer;
	Mod512	: Word;
	Div512	: Word;
Begin
	If FileName='' Then
		FileName:=LastFileName
	Else
		LastFileName:=FileName;
	If XAt(FileName,Offset)=tUnknown Then Begin
		XNext:=-1;
		Exit;
	End Else If XAt(FileName,Offset)=tExeFile Then Begin
		Assign(F,FileName);
		{$I-}
		OldMode:=FileMode;
		FileMode:=0;
		ReSet(F,1);
		FileMode:=OldMode;
		{$I+}
		Temp:=IOResult;
		If Temp<>0 Then Begin
			XNext:=-1;
			Exit;
		End;
		Seek(F,Offset+2);
		BlockRead(F,Mod512,2);
		BlockRead(F,Div512,2);
		Close(F);
		If Mod512<>0 Then
			XNext:=Offset+(Div512-1)*512+Mod512
		Else
			XNext:=Offset+Div512*512;
	End Else Begin
		Assign(F,FileName);
		{$I-}
		OldMode:=FileMode;
		FileMode:=0;
		ReSet(F,1);
		FileMode:=OldMode;
		{$I+}
		Temp:=IOResult;
		If Temp<>0 Then Begin
			XNext:=-1;
			Exit;
		End;
		Seek(F,Offset+4);
		BlockRead(F,Size,4);
		Close(F);
		XNext:=Offset+Size;
	End;
End;

Procedure XSkip(FileName : String; Var Offset : LongInt);
Begin
	Offset:=XNext(FileName,Offset);
End;

Var
	I	: Byte;

Begin
	For I:=0 To 255 Do
		TransTabl[I]:=I;
End.

Image compression types:
	0 - Stored
	1 - RLE
	2 - LZHuf
