/*
  column.c, a program that creates input for POVRAY 3.0
            by Garry J. Vass

  So you had a vision to create a POVRAY of some ancient,
  forgotten, mysterious temple comfortably nestled in 
  some fractal mountain range on a oddly uninhabited planet..

  Perhaps your temple is perched over a luminescent lake...

  All very surreal...  All very peaceful...

  And, of course, a few moons in serene orbit drifting
  through a color_map of clouds...

  But to build your temple, you need some columns...

  I wandered down this path once, and decided to share
  my results.  This file is dedicated to producing 
  a column.  But how to do that?  This brings us to the 
  pre-Socratic architects, our forerunners, who fashioned 
  the classic Greek column styles of Doric, Corinthian, 
  and Ionic.  

  To these unnamed architects, we humbly add another classic
  Greek column style, the POVRAY column:  Four torii and a 
  prism...

  If you compile this file and execute it, it will create
  some POV source called "killme.pov".  If you render that
  file in POVRAY 3, you will get a column.  After that, it 
  is up to you.

  This is a "C" language source file given on an "as is"
  basis.  Use it at your own risk.  If you get something
  you like, count yourself lucky.  Otherwise, too bad.  I
  do not support this file, and any outcome you get from
  using it (good or bad) are your your problem - you 
  compiled it, you ran it, and you rendered it.  If you 
  think I did it wrong, fine.  If you
  think you can improve upon it, fine.  Go for it.  
  Please don't write messages to me about some nitpick
  efficiency improvement.  And further, do not write messages to
  me about how you don't know "C".  I get lots of messages
  like that, they go straight into my kill file, and I simply 
  don't care...  I am an Amatuer.  Comming from "Amat", meaning
  "love".  I do it for love.
*/
#include  <stdio.h>
#include <stdlib.h>
#include   <math.h>

#define RADIANS_PER_DEGREE   0.0174533

double degrees_to_radians( double degrees )
{
    double angle;
    while( degrees >= 360 ) degrees -= 360;
    while( degrees < 0 )    degrees+=360;
    angle = RADIANS_PER_DEGREE * degrees;
    return angle;
}
double cosd( double x )
{
    double r = degrees_to_radians( x );
    double r2 = cos( r );
    return( r2 );
}
double sind( double x )
{
    double r = degrees_to_radians( x );
    double r2 = sin( r );
    return( r2 );
}
void rotate( double angle, double mag, double originx, double originy, double x1, double y1, double *x2, double *y2 )
{
    if( fabs( angle ) > 1e4 )
        angle = 60;
    *x2  = mag * ( ( x1 - originx ) * cosd( angle ) + ( y1 - originy ) * sind( angle ) ) + originx;
    *y2  = mag * ( ( y1 - originy ) * cosd( angle ) - ( x1 - originx ) * sind( angle ) ) + originy;
}
void to_polar( int x, int y, double *dx, double *dy )
{
    *dx = cosd( x );
    *dy = sind( y );
}
char *P1 = "#include \"colors.inc\"        \n\
#include \"textures.inc\"\n\
light_source{ <0, 0, -200 > color White }\n\
camera {\n\
    location  < 0.0, 0.0, -45.0 >\n\
    look_at   < 0.0, 0.0,   0.0 >\n\
}\n\
#declare CaryatidTexture = texture {\n\
    pigment { Gray75 }\n\
    finish {\n\
        ambient 0.2\n\
        diffuse 0.8\n\
        specular 1\n\
        roughness 0.5\n\
    }\n\
}\n\
#declare ColumnTexture = texture {\n\
    pigment { White }\n\
    finish {\n\
       ambient 0.2\n\
       diffuse 0.8\n\
       specular 0.2\n\
       roughness 0.5\n\
       //reflection 0.1\n\
       //brilliance 0\n\
    }\n\
}\n\
#declare Column = union {\n\
    prism {\n\
        linear_sweep\n\
        linear_spline\n\
        -5, 5,\n\
        361,\n";
char *P2 = "        texture { ColumnTexture }\n\
    }\n\
    torus {\n\
        1, 0.9\n\
        translate < 0, 5, 0 >\n\
        texture { CaryatidTexture }\n\
    }\n\
    torus {\n\
        1.5, 0.9\n\
        translate < 0, 6, 0 >\n\
        texture { CaryatidTexture }\n\
    }\n\
    torus {\n\
        1, 0.9\n\
        translate < 0, -5, 0 >\n\
        texture { CaryatidTexture }\n\
    }\n\
    torus {\n\
        1.5, 0.9\n\
        translate < 0, -6, 0 >\n\
        texture { CaryatidTexture }\n\
    }\n\
}\n\n\n";
int main( int argc, char *argv[] )
{
    FILE *fp;
    double angle = 0;
    double radius = 1;
    double increment = 1;  /* must be odd */
    double x = 1;
    double y = 1;
    double z = 1;
    double nx;
    double ny;
    double rad;
    int npts = 0;
    char *cmt;
    fp = fopen( "killme.pov", "wt" );
    rewind( fp );
    fprintf( fp, P1 );
    while( angle <= 360 ) {
           switch( npts % 10 ) {
                   case 0:
                   case 1:
                   case 2:
                   case 3:
                   case 4:
                        rad = radius;
                        cmt = "// normal radius";
                        break;
                   case 5:
                   case 7:
                        rad = 0.97 * radius;
                        cmt = "// cut in";
                        break;
                   case 6:
                        rad = 0.95 * radius;
                        cmt = "// deep cut";
                        break;
           }
           rotate( angle, rad, 0, 0, x, y, &nx, &ny );
           fprintf( fp, " < %6.4f, %6.4f > %s\n", nx, ny, cmt );
           angle += increment;
           ++npts;
    }
    fprintf( fp, "// %d points\n", npts );
    fprintf( fp, P2 );
    fprintf( fp, "object { Column } \n" );
    fclose( fp );
    return( 0 );
}
/*
**  end of source
*/