/*
	POB-File-Format by Thomas Baier April/97


	Description:
	Complete pob-writer implementation for povrays internal object/texture structure.

	History:
	01.04.97	Version 0.8	First Version, all things implemented, 
					rarly tested

	Comments:


	To Do List:

	- why can be patterned pigment of skysphere without colormap ??
	- test for each object and each kind of texture
	- optimized mesh output
*/

#include <stdio.h>
#include <math.h>
#include <stdarg.h>


#include "frame.h"
#include "vector.h"
#include "povproto.h"
#include "parse.h"
#include "parstxtr.h"
#include "atmosph.h"
#include "bezier.h"   
#include "blob.h"     
#include "boxes.h"
#include "colour.h"
#include "cones.h"    
#include "csg.h"      
#include "discs.h"
#include "express.h"  
#include "fractal.h"
#include "gif.h"      
#include "halos.h"
#include "hfield.h"
#include "iff.h"      
#include "image.h"    
#include "lathe.h"    
#include "polysolv.h"
#include "matrices.h"
#include "mesh.h"
#include "normal.h"
#include "objects.h"
#include "octree.h"
#include "pigment.h"
#include "planes.h"
#include "poly.h"
#include "polygon.h"
#include "povray.h"   
#include "pgm.h"      
#include "ppm.h"      
#include "prism.h"    
#include "quadrics.h" 
#include "radiosit.h"      
#include "render.h"   
#include "sor.h"      
#include "spheres.h"  
#include "super.h"
#include "targa.h"    
#include "texture.h"  
#include "tokenize.h" 
#include "torus.h"
#include "triangle.h" 
#include "truetype.h" 

#include "pobwrite.h"
#include "pob_tok.h"
#include "pob_tok1.h"

/* Platform stuff */
/*
	if your Platform does not support IEEE with DOUBLE == 8 Bytes
	you have to rewrite following procedures:
		write_double()
*/

#define __IEEE			// switch on IEEE floating point support

#define POB_VERSION (0x0008) 	// Major Version 00, Minor Version 08
#define POV_VERSION (0x0300) 	// Major Version 03, Minor Version 00

typedef struct link_str
{
	TEXTURE *ptext;
	unsigned int number;
	struct link_str *next;
}link_type;


link_type *add_link( link_type *next_node, link_type *tmp_node );
link_type *create_link( TEXTURE *ptext);
void free_link( link_type *node);

void print_texture_list(link_type *node);
void print_const_list( void );
void print_global_settings(FRAME *Frame);
void print_global_environment(FRAME *Frame);
void print_camera(CAMERA *Camera);

static FILE *fptr = NULL;

static link_type *start_node;
static link_type *next_node;

static unsigned int global_texture_number = 0x10000;
static unsigned int global_pigment_number = 0;
static unsigned int global_tnormal_number = 0;
static unsigned int global_finish_number = 0;
static unsigned int global_halo_number = 0;
static unsigned int pass_level = 0;
unsigned int pob_write_active = 0;


/* chunk related 	*/

void write_dvector(P_DVECTOR v);
void write_dvector_uv(UV_VECT t);
void write_dvector_sngl(SNGL_VECT t);
void write_string(char *data);

void write_double(P_DOUBLE data);
void write_dword(P_DWORD data);
void write_word(P_WORD data);
void write_byte(P_BYTE data);

static void start_chunk(CHUNK *main_chunk, P_WORD tag);
static void end_chunk(CHUNK *main_chunk);



#define MOPEN_TOKEN(a)		{CHUNK chunk;start_chunk(&chunk, a);
#define MCLOSE_TOKEN()		end_chunk(&chunk);}

#define MWRITE_NAME(a)		{static int obj_number = 0;char name[1024];if(pass_level == 1){sprintf(name, "%s_%d",get_token_str(a), obj_number++);write_string(name);}}
#define MWRITE_TNAME(a,t)	{char name[1024];sprintf(name, "%s_%d",get_token_str(a), Texture->txt_number);write_string(name);}
#define MWRITE_PNAME(a,t)	{char name[1024];sprintf(name, "%s_%d",get_token_str(a), Pigment->pigment_number);write_string(name);}
#define MWRITE_NNAME(a,t)	{char name[1024];sprintf(name, "%s_%d",get_token_str(a), Tnormal->tnormal_number);write_string(name);}
#define MWRITE_FNAME(a,t)	{char name[1024];sprintf(name, "%s_%d",get_token_str(a), Finish->finish_number);write_string(name);}
#define MWRITE_HNAME(a,t)	{char name[1024];sprintf(name, "%s_%d",get_token_str(a), Halo->halo_number);write_string(name);}

/* 			*/

void init_pob( char * file_name )
{

        if(pob_write_active == 0)
                return;


        fptr = fopen(file_name, "wb");
        if(fptr == NULL)
        {
                fprintf(stderr, "error open file\n");
                pob_write_active = 0;
        }
}


void exit_pob( void )
{

        if(pob_write_active == 0)
                return;

        if(fptr)
                fclose(fptr);
}

void do_pob( void )
{
	unsigned int lib_flag = 0;
        
        if(pob_write_active == 0)
                return;

        if(fptr == NULL)
                return;

	if(lib_flag)
	{
		pass_level = 1;
//		print_const_list();
	}
	else
	{
	        OBJECT *tmp_obj;
		OBJECT **obj_arr;
		unsigned int count;
		int i;
		
		CHUNK pobfile_sect;
		CHUNK povray_sect;

		for(tmp_obj = Frame.Objects,count = 0; tmp_obj != NULL; tmp_obj = tmp_obj->Sibling,count++);

		obj_arr = (OBJECT **)POV_MALLOC(count * sizeof(OBJECT *) ," Object List");

		for(tmp_obj = Frame.Objects,count = 0; tmp_obj != NULL; tmp_obj = tmp_obj->Sibling,count++)
			obj_arr[count] = tmp_obj;


		// pass 0 collect used textures
        	start_node = next_node = create_link(NULL); // first item is empty
		pass_level = 0;
        	for(i = (count - 1);i > -1;i--)
                	Print(obj_arr[i]);

		pass_level = 1;
		// pass 1 print used texture
		start_chunk(&pobfile_sect, POB_FILESECTION_TAG);
		write_word(POB_VERSION);
		start_chunk(&povray_sect, POB_POVRAYSECTION_TAG);
		write_word(POV_VERSION);
		write_string("Generated by modified povray 3.0 version (thbaier@compuserve.com)");

//		print_const_list();

		// Global Settings
		{
			CHUNK global_sect;

			start_chunk(&global_sect, POB_GLOBALSECTION_TAG);
			print_global_settings(&Frame);
			end_chunk(&global_sect);
		}
		
		// Global Environment
		{
			CHUNK environment_sect;

			start_chunk(&environment_sect, POB_ENVIRONMENTSECTION_TAG);
			print_global_environment(&Frame);
			end_chunk(&environment_sect);
		}

		// Camera
		{
			CHUNK camera_sect;
	
			start_chunk(&camera_sect, POB_CAMERASECTION_TAG);
			print_camera(Frame.Camera);
			end_chunk(&camera_sect);
		}

		// used Textures
		{
			CHUNK material_sect;

			start_chunk(&material_sect, POB_MATERIALSECTION_TAG);
			print_texture_list(start_node);
			end_chunk(&material_sect);
	
		}

		// used Objects
		{
			CHUNK objects_sect;

			start_chunk(&objects_sect, POB_OBJECTSECTION_TAG);
        		for(i = (count - 1);i > -1;i--)
                		Print(obj_arr[i]);
			end_chunk(&objects_sect);
		}


		end_chunk(&povray_sect);

		// test
/*
		{
		        CHUNK modeller_sect;
			
			start_chunk(&modeller_sect, POB_MODELLERSECTION_TAG);
			write_word(0x0100);
			write_string("Moray 2.0!");
			end_chunk(&modeller_sect);
		}
*/
		end_chunk(&pobfile_sect);

		free_link(start_node);
	
		POV_FREE(obj_arr);
	}
}


void print_transform(TRANSFORM *trans)
{

        if(trans != NULL)
        {
		MOPEN_TOKEN(MATRIX_POB_TOKEN);
                write_double( (trans->matrix)[0][0] );
                write_double( (trans->matrix)[0][1] );
                write_double( (trans->matrix)[0][2] );
                write_double( (trans->matrix)[1][0] );
                write_double( (trans->matrix)[1][1] );
                write_double( (trans->matrix)[1][2] );
                write_double( (trans->matrix)[2][0] );
		write_double( (trans->matrix)[2][1] );
                write_double( (trans->matrix)[2][2] );
                write_double( (trans->matrix)[3][0] );
                write_double( (trans->matrix)[3][1] );
                write_double( (trans->matrix)[3][2] );
		MCLOSE_TOKEN();
        }
        else
        {
		MOPEN_TOKEN(MATRIX_POB_TOKEN);
                write_double( 1.0 );
                write_double( 0.0 );
                write_double( 0.0 );
                write_double( 0.0 );
                write_double( 1.0 );
                write_double( 0.0 );
                write_double( 0.0 );
		write_double( 0.0 );
                write_double( 1.0 );
                write_double( 0.0 );
                write_double( 0.0 );
                write_double( 0.0 );
		MCLOSE_TOKEN();
	}
        
  }

void print_bound(OBJECT *Bound)
{
	if(Bound == NULL)
		return;
	MOPEN_TOKEN(BOUNDED_BY_POB_TOKEN);
        Print(Bound);
	MCLOSE_TOKEN();

}

void print_clipped(OBJECT *Clip)
{
	if(Clip == NULL)
		return;
	MOPEN_TOKEN(CLIPPED_BY_POB_TOKEN);
        Print(Clip);
	MCLOSE_TOKEN();
}

void print_flags(OBJECT *tmp_obj)
{
	CHUNK chunk;
	start_chunk(&chunk, FLAGS_POB_TOKEN);
	write_word(tmp_obj->Flags);
	end_chunk(&chunk);
}

void print_image(unsigned int type, char *name)
{
        switch(type)
        {
            case IFF_FILE:
		write_word(IFF_POB_TOKEN);
                break;
            case GIF_FILE:
		write_word(GIF_POB_TOKEN);
                break;
            case POT_FILE:
		write_word(POT_POB_TOKEN);
                break;
            case SYS_FILE:
		write_word(SYS_POB_TOKEN);
                break;
            case TGA_FILE:
		write_word(TGA_POB_TOKEN);
                break;
            case PNG_FILE:
		write_word(PNG_POB_TOKEN);
                break;
            case PGM_FILE:
		write_word(PGM_POB_TOKEN);
                break;
            case PPM_FILE:
		write_word(PPM_POB_TOKEN);
                break;
            default:
                break;                
        }
	write_string(name);
}


void print_color(COLOUR Colour)
{
	write_double(Colour[RED]);
	write_double(Colour[GREEN]);
	write_double(Colour[BLUE]);
	write_double(Colour[FILTER]);
	write_double(Colour[TRANSM]);
}


void print_blendmap_entry(BLEND_MAP *blend_map, unsigned int i)
{

	if(blend_map == NULL)
		return;
    
        switch(blend_map->Type)
        {
                case COLOUR_TYPE:
                        print_color(blend_map->Blend_Map_Entries[i].Vals.Colour);
                break;
                case PIGMENT_TYPE:
			print_pigment( blend_map->Blend_Map_Entries[i].Vals.Pigment);
                break;
                case NORMAL_TYPE:
			print_tnormal( blend_map->Blend_Map_Entries[i].Vals.Tnormal);
                break;

                case TEXTURE_TYPE:
                break;
        }
}



void print_wave_type(TPATTERN *New)
{
	MOPEN_TOKEN(WAVE_TYPE_POB_TOKEN);

	switch(New->Wave_Type)
	{
		case RAMP_WAVE:
        		write_word(RAMP_WAVE_POB_TOKEN);
			break;
		case SINE_WAVE:
        		write_word(SINE_WAVE_POB_TOKEN);
			break;									
		case TRIANGLE_WAVE:
        		write_word(TRIANGLE_WAVE_POB_TOKEN);
			break;
		case SCALLOP_WAVE:
        		write_word(SCALLOP_WAVE_POB_TOKEN);
			break;
	}
	MCLOSE_TOKEN();          
}

mprint_texture_trans(WARP *New)
{
	if(New == NULL)
		return;
	
	switch(New->Warp_Type)
	{
   		case TRANSFORM_WARP:
		{
			TRANS *trans;
      			trans = (TRANS *)New;
			print_transform(&(trans->Trans));
		}
		break;
	}
	if(New->Next_Warp)
		mprint_texture_trans(New->Next_Warp);
}

void print_warp(WARP *New)
{
	if(New == NULL)
		return;

	switch(New->Warp_Type)
	{
   		case CLASSIC_TURB_WARP:
   		case EXTRA_TURB_WARP:
		{
			TURB *Turb;
      			Turb=(TURB *)New;

			MOPEN_TOKEN(CLASSIC_TURB_WARP_POB_TOKEN);
   			
			write_dvector(Turb->Turbulence);
        		write_word(Turb->Octaves);
        		write_double(Turb->Omega);
        		write_double(Turb->Lambda);
			
			MCLOSE_TOKEN();          
		}
		break;

   		case REPEAT_WARP:
		{
			REPEAT *Repeat;
	      		Repeat=(REPEAT *)New;

			MOPEN_TOKEN(REPEAT_WARP_POB_TOKEN);

			switch(Repeat->Axis)
			{
				case X:
					write_double(Repeat->Width);
					write_double(0);
					write_double(0);
					break;
				case Y:
					write_double(0);
					write_double(Repeat->Width);
					write_double(0);
					break;
				case Z:
					write_double(0);
					write_double(0);
					write_double(Repeat->Width);
					break;
			}
			write_dvector(Repeat->Offset);
			write_dvector(Repeat->Flip);
		
			MCLOSE_TOKEN();          
		}
		break;

   		case BLACK_HOLE_WARP:
		{
			BLACK_HOLE *Black_Hole;
      			Black_Hole = (BLACK_HOLE *)New ;

			MOPEN_TOKEN(BLACK_HOLE_WARP_POB_TOKEN);

			write_dvector(Black_Hole->Center);
			write_double(Black_Hole->Radius);
			write_double(Black_Hole->Strength);
			write_double(Black_Hole->Power);

        		write_word(Black_Hole->Inverted);
        		write_word(Black_Hole->Type);
        		
        		write_word(Black_Hole->Repeat);
			if(Black_Hole->Repeat == TRUE)
			{
				write_dvector(Black_Hole->Repeat_Vector);
			}
        		
        		write_word(Black_Hole->Uncertain);
			if(Black_Hole->Uncertain == TRUE)
			{
				write_dvector(Black_Hole->Uncertainty_Vector);
			}
		
			MCLOSE_TOKEN();          
		}
		break;
	}

	if(New->Next_Warp)
		print_warp(New->Next_Warp);
}

void print_pattern_modifier(TPATTERN *New)
{

	MOPEN_TOKEN(PATTERN_MOD_POB_TOKEN);

	if(New != NULL)
	{
		print_wave_type(New);
		print_warp(New->Warps);

		MOPEN_TOKEN(FREQUENCY_POB_TOKEN);
		write_double(New->Frequency);
		MCLOSE_TOKEN();          
		MOPEN_TOKEN(PHASE_POB_TOKEN);
		write_double(New->Phase);
		MCLOSE_TOKEN();          
	}
	MCLOSE_TOKEN();          
}

void print_pattern(TPATTERN *New)
{
	if(New == NULL)
		return;

	switch(New->Type)
	{
		case AGATE_PATTERN :
			write_word(AGATE_POB_TOKEN);
			write_double(New->Vals.Agate_Turb_Scale);
		break;

		case BOZO_PATTERN :
			write_word(BOZO_POB_TOKEN);
		break;
		case GRANITE_PATTERN :
			write_word(GRANITE_POB_TOKEN);
		break;
		case LEOPARD_PATTERN :
			write_word(LEOPARD_POB_TOKEN);
		break;
		case MARBLE_PATTERN :
			write_word(MARBLE_POB_TOKEN);
		break;
		case ONION_PATTERN :
			write_word(ONION_POB_TOKEN);
		break;
		case PATTERN1_PATTERN :
			write_word(PATTERN1_POB_TOKEN);
		break;
		case PATTERN2_PATTERN :
			write_word(PATTERN2_POB_TOKEN);
		break;
		case PATTERN3_PATTERN :
			write_word(PATTERN3_POB_TOKEN);
		break;
		case BUMPY1_PATTERN :
			write_word(BUMPY1_POB_TOKEN);
		break;
		case BUMPY2_PATTERN :
			write_word(BUMPY2_POB_TOKEN);
		break;
		case BUMPY3_PATTERN :
			write_word(BUMPY3_POB_TOKEN);
		break;
		case SPIRAL1_PATTERN :
			write_word(SPIRAL1_POB_TOKEN);
			write_double( New->Vals.Arms);
		break;
		case SPIRAL2_PATTERN :
			write_word(SPIRAL2_POB_TOKEN);
			write_double( New->Vals.Arms);
		break;
		case SPOTTED_PATTERN :
			write_word(BUMPY3_POB_TOKEN);
		break;
		case WOOD_PATTERN :
			write_word(WOOD_POB_TOKEN);
		break;
		case GRADIENT_PATTERN :
			write_word(GRADIENT_POB_TOKEN);
			write_dvector(New->Vals.Gradient);
		break;
		case RADIAL_PATTERN :
			write_word(RADIAL_POB_TOKEN);
		break;
		case CRACKLE_PATTERN :
			write_word(CRACKLE_POB_TOKEN);
		break;
		case WAVES_PATTERN :
			write_word(WAVES_POB_TOKEN);
		break;
		case RIPPLES_PATTERN :
			write_word(RIPPLES_POB_TOKEN);
		break;
		case WRINKLES_PATTERN :
			write_word(WRINKLES_POB_TOKEN);
		break;
		case BUMPS_PATTERN :
			write_word(BUMPS_POB_TOKEN);
		break;
		case DENTS_PATTERN :
			write_word(DENTS_POB_TOKEN);
		break;
		case QUILTED_PATTERN :
			write_word(QUILTED_POB_TOKEN);
			write_double(New->Vals.Quilted.Control0);
			write_double(New->Vals.Quilted.Control1);
		break;
		case AVERAGE_PATTERN :
			write_word(AVERAGE_POB_TOKEN);
		break;

	}
}


/* Texture */

void print_shortref_pigment(PIGMENT *Pigment)
{
	if(Pigment == NULL)
		return;
	
	MWRITE_PNAME(PIGMENT_POB_TOKEN, Pigment);
}


void print_ref_pigment(PIGMENT *Pigment)
{
	if(Pigment == NULL)
		return;

	MOPEN_TOKEN(PIGMENTREF_POB_TOKEN);
	MWRITE_PNAME(PIGMENT_POB_TOKEN, Pigment);
	mprint_texture_trans(((TPATTERN *)Pigment)->Warps);
	MCLOSE_TOKEN();
}

void print_decl_pigment(PIGMENT *Pigment)
{
	if(Pigment == NULL)
		return;

	Pigment->pigment_number = global_pigment_number++;
}


void print_pigment(PIGMENT *Pigment)
{
        if(Pigment == NULL)
                return;

	switch(Pigment->Type)
	{
		case PLAIN_PATTERN:
			print_decl_pigment(Pigment);

        		MOPEN_TOKEN(SIMPLEPIGMENT_POB_TOKEN);
        		MWRITE_PNAME(PIGMENT_POB_TOKEN, Pigment);

			print_color(Pigment->Colour);

			print_pattern_modifier((TPATTERN *)Pigment);

			MCLOSE_TOKEN();          
			break;

		case CHECKER_PATTERN:
        		switch(Pigment->Blend_Map->Type)
        		{
				case COLOUR_TYPE:

					print_decl_pigment(Pigment);

        				MOPEN_TOKEN(CHECKERPIGMENT_POB_TOKEN);
        				MWRITE_PNAME(PIGMENT_POB_TOKEN, Pigment);
					
					write_word(POB_PIGMENT_COLORMAP); // color_type

					print_color(Pigment->Blend_Map->Blend_Map_Entries[0].Vals.Colour);
					print_color(Pigment->Blend_Map->Blend_Map_Entries[1].Vals.Colour);

					print_pattern_modifier((TPATTERN *)Pigment);

					MCLOSE_TOKEN();          
				break;

				case PIGMENT_TYPE:

		        		print_blendmap_entry(Pigment->Blend_Map, 0);
					print_blendmap_entry(Pigment->Blend_Map, 1);

					print_decl_pigment(Pigment);

        				MOPEN_TOKEN(CHECKERPIGMENT_POB_TOKEN);
        				MWRITE_PNAME(PIGMENT_POB_TOKEN, Pigment);
					
					write_word(POB_PIGMENT_PIGMENTMAP); // pigment_type

					print_ref_pigment(Pigment->Blend_Map->Blend_Map_Entries[0].Vals.Pigment);
					print_ref_pigment(Pigment->Blend_Map->Blend_Map_Entries[1].Vals.Pigment);

					print_pattern_modifier((TPATTERN *)Pigment);

					MCLOSE_TOKEN();          
				break;
			}
		break;

		case BRICK_PATTERN:
        		switch(Pigment->Blend_Map->Type)
        		{
				case COLOUR_TYPE:

					print_decl_pigment(Pigment);

        				MOPEN_TOKEN(BRICKPIGMENT_POB_TOKEN);
        				MWRITE_PNAME(PIGMENT_POB_TOKEN, Pigment);
					write_word(POB_PIGMENT_COLORMAP); // color_type

					write_dvector(Pigment->Vals.Brick.Size);
					write_double(Pigment->Vals.Brick.Mortar + Small_Tolerance*2.0);

					print_color(Pigment->Blend_Map->Blend_Map_Entries[0].Vals.Colour);
					print_color(Pigment->Blend_Map->Blend_Map_Entries[1].Vals.Colour);
				
					print_pattern_modifier((TPATTERN *)Pigment);

					MCLOSE_TOKEN();          
				break;

				case PIGMENT_TYPE:

	        			print_blendmap_entry(Pigment->Blend_Map, 0);
					print_blendmap_entry(Pigment->Blend_Map, 1);

					print_decl_pigment(Pigment);

        				MOPEN_TOKEN(BRICKPIGMENT_POB_TOKEN);
        				MWRITE_PNAME(PIGMENT_POB_TOKEN, Pigment);
					write_word(POB_PIGMENT_PIGMENTMAP); // pigment_type

					write_dvector(Pigment->Vals.Brick.Size);
					write_double(Pigment->Vals.Brick.Mortar + Small_Tolerance*2.0);

					print_ref_pigment(Pigment->Blend_Map->Blend_Map_Entries[0].Vals.Pigment);
					print_ref_pigment(Pigment->Blend_Map->Blend_Map_Entries[1].Vals.Pigment);
				
					print_pattern_modifier((TPATTERN *)Pigment);

					MCLOSE_TOKEN();          
				break;
			}
		break;

		case HEXAGON_PATTERN:
        		switch(Pigment->Blend_Map->Type)
        		{
				case COLOUR_TYPE:

					print_decl_pigment(Pigment);
        				
					MOPEN_TOKEN(HEXAGONPIGMENT_POB_TOKEN);
        				MWRITE_PNAME(PIGMENT_POB_TOKEN, Pigment);
					
					write_word(POB_PIGMENT_COLORMAP); // color_type
				
					print_color(Pigment->Blend_Map->Blend_Map_Entries[0].Vals.Colour);
					print_color(Pigment->Blend_Map->Blend_Map_Entries[1].Vals.Colour);
					print_color(Pigment->Blend_Map->Blend_Map_Entries[2].Vals.Colour);

					print_pattern_modifier((TPATTERN *)Pigment);

					MCLOSE_TOKEN();          
				break;

				case PIGMENT_TYPE:

	        			print_blendmap_entry(Pigment->Blend_Map, 0);
					print_blendmap_entry(Pigment->Blend_Map, 1);
					print_blendmap_entry(Pigment->Blend_Map, 2);

					print_decl_pigment(Pigment);

					MOPEN_TOKEN(HEXAGONPIGMENT_POB_TOKEN);
        				MWRITE_PNAME(PIGMENT_POB_TOKEN, Pigment);
					
					write_word(POB_PIGMENT_PIGMENTMAP); // pigment_type

					print_ref_pigment(Pigment->Blend_Map->Blend_Map_Entries[0].Vals.Pigment);
					print_ref_pigment(Pigment->Blend_Map->Blend_Map_Entries[1].Vals.Pigment);
					print_ref_pigment(Pigment->Blend_Map->Blend_Map_Entries[2].Vals.Pigment);

					print_pattern_modifier((TPATTERN *)Pigment);

					MCLOSE_TOKEN();          
				break;

			}
		break;

		case BITMAP_PATTERN:

			print_decl_pigment(Pigment);
			MOPEN_TOKEN(BITMAPPIGMENT_POB_TOKEN);
        		MWRITE_PNAME(PIGMENT_POB_TOKEN, Pigment);
					
			write_word(POB_PIGMENT_COLORMAP); // color_type

			print_image(Pigment->Vals.Image->File_Type, Pigment->Vals.Image->filename);

			write_word(Pigment->Vals.Image->Once_Flag);

			write_word(Pigment->Vals.Image->Interpolation_Type);
			write_word(Pigment->Vals.Image->Map_Type);

			print_pattern_modifier((TPATTERN *)Pigment);

			MCLOSE_TOKEN();          

		break;

		default:
		{			
                	unsigned int i;
			
        		switch(Pigment->Blend_Map->Type)
        		{
                		case COLOUR_TYPE:
					
					print_decl_pigment(Pigment);
					MOPEN_TOKEN(PATTERNPIGMENT_POB_TOKEN);
        				MWRITE_PNAME(PIGMENT_POB_TOKEN, Pigment);
					
					write_word(POB_PIGMENT_COLORMAP); // color_type
					
					print_pattern((TPATTERN *)Pigment);

					write_word(Pigment->Blend_Map->Number_Of_Entries);
			
					if(Pigment->Blend_Map->Number_Of_Entries)
					{
						for(i = 0; i < Pigment->Blend_Map->Number_Of_Entries;i++)
                				{
                        				write_double(Pigment->Blend_Map->Blend_Map_Entries[i].value);
                        				print_color(Pigment->Blend_Map->Blend_Map_Entries[i].Vals.Colour);
                				}
					}

					print_pattern_modifier((TPATTERN *)Pigment);

					MCLOSE_TOKEN();          
                		break;
        
				case PIGMENT_TYPE:
					for(i = 0; i < Pigment->Blend_Map->Number_Of_Entries;i++)
						print_blendmap_entry( Pigment->Blend_Map, i);
			
					print_decl_pigment(Pigment);

					MOPEN_TOKEN(PATTERNPIGMENT_POB_TOKEN);
        				MWRITE_PNAME(PIGMENT_POB_TOKEN, Pigment);
					
					write_word(POB_PIGMENT_PIGMENTMAP); // pigment_type

					print_pattern((TPATTERN *)Pigment);

					write_word(Pigment->Blend_Map->Number_Of_Entries);
			
					if(Pigment->Blend_Map->Number_Of_Entries)
					{
						for(i = 0; i < Pigment->Blend_Map->Number_Of_Entries;i++)
                				{
							MOPEN_TOKEN(MAPVALUE_POB_TOKEN);
		                        		write_double(Pigment->Blend_Map->Blend_Map_Entries[i].value);
							MCLOSE_TOKEN();

							print_ref_pigment( Pigment->Blend_Map->Blend_Map_Entries[i].Vals.Pigment);
			        		}
					}

					print_pattern_modifier((TPATTERN *)Pigment);

					MCLOSE_TOKEN();          
				break;

				default: // only needed for skypshere ??

					print_decl_pigment(Pigment);
					MOPEN_TOKEN(PATTERNPIGMENT_POB_TOKEN);
       					MWRITE_PNAME(PIGMENT_POB_TOKEN, Pigment);

					write_word(POB_PIGMENT_NOMAP); // without map

					print_pattern((TPATTERN *)Pigment);

					print_pattern_modifier((TPATTERN *)Pigment);

					MCLOSE_TOKEN();          

				break;
			}
		}
		break;
	}
}

void print_shortref_tnormal(TNORMAL *Tnormal)
{
	if(Tnormal == NULL)
		return;

	MWRITE_NNAME(NORMAL_POB_TOKEN, Tnormal);
}


void print_ref_tnormal(TNORMAL *Tnormal)
{
	if(Tnormal == NULL)
		return;
	
	MOPEN_TOKEN(NORMALREF_POB_TOKEN);
	MWRITE_NNAME(NORMAL_POB_TOKEN, Tnormal);
	mprint_texture_trans(((TPATTERN *)Tnormal)->Warps);
	MCLOSE_TOKEN();
}

void print_decl_tnormal(TNORMAL *Tnormal)
{
	if(Tnormal == NULL)
		return;

	Tnormal->tnormal_number = global_tnormal_number++;
}


void print_tnormal(TNORMAL *Tnormal)
{
        if(Tnormal == NULL)
                return;

	switch(Tnormal->Type)
	{
		case BUMPS_PATTERN:

			print_decl_tnormal(Tnormal);
        		MOPEN_TOKEN(SIMPLENORMAL_POB_TOKEN);
        		MWRITE_NNAME(NORMAL_POB_TOKEN, Tnormal);

			write_double(Tnormal->Amount);

			print_pattern((TPATTERN *)Tnormal);
			print_pattern_modifier((TPATTERN *)Tnormal);

			MCLOSE_TOKEN();          
			break;

		case CHECKER_PATTERN:
        		print_blendmap_entry(Tnormal->Blend_Map, 0);
			print_blendmap_entry(Tnormal->Blend_Map, 1);

			print_decl_tnormal(Tnormal);
        		MOPEN_TOKEN(CHECKERNORMAL_POB_TOKEN);
        		MWRITE_NNAME(NORMAL_POB_TOKEN, Tnormal);

			write_double(Tnormal->Amount);

			if(Tnormal->Blend_Map)
			{
				write_word(POB_NORMAL_NORMALMAP); // Blendmap
				print_ref_tnormal(Tnormal->Blend_Map->Blend_Map_Entries[0].Vals.Tnormal);
				print_ref_tnormal(Tnormal->Blend_Map->Blend_Map_Entries[1].Vals.Tnormal);
			}
			else
				write_word(POB_NORMAL_NOMAP); // no Blendmap

			print_pattern_modifier((TPATTERN *)Tnormal);

			MCLOSE_TOKEN();          
		break;

		case BRICK_PATTERN:
        		print_blendmap_entry(Tnormal->Blend_Map, 0);
			print_blendmap_entry(Tnormal->Blend_Map, 1);

			print_decl_tnormal(Tnormal);

        		MOPEN_TOKEN(BRICKNORMAL_POB_TOKEN);
        		MWRITE_NNAME(NORMAL_POB_TOKEN, Tnormal);

			write_double(Tnormal->Amount);
			write_dvector(Tnormal->Vals.Brick.Size);
			write_double(Tnormal->Vals.Brick.Mortar + Small_Tolerance*2.0);

			if(Tnormal->Blend_Map)
			{
				write_word(POB_NORMAL_NORMALMAP); // Blendmap
				print_ref_tnormal(Tnormal->Blend_Map->Blend_Map_Entries[0].Vals.Tnormal);
				print_ref_tnormal(Tnormal->Blend_Map->Blend_Map_Entries[1].Vals.Tnormal);
			}
			else
				write_word(POB_NORMAL_NOMAP); // no Blendmap

			print_pattern_modifier((TPATTERN *)Tnormal);

			MCLOSE_TOKEN();          
		break;

		case HEXAGON_PATTERN:
        		print_blendmap_entry(Tnormal->Blend_Map, 0);
			print_blendmap_entry(Tnormal->Blend_Map, 1);
			print_blendmap_entry(Tnormal->Blend_Map, 2);

			print_decl_tnormal(Tnormal);
        		MOPEN_TOKEN(HEXAGONNORMAL_POB_TOKEN);
        		MWRITE_NNAME(NORMAL_POB_TOKEN, Tnormal);

			write_double(Tnormal->Amount);

			if(Tnormal->Blend_Map)
			{
				write_word(POB_NORMAL_NORMALMAP); // Blendmap
				print_ref_tnormal(Tnormal->Blend_Map->Blend_Map_Entries[0].Vals.Tnormal);
				print_ref_tnormal(Tnormal->Blend_Map->Blend_Map_Entries[1].Vals.Tnormal);
				print_ref_tnormal(Tnormal->Blend_Map->Blend_Map_Entries[2].Vals.Tnormal);
			}
			else
				write_word(POB_NORMAL_NOMAP); // no Blendmap

			print_pattern_modifier((TPATTERN *)Tnormal);

			MCLOSE_TOKEN();          
		break;
		case BITMAP_PATTERN:

			print_decl_tnormal(Tnormal);

			MOPEN_TOKEN(BITMAPNORMAL_POB_TOKEN);
        		MWRITE_NNAME(NORMAL_POB_TOKEN, Tnormal);

			write_double(Tnormal->Amount);
			
			print_image(Tnormal->Vals.Image->File_Type, Tnormal->Vals.Image->filename);

			write_word(Tnormal->Vals.Image->Once_Flag);

			write_word(Tnormal->Vals.Image->Interpolation_Type);
			write_word(Tnormal->Vals.Image->Map_Type);
              
			print_pattern_modifier((TPATTERN *)Tnormal);

			MCLOSE_TOKEN();          

		break;

		default:
		{			
                	unsigned int i;
			
			if(Tnormal->Blend_Map)
			{

	        		switch(Tnormal->Blend_Map->Type)
        			{
					case SLOPE_TYPE:
				
					print_decl_tnormal(Tnormal);
					MOPEN_TOKEN(PATTERNNORMAL_POB_TOKEN);
        				MWRITE_NNAME(NORMAL_POB_TOKEN, Tnormal);

					write_double(Tnormal->Amount);

					print_pattern((TPATTERN *)Tnormal);

					write_word(POB_NORMAL_SLOPEMAP); // Slopmap

					write_word(Tnormal->Blend_Map->Number_Of_Entries);

					for(i = 0; i < Tnormal->Blend_Map->Number_Of_Entries;i++)
                			{
                        			write_double(Tnormal->Blend_Map->Blend_Map_Entries[i].value);
						write_dvector_uv( Tnormal->Blend_Map->Blend_Map_Entries[i].Vals.Point_Slope);
                			}

					print_pattern_modifier((TPATTERN *)Tnormal);

					MCLOSE_TOKEN();          
					break;
 
					case NORMAL_TYPE:

					for(i = 0; i < Tnormal->Blend_Map->Number_Of_Entries;i++)
						print_blendmap_entry( Tnormal->Blend_Map, i);
			
					print_decl_tnormal(Tnormal);

					MOPEN_TOKEN(PATTERNNORMAL_POB_TOKEN);
        				MWRITE_NNAME(NORMAL_POB_TOKEN, Tnormal);

					write_double(Tnormal->Amount);

					print_pattern((TPATTERN *)Tnormal);

					write_word(POB_NORMAL_NORMALMAP); // Blendmap

					write_word(Tnormal->Blend_Map->Number_Of_Entries);
			
					for(i = 0; i < Tnormal->Blend_Map->Number_Of_Entries;i++)
        	        		{
						MOPEN_TOKEN(MAPVALUE_POB_TOKEN);
	                        		write_double(Tnormal->Blend_Map->Blend_Map_Entries[i].value);
						MCLOSE_TOKEN();

						print_ref_tnormal( Tnormal->Blend_Map->Blend_Map_Entries[i].Vals.Tnormal);
                			}

					print_pattern_modifier((TPATTERN *)Tnormal);

					MCLOSE_TOKEN();          
					break;
				}
			}
			
			else
			{
				print_decl_tnormal(Tnormal);

        			MOPEN_TOKEN(PATTERNNORMAL_POB_TOKEN);
        			MWRITE_NNAME(NORMAL_POB_TOKEN, Tnormal);

				write_double(Tnormal->Amount);
	
				print_pattern((TPATTERN *)Tnormal);
				write_word(POB_NORMAL_NOMAP); // no Blendmap

				print_pattern_modifier((TPATTERN *)Tnormal);

				MCLOSE_TOKEN();          
			}

		}
		break;
	}
}

void print_shortref_finish(FINISH *Finish)
{
	if(Finish == NULL)
		return;

        MWRITE_FNAME(FINISH_POB_TOKEN, Finish);
}


void print_ref_finish(FINISH *Finish)
{
	if(Finish == NULL)
		return;

	MOPEN_TOKEN(FINISHREF_POB_TOKEN);
        MWRITE_FNAME(FINISH_POB_TOKEN, Finish);
	MCLOSE_TOKEN();



}

void print_decl_finish(FINISH *Finish)
{
	if(Finish== NULL)
		return;

	Finish->finish_number = global_finish_number++;
}


void print_finish(FINISH *Finish)
{
        if(Finish == NULL)
                return;

	print_decl_finish(Finish);

	MOPEN_TOKEN(SIMPLEFINISH_POB_TOKEN);
        MWRITE_FNAME(FINISH_POB_TOKEN, Finish);

	print_color(Finish->Ambient);
	write_double(Finish->Brilliance);
	write_double(Finish->Diffuse);
	print_color(Finish->Reflection);
	write_double(Finish->Refraction);
	write_double(Finish->Index_Of_Refraction);
	write_double(Finish->Phong);
	write_double(Finish->Phong_Size);
	write_double(Finish->Specular);

	if(Finish->Roughness != 0.0)
		write_double(1.0 / Finish->Roughness);
	else
		write_double(0.0);

	write_double(Finish->Metallic);
	write_double(Finish->Caustics / 45.0);
	write_double(Finish->Crand);

	write_double(Finish->Irid);
	write_double(Finish->Irid_Film_Thickness);

	write_double(Finish->Irid_Turb); // irid_turb vector
	write_double(0);
	write_double(0);

	write_double(Finish->Fade_Distance);
	write_double(Finish->Fade_Power);

	MCLOSE_TOKEN();          
}

void print_shortref_halo(HALO *Halo)
{
	if(Halo== NULL)
		return;
        MWRITE_HNAME(HALO_POB_TOKEN, Halo);
}


void print_ref_halo(HALO *Halo)
{
	if(Halo == NULL)
		return;

	MOPEN_TOKEN(HALOREF_POB_TOKEN);
        MWRITE_HNAME(HALO_POB_TOKEN, Halo);
	print_transform(Halo->Trans);
	MCLOSE_TOKEN();
}

void print_decl_halo(HALO *Halo)
{
	if(Halo== NULL)
		return;

	Halo->halo_number = global_halo_number++;
}

void print_halo_render_type(HALO *Halo)
{
	if(Halo== NULL)
		return;

	switch(Halo->Rendering_Type)
	{
		case HALO_ATTENUATING :
		write_word(ATTENUATING_POB_TOKEN);
		break;
		case HALO_EMITTING :
		write_word(EMITTING_POB_TOKEN);
		break;
		case HALO_GLOWING :
		write_word(GLOWING_POB_TOKEN);
		break;
		case HALO_DUST :
		write_word(DUST_POB_TOKEN);
		break;
		case HALO_VOLUME_RENDERED :
		write_word(VOLUME_RENDERED_POB_TOKEN);
		break;
		case HALO_VOL_REND_WITH_LIGHT :
		write_word(VOL_WITH_LIGHT_POB_TOKEN);
		break;
	}
}

void print_halo_type(HALO *Halo)
{
	if(Halo == NULL)
		return;

	switch(Halo->Type)
	{
		case HALO_NO_HALO :
		write_word(HALO_NO_HALO_POB_TOKEN);
		break;
		case HALO_CONSTANT :
		write_word(CONSTANT_POB_TOKEN);
		break;
		case HALO_LINEAR :
		write_word(LINEAR_POB_TOKEN);
		break;
		case HALO_CUBIC :
		write_word(CUBIC_POB_TOKEN);
		break;
		case HALO_POLY :
		write_word(POLY_POB_TOKEN);
		break;
		case HALO_VOLUME_OBJECT :
		write_word(VOLUME_OBJECT_POB_TOKEN);
		break;
	}
}

void print_halo_mapping_type(HALO *Halo)
{
	if(Halo == NULL)
		return;

	switch(Halo->Mapping_Type)
	{
		case HALO_PLANAR_MAP :
		write_word(PLANAR_MAPPING_POB_TOKEN);
		break;
		case HALO_SPHERICAL_MAP :
		write_word(SPHERICAL_MAPPING_POB_TOKEN);
		break;
		case HALO_CYLINDRICAL_MAP :
		write_word(CYLINDRICAL_MAPPING_POB_TOKEN);
		break;
		case HALO_BOX_MAP :
		write_word(BOX_MAPPING_POB_TOKEN);
		break;
	}
}


void print_halo(HALO *Halo)
{
        unsigned int i;
	COLOUR tmp_Colour;
	
	if(Halo == NULL)
                return;

	print_decl_halo(Halo);


	MOPEN_TOKEN(SIMPLEHALO_POB_TOKEN);
        MWRITE_HNAME(HALO_POB_TOKEN, Halo);

	print_halo_render_type(Halo);
	print_halo_type(Halo);
	print_halo_mapping_type(Halo);

	write_double(Halo->Samples);
	write_word(Halo->AA_Level);
	write_double(Halo->AA_Threshold);
	write_double(Halo->Jitter);
	write_word(Halo->Dust_Type);
	write_double(Halo->Max_Value);
	write_double(Halo->Exponent);
	write_double(Halo->Eccentricity);

     	
	if(Halo->Blend_Map)
		write_word(Halo->Blend_Map->Number_Of_Entries);
	else
		write_word(0);

	if(Halo->Blend_Map)
	{
	        for(i = 0; i < Halo->Blend_Map->Number_Of_Entries;i++)
	        {
			tmp_Colour[0] = Halo->Blend_Map->Blend_Map_Entries[i].Vals.Colour[0];
			tmp_Colour[1] = Halo->Blend_Map->Blend_Map_Entries[i].Vals.Colour[1];
			tmp_Colour[2] = Halo->Blend_Map->Blend_Map_Entries[i].Vals.Colour[2];
			tmp_Colour[3] = Halo->Blend_Map->Blend_Map_Entries[i].Vals.Colour[3];
			tmp_Colour[4] = Halo->Blend_Map->Blend_Map_Entries[i].Vals.Colour[4];
			tmp_Colour[TRANSM] = 1.0 - tmp_Colour[TRANSM]; 

        		write_double(Halo->Blend_Map->Blend_Map_Entries[i].value);
			print_color(tmp_Colour);
	        }
	}

	if(Halo->Turb)
	{
		write_word(1); // Turb active

		write_dvector(Halo->Turb->Turbulence);
		write_word(Halo->Turb->Octaves);
		write_double(Halo->Turb->Omega);
		write_double(Halo->Turb->Lambda);
	}
	else
		write_word(0);// Turb not active
	

	write_double(Halo->Frequency);
	write_double(Halo->Phase);

	MCLOSE_TOKEN();          
}

unsigned int count_layered_texture(TEXTURE *Texture)
{
        unsigned int i = 0;
        TEXTURE *local;

	for(local = Texture; local != NULL; local = (TEXTURE *)local->Next)
                i++;
        return i;        
}

/*
Povray 3 Texture concept:

	texture 		-> 	plain_texture
					layered_texture
					material_mapped_texture
					texture_mapped_texture

	plain_texture 		-> 	pigment normal finish halo

	layered_texture 	-> 	texture	  (only plain/layered texture allowed)
				   	texture					"
					...
					( layering only possible via declare)

	material_mapped_texture ->	image
					texture	
					texture	
					...

	texture_mapped_texture	-> 	keyword
					texture	
					texture
					...
*/

void print_shortref_texture(TEXTURE *Texture)
{
	if(Texture== NULL)
		return;

	MWRITE_TNAME(TEXTURE_POB_TOKEN, Texture);
}


void print_ref_texture(TEXTURE *Texture)
{
	if(Texture == NULL)
		return;

	MOPEN_TOKEN(TEXTUREREF_POB_TOKEN);
	MWRITE_TNAME(TEXTURE_POB_TOKEN, Texture);
	mprint_texture_trans(((TPATTERN *)Texture)->Warps);
	MCLOSE_TOKEN();
}

void print_decl_texture(TEXTURE *Texture)
{
	if(Texture == NULL)
		return;
		
//	if(Texture->txt_number == 0)
		Texture->txt_number = global_texture_number++;
}




void print_texture( TEXTURE *Texture)
{
	if(Texture == NULL)
		return;
	
	print_ref_texture(Texture);

	if(pass_level == 0)
		next_node = add_link(next_node, create_link( Texture));
}

void print_mtexture( TEXTURE *Texture)
{
	CHUNK layered_texture;
        
	static unsigned int texture_depth = 1;
		
	if(Texture == NULL)
		return;

        switch(Texture->Type)
        {
                // normal texture
                case PLAIN_PATTERN:
                {
			HALO *local_halo;
			HALO **halo_arr;
			unsigned int count;        
			int i;        

			if(texture_depth == 1)
			{
				TEXTURE *local_texture;

				for(local_texture = Texture; local_texture != NULL;local_texture = (TEXTURE *)local_texture->Next)
				{
					print_pigment(local_texture->Pigment);
					print_tnormal(local_texture->Tnormal);
					print_finish(local_texture->Finish);


					//reverse order of halo
					for(local_halo = local_texture->Halo,count = 0; local_halo != NULL;local_halo = local_halo->Next_Halo, count++);
					if(count)
					{
						halo_arr = (HALO **)POV_MALLOC(count * sizeof(HALO *) ," Halo List");
						for(local_halo = local_texture->Halo,count = 0; local_halo != NULL;local_halo = local_halo->Next_Halo, count++)
							halo_arr[count] = local_halo;
						for(i = (count - 1);i > -1;i--)
        	        				print_halo(halo_arr[i]);
						POV_FREE(halo_arr);
					}
				}

	       			print_decl_texture(Texture);

				if((TEXTURE *)Texture->Next)
				{
					start_chunk(&layered_texture, LAYEREDTEXTURE_POB_TOKEN);
					MWRITE_TNAME(TEXTURE_POB_TOKEN, Texture);
				}

			}
                        
                	texture_depth++;

        		MOPEN_TOKEN(SIMPLETEXTURE_POB_TOKEN);
        		MWRITE_TNAME(TEXTURE_POB_TOKEN, Texture);


                	print_ref_pigment(Texture->Pigment);
                	print_ref_tnormal(Texture->Tnormal);
                	print_ref_finish(Texture->Finish);

			//reverse order of halo
			for(local_halo = Texture->Halo,count = 0; local_halo != NULL;local_halo = local_halo->Next_Halo, count++);
			if(count)
			{
				halo_arr = (HALO **)POV_MALLOC(count * sizeof(HALO *) ," Halo List");
				for(local_halo = Texture->Halo,count = 0; local_halo != NULL;local_halo = local_halo->Next_Halo, count++)
					halo_arr[count] = local_halo;
				for(i = (count - 1);i > -1;i--)
					print_ref_halo(halo_arr[i]);
				POV_FREE(halo_arr);
			}
      			
			MCLOSE_TOKEN();          

                	if((TEXTURE *)Texture->Next)
                        	print_mtexture((TEXTURE *)Texture->Next);
                
			texture_depth--;
			if(texture_depth == 1)
			{
				if((TEXTURE *)Texture->Next)
				{
					end_chunk(&layered_texture);
				}
                	}
		}
                break;

                // material_map
                case BITMAP_PATTERN:
                {
			TEXTURE *local_txt;

                	for(local_txt = Texture->Materials; local_txt != NULL; local_txt = local_txt->Next_Material)
	                        print_mtexture(local_txt);
        
			print_decl_texture(Texture);

        		MOPEN_TOKEN(MATERIAL_MAPTEXTURE_POB_TOKEN);
        		MWRITE_TNAME(TEXTURE_POB_TOKEN, Texture);
			
			print_image(Texture->Vals.Image->File_Type, Texture->Vals.Image->filename);
			write_word(Texture->Vals.Image->Once_Flag);

			write_word(Texture->Vals.Image->Interpolation_Type);
			write_word(Texture->Vals.Image->Map_Type);

                	for(local_txt = Texture->Materials; local_txt != NULL; local_txt = local_txt->Next_Material)
	                        print_ref_texture(local_txt);
			MCLOSE_TOKEN();          
                }
                break;

                // checker map
                case CHECKER_PATTERN:
                {
                	print_mtexture(Texture->Blend_Map->Blend_Map_Entries[0].Vals.Texture);
			print_mtexture(Texture->Blend_Map->Blend_Map_Entries[1].Vals.Texture);

       			print_decl_texture(Texture);
                
        		MOPEN_TOKEN(CHECKERTEXTURE_POB_TOKEN);
        		MWRITE_TNAME(TEXTURE_POB_TOKEN, Texture);

                	print_ref_texture(Texture->Blend_Map->Blend_Map_Entries[0].Vals.Texture);
                	print_ref_texture(Texture->Blend_Map->Blend_Map_Entries[1].Vals.Texture);

			print_pattern_modifier((TPATTERN *)Texture);
                
			MCLOSE_TOKEN();          

                }
                break;
                
                // brick map
                case BRICK_PATTERN:
                {
			print_mtexture(Texture->Blend_Map->Blend_Map_Entries[0].Vals.Texture);
                	print_mtexture(Texture->Blend_Map->Blend_Map_Entries[1].Vals.Texture);
        
			print_decl_texture(Texture);
                	
        		MOPEN_TOKEN(BRICKTEXTURE_POB_TOKEN);
        		MWRITE_TNAME(TEXTURE_POB_TOKEN, Texture);

                	print_ref_texture(Texture->Blend_Map->Blend_Map_Entries[0].Vals.Texture);
                	print_ref_texture(Texture->Blend_Map->Blend_Map_Entries[1].Vals.Texture);

        		MOPEN_TOKEN(BRICK_SIZE_POB_TOKEN);
			write_dvector(Texture->Vals.Brick.Size);
			MCLOSE_TOKEN();          

        		MOPEN_TOKEN(MORTAR_POB_TOKEN);
			write_double(Texture->Vals.Brick.Mortar + Small_Tolerance*2.0);
			MCLOSE_TOKEN();          
                
			print_pattern_modifier((TPATTERN *)Texture);
			MCLOSE_TOKEN();          

                }
                break;

                // hexagon map
                case HEXAGON_PATTERN:
                {
			print_mtexture(Texture->Blend_Map->Blend_Map_Entries[0].Vals.Texture);
			print_mtexture(Texture->Blend_Map->Blend_Map_Entries[1].Vals.Texture);
			print_mtexture(Texture->Blend_Map->Blend_Map_Entries[2].Vals.Texture);
        
			print_decl_texture(Texture);

        		MOPEN_TOKEN(HEXAGONTEXTURE_POB_TOKEN);
        		MWRITE_TNAME(TEXTURE_POB_TOKEN, Texture);

                	print_ref_texture(Texture->Blend_Map->Blend_Map_Entries[0].Vals.Texture);
                	print_ref_texture(Texture->Blend_Map->Blend_Map_Entries[1].Vals.Texture);
                	print_ref_texture(Texture->Blend_Map->Blend_Map_Entries[2].Vals.Texture);

			print_pattern_modifier((TPATTERN *)Texture);
			MCLOSE_TOKEN();          

                }
                break;
                default:
		{
			// rest of texture mapped texture
			unsigned int i;

                	for(i = 0; i < Texture->Blend_Map->Number_Of_Entries;i++)
                        	print_mtexture( Texture->Blend_Map->Blend_Map_Entries[i].Vals.Texture);

			print_decl_texture(Texture);

        		MOPEN_TOKEN(PATTERNTEXTURE_POB_TOKEN);
        		MWRITE_TNAME(TEXTURE_POB_TOKEN, Texture);
			
			print_pattern((TPATTERN *)Texture);

			write_word(Texture->Blend_Map->Number_Of_Entries);

			if(Texture->Blend_Map->Number_Of_Entries)
			{
                		for(i = 0; i < Texture->Blend_Map->Number_Of_Entries;i++)
                		{
                        		write_double(Texture->Blend_Map->Blend_Map_Entries[i].value);
                        		print_shortref_texture(Texture->Blend_Map->Blend_Map_Entries[i].Vals.Texture);
                		}
			}

			print_pattern_modifier((TPATTERN *)Texture);
			MCLOSE_TOKEN();          
		}
		break;

        }                 
}

void print_texture_list(link_type *node)
{
	link_type *tmp_node;
	link_type *next_node;

	tmp_node = node;

	while(tmp_node != NULL)
	{
		next_node = tmp_node->next;
		if(tmp_node->ptext)
			print_mtexture(tmp_node->ptext);

		tmp_node = next_node;
	}
}

#ifdef CONST_LIST
/* only for textures from declare List */

typedef struct Reserved_Word_Struct RESERVED_WORD;
typedef struct Hash_Table_Struct HASH_TABLE;

struct Hash_Table_Struct
{
  RESERVED_WORD Entry;
  HASH_TABLE *next;
};

#define HASH_TABLE_SIZE 257

extern int Number_Of_Symbols;
extern HASH_TABLE *Symbol_Table_Hash_Table[HASH_TABLE_SIZE];


char *get_symbol(unsigned int Token_ID)
{
	unsigned int i;
	HASH_TABLE *p;

  	for(i = 0;i < Number_Of_Symbols;i++)
	{
		p = Symbol_Table_Hash_Table[i];

  		while (p)
  		{
    			if (Token_ID == (p->Entry.Token_Number))
      				return(p->Entry.Token_Name);
	
    			p = p->next;
		}
  	}
	return "HALLO";
}


// print const texture
void print_const_list(void)
{
	unsigned int i;
	TEXTURE *ptxt;

	for(i = 1; i < (Number_Of_Constants + 1);i++)
	{
		if(Constants[i].Constant_Type == TEXTURE_ID_TOKEN)
		{
			ptxt = (TEXTURE *)Constants[i].Constant_Data;
		}
	}
}

#endif

/* Frame objects */

void print_rainbow(RAINBOW *Rainbow)
{
	P_WORD i;

	if(Rainbow == NULL)
		return;

        MOPEN_TOKEN(RAINBOW_POB_TOKEN);
        MWRITE_NAME(RAINBOW_POB_TOKEN);

	write_dvector(Rainbow->Antisolar_Vector);
	write_double(Rainbow->Angle);
	write_double(Rainbow->Distance);
	write_double(Rainbow->Jitter);
	write_double(Rainbow->Width);
	write_dvector(Rainbow->Up_Vector);
	write_double(Rainbow->Falloff_Angle);
	write_double(Rainbow->Arc_Angle );

        if(Rainbow->Pigment)
	{
		write_word(Rainbow->Pigment->Blend_Map->Number_Of_Entries);
		
		if(Rainbow->Pigment->Blend_Map->Number_Of_Entries)
		{
			for(i = 0; i < Rainbow->Pigment->Blend_Map->Number_Of_Entries;i++)
        		{
        			write_double(Rainbow->Pigment->Blend_Map->Blend_Map_Entries[i].value);
                		print_color(Rainbow->Pigment->Blend_Map->Blend_Map_Entries[i].Vals.Colour);
                	}
		}
	}
	else
		write_word(0);

      	MCLOSE_TOKEN();          
}

void print_skysphere(SKYSPHERE *Skysphere)
{
        unsigned int i;

	if(Skysphere == NULL)
		return;
    
        for(i = 0; i < Skysphere->Count;i++)
                print_pigment(Skysphere->Pigments[i]);

        MOPEN_TOKEN(SKY_SPHERE_POB_TOKEN);
        MWRITE_NAME(SKY_SPHERE_POB_TOKEN);

        for(i = 0; i < Skysphere->Count;i++)
                print_ref_pigment(Skysphere->Pigments[i]);

        print_transform(Skysphere->Trans);

      	MCLOSE_TOKEN();          
}

void print_fog(FOG *Fog)
{
	if(Fog == NULL)
		return;

        MOPEN_TOKEN(FOG_POB_TOKEN);
        MWRITE_NAME(FOG_POB_TOKEN);

        print_color(Fog->Colour);
	write_double(Fog->Distance);
	write_word(Fog->Type);
	write_double(Fog->Alt);
	write_double(Fog->Offset);
	write_double(Fog->Turb_Depth);

        if(Fog->Turb)
        {
		write_word(1);// Turb
		write_dvector(Fog->Turb->Turbulence);
		write_word(Fog->Turb->Octaves);
		write_double(Fog->Turb->Omega);
		write_double(Fog->Turb->Lambda);
        }        
	else
		write_word(0);//no Turb

	write_dvector(Fog->Up);

      	MCLOSE_TOKEN();          
}

void print_atmosphere(ATMOSPHERE *Atmosphere)
{
	if(Atmosphere == NULL)
		return;

        MOPEN_TOKEN(ATMOSPHERE_POB_TOKEN);
        MWRITE_NAME(ATMOSPHERE_POB_TOKEN);

	write_word(Atmosphere->Type);
	write_double(Atmosphere->Distance);
	write_word(Atmosphere->Samples);
	write_double(Atmosphere->Scattering);
	write_word(Atmosphere->AA_Level);
	write_double(Atmosphere->AA_Threshold);
	write_double(Atmosphere->Jitter);
	write_double(Atmosphere->Eccentricity);
        
	print_color(Atmosphere->Colour);

      	MCLOSE_TOKEN();          
}


void print_global_environment(FRAME *Frame)
{
	if(Frame == NULL)
		return;

        if(Frame->Rainbow)
                print_rainbow(Frame->Rainbow);

        if(Frame->Skysphere)
                print_skysphere(Frame->Skysphere);
        
        if(Frame->Fog)
        {
		unsigned int count;        
		int i;
		FOG **fog_arr;
        	FOG *local_fog;
        
		// reverse fog order to do it correct
        	for(local_fog = Frame->Fog,count = 0; local_fog != NULL; local_fog = local_fog->Next,count++);

		fog_arr = (FOG **)POV_MALLOC(count * sizeof(FOG *) ," Fog List");

        	for(local_fog = Frame->Fog, count = 0; local_fog != NULL; local_fog = local_fog->Next, count++)
			fog_arr[count] = local_fog;


        	for(i = (count - 1);i > -1;i--)
                	print_fog(fog_arr[i]);

		POV_FREE(fog_arr);
        }
                        
        if(Frame->Atmosphere)
                print_atmosphere(Frame->Atmosphere);
                
        MOPEN_TOKEN(BACKGROUND_POB_TOKEN);
        MWRITE_NAME(BACKGROUND_POB_TOKEN);
        print_color(Frame->Background_Colour);
      	MCLOSE_TOKEN();          
}

void print_global_settings(FRAME *Frame)
{
	if(Frame == NULL)
		return;

        print_color(Frame->Irid_Wavelengths);
        write_word(opts.Options & GAMMA_CORRECT);
	write_double(opts.GammaFactor * opts.DisplayGamma);

	write_word(Max_Trace_Level);
	write_word(Max_Intersections);
	write_double(ADC_Bailout);
	write_word(Number_Of_Waves);
        print_color(Frame->Ambient_Light);
        write_word(opts.Options & HF_GRAY_16);

	write_double(opts.Radiosity_Brightness);
	write_word(opts.Radiosity_Count);
	write_double(opts.Radiosity_Dist_Max);
	write_double(opts.Radiosity_Error_Bound);
	write_double(opts.Radiosity_Gray);
	write_double(opts.Radiosity_Low_Error_Factor);
	write_double(opts.Radiosity_Min_Reuse);
	write_word(opts.Radiosity_Nearest_Count);
	write_word(opts.Radiosity_Recursion_Limit);
}



void print_camera(CAMERA *Camera)
{
        DBL focal_len;

	if(Camera == NULL)
		return;

        if(Camera->Tnormal)
                print_tnormal(Camera->Tnormal);

        MOPEN_TOKEN(CAMERA_POB_TOKEN);
        MWRITE_NAME(CAMERA_POB_TOKEN);

        switch(Camera->Type)
        {
            case PERSPECTIVE_CAMERA:
		write_word(PERSPECTIVE_POB_TOKEN);
                break;
            case ORTHOGRAPHIC_CAMERA:
		write_word(ORTHOGRAPHIC_POB_TOKEN);
                break;
            case FISHEYE_CAMERA:
		write_word(FISHEYE_POB_TOKEN);
                break;
            case ULTRA_WIDE_ANGLE_CAMERA:
		write_word(ULTRA_WIDE_ANGLE_POB_TOKEN);
                break;
            case OMNIMAX_CAMERA:
		write_word(OMNIMAX_POB_TOKEN);
                break;
            case PANORAMIC_CAMERA:
		write_word(PANORAMIC_POB_TOKEN);
                break;
            case CYL_1_CAMERA:
		write_word(CYLINDER_POB_TOKEN);write_word(1);
                break;
            case CYL_2_CAMERA:
		write_word(CYLINDER_POB_TOKEN);write_word(2);
                break;
            case CYL_3_CAMERA:
		write_word(CYLINDER_POB_TOKEN);write_word(3);
                break;
            case CYL_4_CAMERA:
		write_word(CYLINDER_POB_TOKEN);write_word(4);
                break;
            case TEST_CAMERA_1:
		write_word(TEST_CAMERA_1_POB_TOKEN);
                break;
            case TEST_CAMERA_2:
		write_word(TEST_CAMERA_2_POB_TOKEN);
                break;
            case TEST_CAMERA_3:
		write_word(TEST_CAMERA_3_POB_TOKEN);
                break;
            case TEST_CAMERA_4:
		write_word(TEST_CAMERA_4_POB_TOKEN);
                break;
        }

	write_dvector(Camera->Location);
	write_dvector(Camera->Look_At);
	write_dvector(Camera->Sky);
	write_dvector(Camera->Up);
	write_dvector(Camera->Direction);
	write_dvector(Camera->Right);
	
	write_double(Camera->Confidence);
	write_double(Camera->Variance);
	write_double(Camera->Aperture);
        write_word(Camera->Blur_Samples);
	write_double(Camera->used_Angle);

        VLength (focal_len, Camera->Focal_Distance_vector);
	write_double(focal_len);
	write_dvector(Camera->Focal_Distance_vector);

        if(Camera->Tnormal)
                print_ref_tnormal(Camera->Tnormal);

      	MCLOSE_TOKEN();          
}



/* object method */

//ok
void Print_Bicubic_Patch(OBJECT *Object)
{
        int i,j;

        BICUBIC_PATCH *tmp_obj;
        
        tmp_obj = (BICUBIC_PATCH *)Object;

        MOPEN_TOKEN(BICUBIC_PATCH_POB_TOKEN);
        MWRITE_NAME(BICUBIC_PATCH_POB_TOKEN);
        
	write_word(tmp_obj->Patch_Type);
	write_double(tmp_obj->Flatness_Value);
	write_word(tmp_obj->U_Steps);
	write_word(tmp_obj->V_Steps);
        for (i = 0; i < 4; i++)
                for (j = 0; j < 4; j++)
			write_dvector(tmp_obj->Control_Points[i][j]);

        print_flags(Object);
        print_texture(tmp_obj->Texture);
        print_bound(tmp_obj->Bound);
        print_clipped(tmp_obj->Clip);
      	MCLOSE_TOKEN();          
}

//ok
void print_blob_sphere(BLOB_ELEMENT *Entry)
{
        MOPEN_TOKEN(BLOBSPHERE_POB_TOKEN);
	{
		static int obj_number = 0;
		char name[1024];
		if(pass_level == 1)
		{
			sprintf(name, "%s_%d","blob_sphere", obj_number++);
			write_string(name);
		}
	}

	write_dvector(Entry->O);

        write_double(sqrt(Entry->rad2));

        write_double(Entry->c[2]);

        print_transform(Entry->Trans);
        print_texture(Entry->Texture);
      	MCLOSE_TOKEN();          
}

//ok
void print_blob_cylinder(BLOB_ELEMENT *Entry)
{
//        VECTOR tmp;
        
        MOPEN_TOKEN(BLOBCYLINDER_POB_TOKEN);
	{
		static int obj_number = 0;
		char name[1024];
		if(pass_level == 1)
		{
			sprintf(name, "%s_%d","blob_cylinder", obj_number++);
			write_string(name);
		}
	}

	/* vector 1 */
	write_double(0.0);
	write_double(0.0);
	write_double(0.0);

	/* vector 2 */
	write_double(0.0);
	write_double(0.0);
	write_double(Entry->len);

        write_double(sqrt(Entry->rad2));

        write_double(Entry->c[2]);
        print_transform(Entry->Trans);
        print_texture(Entry->Texture);
      	MCLOSE_TOKEN();          
}

//ok
void print_blob_component(BLOB_ELEMENT *Entry)
{
    switch(Entry->Type)
    {
        case BLOB_SPHERE:
                print_blob_sphere(Entry);
                break;
        case BLOB_CYLINDER:
                print_blob_cylinder(Entry);
                break;
        default:
                break;
                
    }
}

//ok
void Print_Blob(OBJECT *Object)
{
        int i;

        BLOB *tmp_obj;
        BLOB_DATA *b_data;        

        tmp_obj = (BLOB *)Object;
        b_data = tmp_obj->Data;

        MOPEN_TOKEN(BLOB_POB_TOKEN);
        MWRITE_NAME(BLOB_POB_TOKEN);

	write_double(b_data->Threshold);
	
	write_word(b_data->Number_Of_Components);

        MOPEN_TOKEN(BLOBCOMPONENT_POB_TOKEN);
        for (i = 0; i < b_data->Number_Of_Components; i++)
        {
                print_blob_component(&(b_data->Entry[i]));
        }
      	MCLOSE_TOKEN();          

        MOPEN_TOKEN(BLOBMOD_POB_TOKEN);
	print_flags(Object);
        print_transform(tmp_obj->Trans);
        print_texture(tmp_obj->Texture);
        print_bound(tmp_obj->Bound);
        print_clipped(tmp_obj->Clip);
      	MCLOSE_TOKEN();          

      	MCLOSE_TOKEN();          
}

//ok
void Print_Box(OBJECT *Object)
{
        BOX *tmp_obj;
        
        tmp_obj = (BOX *)Object;

        MOPEN_TOKEN(BOX_POB_TOKEN);
        MWRITE_NAME(BOX_POB_TOKEN);

        write_dvector(tmp_obj->bounds[0]);
        write_dvector(tmp_obj->bounds[1]);

        print_flags(Object);
        print_transform(tmp_obj->Trans);
        print_texture(tmp_obj->Texture);
        print_bound(tmp_obj->Bound);
        print_clipped(tmp_obj->Clip);
      	MCLOSE_TOKEN();          
}

//ok
void Print_Cone(OBJECT *Object)
{
        CONE *tmp_obj;
        
        tmp_obj = (CONE *)Object;

        if(Test_Flag(tmp_obj, CYLINDER_FLAG))
        {
        	MOPEN_TOKEN(CYLINDER_POB_TOKEN);
        	MWRITE_NAME(CYLINDER_POB_TOKEN);
        	
		write_dvector(tmp_obj->apex);
		write_dvector(tmp_obj->base);
		write_double(tmp_obj->base_radius);

		if(pass_level == 1)
			Invert_Flag(Object, CLOSED_FLAG);
        	
		print_flags(Object);
        	print_transform(tmp_obj->Trans);
        	print_texture(tmp_obj->Texture);
        	print_bound(tmp_obj->Bound);
        	print_clipped(tmp_obj->Clip);
      		MCLOSE_TOKEN();          
        }
        else
        {
        	MOPEN_TOKEN(CONE_POB_TOKEN);
        	MWRITE_NAME(CONE_POB_TOKEN);

		write_dvector(tmp_obj->apex);
		write_double(tmp_obj->apex_radius);

		write_dvector(tmp_obj->base);
		write_double(tmp_obj->base_radius);

		if(pass_level == 1)
			Invert_Flag(Object, CLOSED_FLAG);

		print_flags(Object);
        	print_transform(tmp_obj->Trans);
        	print_texture(tmp_obj->Texture);
        	print_bound(tmp_obj->Bound);
        	print_clipped(tmp_obj->Clip);
      		MCLOSE_TOKEN();          
        }
}

//ok
void Print_CSG_Union(OBJECT *Object)
{
        CSG *tmp_obj;
        OBJECT *csg_obj;

	unsigned int count;        
	int i;
	OBJECT **obj_arr;
        
        tmp_obj = (CSG *)Object;

	// reverse csg order to do it correct
        for(csg_obj = tmp_obj->Children,count = 0; csg_obj != NULL; csg_obj = csg_obj->Sibling,count++);

	obj_arr = (OBJECT **)POV_MALLOC(count * sizeof(OBJECT *) ," Object List");

        for(csg_obj = tmp_obj->Children,count = 0; csg_obj != NULL; csg_obj = csg_obj->Sibling,count++)
		obj_arr[count] = csg_obj;

       	MOPEN_TOKEN(UNION_POB_TOKEN);
       	MWRITE_NAME(UNION_POB_TOKEN);

	MOPEN_TOKEN(GROUP_POB_TOKEN);
        for(i = (count - 1);i > -1;i--)
                Print(obj_arr[i]);
      	MCLOSE_TOKEN();          

	MOPEN_TOKEN(GROUPMOD_POB_TOKEN);
        print_flags(Object);
        print_texture(tmp_obj->Texture);
        print_bound(tmp_obj->Bound);
        print_clipped(tmp_obj->Clip);
      	MCLOSE_TOKEN();          

      	MCLOSE_TOKEN();          

	POV_FREE(obj_arr);
}

//ok
void Print_CSG_Merge(OBJECT *Object)
{
        CSG *tmp_obj;
        OBJECT *csg_obj;

	unsigned int count;        
	int i;
	OBJECT **obj_arr;
        
        tmp_obj = (CSG *)Object;

	// reverse csg order to do it correct
        for(csg_obj = tmp_obj->Children,count = 0; csg_obj != NULL; csg_obj = csg_obj->Sibling,count++);

	obj_arr = (OBJECT **)POV_MALLOC(count * sizeof(OBJECT *) ," Object List");

        for(csg_obj = tmp_obj->Children,count = 0; csg_obj != NULL; csg_obj = csg_obj->Sibling,count++)
		obj_arr[count] = csg_obj;

       	MOPEN_TOKEN(MERGE_POB_TOKEN);
       	MWRITE_NAME(MERGE_POB_TOKEN);
	
	MOPEN_TOKEN(GROUP_POB_TOKEN);
        for(i = (count - 1);i > -1;i--)
                Print(obj_arr[i]);
      	MCLOSE_TOKEN();          

	MOPEN_TOKEN(GROUPMOD_POB_TOKEN);
        print_flags(Object);
        print_texture(tmp_obj->Texture);
        print_bound(tmp_obj->Bound);
        print_clipped(tmp_obj->Clip);
      	MCLOSE_TOKEN();          

      	MCLOSE_TOKEN();          

	POV_FREE(obj_arr);
}

//ok		
void Print_CSG_Inter(OBJECT *Object)
{
        CSG *tmp_obj;
        OBJECT *csg_obj;

	unsigned int count;        
	int i;
	OBJECT **obj_arr;


        tmp_obj = (CSG *)Object;

	// reverse csg order to do it correct
        for(csg_obj = tmp_obj->Children,count = 0; csg_obj != NULL; csg_obj = csg_obj->Sibling,count++);

	obj_arr = (OBJECT **)POV_MALLOC(count * sizeof(OBJECT *) ," Object List");

        for(csg_obj = tmp_obj->Children,count = 0; csg_obj != NULL; csg_obj = csg_obj->Sibling,count++)
		obj_arr[count] = csg_obj;

	if(tmp_obj->csg_type == CSG_INTERSECTION_TYPE)
	{
       		MOPEN_TOKEN(INTERSECTION_POB_TOKEN);
       		MWRITE_NAME(INTERSECTION_POB_TOKEN);

       		MOPEN_TOKEN(GROUP_POB_TOKEN);
        	for(i = (count - 1);i > -1;i--)
                	Print(obj_arr[i]);
      		MCLOSE_TOKEN();          

       		MOPEN_TOKEN(GROUPMOD_POB_TOKEN);
        	print_flags(Object);
        	print_texture(tmp_obj->Texture);
        	print_bound(tmp_obj->Bound);
        	print_clipped(tmp_obj->Clip);
      		MCLOSE_TOKEN();          

      		MCLOSE_TOKEN();          
	}
	else
	{
       		MOPEN_TOKEN(DIFFERENCE_POB_TOKEN);
       		MWRITE_NAME(DIFFERENCE_POB_TOKEN);

       		MOPEN_TOKEN(GROUP_POB_TOKEN);
        	for(i = (count - 1);i > -1;i--)
                	Print(obj_arr[i]);
      		MCLOSE_TOKEN();          

       		MOPEN_TOKEN(GROUPMOD_POB_TOKEN);
        	print_flags(Object);
        	print_texture(tmp_obj->Texture);
        	print_bound(tmp_obj->Bound);
        	print_clipped(tmp_obj->Clip);
      		MCLOSE_TOKEN();          

      		MCLOSE_TOKEN();          
	}


	POV_FREE(obj_arr);
}


//ok
void Print_Disc(OBJECT *Object)
{
        DISC *tmp_obj;
        
        tmp_obj = (DISC *)Object;

       	MOPEN_TOKEN(DISC_POB_TOKEN);
       	MWRITE_NAME(DISC_POB_TOKEN);

	/* vector 1 */
	write_double(0);
	write_double(0);
	write_double(0);

	/* vector 2 */
	write_double(0);
	write_double(0);
	write_double(1);

        write_double(sqrt(tmp_obj->oradius2));
        write_double(sqrt(tmp_obj->iradius2));

        print_flags(Object);
        print_transform(tmp_obj->Trans);
        print_texture(tmp_obj->Texture);
        print_bound(tmp_obj->Bound);
        print_clipped(tmp_obj->Clip);
      	MCLOSE_TOKEN();          
}

void Print_Fractal(OBJECT *Object)
{
        FRACTAL *tmp_obj;
        
        tmp_obj = (FRACTAL *)Object;

       	MOPEN_TOKEN(JULIA_FRACTAL_POB_TOKEN);
       	MWRITE_NAME(JULIA_FRACTAL_POB_TOKEN);

        write_double(tmp_obj->Julia_Parm[0]);
        write_double(tmp_obj->Julia_Parm[1]);
        write_double(tmp_obj->Julia_Parm[2]);
        write_double(tmp_obj->Julia_Parm[3]);

	write_word(tmp_obj->Algebra);
	write_word(tmp_obj->Sub_Type);
        if (tmp_obj->Sub_Type == PWR_STYPE)
        {
                write_double(tmp_obj->exponent.x);
                write_double(tmp_obj->exponent.y);
        }

        write_word(tmp_obj->n);
        write_double(1.0/tmp_obj->Precision);

        write_double(tmp_obj->Slice[0]);
        write_double(tmp_obj->Slice[1]);
        write_double(tmp_obj->Slice[2]);
        write_double(tmp_obj->Slice[3]);

        write_double(tmp_obj->SliceDist);

        print_flags(Object);
        print_transform(tmp_obj->Trans);
        print_texture(tmp_obj->Texture);
        print_bound(tmp_obj->Bound);
        print_clipped(tmp_obj->Clip);
      	MCLOSE_TOKEN();          
}

//ok
void Print_HField(OBJECT *Object)
{
        HFIELD *tmp_obj;
        
        tmp_obj = (HFIELD *)Object;

       	MOPEN_TOKEN(HEIGHT_FIELD_POB_TOKEN);
       	MWRITE_NAME(HEIGHT_FIELD_POB_TOKEN);
        print_image(tmp_obj->File_Type, tmp_obj->filename);
        write_double(tmp_obj->water_level);

        print_flags(Object);
        print_transform(tmp_obj->Trans);
        print_texture(tmp_obj->Texture);
        print_bound(tmp_obj->Bound);
        print_clipped(tmp_obj->Clip);
      	MCLOSE_TOKEN();          
}


//ok
void Print_Lathe(OBJECT *Object)
{
        LATHE *tmp_obj;
        unsigned int i;
        
        tmp_obj = (LATHE *)Object;

       	MOPEN_TOKEN(LATHE_POB_TOKEN);
       	MWRITE_NAME(LATHE_POB_TOKEN);

        write_word(tmp_obj->Spline_Type - 1);

        write_word(tmp_obj->rNumber);

        for(i = 0; i < tmp_obj->rNumber;i++)
        {
		write_dvector_uv(tmp_obj->Points[i]);
        }

        print_flags(Object);
        print_texture(tmp_obj->Texture);
        print_bound(tmp_obj->Bound);
        print_clipped(tmp_obj->Clip);
      	MCLOSE_TOKEN();          
}

//ok
void Print_Mesh(OBJECT *Object)
{
        MESH *tmp_obj;
        MESH_DATA *m_data;

        int i;
        
        tmp_obj = (MESH *)Object;
        m_data = tmp_obj->Data;

       	MOPEN_TOKEN(MESH_POB_TOKEN);
       	MWRITE_NAME(MESH_POB_TOKEN);
        
       	MOPEN_TOKEN(MESHDATA_POB_TOKEN);
	write_word(m_data->Number_Of_Triangles);
	for(i = 0; i < m_data->Number_Of_Triangles;i++)
        {
            	if(m_data->Triangles[i].Smooth)
	    	{
       			MOPEN_TOKEN(SMOOTH_TRIANGLE_POB_TOKEN);
       			MWRITE_NAME(SMOOTH_TRIANGLE_POB_TOKEN);

            		write_dvector_sngl(m_data->Vertices[m_data ->Triangles[i].P1]);
        	        write_dvector_sngl(m_data->Normals[m_data ->Triangles[i].N1]);
            		write_dvector_sngl(m_data->Vertices[m_data ->Triangles[i].P2]);
                	write_dvector_sngl(m_data->Normals[m_data ->Triangles[i].N2]);
            		write_dvector_sngl(m_data->Vertices[m_data ->Triangles[i].P3]);
                	write_dvector_sngl(m_data->Normals[m_data ->Triangles[i].N3]);

			if(m_data->Triangles[i].Texture != -1)
				print_texture(m_data->Textures[m_data->Triangles[i].Texture]);
      			MCLOSE_TOKEN();          
		}
            	else
		{
       			MOPEN_TOKEN(TRIANGLE_POB_TOKEN);
       			MWRITE_NAME(TRIANGLE_POB_TOKEN);

			write_dvector_sngl(m_data->Vertices[m_data ->Triangles[i].P1]);
            		write_dvector_sngl(m_data->Vertices[m_data ->Triangles[i].P2]);
            		write_dvector_sngl(m_data->Vertices[m_data ->Triangles[i].P3]);

			if(m_data->Triangles[i].Texture != -1)
				print_texture(m_data->Textures[m_data->Triangles[i].Texture]);
      			MCLOSE_TOKEN();          
		}
        }        
	MCLOSE_TOKEN();          

       	MOPEN_TOKEN(MESHMOD_POB_TOKEN);
        print_flags(Object);
        print_texture(tmp_obj->Texture);
        print_bound(tmp_obj->Bound);
        print_clipped(tmp_obj->Clip);
      	MCLOSE_TOKEN();          

      	MCLOSE_TOKEN();          
}

//ok
void Print_Plane(OBJECT *Object)
{
        PLANE *tmp_obj;
        
        tmp_obj = (PLANE *)Object;

       	MOPEN_TOKEN(PLANE_POB_TOKEN);
       	MWRITE_NAME(PLANE_POB_TOKEN);
        write_dvector(tmp_obj->Normal_Vector);
        write_double(-tmp_obj->Distance);

        print_flags(Object);
        print_transform(tmp_obj->Trans);
        print_texture(tmp_obj->Texture);
        print_bound(tmp_obj->Bound);
        print_clipped(tmp_obj->Clip);
      	MCLOSE_TOKEN();          
}

//ok
void Print_Light_Source(OBJECT *Object)
{

        LIGHT_SOURCE *tmp_obj;
        
        tmp_obj = (LIGHT_SOURCE *)Object;

       	MOPEN_TOKEN(LIGHT_SOURCE_POB_TOKEN);
       	MWRITE_NAME(LIGHT_SOURCE_POB_TOKEN);
        
	write_word(tmp_obj->Light_Type);

	write_dvector(tmp_obj->Center);
        print_color(tmp_obj->Colour);

        if ((tmp_obj->Light_Type == SPOT_SOURCE) || (tmp_obj->Light_Type == CYLINDER_SOURCE))
        {
                write_dvector(tmp_obj->Points_At);
                write_double(tmp_obj->Coeff);
                write_double(tmp_obj->Radius);
                write_double(tmp_obj->Falloff);
        }

        write_word(tmp_obj->Area_Light);
	if(tmp_obj->Area_Light == TRUE)
        {
                write_dvector(tmp_obj->Axis1);
                write_dvector(tmp_obj->Axis2);
                write_word(tmp_obj->Area_Size1);
                write_word(tmp_obj->Area_Size2);
        }

	write_double(tmp_obj->Fade_Distance);
	write_double(tmp_obj->Fade_Power);
	write_double(tmp_obj->Atmospheric_Attenuation);
	write_double(tmp_obj->Atmosphere_Interaction);
	write_word(tmp_obj->Adaptive_Level);

        write_word(tmp_obj->Jitter);
        write_word(tmp_obj->Track);

        if(tmp_obj->Children != NULL)
        {
       		MOPEN_TOKEN(LOOKS_LIKE_POB_TOKEN);
       		MWRITE_NAME(LOOKS_LIKE_POB_TOKEN);
                Print(tmp_obj->Children);
      		MCLOSE_TOKEN();          
        }

      	MCLOSE_TOKEN();          

}

//ok
void Print_Poly(OBJECT *Object)
{
        int i;
        POLY *tmp_obj;
        
        tmp_obj = (POLY *)Object;

       	MOPEN_TOKEN(POLY_POB_TOKEN);
       	MWRITE_NAME(POLY_POB_TOKEN);
        write_word(tmp_obj->Order);
        for(i = 0; i < term_counts(tmp_obj->Order);i++)
                write_double(tmp_obj->Coeffs[i]);

        print_flags(Object);
        print_transform(tmp_obj->Trans);
        print_texture(tmp_obj->Texture);
        print_bound(tmp_obj->Bound);
        print_clipped(tmp_obj->Clip);
      	MCLOSE_TOKEN();          
}

//ok
void Print_Polygon(OBJECT *Object)
{
        int i;
        POLYGON *tmp_obj;
        POLYGON_DATA *p_data;
        
        tmp_obj = (POLYGON *)Object;
        p_data = tmp_obj->Data;

       	MOPEN_TOKEN(POLYGON_POB_TOKEN);
       	MWRITE_NAME(POLYGON_POB_TOKEN);

        write_word(p_data->Number);
        for(i = 0; i < p_data->Number;i++)
        {
		write_dvector_uv(p_data->Points[i]);
        }

        print_flags(Object);
        print_transform(tmp_obj->Trans);
        print_texture(tmp_obj->Texture);
        print_bound(tmp_obj->Bound);
        print_clipped(tmp_obj->Clip);
      	MCLOSE_TOKEN();          
}

void Print_Prism(OBJECT *Object)
{
        int i;
        PRISM *tmp_obj;
        PRISM_SPLINE *p_data;
        
        
        tmp_obj = (PRISM *)Object;
        p_data = tmp_obj->Spline;

       	MOPEN_TOKEN(PRISM_POB_TOKEN);
       	MWRITE_NAME(PRISM_POB_TOKEN);

        write_word(tmp_obj->Spline_Type - 1);
        write_word(tmp_obj->Sweep_Type - 1);
        write_double(tmp_obj->Height1);
        write_double(tmp_obj->Height2);
        write_word(tmp_obj->rNumber);

        for(i = 0; i < tmp_obj->rNumber;i++)
        {
		write_dvector_uv(tmp_obj->Points[i]);
        }

	if(pass_level == 1)
		Invert_Flag(Object, CLOSED_FLAG);

        print_flags(Object);
        print_transform(tmp_obj->Trans);
        print_texture(tmp_obj->Texture);
        print_bound(tmp_obj->Bound);
        print_clipped(tmp_obj->Clip);
      	MCLOSE_TOKEN();          
}

//ok
void Print_Quadric(OBJECT *Object)
{
        QUADRIC *tmp_obj;
        
        tmp_obj = (QUADRIC *)Object;

       	MOPEN_TOKEN(QUADRIC_POB_TOKEN);
       	MWRITE_NAME(QUADRIC_POB_TOKEN);

        write_dvector(tmp_obj->Square_Terms);
        write_dvector(tmp_obj->Mixed_Terms);
        write_dvector(tmp_obj->Terms);
        write_double(tmp_obj->Constant);

        print_flags(Object);
        print_texture(tmp_obj->Texture);
        print_bound(tmp_obj->Bound);
        print_clipped(tmp_obj->Clip);
      	MCLOSE_TOKEN();          
}

//ok
void Print_Sor(OBJECT *Object)
{
        int i;
        SOR *tmp_obj;
        
        tmp_obj = (SOR *)Object;
    
       	MOPEN_TOKEN(SOR_POB_TOKEN);
       	MWRITE_NAME(SOR_POB_TOKEN);

        write_word(tmp_obj->rNumber);
        for(i = 0; i < tmp_obj->rNumber;i++)
                write_dvector_uv(tmp_obj->Points[i]);
        
	if(pass_level == 1)
		Invert_Flag(Object, CLOSED_FLAG);
        
	print_flags(Object);
        print_transform(tmp_obj->Trans);
        print_texture(tmp_obj->Texture);
        print_bound(tmp_obj->Bound);
        print_clipped(tmp_obj->Clip);
      	MCLOSE_TOKEN();          
}

//ok
void Print_Sphere(OBJECT *Object)
{
        SPHERE *tmp_obj;
	        
        tmp_obj = (SPHERE *)Object;
	
        MOPEN_TOKEN(SPHERE_POB_TOKEN);
        MWRITE_NAME(SPHERE_POB_TOKEN);

	write_dvector(tmp_obj->Center);
        write_double(tmp_obj->Radius);

        print_flags(Object);
        print_transform(tmp_obj->Trans);
        print_texture(tmp_obj->Texture);
        print_bound(tmp_obj->Bound);
        print_clipped(tmp_obj->Clip);
	MCLOSE_TOKEN();
}

void Print_Ellipsoid(OBJECT *Object)
{
        SPHERE *tmp_obj;
        
        tmp_obj = (SPHERE *)Object;

        MOPEN_TOKEN(SPHERE_POB_TOKEN);
        MWRITE_NAME(SPHERE_POB_TOKEN);

	write_dvector(tmp_obj->Center);
        write_double(tmp_obj->Radius);

        print_flags(Object);
        print_transform(tmp_obj->Trans);
        print_texture(tmp_obj->Texture);
        print_bound(tmp_obj->Bound);
        print_clipped(tmp_obj->Clip);
      	MCLOSE_TOKEN();          
}

//ok
void Print_Superellipsoid(OBJECT *Object)
{
        SUPERELLIPSOID *tmp_obj;
        
        tmp_obj = (SUPERELLIPSOID *)Object;

        MOPEN_TOKEN(SUPERELLIPSOID_POB_TOKEN);
        MWRITE_NAME(SUPERELLIPSOID_POB_TOKEN);

        write_double(2.0 / tmp_obj->Power[0]);
        write_double(2.0 / tmp_obj->Power[2]);

        print_flags(Object);
        print_transform(tmp_obj->Trans);
        print_texture(tmp_obj->Texture);
        print_bound(tmp_obj->Bound);
        print_clipped(tmp_obj->Clip);
      	MCLOSE_TOKEN();          
}

//ok
void Print_Torus(OBJECT *Object)
{
        TORUS *tmp_obj;
        
        tmp_obj = (TORUS *)Object;

        MOPEN_TOKEN(TORUS_POB_TOKEN);
        MWRITE_NAME(TORUS_POB_TOKEN);

        write_double(tmp_obj->R);
        write_double(tmp_obj->r);

        print_flags(Object);
        print_transform(tmp_obj->Trans);
        print_texture(tmp_obj->Texture);
        print_bound(tmp_obj->Bound);
        print_clipped(tmp_obj->Clip);
      	MCLOSE_TOKEN();          
}

//ok
void Print_Triangle(OBJECT *Object)
{
        TRIANGLE *tmp_obj;
        
        tmp_obj = (TRIANGLE *)Object;

        MOPEN_TOKEN(TRIANGLE_POB_TOKEN);
        MWRITE_NAME(TRIANGLE_POB_TOKEN);

        write_dvector(tmp_obj->P1);
        write_dvector(tmp_obj->P2);
        write_dvector(tmp_obj->P3);

        print_flags(Object);
        print_texture(tmp_obj->Texture);
        print_bound(tmp_obj->Bound);
        print_clipped(tmp_obj->Clip);
      	MCLOSE_TOKEN();          
}

//ok
void Print_Smooth_Triangle(OBJECT *Object)
{
        SMOOTH_TRIANGLE *tmp_obj;
        
        tmp_obj = (SMOOTH_TRIANGLE *)Object;

        MOPEN_TOKEN(SMOOTH_TRIANGLE_POB_TOKEN);
        MWRITE_NAME(SMOOTH_TRIANGLE_POB_TOKEN);

        write_dvector(tmp_obj->P1);
        write_dvector(tmp_obj->N1);
        write_dvector(tmp_obj->P2);
        write_dvector(tmp_obj->N2);
        write_dvector(tmp_obj->P3);
        write_dvector(tmp_obj->N3);

        print_flags(Object);
        print_texture(tmp_obj->Texture);
        print_bound(tmp_obj->Bound);
        print_clipped(tmp_obj->Clip);
      	MCLOSE_TOKEN();          
}

//ok
void Print_TTF(OBJECT *Object)
{
        TTF *tmp_obj;
        
        tmp_obj = (TTF *)Object;

        MOPEN_TOKEN(TEXT_POB_TOKEN);
        MWRITE_NAME(TEXT_POB_TOKEN);
	
	write_word(TTF_POB_TOKEN);
	write_string(tmp_obj->filename);
	write_string(tmp_obj->string);
        write_double(tmp_obj->depth);
        write_dvector(tmp_obj->offset);

        print_flags(Object);
        print_transform(tmp_obj->Trans);
        print_texture(tmp_obj->Texture);
        print_bound(tmp_obj->Bound);
        print_clipped(tmp_obj->Clip);
      	MCLOSE_TOKEN();          

}

void free_link( link_type *node)
{
	link_type *tmp_node;
	link_type *next_node;

	tmp_node = node;

	while(tmp_node != NULL)
	{
		next_node = tmp_node->next;
		POV_FREE(tmp_node);
		tmp_node = next_node;
	}
}

link_type *create_link( TEXTURE *ptext)
{
	link_type *tmp_node;

	tmp_node = (link_type *)POV_MALLOC(sizeof(link_type),"Texture link");
	tmp_node->next = NULL;
	tmp_node->ptext = ptext;
	return tmp_node;
}

link_type *add_link( link_type *next_node, link_type *tmp_node )
{
	next_node->next = tmp_node;
	return tmp_node;
}


/* CHUNK related */

void write_byte(P_BYTE data)
{
	if(pass_level == 0) return;

	fputc (data,fptr);
}


void write_word(P_WORD data)
{
	if(pass_level == 0) return;

    	fputc(data & 0xFF, fptr);
    	fputc((data >> 8) & 0xFF, fptr);
}

void write_dword(P_DWORD data)
{
	if(pass_level == 0) return;

    	fputc(data & 0xFF, fptr);
    	fputc((data >> 8) & 0xFF, fptr);
    	fputc((data >> 16) & 0xFF, fptr);
    	fputc((data >> 24) & 0xFF, fptr);
}

void write_double(P_DOUBLE data)
{
	if(pass_level == 0) return;
#ifdef __IEEE
	fwrite (&data, 8, 1, fptr);
#endif //__IEEE
}

void write_string(char *data)
{
	unsigned int i;
	unsigned int len;

	if(pass_level == 0) return;

	len = strlen(data);
	if(len > 255)
		len = 255;

	write_byte((P_BYTE)len);

	for (i = 0; i < len; i++)
		write_byte((P_BYTE)data[i]);

	if(!(len & 1))	// word alignment
		write_byte(0x00);
}

void write_dvector(P_DVECTOR v)
{
	if(pass_level == 0) return;

	write_double(v[0]);
	write_double(v[1]);
	write_double(v[2]);
}

void write_dvector_uv(UV_VECT v)
{
	if(pass_level == 0) return;

	write_double(v[0]);
	write_double(v[1]);
}

void write_dvector_sngl(SNGL_VECT v)
{
	if(pass_level == 0) return;

	write_double((double)v[0]);
	write_double((double)v[1]);
	write_double((double)v[2]);
}

void start_chunk(CHUNK *main_chunk, P_WORD tag)
{
	if(pass_level == 0) return;

	main_chunk->start  = ftell(fptr);
	write_word(tag);
	write_dword(0L);
}

void end_chunk(CHUNK *main_chunk)
{
	P_DWORD tmp_pos;

	if(pass_level == 0) return;

	tmp_pos = ftell(fptr);

	/* check for chunk aligment */
	if(tmp_pos & 1)
		write_byte(0x00);
	tmp_pos = ftell(fptr);

	fseek (fptr, main_chunk->start + 2, 0);
	write_dword(tmp_pos - main_chunk->start);
	fseek (fptr, tmp_pos, 0);
}

