/* @(#)gifload.c 1.4   8/27/96 */

/******************************************************************************
* Threedom: a 3D polygon renderer                                             *
* (C) Copyright 1996 by Philip Stephens                                       *
* (C) Copyright 1996 by the IBM Corporation                                   *
* All Rights Reserved                                                         *
*                                                                             *
* Permission to use, copy, modify, and distribute this software and its       *
* documentation without fee for any non-commerical purpose is hereby granted, *
* provided that the above copyright notice appears on all copies and that     *
* both that copyright notice and this permission notice appear in all         *
* supporting documentation.                                                   *
*                                                                             *
* NO REPRESENTATIONS ARE MADE ABOUT THE SUITABILITY OF THIS SOFTWARE FOR ANY  *
* PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.       *
* NEITHER PHILIP STEPHENS OR IBM SHALL BE LIABLE FOR ANY DAMAGES SUFFERED BY  *
* THE USE OF THIS SOFTWARE.                                                   *
******************************************************************************/

/*
 * gifload.c - based strongly on...
 *
 * gif2ras.c - Converts from a Compuserve GIF (tm) image to a Sun Raster image.
 *
 * Copyright (c) 1988, 1989 by Patrick J. Naughton
 *
 * Author: Patrick J. Naughton
 * naughton@wind.sun.com
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * This file is provided AS IS with no warranties of any kind.  The author
 * shall have no liability with respect to the infringement of copyrights,
 * trade secrets or any patents by this file or any part thereof.  In no
 * event will the author be liable for any lost revenue or profits or
 * other special, indirect and consequential damages.
 *
 * The Graphics Interchange Format(c) is the Copyright property of CompuServe
 * Incorporated.  GIF(sm) is a Service Mark property of CompuServe
 * Incorporated.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "typedef.h"
#include "glib.h"
#include "fileio.h"

#define NEXTBYTE	(*ptr++)
#define IMAGESEP	0x2c
#define GRAPHIC_EXT	0xf9
#define PLAINTEXT_EXT	0x01
#define APPLICATION_EXT	0xff
#define COMMENT_EXT	0xfe
#define START_EXTENSION	0x21
#define INTERLACEMASK	0x40
#define COLORMAPMASK	0x80

static int
    BitOffset = 0,		/* Bit Offset of next code */
    XC = 0, YC = 0,		/* Output X and Y coords of current pixel */
    Pass = 0,			/* Used by output routine if interlaced pic */
    OutCount = 0,		/* Decompressor output 'stack count' */
    Width, Height,		/* image dimensions */
    LeftOfs, TopOfs,		/* image offset */
    BitsPerPixel,		/* Bits per pixel, read from GIF header */
    BytesPerScanline,		/* bytes per scanline in output raster */
    ColorMapSize,		/* number of colors */
    CodeSize,			/* Code size, read from GIF header */
    InitCodeSize,		/* Starting code size, used during Clear */
    Code,			/* Value returned by ReadCode */
    MaxCode,			/* limiting value for current code size */
    ClearCode,			/* GIF clear code */
    EOFCode,			/* GIF end-of-information code */
    CurCode, OldCode, InCode,	/* Decompressor variables */
    FirstFree,			/* First free code, generated per GIF spec */
    FreeCode,			/* Decompressor, next free slot in hash table*/
    FinChar,			/* Decompressor variable */
    BitMask,			/* AND mask for data size */
    ReadMask;			/* Code AND mask for current code size */

static int Interlace, HasColormap;

static pixel *Image;		/* The result array */
static byte *RawGIF;		/* The heap array to hold it, raw */
static byte *Raster;		/* The raster data stream, unblocked */

    /* The hash table used by the decompressor */
static int Prefix[4096];
static int Suffix[4096];

    /* An output array used by the decompressor */
static int OutCode[1025];

    /* The color map, read from the GIF header */
static byte Red, Green, Blue;
static pixel localColor[256];

static const char *id87 = "GIF87a";
static const char *id89 = "GIF89a";

/*
 * Forward declarations.
 */
static int	ReadCode(void);
static void	AddToPixel(byte);

void
load_GIF(FILE *fp, pixel **image_ptr, int *image_width, int *image_height)
{
	int filesize;
	byte ch, ch1;
	byte *ptr, *ptr1;
	int i;

	/*
	 * Initialise global variables.
	 */
	BitOffset = 0;
	XC = 0;
	YC = 0;
	Pass = 0;
	OutCount = 0;

	/*
	 * Determine the size of the file.
	 */
	fseek(fp, 0L, 2);
	filesize = ftell(fp);
	fseek(fp, 0L, 0);

	/*
	 * Allocate memory for the raw GIF data and the raster image.
	 */
	if ((RawGIF = (byte *)malloc(filesize)) == NULL)
		memory_error("raw GIF data");
	if ((Raster = (byte *)malloc(filesize)) == NULL) {
		free(RawGIF);
		memory_error("raster image");
	}

	/*
	 * Read the GIF into the RawGIF buffer.
	 */
	if (fread(RawGIF, filesize, 1, fp) != 1)
		file_error("GIF data read failed");

	/*
	 * Verify the header, then skip over it.
	 */
	ptr = RawGIF;
	if (strncmp((const char *)ptr, id87, 6) &&
	    strncmp((const char *)ptr, id89, 6))
		file_error("not a GIF file");
	ptr += 6;

	/*
	 * Skip over screen dimensions in GIF screen descriptor.
	 */
	ptr += 4;

	/*
	 * Check whether GIF has it's own colourmap, then determine the number
	 * of bits per pixel, and hence the bit mask and size of the colourmap.
	 */
	ch = NEXTBYTE;
	HasColormap = ((ch & COLORMAPMASK) ? TRUE : FALSE);
	BitsPerPixel = (ch & 7) + 1;
	ColorMapSize = 1 << BitsPerPixel;
	BitMask = ColorMapSize - 1;

	/*
	 * Skip over background colour.
	 */
	ptr++;
  
	/*
	 * Next byte should be zero.
	 */
	if (NEXTBYTE)
		file_error("corrupt GIF file (bad screen descriptor)");

	/*
	 * Read in GIF colormap, if there is one.
	 * XXX -- what should we do if there isn't one?
	 */
	if (HasColormap)
		for (i = 0; i < ColorMapSize; i++) {
			Red = NEXTBYTE;
			Green = NEXTBYTE;
			Blue = NEXTBYTE;	
			localColor[i] = get_pixel(Red, Green, Blue);
		}

	/*
	 * Look for image seperator, processing any useful image extensions
	 * along the way.
	 */
	for (ch = NEXTBYTE; ch != IMAGESEP; ch = NEXTBYTE) {
		if (ch != START_EXTENSION)
			file_error("corrupt GIF89a file");
		switch (ch = NEXTBYTE) {
		case GRAPHIC_EXT:
			ch = NEXTBYTE;
			ptr += ch;
			break;
		case PLAINTEXT_EXT:
			break;
		case APPLICATION_EXT:
			break;
		case COMMENT_EXT:
			break;
		default:
			file_error("invalid GIF89 extension");
	  	}
		while ((ch = NEXTBYTE) != 0)
			ptr += ch;
	}

	/*
	 * Now read in values from the image descriptor.
	 */
	ch = NEXTBYTE;
	LeftOfs   = ch + (NEXTBYTE << 8);
	ch = NEXTBYTE;
	TopOfs = ch + (NEXTBYTE << 8);
	ch = NEXTBYTE;
	Width  = ch + (NEXTBYTE << 8);
	ch = NEXTBYTE;
	Height = ch + (NEXTBYTE << 8);
	Interlace = ((NEXTBYTE & INTERLACEMASK) ? TRUE : FALSE);

	/*
	 * Note that I ignore the possible existence of a local color map.
	 * I'm told there aren't many files around that use them, and the spec
	 * says it's defined for future use.  This could lead to an error
	 * reading some files. 
	 */

	/*
	 * Start reading the raster data. First we get the intial code size
 	 * and compute decompressor constant values, based on this code size.
	 */

	CodeSize = NEXTBYTE;
	ClearCode = 1 << CodeSize;
	EOFCode = ClearCode + 1;
	FirstFree = ClearCode + 2;
	FreeCode = FirstFree;

	/*
	 * The GIF spec has it that the code size used to compute the above
	 * values is the code size given in the file, but the code size used
	 * in compression/decompression is one more than this. (thus the ++).
	 */
	CodeSize++;
	InitCodeSize = CodeSize;
	MaxCode = 1 << CodeSize;
	ReadMask = MaxCode - 1;

	/*
	 * Read the raster data.  Here we just transpose it from the GIF array
	 * to the Raster array, turning it from a series of blocks into one
	 * long data stream, which makes life much easier for ReadCode().
	 */
	ptr1 = Raster;
	do {
		ch1 = NEXTBYTE;
		ch = ch1;
		while (ch--)
			*ptr1++ = NEXTBYTE;
		if ((ptr1 - Raster) > filesize)
			file_error("corrupt GIF file (unblock)");
	} while(ch1);

	/*
	 * We're done with the raw data now.
	 */
	free(RawGIF);
	if ((Image = (pixel *)malloc(Width * Height * sizeof(pixel))) == NULL)
		memory_error("GIF image");
	BytesPerScanline = Width * sizeof(pixel);

	/*
	 * Decompress the file, continuing until you see the GIF EOF code.
	 * One obvious enhancement is to add checking for corrupt files here.
	 */
	Code = ReadCode();
	while (Code != EOFCode) {
		/*
		 * Clear code sets everything back to its initial value, then
		 * reads the immediately subsequent code as uncompressed data.
		 */
		if (Code == ClearCode) {
			CodeSize = InitCodeSize;
			MaxCode  = 1 << CodeSize;
			ReadMask = MaxCode - 1;
			FreeCode = FirstFree;
			Code = ReadCode();
			CurCode = Code;
			OldCode = Code;
			FinChar = CurCode & BitMask;
			AddToPixel((byte)FinChar);
		}
		/*
		 * If not a clear code, then must be data: save same as CurCode
		 * and InCode.
		 */
		 else {
			InCode = Code;
			CurCode = Code;

			/*
			 * If greater or equal to FreeCode, not in the hash
			 * table yet; repeat the last character decoded.
			 */
			if (CurCode >= FreeCode) {
				CurCode = OldCode;
				OutCode[OutCount++] = FinChar;
			}

			/*
			 * Unless this code is raw data, pursue the chain
			 * pointed to by CurCode through the hash table to its
			 * end; each code in the chain puts its associated
			 * output code on the output queue.
			 */
	    		while (CurCode > BitMask) {
				if (OutCount > 1024)
					file_error("Corrupt GIF file!\n");
				OutCode[OutCount++] = Suffix[CurCode];
				CurCode = Prefix[CurCode];
			}

			/*
			 * The last code in the chain is treated as raw data.
			 */
			FinChar = CurCode & BitMask;
			OutCode[OutCount++] = FinChar;

			/*
			 * Now we put the data out to the Output routine.
			 * It's been stacked LIFO, so deal with it that way...
			 */
			for (i = OutCount - 1; i >= 0; i--)
				AddToPixel((byte)OutCode[i]);
			OutCount = 0;

			/*
			 * Build the hash table on-the-fly. No table is stored
			 * in the file.
			 */

			Prefix[FreeCode] = OldCode;
			Suffix[FreeCode] = FinChar;
			OldCode = InCode;

			/* Point to the next slot in the table.  If we exceed
			 * the current MaxCode value, increment the code size
			 * unless it's already 12.  If it is, do nothing: the 
			 * next code decompressed better be CLEAR.
			 */
			FreeCode++;
			if (FreeCode >= MaxCode && CodeSize < 12) {
				CodeSize++;
				MaxCode *= 2;
				ReadMask = (1 << CodeSize) - 1;
			}
		}
		Code = ReadCode();
	}

	/*
	 * Free the raster buffer and return the image pointer and dimensions.
	 */
    	free(Raster);
	*image_ptr = Image;
	*image_width = Width;
	*image_height = Height;
}

/*
 * Fetch the next code from the raster data stream.  The codes can be
 * any length from 3 to 12 bits, packed into 8-bit bytes, so we have to
 * maintain our location in the Raster array as a BIT Offset.  We compute
 * the byte Offset into the raster array by dividing this by 8, pick up
 * three bytes, compute the bit Offset into our 24-bit chunk, shift to
 * bring the desired code to the bottom, then mask it off and return it. 
 */
static int
ReadCode(void)
{
	int RawCode, ByteOffset;

	ByteOffset = BitOffset / 8;
	RawCode = Raster[ByteOffset] + (Raster[ByteOffset + 1] << 8);

	if (CodeSize >= 8)
		RawCode += (Raster[ByteOffset + 2] << 16);

	RawCode  >>= (BitOffset % 8);
	BitOffset += CodeSize;

	return(RawCode & ReadMask);
}

static void
AddToPixel(byte Index)
{
	/*
	 * Convert pixel value from an index into the GIF's colourmap to an
	 * index into our standard 6x6x6 colourmap.
	 */
	if (YC < Height)
		*(Image + XC * Height + YC) = localColor[Index];

	/*
	 * Update the X-coordinate, and if it overflows, update the
	 * Y-coordinate.  If a non-interlaced picture, just increment YC to the
	 * next scan line. If it's interlaced, deal with the interlace as
	 * described in the GIF spec.  Put the decoded scan line out to the
	 * image buffer if we haven't gone past the bottom of it.
	 */
	if (++XC == Width) {
		XC = 0;
		if (!Interlace)
			YC++;
		else {
			switch (Pass) {
			case 0:
				YC += 8;
				if (YC >= Height) {
					Pass++;
					YC = 4;
				}
				break;
			case 1:
				YC += 8;
				if (YC >= Height) {
					Pass++;
					YC = 2;
				}
				break;
			case 2:
				YC += 4;
				if (YC >= Height) {
					Pass++;
					YC = 1;
				}
				break;
			case 3:
				YC += 2;
	    		}
		}
    	}
}
