/* @(#)render.c 1.13   8/27/96 */

/******************************************************************************
* Threedom: a 3D polygon renderer                                             *
* (C) Copyright 1996 by Philip Stephens                                       *
* (C) Copyright 1996 by the IBM Corporation                                   *
* All Rights Reserved                                                         *
*                                                                             *
* Permission to use, copy, modify, and distribute this software and its       *
* documentation without fee for any non-commerical purpose is hereby granted, *
* provided that the above copyright notice appears on all copies and that     *
* both that copyright notice and this permission notice appear in all         *
* supporting documentation.                                                   *
*                                                                             *
* NO REPRESENTATIONS ARE MADE ABOUT THE SUITABILITY OF THIS SOFTWARE FOR ANY  *
* PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.       *
* NEITHER PHILIP STEPHENS OR IBM SHALL BE LIABLE FOR ANY DAMAGES SUFFERED BY  *
* THE USE OF THIS SOFTWARE.                                                   *
******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#ifdef BSD
#include <strings.h>
#endif
#include <stdarg.h>
#include <math.h>

#include "typedef.h"
#include "main.h"
#include "glib.h"
#include "fileio.h"
#include "dither.h"

/*
 * Pointer to last screen point.
 */
static spoint *last_spoint_ptr;

/**********************************************************
 * Transform a vertex by the viewing position and angles. *
 **********************************************************/

static void
transform_vertex(vertex *vertex_ptr)
{
	fixed dx, dy, dz;
	fixed tx, ty, tz;
	
	/*
	 * Translate vertex so that viewpoint is at origin.
	 */
	dx = vertex_ptr->x - view_x;
	dy = vertex_ptr->y - view_y;
	dz = vertex_ptr->z - view_z;

	/*
	 * Rotate vertex around y-axis first.
	 */
	tx = fmul(dx, cosine[view_angle_y]) - fmul(dz, sine[view_angle_y]);
	ty = dy;
	tz = fmul(dx, sine[view_angle_y]) + fmul(dz, cosine[view_angle_y]);

	/*
	 * Rotate vertex around x-axis second.
	 */
	vertex_ptr->tx = tx;
	vertex_ptr->ty = fmul(ty, cosine[view_angle_x]) +
		fmul(tz, sine[view_angle_x]);
	vertex_ptr->tz = fmul(tz, cosine[view_angle_x]) -
		fmul(ty, sine[view_angle_x]);
}

/************************************************************************
 * Check to see if a polygon is behind the viewing plane or facing away *
 * from the viewer.							*
 ************************************************************************/

static boolean
polygon_visible(polygon *polygon_ptr)
{
	vertex **polygon_vertex_ptr_list;
	vertex *vertex2_ptr, *normal_vertex_ptr;
	int polygon_vertices, vertex_no;
	boolean invisible;
	fixed Ntx, Nty, Ntz;
	fixed dot_product;

	/*
	 * Step through the list of polygon vertices and check their Z axis
	 * coordinates.  If all are in front of the viewing plane, return
	 * FALSE.
	 */
	polygon_vertex_ptr_list = polygon_ptr->vertex_ptr_list;
	polygon_vertices = polygon_ptr->vertices;
	invisible = TRUE;
	for (vertex_no = 0; vertex_no < polygon_vertices; vertex_no++) {
		vertex *vertex_ptr;
	
		vertex_ptr = polygon_vertex_ptr_list[vertex_no];
		if (vertex_ptr->tz >= FIXED_ONE) {
			invisible = FALSE;
			break;
		}
	}
	if (invisible)
		return(!invisible);

	/*
	 * Next transform the normal vertex, and subtract it from transformed
	 * vertex 2 of the polygon to obtain the transformed normal vector.
	 */
	normal_vertex_ptr = &polygon_ptr->normal_vertex;
	transform_vertex(normal_vertex_ptr);
	vertex2_ptr = polygon_vertex_ptr_list[1];
	Ntx = normal_vertex_ptr->tx - vertex2_ptr->tx;
	Nty = normal_vertex_ptr->ty - vertex2_ptr->ty;
	Ntz = normal_vertex_ptr->tz - vertex2_ptr->tz;

	/*
	 * We now calculate the dot product between the view vector, which is
	 * the vector from the origin to the transformed vertex 2 of the
	 * polygon, and the transformed normal vector.  If the result is
	 * negative, we're looking at the front of the polygon and so it's
	 * visible.
	 */
	dot_product = fmul(vertex2_ptr->tx, Ntx) +
		      fmul(vertex2_ptr->ty, Nty) +
		      fmul(vertex2_ptr->tz, Ntz);
	return(dot_product < 0);
}

/************************************************************************
 * Find the top or bottom screen point in the global screen point list. *
 ************************************************************************/

static spoint *
find_spoint(int find_flag)
{
	spoint *spoint_ptr;
	spoint *best_spoint_ptr;

	best_spoint_ptr = spoint_list;
	for (spoint_ptr = spoint_list + 1; spoint_ptr <= last_spoint_ptr;
	     spoint_ptr++)
		if (find_flag == FIND_TOP) {
			if (spoint_ptr->sy < best_spoint_ptr->sy)
				best_spoint_ptr = spoint_ptr;
		} else {
			if (spoint_ptr->sy > best_spoint_ptr->sy)
				best_spoint_ptr = spoint_ptr;
		}
	return(best_spoint_ptr);
}

/*************************************
 * Return the previous screen point. *
 *************************************/

static spoint *
prev_spoint(spoint *spoint_ptr)
{
	if (spoint_ptr == spoint_list)
		spoint_ptr = last_spoint_ptr;
	else
		spoint_ptr--;
	return(spoint_ptr);
}

/*********************************
 * Return the next screen point. *
 *********************************/

static spoint *
next_spoint(spoint *spoint_ptr)
{
	if (spoint_ptr == last_spoint_ptr)
		spoint_ptr = spoint_list;
	else
		spoint_ptr++;
	return(spoint_ptr);
}

/************************************************************************
 * Clip an edge by finding the intersection for both the vertex and the *
 * texture point.                                                       *
 ************************************************************************/

static void
clip_edge(vertex *vertex1_ptr, vertex *vertex2_ptr,
	  tpoint *tpoint1_ptr, tpoint *tpoint2_ptr,
	  vertex *clipped_vertex_ptr, tpoint *clipped_tpoint_ptr)
{
	fixed fraction;

	/*
	 * Find the fraction of the edge that is behind the viewing plane.
	 */
	fraction = fdiv(FIXED_ONE - vertex1_ptr->tz,
		        vertex2_ptr->tz - vertex1_ptr->tz);

	/*
	 * Compute the vertex intersection.
	 */
	clipped_vertex_ptr->tx = vertex1_ptr->tx +
		fmul(fraction, vertex2_ptr->tx - vertex1_ptr->tx);
	clipped_vertex_ptr->ty = vertex1_ptr->ty +
		fmul(fraction, vertex2_ptr->ty - vertex1_ptr->ty);
	clipped_vertex_ptr->tz = FIXED_ONE;

	/*
	 * Compute the tpoint intersection.
	 */
	clipped_tpoint_ptr->u = tpoint1_ptr->u +
		fmul(fraction, tpoint2_ptr->u - tpoint1_ptr->u);
	clipped_tpoint_ptr->v = tpoint1_ptr->v +
		fmul(fraction, tpoint2_ptr->v - tpoint1_ptr->v);
}

/*****************************************************************************
 * Project a vertex into screen space, and add the resulting screen point to *
 * the global screen point list.  Also compute 1/tz, u/tz and v/tz for that  *
 * screen point.							     *
 *****************************************************************************/

static void
add_spoint_to_list(vertex *vertex_ptr, tpoint *tpoint_ptr)
{
	spoint *spoint_ptr;
	fixed px, py, one_on_tz;

	/*
	 * Project vertex into screen space.
	 */
	spoint_ptr = &spoint_list[spoints];
	px = fdiv(vertex_ptr->tx, vertex_ptr->tz);
	py = fdiv(vertex_ptr->ty, vertex_ptr->tz);
	spoint_ptr->sx = half_window_width + fmul(px, half_window_width);
	spoint_ptr->sy = half_window_height - fmul(py, half_window_height);
		
	/*
	 * Compute 1/tz.
	 */
	one_on_tz = fdiv(FIXED_ONE, vertex_ptr->tz);
	spoint_ptr->one_on_tz = one_on_tz;

	/*
	 * Generate u/tz and v/tz for this screen point.
	 */
	spoint_ptr->u_on_tz = fmul(tpoint_ptr->u, one_on_tz);
	spoint_ptr->v_on_tz = fmul(tpoint_ptr->v, one_on_tz);
	spoints++;
}

/********************************************************************
 * Clip a polygon against the viewing plane at z = 1, and store the *
 * projected screen points of the new polygon in the global screen  *
 * point list.                                                      *
 ********************************************************************/

static void
clip_polygon(polygon *polygon_ptr)
{
	vertex **polygon_vertex_ptr_list;
	tpoint **polygon_tpoint_ptr_list;
	int polygon_vertices;
	int vertex1_no, vertex2_no;

	/*
	 * Initialise some local variables.
	 */
	polygon_vertex_ptr_list = polygon_ptr->vertex_ptr_list;
	polygon_tpoint_ptr_list = polygon_ptr->tpoint_ptr_list;
	polygon_vertices = polygon_ptr->vertices;

	/*
	 * Step through the vertices of the polygon, checking the status of
	 * each edge compared against the viewing plane, and output the new
	 * set of vertices.
	 */
	spoints = 0;
	vertex1_no = polygon_vertices - 1;
	for (vertex2_no = 0; vertex2_no < polygon_vertices; vertex2_no++) {
		vertex *vertex1_ptr, *vertex2_ptr, vertex_clipped;
		tpoint *tpoint1_ptr, *tpoint2_ptr, tpoint_clipped;

		/*
		 * Get the vertex and tpoint pointers for the current edge.
		 */
		vertex1_ptr = polygon_vertex_ptr_list[vertex1_no];
		vertex2_ptr = polygon_vertex_ptr_list[vertex2_no];
		tpoint1_ptr = polygon_tpoint_ptr_list[vertex1_no];
		tpoint2_ptr = polygon_tpoint_ptr_list[vertex2_no];

		/*
		 * Compare the edge against the viewing plane at z=1 and add
		 * zero, one or two vertices to the global vertex list.
		 */
		if (vertex1_ptr->tz >= FIXED_ONE) {
			/*
			 * If the edge is entirely behind the viewing plane,
			 * add the end point to the global vertex list.
			 */
			if (vertex2_ptr->tz >= FIXED_ONE)
				add_spoint_to_list(vertex2_ptr, tpoint2_ptr);
			/*
			 * If the edge is crossing over to the front of the
			 * viewing plane, add the intersection point to the
			 * global vertex list.
			 */
			else {
				clip_edge(vertex1_ptr, vertex2_ptr,
					tpoint1_ptr, tpoint2_ptr,
					&vertex_clipped, &tpoint_clipped);
				add_spoint_to_list(&vertex_clipped,
					&tpoint_clipped);
			}			
		} else {
			/*
			 * If the edge is crossing over to the back of the
			 * viewing plane, add the intersection point and the
			 * edge's end point to the global vertex list.
			 */
			if (vertex2_ptr->tz >= FIXED_ONE) {
				clip_edge(vertex1_ptr, vertex2_ptr,
					tpoint1_ptr, tpoint2_ptr,
					&vertex_clipped, &tpoint_clipped);
				add_spoint_to_list(&vertex_clipped,
					&tpoint_clipped);
				add_spoint_to_list(vertex2_ptr, tpoint2_ptr);
			}
		}
		
		/*
		 * Move onto the next edge.
		 */
		vertex1_no = vertex2_no;
	}

	/*
	 * Generate a pointer to the end of the list.
	 */
	last_spoint_ptr = &spoint_list[spoints - 1];
}

/*****************************************************************************
 * Compute the slopes for the values to be interpolated down a polygon edge. *
 *****************************************************************************/

static void
compute_slopes(spoint *spoint1_ptr, spoint *spoint2_ptr, fixed sy,
	       fixed *sx, fixed *sx_slope,
	       fixed *one_on_tz, fixed *one_on_tz_slope,
	       fixed *u_on_tz, fixed *u_on_tz_slope,
	       fixed *v_on_tz, fixed *v_on_tz_slope)
{
	fixed delta_sy;

	/*
	 * Compute the slopes of the interpolants.
	 */
	delta_sy = ABS(spoint2_ptr->sy - spoint1_ptr->sy);
	*sx_slope = fdiv(spoint2_ptr->sx - spoint1_ptr->sx, delta_sy);
	*one_on_tz_slope = fdiv(spoint2_ptr->one_on_tz - spoint1_ptr->one_on_tz,
		delta_sy);
	*u_on_tz_slope = fdiv(spoint2_ptr->u_on_tz - spoint1_ptr->u_on_tz,
		delta_sy);
	*v_on_tz_slope = fdiv(spoint2_ptr->v_on_tz - spoint1_ptr->v_on_tz,
		delta_sy);

	/*
	 * Determine the initial values for the interpolants.
	 */
	*sx = spoint1_ptr->sx;
	*one_on_tz = spoint1_ptr->one_on_tz;
	*u_on_tz = spoint1_ptr->u_on_tz;
	*v_on_tz = spoint1_ptr->v_on_tz;
	delta_sy = sy - spoint1_ptr->sy;
	if (delta_sy > 0) {
		*sx += fmul(*sx_slope, delta_sy);
		*one_on_tz += fmul(*one_on_tz_slope, delta_sy);
		*u_on_tz += fmul(*u_on_tz_slope, delta_sy);
		*v_on_tz += fmul(*v_on_tz_slope, delta_sy);
	}
}

/*********************
 * Render a polygon. *
 *********************/

static void
render_polygon(polygon *polygon_ptr)
{
	spoint *top_spoint_ptr, *bottom_spoint_ptr;
	spoint *left_spoint1_ptr, *left_spoint2_ptr;
	spoint *right_spoint1_ptr, *right_spoint2_ptr;
	fixed sy;
	fixed left_sx, right_sx;
	fixed left_one_on_tz, right_one_on_tz;
	fixed left_u_on_tz, right_u_on_tz;
	fixed left_v_on_tz, right_v_on_tz;
	fixed left_sx_slope, right_sx_slope;
	fixed left_one_on_tz_slope, right_one_on_tz_slope;
	fixed left_u_on_tz_slope, right_u_on_tz_slope;
	fixed left_v_on_tz_slope, right_v_on_tz_slope;
	texture *texture_ptr;
	pixel *image_ptr;
	int width_mask, height_mask, column_shift;
	int light_level;

	/*
	 * Clip the polygon against the viewing plane, creating a new polygon
	 * whose screen points are in the global screen point list.
	 */
	clip_polygon(polygon_ptr);

	/*
	 * Find the topmost screen point.  If it's below the bottom edge of
	 * the screen, there is nothing to render.
	 */
	top_spoint_ptr = find_spoint(FIND_TOP);
	if (INT(CEIL(top_spoint_ptr->sy)) >= window_height)
		return;

	/*
	 * Find the bottommost screen point.  If it's above the top edge of the
	 * screen, there is nothing to render.
	 */
	bottom_spoint_ptr = find_spoint(FIND_BOTTOM);
	if (bottom_spoint_ptr->sy < 0)
		return;

	/*
	 * If the top screen point is below the top edge of the screen, the
	 * initial left and right edge is that with the top screen point as a
	 * start point.  Otherwise we find the first left and right edge
	 * crossing the top edge of the screen.
	 */
	left_spoint1_ptr = top_spoint_ptr;
	left_spoint2_ptr = prev_spoint(top_spoint_ptr);
	right_spoint1_ptr = top_spoint_ptr;
	right_spoint2_ptr = next_spoint(top_spoint_ptr);
	if (top_spoint_ptr->sy < 0) {
		while (left_spoint2_ptr->sy < 0) {
			left_spoint1_ptr = left_spoint2_ptr;
			left_spoint2_ptr = prev_spoint(left_spoint2_ptr);
		}
		while (right_spoint2_ptr->sy < 0) {
			right_spoint1_ptr = right_spoint2_ptr;
			right_spoint2_ptr = next_spoint(right_spoint2_ptr);
		}
	}

	/*
	 * If either the left or right edge is horizontal, skip over it.
	 */
	if (left_spoint1_ptr->sy == left_spoint2_ptr->sy) {
		left_spoint1_ptr = left_spoint2_ptr;
		left_spoint2_ptr = prev_spoint(left_spoint2_ptr);
	}
	if (right_spoint1_ptr->sy == right_spoint2_ptr->sy) {
		right_spoint1_ptr = right_spoint2_ptr;
		right_spoint2_ptr = next_spoint(right_spoint2_ptr);
	}

	/*
	 * Determine the initial value for sy.
	 */
	sy = CEIL(top_spoint_ptr->sy);
	if (sy < 0)
		sy = 0;

	/*
	 * Compute the initial values and slopes of all the interpolants.
	 */
	compute_slopes(left_spoint1_ptr, left_spoint2_ptr, sy,
		&left_sx, &left_sx_slope,
		&left_one_on_tz, &left_one_on_tz_slope,
		&left_u_on_tz, &left_u_on_tz_slope,
		&left_v_on_tz, &left_v_on_tz_slope);
	compute_slopes(right_spoint1_ptr, right_spoint2_ptr, sy,
		&right_sx, &right_sx_slope,
		&right_one_on_tz, &right_one_on_tz_slope,
		&right_u_on_tz, &right_u_on_tz_slope,
		&right_v_on_tz, &right_v_on_tz_slope);

	/*
	 * Enter rendering loop.
	 *
	 * We render the polygon row by row, recomputing the slopes of the
	 * interpolants as as we reach the next left and right vertex, until
	 * we've reached the bottom vertex.
	 *
	 * XXX -- At each new vertex we should refresh the values of the
	 * interpolants to counteract any roundoff errors that might otherwise
	 * result over time.
	 */
	texture_ptr = polygon_ptr->texture_ptr;
	image_ptr = texture_ptr->image_ptr;
	width_mask = texture_ptr->width_mask;
	height_mask = texture_ptr->height_mask;
	column_shift = texture_ptr->column_shift;
	light_level = polygon_ptr->light_level;
	do {
		if (INT(sy) >= window_height - 1 ||
		    sy >= bottom_spoint_ptr->sy)
			break;
		if (sy >= left_spoint2_ptr->sy) {
			left_spoint1_ptr = left_spoint2_ptr;
			left_spoint2_ptr = prev_spoint(left_spoint2_ptr);
			compute_slopes(left_spoint1_ptr, left_spoint2_ptr, sy,
				&left_sx, &left_sx_slope,
				&left_one_on_tz, &left_one_on_tz_slope,
				&left_u_on_tz, &left_u_on_tz_slope,
				&left_v_on_tz, &left_v_on_tz_slope);
		}
		if (sy >= right_spoint2_ptr->sy) {
			right_spoint1_ptr = right_spoint2_ptr;
			right_spoint2_ptr = next_spoint(right_spoint2_ptr);
			compute_slopes(right_spoint1_ptr, right_spoint2_ptr, sy,
				&right_sx, &right_sx_slope,
				&right_one_on_tz, &right_one_on_tz_slope,
				&right_u_on_tz, &right_u_on_tz_slope,
				&right_v_on_tz, &right_v_on_tz_slope);
		}
		render_polygon_row(INT(sy), left_sx, right_sx,
			left_one_on_tz, right_one_on_tz,
			left_u_on_tz, right_u_on_tz,
			left_v_on_tz, right_v_on_tz,
			image_ptr, width_mask, height_mask, column_shift,
			light_level);
		left_sx += left_sx_slope;
		left_one_on_tz += left_one_on_tz_slope;
		left_u_on_tz += left_u_on_tz_slope;
		left_v_on_tz += left_v_on_tz_slope;
		right_sx += right_sx_slope;
		right_one_on_tz += right_one_on_tz_slope;
		right_u_on_tz += right_u_on_tz_slope;
		right_v_on_tz += right_v_on_tz_slope;
		sy += FIXED_ONE;
	} while (TRUE);
}

void
render_frame(void)
{
	int vertex_no;
	int polygon_no;

	/*
	 * Transform all vertices in the vertex list.
	 */
	for (vertex_no = 0; vertex_no < vertices; vertex_no++)
		transform_vertex(vertex_ptr_list[vertex_no]);

	/*
	 * Clear the frame buffer (eventually this should go away).
	 */
	MEMZERO(frame_buffer, window_width * window_height);

	/*
	 * Clear Z buffer and reset the z-factor once in a very great while
	 * (specifically, everytime the z-factor is about to wrap around to
	 * zero).
	 */
	if (z_factor == FIXED(MAX_FIXED)) {
		MEMZERO(z_buffer, window_width * window_height * sizeof(fixed));
		z_factor = 0;
	}

	/*
	 * Render each polygon in turn, unless it's trivially not visible
	 * (i.e. behind the viewing plane or facing away from us).
	 */
	for (polygon_no = 0; polygon_no < polygons; polygon_no++) {
		polygon *polygon_ptr;

		polygon_ptr = polygon_ptr_list[polygon_no];
		if (polygon_visible(polygon_ptr))
			render_polygon(polygon_ptr);
	}

	z_factor += FIXED_ONE;
}
