/* @(#)typedef.h 1.15   8/28/96 */

/******************************************************************************
* Threedom: a 3D polygon renderer                                             *
* (C) Copyright 1996 by Philip Stephens                                       *
* (C) Copyright 1996 by the IBM Corporation                                   *
* All Rights Reserved                                                         *
*                                                                             *
* Permission to use, copy, modify, and distribute this software and its       *
* documentation without fee for any non-commerical purpose is hereby granted, *
* provided that the above copyright notice appears on all copies and that     *
* both that copyright notice and this permission notice appear in all         *
* supporting documentation.                                                   *
*                                                                             *
* NO REPRESENTATIONS ARE MADE ABOUT THE SUITABILITY OF THIS SOFTWARE FOR ANY  *
* PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.       *
* NEITHER PHILIP STEPHENS OR IBM SHALL BE LIABLE FOR ANY DAMAGES SUFFERED BY  *
* THE USE OF THIS SOFTWARE.                                                   *
******************************************************************************/

/*
 * Type definitions for various standard integer types.
 */
typedef int boolean;
typedef unsigned char byte;
typedef unsigned short word;
#ifdef	NO_FIXED_MATH
typedef double fixed;
#else
typedef long long fixed;
#endif
typedef unsigned long time_ms;

/*
 * Definition of a pixel and frame buffer pixel.
 */
typedef word pixel;
typedef byte fbpixel;

/*
 * Pixel colour constants.
 */
#define PIXEL_COLOURS		4096

#define PIXEL_REDS		16
#define PIXEL_RED_MASK		0xf000
#define PIXEL_RED_SHIFT		4

#define PIXEL_GREENS		16
#define PIXEL_GREEN_MASK	0xf000
#define PIXEL_GREEN_SHIFT	8

#define PIXEL_BLUES		16
#define PIXEL_BLUE_MASK		0xf000
#define PIXEL_BLUE_SHIFT	12

/*
 * Frame buffer colour constants.
 */
#define FB_COLOURS		192
#define FB_MAPS			3
#define FB_COLOURS_PER_MAP	64
#define FB_MAP_OFFSET		32

#define FB_BLACK		FB_MAP_OFFSET
#define FB_WHITE		FB_MAP_OFFSET + 63

#ifndef WIN32
#define FB_REDS			4
#define FB_RED_SHIFT		14
#define FB_GREENS		4
#define FB_GREEN_SHIFT		14
#define FB_BLUES		4
#define FB_BLUE_SHIFT		14
#else
#define FB_REDS			4
#define FB_RED_SHIFT		6
#define FB_GREENS		4
#define FB_GREEN_SHIFT		6
#define FB_BLUES		4
#define FB_BLUE_SHIFT		6
#endif

/*
 * Lighting constants
 */
#define LIGHT_MAX	5

/*
 * Miscellaneous macros and constants.
 */
#define MYMIN(A,B) 	((A) < (B) ? (A) : (B))
#define MYMAX(A,B) 	((A) > (B) ? (A) : (B))

/*
 * OS-dependent macros.
 */
#ifdef BSD
#define MEMCOPY(ptr1, ptr2, size)	bcopy(ptr1, ptr2, size)
#define MEMZERO(ptr, size)		bzero(ptr, size)
#else
#define MEMCOPY(ptr1, ptr2, size)	memcpy(ptr2, ptr1, size)
#define MEMZERO(ptr, size)		memset(ptr, 0, size)
#endif

/*
 * Boolean type definition and macros.
 */
#ifndef TRUE
#define TRUE	1
#endif
#ifndef FALSE
#define FALSE	0
#endif

/*
 * Fixed-point constants.
 */
#define MIN_FIXED	-32768.0
#define MAX_FIXED	32767.0

/*
 * Miscellaneous macros.
 */
#define RAD(x)		((x) * M_PI / 180.0)
#define ABS(x)		((x) < 0 ? -(x) : (x))

/*
 * Flags for find_vertex.
 */
#define FIND_TOP	0
#define FIND_BOTTOM	1

/*
 * Motion defines.
 */
#define TURN_RATE	1

/*
 * Directory seperator.
 */
#ifndef WIN32
#define DIR_SEPARATOR_CHAR	'/'
#define DIR_SEPARATOR_STR	"/"
#else
#define DIR_SEPARATOR_CHAR	'\\'
#define DIR_SEPARATOR_STR	"\\"
#endif

#if defined(__GNUC__) && defined(ARCH_i86)

/*
 * Inline assembly functions to perform fixed-point math.
 */
static inline fixed fmul(fixed r1, fixed r2)
{
     fixed result;

     __asm__ ("imull %2\n\t"
	      "shrd  $16, %%edx, %%eax\n\t"
	      :"=a" (result):"a" (r1), "d" (r2):"eax", "edx");

     return result;
}

static inline fixed fmul2_30(fixed r1, fixed r2)
{
     fixed result;

     __asm__ ("imull %2\n\t"
	      "shrd  $30, %%edx, %%eax\n\t"
	      :"=a" (result):"a" (r1), "d" (r2):"eax", "edx");

     return result;
}


static inline fixed fdiv(fixed dividend, fixed divisor)
{
     fixed result;

     /* no checking for overflow is done yet */
     __asm__("movl %%edx, %%eax\n\t"
	     "sar  $16, %%edx\n\t"
	     "shl  $16, %%eax\n\t"
	     "idivl %%ecx, %%eax\n\t"
	     :"=a" (result):"d" (dividend), "c" (divisor):"eax", "ecx", "edx");

     return result;
}

#else

/*
 * Macros to perform fixed-point math.
 */
#ifdef	NO_FIXED_MATH

#define FIXED_ONE	1.0
#define MOVE_RATE	0.125

#define fmul(x,y)	(fixed)((double)(x) * (double)(y))
#define fdiv(x,y)	(fixed)((double)(x) / (double)(y))
#define FIXED(x)	(fixed)(x)
#define INT(x)		(int)(x)
#define INT_TO_FIXED(x)	(fixed)(x)
#define FLOAT(x)	(float)(x)
#define CEIL(x)		(fixed)((int)(x) + 1)

#else

#define FIXED_ONE	16777216
#define MOVE_RATE	2097152

#define FIXED(x)	(fixed)((double)(x) * 16777216.0)
#define FLOAT(x)	((double)(x) / 16777216.0)
#define fmul(x,y)	(fixed)((double)(x) * (double)(y) / 16777216.0)
#define fdiv(x,y)	(fixed)(((double)(x) * 16777216.0) / (double)(y))
#define INT(x)		(int)((x) >> 24)
#define INT_TO_FIXED(x)	((fixed)(x) << 24)
#define CEIL(x)		((((x) + FIXED_ONE) >> 24) << 24)

#endif

#endif	/* GNU_C && i86 */

/*
 * Movement and collision flags.
 */
#define NONE		0

#define FORWARD		1
#define BACKWARD	2
#define LEFT		3
#define RIGHT		4

#define TURNLEFT	1
#define TURNRIGHT	2
#define LOOKUP		3
#define LOOKDOWN	4

/*
 * A generic event record, used for the graphics API.
 */
typedef struct {
	boolean resize;		/* if TRUE, window_width and window_height */
	int window_width;	/* are valid. */
	int window_height;
	int move;	
	int turn;
	boolean fast;
	char *world_file_name;	/* valid if not NULL */
} event_record;

/*
 * Data type required for the GIF loader.
 */
typedef struct {
	byte red, green, blue;
} RGBcolor;

/*
 * Vertex structure.
 */
typedef struct {
	fixed x, y, z;			/* original vertex */
	fixed tx, ty, tz;		/* transformed vertex */
} vertex;

/*
 * Texture point structure.
 */
typedef struct {
	fixed u, v;			/* original texture point */
} tpoint;

/*
 * Screen point structure.
 */
typedef struct {
	fixed sx, sy;			/* projected screen point */
	fixed one_on_tz;		/* 1/tz of point */
	fixed u_on_tz, v_on_tz;		/* u/tz and v/tz for texture point */
} spoint;

/*
 * Texture map structure.
 */
typedef struct {
	pixel *image_ptr;
	fixed width, height;
	int width_mask, height_mask;
	int column_shift;
} texture;

/*
 * Polygon structure.
 */
typedef struct {
	int vertices;			/* Number of edges */
	vertex **vertex_ptr_list;	/* List of edges in clockwise order */
	tpoint **tpoint_ptr_list;	/* List of texture points */
	texture *texture_ptr;		/* Texture to render on polygon */
	vertex normal_vertex;		/* Vertex representing surface normal */
	int light_level;		/* Light level for this polygon */
} polygon;
