/* @(#)win95.c 1.1   8/27/96 */

/******************************************************************************
* Threedom: a 3D polygon renderer                                             *
* (C) Copyright 1996 by Philip Stephens                                       *
* (C) Copyright 1996 by the IBM Corporation                                   *
* All Rights Reserved                                                         *
*                                                                             *
* Permission to use, copy, modify, and distribute this software and its       *
* documentation without fee for any non-commerical purpose is hereby granted, *
* provided that the above copyright notice appears on all copies and that     *
* both that copyright notice and this permission notice appear in all         *
* supporting documentation.                                                   *
*                                                                             *
* NO REPRESENTATIONS ARE MADE ABOUT THE SUITABILITY OF THIS SOFTWARE FOR ANY  *
* PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.       *
* NEITHER PHILIP STEPHENS OR IBM SHALL BE LIABLE FOR ANY DAMAGES SUFFERED BY  *
* THE USE OF THIS SOFTWARE.                                                   *
******************************************************************************/

#define STRICT
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <windowsx.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include "..\winlib\winlib.h"
#include "typedef.h"
#include "win95.h"

/*
 * I don't care to hear about parameters or local variables that
 * aren't used in a function.
 */
#pragma warn -aus
#pragma warn -par
#pragma warn -sig

static HWND main_window;
static HWND bitmap_window;
static HWND status_window;
static fast_bitmap *bitmap_ptr = NULL;
static RGBQUAD palette[256];
static event_record event;
static void (*event_proc_ptr)(event_record *event);
static void (*work_proc_ptr)(void);
static void (*quit_proc_ptr)(void);

/****************************************
 * Function to destroy the main window. *
 ****************************************/

static void
destroy_main_window(HWND window_handle)
{
	PostQuitMessage(0);
	(*quit_proc_ptr)();
}

/**********************************
 * Function to handle key events. *
 **********************************/

static void
process_key_event(HWND window_handle, UINT keycode, BOOL keydown, int repeat,
		  UINT flags)
{
	BOOL send_event;

	send_event = FALSE;
	if (keydown)
		switch(keycode) {
		case VK_UP:
			event.move = FORWARD;
			send_event = TRUE;
			break;
		case VK_DOWN:
			event.move = BACKWARD;
			send_event = TRUE;
			break;
		case 'x':
		case 'X':
			event.move = LEFT;
			send_event = TRUE;
			break;
		case 'c':
		case 'C':
			event.move = RIGHT;
			send_event = TRUE;
			break;
		case VK_LEFT:
			event.turn = TURNLEFT;
			send_event = TRUE;
			break;
		case VK_RIGHT:
			event.turn = TURNRIGHT;
			send_event = TRUE;
			break;
		case 'a':
		case 'A':
			event.turn = LOOKUP;
			send_event = TRUE;
			break;
		case 'z':
		case 'Z':
			event.turn = LOOKDOWN;
			send_event = TRUE;
			break;
		case VK_SHIFT:
			event.fast = TRUE;
			break;
		}
	else
		switch(keycode) {
		case VK_UP:
		case VK_DOWN:
		case 'x':
		case 'X':
		case 'c':
		case 'C':
			event.move = NONE;
			send_event = TRUE;
			break;
		case VK_LEFT:
		case VK_RIGHT:
		case 'a':
		case 'A':
		case 'z':
		case 'Z':
			event.turn = NONE;
			send_event = TRUE;
			break;
		case VK_SHIFT:
			event.fast = FALSE;
			break;
		}

	if (send_event)
		(*event_proc_ptr)(&event);
}

/***********************************************
 * Function to handle gaining of window focus. *
 ***********************************************/

static BOOL
handle_window_focus(HWND window_handle)
{
	SetFocus(window_handle);
	if (bitmap_ptr)
		return(InstallFastBitmapPalette(bitmap_ptr));
	return(FALSE);
}

/*************************************************
 * Function to process a command from a control. *
 *************************************************/

void
process_command_event(HWND window_handle, int menu_id, HWND control_handle,
		      UINT code_notify)
{
	char *world_file_name;

	switch(menu_id) {
	case CM_OPEN:
		if ((world_file_name = OpenFileDialog("Open Threedom world",
				 "Threedom world (*.w3d)\0*.w3d\0")) != NULL) {
			event.world_file_name = world_file_name;
			(*event_proc_ptr)(&event);
			event.world_file_name = NULL;
		}
		break;
	case CM_EXIT:
		PostQuitMessage(0);
		(*quit_proc_ptr)();
		break;
	}
}

/******************************************
 * Function to start up graphics library. *
 ******************************************/

void
graphics_startup(char *program_name, void (*event_proc)(event_record *event),
		 void (*work_proc)(void), void (*quit_proc)(void),
		 HINSTANCE instance_handle)
{
	WinLibInit(instance_handle);
	event_proc_ptr = event_proc;
	work_proc_ptr = work_proc;
	quit_proc_ptr = quit_proc;
	event.move = NONE;
	event.turn = NONE;
	event.fast = FALSE;
	event.resize = FALSE;
	event.world_file_name = NULL;
}

/*******************************************
 * Function to shut down graphics library. *
 *******************************************/

void
graphics_shutdown(void)
{
	WinLibCleanUp();
}

/***********************************
 * Function to create main window. *
 ***********************************/

/*
 * List of event function handlers for main window.
 */
static event_callbacks callback_list = {
	NULL,
	destroy_main_window,
	NULL,
	NULL,
	process_key_event,
	NULL,
	NULL,
	handle_window_focus,
	process_command_event
};

/*
 * Callback list for status window.
 */
static event_callbacks status_callback_list = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

void
create_window(int window_width, int window_height, char *window_title,
	      int window_state)
{
	/*
	 * Create a main window that is big enough to hold the bitmap and status
	 * windows.
	 */
	main_window = CreateMainWindow(window_title, window_width + 30,
				       window_height + 170, window_state,
				       "menu", &callback_list);

	/*
	 * Create the bitmap window and status window.
	 */
	bitmap_window = CreateChildWindow(main_window, 10, 10, window_width,
					  window_height, WS_BORDER, NULL);
	status_window = CreateEditControl(main_window, 10, window_height + 20,
					  window_width, 100, WS_BORDER | ES_READONLY
					  | ES_WANTRETURN, &status_callback_list);
}

/**************************************
 * Function to create a frame buffer. *
 **************************************/

fbpixel *
create_frame_buffer(int window_width, int window_height)
{
	int index;
	int map;
	float factor;
	word red, green, blue;

	/*
	 * Create the logical palette.
	 */
	index = FB_MAP_OFFSET;
	for (map = 0, factor = 1.0; map < FB_MAPS; map++, factor *= 0.75)
		 for (red = 0; red < FB_REDS; red++)
			for (green = 0; green < FB_GREENS; green++)
				for (blue = 0; blue < FB_BLUES; blue++) {
					palette[index].rgbRed = (BYTE)((red <<
						FB_RED_SHIFT) * factor);
					palette[index].rgbGreen = (BYTE)((green <<
						FB_GREEN_SHIFT) * factor);
					palette[index].rgbBlue = (BYTE)((blue <<
						FB_BLUE_SHIFT) * factor);
					index++;
				}

	/*
	 * Create the fast bitmap.
	 */
	bitmap_ptr = CreateFastBitmap(bitmap_window, window_width, window_height,
				      (RGBQUAD *)&palette, FB_COLOURS, FB_MAP_OFFSET);
	(void)InstallFastBitmapPalette(bitmap_ptr);


	/*
	 * Return a pointer to the bitmap array.
	 */
	return((fbpixel *)bitmap_ptr->bits);
}

/***************************************
 * Function to display a frame buffer. *
 ***************************************/

void
display_frame_buffer(void)
{
	DisplayFastBitmap(bitmap_ptr);
}

/***************************************
 * Function to destroy a frame buffer. *
 ***************************************/

void
destroy_frame_buffer(void)
{
	DestroyFastBitmap(bitmap_ptr);
}

/*************************
 * Enter the event loop. *
 *************************/

void
enter_event_loop(void)
{
	EnterEventLoop(work_proc_ptr);
}

/**********************************************
 * Get the pixel index for a given RGB value. *
 **********************************************/

pixel
get_pixel(byte r, byte g, byte b)
{
	pixel index;
	word red = (word)r << 8;
	word green = (word)g << 8;
	word blue = (word)b << 8;

	index = ((red & PIXEL_RED_MASK) >> PIXEL_RED_SHIFT) +
		((green & PIXEL_GREEN_MASK) >> PIXEL_GREEN_SHIFT) +
		((blue & PIXEL_BLUE_MASK) >> PIXEL_BLUE_SHIFT);
	return(index);
}

/*******************************************
 * Display a message on the status screen. *
 *******************************************/

void
system_message(char *format, ...)
{
	va_list arg_ptr;
	char oldmessage[1024];
	char newmessage[2048];
	char *old_ptr, *new_ptr;

	va_start(arg_ptr, format);
	vsprintf(oldmessage, format, arg_ptr);
	va_end(arg_ptr);

	old_ptr = oldmessage;
	new_ptr = newmessage;
	while (*old_ptr) {
		if (*old_ptr == '\n')
			*new_ptr++ = '\r';
		*new_ptr++ = *old_ptr++;
	}
	*new_ptr = '\0';

	DisplayMessage(status_window, newmessage);
}
