//
// Written by: Robert C. Pendleton
// 
// Copyright 1993, 1994 by Robert C. Pendleton, all right reserved
//
// Non-commercial use by individuals is permitted.
//
//

//----------------------------------------------
//
// Inspired by code from VOGLE and "Graphic Gems"
//
//----------------------------------------------

#include <stdio.h>
#include <stdlib.h>

#include "fixed.h"
#include "ptypes.h"
#include "3d.h"


matrix4x3 *
identity(matrix4x3 *m)
{
    m->el[0][0] = fpOne;
    m->el[0][1] = fpZero;
    m->el[0][2] = fpZero;

    m->el[1][0] = fpZero;
    m->el[1][1] = fpOne;
    m->el[1][2] = fpZero;

    m->el[2][0] = fpZero;
    m->el[2][1] = fpZero;
    m->el[2][2] = fpOne;

    m->el[3][0] = fpZero;
    m->el[3][1] = fpZero;
    m->el[3][2] = fpZero;

    return m;
}

matrix4x3 *
rotatex(matrix4x3 *m, int32 angle)
{
    fp14 sinX, cosX;
    fp14 temp;
    int32 i;

    cosX = fpCos(angle);
    sinX = fpSin(angle);

    for (i = 0; i < 4; i++)
    {
        temp = m->el[i][1];
        m->el[i][1] = fpMul(temp, cosX) + fpMul(m->el[i][2], sinX);
        m->el[i][2] = fpMul(m->el[i][2], cosX) - fpMul(temp, sinX);
    }
    return m;
}

matrix4x3 *
rotatey(matrix4x3 *m, int32 angle)
{
    fp14 cosY, sinY;
    fp14 temp;
    int32 i;

    cosY = fpCos(angle);
    sinY = fpSin(angle);

    for (i = 0; i < 4; i++)
    {
        temp = m->el[i][0];
        m->el[i][0] = fpMul(temp, cosY) - fpMul(m->el[i][2], sinY);
        m->el[i][2] = fpMul(temp, sinY) + fpMul(m->el[i][2], cosY);
    }
    return m;
}

matrix4x3 *
rotatez(matrix4x3 *m, int32 angle)
{
    fp14 cosZ, sinZ;
    fp14 temp;
    int32 i;

    cosZ = fpCos(angle);
    sinZ = fpSin(angle);

    for (i = 0; i < 4; i++)
    {
        temp = m->el[i][0];
        m->el[i][0] = fpMul(temp, cosZ) + fpMul(m->el[i][1], sinZ);
        m->el[i][1] = fpMul(m->el[i][1], cosZ) - fpMul(temp, sinZ);
    }
    return m;
}

matrix4x3 *
scale(matrix4x3 *m, fp14 scaleX, fp14 scaleY, fp14 scaleZ)
{
    m->el[0][0] = fpMul(m->el[0][0], scaleX);
    m->el[0][1] = fpMul(m->el[0][1], scaleY);
    m->el[0][2] = fpMul(m->el[0][2], scaleZ);

    m->el[1][0] = fpMul(m->el[1][0], scaleX);
    m->el[1][1] = fpMul(m->el[1][1], scaleY);
    m->el[1][2] = fpMul(m->el[1][2], scaleZ);

    m->el[2][0] = fpMul(m->el[2][0], scaleX);
    m->el[2][1] = fpMul(m->el[2][1], scaleY);
    m->el[2][2] = fpMul(m->el[2][2], scaleZ);

    m->el[3][0] = fpMul(m->el[3][0], scaleX);
    m->el[3][1] = fpMul(m->el[3][1], scaleY);
    m->el[3][2] = fpMul(m->el[3][2], scaleZ);

    return m;
}


matrix4x3 *
translate(matrix4x3 *m, fp14 transX, fp14 transY, fp14 transZ)
{
    m->el[3][0] += transX;
    m->el[3][1] += transY;
    m->el[3][2] += transZ;

    return m;
}


//----------------------------------------------
/*
Map from world coordinates to screen coordinates. The screen area is
assumed to have coordinates in the range 0 .. width and 0 .. height.
Width and height are given in pixels.  The viewing window has the
specified left, right, top, and bottom limits. The window bounds are
given in world coordinates.  The eye is distance units from the
viewing window. 
*/

fp14 sXCenter = 0;
fp14 sYCenter = 0;
fp14 sXScale = 0;
fp14 sYScale = 0;

matrix4x3 *
view(matrix4x3 *m, 
     fp14 scrWidth, fp14 scrHeight, 
     fp14 wLeft, fp14 wRight, fp14 wTop, fp14 wBottom, 
     fp14 eyeDist)
{
    if(!((wLeft != wRight) && (wBottom != wTop) &&
         (scrWidth > fpZero) && (scrHeight > fpZero)))
    {
        printf("invalid view\n");
        exit(0);
    }

    sXCenter = scrWidth >> 1;
    sYCenter = scrHeight >> 1;

    sXScale = fpMul(sXCenter, fpDiv(eyeDist, (wRight - wLeft) >> 1));
    sYScale = fpMul(sYCenter, fpDiv(eyeDist, (wTop - wBottom) >> 1));

    scale(m, sXScale, sYScale, fpOne);

    translate(m, fpZero, fpZero, eyeDist);

    return m;
}

void
printMat4x3(char *label, matrix4x3 *m)
{
    int32 i;

    printf("\n%s\n", label);
    for (i = 0; i < 4; i++)
    {
        printf("[ %8.4f %8.4f %8.4f ]\n", 
               fp2float(m->el[i][0]),
               fp2float(m->el[i][1]),
               fp2float(m->el[i][2]));
    }
}

void
printPoint3(char *label, point3 *p)
{
    printf("%s (%8.4f %8.4f %8.4f)\n", 
           label,
           fp2float(p->x),
           fp2float(p->y),
           fp2float(p->z));
}

void
printPoint2(char *label, point2 *p)
{
    printf("%s (%8.4f %8.4f)\n", 
           label,
           fp2float(p->x),
           fp2float(p->y));
}

