#include "stdlib.h"
#include "stdio.h"
#include "sys\types.h"
#include "sys\stat.h"
#include "fcntl.h"

#include "ptypes.h"
#include "tg.h"
#include "pcx.h"

//---------------------------------------------------
//
// load and decode a 256 color 1 plane version 5 PCX
// format file. You pass in a file name and pointers
// to fill in with the header, the palette, and the
// image.
//

boolean
loadPcx(char *file, 
        int32 *imgWidth, 
        int32 *imgHeight, 
        color *palette, 
        uint8 **image)
{
    int32 pcx;
    tPCXHdr header;
    struct stat stats;

    int32 count;
    uint8 pal[3 * 256];
    int16 r, g, b;
    uint8 *next;

    int32 width;
    int32 paddedWidth;
    int32 height;

    int32 rawSize;
    int32 imgSize;

    uint8 *raw, *src;
    uint8 *img;

    int32 repeat;
    uint8 byte;

    int32 i, j, k;

    //
    // first open the file
    //
    if (-1 == (pcx = open(file, O_RDONLY | O_BINARY)))
    {
        printf("couldn't open file \"%s\"\n", file);
        return FALSE;
    }

    //
    // get the file stats
    //
    fstat(pcx, &stats);
    stat(file, &stats);

    //
    // read the header of a pcx file
    //
    if (sizeof(tPCXHdr) != (count = read(pcx, &header, sizeof(tPCXHdr))))
    {
        printf("error reading header of file \"%s\"\n", file);
        close(pcx);
        return FALSE;
    }

#ifdef COMMENT
    printf("flag           = %02x\n", header.zSoftFlag);
    printf("version        = %d\n", header.version);
    printf("encoding       = %d\n", header.encoding);
    printf("bits/pixel     = %d\n", header.bitsPerPixel);
    printf("xMin           = %d\n", header.xMin);
    printf("xMax           = %d\n", header.xMax);
    printf("yMin           = %d\n", header.yMin);
    printf("yMax           = %d\n", header.yMax);
    printf("# planes       = %d\n", header.planes);
    printf("bytes/line     = %d\n", header.bytesPerLine);
#endif

    //
    // make sure that it is the kind of
    // file that we want. We're being
    // pretty picky here. But, that makes
    // life simpler.
    //
    if (header.zSoftFlag != 0x0a ||
        header.version != 5 ||
        header.bitsPerPixel != 8 ||
        header.planes != 1)
    {
        printf("This program can only read version 5 pcx files\n");
        printf("with 1 plane and 8 bits/pixel\n");
        close(pcx);
        return FALSE;
    }

    //
    // Now go pick up the palette
    //
    // Could do this later, but what the heck
    //
    if (-1 == lseek(pcx, stats.st_size - sizeof(pal), SEEK_SET))
    {
        printf("error seeking to palette of file \"%s\"\n", file);
        close(pcx);
        return FALSE;
    }

    if (sizeof(pal) != (count = read(pcx, &pal, sizeof(pal))))
    {
        printf("error reading palette of file \"%s\" wanted %d got %d\n", 
                file, sizeof(pal), count);

        close(pcx);
        return FALSE;
    }

    //
    // convert the palette to the form I want
    //
    next = &pal[0];
    for (i = 0; i < 256; i++)
    {
        r = *next++;
        g = *next++;
        b = *next++;

        palette->red = r >> 2;
        palette->green = g >> 2;
        palette->blue = b >> 2;

        palette++;
    }

    //
    // compute the size of the raw image,
    // get space for it, and read it in
    //
    *imgWidth = width = header.xMax - header.xMin + 1;
    paddedWidth = width + (width & 1);

    *imgHeight = height = header.yMax - header.yMin + 1;

    rawSize = stats.st_size - sizeof(tPCXHdr) - sizeof(pal);
    raw = malloc(rawSize);
    if (raw == NULL)
    {
        printf("unable to allocate %d bytes\n", rawSize);
        close(pcx);
        return FALSE;
    }

    if (-1 == lseek(pcx, sizeof(tPCXHdr), SEEK_SET))
    {
        printf("error seeking to raw image if file \"%s\"\n", file);
        free(raw);
        close(pcx);
        return FALSE;
    }

    if (rawSize != (count = read(pcx, raw, rawSize)))
    {
        printf("error reading raw image from file \"%s\" wanted %d got %d\n", 
                file, rawSize, count);
        free(raw);
        close(pcx);
        return FALSE;
    }

    //
    // get space for the uncompressed image
    //
    imgSize = paddedWidth * height;
    *image = img = malloc(imgSize);

    if (img == NULL)
    {
        printf("unable to allocate %d bytes\n", imgSize);
        free(raw);
        close(pcx);
        return FALSE;
    }

    //
    // decompress the raw image
    //

    i = 0;
    k = 0;
    src = raw;
    while (i < rawSize && k < imgSize)
    {
        if ((*src & 0xc0) == 0xc0)
        {
            repeat = *src & 0x3f;
            src++;
            byte = *src++;
            i += 2;

            for (j = 0; j < repeat; j++)
            {
                *img++ = byte;
                k++;
            }
        }
        else
        {
            *img++ = *src++;
            i++;
            k++;
        }
    }

    free(raw);
    close(pcx);
    return TRUE;
}

