//
// Written by: Robert C. Pendleton
// 
// Copyright 1993 by Robert C. Pendleton, all right reserved
//
// Non-commercial use by individuals is permitted.
//
//

#include <stdlib.h>
#include <stdio.h>

#include <dos.h>
#include <conio.h>

#include "fixed.h"
#include "tg.h"
#include "3d.h"

#define vgapage ((uint8 *) 0xa0000)

#define MOR_ADDR  (0x3c2)           // misc. output register
#define MOR_READ  (0x3cc)           // misc. output register

#define IST1_ADDR (0x3da)           // input status #1 register

#define SEQ_ADDR  (0x3c4)           // base port of the Sequencer
#define SEQ_DATA  (0x3c5)           // data port of the Sequencer

#define CRTC_ADDR (0x3d4)           // base port of the CRT Controller (color)
#define CRTC_DATA (0x3d5)           // data port of the CRT Controller (color)

#define GCR_ADDR  (0x3ce)           // graphics controller address register
#define GCR_DATA  (0x3cf)           // graphics controller data register

#define ACR_ADDR  (0x3c0)           // attribute registers
#define ACR_DATA  (0x3c1)           // attribute registers

#define PAL_WRITE_ADDR (0x3c8)      // palette write address
#define PAL_READ_ADDR  (0x3c7)      // palette write address
#define PAL_DATA       (0x3c9)      // palette data register

#define exchange(type, a, b)    \
{               \
    type _t_;   \
    _t_ = a;    \
    a = b;      \
    b = _t_;    \
}

/*-------------------------------------------------------------------*/

typedef struct
{
    fp14 x, y, z;
} point;

typedef struct
{
    fp14 y;
} edgeRec;

int
adjust(fp14 val)
{
    if (fpFract(val) == fpHalf)
    {
        val++;
    }

    return fp2int(val + fpHalf);
}

static char texture[128][128];

void
makeTexture(int argc, char *argv[])
{
    int x, y;
    color palette[256];
    long width;
    long height;
    unsigned char *image;
    int red, green, blue;

    red = 1;
    setColor(red, 0xff, 0, 0);

    green = 2;
    setColor(green, 0, 0xff, 0);

    blue = 3;
    setColor(blue, 0, 0, 0xff);

    if (argc == 1)
    {
        for (y = 0; y < 128; y++)
        {
            for (x = 0; x < 128; x++)
            {
                texture[y][x] = 1;
                if ((x & 0x4) && (y & 0x4))
                {
                    texture[y][x] = 2;
                }
                else if (x & 0x4)
                {
                    texture[y][x] = 3;
                }
            }
        }
    }
    else
    {
        if (!loadPcx(argv[1], &width, &height, &palette, &image))
        {
            exit(0);
        }

        
        for (y = 0; y < 128; y++)
        {
            for (x = 0; x < 128; x++)
            {
                texture[y][x] = *(image + x);
            }
            image += width;
        }

        setPalette(0, 256, &palette);
    }
}

static float ox, oy, oz;
static float hx, hy, hz;
static float vx, vy, vz;

void
tspan(int ix, edgeRec *pt1, edgeRec *pt2)
{
    int idy, iy1, iy2;

    float x, y, z;
    int iu, iv;

    char *col, *end;

    if (pt2->y < pt1->y)
    {
        exchange(edgeRec *, pt1, pt2);
    }

    iy1 = adjust(pt1->y);
    iy2 = adjust(pt2->y);

    if ((iy1 < 0) && (iy2 < 0))
    {
        return;
    }

    if ((iy1 > height) && (iy2 > height))
    {
        return;
    }

    iy1 = max(0, iy1);
    iy2 = min(height, iy2);

    idy = iy2 - iy1;

    if (idy == 0)
    {
        return;
    }

    x = ox + 
        (hx * (ix - fp2float(sXCenter))) + 
        (vx * (iy1 - fp2float(sYCenter)));

    y = oy + 
        (hy * (ix - fp2float(sXCenter))) + 
        (vy * (iy1 - fp2float(sYCenter)));

    z = oz + 
        (hz * (ix - fp2float(sXCenter))) + 
        (vz * (iy1 - fp2float(sYCenter)));

    outp(SEQ_ADDR, 0x02);
    outp(SEQ_DATA, 1 << (ix & 0x3));

    col = (activeOffset + (80 * iy1) + (ix >> 2));
    end = (activeOffset + (80 * iy2) + (ix >> 2));

    do
    {
        iu = ((long)(x / z * 128)) & 0x7f;
        iv = ((long)(y / z * 128)) & 0x7f;

        *col = texture[iv][iu];

        x += vx;
        y += vy;
        z += vz;

        col += 80;

    } while (col < end);
}

#define step (15)

void
lspan(int ix, edgeRec *pt1, edgeRec *pt2)
{
    int idy, iy1, iy2;

    float x, y, z;
    float u1, u2, du;
    float v1, v2, dv;
    float u, v;
    int iu, iv;
    int len, count;

    char *col;

    if (pt2->y < pt1->y)
    {
        exchange(edgeRec *, pt1, pt2);
    }

    iy1 = adjust(pt1->y);
    iy2 = adjust(pt2->y);

    if ((iy1 < 0) && (iy2 < 0))
    {
        return;
    }

    if ((iy1 >= height) && (iy2 >= height))
    {
        return;
    }

    iy1 = max(0, iy1);
    iy2 = min(height, iy2);

    idy = iy2 - iy1;

    if (idy == 0)
    {
        return;
    }

    x = ox + 
        (hx * (ix - fp2float(sXCenter))) + 
        (vx * (iy1 - fp2float(sYCenter)));

    y = oy + 
        (hy * (ix - fp2float(sXCenter))) + 
        (vy * (iy1 - fp2float(sYCenter)));

    z = oz + 
        (hz * (ix - fp2float(sXCenter))) + 
        (vz * (iy1 - fp2float(sYCenter)));

    if (z == 0.0)
    {
        return;
    }

    u2 = (x / z);
    v2 = (y / z);

    outp(SEQ_ADDR, 0x02);
    outp(SEQ_DATA, 1 << (ix & 0x3));

    col = (activeOffset + (80 * iy1) + (ix >> 2));

    len = iy2 - iy1;

    while (len > 0)
    {
        count = min(step, len);
        len -= step;

        x += (vx * count);
        y += (vy * count);
        z += (vz * count);

        if (z == 0.0)
        {
            return;
        }

        u = u1 = u2;
        v = v1 = v2;

        u2 = x / z;
        v2 = y / z;

        du = (u2 - u1) / count;
        dv = (v2 - v1) / count;

        u *= 128;
        du *= 128;

        v *= 128;
        dv *= 128;

        do
        {
            iu = ((long)u) & 0x7f;
            iv = ((long)v) & 0x7f;

            *col = texture[iv][iu];

            u += du;
            v += dv;

            col += 80;

        } while (--count);
    }
}

void
qspan(int ix, edgeRec *pt1, edgeRec *pt2)
{
    float fdy, fy1, fy2;
    int idy, iy1, iy2;
    float first, middle, last;

    float x1, y1, z1;
    float x2, y2, z2;
    float x3, y3, z3;

    float u1, u2, u3;
    float v1, v2, v3;
    float k, s, t;

    float u, du, ddu;
    float v, dv, ddv;
    int iu, iv;

    char *col;

    if (pt2->y < pt1->y)
    {
        exchange(edgeRec *, pt1, pt2);
    }

    iy1 = adjust(pt1->y);
    iy2 = adjust(pt2->y);

    if ((iy1 < 0) && (iy2 < 0))
    {
        return;
    }

    if ((iy1 > height) && (iy2 > height))
    {
        return;
    }

    iy1 = max(0, iy1);
    iy2 = min(height, iy2);

    idy = (iy2 - 1) - iy1;

    if (idy <= 0)
    {
        return;
    }

    fy1 = max(0.0, fp2float(pt1->y));
    fy2 = min(height, fp2float(pt2->y));

    fdy = fy2 - fy1;

    first = fy1 - fp2float(sYCenter);
    
    x1 = ox + 
         (hx * (ix - fp2float(sXCenter))) + 
         (vx * first);

    y1 = oy + 
         (hy * (ix - fp2float(sXCenter))) + 
         (vy * first);

    z1 = oz + 
         (hz * (ix - fp2float(sXCenter))) + 
         (vz * first);

    if (z1 != 0.0)
    {
        u1 = (x1 / z1);
        v1 = (y1 / z1);
    }
    else
    {
        return;
    }

    middle = (((fy2 + fy1) / 2.0) - fp2float(sYCenter)) - first;

    x2 = x1 + (vx * middle);

    y2 = y1 + (vy * middle);

    z2 = z1 + (vz * middle);

    if (z2 != 0.0)
    {
        u2 = (x2 / z2);
        v2 = (y2 / z2);
    }
    else
    {
        return;
    }

    last = (fy2 - fp2float(sYCenter)) - first;

    x3 = x1 + (vx * last);

    y3 = y1 + (vy * last);

    z3 = z1 + (vz * last);

    if (z3 != 0.0)
    {
        u3 = (x3 / z3);
        v3 = (y3 / z3);
    }
    else
    {
        return;
    }

    k = u1 + u3 - (u2 * 2.0);
    s = ((u3 - u1 - (k * 2.0)) / fdy);
    t = ((k * 2.0) / fdy);
    t = (t / fdy);

    du = s + t;
    ddu = t + t;

    k = v1 + v3 - (v2 * 2.0);
    s = ((v3 - v1 - (k * 2.0)) / fdy);
    t = ((k * 2.0) / fdy);
    t = (t / fdy);

    dv = s + t;
    ddv = t + t;

    u = u1;
    v = v1;

    outp(SEQ_ADDR, 0x02);
    outp(SEQ_DATA, 1 << (ix & 0x3));

    col = (activeOffset + (80 * iy1) + (ix >> 2));

    u *= 128;
    du *= 128;
    ddu *= 128;

    v *= 128;
    dv *= 128;
    ddv *= 128;

    do
    {
        iu = ((long)u) & 0x7f;
        iv = ((long)v) & 0x7f;

        *col = texture[iv][iu];

        u += du;
        du += ddu;

        v += dv;
        dv += ddv;

        col += 80;

    } while (--idy);
}

void
tedge(edgeRec *table, point *pt1, point *pt2)
{
    fp14 y, dy;
    fp14 fdx;
    int idx, ix1, ix2;

    if (pt2->x < pt1->x)
    {
        exchange(point *, pt1, pt2)
    }

    ix1 = adjust(pt1->x);
    ix2 = adjust(pt2->x);
    idx = ix2 - ix1;

    if (idx == 0)
    {
        return;
    }

    if (ix1 < 0 && ix2 < 0)
    {
        return;
    }

    if (ix1 >= width && ix2 >= width)
    {
        return;
    }

    idx = max(2, idx - 1);
    fdx = int2fp(idx);

    y = pt1->y;
    dy = fpDiv((pt2->y - pt1->y), fdx); 

    do
    {
        if (ix1 >= 0 && ix1 < width)
        {
            table[ix1].y = y;
        }

        y += dy;
        ix1++;

    } while(ix1 < ix2);
}

edgeRec table1[320];
edgeRec table2[320];

void
tpoly(int type, int n, point *ppnts, point *tpnts)
{
    fp14 maxx, minx;
    int imaxx, iminx;
    int ix1, ix2;

    int pnt1, pnt2;
    int i;

    float px, py, pz;
    float mx, my, mz;
    float nx, ny, nz;

    if (n < 3)
    {
        return;
    }

    //
    // find the the basis vectors for the polygon.
    // use the first point, the second point, and 
    // the last point in the polygon definition
    // to determine the vectors.
    //

    px = fp2float(tpnts[0].x);
    py = fp2float(tpnts[0].y);
    pz = fp2float(tpnts[0].z);

    mx = fp2float(tpnts[1].x) - px;
    my = fp2float(tpnts[1].y) - py;
    mz = fp2float(tpnts[1].z) - pz;

    nx = px - fp2float(tpnts[n - 1].x);
    ny = py - fp2float(tpnts[n - 1].y);
    nz = pz - fp2float(tpnts[n - 1].z);

    hx = (py * mz) - (pz * my);
    vx = (pz * mx) - (px * mz);
    ox = (px * my) - (py * mx);

    hy = (py * nz) - (pz * ny);
    vy = (pz * nx) - (px * nz);
    oy = (px * ny) - (py * nx);

    hz = (ny * mz) - (nz * my);
    vz = (nz * mx) - (nx * mz);
    oz = (nx * my) - (ny * mx);

    //
    // find the point with the max y value and the
    // point with the min y value 
    //

    maxx = minx = ppnts[0].x;
    imaxx = iminx = 0;

    for (i = 1; i < n; i++)
    {
        if (ppnts[i].x > maxx)
        {
            maxx = ppnts[i].x;
            imaxx = i;
        }

        if (ppnts[i].x < minx)
        {
            minx = ppnts[i].x;
            iminx = i;
        }
    }

    //
    // get ready to walk the edge list
    // filling in the edge tables.
    //

    ix1 = adjust(minx);
    ix2 = adjust(maxx);

    if (ix1 == ix2)
    {
        return;
    }

    // 
    // walk from the min Y point to the max Y point
    //

    pnt1 = iminx;
    pnt2 = iminx + 1;
    if (pnt2 >= n)
    {
        pnt2 = 0;
    }

    do
    {
        tedge(table1, &ppnts[pnt1], &ppnts[pnt2]);

        pnt1 = pnt2;
        pnt2 = pnt2 + 1;
        if (pnt2 >= n)
        {
            pnt2 = 0;
        }
    } while (pnt1 != imaxx);

    //
    // walk from the max Y point to the min Y point
    //

    pnt1 = imaxx;
    pnt2 = imaxx + 1;
    if (pnt2 >= n)
    {
        pnt2 = 0;
    }

    do
    {
        tedge(table2, &ppnts[pnt1], &ppnts[pnt2]);

        pnt1 = pnt2;
        pnt2 = pnt2 + 1;
        if (pnt2 >= n)
        {
            pnt2 = 0;
        }
    } while (pnt1 != iminx);

    //
    // walk the edge table and fill in all the spans
    //

    ix1 = max(ix1, 0);
    ix2 = min(ix2, 320);

    switch (type)
    {
    case 0:
        do
        {
            tspan(ix1, &table1[ix1], &table2[ix1]);
            ix1++;
        } while (ix1 < ix2);
        break;

    case 1:
        do
        {
            lspan(ix1, &table1[ix1], &table2[ix1]);
            ix1++;
        } while (ix1 < ix2);
        break;

    case 2:
        do
        {
            qspan(ix1, &table1[ix1], &table2[ix1]);
            ix1++;
        } while (ix1 < ix2);
        break;

    }
}

char *label[3] = {
    "True Texture",
    "Linear Interp.",
    "Quadratic Interp."
};

void
quadTest()
{
    matrix4x3 mat;
    point quad[4], tquad[4], pquad[4];
    int i, j;
    int key = -1;
    int page;
    int xa = 0, xda = 0;
    int ya = 0, yda = 0;
    int za = 0, zda = 0;
    fp14 s = fpOne;

    int x, y, c;

    int type = 0;

    quad[0].x = int2fp(0);
    quad[0].y = int2fp(0);
    quad[0].z = int2fp(0);

    quad[1].x = int2fp(0);
    quad[1].y = int2fp(1);
    quad[1].z = int2fp(0);

    quad[2].x = int2fp(1);
    quad[2].y = int2fp(1);
    quad[2].z = int2fp(0);

    quad[3].x = int2fp(1);
    quad[3].y = int2fp(0);
    quad[3].z = int2fp(0);

    page = 1;
    setActivePage(page);
    setVisiblePage(page ^ 1);

    while (key != 27)
    {
        switch(key)
        {
        case '1':
            type = 0;
            break;

        case '2':
            type = 1;
            break;

        case '3':
            type = 2;
            break;

        case 'x':
            xda--;
            break;

        case 'X':
            xda++;
            break;

        case 'y':
            yda--;
            break;

        case 'Y':
            yda++;
            break;

        case 'z':
            zda--;
            break;

        case 'Z':
            zda++;
            break;

        case '+':
            s += float2fp(0.1);
            break;

        case '-':
            s -= float2fp(0.1);
            break;

        case 's':
            xda = yda = zda = 0;
            break;

        case 'S':
            xda = yda = zda = 0;
            xa = ya = za = 0;
            s = fpOne;
            break;
        }

        identity(&mat);
        translate(&mat, -fpHalf, -fpHalf, fpZero);

        xa = 0x3ff & (xa + xda);
        ya = 0x3ff & (ya + yda);
        za = 0x3ff & (za + zda);

        rotatex(&mat, xa);
        rotatey(&mat, ya);
        rotatez(&mat, za);

        scale(&mat, s, s, s);

        translate(&mat, fpZero, fpZero, fpZero);

        view(&mat, 
             int2fp(width), int2fp(height), 
             float2fp(2), float2fp(-2), 
             float2fp(1.5), float2fp(-1.5),
             int2fp(1));

        for (j = 0; j < 4; j++)
        {
            transform(mat, tquad[j], quad[j]);
            project(pquad[j], tquad[j]);
        }

        clear(0);
        drawText(0, 8, 255, label[type]);
        tpoly(type, 4, &pquad, &tquad);

        setActivePage(page);
        setVisiblePage(page ^ 1);
        page ^= 1;

        key = -1;
        if (kbhit())
        {
            key = getch();
        }
    }
}

void
main(int argc, char *argv[])
{
    tgInit();
    mode(t320x240);

    makeTexture(argc, argv);

    quadTest();

    tgFinit();
}
