//
// Written by: Robert C. Pendleton
// 
// This code is derived from code I wrote
// myself and from several public domain
// sources. 
//
// Placed in the public domain by the author.
//

#include <dos.h>
#include <conio.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include "tg.h"

#include "ptypes.h"

#define max(a,b) (((a) > (b)) ? (a) : (b))
#define min(a,b) (((a) < (b)) ? (a) : (b))
#define abs(a) (((a)<0) ? -(a) : (a))
#define sign(a) (((a)<0) ? -1 : (a)>0 ? 1 : 0)

#define vgapage ((uint8 *) 0xa0000)

#define MOR_ADDR  (0x3c2)           // misc. output register
#define MOR_READ  (0x3cc)           // misc. output register

#define IST1_ADDR (0x3da)           // input status #1 register

#define SEQ_ADDR  (0x3c4)           // base port of the Sequencer
#define SEQ_DATA  (0x3c5)           // data port of the Sequencer

#define CRTC_ADDR (0x3d4)           // base port of the CRT Controller (color)
#define CRTC_DATA (0x3d5)           // data port of the CRT Controller (color)

#define GCR_ADDR  (0x3ce)           // graphics controller address register
#define GCR_DATA  (0x3cf)           // graphics controller data register

#define ACR_ADDR  (0x3c0)           // attribute registers
#define ACR_DATA  (0x3c1)           // attribute registers

#define PAL_WRITE_ADDR (0x3c8)      // palette write address
#define PAL_READ_ADDR  (0x3c7)      // palette write address
#define PAL_DATA       (0x3c9)      // palette data register

//---------------------------------------------------
//
// tg global variables
//

static int32 origMode;             // the original video mode
static uint8 far *fontAddr;        // address of the current font

uint8 *activeOffset;        // address of the active page

int32 lineSize;             // offset to next scan line
int32 pageSize;             // offset to a page

static void mode296x220();
static void mode320x200();
static void mode320x240();

static void setTweakMode(int32 mode);
static boolean clipLine(int32 *x1, int32 *y1, int32 *x2, int32 *y2);

int32 error;                // the last error 

int32 maxx;                 // maximum x coord
int32 maxy;                 // maximum y coord

int32 width;                // the width in pixels
int32 height;               // the height in pixels

int32 pages;                // the number of graphic pages

char *modeName;             // printable name of the tweaked mode


//---------------------------------------------------
//
// low level long word fill routine
//

void
fillLong(void *addr, uint32 value, int32 count);
#pragma aux fillLong = \
    "cld" \
    "repe stosd" \
    parm [edi] [eax] [ecx];

//---------------------------------------------------
//
// Multiply two 32 bit ints and divide by a third
// making use of the 64 bit intermediate value
// to increase the range of number that can be used.
//

int32
mulDiv(int32 value, int32 mulBy, int32 divBy);
#pragma aux mulDiv = \
    "imul   ecx" \
    "idiv   ebx" \
    parm [eax] [ecx] [ebx]\
    modify [edx] \
    value [eax];

//---------------------------------------------------
//
// Use the bios to make sure we have a VGA display
//
// Not really sure this works, but I have two books
// that say it does... 
//

boolean
isVga()
{
    union REGS rg;

    rg.w.ax = 0x1a00;       // if this function is supported
                            // then we have a vga bios
    int386(0x10, &rg, &rg);

    return rg.h.al == 0x1a;
}

//---------------------------------------------------
//
// Use the bios to get the address of the 8x8 font
//
// You need a font if you are going to draw text.
//

uint8 far *
getFont()
{
    union REGPACK rg;
    uint32 seg;
    uint32 off;

    memset(&rg, 0, sizeof(rg));
    rg.w.ax = 0x1130;
    rg.h.bh = 0x03;
    intr(0x10, &rg);
    seg = rg.w.es;
    off = rg.w.bp;
    

    return (uint8 far *)MK_FP(seg, off);
}

//---------------------------------------------------
//
// Use the bios to get the current video mode
//

long
getMode()
{
    union REGS rg;

    rg.h.ah = 0x0f;
    int386(0x10, &rg, &rg);

    return rg.h.al;
}

//---------------------------------------------------
//
// Use the bios to set the video mode
//

void
setMode(long mode)
{
    union REGS rg;

    rg.h.ah = 0x00;
    rg.h.al = mode;
    int386(0x10, &rg, &rg);
}

//---------------------------------------------------
//
// Set a 296x220 unchained 256 color mode
//

void
mode296x220()
{
    int crtc11;

    outp(0x3d4, 0x11); // unlock crtc
    crtc11 = inp(0x3d5) & 0x7f;
    outp(0x3d4, 0x11);
    outp(0x3d5, crtc11);

    width   = 296;
    height  = 220;
    maxx    = 295;
    maxy    = 219;
    pages   = 4;
    lineSize = 74;
    pageSize = 65120;
    modeName = "296x220";

    outp(0x3c2, 0xe3);   // mor

    outp(0x3d4, 0x00); // crtc
    outp(0x3d5, 0x5f);

    outp(0x3d4, 0x01); // crtc
    outp(0x3d5, 0x49);

    outp(0x3d4, 0x02); // crtc
    outp(0x3d5, 0x50);

    outp(0x3d4, 0x03); // crtc
    outp(0x3d5, 0x82);

    outp(0x3d4, 0x04); // crtc
    outp(0x3d5, 0x53);

    outp(0x3d4, 0x05); // crtc
    outp(0x3d5, 0x80);

    outp(0x3d4, 0x06); // crtc
    outp(0x3d5, 0x0d);

    outp(0x3d4, 0x07); // crtc
    outp(0x3d5, 0x3e);

    outp(0x3d4, 0x08); // crtc
    outp(0x3d5, 0x00);

    outp(0x3d4, 0x09); // crtc
    outp(0x3d5, 0x41);

    outp(0x3d4, 0x10); // crtc
    outp(0x3d5, 0xd7);

    outp(0x3d4, 0x11); // crtc
    outp(0x3d5, 0xac);

    outp(0x3d4, 0x12); // crtc
    outp(0x3d5, 0xb7);

    outp(0x3d4, 0x13); // crtc
    outp(0x3d5, 0x25);

    outp(0x3d4, 0x14); // crtc
    outp(0x3d5, 0x00);

    outp(0x3d4, 0x15); // crtc
    outp(0x3d5, 0xe7);

    outp(0x3d4, 0x16); // crtc
    outp(0x3d5, 0x06);

    outp(0x3d4, 0x17); // crtc
    outp(0x3d5, 0xe3);

    outp(0x3c4, 0x01); // seq
    outp(0x3c5, 0x01);

    outp(0x3c4, 0x04); // seq
    outp(0x3c5, 0x06);

    outp(0x3ce, 0x05); // gcr
    outp(0x3cf, 0x40);

    outp(0x3ce, 0x06); // gcr
    outp(0x3cf, 0x05);

    inp(0x3da);          // acr
    outp(0x3c0, 0x10 | 0x20);
    outp(0x3c0, 0x41);

    inp(0x3da);          // acr
    outp(0x3c0, 0x13 | 0x20);
    outp(0x3c0, 0x00);

    outp(0x3d4, 0x11); // lock crtc
    crtc11 = inp(0x3d5) | 0x80;
    outp(0x3d4, 0x11);
    outp(0x3d5, crtc11);
}

//---------------------------------------------------
//
// Set an unchained version of mode 0x13
//

void
mode320x200()
{
    setMode(0x13);

    width   = 320;
    height  = 200;
    maxx    = 319;
    maxy    = 199;
    pages   = 4;
    lineSize = 80;
    pageSize = 64000;
    modeName = "320x200";

    /*

    Turn on the "mode X" effect by turning off chain 4,
    word, and double word mode. 

    */
    //
    // Turn off the Chain-4 bit in SEQ #4
    //
    outp(SEQ_ADDR, 0x04);
    outp(SEQ_DATA, 0x06);

    //
    // Turn off word mode, by setting the Mode Control register
    // of the CRTC #17
    //
    outp(CRTC_ADDR, 0x17);
    outp(CRTC_DATA, 0xE3);

    //
    // Turn off doubleword mode, by setting CRTC #14
    //
    outp(CRTC_ADDR, 0x14);
    outp(CRTC_DATA, 0x00);
}

//---------------------------------------------------
//
// This is the famous mode-x used in so many video
// games. It is a 320x240 unchained 256 color mode.
//

void
mode320x240()
{
    int crtc11;

    outp(0x3d4, 0x11); // unlock crtc
    crtc11 = inp(0x3d5) & 0x7f;
    outp(0x3d4, 0x11);
    outp(0x3d5, crtc11);

    width   = 320;
    height  = 240;
    maxx    = 319;
    maxy    = 239;
    pages   = 3;
    lineSize = 80;
    pageSize = 76800;
    modeName = "320x240";

    outp(0x3c2, 0xe3);   // mor

    outp(0x3d4, 0x00); // crtc
    outp(0x3d5, 0x5f);

    outp(0x3d4, 0x01); // crtc
    outp(0x3d5, 0x4f);

    outp(0x3d4, 0x02); // crtc
    outp(0x3d5, 0x50);

    outp(0x3d4, 0x03); // crtc
    outp(0x3d5, 0x82);

    outp(0x3d4, 0x04); // crtc
    outp(0x3d5, 0x54);

    outp(0x3d4, 0x05); // crtc
    outp(0x3d5, 0x80);

    outp(0x3d4, 0x06); // crtc
    outp(0x3d5, 0x0d);

    outp(0x3d4, 0x07); // crtc
    outp(0x3d5, 0x3e);

    outp(0x3d4, 0x08); // crtc
    outp(0x3d5, 0x00);

    outp(0x3d4, 0x09); // crtc
    outp(0x3d5, 0x41);

    outp(0x3d4, 0x10); // crtc
    outp(0x3d5, 0xea);

    outp(0x3d4, 0x11); // crtc
    outp(0x3d5, 0xac);

    outp(0x3d4, 0x12); // crtc
    outp(0x3d5, 0xdf);

    outp(0x3d4, 0x13); // crtc
    outp(0x3d5, 0x28);

    outp(0x3d4, 0x14); // crtc
    outp(0x3d5, 0x00);

    outp(0x3d4, 0x15); // crtc
    outp(0x3d5, 0xe7);

    outp(0x3d4, 0x16); // crtc
    outp(0x3d5, 0x06);

    outp(0x3d4, 0x17); // crtc
    outp(0x3d5, 0xe3);

    outp(0x3c4, 0x01); // seq
    outp(0x3c5, 0x01);

    outp(0x3c4, 0x04); // seq
    outp(0x3c5, 0x06);

    outp(0x3ce, 0x05); // gcr
    outp(0x3cf, 0x40);

    outp(0x3ce, 0x06); // gcr
    outp(0x3cf, 0x05);

    inp(0x3da);          // acr
    outp(0x3c0, 0x10 | 0x20);
    outp(0x3c0, 0x41);

    inp(0x3da);          // acr
    outp(0x3c0, 0x13 | 0x20);
    outp(0x3c0, 0x00);

    outp(0x3d4, 0x11); // lock crtc
    crtc11 = inp(0x3d5) | 0x80;
    outp(0x3d4, 0x11);
    outp(0x3d5, crtc11);
}

//---------------------------------------------------
//
// Set up one of the 7 tweaked modes.
//
// Start out by setting mode 0x13 to make sure that
// the palette and fonts are set up correctly for
// 256 color modes.
//

void
setTweakMode(int32 mode)
{
    setMode(0x13);

    switch (mode)
    {
    case t296x220:
        mode296x220();
        break;

    case t320x200:
        mode320x200();
        break;

    case t320x240:
        mode320x240();
        break;

    default:
        error = modeNotSupported;
    }
}

//---------------------------------------------------
//
// Remember the mode we were in before we started.
//

tgInit()
{
    error = none;

    origMode = getMode();                   // save the current mode
    if (!isVga())
    {
        error = notVgaDisplay;
        return;
    }
}

//---------------------------------------------------
//
// Put things back the way they were before we tried
// to set up a new video mode.
//

tgFinit()
{
    setMode(origMode);
}

//---------------------------------------------------
//
// Put the graphcs system into the requested mode and
// clear 256k of video memory to pixel value 0.
//

void
mode(int32 mode)
{
    modeName = NULL;
    fontAddr = getFont();
    setTweakMode(mode);                 // set the new mode
    if (error != none)
    {
        return;
    }
    outp(SEQ_ADDR, 0x02);               // enable all planes
    outp(SEQ_DATA, 0x0F);               // enable all planes
    fillLong(vgapage, 0, 16 * 1024);    // clear the whole page

    activeOffset = vgapage;
}

//---------------------------------------------------
//
//

void
resetMode()
{
    setMode(origMode);

    width = 0;
    height = 0;
    maxx = 0;
    maxy = 0;
    pages = 0;
    lineSize = 0;
    pageSize = 0;
    modeName = NULL;
}

//---------------------------------------------------
//
// Clear the current page to the given pixel value.
//

void
clear(uint8 color)
{
    int32 c = 0;

    c = (color & 0xff);
    c = (c << 8) | (color & 0xff);
    c = (c << 8) | (color & 0xff);
    c = (c << 8) | (color & 0xff);

    outp(SEQ_ADDR, 0x02);     // enable all planes
    outp(SEQ_DATA, 0x0F);     // enable all planes
    fillLong(activeOffset, c, pageSize / 16);
}

//---------------------------------------------------
//
// Make all drawing go the the selected page.
//

void
setActivePage(int32 page)
{
    if (page < 0 || page >= pages)
    {
        return;
    }
    activeOffset = vgapage + (page * pageSize / 4);
}

//---------------------------------------------------
//
// Make the selected page visible.
//

void
setVisiblePage(int32 page)
{
    int32 offset;

    if (page < 0 || page >= pages)
    {
        return;
    }

    offset = page * pageSize / 4;
    
    while(inp(IST1_ADDR) & 0x01);       // wait for display disable

    outp(CRTC_ADDR, 0x0c);
    outp(CRTC_DATA, (offset & 0xff00) >> 8);

    outp(CRTC_ADDR, 0x0d);
    outp(CRTC_DATA, offset &0x00ff);

    //
    // you can spend a lot of CPU time in this little loop
    //
    while(!(inp(IST1_ADDR) & 0x08));    // wait for vertical retrace
}

//---------------------------------------------------
//
// Draw a point in the current page with the given
// pixel value.
//

void
setPixel(int32 x, int32 y, uint8 color)
{
    //
    // first clip to the mode width and height
    //
    if (x < 0 || x > maxx || y < 0 || y > maxy)
    {
        return;
    }

    //
    // set the mask so that only one pixel gets written
    //
    outp(SEQ_ADDR, 0x02);
    outp(SEQ_DATA, 1 << (x & 0x3));

    *(activeOffset + (lineSize * y) + (x >> 2)) = color;
}

//---------------------------------------------------
//
// Read the value of a pixel.
//

int32
getPixel(int32 x, int32 y)
{
    //
    // first clip to the mode width and height
    //
    if (x < 0 || x > maxx || y < 0 || y > maxy)
    {
        return -1;
    }

    //
    // set the mask so that only one pixel gets read
    //
    outp(GCR_ADDR, 0x04);
    outp(GCR_DATA, x & 0x3);

    return *(activeOffset + (lineSize * y) + (x >> 2));
}

//---------------------------------------------------
//
// draw lines clipped to the screen.
//

//---------------------------------------------------
//
// return a bit mask that encodes which sides of the
// limits the point is on. If the coded value is zero
// the point is within the limits rectangle.
//

#define leftEdge    (0)
#define rightEdge   (maxx)
#define topEdge     (0)
#define bottomEdge  (maxy)

//---------------------------------------------------
//
// Do a simple line clip. Return true if there is a
// line left to draw and false if the line was clipped
// completely out of the picture.
//

boolean
clipLine(int32 *x1, int32 *y1, int32 *x2, int32 *y2)
{
    int32 dy, dx;

    if ((*x1 < leftEdge && *x2 < leftEdge) ||
        (*x1 > rightEdge && *x2 > rightEdge) ||
        (*y1 < topEdge && *y2 < topEdge) ||
        (*y1 > bottomEdge && *y2 > bottomEdge))
    {
        return FALSE;
    }

    dy = *y2 - *y1;
    dx = *x2 - *x1;

    if (*x1 < leftEdge)
    {
        *y1 = *y1 + mulDiv((leftEdge - *x1), dy, dx);
        *x1 = leftEdge;
    }
    else if (*x1 > rightEdge)
    {
        *y1 = *y1 + mulDiv((rightEdge - *x1), dy, dx);
        *x1 = rightEdge;
    }

    if (*y1 < topEdge)
    {
        *x1 = *x1 + mulDiv((topEdge - *y1), dx, dy);
        *y1 = topEdge;
    }
    else if (*y1 > bottomEdge)
    {
        *x1 = *x1 + mulDiv((bottomEdge - *y1), dx, dy);
        *y1 = bottomEdge;
    }

    if (*x1 < leftEdge ||
        *x1 > rightEdge)
    {
        return FALSE;
    }

    if (*x2 < leftEdge)
    {
        *y2 = *y2 + mulDiv((leftEdge - *x2), dy, dx);
        *x2 = leftEdge;
    }
    else if (*x2 > rightEdge)
    {
        *y2 = *y2 + mulDiv((rightEdge - *x2), dy, dx);
        *x2 = rightEdge;
    }

    if (*y2 < topEdge)
    {
        *x2 = *x2 + mulDiv((topEdge - *y2), dx, dy);
        *y2 = topEdge;
    }
    else if (*y2 > bottomEdge)
    {
        *x2 = *x2 + mulDiv((bottomEdge - *y2), dx, dy);
        *y2 = bottomEdge;
    }

    if (*x2 < leftEdge ||
        *x2 > rightEdge)
    {
        return FALSE;
    }

    return TRUE;
}

/*
 * Derived from:
 *
 * Digital Line Drawing
 * by Paul Heckbert
 * from "Graphics Gems", Academic Press, 1990
 */

void
line(int32 x1, int32 y1, int32 x2, int32 y2, uint8 color)
{
    int32 d;
    int32 x;
    int32 y;
    int32 ax;
    int32 ay;
    int32 sx;
    int32 sy;
    int32 dx;
    int32 dy;

    uint8 *lineAddr;
    int32 yOffset;

    if (!clipLine(&x1, &y1, &x2, &y2))
    {
        return;
    }

    dx = x2 - x1;  
    ax = abs(dx) << 1;  
    sx = sign(dx);

    dy = y2 - y1;  
    ay = abs(dy) << 1;  
    sy = sign(dy);
    yOffset = sy * lineSize;

    x = x1;
    y = y1;

    lineAddr = (activeOffset + (lineSize * y));
    if (ax>ay)
    {                       /* x dominant */
        d = ay - (ax >> 1);
        for (;;)
        {
            outp(SEQ_ADDR, 0x02);
            outp(SEQ_DATA, 1 << (x & 0x3));
            *((lineAddr) + (x >> 2)) = color;

            if (x == x2)
            {
                return;
            }
            if (d>=0)
            {
                y += sy;
                lineAddr += yOffset;
                d -= ax;
            }
            x += sx;
            d += ay;
        }
    }
    else
    {                       /* y dominant */
        d = ax - (ay >> 1);
        for (;;)
        {
            outp(SEQ_ADDR, 0x02);
            outp(SEQ_DATA, 1 << (x & 0x3));
            *((lineAddr) + (x >> 2)) = color;

            if (y == y2)
            {
                return;
            }
            if (d>=0) 
            {
                x += sx;
                d -= ay;
            }
            y += sy;
            lineAddr += yOffset;
            d += ax;
        }
    }
}

//---------------------------------------------------
//
// Draw a character in the active page using the
// 8x8 character font.
//

void
drawChar(int32 x, int32 y, uint8 color, char c)
{
    uint32 i, j;
    uint8 mask;
    uint8 far *font = fontAddr + (c * 8);

    for (i = 0; i < 8; i++)
    {
        mask = *font;
        for (j = 0; j < 8; j++)
        {
            if (mask & 0x80)
            {
                setPixel(x + j, y + i, color);
            }
            mask <<= 1;
        }
        font++;
    }
}

//---------------------------------------------------
//
// Draw a text string on the screen.
//

void
drawText(int32 x, int32 y, uint8 color, char *string)
{
    while (*string)
    {
        drawChar(x, y, color, *string);
        x += 8;
        string++;
    }
}

//---------------------------------------------------
//
// Fill a rectangle on the screen. 
//

int32 middleMask[4][4] = 
{
    {0x1, 0x3, 0x7, 0xf},
    {0x0, 0x2, 0x6, 0xe},
    {0x0, 0x0, 0x4, 0xc},
    {0x0, 0x0, 0x0, 0x8},
};

int32 leftMask[4] = {0xf, 0xe, 0xc, 0x8};

int32 rightMask[4] = {0x1, 0x3, 0x7, 0xf};

void
fillRect(int32 x, int32 y, int32 width, int32 height, uint8 color)
{
    int32 i;

    int32 x1;
    int32 x2;
    int32 y1;
    int32 y2;

    int32 leftBand;
    int32 rightBand;
    int32 leftBit;
    int32 rightBit;
    int32 mask;
    int32 bands;

    char *top;
    char *where;

    x1 = x;
    x2 = x + width - 1;
    y1 = y;
    y2 = y + height - 1;

    //
    // first off, clip to the screen.
    //
    if (x1 < 0)
    {
        x1 = 0;
    }

    if (x2 > maxx)
    {
        x2 = maxx;
    }

    if (y1 < 0)
    {
        y1 = 0;
    }

    if (y2 > maxy)
    {
        y2 = maxy;
    }

    //
    // is there a rectangle left?
    //
    if (y2 < y1 || x2 < x1)
    {
        return;
    }

    //
    // in mode X we want to paint from the top down
    // and make use of the 4 pixel wide vertical bands
    //

    leftBand = x1 >> 2; 
    rightBand = x2 >> 2;

    leftBit = x1 & 3; 
    rightBit = x2 & 3;

    if (leftBand == rightBand)  // the whole rectangle is in one band
    {
        mask = middleMask[leftBit][rightBit];
        outp(SEQ_ADDR, 0x02);
        outp(SEQ_DATA, mask);

        top = activeOffset + (lineSize * y1) + leftBand;
        for (i = y1; i <= y2; i++)
        {
            *top = color;
            top += lineSize;
        }
    }
    else                        // spans 2 or more bands
    {
        mask = leftMask[leftBit];
        outp(SEQ_ADDR, 0x02);
        outp(SEQ_DATA, mask);

        top = activeOffset + (lineSize * y1) + leftBand;
        where = top;
        for (i = y1; i <= y2; i++)          // fill the left edge
        {
            *where = color;
            where += lineSize;
        }
        top++;

        outp(SEQ_ADDR, 0x02);
        outp(SEQ_DATA, 0x0f);

        bands = rightBand - (leftBand + 1);
        if (bands > 0)
        {
            where = top;
            for (i = y1; i <= y2; i++)      // fill the middle
            {
                memset(where, color, bands);
                where += lineSize;
            }
            top += bands;
        }

        mask = rightMask[rightBit];

        outp(SEQ_ADDR, 0x02);
        outp(SEQ_DATA, mask);

        where = top;
        for (i = y1; i <= y2; i++)          // fill the right edge
        {
            *where = color;
            where += lineSize;
        }

    }
}

//---------------------------------------------------
//
// Set one entry in the color palatte.
//

void
setColor(int32 index, uint32 r, uint32 g, uint32 b)
{
    if (index < 0 || index > 255)
    {
        return;
    }

    while(!(inp(0x3da) & 0x01));    // wait for blanking

    outp(PAL_WRITE_ADDR, index);
    outp(PAL_DATA, r);
    outp(PAL_DATA, g);
    outp(PAL_DATA, b);
}

//---------------------------------------------------
//
// Set a range of entries in the color palette.
//

void
setPalette(int32 start, int32 count, color *p)
{
    int32 i;

    if (start < 0 || (start + count - 1) > 255)
    {
        return;
    }

    while(!(inp(0x3da) & 0x08));    // wait vertical retrace

    outp(PAL_WRITE_ADDR, start);
    for (i = 0; i < count; i++)
    {
        outp(PAL_DATA, p->red);
        outp(PAL_DATA, p->green);
        outp(PAL_DATA, p->blue);
        p++;
    }
}

//---------------------------------------------------
