//Direct3D Immediate Mode Example Shell built with MSVC++ v4.0

//see accompanying text for further explanation and build notes

																					   

#include "d3dimex.h"



LRESULT CALLBACK WndProc (HWND hwnd, UINT iMsg, WPARAM wParam, LPARAM lParam);



//WinMain is standard Windows creation code. Note that the PeekMessage

//form is used for the main event loop. When there are no messages pending,

//control drops into RenderScene 

int WINAPI WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance,

					PSTR szCmdLine, int iCmdShow)

{

	WNDCLASSEX wndclass;

	const char *pszClassName = "D3D Example";

	const char *pszWindowCaption = "D3D Background Example";

	MSG msg;

	HWND hwnd;

	RECT rc;

	DWORD dwStyle;



	memset (&wndclass, 0, sizeof(WNDCLASSEX));

	wndclass.cbSize = sizeof(WNDCLASSEX);

	wndclass.style = CS_HREDRAW | CS_VREDRAW;

	wndclass.lpfnWndProc = WndProc;

	wndclass.cbClsExtra = 0;

	wndclass.cbWndExtra = 0;

	wndclass.hInstance = hInstance;

	wndclass.hIcon = LoadIcon (NULL, IDI_APPLICATION);

	wndclass.hCursor = LoadCursor (NULL, IDC_ARROW);

   	wndclass.hbrBackground = (HBRUSH) GetStockObject (GRAY_BRUSH);

	wndclass.lpszMenuName = pszClassName;

	wndclass.lpszClassName = pszClassName;

	wndclass.hIconSm = LoadIcon (NULL, IDI_APPLICATION);



	RegisterClassEx (&wndclass);



	InitGlobals ();



	dwStyle = WS_OVERLAPPED | WS_CAPTION | WS_SYSMENU;



   	rc.top = rc.left = 0;

	rc.bottom = g_iHeight;

	rc.right = g_iWidth;



  	//AdjustWindowRect tells us what our adjusted window

	//coords should be so that our client area is g_iWidth 

	//by g_iHeight -- but it doesn't actually set up the window.

	//We must use the new rect coords in CreateWindowEx

	AdjustWindowRectEx (&rc, dwStyle, FALSE, WS_EX_APPWINDOW);



	hwnd = CreateWindowEx (WS_EX_APPWINDOW,

							pszClassName,

   							pszWindowCaption,

							dwStyle,

							50,

							50,

							rc.right - rc.left,

							rc.bottom - rc.top,

							NULL,

							NULL,

							hInstance,

							NULL);



	InitApp (hwnd);



	ShowWindow (hwnd, iCmdShow);

   	UpdateWindow (hwnd);



	while (TRUE)

	{

		if (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))

		{

			if (msg.message == WM_QUIT)

				break;



			TranslateMessage (&msg);

			DispatchMessage (&msg);

		}

		else

		{

		  RenderScene (hwnd);

		}

	}



	return msg.wParam;

}





void InitGlobals (void)

{

	g_iWidth = OUR_WIDTH;	 //client area of window

	g_iHeight = OUR_HEIGHT;	 //client height of window

	g_bIsSoftware = FALSE;	 //use HW or SW renderer?

	g_pDD = NULL;			 //DirectDraw object

	g_pD3D = NULL;			 //Direct3D object

	g_pD3DDevice = NULL;	 //Direct3D device

	g_pD3DViewport = NULL;	 //Direct3D vieport

	g_pMatBack = NULL;		 //background material

	g_pFrontBuffer = NULL;	 //primary buffer

	g_pBackBuffer = NULL;	 //back buffer

	g_pClip = NULL;		     //clipper object 

	g_pPal = NULL;	         //palette

	g_NumDrivers = 0;		 //number of enumerated drivers

}





//All the DirectDraw and Direct3D initialization is done here.

//Direct3D is an interface obtained from a DirectDraw object,

//so the first thing to do is create a DirectDraw object. Then 

//set up all our surfaces, create a Direct3D object, and set 

//up a device based on a device driver which will be chosen 

//from those installed (see SetUpDeviceDriver). Create and 

//configure a viewport, and set up a background material	

void InitApp (HWND hwnd)

{

	HRESULT rval;



	//Create a DirectDraw object, set our cooperative level, 

	//create all our surfaces, set up a clipper and, 

	//if necessary, a palette

	rval = SetModeAndSurfaces (hwnd);

	ASSERT (rval == TRUE);

 

	//Create Direct3D object on the DirectDraw object

	rval = g_pDD->QueryInterface (IID_IDirect3D, (LPVOID *)&g_pD3D);

	ASSERT (rval == DD_OK);

	ASSERT (g_pD3D);



	//Enumerate all installed device drivers and pick one

	//based on (in this case) color model

	rval = SetUpDeviceDriver ();

	ASSERT (rval == TRUE);



	//Now we need to create a viewport, associate it with a 

	//device, and set up the viewing area. The device size

	//is based on current mode and will be the size of the

	//screen; it can't be resized. However, the viewport 

	//settings can be changed, and more than one viewport 

	//can be set up for a single device. This example uses one.

	rval = g_pD3D->CreateViewport (&g_pD3DViewport, NULL);

	ASSERT (rval == D3D_OK);

	ASSERT (g_pD3DViewport);

	

	//Attach viewport to our device

	rval = g_pD3DDevice->AddViewport (g_pD3DViewport);

	ASSERT (rval == D3D_OK);



   	//Configure viewport and give it some default settings,

	D3DVIEWPORT vd;

	memset (&vd, 0, sizeof(D3DVIEWPORT));



	//Direct3D needs the size to validate the viewport desc

	vd.dwSize = sizeof(D3DVIEWPORT);	

	

	//Define viewport area on device; 

    vd.dwX = 0;	   

    vd.dwY = 0;    

    vd.dwWidth = g_iWidth;    

    vd.dwHeight = g_iHeight;  

	

	//Normally we would configure our viewport here, but this simple 

	//example doesn't need to. Code showing some reasonable values is

	//provided at the end of this listing



 	//Apply setings to viewport

	rval = g_pD3DViewport->SetViewport (&vd);

	ASSERT (rval == D3D_OK);



	//Create a red background material: Fill out a material 

	//description (D3DMATERIAL) and associate it with the 

	//material. Get a material handle for it from the device 

	//and then attach it to the viewport as a background

	D3DMATERIAL MatBack;

	D3DMATERIALHANDLE hMatBack;



	//Create the material interface

	rval = g_pD3D->CreateMaterial (&g_pMatBack, NULL);

	ASSERT (rval == D3D_OK);

	ASSERT (g_pMatBack);



	//Fill out a material description; we'll make it red

	memset (&MatBack, 0, sizeof(D3DMATERIAL));

	MatBack.dwSize = sizeof(D3DMATERIAL);

   	MatBack.diffuse.r = D3DVAL (1);

	MatBack.diffuse.g = D3DVAL (0);

	MatBack.diffuse.b = D3DVAL (0);

	//For the ramp color model, rampsize must be set

	//to one for a background material, otherwise 

	//the background will be black

	MatBack.dwRampSize = 1;

	

	//Set up the background material using the description

	rval = g_pMatBack->SetMaterial (&MatBack);

	ASSERT (rval == D3D_OK);



	//Get a material handle for it from the device

	rval = g_pMatBack->GetHandle (g_pD3DDevice, &hMatBack);	  

	ASSERT (rval == D3D_OK);

	ASSERT (hMatBack);



	//Associate this background material with our viewport

    g_pD3DViewport->SetBackground (hMatBack);

}	



//Window Procedure: since we're doing everything 

//in RenderScene, this just handles window close message

LRESULT CALLBACK WndProc (HWND hwnd, UINT iMsg, WPARAM wParam, LPARAM lParam)

{



	switch (iMsg)

	{

		

		case WM_DESTROY:

			ReleaseAll ();

			PostQuitMessage (0);

			return 0;

	}



	return DefWindowProc (hwnd, iMsg, wParam, lParam);

}



//Release all Direct3D and DirectDraw objects

void ReleaseAll (void)

{

	RELEASE (g_pMatBack);

	RELEASE (g_pD3DViewport);

	RELEASE (g_pD3DDevice);

	RELEASE (g_pD3D);

	RELEASE (g_pPal);

	RELEASE (g_pClip);

	RELEASE (g_pBackBuffer);

	RELEASE (g_pFrontBuffer);

	RELEASE (g_pDD);	 

}											 





//First set up a DirectDraw object. Then, let Windows know that 

//we will be sharing the GDI (via SetCooperativeLevel using the 

//DDSCL_NORMAL flag). Create the primary and back buffers. 

//Since this is a windowed mode we must create a clipper. 

//If the back buffer was created in system memory, we dont 

//want to set up a hardware device driver, so we will set a flag 

//indicating this. If we are in an 8-bit mode, we need to set a

//palette to each of our surfaces. Note that the primary

//surface will automatically be allocated to the size of the screen,

//so it is not necessary to set up the .dwWidth and .dwHeight fields

//in the surface description (DDSURFACEDESC). Also note that 

//surfaces will be allocated in video memory if possible,

//in system memory otherwise, so we dont need to explicitly set

//DDSCAPS_VIDEOMEMORY

BOOL SetModeAndSurfaces (HWND hWnd)

{ 

	HRESULT rval;



	//Create a DirectDraw object

	rval = DirectDrawCreate (NULL, &g_pDD, NULL);

	ASSERT (rval == DD_OK);

	ASSERT (g_pDD);



  	//Tell Windows we will be sharing video resources

	rval = g_pDD->SetCooperativeLevel (hWnd, DDSCL_NORMAL);

	ASSERT (rval == DD_OK);



	//Set up front buffer surface description

	DDSURFACEDESC sd;

	memset (&sd, 0, sizeof(DDSURFACEDESC));

	sd.dwSize = sizeof(DDSURFACEDESC);	//must be set for DirectDraw to validate description

	sd.dwFlags = DDSD_CAPS;

   	sd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;

	

	//Now create it using the above description

	rval = g_pDD->CreateSurface (&sd, &g_pFrontBuffer, NULL);

	ASSERT (rval == DD_OK);

	ASSERT (g_pFrontBuffer);



	//Create back buffer surface. We need DDSCAPS_3DDEVICE					   

	//to create a surface that gets attached to a device

	sd.dwFlags = DDSD_WIDTH | DDSD_HEIGHT | DDSD_CAPS;

	sd.dwWidth = g_iWidth;

	sd.dwHeight = g_iHeight;

	sd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_3DDEVICE;



	rval = g_pDD->CreateSurface (&sd, &g_pBackBuffer, NULL);

	ASSERT (rval == DD_OK);

	ASSERT (g_pBackBuffer);



#if 1

	//Comment this out and play with the viewport settings

	//in InitApp and in RenderScene to see some really werd 

	//window effects. Don't forget to re-enable it.

	rval = g_pDD->CreateClipper (0, &g_pClip, NULL);

	ASSERT (rval == DD_OK);

	ASSERT (g_pClip);



	//Associate the clipper to our window

	rval = g_pClip->SetHWnd (0, hWnd);

	ASSERT (rval == DD_OK);



	//Attach the clipper to front buffer

	rval = g_pFrontBuffer->SetClipper (g_pClip);

	ASSERT (rval == DD_OK);

#endif



	//If the back buffer is in system memory, make sure we render to 

	//it using a software renderer 

	g_pBackBuffer->GetSurfaceDesc (&sd);

	if (sd.ddsCaps.dwCaps & DDSCAPS_SYSTEMMEMORY)	{

		g_bIsSoftware = TRUE;

	}



	//If we're in an 8-bit mode, set a palettte for each surface

	if (sd.ddpfPixelFormat.dwRGBBitCount < 16)

	{

		int i;

		HDC hdc;

		PALETTEENTRY pe[256];



		//Set up palette

		//Save our current hardware palette

		hdc = GetDC (NULL);

		GetSystemPaletteEntries (hdc, 0, 256, pe);

		ReleaseDC (NULL, hdc);



		//Remember that, in a windowed mode, the first and last 10 palette 

		//entries should be left for system use, the middle 236 are ours

	   	for (i = 0; i < 10; i++) pe[i].peFlags = D3DPAL_READONLY;

	   	for (i = 10; i < 246; i++) pe[i].peFlags = PC_RESERVED;

	   	for (i = 246; i < 256; i++) pe[i].peFlags = D3DPAL_READONLY;

		 

		rval = g_pDD->CreatePalette (DDPCAPS_8BIT | DDPCAPS_INITIALIZE, pe, &g_pPal, NULL);

		ASSERT (rval == DD_OK);

		ASSERT (g_pPal);



		//Now attach the palette to each buffer

		rval = g_pFrontBuffer->SetPalette (g_pPal);

		ASSERT (rval == DD_OK);



		rval = g_pBackBuffer->SetPalette (g_pPal);

		ASSERT (rval == DD_OK);



	}

   	return TRUE;

}





//Choose a driver from the list generated by EnumDeviceDriverCallback 

//based on one or more criteria. Our criterion here is simple; 

//we just choose an RGB color model software driver, if it exists,

//otherwise we choose a ramp color model software driver. A slightly 

//more complex example is provided at the end of this listing

int PickDriver (void)

{

    int i;

	int driverNumber = -1;



	//Return the first RGB, SW driver, if any 

	for (i=0; i<g_NumDrivers; i++) 

	{

		if (!Driver[i].bIsHardware && (Driver[i].Desc.dcmColorModel & D3DCOLOR_RGB))

		{

			driverNumber=i;

			break;

		}

	}



	if (driverNumber != -1)

		return (driverNumber);

    

	//That didn't work, so try for the first MONO, SW driver                                               

	for (i=0; i<g_NumDrivers; i++)

	{

		if (!Driver[i].bIsHardware && (Driver[i].Desc.dcmColorModel & D3DCOLOR_MONO))

		{

			driverNumber=i;

			break;

		}

	}



	return (driverNumber);	//If this returns as -1, we are in trouble

}





//Device driver enumeration function -- when EnumDevices is called, it 

//fills in the device driver descriptions for lpHWDesc and lpSWDesc,

//and then uses EnumDeviceDriverCallback to list a driver and save some

//information about it. This procedure is repeated for all device 

//drivers installed on the system. Information for each driver is saved

//in a global array of driverInfo structures. The most important item

//is the GUID. The GUID will be used to get a device interface for the 

//driver. lpUserArg is not used here, it gets passed in as the last 

//argument in EnumDevices, and can be used to pass application specific

//data to the callback (can be used, for example, if you dont want to set

//Driver up as a global)  

static HRESULT

CALLBACK EnumDeviceDriverCallback (LPGUID lpGuid, LPSTR lpDeviceDescription,

                      LPSTR lpDeviceName, LPD3DDEVICEDESC lpHWDesc,

                      LPD3DDEVICEDESC lpHELDesc, LPVOID lpUserArg)

{

	//Save this driver's info; we'll figure out which one we 

	//want to use in PickDriver

    memcpy (&Driver[g_NumDrivers].Guid, lpGuid, sizeof(GUID));

    lstrcpy (Driver[g_NumDrivers].About, lpDeviceDescription);

    lstrcpy (Driver[g_NumDrivers].Name, lpDeviceName);

    

	//If the color model for a HW (HAL) driver is invalid or 0, then the 

	//driver must be a SW (HEL) driver, so use this as a test to see 

	//which type of driver we just saved

    if (lpHWDesc->dcmColorModel)

	{

        Driver[g_NumDrivers].bIsHardware = TRUE;

        memcpy (&Driver[g_NumDrivers].Desc, lpHWDesc, sizeof(D3DDEVICEDESC));

    }

	else

	{

        Driver[g_NumDrivers].bIsHardware = FALSE;

        memcpy (&Driver[g_NumDrivers].Desc, lpHELDesc, sizeof(D3DDEVICEDESC));

	}



    g_NumDrivers++;

    return (D3DENUMRET_OK);

}





//SetUpdeviceDriver enumerates all device drivers in the system, and then 

//uses PickDriver to choose one based on one or more criteria (PickDriver

//uses color model, but other choices could be based on texturing or 

//z-buffering capabilities, for example. Once a driver is decided upon by

//PickDriver, we can get a Direct3D Device interface for it via its 

//GUID (Globally Unique ID)

BOOL SetUpDeviceDriver (void)

{

	HRESULT rval;



  	rval = g_pD3D->EnumDevices (EnumDeviceDriverCallback, NULL);

	ASSERT (rval == D3D_OK);



	g_CurrentDriver = PickDriver ();

	ASSERT (g_CurrentDriver != -1);  //-1 means no driver matched our criteria



	//Note on z-buffering: here would be a good place to set up

	//a z-buffer surface if you want one. If your device driver

	//was set as software, set up a DirectDraw z-buffer surface

	//using the DDSCAPS_SYSTEMMEMORY flag. If the driver was set

	//as hardware then you need to call GetCaps here to get 

	//the z-buf bit depth. Then you can create a z-buffer with 

	//that bit depth in video memory using DDSCAPS_VIDEOMEMORY 

	//as one of the flags in your surface description



	//Get the Direct3D Device interface from the GUID associated 

	//with the chosen driver. If the interface is not supported 

	//by the device, then g_pD3DDevice will be set to NULL and 

	//rval will return as E_NOINTERFACE

	rval = g_pBackBuffer->QueryInterface (Driver[g_CurrentDriver].Guid,

														 (LPVOID *)&g_pD3DDevice);

	ASSERT (rval == D3D_OK);

	ASSERT (g_pD3DDevice);



	return TRUE;

}





//All RenderScene does is clear the viewport to the background material,

//and perform the blit. You might want to call GetSurfaceDesc 

//to verify the height and width of the back buffer (the dwWidth and dwHeight

//members will contain these values) before using them to clear the

//viewport. Remember that DirectDraw defines the width and height of the front

//(primary) buffer to be that of the screen

void RenderScene (HWND hwnd)

{

	HRESULT rval;

	RECT rcFront, rcBack;



   	//Clear viewport (to current background material)

	D3DRECT r;

	r.x1 = 0;

	r.y1 = 0; 

	r.x2 = g_iWidth; 

	r.y2 = g_iHeight;



	//If you have a z-buffer, you would also need the D3DCLEAR_ZBUFFER flag

	rval = g_pD3DViewport->Clear(1, &r, D3DCLEAR_TARGET);

	ASSERT (rval == D3D_OK);

		 

	//Blt copies the back buffer to the primary buffer.

	//Remember that the primary buffer is in screen coords and is the 

	//size of the screen. We want the coords of the rectangle on the 

	//primary buffer that we are blitting to, in screen coords.



	//Describe the dimensions of the back buffer that we will be blitting

	rcBack.left = rcBack.top = 0;

	rcBack.right = g_iWidth;

	rcBack.bottom = g_iHeight;



	//Get rectangle that describes front buffer

	GetClientRect (hwnd, &rcFront);



	POINT pt = {0,0};				//Get upper left point in client area

	ClientToScreen (hwnd, &pt);		//in screen coords

	//Now set up a rectangle that will describe where on the front buffer

	//we will be blitting to. The rectangle's upper left hand corner will be the coord

	//of the point we just got.

	OffsetRect (&rcFront, pt.x, pt.y); 	 



   	//Now we can blit the back buffer to the correct area on the front buffer.

	//If we were using a full screen mode we could use page-flipping. It is

	//rumoured that windowed page-flipping will be supported in the future.

	rval = g_pFrontBuffer->Blt (&rcFront, g_pBackBuffer, &rcBack, DDBLT_WAIT, NULL);

	ASSERT (rval == DD_OK);

}





//ASSERT: use it early, use it often -- only YOU can prevent forest fires!

//(Now ask me what I think of a compiler that doesn't provide its own Assert macro...)

#include <stdio.h>



static void _Assert (const char *file, int line, const char *msg)

{

	int result;	

	static char buf[1024];



	sprintf (buf, "Assertion Failed %s at %d:  %s", file, line, msg);



	result = MessageBox (NULL, buf, "Assertion Failure", MB_OKCANCEL | 

												MB_APPLMODAL | MB_ICONERROR);



	if (result == IDCANCEL)

		PostQuitMessage (0);

}



///////////////////////EXAMPLE CODE ENDS HERE/////////////////////////////////







//Here is a slightly more complicated version of PickDriver that will

//choose a HW, RGB driver, if possible.	You might want to add a check for

//the current mode's bit depth, and then make sure the driver you're about to

//choose can support it. Other capabilities that you could check here are

//z-buffering and texturing. 

#if 0

int PickDriver (void)

{

    int i;

	int driverNumber = -1;



	//If we cant render using HW, dont bother finding a HW driver

	if (g_bIsSoftware == FALSE)

	{

		//Return the first HW, RGB driver, if any 

		for (i=0; i<g_NumDrivers; i++) 

		{

			if (Driver[i].bIsHardware && (Driver[i].Desc.dcmColorModel & D3DCOLOR_RGB))

			{

				driverNumber=i;

				break;

			}

		}



		if (driverNumber != -1)

			return (driverNumber);

    

		//That didn't work, so try for the first HW, MONO driver                                               

		for (i=0; i<g_NumDrivers; i++)

		{

			if (Driver[i].bIsHardware && (Driver[i].Desc.dcmColorModel & D3DCOLOR_MONO))

			{

				driverNumber=i;

				break;

			}

		}



		if (driverNumber != -1)

			return (driverNumber);

	} 

	

	//If we're here, either we didnt find a HW driver, or hardware rendering 

	//is disabled (bIsSoftware is set), so try for a SW driver, we'll settle

	//for any color model...

	for (i=0; i<g_NumDrivers; i++)

	{

		if (!Driver[i].bIsHardware && (Driver[i].Desc.dcmColorModel & D3DCOLOR_RGB))

		{

			driverNumber = i;

			break;

		}

	}



	return (driverNumber);	//if this returns as -1, we are in trouble

}

#endif

 



//Obviously your viewport dimensions are going to depend heavily on the

//units in which you're working. A useful starting point might be:

#if 0

	//The scale settings will tell us how many units across

	//our viewport is -- this makes it 2 x 2

    vd.dvScaleX = D3DVAL (g_iWidth) / D3DVAL (2.0);

    vd.dvScaleY = D3DVAL (g_iHeight) / D3DVAL (2.0);

	

	//If we now set our maximum x and y to 1, we will have a 2 unit range

	//which ends at one, and therefore starts at -1 which puts the 

	//origin at the center of the viewport

	vd.dvMaxX = D3DVAL (1.0);

    vd.dvMaxY = D3DVAL (1.0);



	//We can set our near and far z values to be:

	vd.dvMinZ = D3DVAL (-10.0);

	vd.dvMaxZ = D3DVAL (200.0);

#endif

