#include <windows.h>
#include <ddraw.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include "dxproto.h"
#include "d3d.h"

DDMODE		ModeList[16];
WORD		NumModes;
D3DDEVICE	D3DDevice[16];
WORD		NumDevices;
TEXTUREDESC TextureDesc[16];
WORD		NumTextureDesc;

int InitDDInterface(HWND hWindow, DXDevice *Dev)
{
    DDSURFACEDESC       Description;
    DDSCAPS             Capabilites;
    HRESULT             Result;
	
	Dev->hWindow = hWindow;
	Dev->lpDD = NULL;
	
	DirectDrawEnumerate(DDDeviceCallback, &Dev->lpDD);
	if(!Dev->lpDD)
	{
		Result = DirectDrawCreate(NULL, &Dev->lpDD, NULL);
		if(Result != DD_OK)
		{
			MessageBox(GetFocus(), "Unable to load DirectDraw.",
					   "CRITICAL ERROR", MB_OK | MB_ICONEXCLAMATION);
			return FALSE;
		}	

        NumModes = 0;
	    Dev->lpDD->lpVtbl->EnumDisplayModes(Dev->lpDD,
	                                        0, NULL, 0,
	                                        DDModeCallback);
	}
	ChooseMode(Dev);

	Result = Dev->lpDD->lpVtbl->SetCooperativeLevel(Dev->lpDD,
                                            		hWindow,
		                                            DDSCL_EXCLUSIVE |
                                                    DDSCL_FULLSCREEN |
                                                    DDSCL_NOWINDOWCHANGES);
	if(Result != DD_OK)
	{
		MessageBox(GetFocus(), "Unable to set the Co-op level.",
				   "CRITICAL ERROR", MB_OK | MB_ICONEXCLAMATION);
        Dev->lpDD->lpVtbl->Release(Dev->lpDD);
		return FALSE;
	}	

	Result = Dev->lpDD->lpVtbl->SetDisplayMode(Dev->lpDD,
                                               Dev->Mode.Width,
                                               Dev->Mode.Height,
                                               Dev->Mode.BPP);

	if(Result != DD_OK)
	{
		MessageBox(GetFocus(), "Error setting Display Mode",
                   "CRITICAL ERROR", MB_OK | MB_ICONEXCLAMATION);
        Dev->lpDD->lpVtbl->Release(Dev->lpDD);
		return FALSE;
	}

    memset(&Description, 0, sizeof(DDSURFACEDESC));
    Description.dwSize = sizeof(DDSURFACEDESC);
    Description.dwFlags = DDSD_CAPS | DDSD_BACKBUFFERCOUNT;
    Description.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE |
                                 DDSCAPS_FLIP |
								 DDSCAPS_3DDEVICE |
                                 DDSCAPS_COMPLEX;
    Description.dwBackBufferCount = 1;
    Result = Dev->lpDD->lpVtbl->CreateSurface(Dev->lpDD,
                                              &Description,
                                              &Dev->lpFrontBuffer,
                                              NULL);
	if(Result != DD_OK)
	{
		MessageBox(GetFocus(), "Unable to create the Front Buffer.",
				   "CRITICAL ERROR", MB_OK | MB_ICONEXCLAMATION);
        Dev->lpDD->lpVtbl->Release(Dev->lpDD);
		return FALSE;
	}	

	Capabilites.dwCaps = DDSCAPS_BACKBUFFER;
    Result =
        Dev->lpFrontBuffer->lpVtbl->GetAttachedSurface(Dev->lpFrontBuffer,
													   &Capabilites,
                                                       &Dev->lpBackBuffer);
	if(Result != DD_OK)
	{
		MessageBox(GetFocus(), "Unable to create the Back Buffer.",
				   "CRITICAL ERROR", MB_OK | MB_ICONEXCLAMATION);
        Dev->lpFrontBuffer->lpVtbl->Release(Dev->lpFrontBuffer);
        Dev->lpFrontBuffer = NULL;
        Dev->lpDD->lpVtbl->Release(Dev->lpDD);
		return FALSE;
	}	

	return TRUE;
};

int EndDDInterface(DXDevice *Dev)
{
    if(Dev->lpDD != NULL)
    {
        if(Dev->lpFrontBuffer != NULL)
        {
            Dev->lpFrontBuffer->lpVtbl->Release(Dev->lpFrontBuffer);
            Dev->lpFrontBuffer = NULL;
        }
        Dev->lpDD->lpVtbl->Release(Dev->lpDD);
        Dev->lpDD = NULL;
    }
	return TRUE;
};

int InitD3DInterface(DXDevice *Dev)
{
    HRESULT         Result;
    D3DVIEWPORT     ViewData;
    DDSURFACEDESC   Desc;
    DWORD           zDepth;


    Result = Dev->lpDD->lpVtbl->QueryInterface(Dev->lpDD,
                                               &IID_IDirect3D,
                                               (LPVOID *)&Dev->lpD3D);
	if(Result != DD_OK)
	{
		MessageBox(GetFocus(), "Unable to create Direct3D interface.",
				   "CRITICAL ERROR", MB_OK | MB_ICONEXCLAMATION);
		return FALSE;
	}		

	NumDevices = 0;
	Result = Dev->lpD3D->lpVtbl->EnumDevices(Dev->lpD3D, D3DDeviceCallback,
											 NULL);

	if(Result != DD_OK)
	{
		MessageBox(GetFocus(), "Unable to enumerate Direct3D devices.",
				   "CRITICAL ERROR", MB_OK | MB_ICONEXCLAMATION);
		EndD3DInterface(Dev);
		return FALSE;
	}		
	ChooseDevice(Dev);

	memset(&TextureDesc, 0, sizeof(TextureDesc));
	Result = Dev->lpD3DDevice->lpVtbl->EnumTextureFormats(Dev->lpD3DDevice,
														  EnumTextures,
														  NULL);
	if(Result != DD_OK)
	{
		MessageBox(GetFocus(), "Unable to enumerate textures.",
				   "CRITICAL ERROR", MB_OK | MB_ICONEXCLAMATION);
	}
	ChooseTextureFormat(Dev);

    memset(&Desc, 0, sizeof(DDSURFACEDESC));
    Desc.dwSize = sizeof(DDSURFACEDESC);
    Desc.dwFlags = DDSD_WIDTH | DDSD_HEIGHT |
                   DDSD_ZBUFFERBITDEPTH | DDSD_CAPS;
    Desc.ddsCaps.dwCaps = DDSCAPS_ZBUFFER;
    Desc.dwWidth = Dev->Mode.Width;
    Desc.dwHeight = Dev->Mode.Height;

    zDepth = Dev->D3DDevice.Desc.dwDeviceZBufferBitDepth;
    if(zDepth & DDBD_32)
        Desc.dwZBufferBitDepth = 32;
    else if(zDepth & DDBD_24)
        Desc.dwZBufferBitDepth = 24;
    else if(zDepth & DDBD_16)
        Desc.dwZBufferBitDepth = 16;
    else if(zDepth & DDBD_8)
        Desc.dwZBufferBitDepth = 8;
    else
        Desc.dwZBufferBitDepth = 16;

    Result = Dev->lpDD->lpVtbl->CreateSurface(Dev->lpDD, &Desc,
                                              &Dev->lpZBuffer,
                                              NULL);
    if(Result != DD_OK)
    {
        MessageBox(GetFocus(), "Unable to create the z-buffer.",
				   "CRITICAL ERROR", MB_OK | MB_ICONEXCLAMATION);
    }
    Result = Dev->lpBackBuffer->lpVtbl->AddAttachedSurface(Dev->lpBackBuffer,
                                                           Dev->lpZBuffer);
    if(Result != DD_OK)
    {
        MessageBox(GetFocus(), "Unable to attach the z-buffer.",
				   "CRITICAL ERROR", MB_OK | MB_ICONEXCLAMATION);
    }

    Result = Dev->lpD3D->lpVtbl->CreateViewport(Dev->lpD3D,
                                                &Dev->lpD3DViewport, NULL);
    if(Result != D3D_OK)
    {
        MessageBox(GetFocus(), "Unable to create the viewport.",
				   "CRITICAL ERROR", MB_OK | MB_ICONEXCLAMATION);
		EndD3DInterface(Dev);
		return FALSE;
	}		
    Result = Dev->lpD3DDevice->lpVtbl->AddViewport(Dev->lpD3DDevice,
                                                   Dev->lpD3DViewport);
    if(Result != D3D_OK)
    {
        MessageBox(GetFocus(), "Unable to attach the viewport.",
				   "CRITICAL ERROR", MB_OK | MB_ICONEXCLAMATION);
		EndD3DInterface(Dev);
		return FALSE;
	}		

    memset(&ViewData, 0, sizeof(D3DVIEWPORT));
    ViewData.dwSize = sizeof(D3DVIEWPORT);
    ViewData.dwX = ViewData.dwY = 0;
    ViewData.dwWidth = Dev->Mode.Width;
    ViewData.dwHeight = Dev->Mode.Height;
    ViewData.dvScaleX = ViewData.dwWidth / (float)2.0;
    ViewData.dvScaleY = ViewData.dwHeight / (float)2.0;
    ViewData.dvMaxX = (float)D3DDivide(D3DVAL(ViewData.dwWidth),
                                       D3DVAL(2 * ViewData.dvScaleX));
    ViewData.dvMaxY = (float)D3DDivide(D3DVAL(ViewData.dwHeight),
                                       D3DVAL(2 * ViewData.dvScaleY));
    Result = Dev->lpD3DViewport->lpVtbl->SetViewport(Dev->lpD3DViewport,
                                                     &ViewData);
    if(Result != D3D_OK)
    {
        MessageBox(GetFocus(), "Unable to set the viewport data.",
				   "CRITICAL ERROR", MB_OK | MB_ICONEXCLAMATION);
		EndD3DInterface(Dev);
		return FALSE;
	}		
        
	return TRUE;
};

int EndD3DInterface(DXDevice *Dev)
{
	ReleaseAllTextures(Dev);

	if(Dev->lpZBuffer)
    {
        Dev->lpZBuffer->lpVtbl->Release(Dev->lpZBuffer);
        Dev->lpZBuffer = NULL;
    }

    if(Dev->lpD3DViewport)
    {
        Dev->lpD3DViewport->lpVtbl->Release(Dev->lpD3DViewport);
        Dev->lpD3DViewport = NULL;
    }

    if(Dev->lpD3DDevice)
	{
		Dev->lpD3DDevice->lpVtbl->Release(Dev->lpD3DDevice);
		Dev->lpD3DDevice = NULL;
	}

	if(Dev->lpD3D)
	{
        Dev->lpD3D->lpVtbl->Release(Dev->lpD3D);
		Dev->lpD3D = NULL;
	}

	return TRUE;
};

HRESULT CALLBACK
	DDModeCallback(LPDDSURFACEDESC lpSurfaceDesc, LPVOID lpContext)
{
	int Count;
	lpContext = lpContext;

	if(lpSurfaceDesc->dwWidth > 800)
		return DDENUMRET_OK;
	if(lpSurfaceDesc->dwHeight > 600)
		return DDENUMRET_OK;
	if(lpSurfaceDesc->ddpfPixelFormat.dwRGBBitCount < 16)
        return DDENUMRET_OK;

    ModeList[NumModes].Width = (WORD)lpSurfaceDesc->dwWidth;
    ModeList[NumModes].Height = (WORD)lpSurfaceDesc->dwHeight;
    ModeList[NumModes].BPP = (BYTE)lpSurfaceDesc->ddpfPixelFormat.dwRGBBitCount;
	ModeList[NumModes].RedShift = lpSurfaceDesc->ddpfPixelFormat.dwRBitMask;
	ModeList[NumModes].GreenShift = lpSurfaceDesc->ddpfPixelFormat.dwGBitMask;
	ModeList[NumModes].BlueShift = lpSurfaceDesc->ddpfPixelFormat.dwBBitMask;

	Count = 0;
	while(!(ModeList[NumModes].RedShift & 1))
		ModeList[NumModes].RedShift >>= 1;
	while(ModeList[NumModes].RedShift & 1)
	{
		ModeList[NumModes].RedShift >>= 1;
		Count++;
	}
	ModeList[NumModes].RedShift = Count;

	Count = 0;
	while(!(ModeList[NumModes].GreenShift & 1))
		ModeList[NumModes].GreenShift >>= 1;
	while(ModeList[NumModes].GreenShift & 1)
	{
		ModeList[NumModes].GreenShift >>= 1;
		Count++;
	}
	ModeList[NumModes].GreenShift = Count;

	Count = 0;
	while(!(ModeList[NumModes].BlueShift & 1))
		ModeList[NumModes].BlueShift >>= 1;
	while(ModeList[NumModes].BlueShift & 1)
	{
		ModeList[NumModes].BlueShift >>= 1;
		Count++;
	}
	ModeList[NumModes].BlueShift = Count;

	NumModes++;
    if(NumModes > 15)
        return DDENUMRET_CANCEL;
    return DDENUMRET_OK;
};

BOOL CALLBACK DDDeviceCallback(GUID FAR *lpGUID, LPSTR lpDriverDesc,
							   LPSTR lpDriverName, LPVOID lpContext)
{
	LPDIRECTDRAW lpDD = NULL;
	DDCAPS DCaps, HCaps;
	HRESULT Result;
	int i;
	
	lpDriverDesc = lpDriverDesc;
	lpDriverName = lpDriverName;
	
	if(!lpGUID)
		return DDENUMRET_OK;

	Result = DirectDrawCreate(lpGUID, &lpDD, NULL);
	if(Result != DD_OK)
	{
		lpDD = NULL;
		return DDENUMRET_OK;
	}

	memset(&DCaps, 0, sizeof(DDCAPS));
	memset(&HCaps, 0, sizeof(DDCAPS));
	DCaps.dwSize = sizeof(DDCAPS);
	HCaps.dwSize = sizeof(DDCAPS);

    Result = lpDD->lpVtbl->GetCaps(lpDD, &DCaps, &HCaps);
	if(Result != DD_OK)
	{
        lpDD->lpVtbl->Release(lpDD);
		lpDD = NULL;
		return DDENUMRET_OK;
	}

	if(DCaps.dwCaps & DDCAPS_3D)
	{
		NumModes = 0;
        lpDD->lpVtbl->EnumDisplayModes(lpDD, 0, NULL, 0, DDModeCallback);
		for(i = 0; i < NumModes; i++)
		{
			if((ModeList[i].Width == 640) &&
			   (ModeList[i].Height == 480) &&
			   (ModeList[i].BPP == 16))
			{	
				*(LPDIRECTDRAW *)lpContext = lpDD;
				return DDENUMRET_CANCEL;
			}
		}
	}

    lpDD->lpVtbl->Release(lpDD);
	lpDD = NULL;
	return DDENUMRET_OK;
};

HRESULT CALLBACK D3DDeviceCallback(LPGUID lpGuid, LPSTR lpDDesc,
								   LPSTR lpDName, LPD3DDEVICEDESC lpHWDesc,
								   LPD3DDEVICEDESC lpHELDesc, LPVOID lpVoid)
{
	memcpy(&D3DDevice[NumDevices].Guid, lpGuid, sizeof(GUID));
	if(lpHWDesc->dcmColorModel)
	{
		memcpy(&D3DDevice[NumDevices].Desc, lpHWDesc, sizeof(D3DDEVICEDESC));
		D3DDevice[NumDevices].bHardware = TRUE;
	}
	else
	{
		memcpy(&D3DDevice[NumDevices].Desc, lpHELDesc, sizeof(D3DDEVICEDESC));
		D3DDevice[NumDevices].bHardware = FALSE;
	}
		
	D3DDevice[NumDevices].bTextureMap =
		(D3DDevice[NumDevices].Desc.dpcTriCaps.dwTextureCaps &
		D3DPTEXTURECAPS_PERSPECTIVE) ? TRUE : FALSE;

	if(D3DDevice[NumDevices].Desc.dwDeviceZBufferBitDepth)
		D3DDevice[NumDevices].bZBuffer = TRUE;
	else	
		D3DDevice[NumDevices].bZBuffer = FALSE;
		
	NumDevices++;
	if(NumDevices > 15)
		return DDENUMRET_CANCEL;
	return DDENUMRET_OK;
};

HRESULT CALLBACK EnumTextures(LPDDSURFACEDESC lpDesc, LPVOID lpContext)
{
	DWORD R, G, B, A;
		
	if(!(lpDesc->ddpfPixelFormat.dwFlags & DDPF_RGB))
		return DDENUMRET_OK;

	memcpy(&TextureDesc[NumTextureDesc].Desc, lpDesc, sizeof(DDSURFACEDESC));
	TextureDesc[NumTextureDesc].BPP = lpDesc->ddpfPixelFormat.dwRGBBitCount;
	R = lpDesc->ddpfPixelFormat.dwRBitMask;
	G = lpDesc->ddpfPixelFormat.dwGBitMask;
	B = lpDesc->ddpfPixelFormat.dwBBitMask;
	A = lpDesc->ddpfPixelFormat.dwRGBAlphaBitMask;

	if(R)
	{
		TextureDesc[NumTextureDesc].RBPP = 0;
		while(!(R & 1))
			R >>= 1;
		while(R & 1)
		{
			R >>= 1;
			TextureDesc[NumTextureDesc].RBPP++;
		}
	}

	if(G)
	{
		TextureDesc[NumTextureDesc].GBPP = 0;
		while(!(G & 1))
			G >>= 1;
		while(G & 1)
		{
			G >>= 1;
			TextureDesc[NumTextureDesc].GBPP++;
		}
	}

	if(B)
	{
		TextureDesc[NumTextureDesc].BBPP = 0;
		while(!(B & 1))
			B >>= 1;
		while(B & 1)
		{
			B >>= 1;
			TextureDesc[NumTextureDesc].BBPP++;
		}
	}

	if(A)
	{
		TextureDesc[NumTextureDesc].ABPP = 0;
		while(!(A & 1))
			A >>= 1;
		while(A & 1)
		{
			A >>= 1;
			TextureDesc[NumTextureDesc].ABPP++;
		}
	}

	NumTextureDesc++;
	if(NumTextureDesc > 15)
		return DDENUMRET_CANCEL;
	return DDENUMRET_OK;
};

int ChooseMode(DXDevice *Dev)
{
	int i;
	BOOL bChosen = FALSE;
	
	for(i = 0; i < NumModes; i++)
	{
		if((ModeList[i].Width == 640) &&
		   (ModeList[i].Height == 480) &&
		   (ModeList[i].BPP == 16))
		{
            memcpy(&Dev->Mode, &ModeList[i], sizeof(Dev->Mode));
			bChosen = TRUE;
			break;
		}
	}

    if(!bChosen)
       memcpy(&Dev->Mode, &ModeList[0], sizeof(Dev->Mode));

	return TRUE;
};

int ChooseDevice(DXDevice *Dev)
{
	int i, m = -1;
	HRESULT Res;

	for(i = 0; i < NumDevices; i++)
	{
		if(D3DDevice[i].bHardware == TRUE)
		{
			if(D3DDevice[i].Desc.dcmColorModel & D3DCOLOR_RGB)
			{
				m = i;
				break;
			}
		}
	}

	if(m == -1)
	{
		for(i = 0; i < NumDevices; i++)
		{
			if(D3DDevice[i].Desc.dcmColorModel & D3DCOLOR_RGB)
			{
				m = i;
				break;
			}
		}
	}

    memcpy(&Dev->D3DDevice, &D3DDevice[m], sizeof(Dev->D3DDevice));
	Res =
		Dev->lpBackBuffer->lpVtbl->QueryInterface(Dev->lpBackBuffer,
  										  		  &D3DDevice[m].Guid,
											  	  (LPVOID *)&Dev->lpD3DDevice);
    if(Res != D3D_OK)
	{
		MessageBox(GetFocus(), "Unable to Create Direct3D Device",
				   "CRITICAL ERROR", MB_OK | MB_ICONEXCLAMATION);
		EndD3DInterface(Dev);
		return FALSE;
	}	
	return TRUE;
};

void ChooseTextureFormat(DXDevice *Dev)
{
	int i;
	int m;
	m = 10;	
	for(i = 0; i < NumTextureDesc; i++)
	{
		if(TextureDesc[i].BPP == 32 && TextureDesc[i].ABPP != 0 && m > 3)
		{
			memcpy(&Dev->TextureDesc, &TextureDesc[i], sizeof(TEXTUREDESC));
			m = 3;
		}

		if(TextureDesc[i].BPP == 16 && TextureDesc[i].ABPP != 0 && m > 2)
		{
			memcpy(&Dev->TextureDesc, &TextureDesc[i], sizeof(TEXTUREDESC));
			m = 2;
		}

		if(TextureDesc[i].BPP == 32 && m > 4)
		{
			memcpy(&Dev->TextureDesc, &TextureDesc[i], sizeof(TEXTUREDESC));
			m = 4;
		}

		if(TextureDesc[i].BPP == 16 && m > 1)
		{
			memcpy(&Dev->TextureDesc, &TextureDesc[i], sizeof(TEXTUREDESC));
			m = 1;
		}
	}
	return;
};
