#include <math.h>
#include <defines.h>
#include <world.h>
#include <string.h>
#include <types.h>
#include <math3D.h>
#include <mmsystem.h>
#include <update.h>
#include <malloc.h>

WORLD::WORLD(int Width, int Height)
{
	ScreenInfo.Width = Width;
	ScreenInfo.Height = Height;
	WinGDrv = new WIN_G(ScreenInfo.Width, ScreenInfo.Height);
	GetPaletteEntries(WinGDrv->GetPal(), 0, 256, (PALETTEENTRY far *)PalEntry);
	UpdateMatrix = new MATRIX();
//	OldTime = timeGetTime();
	zBuffer = GlobalAlloc(GPTR, sizeof(int) * (WinGDrv->GetWidth() * WinGDrv->GetHeight()));
};

WORLD::~WORLD(void)
{
	CloseWorld();
	delete WinGDrv;
	delete UpdateMatrix;
	GlobalFree(zBuffer);
};

void WORLD::CloseWorld(void)
{
	int i, j;
	if(WorldOpen)
	{
		for(i = 0; i < NumObjects; i++)
		{
			delete Object[i].Vertex;
			for(j = 0; j < Object[i].NumPolygons; j++)
				delete Object[i].Polygon[j].Vertex;
			delete Object[i].Polygon;
		}
		for(i = 0; i < Number_Textures; i++)
			WinGDrv->FreeBitmap(DibTexture[i]);
		delete DibTexture;
		delete Object;
		delete Camera;
		delete Light;
		delete PolygonList.Polygon;
	}
	WorldOpen = FALSE;
};

void WORLD::DrawWorld(void)
{
	int i;

	if(PolyMap == MP_ZBUFFER)
	{
		long *c;

		c = (long *)zBuffer;
		for(i = 0; i < (WinGDrv->GetWidth() * WinGDrv->GetHeight() >> 2); i++)
		{
			*(long *)c++ = 0x7FFF; *(long *)c++ = 0x7FFF;
			*(long *)c++ = 0x7FFF; *(long *)c++ = 0x7FFF;
		}
	}

	HBRUSH Brush, OBrush;
	CurrentTime = timeGetTime();
	DeltaTime = CurrentTime - OldTime;
	OldTime = CurrentTime;
	Brush = CreateSolidBrush(BackColor);
	OBrush = SelectObject(WinGDrv->WinGDC, Brush);
	PatBlt(WinGDrv->WinGDC, 0, 0, WinGDrv->GetWidth(), WinGDrv->GetHeight(), PATCOPY);
	SelectObject(WinGDrv->WinGDC, OBrush);
	DeleteObject(Brush);
	UpdateObjects();
	AlignWithCamera();
	MakeList();
	if(PolyMap == MP_ZSORT)
		SortList();
	DrawList();
};

void WORLD::MakeList()
{
	register int i;
	register int j;

	PolygonList.NumPolygons = 0;

	for(i = 0; i < NumObjects; i++)
	{
		if(!FrontOfCamera(&Object[i]))
			continue;
		for(j = 0; j < Object[i].NumPolygons; j++)
		{
			Object[i].Polygon[j].Distance = FindPolygonDistance(&(Object[i].Polygon[j]));
			if( (Object[i].Polygon[j].Distance <= Camera[CurrentCamera].FarClipPlane) ||
				(Object[i].Polygon[j].Special == 1) )
			{
				if(!BackfaceCull(&(Object[i].Polygon[j])))
				{
					zClipPolygon(&Object[i].Polygon[j], &PolygonList.Polygon[PolygonList.NumPolygons]);
					if(PolygonList.Polygon[PolygonList.NumPolygons].NumVertices > 1)
						PolygonList.NumPolygons++;
				}
			}
		}
	}
};

void WORLD::SortList()
{
	QSort(PolygonList.Polygon, 0, PolygonList.NumPolygons - 1);
};

void WORLD::BubbleSort()
{
	int i;
	CLIPPOLY Temp;
	int sort = 1;

	if(PolygonList.NumPolygons <= 1)
		return;

	while(sort)
	{
		sort = 0;
		for(i = 0; i < PolygonList.NumPolygons - 1; i++)
		{
			if(PolygonList.Polygon[i].Distance < PolygonList.Polygon[i + 1].Distance)
			{
				Temp = PolygonList.Polygon[i];
				PolygonList.Polygon[i] = PolygonList.Polygon[i + 1];
				PolygonList.Polygon[i + 1] = Temp;
				sort = 1;
			}
		}
	}
};

void WORLD::QSort(CLIPPOLY *item, int left, int right)
{
	int i, j;
	float x;
	CLIPPOLY y;

	i = left; j = right;
	x = item[(left + right) / 2].Distance;

	do {
		while(item[i].Distance > x && i < right) i++;
		while(x > item[j].Distance && j > left) j--;

		if(i <= j) {
			y = item[i];
			item[i] = item[j];
			item[j] = y;
			i++; j--;
		}
	} while(i <= j);

	if(left < j) QSort(item, left, j);
	if(i < right) QSort(item, i, right);
};


void WORLD::DrawList()
{
	register int i;
	register int s;
	CLIPPOLY Temp;

	for(i = 0; i < PolygonList.NumPolygons; i++)
	{
		if(PolygonList.Polygon[i].Special == 1)
		{
			Temp = PolygonList.Polygon[0];
			PolygonList.Polygon[0] = PolygonList.Polygon[i];
			PolygonList.Polygon[i] = Temp;
		}
	}
	for(i = 0; i < PolygonList.NumPolygons; i++)
	{
		if(PolygonList.Polygon[i].NumVertices > 1)
		{
			Project(&PolygonList.Polygon[i]);
			if((PolygonList.Polygon[i].Special == 1))
				xyClipPolygon(&PolygonList.Polygon[i]);
			if(PolygonList.Polygon[i].NumVertices > 1)
				DrawPolygon(&(PolygonList.Polygon[i]));
		}
	}
};

BOOL WORLD::BackfaceCull(POLYGON *p)
{
	if(p->NumVertices < 3)
		return 0;

	if(p->BackfaceType == 0)
		return 0;

	float X1, Y1, Z1;
	float X2, Y2, Z2;
	float X3, Y3, Z3;
	float D;

	X1 = p->Vertex[0]->xCamera; Y1 = p->Vertex[0]->yCamera; Z1 = p->Vertex[0]->zCamera;
	X2 = p->Vertex[1]->xCamera; Y2 = p->Vertex[1]->yCamera; Z2 = p->Vertex[1]->zCamera;
	X3 = p->Vertex[2]->xCamera; Y3 = p->Vertex[2]->yCamera; Z3 = p->Vertex[2]->zCamera;

	D = (X3 * ((Z1 * Y2) - (Y1 * Z2)))
	  + (Y3 * ((X1 * Z2) - (Z1 * X2)))
	  + (Z3 * ((Y1 * X2) - (X1 * Y2)));

	return(D * p->BackfaceType) < 0;
};

BOOL WORLD::FrontOfCamera(OBJECT *o)
{
	int k;

	for(k = 0; k < o->NumVertices; k++)
	{
		if(o->Vertex[k].zCamera >= Camera[CurrentCamera].NearClipPlane)
			return TRUE;
	}
	return FALSE;
};

int WORLD::OnRaster(CLIPPOLY *p)
{
	int i, onscreen = 0;

	for(i = 0; i < p->NumVertices; i++)
	{
		if(p->Vertex[i].xRaster >= 0 && p->Vertex[i].xRaster < ScreenInfo.Width
		&& p->Vertex[i].yRaster >= 0 && p->Vertex[i].yRaster < ScreenInfo.Height)
			onscreen++;
	}

	if(onscreen == 0)
		return NONE_ONSCREEN;
	else if(onscreen == p->NumVertices)
		return ALL_ONSCREEN;
	else
		return PART_ONSCREEN;
};

long WORLD::FindPolygonDistance(POLYGON *p)
{
	float xmin, xmax;
	float ymin, ymax;
	float zmin, zmax;
	float zmed;
	int i;
	float total = 0;

	if(p->Special == 1)
		return(2000000);

	xmin = p->Vertex[0]->xCamera; xmax = p->Vertex[1]->xCamera;
	ymin = p->Vertex[0]->yCamera; ymax = p->Vertex[1]->yCamera;
	zmin = p->Vertex[0]->zCamera; zmax = p->Vertex[1]->zCamera;

	for(i = 0; i < p->NumVertices; i++)
	{
		total += p->Vertex[i]->zCamera;
		if(p->Vertex[i]->xCamera < xmin)
			xmin = p->Vertex[i]->xCamera;
		if(p->Vertex[i]->yCamera < ymin)
			ymin = p->Vertex[i]->yCamera;
		if(p->Vertex[i]->zCamera < zmin)
			zmin = p->Vertex[i]->zCamera;

		if(p->Vertex[i]->xCamera > xmax)
			xmax = p->Vertex[i]->xCamera;
		if(p->Vertex[i]->yCamera > ymax)
			ymax = p->Vertex[i]->yCamera;
		if(p->Vertex[i]->zCamera > zmax)
			zmax = p->Vertex[i]->zCamera;
	}
	total /= p->NumVertices;
	p->xMin = xmin; p->yMin = ymin; p->zMin = zmin;
	p->xMax = xmax; p->yMax = ymax; p->zMax = zmax;
	zmed = (zmax + zmin) / 2;

	return(fabs(total)); // Seems to work better than zmed
};

void WORLD::UpdateObjects()
{
	register int     Counter1;
	register OBJECT  *o;

	for(Counter1 = 0; Counter1 < NumObjects; Counter1++)
	{
		o = &Object[Counter1];
		o->Old_xTranslate = o->xTranslate;
		o->Old_yTranslate = o->yTranslate;
		o->Old_zTranslate = o->zTranslate;
	}

	for(Counter1 = 0; Counter1 < NumObjects; Counter1++)
	{
		o = &Object[Counter1];
		o->ObjFunction(o);
	}
};

void WORLD::AlignWithCamera(void)
{
	register int Counter1;

//	Camera[CurrentCamera].Pitch &= (NumDegrees - 1);
//	Camera[CurrentCamera].Yaw &= (NumDegrees - 1);
//	Camera[CurrentCamera].Roll &= (NumDegrees - 1);

	UpdateMatrix->Identity();
	UpdateMatrix->Translate(-Camera[CurrentCamera].xTranslate, -Camera[CurrentCamera].yTranslate, -Camera[CurrentCamera].zTranslate);
	UpdateMatrix->Rotate(-Camera[CurrentCamera].Pitch, -Camera[CurrentCamera].Yaw, -Camera[CurrentCamera].Roll);

	for(Counter1 = 0; Counter1 < NumObjects; Counter1++)
		UpdateMatrix->TransformWithCamera(&Object[Counter1]);
};

void WORLD::Project(CLIPPOLY *poly)
{
	UINT    v;
	float    zVal;
	long    distance;

	distance = (long)Camera[CurrentCamera].ViewPlane;

	for(v = 0; v < poly->NumVertices; v++)
	{
		zVal = poly->Vertex[v].zCamera;

		if(zVal == 0.0)
		{
			poly->Vertex[v].xRaster = (long)(poly->Vertex[v].xCamera + (ScreenInfo.Width / 2));
			poly->Vertex[v].yRaster = (long)(poly->Vertex[v].yCamera + (ScreenInfo.Height / 2));
		}

		else
		{
			poly->Vertex[v].xRaster = (long)((distance * poly->Vertex[v].xCamera) / zVal + (ScreenInfo.Width / 2));
			poly->Vertex[v].yRaster = (long)((distance * poly->Vertex[v].yCamera) / zVal + (ScreenInfo.Height / 2));
		}
	}
};

void WORLD::CalcPolyNormal(CLIPPOLY *p)
{
	VECTOR Vect1, Vect2;

	Vect1.xComp = (p->Vertex[0].xWorld - p->Vertex[1].xWorld);
	Vect1.yComp = (p->Vertex[0].yWorld - p->Vertex[1].yWorld);
	Vect1.zComp = (p->Vertex[0].zWorld - p->Vertex[1].zWorld);

	Vect2.xComp = (p->Vertex[2].xWorld - p->Vertex[1].xWorld);
	Vect2.yComp = (p->Vertex[2].yWorld - p->Vertex[1].yWorld);
	Vect2.zComp = (p->Vertex[2].zWorld - p->Vertex[1].zWorld);

	p->Normal = Vect1 CROSS Vect2;
};

void WORLD::SetCurrentCamera(char *CName)
{
	int i;

	for(i = 0; i < NumCameras; i++)
	{
		if(strcmp(Camera[i].Name, CName) == 0)
		{
			CurrentCamera = i;
			return;
		}
	}
	CurrentCamera = 0;
};