// main.cpp - 12-13-96 - Jason Zisk
// This is the main module that has "game specific" stuff.  Sets up a simple scene
// and lets you fly around.
#include <windows.h>
#include <stdio.h>
#include <d3drmwin.h>
#include <math.h>		// we use sqrt for the mouse stuff

#include "rm.h"

// Defines

#define	CLASSNAME	"SpaceFloater_Class"	// Used in Init()
#define WINDOWNAME	"Space Floater"

// Globals
HINSTANCE				g_hInst = NULL;		// Instance of this app
HWND					g_hWnd  = NULL;		// Handle of the window

BOOL					g_bActive = FALSE;	// Active?

LPDIRECT3DRMFRAME       g_Scene = NULL;     // Scene frame
LPDIRECT3DRMFRAME       g_Camera = NULL;    // Camera frame
LPDIRECT3DRMFRAME		g_Objs[7];			// The seven objects we get to fly around
LPDIRECT3DRMVIEWPORT    g_lpD3DRMViewport = NULL; // Direct3D RM Viewport

BOOL					Filtering=FALSE;	// Whether texture filtering is on or not

// Frame rate stuff
DWORD                   g_dwLastTime = 0;
DWORD                   g_dwCurTime = 0;
DWORD                   g_dwFpsTime = 0;
DWORD                   g_dwDeltaTime = 0;
DWORD                   g_dwFramesRendered = 0;
DWORD                   g_dwFps = 0;

BOOL					LBut=FALSE,RBut=FALSE;	// Left or right button down?

double					Vel=0.0;			// current velocity

// Functions
BOOL Init(HINSTANCE hInst, int nCmdShow);
static BOOL CreateScene(void);

// Handles all the windows messages we might receive
long FAR PASCAL WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{	
    // Handle messages
    switch (message) {		
		case WM_KEYDOWN:
			switch(wParam) {				
				case VK_F1:	// change video modes
				case VK_F2:
				case VK_F3:
				case VK_F4:						
					// kill viewport because it will change
					if (g_lpD3DRMViewport) {   
						g_lpD3DRMViewport->Release();
						g_lpD3DRMViewport = NULL;
					}													
					// Higher/lower mode
					if(wParam == VK_F1) EnterPrevVideoMode();
					if(wParam == VK_F2) EnterNextVideoMode();
					// highest/lowest mode
					if(wParam == VK_F3) EnterLowestVideoMode();
					if(wParam == VK_F4) EnterHighestVideoMode();
					
					// recreate the viewport
					TRY_D3DRM(g_lpD3DRM->CreateViewport(g_lpD3DRMDevice, 
														g_Camera,
														0,
														0,
														g_vidModeX,
														g_vidModeY,
														&g_lpD3DRMViewport))

					// Set the back clipping plane(don't forget this!)
					g_lpD3DRMViewport->SetBack(D3DVAL(5000.0f));
					break;	
				case VK_F5:	// toggle RGB/MONO
					// destroy viewport
					if (g_lpD3DRMViewport) {   
						g_lpD3DRMViewport->Release();
						g_lpD3DRMViewport = NULL;
					}
					if(g_colorModel==D3DCOLOR_MONO)
						InitDirectX(g_vidModeX,g_vidModeY,g_vidModeBIT,D3DCOLOR_RGB);
					else
						InitDirectX(g_vidModeX,g_vidModeY,g_vidModeBIT,D3DCOLOR_MONO);					
					// recreate the viewport
					TRY_D3DRM(g_lpD3DRM->CreateViewport(g_lpD3DRMDevice, 
														g_Camera,
														0,
														0,
														g_vidModeX,
														g_vidModeY,
														&g_lpD3DRMViewport))
					// Set the back clipping plane
					g_lpD3DRMViewport->SetBack(D3DVAL(5000.0f));
					break;
				case VK_ESCAPE:	// quit out
					PostMessage(hWnd, WM_CLOSE, 0, 0);
					break;						
				case 'R':	// reset everything					
					g_Camera->SetVelocity(g_Scene,
						D3DVALUE(0.0), D3DVALUE(0.0), D3DVALUE(0.0),FALSE);
					g_Camera->SetPosition(g_Scene,
						D3DVALUE(200.0),D3DVALUE(200.0),D3DVALUE(-200.0));
					g_Camera->LookAt(g_Objs[0],g_Scene,D3DRMCONSTRAIN_Z);					
					RBut=LBut=FALSE;
					Vel = 0.0;
					break;
				case 'F':	// texture filtering toggle(only if in RGB mode)
					if(Filtering) 
						g_lpD3DRMDevice->SetTextureQuality(D3DRMTEXTURE_NEAREST);
					else 
						g_lpD3DRMDevice->SetTextureQuality(D3DRMTEXTURE_LINEAR);
					Filtering = !Filtering;
					break;
			}
			break;		
		case WM_RBUTTONDOWN:			
			RBut = TRUE;
			break;
		case WM_LBUTTONDOWN:			
			LBut = TRUE;			
			break;
		case WM_RBUTTONUP:
			RBut=FALSE;
			break;
		case WM_LBUTTONUP:
			LBut=FALSE;
			break;		
        case WM_SYSCOMMAND:
            switch (wParam) {
                // Trap ALT so it doesn't pause the app
                case SC_KEYMENU :
                    return 0;
					break;
            }
        case WM_ACTIVATEAPP:
            // Determine whether app is being activated or not
            g_bActive = (BOOL)wParam ? TRUE : FALSE;

            if (g_bActive) {
				// Hide the cursor if we are active
                while (ShowCursor(FALSE) > 0) { };                
            } else {
                ShowCursor(TRUE);                
            }
			break;
        case WM_CLOSE:
			DestroyWindow(g_hWnd);
			break;
        case WM_DESTROY:
			// destroy all the objects created
			if(g_Camera) g_Camera->Release();
			// this will release everything under it too
			if(g_Scene) g_Scene->Release();
			
			// destroy the viewport we created
			if (g_lpD3DRMViewport) {   
				g_lpD3DRMViewport->Release();
				g_lpD3DRMViewport = NULL;
			}            
			// Show the mouse
            ShowCursor(TRUE);
			// Print out any final error messages
			FinalWord();
			// Clean up DirectX
			KillDirectX();
			// And finally get out for good.
			PostQuitMessage(0);
			break;
    }
    
    return DefWindowProc(hWnd, message, wParam, lParam);
}

int FAR PASCAL WinMain(HINSTANCE hInst, HINSTANCE hPrevInst, LPSTR lpCmdLine, int nCmdShow)
{
    MSG msg;

	double DeltaR,radius,denom;	// these used for mouse movement
	int DeltaX,DeltaY;

    // Set global handle
    g_hInst = hInst;
	
	// Initialize the window and stuff
    if (!Init(hInst, nCmdShow)) return 1;

	// Try to start in 640x480x8, let the init code decide driver(hopefully hardware)
	if(!InitDirectX(640,480,8)) return FALSE;

	// Create the Scene
    if (!CreateScene()) {
		ErrorMsg("Creation of Scene failed");
		return 1;
	}

	// Main loop
    while (TRUE) {        
		if (PeekMessage(&msg,NULL,0,0,PM_REMOVE)) {	// only process 1 message
			if (msg.message == WM_QUIT) {
                PostQuitMessage(msg.wParam);
                return 1;
            }
            TranslateMessage(&msg);
            DispatchMessage(&msg);
		}
		// Update frame
        if (g_bActive) {
			// Make sure all the surfaces are still there(for those alt-tabbing fools)
			if(!CheckSurfaces())
				TRY_D3DRM(g_lpD3DRMViewport->ForceUpdate(0, 0, g_vidModeX, g_vidModeY))

			// Doom-style mouse.  It could be better, but this is just an example.
			// Maybe someone will convert this to DirectInput and send it to me?
			// zisk@uhavax.hartford.edu
			POINT MousePos;
			GetCursorPos(&MousePos);
			int midX = g_vidModeX>>1;
			int midY = g_vidModeY>>1;
			DeltaX = MousePos.x - midX;
			DeltaY = MousePos.y - midY;
			SetCursorPos(midX,midY);
			if(DeltaX || DeltaY) {
				DeltaR = sqrt((float)DeltaX * (float)DeltaX + (float)DeltaY * (float)DeltaY);
				radius = 1024;	// bigger = slower rotation
				denom = sqrt(DeltaR * DeltaR + radius * radius);
				g_Camera->AddRotation(D3DRMCOMBINE_BEFORE,
					D3DVAL(DeltaY),D3DVAL(DeltaX),D3DVAL(0.0),
					D3DDivide(D3DVAL(DeltaR),D3DVAL(denom)));
			}

			// Do some spaceship stuff(camera movement)
			// if we are drifting slowly, just stop us
			if(Vel > -0.3 && Vel < 0.3) Vel = 0.0;
			// if left button retroes on
			if(LBut) Vel -= 0.4;
			// if right button, thrusters on
			if(RBut) Vel += 0.4;
			// cap the speeds here
			if(Vel <= -3.0) Vel = -3.0;
			if(Vel >= 3.0) Vel = 3.0;
			
			// if we hit burn register it
			if(RBut || LBut) {
				// stop the camera completely
				g_Camera->SetVelocity(g_Scene,
					D3DVALUE(0.0), D3DVALUE(0.0), D3DVALUE(0.0),FALSE);
				// now set it's new velocity
				g_Camera->SetVelocity(g_Camera,
					D3DVALUE(0.0), D3DVALUE(0.0), D3DVALUE(Vel),FALSE);
			} // otherwise just continue in same direction, thus it seems like space			

			// do everything(renders scene and updates)
			// notice how it scales the tick time by how long it took to render the
			// last frame.  Why divide by 12?  It "felt good".
			TRY_D3DRM(g_lpD3DRM->Tick(D3DVALUE((g_dwDeltaTime/12))))

			// Perform some timing stuff        
			g_dwCurTime   = timeGetTime();
			g_dwDeltaTime = g_dwCurTime - g_dwLastTime;
			g_dwLastTime  = g_dwCurTime;
			g_dwFpsTime  += g_dwDeltaTime;

			g_dwFramesRendered ++;

			char str[256];               
			if (g_dwFpsTime > 1000) {	// update every second
				g_dwFps             = g_dwFramesRendered;
				g_dwFramesRendered  = 0;
				g_dwFpsTime         = 0;
			}

			// print info
			HDC hDC;
			g_lpBackBuffer->GetDC(&hDC);
			if (!hDC) return FALSE;
			SetBkColor(hDC, 0);
			SetTextColor(hDC, RGB(0,255,0));

			char HardBool = 'N';
			char Driver[8];
			if(g_bHardware3D) HardBool = 'Y';
			if(g_colorModel == D3DCOLOR_MONO) sprintf(Driver,"Ramp");
			else sprintf(Driver,"RGB");
			sprintf(str, "FPS: %d, Hardware: %c            ",g_dwFps,HardBool);
			TextOut(hDC, 0, 0, str, strlen(str));
			sprintf(str, "Driver: %s, BitDepth: %d     ",Driver,g_vidModeBIT);
			TextOut(hDC, 0, 20, str, strlen(str));
			sprintf(str, "X: %d, Y: %d",g_vidModeX,g_vidModeY);
			TextOut(hDC, 0, 40, str, strlen(str));
					
			g_lpBackBuffer->ReleaseDC(hDC);
			
			// Finally, flip the back buffer onto the primary surface
			TRY_DD(g_lpPrimary->Flip(NULL, DDFLIP_WAIT))
        }
    }

    // Exit WinMain and terminate the app....
    return msg.wParam;
}

// Initializes all that windows junk, creates class then shows main window
BOOL Init(HINSTANCE hInst, int nCmdShow)
{
	WNDCLASS    wndClass;

    // Fill out WNDCLASS info
    wndClass.style              = CS_HREDRAW | CS_VREDRAW;
    wndClass.lpfnWndProc        = WndProc;
    wndClass.cbClsExtra         = 0;
    wndClass.cbWndExtra         = 0;
    wndClass.hInstance          = hInst;
    wndClass.hIcon              = LoadIcon(hInst, "D3DTest");
    wndClass.hCursor            = LoadCursor(NULL, IDC_ARROW);
    wndClass.hbrBackground      = GetStockObject(BLACK_BRUSH);
    wndClass.lpszMenuName       = NULL;
    wndClass.lpszClassName      = CLASSNAME;
    
	// can only run one at a time anyway, so just quit if another is running
    if (!RegisterClass(&wndClass)) return FALSE;

    // Create a window
    g_hWnd = CreateWindowEx(WS_EX_APPWINDOW,
                            CLASSNAME, 
                            WINDOWNAME,
                            WS_POPUP | WS_SYSMENU,
                            0, 0,
                            GetSystemMetrics(SM_CXSCREEN),
                            GetSystemMetrics(SM_CYSCREEN),
                            NULL,
                            NULL,
                            hInst,
                            NULL);

    // Return false if window creation failed
    if (!g_hWnd) return FALSE;
    
    // Show the window
    ShowWindow(g_hWnd, SW_SHOWNORMAL);

    // Update the window
    UpdateWindow(g_hWnd);

	return TRUE;
}

// Creates the scene
static BOOL CreateScene(void)
{
	// in case of re-call
	if(g_Scene) g_Scene->Release();

	// temps, the scene will really manage all these
    LPDIRECT3DRMFRAME light = NULL;
    LPDIRECT3DRMLIGHT light1 = NULL;
    LPDIRECT3DRMLIGHT light2 = NULL;

	// create the scene frame
    TRY_D3DRM(g_lpD3DRM->CreateFrame(NULL, &g_Scene))
    
	// create 2 lights
	TRY_D3DRM(g_lpD3DRM->CreateLightRGB(D3DRMLIGHT_PARALLELPOINT, D3DVAL(0.8), D3DVAL(0.8), D3DVAL(0.8), &light1))
    TRY_D3DRM(g_lpD3DRM->CreateLightRGB(D3DRMLIGHT_AMBIENT, D3DVAL(0.4), D3DVAL(0.4), D3DVAL(0.4), &light2))

	// create the frame the lights will be in, a child of the scene
    TRY_D3DRM(g_lpD3DRM->CreateFrame(g_Scene, &light))
	
	// add the directional light to the light frame
	TRY_D3DRM(light->AddLight(light1))
    light1->Release();
	// add the ambient light to the scene directly
    TRY_D3DRM(g_Scene->AddLight(light2))
    light2->Release();

	// load the object once and add it to a bunch of frames
	// this is much much much faster than loading from the hard drive 7 times
	for(int i=0;i<7;i++) {
		TRY_D3DRM(g_lpD3DRM->CreateFrame(g_Scene, &g_Objs[i]))
	}
	TRY_D3DRM(g_Objs[0]->Load("SPHERE.X",NULL, D3DRMLOAD_FROMFILE,
							NULL,NULL))	
	for(i=1;i<7;i++) {
		g_Objs[i]->AddVisual(g_Objs[0]);
	}

	// Move them around
	g_Objs[1]->SetPosition(g_Scene,D3DVALUE(0.0),D3DVALUE(0.0),D3DVALUE(100.0));
	g_Objs[2]->SetPosition(g_Scene,D3DVALUE(0.0),D3DVALUE(0.0),D3DVALUE(-100.0));
	g_Objs[3]->SetPosition(g_Scene,D3DVALUE(100.0),D3DVALUE(0.0),D3DVALUE(0.0));
	g_Objs[4]->SetPosition(g_Scene,D3DVALUE(-100.0),D3DVALUE(0.0),D3DVALUE(0.0));
	g_Objs[5]->SetPosition(g_Scene,D3DVALUE(0.0),D3DVALUE(100.0),D3DVALUE(0.0));
	g_Objs[6]->SetPosition(g_Scene,D3DVALUE(0.0),D3DVALUE(-100.0),D3DVALUE(0.0));

	// Create the camera, child of the scene
    TRY_D3DRM(g_lpD3DRM->CreateFrame(g_Scene, &g_Camera))
	// Set the camera's position
    TRY_D3DRM(g_Camera->SetPosition(g_Scene, 
		D3DVAL(200.0), D3DVAL(200.0), D3DVAL(-200.0)))
	// Make it look at the center object
	TRY_D3DRM(g_Camera->LookAt(g_Objs[0],g_Scene,D3DRMCONSTRAIN_Z))

	// set the position of the light frame
    TRY_D3DRM(light->SetPosition(g_Scene, D3DVAL(0.0), D3DVAL(50.0), D3DVAL(0.0)))
	
	// Create RM viewport from device and camera(full screen)
    TRY_D3DRM(g_lpD3DRM->CreateViewport(g_lpD3DRMDevice, 
                                        g_Camera,
                                        0,
                                        0,
                                        g_vidModeX,
                                        g_vidModeY,
                                        &g_lpD3DRMViewport))

    // Set the back clipping plane
    g_lpD3DRMViewport->SetBack(D3DVAL(5000.0f));

    return TRUE;
}