// rm.cpp - 12-13-96 - Jason Zisk
// This has all the "technology" in it.  Most of this can be used as-is in Direct3D
// programs, just call and go.  The enumeration/video mode code is pretty good as far
// as I can tell.  A little bit more complex than the old Floater, but much more
// robust.
#define INITGUID								// Needed by Direct3D
#define NUM_VID_MODES   50						// Maximum number of video modes

#include <windows.h>
#include <d3drmwin.h>
#include <stdio.h>

#include "rm.h"

// Structures

// This one holds the D3D device information, very handy
typedef struct _D3DDeviceInfo
{
    D3DCOLORMODEL cm;
    LPGUID        lpHWGuid;
    D3DDEVICEDESC HWDeviceDesc;
    LPGUID        lpSWGuid;
    D3DDEVICEDESC SWDeviceDesc;
} D3DDeviceInfo;

// Holds information about a particular video mode, an array of these will be created
struct VideoMode
{
    int         width;
    int         height;
    int         bpp;
    BOOL        bUsable;
};

// Globals
LPDIRECTDRAW            g_lpDD = NULL;          // DirectDraw object
LPDIRECT3D              g_lpD3D = NULL;         // Direct3D object
LPDIRECT3DRM            g_lpD3DRM = NULL;       // Direct3D RM object
DWORD                   g_dwGDIMem = 0;         // Memory available from GDI's surface

D3DDeviceInfo           g_deviceInfo;           // 3D device info
LPGUID                  g_lpD3DDeviceGuid = NULL; // Guid to 3D device

VideoMode               g_vidModes[NUM_VID_MODES];// Array of available video modes
DWORD                   g_dwCurrMode = 0;		// What mode we are currently in
DWORD                   g_dwNumModes = 0;		// Total # of modes enumerated

LPDIRECTDRAWSURFACE     g_lpPrimary = NULL;     // Primary surface
LPDIRECTDRAWSURFACE     g_lpBackBuffer = NULL;  // BackBuffer surface
LPDIRECTDRAWSURFACE     g_lpZBuffer = NULL;     // ZBuffer
LPDIRECTDRAWPALETTE     g_lpPalette = NULL;     // Palette
PALETTEENTRY            g_rPal[768];            // Temp palette used later on

LPDIRECT3DDEVICE        g_lpD3DDevice = NULL;   // Direct3D device
LPDIRECT3DRMDEVICE      g_lpD3DRMDevice = NULL; // Direct3D RM Device

DDCAPS                  g_driverCaps;           // Driver capabilities
DDCAPS                  g_HELcaps;              // HEL capabilities

DWORD                   g_dwZBufferBitDepth = 16;// ZBuffer bit depth
DWORD                   g_dwZBufferMemType = DDSCAPS_SYSTEMMEMORY;// Type of ZBuffer

D3DCOLORMODEL			g_colorModel;			// Current Color Model

// State booleans
BOOL                    g_bHardware3D = FALSE;  // Do we have hardware?

// Video mode globals
DWORD                   g_vidModeX;             // Current X video resolution
DWORD                   g_vidModeY;             // Current Y video resolution
DWORD                   g_vidModeBIT;           // Current video bit depth

// This MUST be defined in the main program for InitDirectX and ErrorMsg
// to work.
extern HWND				g_hWnd;					// Defined in WINMAIN.CPP

// Error message stuff
BOOL					g_bErrorOccured = FALSE;// Whether to print an error or not
char					g_sError[2048];			// The actual error to print

// Sorts the avaiable display modes(in g_vidModes[] array)
void SortDisplayModes(void)
{
    // Sort by width * height
    for (DWORD i = 0; i<g_dwNumModes; i ++) {
        for (DWORD k = 0; k<g_dwNumModes - 1; k ++) {
            int c1 = g_vidModes[k].width * g_vidModes[k].height;
            int c2 = g_vidModes[k + 1].width * g_vidModes[k + 1].height;

            if (c1 > c2) {
                VideoMode tmp;
                
                // Swap the two video modes
                tmp                     = g_vidModes[k];
                g_vidModes[k]           = g_vidModes[k + 1];
                g_vidModes[k + 1]       = tmp;

                // Keep g_dwCurrMode up to date
                if (g_dwCurrMode == k) {
                    g_dwCurrMode = k + 1;
                } else if (g_dwCurrMode == k + 1) {
                    g_dwCurrMode = k;
                }
            }
        }
    }
}

// Finds a DirectDraw driver that has 3D(if any)
BOOL FAR PASCAL DDEnumCallback(GUID FAR* lpGUID, LPSTR lpDriverDesc, LPSTR lpDriverName, LPVOID lpContext)
{
    LPDIRECTDRAW lpDD;
    DDCAPS DriverCaps, HELCaps;

        // Make sure the guid is valid
    if (lpGUID) {
        // Try to create a DirectDraw object
        TRY_DD(DirectDrawCreate(lpGUID, &lpDD, NULL))
        
        // Get the DirectDraw capabilities
        memset(&DriverCaps, 0, sizeof(DDCAPS));
        DriverCaps.dwSize = sizeof(DDCAPS);
        
        memset(&HELCaps, 0, sizeof(DDCAPS));
        HELCaps.dwSize = sizeof(DDCAPS);
        
        TRY_DD(lpDD->GetCaps(&DriverCaps, &HELCaps))

        // Does this driver have 3D hardware capabilites?
        if (DriverCaps.dwCaps & DDCAPS_3D) {
            *(LPDIRECTDRAW*)lpContext = lpDD;
            return DDENUMRET_CANCEL;
        }

        *(LPDIRECTDRAW*)lpContext = NULL;
        lpDD->Release();
    }
    return DDENUMRET_OK;
}

// Enumerates all video modes, filling up the g_vidModes array with all that are
// available
HRESULT CALLBACK DDEnumDisplayModesCallback(LPDDSURFACEDESC pddsd, LPVOID Context)
{       
    // While each mode gets enumerated we have to decide whether 
    // the 3D device and mode are compatible
    if (g_deviceInfo.lpHWGuid) {           
        // Make sure there is enough video ram to support this mode
        // if hardware is in use
        DWORD dwBitDepthMultiplier;
        
        switch(pddsd->ddpfPixelFormat.dwRGBBitCount) {
            case 8  : dwBitDepthMultiplier = 1; break;
            case 16 : dwBitDepthMultiplier = 2; break;
            case 24 : dwBitDepthMultiplier = 3; break;
            case 32 : dwBitDepthMultiplier = 4; break;
        }

        DWORD dwVidRamNeeded = ((pddsd->dwWidth * pddsd->dwHeight) * dwBitDepthMultiplier) * 3;

        if (dwVidRamNeeded > (g_driverCaps.dwVidMemFree + g_dwGDIMem))
          return DDENUMRET_OK;

        // Make sure the Direct3D device can render at a given bit depth
        switch (pddsd->ddpfPixelFormat.dwRGBBitCount) {
            case 8 :             
                if (!(g_deviceInfo.HWDeviceDesc.dwDeviceRenderBitDepth & DDBD_8)) return DDENUMRET_OK;
				break;
            case 16 :            
                if (!(g_deviceInfo.HWDeviceDesc.dwDeviceRenderBitDepth & DDBD_16)) return DDENUMRET_OK;
				break;
            case 24 :             
                if (!(g_deviceInfo.HWDeviceDesc.dwDeviceRenderBitDepth & DDBD_24)) return DDENUMRET_OK;
				break;
            case 32 :            
                if (!(g_deviceInfo.HWDeviceDesc.dwDeviceRenderBitDepth & DDBD_32)) return DDENUMRET_OK;
				break;
        }

        // If we have hardware, start up in requested mode if possible
        if ((pddsd->dwWidth == g_vidModeX) && (pddsd->dwHeight == g_vidModeY) && (pddsd->ddpfPixelFormat.dwRGBBitCount == g_vidModeBIT))
            g_dwCurrMode = g_dwNumModes;
		// At least try to get the size right
		else if ((pddsd->dwWidth == g_vidModeX) && (pddsd->dwHeight == g_vidModeY))
			g_dwCurrMode = g_dwNumModes;
    }

    // Record the video mode information
    g_vidModes[g_dwNumModes].width  = pddsd->dwWidth;
    g_vidModes[g_dwNumModes].height = pddsd->dwHeight;
    g_vidModes[g_dwNumModes].bpp    = pddsd->ddpfPixelFormat.dwRGBBitCount;
    
    g_dwNumModes ++;
            
    return DDENUMRET_OK;
}

HRESULT WINAPI D3DEnumDeviceCallBack(LPGUID lpGuid,     
                                     LPSTR lpDeviceDescription,
                                     LPSTR lpDeviceName,
                                     LPD3DDEVICEDESC lpHWDesc, 
                                     LPD3DDEVICEDESC lpHELDesc, 
                                     LPVOID lpContext)
{
    static BOOL bFoundHardwareDevice = FALSE;   

    // No need to enumerate if we already found the device that supports what we want
    if (bFoundHardwareDevice) return D3DENUMRET_OK;

    D3DDeviceInfo* pInfo = (D3DDeviceInfo *)lpContext;
    
    // Is this a hardware device?
    if (lpHWDesc->dcmColorModel & pInfo->cm) {
        // Driver needs to pass some tests....

        // Make sure the driver has ZBuffering and Texturing capabilities
		// You can put any caps here that are required for you app.
        if (((lpHWDesc->dwDeviceZBufferBitDepth & DDBD_16) || 
            (lpHWDesc->dwDeviceZBufferBitDepth & DDBD_24) ||
            (lpHWDesc->dwDeviceZBufferBitDepth & DDBD_32)) &&
            ((lpHWDesc->dwDevCaps & D3DDEVCAPS_TEXTURESYSTEMMEMORY) ||
			(lpHWDesc->dwDevCaps & D3DDEVCAPS_TEXTUREVIDEOMEMORY))) {                                       
            // Record the HAL description for later use
            memcpy(&pInfo->HWDeviceDesc, lpHWDesc, sizeof(D3DDEVICEDESC));

            // Record the guid for later use
            pInfo->lpHWGuid = lpGuid;
            
            // No need to keep looking for any more devices
            bFoundHardwareDevice = TRUE;
        }
        return D3DENUMRET_OK;
    }

    // Is this a software device?
    if (lpHELDesc->dcmColorModel & pInfo->cm) {
        // Record the HEL description for later use
        memcpy(&pInfo->SWDeviceDesc, lpHELDesc, sizeof(D3DDEVICEDESC));
            
        // Record the guid for later use
        pInfo->lpSWGuid = lpGuid;
            
        g_lpD3DDeviceGuid = lpGuid;
    }
    return D3DENUMRET_OK;
}

// This is the important one, finds the D3D driver we are going to use
BOOL InitD3DDevice()
{    
    // Enumerate the drivers
    TRY_D3D(g_lpD3D->EnumDevices(D3DEnumDeviceCallBack, &g_deviceInfo)) 

	// If the driver we found is Software and RGB, reenumerate for a MONO one.
	if(g_deviceInfo.lpSWGuid && g_deviceInfo.cm==D3DCOLOR_RGB) {
		// Unless we said we wanted RGB no matter what
		if(g_colorModel!=D3DCOLOR_RGB) {
			g_deviceInfo.cm=D3DCOLOR_MONO;
			TRY_D3D(g_lpD3D->EnumDevices(D3DEnumDeviceCallBack, &g_deviceInfo)) 
		}
	}

    // Now do some final cleanup with the driver we selected
    if (g_deviceInfo.lpHWGuid) {
        // We have a hardware driver

        // Make sure to use a video memory based ZBuffer
        g_dwZBufferMemType = DDSCAPS_VIDEOMEMORY;

        // Use 16 bit ZBuffering if possible, higher if not
        if (g_deviceInfo.HWDeviceDesc.dwDeviceZBufferBitDepth & DDBD_16)
            g_dwZBufferBitDepth = 16;
		else if (g_deviceInfo.HWDeviceDesc.dwDeviceZBufferBitDepth & DDBD_24)
            g_dwZBufferBitDepth = 24;
        else if (g_deviceInfo.HWDeviceDesc.dwDeviceZBufferBitDepth & DDBD_32)
            g_dwZBufferBitDepth = 32;
        else g_dwZBufferBitDepth = 0;
        
        // Use Hardware device
        g_lpD3DDeviceGuid = g_deviceInfo.lpHWGuid;
    } else {
        // We have a software driver

        // Use a system memory based ZBuffer
        g_dwZBufferMemType = DDSCAPS_SYSTEMMEMORY;

        // And force the bit depth to 16
        g_dwZBufferBitDepth = 16;

        // Default to the software device
        g_lpD3DDeviceGuid = g_deviceInfo.lpSWGuid;

		// Indicate we are not using hardware(even if available)
		g_bHardware3D=FALSE;
    }
	
	// Set what our final color model is
	g_colorModel = g_deviceInfo.cm;

    return TRUE;
}

// Initializes DirectX, optionally giving it a width, height and bitdepth.
// You can also specifiy a color model, but it is not needed.
BOOL InitDirectX(int W, int H, int BD, D3DCOLORMODEL cm)
{
    BYTE        pal[768];
    DDSURFACEDESC ddsd;
	
	// if we have specified a video mode, try to get that one
	if(W!=0 || H!=0) {
		g_vidModeX = W;
		g_vidModeY = H;
		g_vidModeBIT = BD;
	}

	// if we have specified a color model, make sure we get that one
	if(cm) g_colorModel=cm;

    // Enumerate DirectDraw drivers to see what is installed, preferring one with
    // hardware 3D capabilities
    TRY_DD(DirectDrawEnumerate(DDEnumCallback, &g_lpDD))

    // If g_lpDD is NULL, there isn't a DirectDraw device with hardware 3D capabilities,
    // so create a device using the HEL 
    if (!g_lpDD) TRY_DD(DirectDrawCreate(NULL, &g_lpDD, NULL))

    // Zero out caps structures
    memset(&g_driverCaps, 0, sizeof(DDCAPS));
    g_driverCaps.dwSize = sizeof(DDCAPS);
    memset(&g_HELcaps, 0, sizeof(DDCAPS));
    g_HELcaps.dwSize = sizeof(DDCAPS);

    // Get the current display mode as we can use that memory when full
    // screen exclusive
    memset(&ddsd, 0, sizeof ddsd);
    ddsd.dwSize = sizeof ddsd;
	TRY_DD(g_lpDD->GetDisplayMode(&ddsd));
    g_dwGDIMem = ddsd.lPitch * ddsd.dwHeight *
      (ddsd.ddpfPixelFormat.dwRGBBitCount / 8);

    // Get hardware capabilities
    TRY_DD(g_lpDD->GetCaps(&g_driverCaps, &g_HELcaps))

    // Global to determine whether we have hardware 3D capabilities or not
    g_bHardware3D = g_driverCaps.dwCaps & DDCAPS_3D;

	memset(&g_deviceInfo, 0, sizeof(D3DDeviceInfo));

	// if we have specified a color model try to get that one
	if(cm!=-1) g_deviceInfo.cm = cm;
	// otherwise try to get RGB if hardware, MONO if software
	else g_deviceInfo.cm = g_bHardware3D ? D3DCOLOR_RGB : D3DCOLOR_MONO;

    // Create Direct3D object
    TRY_D3D(g_lpDD->QueryInterface(IID_IDirect3D, (LPVOID *)&g_lpD3D))

    // Enumerate Direct3D devices, preferring hardware rendering over software
    if (!InitD3DDevice()) {
            ErrorMsg("Error locating suitable Direct3D driver");
            return FALSE;
    }

    // Enumerate all the display modes, this is done after locating the 3D device so
    // that any mode that is not compatible with the 3D device does not get added to
    // the list of valid modes
	TRY_DD(g_lpDD->EnumDisplayModes(0, NULL, NULL, DDEnumDisplayModesCallback))

	// Sort display modes into lowest width * height first
	SortDisplayModes();

    // Create Direct3D RM object
    TRY_D3DRM(Direct3DRMCreate(&g_lpD3DRM))

    // Set default amount of texture colours(this should depend on video mode)
    g_lpD3DRM->SetDefaultTextureColors(16);
    
    // Set default amount of texture shades(this should depend on video mode)
    g_lpD3DRM->SetDefaultTextureShades(16);

    // Set up palette 
	// These 3 are read only, D3DRM can't change them.
    g_rPal[0].peFlags   = D3DPAL_READONLY;
    g_rPal[254].peFlags = D3DPAL_READONLY;
    g_rPal[255].peFlags = D3DPAL_READONLY;
    
	// The rest are free for D3DRM's use, it can set then to whatever it wants.
    for (int i = 1; i < 254; i++) {
        g_rPal[i].peRed   = pal[i * 3];
        g_rPal[i].peGreen = pal[(i * 3) + 1];
        g_rPal[i].peBlue  = pal[(i * 3) + 2];                   
        g_rPal[i].peFlags = D3DPAL_FREE;
    }
    
    // Set the entry 254 to a green color for text drawing
    g_rPal[254].peRed   = 0;
    g_rPal[254].peGreen = 255;
    g_rPal[254].peBlue  = 0;

	// Set exlusive mode
    TRY_DD(g_lpDD->SetCooperativeLevel(g_hWnd, DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN | DDSCL_ALLOWMODEX))

	// This is where we actually set the video mode.
	if(W!=0 && H!=0) {	// if we requested a mode
		// See if it is available
		for(unsigned int i=0;i<g_dwNumModes;i++) {
			if(g_vidModes[i].width==W && g_vidModes[i].height==H) {
				// Width and height are correct, now see if we can get bit depth.				
				// Set this one for now, but continue looking as there might be a
				// better one.
				g_dwCurrMode=i;
				if(g_vidModes[i].bpp==BD) {
					// Bit depth is correct too!  Stop looking
					break;
				}
			}
		}
	}

	// Enter video mode set in g_dwCurMode
	if (!EnterVideoMode(g_dwCurrMode)) {
		ErrorMsg("Can't set video mode");
		return 1;
	}

    return TRUE;
}

BOOL EnterVideoMode(int mode)
{
    int width    = g_vidModes[mode].width;
    int height   = g_vidModes[mode].height;
    int bitdepth = g_vidModes[mode].bpp;
    g_dwCurrMode = mode;

    // Try to enter video mode described by width, height and bitdepth
    return EnterVideoModeWHBD(width, height, bitdepth);
}

BOOL EnterVideoModeWHBD(int width, int height, int bitdepth)
{
    DDSURFACEDESC ddsd;
    DDSCAPS ddscaps;

    // Destroy all existing devices and surfaces     

    // Destroy Direct3D RM device
    if (g_lpD3DRMDevice) {
        g_lpD3DRMDevice->Release();
        g_lpD3DRMDevice = NULL;
    }

    // Destroy Direct3D device
    if (g_lpD3DDevice) {
        g_lpD3DDevice->Release();
        g_lpD3DDevice = NULL;
    }

    // Destroy ZBuffer
    if (g_lpZBuffer) {
        g_lpZBuffer->Release();
        g_lpZBuffer = NULL;
    }

    // Destroy Primary surface
    if (g_lpPrimary) {
        g_lpPrimary->Release();
        g_lpPrimary = NULL;
    }

    // Switch video mode
    TRY_DD(g_lpDD->SetDisplayMode(width, height, bitdepth))

    // First, create complex flipping primary surface
    
    // Fill out surface description
    memset(&ddsd, sizeof(DDSURFACEDESC), 0);
    ddsd.dwSize                 = sizeof(DDSURFACEDESC);
    ddsd.dwFlags                = DDSD_CAPS | DDSD_BACKBUFFERCOUNT;
    ddsd.ddsCaps.dwCaps         = DDSCAPS_PRIMARYSURFACE | DDSCAPS_FLIP | DDSCAPS_COMPLEX | DDSCAPS_3DDEVICE;
    ddsd.dwBackBufferCount      = 1;

    // Create the primary surface with 1 back buffer
    TRY_DD(g_lpDD->CreateSurface(&ddsd, &g_lpPrimary, NULL))

    // Get pointer to back buffer
    ddscaps.dwCaps = DDSCAPS_BACKBUFFER;
    TRY_DD(g_lpPrimary->GetAttachedSurface(&ddscaps, &g_lpBackBuffer))

    // Only create a ZBuffer if g_dwZBufferBitDepth > 0
    if (g_dwZBufferBitDepth) {
        // Then, create Z-Buffer. The g_dwZBufferMemType and g_dwZBufferBitDepth variables
        // are set up when the Direct3D device enumeration is done at runtime
        memset(&ddsd, sizeof(DDSURFACEDESC), 0);
        ddsd.dwSize             = sizeof(DDSURFACEDESC);
        ddsd.dwFlags            = DDSD_WIDTH | DDSD_HEIGHT | DDSD_CAPS | DDSD_ZBUFFERBITDEPTH;
        ddsd.dwWidth            = width;
        ddsd.dwHeight           = height;
        ddsd.ddsCaps.dwCaps     = DDSCAPS_ZBUFFER | g_dwZBufferMemType;
        ddsd.dwZBufferBitDepth  = g_dwZBufferBitDepth;

        // Create the ZBuffer
        TRY_DD(g_lpDD->CreateSurface(&ddsd, &g_lpZBuffer, NULL))

        // Attach ZBuffer to the back buffer
        TRY_DD(g_lpBackBuffer->AddAttachedSurface(g_lpZBuffer))
    }

    // Retrieve the caps of the primary surface
    TRY_DD(g_lpPrimary->GetCaps(&ddscaps))

    // Create and attach palette (only if we in 8-bit palettized colour modes)
    if ((bitdepth == 8) && (ddscaps.dwCaps & DDCAPS_PALETTE)) {
        // Create the palette
        TRY_DD(g_lpDD->CreatePalette(DDPCAPS_8BIT | DDPCAPS_INITIALIZE, g_rPal, &g_lpPalette, NULL))
        
        // Set the back buffer's palette
        TRY_DD(g_lpBackBuffer->SetPalette(g_lpPalette))

        // Set the primary surface's palette
        TRY_DD(g_lpPrimary->SetPalette(g_lpPalette))
    }

    // Create Direct3D device
    TRY_D3D(g_lpBackBuffer->QueryInterface(*g_lpD3DDeviceGuid, (LPVOID *)&g_lpD3DDevice))

    // Create Direct3D RM Device from Direct3D Device
    TRY_D3DRM(g_lpD3DRM->CreateDeviceFromD3D(g_lpD3D, g_lpD3DDevice, &g_lpD3DRMDevice))

    // Set the buffer count to 2 so D3DRM can keep track of extents for fullscreen flipping surface
    TRY_D3DRM(g_lpD3DRMDevice->SetBufferCount(2))

	// Set up some defaults
    g_lpD3DRMDevice->SetQuality(D3DRMRENDER_GOURAUD);
    g_lpD3DRMDevice->SetDither(FALSE);
    g_lpD3DRMDevice->SetTextureQuality(D3DRMTEXTURE_NEAREST);
    g_lpD3DRMDevice->SetShades(8);
    
    // Record video mode information
    g_vidModeX = width;
    g_vidModeY = height;
    g_vidModeBIT = bitdepth;

    return TRUE;        
}

// Tries to enter the requested video mode, if it can't nothing happens
BOOL EnterReqVideoMode(int W, int H, int BD)
{
	int vmode=-1;
	// See if it is available
	for(unsigned int i=0;i<g_dwNumModes;i++) {
		if(g_vidModes[i].width==W && g_vidModes[i].height==H && g_vidModes[i].bpp==BD) {			
			vmode=i;
			break;
		}
	}

	// if we found it, set it
	if(vmode!=-1) {
		// Enter video mode set in g_dwCurMode
		if (!EnterVideoMode(vmode)) {
			ErrorMsg("Can't set video mode");
			return FALSE;
		}
	}
	return TRUE;
}

BOOL EnterPrevVideoMode()
{
    if (g_dwCurrMode > 0)
    {
        return EnterVideoMode(-- g_dwCurrMode);
    }
    
    return TRUE;
}

BOOL EnterNextVideoMode()
{
    if (g_dwCurrMode < g_dwNumModes - 1)
    {
        return EnterVideoMode(++ g_dwCurrMode);
    }
    
    return TRUE;
}

BOOL EnterLowestVideoMode()
{
    return EnterVideoMode(0);   
}

BOOL EnterHighestVideoMode()
{
    return EnterVideoMode(g_dwNumModes - 1);    
}

BOOL ReenterCurrentVideoMode()
{
    return EnterVideoMode(g_dwCurrMode);
}

// Checks all the surfaces created, if they are lost it restores them.  Returns FALSE
// if a surface was lost, so the viewport can be recreated if necessary
BOOL CheckSurfaces()
{    
	// Check the primary surface
    if (g_lpPrimary) {
		if (g_lpPrimary->IsLost()) {
			TRY_DD(g_lpPrimary->Restore())
			return FALSE;
		}
	}

	// Check the zbuffer(if there is one)
    if (g_lpZBuffer) {
		if (g_lpZBuffer->IsLost()) {
			TRY_DD(g_lpZBuffer->Restore())
			return FALSE;
		}
	}

    return TRUE;
}

// This kills all the DirectX objects created, and restores the original video mode
void KillDirectX()
{
    // Destroy rendering device
    if (g_lpD3DDevice) {
        g_lpD3DDevice->Release();
        g_lpD3DDevice = NULL;
    }

    // Destroy all surfaces
    if (g_lpZBuffer) {
        g_lpZBuffer->Release();
        g_lpZBuffer = NULL;
    }

    if (g_lpBackBuffer) {
        g_lpBackBuffer->Release();
        g_lpBackBuffer = NULL;
    }

    if (g_lpPrimary) {
        g_lpPrimary->Release();
        g_lpPrimary = NULL;
    }

    // Restore the original video mode  
    if(g_lpDD) g_lpDD->RestoreDisplayMode();

    // Destroy Direct3D RM object
    if (g_lpD3DRM) {
        g_lpD3DRM->Release();
        g_lpD3DRM = NULL;
    }

    // Destroy Direct3D object
    if (g_lpD3D) {
        g_lpD3D->Release();
        g_lpD3D = NULL;
    }

    // Destroy DirectDraw object
    if (g_lpDD) {
        g_lpDD->Release();
        g_lpDD = NULL;
    }
}

// Will cleanup DirectX, close the window and tell the system to print an error
void __cdecl ErrorMsg( LPSTR fmt, ... )
{
    wvsprintf(g_sError, fmt, (char *)(&fmt+1));
    lstrcat(g_sError, "\r\n");
	g_bErrorOccured=TRUE;
	KillDirectX();
	DestroyWindow(g_hWnd);
}

// Called just before final exit, prints out any error messages registered with
// ErrorMsg.  Make sure to call after window is destroyed.  Returns true if error.
BOOL FinalWord(void) 
{
	if(g_bErrorOccured) {
		MessageBox( NULL, g_sError, "Error", MB_OK );	// display it
		return TRUE;
	}
	return FALSE;
}

// The next three functions will catch any DirectX error and convert to english,
// then immediately quit the app with that message.
void TraceErrorDD(HRESULT hErr, char *sFile, int nLine)
{       
    char dderr[256];
    char err[1024];

    switch (hErr)
    {
        case DDERR_ALREADYINITIALIZED : sprintf(dderr, "DDERR_ALREADYINITIALIZED"); break;
        case DDERR_CANNOTATTACHSURFACE : sprintf(dderr, "DDERR_CANNOTATTACHSURFACE"); break;
        case DDERR_CANNOTDETACHSURFACE : sprintf(dderr, "DDERR_CANNOTDETACHSURFACE"); break;
        case DDERR_CURRENTLYNOTAVAIL : sprintf(dderr, "DDERR_CURRENTLYNOTAVAIL"); break;
        case DDERR_EXCEPTION : sprintf(dderr, "DDERR_EXCEPTION"); break;
        case DDERR_GENERIC : sprintf(dderr, "DDERR_GENERIC"); break;
        case DDERR_HEIGHTALIGN : sprintf(dderr, "DDERR_HEIGHTALIGN"); break;
        case DDERR_INCOMPATIBLEPRIMARY : sprintf(dderr, "DDERR_INCOMPATIBLEPRIMARY"); break;
        case DDERR_INVALIDCAPS : sprintf(dderr, "DDERR_INVALIDCAPS"); break;
        case DDERR_INVALIDCLIPLIST : sprintf(dderr, "DDERR_INVALIDCLIPLIST"); break;
        case DDERR_INVALIDMODE : sprintf(dderr, "DDERR_INVALIDMODE"); break;
        case DDERR_INVALIDOBJECT : sprintf(dderr, "DDERR_INVALIDOBJECT"); break;
        case DDERR_INVALIDPARAMS : sprintf(dderr, "DDERR_INVALIDPARAMS"); break;
        case DDERR_INVALIDPIXELFORMAT : sprintf(dderr, "DDERR_INVALIDPIXELFORMAT"); break;
        case DDERR_INVALIDRECT : sprintf(dderr, "DDERR_INVALIDRECT"); break;
        case DDERR_LOCKEDSURFACES : sprintf(dderr, "DDERR_LOCKEDSURFACES"); break;
        case DDERR_NO3D : sprintf(dderr, "DDERR_NO3D"); break;
        case DDERR_NOALPHAHW : sprintf(dderr, "DDERR_NOALPHAHW"); break;
        case DDERR_NOCLIPLIST : sprintf(dderr, "DDERR_NOCLIPLIST"); break;
        case DDERR_NOCOLORCONVHW : sprintf(dderr, "DDERR_NOCOLORCONVHW"); break;
        case DDERR_NOCOOPERATIVELEVELSET : sprintf(dderr, "DDERR_NOCOOPERATIVELEVELSET"); break;
        case DDERR_NOCOLORKEY : sprintf(dderr, "DDERR_NOCOLORKEY"); break;
        case DDERR_NOCOLORKEYHW : sprintf(dderr, "DDERR_NOCOLORKEYHW"); break;
        case DDERR_NODIRECTDRAWSUPPORT : sprintf(dderr, "DDERR_NODIRECTDRAWSUPPORT"); break;
        case DDERR_NOEXCLUSIVEMODE : sprintf(dderr, "DDERR_NOEXCLUSIVEMODE"); break;
        case DDERR_NOFLIPHW : sprintf(dderr, "DDERR_NOFLIPHW"); break;
        case DDERR_NOGDI : sprintf(dderr, "DDERR_NOGDI"); break;
        case DDERR_NOMIRRORHW : sprintf(dderr, "DDERR_NOMIRRORHW"); break;
        case DDERR_NOTFOUND : sprintf(dderr, "DDERR_NOTFOUND"); break;
        case DDERR_NOOVERLAYHW : sprintf(dderr, "DDERR_NOOVERLAYHW"); break;
        case DDERR_NORASTEROPHW : sprintf(dderr, "DDERR_NORASTEROPHW"); break;
        case DDERR_NOROTATIONHW : sprintf(dderr, "DDERR_NOROTATIONHW"); break;
        case DDERR_NOSTRETCHHW : sprintf(dderr, "DDERR_NOSTRETCHHW"); break;
        case DDERR_NOT4BITCOLOR : sprintf(dderr, "DDERR_NOT4BITCOLOR"); break;
        case DDERR_NOT4BITCOLORINDEX : sprintf(dderr, "DDERR_NOT4BITCOLORINDEX"); break;
        case DDERR_NOT8BITCOLOR : sprintf(dderr, "DDERR_NOT8BITCOLOR"); break;
        case DDERR_NOTEXTUREHW : sprintf(dderr, "DDERR_NOTEXTUREHW"); break;
        case DDERR_NOVSYNCHW : sprintf(dderr, "DDERR_NOVSYNCHW"); break;
        case DDERR_NOZBUFFERHW : sprintf(dderr, "DDERR_NOZBUFFERHW"); break;
        case DDERR_NOZOVERLAYHW : sprintf(dderr, "DDERR_NOZOVERLAYHW"); break;
        case DDERR_OUTOFCAPS : sprintf(dderr, "DDERR_OUTOFCAPS"); break;
        case DDERR_OUTOFMEMORY : sprintf(dderr, "DDERR_OUTOFMEMORY"); break;
        case DDERR_OUTOFVIDEOMEMORY : sprintf(dderr, "DDERR_OUTOFVIDEOMEMORY"); break;
        case DDERR_OVERLAYCANTCLIP : sprintf(dderr, "DDERR_OVERLAYCANTCLIP"); break;
        case DDERR_OVERLAYCOLORKEYONLYONEACTIVE : sprintf(dderr, "DDERR_OVERLAYCOLORKEYONLYONEACTIVE"); break;
        case DDERR_PALETTEBUSY : sprintf(dderr, "DDERR_PALETTEBUSY"); break;
        case DDERR_COLORKEYNOTSET : sprintf(dderr, "DDERR_COLORKEYNOTSET"); break;
        case DDERR_SURFACEALREADYATTACHED : sprintf(dderr, "DDERR_SURFACEALREADYATTACHED"); break;
        case DDERR_SURFACEALREADYDEPENDENT : sprintf(dderr, "DDERR_SURFACEALREADYDEPENDENT"); break;
        case DDERR_SURFACEBUSY : sprintf(dderr, "DDERR_SURFACEBUSY"); break;
        case DDERR_CANTLOCKSURFACE : sprintf(dderr, "DDERR_CANTLOCKSURFACE"); break;
        case DDERR_SURFACEISOBSCURED : sprintf(dderr, "DDERR_SURFACEISOBSCURED"); break;
        case DDERR_SURFACELOST : sprintf(dderr, "DDERR_SURFACELOST"); break;
        case DDERR_SURFACENOTATTACHED : sprintf(dderr, "DDERR_SURFACENOTATTACHED"); break;
        case DDERR_TOOBIGHEIGHT : sprintf(dderr, "DDERR_TOOBIGHEIGHT"); break;
        case DDERR_TOOBIGSIZE : sprintf(dderr, "DDERR_TOOBIGSIZE"); break;
        case DDERR_TOOBIGWIDTH : sprintf(dderr, "DDERR_TOOBIGWIDTH"); break;
        case DDERR_UNSUPPORTED : sprintf(dderr, "DDERR_UNSUPPORTED"); break;
        case DDERR_UNSUPPORTEDFORMAT : sprintf(dderr, "DDERR_UNSUPPORTEDFORMAT"); break;
        case DDERR_UNSUPPORTEDMASK : sprintf(dderr, "DDERR_UNSUPPORTEDMASK"); break;
        case DDERR_VERTICALBLANKINPROGRESS : sprintf(dderr, "DDERR_VERTICALBLANKINPROGRESS"); break;
        case DDERR_WASSTILLDRAWING : sprintf(dderr, "DDERR_WASSTILLDRAWING"); break;
        case DDERR_XALIGN : sprintf(dderr, "DDERR_XALIGN"); break;
        case DDERR_INVALIDDIRECTDRAWGUID : sprintf(dderr, "DDERR_INVALIDDIRECTDRAWGUID"); break;
        case DDERR_DIRECTDRAWALREADYCREATED : sprintf(dderr, "DDERR_DIRECTDRAWALREADYCREATED"); break;
        case DDERR_NODIRECTDRAWHW : sprintf(dderr, "DDERR_NODIRECTDRAWHW"); break;
        case DDERR_PRIMARYSURFACEALREADYEXISTS : sprintf(dderr, "DDERR_PRIMARYSURFACEALREADYEXISTS"); break;
        case DDERR_NOEMULATION : sprintf(dderr, "DDERR_NOEMULATION"); break;
        case DDERR_REGIONTOOSMALL : sprintf(dderr, "DDERR_REGIONTOOSMALL"); break;
        case DDERR_CLIPPERISUSINGHWND : sprintf(dderr, "DDERR_CLIPPERISUSINGHWND"); break;
        case DDERR_NOCLIPPERATTACHED : sprintf(dderr, "DDERR_NOCLIPPERATTACHED"); break;
        case DDERR_NOHWND : sprintf(dderr, "DDERR_NOHWND"); break;
        case DDERR_HWNDSUBCLASSED : sprintf(dderr, "DDERR_HWNDSUBCLASSED"); break;
        case DDERR_HWNDALREADYSET : sprintf(dderr, "DDERR_HWNDALREADYSET"); break;
        case DDERR_NOPALETTEATTACHED : sprintf(dderr, "DDERR_NOPALETTEATTACHED"); break;
        case DDERR_NOPALETTEHW : sprintf(dderr, "DDERR_NOPALETTEHW"); break;
        case DDERR_BLTFASTCANTCLIP : sprintf(dderr, "DDERR_BLTFASTCANTCLIP"); break;
        case DDERR_NOBLTHW : sprintf(dderr, "DDERR_NOBLTHW"); break;
        case DDERR_NODDROPSHW : sprintf(dderr, "DDERR_NODDROPSHW"); break;
        case DDERR_OVERLAYNOTVISIBLE : sprintf(dderr, "DDERR_OVERLAYNOTVISIBLE"); break;
        case DDERR_NOOVERLAYDEST : sprintf(dderr, "DDERR_NOOVERLAYDEST"); break;
        case DDERR_INVALIDPOSITION : sprintf(dderr, "DDERR_INVALIDPOSITION"); break;
        case DDERR_NOTAOVERLAYSURFACE : sprintf(dderr, "DDERR_NOTAOVERLAYSURFACE"); break;
        case DDERR_EXCLUSIVEMODEALREADYSET : sprintf(dderr, "DDERR_EXCLUSIVEMODEALREADYSET"); break;
        case DDERR_NOTFLIPPABLE : sprintf(dderr, "DDERR_NOTFLIPPABLE"); break;
        case DDERR_CANTDUPLICATE : sprintf(dderr, "DDERR_CANTDUPLICATE"); break;
        case DDERR_NOTLOCKED : sprintf(dderr, "DDERR_NOTLOCKED"); break;
        case DDERR_CANTCREATEDC : sprintf(dderr, "DDERR_CANTCREATEDC"); break;
        case DDERR_NODC : sprintf(dderr, "DDERR_NODC"); break;
        case DDERR_WRONGMODE : sprintf(dderr, "DDERR_WRONGMODE"); break;
        case DDERR_IMPLICITLYCREATED : sprintf(dderr, "DDERR_IMPLICITLYCREATED"); break;
        case DDERR_NOTPALETTIZED : sprintf(dderr, "DDERR_NOTPALETTIZED"); break;
        case DDERR_UNSUPPORTEDMODE : sprintf(dderr, "DDERR_UNSUPPORTEDMODE"); break;
        case DDERR_NOMIPMAPHW : sprintf(dderr, "DDERR_NOMIPMAPHW"); break;
        case DDERR_INVALIDSURFACETYPE : sprintf(dderr, "DDERR_INVALIDSURFACETYPE"); break;
        case DDERR_DCALREADYCREATED : sprintf(dderr, "DDERR_DCALREADYCREATED"); break;
        case DDERR_CANTPAGELOCK : sprintf(dderr, "DDERR_CANTPAGELOCK"); break;
        case DDERR_CANTPAGEUNLOCK : sprintf(dderr, "DDERR_CANTPAGEUNLOCK"); break;
        case DDERR_NOTPAGELOCKED : sprintf(dderr, "DDERR_NOTPAGELOCKED"); break;
        case DDERR_NOTINITIALIZED : sprintf(dderr, "DDERR_NOTINITIALIZED"); break;

        default : sprintf(dderr, "Unknown Error"); break;
    }
    sprintf(err, "DirectDraw Error %s\nin file %s at line %d", dderr, sFile, nLine);
    ErrorMsg(err);
}

void TraceErrorD3D(HRESULT hErr, char *sFile, int nLine)
{       
    char d3derr[256];
    char err[1024];

    switch (hErr)
    {
        case D3DERR_BADMAJORVERSION : sprintf(d3derr, "D3DERR_BADMAJORVERSION"); break;
        case D3DERR_BADMINORVERSION : sprintf(d3derr, "D3DERR_BADMINORVERSION"); break;
        case D3DERR_EXECUTE_CREATE_FAILED : sprintf(d3derr, "D3DERR_EXECUTE_CREATE_FAILED"); break;
        case D3DERR_EXECUTE_DESTROY_FAILED : sprintf(d3derr, "D3DERR_EXECUTE_DESTROY_FAILED"); break;
        case D3DERR_EXECUTE_LOCK_FAILED : sprintf(d3derr, "D3DERR_EXECUTE_LOCK_FAILED"); break;
        case D3DERR_EXECUTE_UNLOCK_FAILED : sprintf(d3derr, "D3DERR_EXECUTE_UNLOCK_FAILED"); break;
        case D3DERR_EXECUTE_LOCKED : sprintf(d3derr, "D3DERR_EXECUTE_LOCKED"); break;
        case D3DERR_EXECUTE_NOT_LOCKED : sprintf(d3derr, "D3DERR_EXECUTE_NOT_LOCKED"); break;
        case D3DERR_EXECUTE_FAILED : sprintf(d3derr, "D3DERR_EXECUTE_FAILED"); break;
        case D3DERR_EXECUTE_CLIPPED_FAILED : sprintf(d3derr, "D3DERR_EXECUTE_CLIPPED_FAILED"); break;
        case D3DERR_TEXTURE_NO_SUPPORT : sprintf(d3derr, "D3DERR_TEXTURE_NO_SUPPORT"); break;
        case D3DERR_TEXTURE_CREATE_FAILED : sprintf(d3derr, "D3DERR_TEXTURE_CREATE_FAILED"); break;
        case D3DERR_TEXTURE_DESTROY_FAILED : sprintf(d3derr, "D3DERR_TEXTURE_DESTROY_FAILED"); break;
        case D3DERR_TEXTURE_LOCK_FAILED : sprintf(d3derr, "D3DERR_TEXTURE_LOCK_FAILED"); break;
        case D3DERR_TEXTURE_UNLOCK_FAILED : sprintf(d3derr, "D3DERR_TEXTURE_UNLOCK_FAILED"); break;
        case D3DERR_TEXTURE_LOAD_FAILED : sprintf(d3derr, "D3DERR_TEXTURE_LOAD_FAILED"); break;
        case D3DERR_TEXTURE_SWAP_FAILED : sprintf(d3derr, "D3DERR_TEXTURE_SWAP_FAILED"); break;
        case D3DERR_TEXTURE_LOCKED : sprintf(d3derr, "D3DERR_TEXTURE_LOCKED"); break;
        case D3DERR_TEXTURE_NOT_LOCKED : sprintf(d3derr, "D3DERR_TEXTURE_NOT_LOCKED"); break;
        case D3DERR_TEXTURE_GETSURF_FAILED : sprintf(d3derr, "D3DERR_TEXTURE_GETSURF_FAILED"); break;
        case D3DERR_MATRIX_CREATE_FAILED : sprintf(d3derr, "D3DERR_MATRIX_CREATE_FAILED"); break;
        case D3DERR_MATRIX_DESTROY_FAILED : sprintf(d3derr, "D3DERR_MATRIX_DESTROY_FAILED"); break;
        case D3DERR_MATRIX_SETDATA_FAILED : sprintf(d3derr, "D3DERR_MATRIX_SETDATA_FAILED"); break;
        case D3DERR_MATRIX_GETDATA_FAILED : sprintf(d3derr, "D3DERR_MATRIX_GETDATA_FAILED"); break;
        case D3DERR_SETVIEWPORTDATA_FAILED : sprintf(d3derr, "D3DERR_SETVIEWPORTDATA_FAILED"); break;
        case D3DERR_MATERIAL_CREATE_FAILED : sprintf(d3derr, "D3DERR_MATERIAL_CREATE_FAILED"); break;
        case D3DERR_MATERIAL_DESTROY_FAILED : sprintf(d3derr, "D3DERR_MATERIAL_DESTROY_FAILED"); break;
        case D3DERR_MATERIAL_SETDATA_FAILED : sprintf(d3derr, "D3DERR_MATERIAL_SETDATA_FAILED"); break;
        case D3DERR_MATERIAL_GETDATA_FAILED : sprintf(d3derr, "D3DERR_MATERIAL_GETDATA_FAILED"); break;
        case D3DERR_LIGHT_SET_FAILED : sprintf(d3derr, "D3DERR_LIGHT_SET_FAILED"); break;
        case D3DERR_SCENE_IN_SCENE : sprintf(d3derr, "D3DERR_SCENE_IN_SCENE"); break;
        case D3DERR_SCENE_NOT_IN_SCENE : sprintf(d3derr, "D3DERR_SCENE_NOT_IN_SCENE"); break;
        case D3DERR_SCENE_BEGIN_FAILED : sprintf(d3derr, "D3DERR_SCENE_BEGIN_FAILED"); break;
        case D3DERR_SCENE_END_FAILED : sprintf(d3derr, "D3DERR_SCENE_END_FAILED"); break;

        default : sprintf(d3derr, "Unknown Error"); break;
    }
    sprintf(err, "Direct3D Error %s\nin file %s at line %d", d3derr, sFile, nLine);
    ErrorMsg(err);
}

void TraceErrorD3DRM(HRESULT hErr, char *sFile, int nLine)
{       
    char d3drmerr[256];
    char err[1024];

    switch (hErr)
    {
        case D3DRMERR_BADOBJECT : sprintf(d3drmerr, "D3DRMERR_BADOBJECT"); break;
        case D3DRMERR_BADTYPE : sprintf(d3drmerr, "D3DRMERR_BADTYPE"); break;
        case D3DRMERR_BADALLOC : sprintf(d3drmerr, "D3DRMERR_BADALLOC"); break;
        case D3DRMERR_FACEUSED : sprintf(d3drmerr, "D3DRMERR_FACEUSED"); break;
        case D3DRMERR_NOTFOUND : sprintf(d3drmerr, "D3DRMERR_NOTFOUND"); break;
        case D3DRMERR_NOTDONEYET : sprintf(d3drmerr, "D3DRMERR_NOTDONEYET"); break;
        case D3DRMERR_FILENOTFOUND : sprintf(d3drmerr, "D3DRMERR_FILENOTFOUND"); break;
        case D3DRMERR_BADFILE : sprintf(d3drmerr, "D3DRMERR_BADFILE"); break;
        case D3DRMERR_BADDEVICE : sprintf(d3drmerr, "D3DRMERR_BADDEVICE"); break;
        case D3DRMERR_BADVALUE : sprintf(d3drmerr, "D3DRMERR_BADVALUE"); break;
        case D3DRMERR_BADMAJORVERSION : sprintf(d3drmerr, "D3DRMERR_BADMAJORVERSION"); break;
        case D3DRMERR_BADMINORVERSION : sprintf(d3drmerr, "D3DRMERR_BADMINORVERSION"); break;
        case D3DRMERR_UNABLETOEXECUTE : sprintf(d3drmerr, "D3DRMERR_UNABLETOEXECUTE"); break;

        default : sprintf(d3drmerr, "Unknown Error"); break;
    }
    sprintf(err, "Direct3D-RM Error : %s\nin file %s at line %d", d3drmerr, sFile, nLine);
    ErrorMsg(err);
}
