/*
 *  Slick is a modified version of "Cube4" available on the MSDL service.
 *  Cube4 was originally published in the November 94 issue of MSJ.  All
 *  credit for the original code goes to Microsoft, I just goofed with
 *  it a little.
 *  Slick is modified to demonstrate (and play around with!) some of 
 *  the capabilities of OpenGl in WinNT.
 *  Slick uses double buffering to make the cube rotate in space and several
 *  GLfloat variables to alter color and size of the cube as it rotates.  If
 *  nothing else it should give you a headache!
 */

#include <windows.h>
#include <GL/gl.h>
#include <GL/glu.h>


LONG WINAPI WndProc (HWND, UINT, WPARAM, LPARAM);
void SetDCPixelFormat (HDC);
void InitializeRC (void);
void DrawScene (HDC, UINT);

HPALETTE hPalette = NULL;
GLfloat nSize = 0.0f;
GLfloat nCol = 0.0f;
GLfloat nTop = 1.0f;
GLfloat nBottom = 0.0f;
int bFlag = 1;


/*
 *  Function WinMain.
 */

int WINAPI WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance,
                    LPSTR lpszCmdLine, int nCmdShow)
{
    static char szAppName[] = "Slick";
    WNDCLASS wc;
    HWND hwnd;
    MSG msg;


    wc.style = CS_HREDRAW | CS_VREDRAW;
    wc.lpfnWndProc = (WNDPROC) WndProc;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = 0;
    wc.hInstance = hInstance;
    wc.hIcon = LoadIcon (NULL, IDI_APPLICATION);
    wc.hCursor = LoadCursor (NULL, IDC_ARROW);
    wc.hbrBackground = (HBRUSH) (COLOR_WINDOW + 1);
    wc.lpszMenuName = NULL;
    wc.lpszClassName = szAppName;

    RegisterClass (&wc);

    hwnd = CreateWindow (szAppName, szAppName,
        WS_OVERLAPPEDWINDOW | WS_CLIPCHILDREN | WS_CLIPSIBLINGS,
        CW_USEDEFAULT, CW_USEDEFAULT, 200, 200,
        HWND_DESKTOP, NULL, hInstance, NULL);

    ShowWindow (hwnd, nCmdShow);
    UpdateWindow (hwnd);

    while (GetMessage (&msg, NULL, 0, 0)) {
        TranslateMessage (&msg);
        DispatchMessage (&msg);
    }
    return msg.wParam;
}

/*
 *  WndProc processes messages to the main window.
 */
 
LONG WINAPI WndProc (HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
    static HDC hdc;
    static HGLRC hrc;
    PAINTSTRUCT ps;
    GLdouble gldAspect;
    GLsizei glnWidth, glnHeight;
    static UINT nAngle = 0;
    static UINT nTimer;
    int n;
                
    switch (msg) {
    
    case WM_CREATE:
        //
        // Create a rendering context and set a timer.
        //
        hdc = GetDC (hwnd);
        SetDCPixelFormat (hdc);
        hrc = wglCreateContext (hdc);
        wglMakeCurrent (hdc, hrc);
        InitializeRC ();
        nTimer = SetTimer (hwnd, 1, 15, NULL);
        return 0;
    
    case WM_SIZE:
        //
        // Redefine the viewing volume and viewport when the window size
        // changes.
        //
        glnWidth = (GLsizei) LOWORD (lParam);
        glnHeight = (GLsizei) HIWORD (lParam);
        gldAspect = (GLdouble) glnWidth / (GLdouble) glnHeight;

        glMatrixMode (GL_PROJECTION);
        glLoadIdentity ();
        gluPerspective (30.0,           // Field-of-view angle
                        gldAspect,      // Aspect ratio of viewing volume
                        1.0,            // Distance to near clipping plane
                        10.0);          // Distance to far clipping plane

        glViewport (0, 0, glnWidth, glnHeight);
        return 0;

    case WM_PAINT:
        //
        // Draw the scene.
        //
        BeginPaint (hwnd, &ps);
        DrawScene (hdc, nAngle);
        EndPaint (hwnd, &ps);
        return 0;       

    case WM_TIMER:
        //
        // Update the rotation angle and force a repaint.
        //
        nAngle += 2;
        if (nAngle >= 90)
            nAngle = 0;
		if (bFlag == 1)
			nSize += 0.05f;
			nCol += 0.01f;
			if (nSize >= nTop)
				bFlag = 0;
			
		if (bFlag == 0)
			nSize -= 0.05f;
			nCol -= 0.01f;
			if (nSize <= nBottom)
				bFlag = 1;
			
		InvalidateRect (hwnd, NULL, FALSE);         
        return 0;

    case WM_QUERYNEWPALETTE:
        //
        // If the program is using a color palette, realize the palette
        // and update the client area when the window receives the input
        // focus.
        //
        if (hPalette != NULL) {
            if (n = RealizePalette (hdc))
                InvalidateRect (hwnd, NULL, FALSE);
            return n;
        }
        break;
        
    case WM_PALETTECHANGED:
        //
        // If the program is using a color palette, realize the palette
        // and update the colors in the client area when another program
        // realizes its palette.
        //
        if ((hPalette != NULL) && ((HWND) wParam != hwnd)) {
            if (RealizePalette (hdc))
                UpdateColors (hdc);
            return 0;
        }
        break;

    case WM_DESTROY:
        //
        // Clean up and terminate.
        //
        wglMakeCurrent (NULL, NULL);
        wglDeleteContext (hrc);
        ReleaseDC (hwnd, hdc);
        if (hPalette != NULL)
            DeleteObject (hPalette);
        KillTimer (hwnd, nTimer);
        PostQuitMessage (0);
        return 0;
    }
    return DefWindowProc (hwnd, msg, wParam, lParam);
}

/*
 *  SetDCPixelFormat sets the pixel format for a device context in
 *  preparation for creating a rendering context.
 *
 *  Input parameters:
 *    hdc = Device context handle
 *
 *  Returns:
 *    Nothing
 */
 
void SetDCPixelFormat (HDC hdc)
{
    HANDLE hHeap;
    int nColors, i;
    LPLOGPALETTE lpPalette;
    BYTE byRedMask, byGreenMask, byBlueMask;

    static PIXELFORMATDESCRIPTOR pfd = {
        sizeof (PIXELFORMATDESCRIPTOR),             // Size of this structure
        1,                                          // Version number
        PFD_DRAW_TO_WINDOW |                        // Flags
        PFD_SUPPORT_OPENGL |
        PFD_DOUBLEBUFFER,
        PFD_TYPE_RGBA,                              // RGBA pixel values
        24,                                         // 24-bit color
        0, 0, 0, 0, 0, 0,                           // Don't care about these
        0, 0,                                       // No alpha buffer
        0, 0, 0, 0, 0,                              // No accumulation buffer
        32,                                         // 32-bit depth buffer
        0,                                          // No stencil buffer
        0,                                          // No auxiliary buffers
        PFD_MAIN_PLANE,                             // Layer type
        0,                                          // Reserved (must be 0)
        0, 0, 0                                     // No layer masks
    };

    int nPixelFormat;
    
    nPixelFormat = ChoosePixelFormat (hdc, &pfd);
    SetPixelFormat (hdc, nPixelFormat, &pfd);

    DescribePixelFormat (hdc, nPixelFormat, sizeof (PIXELFORMATDESCRIPTOR),
        &pfd);

    if (pfd.dwFlags & PFD_NEED_PALETTE) {
        nColors = 1 << pfd.cColorBits;
        hHeap = GetProcessHeap ();

        (LPLOGPALETTE) lpPalette = HeapAlloc (hHeap, 0,
            sizeof (LOGPALETTE) + (nColors * sizeof (PALETTEENTRY)));
            
        lpPalette->palVersion = 0x300;
        lpPalette->palNumEntries = nColors;

        byRedMask = (1 << pfd.cRedBits) - 1;
        byGreenMask = (1 << pfd.cGreenBits) - 1;
        byBlueMask = (1 << pfd.cBlueBits) - 1;

        for (i=0; i<nColors; i++) {
            lpPalette->palPalEntry[i].peRed =
                (((i >> pfd.cRedShift) & byRedMask) * 255) / byRedMask;
            lpPalette->palPalEntry[i].peGreen =
                (((i >> pfd.cGreenShift) & byGreenMask) * 255) / byGreenMask;
            lpPalette->palPalEntry[i].peBlue =
                (((i >> pfd.cBlueShift) & byBlueMask) * 255) / byBlueMask;
            lpPalette->palPalEntry[i].peFlags = 0;
        }

        hPalette = CreatePalette (lpPalette);
        HeapFree (hHeap, 0, lpPalette);

        if (hPalette != NULL) {
            SelectPalette (hdc, hPalette, FALSE);
            RealizePalette (hdc);
        }
    }
}

/*
 *  InitializeRC initializes the current rendering context.
 *
 *  Input parameters:
 *    None
 *
 *  Returns:
 *    Nothing
 */

void InitializeRC (void)
{
    GLfloat glfLightAmbient[] = { 0.1f, 0.1f, 0.1f, 1.0f };
    GLfloat glfLightDiffuse[] = { 0.9f, 0.9f, 0.9f, 1.0f };
    GLfloat glfLightSpecular[] = { 0.2f, 0.2f, 0.2f, 1.0f };

    //
    // Enable depth testing and backface culling.
    //
    glEnable (GL_DEPTH_TEST);
    glEnable (GL_CULL_FACE);

    //
    // Add a light to the scene.
    //
    glLightfv (GL_LIGHT0, GL_AMBIENT, glfLightAmbient);
    glLightfv (GL_LIGHT0, GL_DIFFUSE, glfLightDiffuse);
    glLightfv (GL_LIGHT0, GL_SPECULAR, glfLightSpecular);
    glEnable (GL_LIGHTING);
    glEnable (GL_LIGHT0);
}

/*
 *  DrawScene uses OpenGL commands to draw a cube.
 *
 *  Input parameters:
 *    hdc = Device context handle
 *    nAngle = Angle of rotation for cube
 *
 *  Returns:
 *    Nothing
 */

void DrawScene (HDC hdc, UINT nAngle)
{
    GLfloat glfMaterialColor[] = { nSize, (1.0f - nCol), (1.0f - nSize), nCol };
    //
    // Clear the color and depth buffers.
    //
    glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    //
    // Define the modelview transformation.
    //
    glMatrixMode (GL_MODELVIEW);
    glLoadIdentity ();
    glTranslatef (0.0f, 0.0f, -8.0f);
    glRotatef (30.0f, 1.0f, 0.0f, 0.0f);
    glRotatef ((GLfloat) nAngle, 0.0f, 1.0f, 0.0f);

    //
    // Define the reflective properties of the cube's faces.
    //
    glMaterialfv (GL_FRONT, GL_AMBIENT_AND_DIFFUSE, glfMaterialColor);

    //
    // Draw the six faces of the cube.
    //
    glBegin (GL_POLYGON);
        glNormal3f (nSize, 0.0f, nSize);
        glVertex3f (nSize, nSize, nSize);
        glVertex3f (-nSize, nSize, nSize);            
        glVertex3f (-nSize, -nSize, nSize);           
        glVertex3f (nSize, -nSize, nSize);            
    glEnd ();

    glBegin (GL_POLYGON);
        glNormal3f (0.0f, 0.0f, -nSize);
        glVertex3f (nSize, nSize, -nSize);
        glVertex3f (nSize, -nSize, -nSize);           
        glVertex3f (-nSize, -nSize, -nSize);          
        glVertex3f (-nSize, nSize, -nSize);           
    glEnd ();

    glBegin (GL_POLYGON);
        glNormal3f (-nSize, 0.0f, 0.0f);
        glVertex3f (-nSize, nSize, nSize);
        glVertex3f (-nSize, nSize, -nSize);           
        glVertex3f (-nSize, -nSize, -nSize);          
        glVertex3f (-nSize, -nSize, nSize);           
    glEnd ();
	
    glBegin (GL_POLYGON);
        glNormal3f (nSize, 0.0f, 0.0f);
        glVertex3f (nSize, nSize, nSize);
        glVertex3f (nSize, -nSize, nSize);            
        glVertex3f (nSize, -nSize, -nSize);           
        glVertex3f (nSize, nSize, -nSize);            
    glEnd ();
	
    glBegin (GL_POLYGON);
        glNormal3f (0.0f, nSize, 0.0f);
        glVertex3f (-nSize, nSize, -nSize);
        glVertex3f (-nSize, nSize, nSize);            
        glVertex3f (nSize, nSize, nSize);             
        glVertex3f (nSize, nSize, -nSize);            
    glEnd ();
	
    glBegin (GL_POLYGON);
        glNormal3f (0.0f, -nSize, 0.0f);
        glVertex3f (-nSize, -nSize, -nSize);
        glVertex3f (nSize, -nSize, -nSize);           
        glVertex3f (nSize, -nSize, nSize);            
        glVertex3f (-nSize, -nSize, nSize);           
    glEnd ();

    //
    // Swap the buffers.
    //
    SwapBuffers (hdc);
}
