/****************************************************************************

    PROGRAM: WinMPEG

    AUTHOR:  Thanassis Tsiodras, summer term of 1995.
             emai: ttsiod@theseas.ntua.gr

    PURPOSE: Display MPEG coded video, using Grayscale or Ordered dither
             output. Can also extract frames to BMP files.

    FUNCTIONS:

    WinMain() - calls initialization function, processes message loop
    InitApplication() - initializes window data and registers window
    InitInstance() - saves instance handle and creates main window
    MainWndProc() - processes messages of main (control) window.
    About() - processes messages for "About" dialog box
    CurtainWndProc() - processes messages of display window (cinema curtain)
    LoadMpg() - Prompts for the filename and gets it
    Draw() - Associated with every timer tick, gets and displays (or saves)
             each and every frame.
    StartPlayMpeg() - Ignites the engine: allocates frame buffers, sets
             the dithering options, initializes variables, starts ticking,
             and more!
    StopPlayMpeg() - Turns everything off...
    PausePlayback() - Freezes playback, ready for...
    RestartPlayback() - a restart.
****************************************************************************/

#define STRICT

#include <windows.h>            /* required for all Windows applications */
#include <mmsystem.h>
#include <commdlg.h>            /* required for File Open Dialog Box */
#include <stdlib.h>
#include "resource.h"
#include "mpeg.h"
#include "video.h"

int quantTbl[4] = {ERROR, 1, 0, 0};
char szMpgExt[] = "MPEG Videos\0*.MPG\0";      /* possible file extensions */
char fNameMpg[128] = "";                       /* the filename */
HANDLE hInst;

BOOL timing=FALSE;                        /* Are we timing frame speed ? */
BOOL running=FALSE;          /* Is a playback on? Even if its freezed... */
BOOL dontshow=FALSE;   /* Used to avoid timing results if user hits stop */
BOOL dither=FALSE;                  /* Ordered dither flag */
BOOL toBMP=FALSE;                 /* Writing BMP files flag */
                                            
HWND hWndBtnPlay,hWndBtnStop,hWndDisplay;
                                 
PBITMAPINFO pbmi=NULL;
HPALETTE hpal;
ImageDesc imag;                    /* Characteristics of current video */
int st,en;                         /* st,en:timing start and end. */
int frames=0;                      /* Current frame reached */
HDC canvas;                        /* The DC of the Cinema Window */
char *FrameBuffer=NULL;            /* Name sez it all */
FILE *fp;                          /* Just to check if it's there... */
extern VidStream *theStream;       /* MPEG decoder special kind'o'stream */

/*************************** Function Prototypes *************************/
int PASCAL WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow) ;
BOOL InitApplication(HANDLE hInstance);
BOOL InitInstance(HANDLE hInstance, int nCmdShow);
LRESULT CALLBACK MainWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);
LRESULT CALLBACK CurtainWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);
BOOL CALLBACK About(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam);

void Draw(HWND hw);
void LoadMpg(HWND hWnd);
void StartPlayMpeg(HWND hw);
void StopPlayMpeg(HWND hWnd);
void PausePlayback(HWND hWnd);
void RestartPlayback(HWND hWnd);

void WriteBMPFile(void);
void MakeGrayPalette(void);
void MakeOrderedPalette(void);
/*
Boolean GetMPEGFrame (char *Frame);
void SetMPEGOption (MPEGOptionEnum Option, int Value);
Boolean OpenMPEG (FILE *MPEGfile, ImageDesc *ImgInfo);
void CloseMPEG (void);
*/

extern void InitOrderedDither(void);
/****************************************************************************
    FUNCTION: WinMain(HANDLE, HANDLE, LPSTR, int)

    PURPOSE: calls initialization function, processes message loop

    COMMENTS:

        Windows recognizes this function by name as the initial entry point
        for the program.  This function calls the application initialization
        routine, if no other instance of the program is running, and always
        calls the instance initialization routine.  It then executes a message
        retrieval and dispatch loop that is the top-level control structure
        for the remainder of execution.  The loop is terminated when a WM_QUIT
        message is received, at which time this function exits the application
        instance by returning the value passed by PostQuitMessage().

        If this function must abort before entering the message loop, it
        returns the conventional value NULL.
****************************************************************************/

int PASCAL WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
    MSG msg;                     /* message              */

    if (!hPrevInstance)          /* Other instances of app running? */
    if (!InitApplication(hInstance)) /* Initialize shared things */
        return (FALSE);      /* Exits if unable to initialize     */

    /* Perform initializations that apply to a specific instance */

    if (!InitInstance(hInstance, nCmdShow))
        return (FALSE);

    /* Acquire and dispatch messages until a WM_QUIT message is received. */

    while (GetMessage(&msg,    /* message structure              */
        NULL,          /* handle of window receiving the message */
        NULL,          /* lowest message to examine          */
        NULL))         /* highest message to examine         */
    {
    TranslateMessage(&msg);    /* Translates virtual key codes       */
    DispatchMessage(&msg);     /* Dispatches message to window       */
    }
    return (msg.wParam);       /* Returns the value from PostQuitMessage */
}


/****************************************************************************
    FUNCTION: InitApplication(HANDLE)

    PURPOSE: Initializes window data and registers window class

    COMMENTS:

        This function is called at initialization time only if no other
        instances of the application are running.  This function performs
        initialization tasks that can be done once for any number of running
        instances.

        In this case, we initialize a window class by filling out a data
        structure of type WNDCLASS and calling the Windows RegisterClass()
        function.  Since all instances of this application use the same window
        class, we only need to do this when the first instance is initialized.
****************************************************************************/

BOOL InitApplication(hInstance)
HANDLE hInstance;                  /* current instance       */
{
    WNDCLASS  wc;

    /* Fill in window class structure with parameters that describe the       */
    /* main window.                                                           */

    wc.style = NULL;                    /* Class style(s).                    */
    wc.lpfnWndProc =(LPVOID) MainWndProc;  /* Function to retrieve messages for  */
                                        /* windows of this class.             */
    wc.cbClsExtra = 0;                  /* No per-class extra data.           */
    wc.cbWndExtra = 0;                  /* No per-window extra data.          */
    wc.hInstance = hInstance;           /* Application that owns the class.   */
    wc.hIcon = LoadIcon(NULL, IDI_APPLICATION);
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground = (HBRUSH) (COLOR_BTNFACE+1);
    wc.lpszMenuName = "VisFliMenu";   /* Name of menu resource in .RC file. */
    wc.lpszClassName = "ControlWClass"; /* Name used in call to CreateWindow. */

    /* Register the window class and return success/failure code. */

    if (!(RegisterClass(&wc)))
       return FALSE;
       
    wc.style = NULL;                    /* Class style(s).                    */
    wc.lpfnWndProc = (LPVOID) CurtainWndProc;   /* Function to retrieve messages for  */
                                        /* windows of this class.             */
    wc.cbClsExtra = 0;                  /* No per-class extra data.           */
    wc.cbWndExtra = 0;                  /* No per-window extra data.          */
    wc.hInstance = hInstance;           /* Application that owns the class.   */
    wc.hIcon = LoadIcon(NULL, IDI_APPLICATION);
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground = (HBRUSH) GetStockObject(BLACK_BRUSH);
    wc.lpszMenuName = NULL;          /* Name of menu resource in .RC file. */
    wc.lpszClassName = "BigScreen"; /* Name used in call to CreateWindow. */

    /* Register the window class and return success/failure code. */

    return (RegisterClass(&wc));
}


/****************************************************************************
    FUNCTION:  InitInstance(HANDLE, int)

    PURPOSE:  Saves instance handle and creates main window

    COMMENTS:

        This function is called at initialization time for every instance of
        this application.  This function performs initialization tasks that
        cannot be shared by multiple instances.

        In this case, we save the instance handle in a static variable and
        create and display the main program window.
****************************************************************************/

BOOL InitInstance(hInstance, nCmdShow)
    HANDLE          hInstance;          /* Current instance identifier.       */
    int             nCmdShow;           /* Param for first ShowWindow() call. */
{
    HWND            hWnd;               /* Main window handle.                */

    /* Save the instance handle in static variable, which will be used in  */
    /* many subsequence calls from this application to Windows.            */

    hInst = hInstance;

    /* Create a main window for this application instance.  */

    hWnd = CreateWindow(
        "ControlWClass",                /* See RegisterClass() call.          */
        "Mpeg Player",       /* Text for window title bar.         */
        WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX, /* Window style.           */
        100,                            /* Default horizontal position.      */
        300,                            /* Default vertical position.        */
        352,                          /* Default width.                     */
        GetSystemMetrics(SM_CYCAPTION)+GetSystemMetrics(SM_CYMENU)+50,
                                        /* Default height.                    */

        NULL,                           /* Overlapped windows have no parent. */
        NULL,                           /* Use the window class menu.         */
        hInstance,                      /* This instance owns this window.    */
        NULL                            /* Pointer not needed.                */
    );
           
    /* If window could not be created, return "failure" */

    if (!hWnd)
        return (FALSE);

    hWndBtnPlay = CreateWindow(
        "BUTTON",
        "&Play",       
        WS_CHILD | WS_VISIBLE | BS_PUSHBUTTON,
        0,0,
        176,48,                             
        hWnd,                           /* Daddy !!!. */
        (HMENU) IDB_PLAY,
        hInstance,                      /* This instance owns this window.    */
        NULL                            /* Pointer not needed.                */
    );
    if (!hWndBtnPlay)
        return (FALSE);

    hWndBtnStop = CreateWindow(
        "BUTTON",
        "&Stop",
        WS_CHILD | WS_VISIBLE | BS_PUSHBUTTON,
        176,0,
        176,48,
        hWnd,                           /* Daddy !!!. */
        (HMENU) IDB_STOP,
        hInstance,                      /* This instance owns this window.    */
        NULL                            /* Pointer not needed.                */
    );
    if (!hWndBtnStop)
        return (FALSE);

    /* Make the window visible; update its client area; and return "success" */
    hWndDisplay = CreateWindow(
        "BigScreen",
        "MPEG Cinema",
        WS_CAPTION | WS_VISIBLE,
        0,0,
        352,260+GetSystemMetrics(SM_CYCAPTION),
        NULL,                           
        NULL,
        hInstance,                      /* This instance owns this window.    */
        NULL                            /* Pointer not needed.                */
    );
    if (!hWndBtnStop)
        return (FALSE);


    ShowWindow(hWnd, nCmdShow);  /* Show the window                        */
    UpdateWindow(hWnd);          /* Sends WM_PAINT message                 */

    ShowWindow(hWndDisplay, SW_RESTORE);  /* Show the window                        */
    UpdateWindow(hWndDisplay);          /* Sends WM_PAINT message                 */

    EnableWindow(hWndBtnStop, FALSE);
    EnableWindow(hWndBtnPlay, FALSE);

    return (TRUE);               /* Returns the value from PostQuitMessage */

}


/**********************************************************************
    FUNCTION: MakeGrayPalette()

    PURPOSE: Prepares a Gray Palette and stores it in hPal.

    COMMENTS: It's a 256 color palette from (0,0,0) to (255,255,255).
***********************************************************************/
void MakeGrayPalette(void)
{
  LOGPALETTE *plgpl;
  short      *pPalIndex;
  int         i;

  plgpl = (LOGPALETTE*) malloc(sizeof(LOGPALETTE) + 256 * sizeof(PALETTEENTRY));
  plgpl->palNumEntries = 256;
  plgpl->palVersion = 0x300;
  pPalIndex=(short *)pbmi->bmiColors;

  for (i=0; i<256; i++)
  {
    plgpl->palPalEntry[i].peRed   = (unsigned char)i;
    plgpl->palPalEntry[i].peGreen = (unsigned char)i;
    plgpl->palPalEntry[i].peBlue  = (unsigned char)i;
    plgpl->palPalEntry[i].peFlags = NULL;
    pPalIndex[i]=i;
  }
  if (hpal)
    DeleteObject(hpal);
  hpal = CreatePalette(plgpl);
  free(plgpl);
}

/*********************************************************************
    FUNCTION: MakeOrderedPalette()

    PURPOSE: Prepares a special Palette and stores it in hPal.

    COMMENTS: It's a 256 color palette used for the ordered dither.
              It needs the following table: convmat
********************************************************************/
int convmat[8][4]=
{
  {117504, 138453, 13954, 34903}, /* no sequence_display_extension */
  {117504, 138453, 13954, 34903}, /* ITU-R Rec. 709 (1990) */
  {104597, 132201, 25675, 53279}, /* unspecified */
  {104597, 132201, 25675, 53279}, /* reserved */
  {104448, 132798, 24759, 53109}, /* FCC */
  {104597, 132201, 25675, 53279}, /* ITU-R Rec. 624-4 System B, G */
  {104597, 132201, 25675, 53279}, /* SMPTE 170M */
  {117579, 136230, 16907, 35559}  /* SMPTE 240M (1987) */
};

void MakeOrderedPalette(void)
{
  LOGPALETTE *plgpl;
  short      *pPalIndex;
  int         i;
  int  crv, cbu, cgu, cgv;
  int  y, u, v;
  unsigned char *clp;

  InitOrderedDither();
  clp=(char *) malloc(1024);
  clp+=384;
  for (i=-384; i<640; i++)
    clp[i] = (i<0) ? 0 : ((i>255) ? 255 : i);

  crv = convmat[5][0];
  cbu = convmat[5][1];
  cgu = convmat[5][2];
  cgv = convmat[5][3];

  plgpl = (LOGPALETTE*) malloc(sizeof(LOGPALETTE) + 256 * sizeof(PALETTEENTRY));
  plgpl->palNumEntries = 256;
  plgpl->palVersion = 0x300;
  pPalIndex=(short *)pbmi->bmiColors;

  for (i=16; i<240; i++)
  {
      /* color space conversion */

      y = 16*((i>>4)&15) + 8;
      u = 32*((i>>2)&3)  - 48;
      v = 32*(i&3)       - 48;

      y = 76309 * (y - 16); /* (255/219)*65536 */

      plgpl->palPalEntry[i].peRed   = clp[(y + crv*v + 32768)>>16];
      plgpl->palPalEntry[i].peGreen = clp[(y - cgu*u -cgv*v + 32768)>>16];
      plgpl->palPalEntry[i].peBlue  = clp[(y + cbu*u + 32786)>>16];
      plgpl->palPalEntry[i].peFlags = NULL;
      pPalIndex[i]=i;
  }

  clp-=384;
  free(clp);

  if (hpal)
    DeleteObject(hpal);
  hpal = CreatePalette(plgpl);
  free(plgpl);
}


/****************************************************************************
    FUNCTION: CurtainWndProc(HWND, UINT, WPARAM, LPARAM)

    PURPOSE:  Processes messages of the Cinema Window.

    MESSAGES: None.

    COMMENTS: Everything is done automagically by DefWindowProc.
              We don't handle anything.
****************************************************************************/
LRESULT CALLBACK CurtainWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    return (DefWindowProc(hWnd, message, wParam, lParam));
}


/****************************************************************************
    FUNCTION: MainWndProc(HWND, UINT, WPARAM, LPARAM)

    PURPOSE:  Processes messages

    MESSAGES:

    WM_CREATE     - Allocate pbmi, prepare a gray palette.
    WM_COMMAND    - application menu
    WM_DESTROY    - destroy window (if needed, stop playback)
    WM_TIMER      - calls Draw function

    COMMENTS: What can I say? It's huge and it's all mine!
****************************************************************************/

LRESULT CALLBACK MainWndProc(hWnd, message, wParam, lParam)
HWND hWnd;                      /* window handle                 */
UINT message;                   /* type of message               */
WPARAM wParam;                  /* additional information        */
LPARAM lParam;                  /* additional information        */
{
    switch (message)
    {       
        case WM_CREATE:
          {
            pbmi    = (PBITMAPINFO)malloc(sizeof(BITMAPINFOHEADER) + 256 * sizeof(RGBQUAD));
            pbmi->bmiHeader.biSize = (LONG)sizeof(BITMAPINFOHEADER);
            pbmi->bmiHeader.biPlanes = 1;
            pbmi->bmiHeader.biCompression = 0;
            pbmi->bmiHeader.biSizeImage = 0;
            pbmi->bmiHeader.biXPelsPerMeter = 0;
            pbmi->bmiHeader.biYPelsPerMeter = 0;
            pbmi->bmiHeader.biClrUsed = 256;
            pbmi->bmiHeader.biClrImportant = 0;
            pbmi->bmiHeader.biBitCount = 8;
            MakeGrayPalette();
          }
          break;

        case WM_COMMAND:       /* message: command from application menu */
            if (wParam == IDM_ABOUT)
            {
                DLGPROC dlgproc;
                dlgproc=(DLGPROC) MakeProcInstance(About,hInst);        
                DialogBox(hInst,        /* current instance          */
                    "AboutBox",         /* resource to use           */
                    hWnd,               /* parent handle             */
                    dlgproc);             /* About() instance address  */
                FreeProcInstance(dlgproc);
                break;
            }
            else if (wParam == IDM_HELP) {
                WinHelp(hWnd, "WINMPEG.HLP", HELP_INDEX, 0L);
                break;
            }
            else if (wParam == IDM_OPENMPG)
            {
                if (running)
                   StopPlayMpeg(hWnd);
                LoadMpg(hWnd);
                break;
            }
            else if (wParam == IDM_EXIT)
            {
                PostMessage(hWnd, WM_SYSCOMMAND, SC_CLOSE, 0L);
                break;
            }
            else if (wParam == IDM_TIME)
            {
                unsigned mcstatus;

                if (timing)
                    mcstatus=MF_BYCOMMAND | MF_UNCHECKED;
                else
                    mcstatus=MF_BYCOMMAND | MF_CHECKED;
                CheckMenuItem(GetMenu(hWnd),IDM_TIME,mcstatus);
                timing=!timing;
                break;
            }
            else if (wParam == IDM_COLOR) 
            {
                unsigned mcstatus;

                if (dither) {
                    mcstatus=MF_BYCOMMAND | MF_UNCHECKED;
                    MakeGrayPalette();
		}
                else {
                    mcstatus=MF_BYCOMMAND | MF_CHECKED;
                    MakeOrderedPalette();
		}
                CheckMenuItem(GetMenu(hWnd),IDM_COLOR,mcstatus);
                dither=!dither;
		if (running) {
		   StopPlayMpeg(hWnd);
		}
                break;
            }
	    else if (wParam == IDM_BMP)
	    {
              unsigned mcstatus;

	      toBMP=!toBMP;
	      if (running) 
	        StopPlayMpeg(hWnd);
              if (toBMP)  {
                mcstatus=MF_BYCOMMAND | MF_CHECKED;
	        MessageBox(GetFocus(),
                  "Files will be output in the MPEG File's "\
                  "directory as 1.bmp, 2.bmp, etc... If for example "\
                  "you loaded file test.mpg from \\grfx\\mpeg, "\
		  "files will be output in the same directory.",
	          "Caution!",
	          MB_OK | MB_ICONEXCLAMATION);
              }
              else 
                mcstatus=MF_BYCOMMAND | MF_UNCHECKED;
              CheckMenuItem(GetMenu(hWnd),IDM_BMP,mcstatus);
	      if (toBMP)
	        mcstatus=MF_BYCOMMAND | MF_GRAYED | MF_DISABLED;
	      else
	        mcstatus=MF_BYCOMMAND | MF_ENABLED;
	      EnableMenuItem(GetMenu(hWnd),IDM_COLOR,mcstatus);
	      break;
	    }
            else if (wParam == IDM_COPY)
            {
		if (FrameBuffer!=NULL) {
                   HANDLE hDIB;
                   BITMAPINFO far *pb;
                   int idx,cnt;
		   char *src;
		   char *dest;

		   if (dither) {
		      MessageBox(GetFocus(),
			"If you want a color snap, try the BMP option! "\
                        "Try again with no dither on (gray mode) for a gray snap.",
		        "Sorry",MB_OK);
		      goto adios;
		   }
                   hDIB=GlobalAlloc(
                     GMEM_MOVEABLE | GMEM_DDESHARE,
                     sizeof(BITMAPINFOHEADER)+256*sizeof(RGBQUAD)+
                     theStream->h_size * theStream->v_size
                   );
                   if (!hDIB)
                      MessageBox(GetFocus(),
                        "Can't allocate memory for image copying...",
                        "MEMORY GONG!",
                        MB_OK);
                   else {
		      pb=(BITMAPINFO far *)  GlobalLock(hDIB);
                      pb->bmiHeader.biSize = (LONG)sizeof(BITMAPINFOHEADER);
                      pb->bmiHeader.biPlanes = 1;
                      pb->bmiHeader.biCompression = 0;
                      pb->bmiHeader.biSizeImage = 0;
                      pb->bmiHeader.biXPelsPerMeter = 0;
                      pb->bmiHeader.biYPelsPerMeter = 0;
                      pb->bmiHeader.biClrUsed = 256;
                      pb->bmiHeader.biClrImportant = 0;
                      pb->bmiHeader.biBitCount = 8;
		      pb->bmiHeader.biWidth=theStream->h_size;
		      pb->bmiHeader.biHeight=theStream->v_size;
		      
		      if (!dither)
                         for (idx=0;idx<256;idx++) {
                           pb->bmiColors[idx].rgbBlue  = idx;
                           pb->bmiColors[idx].rgbGreen = idx;
                           pb->bmiColors[idx].rgbRed   = idx;
                           pb->bmiColors[idx].rgbReserved = 0;
                         }
		      else {

		      }
		      cnt=theStream->h_size * theStream->v_size;
		      src=FrameBuffer;
		      dest=((char *)pb)+sizeof(BITMAPINFOHEADER)+256*sizeof(RGBQUAD);
		      for(;cnt>0; cnt--)
                        *dest++=*src++;
                      GlobalUnlock(hDIB);

                      OpenClipboard(hWnd);
                      EmptyClipboard();
                      SetClipboardData(CF_DIB,hDIB);
                      CloseClipboard();
                   }
		   adios:;
                }
                else
                   MessageBeep(0);
                break;
            }
            else if (wParam == IDB_PLAY)
            {
                if (!running)
                   StartPlayMpeg(hWnd);
                else
                   RestartPlayback(hWnd);
                break;
            }
            else if (wParam == IDB_STOP)
            {
                PausePlayback(hWnd);
                break;
            }
            else                        /* Lets Windows process it   */
                return (DefWindowProc(hWnd, message, wParam, lParam));
                
        case WM_TIMER:
            Draw(hWnd);       
	    break;
        
        case WM_DESTROY:          /* message: window being destroyed */
            if (running) 
               StopPlayMpeg(hWnd);
            DeleteObject(hpal);
            free(pbmi);
            DestroyWindow(hWndDisplay);

            WinHelp(hWnd, "WINMPEG.HLP", HELP_QUIT, 0L);

            PostQuitMessage(0);
            break;           
        default:                  /* Passes it on if unproccessed    */
            return (DefWindowProc(hWnd, message, wParam, lParam));
    }
    return (NULL);
}

/****************************************************************************
    FUNCTION: About(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages for "About" dialog box

    MESSAGES:

    WM_INITDIALOG - initialize dialog box
    WM_COMMAND    - Input received

    COMMENTS:

    No initialization is needed for this particular dialog box, but TRUE
    must be returned to Windows.

    Wait for user to click on "Ok" button, then close the dialog box.
****************************************************************************/
BOOL CALLBACK About(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
    switch (message)
    {
        case WM_INITDIALOG:            /* message: initialize dialog box */
            return (TRUE);

        case WM_COMMAND:               /* message: received a command */
            if (wParam == IDOK         /* "OK" box selected?          */
                || wParam == IDCANCEL) /* System menu close command?  */
            {
                EndDialog(hDlg, TRUE); /* Exits the dialog box        */
                return (TRUE);
            }
            break;
    }
    return (FALSE);               /* Didn't process a message    */
}

/*******************************************************************
    FUNCTION: LoadMpg(HWND)

    PURPOSE:  Pops a File Open Dialog Box, and gets a filename.

    COMMENTS: Also sets the focus on the play button.
*******************************************************************/
void LoadMpg(HWND hWnd)
{
   OPENFILENAME ofn;
   int fh;

   ofn.lStructSize = sizeof(OPENFILENAME);
   ofn.hwndOwner = hWnd;
   ofn.hInstance = hInst;
   ofn.lpstrFilter = szMpgExt;
   ofn.lpstrCustomFilter = NULL;
   ofn.nMaxCustFilter = 0L;
   ofn.nFilterIndex = 1L;
   ofn.lpstrFile = (LPSTR) fNameMpg;
   ofn.nMaxFile = 128;
   ofn.lpstrInitialDir = "\\grfx\\mpeg";
   ofn.lpstrTitle = NULL;
   ofn.lpstrFileTitle = NULL;
   ofn.lpstrDefExt = NULL;
   ofn.Flags = OFN_HIDEREADONLY;

   fh=GetOpenFileName((LPOPENFILENAME)&ofn);

   if (fh > 0) {
      EnableWindow(hWndBtnPlay, TRUE);   
      SetFocus(hWndBtnPlay);
   }
}               

/**********************************************************************
    FUNCTION: StartPlayMpg(HWND)

    PURPOSE:  Vroom, vroom... Go!

    COMMENTS: Check if the file exists. Set various booleans.
              Set Dithering state. Call OpenMPEG, allocate FrameBuffer,
              set the play-stop buttons, get the curtain DC,
              check the time, start ticking!
***********************************************************************/
void StartPlayMpeg(HWND hw)
{
  dontshow=FALSE;
  if ((fp=fopen(fNameMpg,"rb"))==NULL)
    return;

  running=TRUE;
  if (!toBMP) {
    if (!dither)
      SetMPEGOption (MPEG_DITHER, GRAY_DITHER);
    else
      SetMPEGOption (MPEG_DITHER, ORDERED_DITHER);
  }
  else 
    SetMPEGOption (MPEG_DITHER, FULL_COLOR_DITHER);

  OpenMPEG(fp, &imag);
  FrameBuffer=malloc(imag.Size);
  pbmi->bmiHeader.biWidth = theStream->h_size;
  pbmi->bmiHeader.biHeight= theStream->v_size;  /*imag.Height;*/

  EnableWindow(hWndBtnStop, TRUE);
  EnableWindow(hWndBtnPlay, FALSE);
  canvas=GetDC(hWndDisplay);
  SetFocus(hWndDisplay);
  frames=0;
  st=timeGetTime();
  SetTimer(hw,0,1,NULL);
}

/**********************************************************************
    FUNCTION: Draw(HWND)

    PURPOSE:  Zap a frame on the Cinema window (or on the disk)

    COMMENTS: None.
***********************************************************************/
void Draw(HWND hw)
{
  HPALETTE hPalPrev;
  BOOL boo;
  char s[80];

  boo=GetMPEGFrame(FrameBuffer);
  if (!boo) {
    StopPlayMpeg(hw);
    return;
  }
  if (!toBMP) {
    hPalPrev=SelectPalette(canvas,hpal,FALSE);
    RealizePalette(canvas);
    SetDIBitsToDevice(
        canvas,
        (352-theStream->h_size)>>1, (260-theStream->v_size)>>1,
        theStream->h_size,theStream->v_size,
        0,0,
        0,theStream->v_size,
        FrameBuffer,pbmi,DIB_PAL_COLORS
    );
    SelectPalette(canvas,hPalPrev,FALSE);
  }
  else {
    sprintf(s,"Frame %d.bmp being written...",frames);
    TextOut(canvas,80,120,s,strlen(s));
    WriteBMPFile();
  }
  frames++;
}

/**********************************************************************
    FUNCTION: WriteBMPFile.

    PURPOSE:  Save the contents in FrameBuffer to disk.

    COMMENTS: Writes the needed header, zaps data on the disk.
              No smartdrv means REAALLLY slow output.
***********************************************************************/


void WriteBMPFile(void)
{
  struct TBITMAPFILEHEADER {
    unsigned   short   bftype;
    unsigned   bfsize;
    unsigned   short   bfreserved1,bfreserved2;
    unsigned   bfoffbits;
  } fhdr;
  struct TBITMAPINFOHEADER {
    unsigned   biSize;
    unsigned   biWidth;
    unsigned   biHeight;
    unsigned   short   biPlanes;
    unsigned   short   biBitCount;
    unsigned   biCompression;
    unsigned   biSizeImage;
    unsigned   biXPelsPerMeter;
    unsigned   biYPelsPerMeter;
    unsigned   biClrUsed;
    unsigned   biClrImportant;
  } bhdr;
  char s[80];
  char *data;
  FILE *fp;
  unsigned short x,y,i,j;
  unsigned char c;

  x=imag.Width;
  y=theStream->v_size;

  sprintf(s,"%d.bmp",frames);
  fp=fopen(s,"wb");

  fhdr.bftype='B'+256*'M';
  fhdr.bfsize=sizeof(fhdr)+sizeof(bhdr)+3*x*y;
  fhdr.bfreserved1=fhdr.bfreserved2=0;
  fhdr.bfoffbits=sizeof(fhdr)+sizeof(bhdr);

  bhdr.biSize=sizeof(bhdr);
  bhdr.biWidth=x;
  bhdr.biHeight=y;
  bhdr.biPlanes=1;
  bhdr.biBitCount=24;
  bhdr.biCompression=0;  /* BI_RGB */
  bhdr.biSizeImage=0;
  bhdr.biXPelsPerMeter=0;
  bhdr.biYPelsPerMeter=0;
  bhdr.biClrUsed=0;
  bhdr.biClrImportant=0;

  fwrite(&fhdr,1,sizeof(fhdr),fp);
  fwrite(&bhdr,1,sizeof(bhdr),fp);

  for(j=0;j<y;j++) {
    data=FrameBuffer+(y-1-j)*x*4;
    for(i=0;i<x;i++) {
      fwrite(data, 1,3,fp);
      data+=4;
    }
  }
  fclose(fp);
}

/**********************************************************************
    FUNCTION: PausePlayback(HWND)

    PURPOSE:  Ditto.

    COMMENTS: None.
***********************************************************************/
void PausePlayback(HWND hWnd)
{
  KillTimer(hWnd,0);
  dontshow=TRUE;
  EnableWindow(hWndBtnStop, FALSE);
  EnableWindow(hWndBtnPlay, TRUE);
}

/**********************************************************************
    FUNCTION: RestartPlayback(HWND)

    PURPOSE:  Name sez it all.

    COMMENTS: None.
***********************************************************************/
void RestartPlayback(HWND hWnd)
{
  SetTimer(hWnd,0,1,NULL);
  EnableWindow(hWndBtnStop, TRUE);
  EnableWindow(hWndBtnPlay, FALSE);
}

/**********************************************************************
    FUNCTION: StopPlayMpeg(HWND)

    PURPOSE:  Engine stop.

    COMMENTS: Check the time. Stop the timer. CloseMPEG and close file.
              Free memory of the FrameBuffer. Release the canvas.
              Reset the buttons. Clear the screen. If needed, show
              results.
***********************************************************************/
void StopPlayMpeg(HWND hWnd)
{
  char mesg[80];

  en=timeGetTime();
  KillTimer(hWnd,0);
  CloseMPEG();
  fclose(fp);
  free(FrameBuffer);
  FrameBuffer=NULL;
  ReleaseDC(hWndDisplay,canvas);
  EnableWindow(hWndBtnStop, FALSE);
  EnableWindow(hWndBtnPlay, TRUE);
  InvalidateRect(hWndDisplay,NULL,TRUE);
  if (timing && !dontshow) {
     sprintf(mesg,"Frame Ratio : %3.1f",frames/((en-st)/1000.0));
     MessageBox(GetFocus(),mesg,"Playback speed",MB_OK);
  }
  running=FALSE;
}
