//
//  Genetic Algorithm (GA) Demo
//
//  Traveling Salesman Problem
//
//  Copyright Cygron 1997
//

unit main;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  Dialogs, Ga, StdCtrls, ExtCtrls, ComCtrls;

const
  MAX = 10; // maximum number of towns
  NOROAD = 99;

type
  TForm1 = class(TForm)
    Panel1: TPanel;
    PaintBox1: TPaintBox;
    Panel2: TPanel;
    GeneticAlg1: TGeneticAlg;
    ButtonInitMap: TButton;
    ButtonReset: TButton;
    ButtonStep: TButton;
    Label1: TLabel;
    ProgressBar1: TProgressBar;
    Panel3: TPanel;
    ListBox1: TListBox;
    Panel4: TPanel;
    Label2: TLabel;
    Label3: TLabel;
    Label4: TLabel;
    procedure FormCreate(Sender: TObject);
    procedure PaintBox1Paint(Sender: TObject);
    procedure PaintBox1MouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure ButtonResetClick(Sender: TObject);
    procedure ButtonStepClick(Sender: TObject);
    procedure ButtonInitMapClick(Sender: TObject);
    function GeneticAlg1Born: Pointer;
    function GeneticAlg1Fitness(x: Pointer): Double;
  public
    pos: array[1..MAX,1..2] of Double;
    cost: array[1..MAX,1..MAX] of Double;
    n: Word;
    valid: Boolean;
    procedure AddTown(x,y: Double);
    procedure SetRoad(i,j: Word);
    procedure ShowBest;
    procedure ShowInds;
  end;

var
  Form1: TForm1;

implementation

{$R *.DFM}

//////////////////////////////////////////////////////////////////////
// add a town with the given coordinates
procedure TForm1.AddTown(x,y: Double);
var
  i: Integer;
begin
  if n<MAX
  then begin
    Inc(n);
    pos[n,1] := x;
    pos[n,2] := y;
    for i := 1 to MAX do begin
      cost[n,i] := NOROAD;
      cost[i,n] := NOROAD;
    end;
    PaintBox1.Invalidate;
    valid := False;
  end;
end;

//////////////////////////////////////////////////////////////////////
// set a road betwwen two towns
procedure TForm1.SetRoad(i,j: Word);
var
  d: Double;
begin
  d := sqrt(sqr(pos[i,1]-pos[j,1]) + sqr(pos[i,2]-pos[j,2]));
  cost[i,j] := d;
  cost[j,i] := d;
  valid := False;
  PaintBox1.Invalidate;
end;

//////////////////////////////////////////////////////////////////////
// show best individual
procedure TForm1.ShowBest;
var
  best: TPermutationInd;
  y: Double;
begin
  best := TPermutationInd(GeneticAlg1.Individuals[0]);
  y := best.m_Fitness-1000;
  Label1.Caption := 'Best tour: 0,' + best.toString(',')
     + ',0  Length: '+ Format('%.2f',[-10*y]);
  Label1.Update;
end;

//////////////////////////////////////////////////////////////////////
// show population
procedure TForm1.ShowInds;
var
  i: Integer;
  x: TPermutationInd;
  s: String;
begin
  ListBox1.Clear;
  for i := 1 to GeneticAlg1.Size do begin
    x := TPermutationInd(GeneticAlg1.Individuals[i-1]);
    s := x.toString(',');
    s := s + Format(' : %.2f',[10*(1000-x.m_Fitness)]);
    ListBox1.Items.Add(s);
  end;
end;

//////////////////////////////////////////////////////////////////////
// FormCreate
procedure TForm1.FormCreate(Sender: TObject);
begin
  Randomize;
  n := 0;
  ButtonInitMapClick(self);
  ButtonResetClick(self);
end;

//////////////////////////////////////////////////////////////////////
// paint graph
procedure TForm1.PaintBox1Paint(Sender: TObject);
var
  R: TRect;
  C: TCanvas;
  x,y,x1,y1,i,j,k: Integer;
  s: String;
  ind: TPermutationInd;
begin
  C := PaintBox1.Canvas;
  R := PaintBox1.ClientRect;
  // paint roads
  C.Pen.Color := clLime;
  C.Pen.Width := 3;
  C.Pen.Style := psSolid;
  for i := 1 to n-1 do begin
    for j := i+1 to n do begin
      if cost[i,j]<>NOROAD then begin
        x := Round(R.Right*pos[i,1]);
        y := Round(R.Bottom*pos[i,2]);
        x1 := Round(R.Right*pos[j,1]);
        y1 := Round(R.Bottom*pos[j,2]);
        C.MoveTo(x,y);
        C.LineTo(x1,y1);
      end;
    end;
  end;
  // paint best line
  if valid then begin
    C.Pen.Color := clRed;
    C.Pen.Width := 0;
    C.Pen.Style := psDash;
    ind := TPermutationInd(GeneticAlg1.Individuals[0]);
    x := Round(R.Right*pos[1,1]);
    y := Round(R.Bottom*pos[1,2]);
    C.MoveTo(x,y);
    for i := 1 to n-1 do begin
      j := ind.m_Data[i]+1;
      x := Round(R.Right*pos[j,1]);
      y := Round(R.Bottom*pos[j,2]);
      C.LineTo(x,y);
    end;
    x := Round(R.Right*pos[1,1]);
    y := Round(R.Bottom*pos[1,2]);
    C.LineTo(x,y);
  end;
  // paint towns
  C.Font.Color := clBlack;
  C.Pen.Color := clBlack;
  C.Pen.Width := 2;
  C.Pen.Style := psSolid;
  for i := 1 to n do begin
    s := IntToStr(i-1);
    x := Round(R.Right*pos[i,1]);
    y := Round(R.Bottom*pos[i,2]);
    C.Ellipse(x-10,y-10,x+11,y+11);
    C.TextOut(x-C.TextWidth(s) div 2,y-C.TextHeight(s) div 2,s);
  end;
end;

//////////////////////////////////////////////////////////////////////
// MouseDown
procedure TForm1.PaintBox1MouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  i: Integer;
  d: Double;
begin
  if n=MAX then exit;
  AddTown(X/PaintBox1.ClientRect.Right,Y/PaintBox1.ClientRect.Bottom);
  // road to previous
  SetRoad(n,n-1);
  // road some neighbour
  for i := 1 to n do begin
    d := sqrt(sqr(pos[n,1]-pos[i,1]) + sqr(pos[n,2]-pos[i,2]));
    if d<0.4 then SetRoad(n,i);
  end;
end;

//////////////////////////////////////////////////////////////////////
// OnFitness
//
// If there is no road between two towns in a solution,
// that solution gets an extremely low fitness.
function TForm1.GeneticAlg1Fitness(x: Pointer): Double;
var
  y: Double;
  i,j,k: Integer;
begin
  if n<2
  then begin
    Result := 0;
    Exit;
  end;
  y := 0;
  j := TPermutationInd(x).m_Data[1]+1;
  y := y + cost[1,j];
  j := TPermutationInd(x).m_Data[n-1]+1;
  y := y + cost[1,j];
  for i := 2 to n-1 do begin
    j := TPermutationInd(x).m_Data[i-1]+1;
    k := TPermutationInd(x).m_Data[i]+1;
    y := y + cost[j,k];
  end;
  if y>=99
  then Result := 100-y/10
  else Result := 1000-y;
end;

//////////////////////////////////////////////////////////////////////
// OnBorn
function TForm1.GeneticAlg1Born: Pointer;
begin
  if n=1
  then Result := TPermutationInd.Create(1)
  else Result := TPermutationInd.Create(n-1);
end;

//////////////////////////////////////////////////////////////////////
// reset population
procedure TForm1.ButtonResetClick(Sender: TObject);
begin
  SetRoad(1,n);
  valid := True;
  GeneticAlg1.Reset;
  PaintBox1.Invalidate;
  ShowBest;
  ShowInds;
end;

//////////////////////////////////////////////////////////////////////
// step 50
procedure TForm1.ButtonStepClick(Sender: TObject);
var
  i: Integer;
begin
  if not Valid then ButtonResetClick(self);
  for i := 1 to 50 do begin
    GeneticAlg1.Step;
    if i mod 10 = 0
    then begin
      ShowBest;
      PaintBox1.Repaint;
    end;
    ProgressBar1.Position := Round(i/5);
  end;
  ProgressBar1.Position := 0;
  PaintBox1.Invalidate;
  ShowBest;
  ShowInds;
end;

//////////////////////////////////////////////////////////////////////
// initialize map
procedure TForm1.ButtonInitMapClick(Sender: TObject);
begin
  n := 0;
  AddTown(0.5,0.5);
end;

end.
