;;; -*- Mode: LISP; Package: atn; Syntax: Common-lisp; -*-
;;; **************************************************************************
;;;
;;; PORTABLE AI LAB - IDSIA
;;;
;;; **************************************************************************
;;;
;;; Filename:   atn-graph
;;; Short Desc: Graphing Tool for atns
;;; Version:    1.0
;;; Status:     Experimental
;;; Last Mod:   16 June 1992
;;; Author:     Paolo Cattaneo
;;;


(in-package :atn)



(defclass graphic-node ()
      ((node-name
           :accessor node-name
           :initarg :node-name
           :initform nil)
       (x 
           :accessor x
           :initarg :x
           :initform 0)
       (y 
           :accessor y
           :initarg :y
           :initform 0)
       (arcs 
           :accessor arcs
           :initarg :arcs
           :initform nil)
       (radpos
           :accessor radpos
           :initarg :radpos
           :initform 0)))



(defclass network-window ()
      ((net-window
            :accessor net-window
            :initarg :net-window
            :initform nil)
       (node-list
            :accessor node-list
            :initarg :node-list
            :initform nil)))

(defclass q-window (bitmap-window)
      ())



(defmethod initialize-instance :after ((q q-window) &rest whatever)
     (declare (ignore whatever))
     (push q *open-atn-windows*))

(defmethod window-close ((q q-window))
     (delete  q *open-atn-windows*)
     (close q))



(defun net-graphics ()
     (create-windows *net*))



(defun create-windows (net)
      (let* ((width (box-width (window-interior *atn-main-window*)))
                (height (box-height (window-interior *atn-main-window*)))
                (number-of-windows (length net))
                (wh (round (/ width number-of-windows)))
                (counter 0)
                win)
          (dolist (subnet net)
                (let* ((netname (car subnet))
                          (new-window
                           (open-stream 'q-window *atn-main-window* :output
                                 :title (write-to-string netname)
                                 :window-border :dialog-box
                                 :user-closable nil :user-resizable nil
                                 :user-movable t :user-scrollable nil
                                 :user-shrinkable nil
                                 :window-exterior (make-box
                                                                          (* counter wh)
                                                                          (- height wh)
                                                                          (+ (* counter wh) wh)
                                                                          height)
                                 :view-font *nodefont*))
                          (nlist (create-and-place-nodes subnet)))
                    (push
                          (setq win
                                (make-instance 'network-window
                                       :net-window new-window
                                       :node-list nlist))
                          *usedwindows*)
                    (draw-nodes win)
                    (treat-arcs win)
                    (setq counter (1+ counter))))))
                


(defmethod center-w ((w window))
     (make-position
           (round (/ (box-width (window-interior w)) 2))
           (round (/ (box-height (window-interior w)) 2))))


(defun width (point)
     (position-x point))

(defun height (point)
     (position-y point))

(defun create-and-place-nodes (subnet)
     (setq *lastnet* nil)
     (let* ((r (- pi))
             (nodes-objects nil)
             (nodes (cdr subnet))
             (counter 0)
             (step nil))
         (if (eql (caar nodes) 'regs)
            (setq nodes (cdr nodes)))
         (if (eql (caar nodes) 'init)
            (setq nodes (cdr nodes)))
         (setq step (/ (* 2 pi) (length nodes)))
         (dolist (anode nodes)
              (push (make-instance 'graphic-node
                              :radpos (+ r (* step counter))
                              :node-name (car anode)
                              :arcs (cdr anode))
                   nodes-objects)
              (setq counter (1+ counter)))
         nodes-objects))



(defun close-net ()
     (dolist (nwindow *usedwindows*)
           (when (windowp (net-window nwindow))
                  (kill-window nwindow)))
     (setq *usedwindows* nil))

(defmethod kill-window ((n network-window))
     (window-close (net-window n)))



(defmethod draw-node (lista)
     (dolist (item lista)
          (draw-nodes item)))



(defun draw-inverted-oval (win h v radius node)
     (let ((cfont (font win)))
         (fill-circle
            win
            (make-position h v)
            radius)
         (sleep 0.5)
         (erase-contents-circle
             win
             (make-position h v)
             radius)
         (setf (font win) *nodefont*)
         (move-to win
               (make-position
                     (- (x node) (round (/ (font-size (font win)) 5)))
                     (- (y node) (round (/ (font-size (font win)) 2)))))
         (princ (char (write-to-string (node-name node)) 0) win)
         (setf (font win) cfont)))
         



 (defun draw-oval (win h v radius)
     (draw-circle
         win
         (make-position h v)
         radius))
               
         

               


(defmethod draw-nodes ((n network-window))
      (let* ((win (net-window n))
                (first-node t)
                (cfont (font win))
                (center-h (position-x (center-w win)))
                (center-v (position-y (center-w win)))
                (mid-h (/ center-h 1.75))
                (mid-v (/ center-v 1.75)))
          (setf (font win) *nodefont*)
          (dolist (item (reverse (node-list n)))
                (move-to win (center-w win))
                (let ((xc (- center-h (round (* -1 (sin (radpos item)) mid-h))))
                        (yc (- center-v (round (* -1 (cos (radpos item)) mid-v)))))
                    (move-to win (make-position xc yc))
                    (setf (x item) xc)
                    (setf (y item) yc)               
                    (draw-oval win xc yc
                         ;                        (point-h (pen-position win))
                         ;                        (point-v (pen-position win))
                         *nodesize*)
                    (when first-node
                           (draw-arrow win xc (- yc (* 2 *nodesize*)) xc (- yc *nodesize*))
                           (setf first-node nil))
                    (move-to win
                           (make-position
                                  (- xc (round (/ (font-size (font win)) 5)))
                                  (- yc (round (/ (font-size (font win)) 2)))))
                    (princ (char (write-to-string (node-name item)) 0) win)
                    ))
          (setf (font win) cfont)))




(defun draw-arrow (win x1 y1 x2 y2)
     (let* ((angle (if (= x1 x2) 
                           (if (< y1 y2) (- (/ pi 2)) (/ pi 2))
                           (atan (/ (- y2 y1) (- x2 x1)))))
             (posangle (cond ((> y2 y1) (if (> x2 x1)
                                                          (+ angle pi)
                                                          angle))
                                       ((> x2 x1) (+ angle pi))
                                       (t angle)))
             (angle-near (- posangle *arrowtheta*))
             (angle-far (+ posangle *arrowtheta*))
             (sx1 (+ x2 (round (* (cos angle-near) *arrowx*))))
             (sy1 (+ y2 (round (* (sin angle-near) *arrowx*))))
             (sx2 (+ x2 (round (* (cos angle-far) *arrowx*))))
             (sy2 (+ y2 (round (* (sin angle-far) *arrowx*))))
             )
         ;;(print posangle)
         (fill-polygon win
            (list 
               (make-position x2 y2)
               (make-position sx1 sy1)
               (make-position sx2 sy2)))))



(defun str-arc-label (arc)
      (let* ((arc-type (car arc))
                (lab (cadr arc))
                (wlab (unless (or
                                                 (eq arc-type 'jump)
                                                 (eq arc-type 'tst))
                                  (if (or
                                             (symbolp lab)
                                             (stringp lab))
                                     lab
                                     (first lab))))) ;;; Here it is a problem with
                                                            ;;; "multiple labels"
          (cond ((eq arc-type 'jump)
                       (string-downcase (write-to-string 'l)))
                      ((eq arc-type 'tst)
                       (string-upcase (write-to-string 'l)))
                      ((eq arc-type 'cat)
                       (string-downcase (write-to-string lab)))
                      ((eq arc-type 'push)
                       (string-upcase (write-to-string lab)))
                      ((eq arc-type 'word)
                       (string-upcase (write-to-string wlab)))
                      (t (write-to-string lab)))))



(defun draw-arc (win x1 y1 x2 y2 arc)
      (let* ((arc-type (car arc))
                (posangle (if (= x1 x2)
                                        (if (> y1 y2) (- (/ pi 2)) (/ pi 2))
                                        (atan (/ (- y2 y1) (- x2 x1)))))
                (angle (if (> x1 x2)
                                 (+ pi posangle)
                                 posangle))
                (delta-x (* (cos angle) *nodesize*))
                (delta-y (* (sin angle) *nodesize*))
                (sx1 (round (+ x1 (* (cos angle) *nodesize*))))
                (sx2 (round (- x2 (* (cos angle) *nodesize*))))
                (sy1 (round (+ y1 (* (sin angle) *nodesize*))))
                (sy2 (round (- y2 (* (sin angle) *nodesize*))))
                (mid-sx (round (/ (+ sx1 sx2) 2)))
                (mid-sy (round (/ (+ sy1 sy2) 2)))
                (cfont (font win))
                (lab (str-arc-label arc)))
          (move-to win (make-position sx1 sy1))
          (draw-to win (make-position sx2 sy2))
          (setf (font win) (label-font arc-type))
          (move-to win
                 (make-position
                        (- mid-sx (round (/ (max-string-size lab) 2)))
                        (- mid-sy (/ (font-size (font win)) 2))))
          (display-stringa win lab)
          (draw-arrow win sx1 sy1 sx2 sy2)
          (setf (font win) cfont)))



(defun draw-arc1 (win x1 y1 x2 y2 arc arc-no)
      (let* ((arc-type (car arc))
                (arc-offset (* (if (oddp arc-no) (1+ (truncate arc-no 2)) (truncate arc-no 2)) *nodesize*))
                (arc-angle (atan (/ arc-offset *arc-start*)))
                (delta-angle (if (oddp arc-no) (- arc-angle) arc-angle))
                (arc-start (/ arc-offset (sin arc-angle)))
                (posangle (if (= x1 x2)
                                        (if (> y1 y2) (- (/ pi 2)) (/ pi 2))
                                        (atan (/ (- y2 y1) (- x2 x1)))))
                (angle (if (> x1 x2) (+ pi posangle) posangle))
                (angle1 (if (> posangle 0) (- angle delta-angle) (+ angle delta-angle)))
                (angle2 (if (> posangle 0) (+ angle delta-angle) (- angle delta-angle)))
                (delta-x1 (* (cos angle1) *nodesize*))
                (delta-x2 (* (cos angle2) *nodesize*))
                (delta-y1 (* (sin angle1) *nodesize*))
                (delta-y2 (* (sin angle2) *nodesize*))
                (delta-xx1 (* (cos angle1) arc-start))
                (delta-xx2 (* (cos angle2) arc-start))
                (delta-yy1 (* (sin angle1) arc-start))
                (delta-yy2 (* (sin angle2) arc-start))
                (sx1 (round (+ x1 delta-x1)))
                (sx2 (round (- x2 delta-x2)))
                (sy1 (round (+ y1 delta-y1)))
                (sy2 (round (- y2 delta-y2)))
                (sxx1 (round (+ x1 delta-xx1)))
                (sxx2 (round (- x2 delta-xx2)))
                (syy1 (round (+ y1 delta-yy1)))
                (syy2 (round (- y2 delta-yy2)))
                (mid-sx (round (/ (+ sxx1 sxx2) 2)))
                (mid-sy (round (/ (+ syy1 syy2) 2)))
                (cfont (font win))
                (lab (str-arc-label arc)))
          (move-to win (make-position sx1 sy1))
          (draw-to win (make-position sxx1 syy1))
          (draw-to win (make-position sxx2 syy2))
          (draw-to win (make-position sx2 sy2))
          (setf (font win) (label-font arc-type))
          (move-to win
                 (make-position
                        (- mid-sx (round (/ (max-string-size lab) 2)))
                        (- mid-sy (round (/ (font-size (font win)) 2)))))
          (display-stringa win lab)
          (draw-arrow win sxx2 syy2 sx2 sy2)
          (setf (font win) cfont)))



(defun label-font (arc-type)
      (cond ((eq arc-type 'push) *push-font*)
                  ((or (eq arc-type 'jump)
                          (eq arc-type 'tst))
                   *jump-font*)
                  ((eq arc-type 'cat) *cat-font*)
                  (t *word-font*)))



(defun max-string-size (stringa)
     (stream-string-width *atn-main-window* stringa))

(defun display-stringa (win stringa)
     (let ((string-list (coerce stringa 'list)))
         (dolist (c string-list)
              (princ c win))))


(defun draw-arcs (lista)
     (dolist (item lista)
          (treat-arcs item)))



(defun how-many (elm lst)
      (cond ((null lst) 0)
                  ((eq (first lst) elm) (1+ (how-many elm (rest lst))))
                  ((how-many elm (rest lst)))))



(defmethod treat-arcs ((n network-window))
      (let* ((win (net-window n))
                nodes-to)
          (dolist (node (node-list n))
                (setq nodes-to nil)
                (dolist (arc (arcs node))
                      (cond
                                  ((eq (car arc) 'pop)
                                   (draw-oval win (x node) (y node) (+ 2 *nodesize*)))
                                  ((or
                                         (eq (car arc) 'push)
                                         (eq (car arc) 'cat)
                                         (eq (car arc) 'word)
                                         (eq (car arc) 'jump)
                                         (eq (car arc) 'tst))
                                   (let ((node-to-name
                                            (cond
                                                        ((or
                                                               (eq (car arc) 'jump)
                                                               (eq (car arc) 'tst))
                                                         (cadr arc))
                                                        (t (caddr arc)))))
                                       (if (eq (node-name node) node-to-name)
                                          (draw-arc-to-itself node win arc)
                                          (let* ((node-to (search-node-by-name node-to-name (node-list n)))
                                                    (x-to (x node-to))
                                                    (y-to (y node-to))
                                                    (arc-no (how-many node-to nodes-to)))
                                              (if (zerop arc-no)
                                                 (draw-arc win (x node) (y node) x-to y-to arc)
                                                 (draw-arc1 win (x node) (y node) x-to y-to arc arc-no)) 
                                              (push node-to nodes-to))))))))))



(defun search-node-by-name (node nodelist)
     (if (eq (node-name (car nodelist)) node) (car nodelist)
        (search-node-by-name node (cdr nodelist))))

(defun search-net-by-name (net netlist)
     (if (string= (stream-title (net-window (car netlist))) 
             (write-to-string net)) (car netlist)
        (search-net-by-name net (cdr netlist))))



(defmethod draw-arc-to-itself ((node graphic-node) win arc)
      (let* ((arc-type (car arc))
                (cfont (font win))
                (center-h (position-x (center-w win)))
                (center-v (position-y (center-w win)))
                (mid-h (/ center-h 1.75))
                (mid-v (/ center-v 1.75))
                (sin-an (sin (radpos node)))
                (cos-an (cos (radpos node)))
                (radius (truncate
                                   (- (box-width (window-interior win)) 
                                      (* 2 (+ *nodesize* *arc-offset* *window-border* *window-offset*)))
                                   2))
                (xca (round (- center-h (* -1 (sin (radpos node)) (+ *arc-offset* mid-h)))))
                (yca (round (- center-v (* -1 (cos (radpos node)) (+ *arc-offset* mid-v)))))
                (xcl (- xca (round (* -1 sin-an *nodesize*))))
                (ycl (- yca (round (* -1 cos-an *nodesize*))))
                (sin-ar (round (* sin-an *nodesize*)))
                (cos-ar (round (* -1 cos-an *nodesize*)))
                (x1 (- (x node) cos-ar))
                (x2 (- xca cos-ar))
                (y1 (- (y node) sin-ar))
                (y2 (- yca sin-ar))
                (lab (str-arc-label arc))
                (new-font (label-font arc-type)))
          
          (draw-oval win xca yca *nodesize*)             ;; draw the arc
          (setf (font win) *nodefont*)
          (erase-contents-circle
               win
               (make-position (x node) (y node))
               (1- *nodesize*))
          (setf (font win) *nodefont*)
          (move-to win
                 (make-position
                        (- (x node) (round (/ (font-size (font win)) 5)))
                        (- (y node) (round (/ (font-size (font win)) 2)))))
          (princ (char (write-to-string (node-name node)) 0) win)
          (setf (font win) new-font)
          (move-to win
                 (make-position
                        (- xcl (round (/ (max-string-size lab) 2)))
                        (- ycl (round (/ (font-size new-font) 2)))))
          (display-stringa win lab)
          (setf (font win) cfont)                      ;; reset window font
          (draw-arrow win x1 y1 x2 y2)))



(defun move-cursor-to (subnet node)
     (if *hilitenet*
        (highlight-node *hilitenet* *hilitenode*))
     (if subnet
        (highlight-node subnet node))
     (setq *hilitenet* subnet)
     (setq *hilitenode* node))


(defun highlight-node (subnet node)
     (let* ((net (search-net-by-name subnet *usedwindows*))
             (win (net-window net))
             (rnode (search-node-by-name node (node-list net))))
         (if (not (same-net *lastnet* win))
                (setq *lastnet* win))
         (if win
                (draw-inverted-oval win (x rnode) (y rnode) (1- *nodesize*) rnode))))



(defmethod set-zoom ((win window))
     (let* ((size (make-point  (/ *screen-width* 2) (+ 40 (/ *screen-width* 4))))
             (position (make-point (/ *screen-width* 4) 40)))
         (set-window-zoom-position win position)
         (set-window-zoom-size win size)))

(defun same-net (n1 n2)
     (eq n1 n2))

