;;; -*- Mode: LISP; Package: atn; Syntax: Common-lisp; -*-
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - IDSIA LUGANO
;;;  
;;; ************************************************************************
;;;
;;; Filename:   atp.lex.cl
;;; Short Desc:  Lexical data structure manager
;;; Version:    1.0
;;; Status:     Review
;;; Last Mod:   8.9.91 - FB
;;; Author:     Mike Lenz
;;;
;;; --------------------------------------------------------------------------
;;; Change History:
;;;
;;;
;;; --------------------------------------------------------------------------


;;; ==========================================================================
;;; PACKAGE DECLARATIONS
;;; ==========================================================================

(in-package :atn)

;;
;; M. Lenz
;; pAIL ATN module
;;
;; Lexical data structure manager
;; 19-8-91
;;



(defun loadlex (filename)
      (format-display *atn-output-window*
          "Loading lexicon from ~A~%" filename)
      (when filename
             (with-open-file (infile filename :direction :input) 
                  (setq *lexicon* nil)
                  (do ((entry (read infile nil 'eof)
                           (read infile nil 'eof)))
                         ((eq 'eof entry) (setq *lexicon* (nreverse *lexicon*)))
                        (when entry (push entry *lexicon*)))
                  (store-lexicon *lexicon*))
             *lexicon*))



(defun create-lex-window (lexicon)
      (let (not-close)
          (when (windowp *lex-dialog*)
                 (setq not-close (not (window-close *lex-dialog*))))
          (unless not-close
                (setq *lex-dialog*
                      (open-dialog ()
                            'atn-lex-window *atn-main-window* :pop-up-p nil
                            :background-color (make-rgb :red 0 :green 128 :blue 128) 
                            :user-closable nil :user-scrollable nil :user-resizable t
                            :user-shrinkable t 
                            :sequence (lexicon-to-sequence lexicon)
                            :button-menu-bar '(("Save" 'save-lex-window)
                                                                 ("Close" 'close-lex-window)
                                                                 ("Add" 'new-word)
                                                                 ("Delete" 'delete-word)
                                                                 ("Find" 'find-word))
                            :title (concatenate 'string
                                           "Lexicon: "
                                           (file-namestring *current-lex-file*))
                            :visible-box (make-box 0 0 0 0))))))



(defun close-lex-window (&rest x)
      (when (window-close *lex-dialog*)
             (set-menu-before-load-lexicon))
      t)

(defun save-lex-window (&rest x)
     (save-atn-lexicon))


(defun save-atn-lexicon () 
      (when (save-lexicon) 
             (setf (stream-title *lex-dialog*)
                       (concatenate 'string 
                             "Lexicon : "  
                             (file-namestring *current-lex-file*)))))



(defun save-lexicon ()
      (let ((filename (pop-up-save-file-dialog
                                        *atn-main-window*
                                        (make-pathname :host nil :name *pail-directory*)
                                        "Save Lexicon File"
                                        "Please, type in the new lexicon file."
                                        (file-namestring *current-lex-file*)
                                        (concatenate 'string *pail-directory* "atn\\lexicon\\"))))
          (when filename
                 (savedata filename)
               (setf *current-lex-file* filename))))



(defun savedata (filename)
      (when filename
             (progn
                  (with-open-file 
                       (port filename :direction :output :if-exists :supersede)
                       (format-display *atn-output-window*
                            "~%Saving Lexicon to ~A~%~%" filename)
                       (dumpdata *lexicon* port))
                  (setf (save-p (wsequence *lex-dialog*)) nil)
                  t)))


(defun dumpdata (lexicon port)
      (mapc #'(lambda (entry)
                           (print entry port))
             lexicon))



(defun new-word (&rest x)
      (multiple-value-bind (word other value)
              (ask-user-for-string
                    "Please, enter new word?"
                    nil "OK" "Cancel")
             (when (and word (string= value "OK") (string/= word ""))
                    (new-entry word))))
 


(defun new-entry (word)
      (when word
             (multiple-value-bind (category other value)
                     (ask-user-for-string "Category?" nil "OK" "Cancel")
                    (when (string= value "OK")
                           (multiple-value-bind (features other value)
                                   (ask-user-for-string "Features?" nil "Add" "Cancel")
                                  (when (string= value "Add")
                                       (when (not features)
                                            (setq features "" ))
                                       (format t "FEatures ~a~%" features)
                                         (let* ((table (wsequence *lex-dialog*))
                                                   (cells (ordered-insert
                                                                   (show-entry (cons (string-upcase word)
                                                                                                 (cons (string-upcase category)
                                                                                                       (list (string-upcase features)))))
                                                                   (dialog-item-range table)))
                                                   the-features)
                                             (set-dialog-item-range table cells)
                                             (setf (save-p table) t)
                                             (do ((feature) (start 0))
                                                    ((equal feature 'done))
                                                   (multiple-value-setq (feature start)
                                                          (read-from-string (string-upcase features) nil 'done :start start))
                                                   (unless (equal feature 'done)
                                                         (push feature the-features)))
                                             (when (string= features "")
                                                  (setq features nil))
                                             (setq *lexicon* 
                                                   (ordered-insert1
                                                        (cons (read-from-string word)
                                                              (cons (read-from-string category)
                                                                    (reverse the-features)))
                                                        *lexicon*))
                                             (store-lexicon *lexicon*))))))))
                         
  

(defun delete-word (&rest x) 
      (if (dialog-item-value (wsequence *lex-dialog*))
         (let ((rules (delete-from-sequence
                               *lex-dialog*
                               (dialog-item-value (wsequence *lex-dialog*)))))
             (when rules
                    (setf *lexicon* (sequence-to-lexicon rules))))
         (pop-up-message-dialog *atn-main-window* nil
               "You must select a item first!" nil "OK")))
 

 
(defun select-entries-to-delete (lista)
      (when lista
             (select-item-from-list
                  *atn-main-window* 
                   lista 
                   "Select entry to delete:")))
  

                           
(defun delete-entries (lista)
      (when lista
             (let ((table (wsequence *lex-dialog*)))
                 (set-dialog-item-range table
                       (my-delete lista (dialog-item-range table)))
                 (reset-table table)
                 (setf *lexicon*
                           (sequence-to-lexicon (dialog-item-range table))))))


    
(defun find-word (&rest x)
      (multiple-value-bind (word other value)
              (ask-user-for-string
                    "Which word would you like to find?"
                    nil "Find" "Cancel")
             (when (and (string= value "Find") word (string/= word ""))
                    (let ((entries (find-entries (string-upcase word)))) 
                        (if entries
                           (list-entries word entries) 
                           (pop-up-message-dialog *atn-main-window*
                                 nil
                                 (concatenate 'string
                                       "Sorry, no entries for " 
                                       word ".")
                                 nil "OK"))))))
 


;;; (defun find-entries (word)
;;;       (let* ((current-word-list (when *lex-dialog*
;;;                                                           (dialog-item-range (wsequence *lex-dialog*))))
;;;                 (result nil))
;;;           (when *lex-dialog*
;;;                  (dolist (item current-word-list (reverse result))
;;;                        (when (string= word item :end2 (length (coerce word 'list)))
;;;                               (push item result))))))
      
      
      
(defun find-entries (word)
      (let* ((current-word-list (when *lex-dialog* *lexicon*))
                (result nil))
          (when *lex-dialog*
                 (dolist (item current-word-list (reverse result))
                       (when (string=
                                         word
                                         (write-to-string (car item))
                                         :end2 (length (coerce word 'list)))
                              (push
                                    (show-entry (list
                                                                (first item)
                                                                (second item)))
                                    result))))))
            


(defun list-entries (word lista)
      (when lista
             (let ((item (select-item-from-list
                                      *atn-main-window* 
                                      lista
                                      (concatenate 'string "Entries for " word))))
                 (when item
                        (dolist (value (dialog-item-range (wsequence *lex-dialog*)))
                              (when (string= item value
                                                :end2 (length item))
                                     (set-dialog-item-value (wsequence *lex-dialog*) value)))))))



(defun sequence-to-lexicon (sequence)
      (let (result)
          (dolist (entry sequence (reverse result))
                (let (word category done features (start 0))
                    (multiple-value-setq (word start)
                           (read-from-string entry nil 'done :start start))
                    (multiple-value-setq (category start)
                           (read-from-string entry nil 'done :start (+ 2 start)))
                    (multiple-value-setq (done start)
                           (read-from-string entry nil 'done :start start))
                    (if (equal done 'done)
                       (push (list word category) result)
                       (progn 
                            (do ((feature) (start1 start))
                                   ((equal feature 'done))
                                  (multiple-value-setq (feature start1)
                                         (read-from-string entry nil 'done :start start1))
                                  (unless (equal feature 'done)
                                        (push feature features)))
                            (push (append
                                              (list word category)
                                              (reverse features))
                                  result)))))))
                    
                           
                
;;;                 (push (list (multiple-value-bind (a b)
;;;                                             (read-from-string entry)
;;;                                            (setq second b)
;;;                                            a)
;;;                                (read-from-string entry nil 'done :start (+ 2 second)))
;;;                       result))
;;;           (reverse result)))
;;; 



(defun lexicon-to-sequence (lexicon)
      (let (result)
          (setq common-lisp::*print-pretty* nil)
          (dolist (item lexicon result)
                (setq result
                      (ordered-insert (show-entry item) result)))))

;          (mapcar (function show-entry)
;                 lexicon))       



(defun show-entry (entry)
      (let ((features ""))
          (if (cddr entry)
             (format nil  "~A : ~A :~A"
                  (car entry)
                  (cadr entry)
                  (dolist (item (cddr entry) features)
                        (setq features
                              (concatenate 'string
                                    features " "
                                    (write-to-string item)))))
             (format nil  "~A : ~A"
                  (car entry)
                  (cadr entry)))))





;;
;; Defined herein:
;;
;; .  read-word
;; .  lex-word
;; .  lex-string
;; .  lookup-word
;; .  lookup-string
;; .  clear-lexicon
;;

;;
;; Returns next word of *input* as a string OR a list,
;; ignoring punctuation and spaces.
;; (Assumes global *input-length* has also been set.)
;; Also returns next input-head position of the string
;; via *nextpos*.
;; Note 2 loops: the first swallows any whitespace to the left
;; of the next word; the 2nd gets the word.
;;


(defun read-word (startpos)
  (do ((pos startpos (+ pos 1)))
      ((or (= pos *input-length*)
	   (alphanumericp (elt *input* pos)))
       (if (= pos *input-length*)
	   (return-from read-word nil))
       (setq startpos pos))
      )
  (do ((pos startpos (+ pos 1)))
      ((or (= pos *input-length*)
	   (not (alphanumericp (elt *input* pos))))
       (setq *nextpos* pos)
       (subseq *input* startpos pos))
      ))


;;
;; *lextree* stores our lexical data structure: a letter tree.
;; Each <node> is of the form
;;   (<character> (<node>)* <value1> <value2> ... <valuen>)
;;
;; e.g. after
;;   (lex-string "dog" 33)
;;   (lex-string "dog" 44)
;;   (lex-string "do" 55)
;;   (lex-string "" 66)
;; *lextree* looks like:
;;
;; (((#\d ((#\o ((#\g nil 44 33)) 55)))) 66)
;;
;; (Note that <character> of the root node is nil, for attaching
;;  values to the string "".)
;;



(defun clear-lexicon ()
  (setf (car *lextree*) nil)
  (setf (cdr *lextree*) nil))

;;
;; Associates value with charlist (a list of characters)
;; in the lexical tree.
;;

(defun lex-word (charlist value)
  (let ((curlist *lextree*))
    (dolist (achar charlist)
      (let ((entry (assoc achar (car curlist) :test #'char=)))
	(if entry
	    (setq curlist (cdr entry))
	  (let* ((newtree (copy-list '(nil))))
	    (push (append (list achar) newtree) (car curlist))
	    (setq curlist newtree)))))
    (push value (cdr curlist))))

;;
;; Returns the list (value1 value2 ... valuen) of values
;; associated with charlist in the lexical tree; or nil
;; if the word is not in the lexicon (i.e. no values have
;; been associated).
;;

(defun lookup-word (charlist)
  (let ((curlist *lextree*))
    (dolist (achar charlist)
      (if (null curlist)
	  (return-from lookup-word nil))
      (setq curlist (cdr (assoc achar (car curlist) :test #'char=))))
    (cdr curlist)))

;;
;; Like lex-word and lookup-word, respectively, but these
;; accept a string argument rather than a character list.
;;

(defun lex-string (str val)
  (lex-word (str-to-list str) val))

(defun lookup-string (str)
  (lookup-word (str-to-list str)))


;;
;; Utility routines
;;

(defun str-to-list (str)
  (coerce str 'list))

(defun ordered-insert (elem list)
      (if list
         (if (string< elem (car list))
            (cons elem list)
            (cons (car list) (ordered-insert elem (cdr list))))
         (list elem)))



(defun ordered-insert1 (elem list)
      (if list
         (if (string< (car elem) (caar list))
            (cons elem list)
            (cons (car list) (ordered-insert1 elem (rest list))))
         (list elem)))

